# -----------------------------------------------------
# Utility functions for spyder editor integration
# -----------------------------------------------------

import sys

# -----------------------------------------------------
def isInstalledSpyder():
    "Function allowing to check wether spyder editor is installed"
   
    try:
        import spyder
        return True
    except:
        return False
        
# -----------------------------------------------------
def isSpyderIntegrationSupported():
    "Function allowing to check wether spyder editor integration is supported"
   
    try:
        import spyder
        if spyder.version_info[0] >= 3:
            return True
    except:
        return False
        
# -----------------------------------------------------
class IPSDKRopePatch:
    "Class allowing to retrieve hack rope autocompletion sub module"
    
    # list of ipsdk sub modules
    PyIPSDKSubModules = []
    
    # rope initial extension modules function
    old_extension_modules = None
    
    # initialize rope hack
    @staticmethod
    def apply():
        try:
            # apply rope patch
            import rope.base.project
            from rope.base import pycore
            IPSDKRopePatch.old_extension_modules = pycore.PyCore.extension_modules.fget
            pycore.PyCore.extension_modules = property(IPSDKRopePatch.new_extension_modules)
            print("IPSDK Rope Patch applied");
        except:
            print("Unexpected error while apply ipsdk rope patch:\n" + str(sys.exc_info()[0]) + " : " + str(sys.exc_info()[1]))
            return False
            
    # clear rope hack
    @staticmethod
    def clear():
        try:
            # remove rope patch
            if IPSDKRopePatch.old_extension_modules:
            
                from rope.base import pycore
                pycore.PyCore.extension_modules = property(IPSDKRopePatch.old_extension_modules)
                print("IPSDK Rope Patch removed");
        except:
            print("Unexpected error while removing ipsdk rope patch:\n" + str(sys.exc_info()[0]) + " : " + str(sys.exc_info()[1]))
            return False
    
    @staticmethod
    def new_extension_modules(PyCoreObject):
        
        try:
        
            # initialize py ipsdk sub module list if needed
            if not IPSDKRopePatch.PyIPSDKSubModules:
            
                # find PyIPSDK sub modules
                import PyIPSDK
                import pkgutil
                def catch_exceptions(module):
                    pass
                submods = pkgutil.walk_packages(PyIPSDK.__path__,
                                                PyIPSDK.__name__ + '.',
                                                catch_exceptions)
                                                
                # parse and find valid PyIPSDK sub modules
                for sm in submods:
                    sm_name = sm[1]
                    seq = sm_name.split('.')
                    if len(seq) == 2 and seq[0][0:len("PyIPSDKBase_")] == "PyIPSDKBase_" and seq[1] != "PyIPSDKBase":
                        curModPrettyName = "PyIPSDK." + seq[1]
                        IPSDKRopePatch.PyIPSDKSubModules.append(curModPrettyName)
            
            # call of initial method
            result = IPSDKRopePatch.old_extension_modules(PyCoreObject)
            
            # update of returned list with ipsdk sub modules
            result.update(IPSDKRopePatch.PyIPSDKSubModules)
            
            return result
        except:
            print("Unexpected error while searching for rope extension modules:\n" + str(sys.exc_info()[0]) + " : " + str(sys.exc_info()[1]))
            return []
                    
# -----------------------------------------------------
def is_disabled_autocompletion():
    "function allowing to check wether autocompletion for IPSDK libraries under spyder editor has been explicitly disabled"
    
    # check for spyder installation
    if isInstalledSpyder() == False:
        return True
    
    # check wether spyder integration is supported by version
    if isSpyderIntegrationSupported() == False:
        print("Spyder editor version is <3.0.0, can't manage auto completion")
        return True
        
    try:
        # load prefered modules database
        from spyder.config.base import get_conf_path
        from pickleshare import PickleShareDB
        from spyder.utils.introspection.module_completion import get_submodules
        MODULES_PATH = get_conf_path('db')
        modules_db = PickleShareDB(MODULES_PATH)
        
        # check for disable tag
        if not 'submodules' in modules_db:
            return False
        if "Disable_PyIPSDK" in modules_db['submodules']:
            return True
        else:
            return False
    except:
        print("Unexpected error:\n" + str(sys.exc_info()[0]) + " : " + str(sys.exc_info()[1]))
        return True
            
# -----------------------------------------------------
def active_autocompletion():
    "function allowing to activate autocompletion for IPSDK libraries under spyder editor"
    
    # check for spyder installation
    if isInstalledSpyder() == False:
        print("Spyder seems not to be installed (or editor version is <3), can't integrate auto completion")
        return False
        
    try:
        # load prefered modules database
        from spyder.config.base import get_conf_path
        from pickleshare import PickleShareDB
        MODULES_PATH = get_conf_path('db')
        modules_db = PickleShareDB(MODULES_PATH)
        if not 'submodules' in modules_db:
            return False
        subModules = modules_db['submodules']
        
        # remove disable tag if needed
        if "Disable_PyIPSDK" in subModules:
            subModules.remove("Disable_PyIPSDK")
          
        # add autocompletion capabilities
        if not 'PyIPSDK' in subModules:
            subModules.append('PyIPSDK')
            print("Autocompletion added for : " + 'PyIPSDK')
        modules_db['submodules'] = subModules
        
        # apply ipsdk rope patch if needed
        IPSDKRopePatch.apply()
        
        return True
    except:
        from spyder.utils.introspection.manager import (
            DEBUG_EDITOR, LOG_FILENAME, IntrospectionPlugin)
        from spyder.utils.debug import log_last_error
        log_last_error(LOG_FILENAME, "Unexpected error:\n %s : %s" % str(sys.exc_info()[0]) % str(sys.exc_info()[1]))
        print("Unexpected error:\n" + str(sys.exc_info()[0]) + " : " + str(sys.exc_info()[1]))
        return False
  
# -----------------------------------------------------
def deactive_autocompletion():
    "function allowing to de activate autocompletion for IPSDK libraries under spyder editor"
    
    # check for spyder installation
    if isInstalledSpyder() == False:
        print("Spyder seems not to be installed (or editor version is <3), can't manage auto completion")
        return False
        
    try:
        # load prefered modules database
        from spyder.config.base import get_conf_path
        from pickleshare import PickleShareDB
        from spyder.utils.introspection.module_completion import get_preferred_submodules
        MODULES_PATH = get_conf_path('db')
        modules_db = PickleShareDB(MODULES_PATH)
        
        # print info
        print("Removing autocompletion for spyder editor")
        
        # clear existing modules
        modules_db.pop('submodules')
        
        # update modules list
        get_preferred_submodules()
        
        # add disabled module tag
        modules_db['submodules'] += ["Disable_PyIPSDK"]
        
        # removing ipsdk rope patch if needed
        IPSDKRopePatch.clear()
        
        return True
    except:
        print("Unexpected error:\n" + str(sys.exc_info()[0]) + " : " + str(sys.exc_info()[1]))
        return False