# ------------------------------------------------------------------------------
# Installation script for IPSDK library python wrapping to DragonFly software
# 
# DragonFly is a trademark of ORS, Object Research Systems
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# function allowing to update ipsdk module
# ------------------------------------------------------------------------------

# taille de voxel
IPSDK_spacingDx = 1.0
IPSDK_spacingDy = 1.0
IPSDK_spacingDz = 1.0

# origine en 
IPSDK_spacingOx = 1.0
IPSDK_spacingOy = 1.0
IPSDK_spacingOz = 1.0


def updateIPSDKForDragonFly(ipsdkBaseModule):
    "function allowing to update ipsdk module"
    
    try:
        import ORSModel
    except:
        return
        
    # update of ipsdk base module
    import sys
    thisModule = sys.modules[__name__]
    ipsdkBaseModule.__dict__["fromDragonFly"] = thisModule.__dict__["fromDragonFly"]
    ipsdkBaseModule.__dict__["toDragonFly"] = thisModule.__dict__["toDragonFly"]
    ipsdkBaseModule.__dict__["toDragonFlyMultiROI"] = thisModule.__dict__["toDragonFlyMultiROI"]
    
# ------------------------------------------------------------------------------
# function allowing to import an image from DragonFly to IPSDK
# ------------------------------------------------------------------------------

def fromDragonFly(dragonFlyImageName):
    "function allowing to import an image from DragonFly to IPSDK"
    
    from ORSModel.ors import Channel, Managed
    import PyIPSDK

    global IPSDK_spacingDx
    global IPSDK_spacingDy
    global IPSDK_spacingDz
    global IPSDK_spacingOx
    global IPSDK_spacingOy
    global IPSDK_spacingOz
    
    # retrieve associated image
    objList = Managed.getAllInstancesOf(Channel.getClassNameStatic())
    for obj in objList:
         if obj.getTitle() == dragonFlyImageName:
             IPSDK_spacingDx = obj.getXSpacing()
             IPSDK_spacingDy = obj.getYSpacing()
             IPSDK_spacingDz = obj.getZSpacing()
             Img = PyIPSDK.fromArray(obj.getNDArray(0))
             calib = PyIPSDK.createGeometricCalibration3d(IPSDK_spacingDx,IPSDK_spacingDy,IPSDK_spacingDz,'mm')
             Img.setGeometricCalibration(calib)
             
             ori = obj.getOrigin()
             IPSDK_spacingOx = ori[0]
             IPSDK_spacingOy = ori[1]
             IPSDK_spacingOz = ori[2]
                 
             return Img
        
    raise NameError("Image " + dragonFlyImageName + " has not been found")
    return None
    
# ------------------------------------------------------------------------------
# function allowing to export an image from IPSDK to DragonFly
# ------------------------------------------------------------------------------

def toDragonFly(ipsdkImage, dragonFlyImageName = ""):
    "function allowing to export an image from IPSDK to DragonFly"
    
    from ORSModel import createChannelFromNumpyArray
    from ORSModel import orsVect
    
    import PyIPSDK
  
    global IPSDK_spacingDx
    global IPSDK_spacingDy
    global IPSDK_spacingDz
    global IPSDK_spacingOx
    global IPSDK_spacingOy
    global IPSDK_spacingOz
    
    # create and initialize channel from array
    dragonFlyChannel = createChannelFromNumpyArray(ipsdkImage.array)

    if dragonFlyImageName != "":
        dragonFlyChannel.setTitle(dragonFlyImageName)
        dragonFlyChannel.setOrigin(orsVect(IPSDK_spacingOx,IPSDK_spacingOy,IPSDK_spacingOz))
            
        if (ipsdkImage.hasGeometricCalibration()):
            calib = ipsdkImage.getGeometricCalibration()	        
            dragonFlyChannel.setXSpacing(calib.getXScale())
            dragonFlyChannel.setYSpacing(calib.getYScale())
            dragonFlyChannel.setZSpacing(calib.getZScale())
        else:
            dragonFlyChannel.setXSpacing(IPSDK_spacingDx)
            dragonFlyChannel.setYSpacing(IPSDK_spacingDy)
            dragonFlyChannel.setZSpacing(IPSDK_spacingDz)
    dragonFlyChannel.publish()
    return None
    
def toDragonFlyMultiROI(ipsdkImage, dragonFlyImageName = ""):
    "function allowing to export an image from IPSDK to DragonFly as a multi ROI"
    
    from ORSModel import createChannelFromNumpyArray
    from ORSModel import orsVect

    import PyIPSDK
  
    global IPSDK_spacingDx
    global IPSDK_spacingDy
    global IPSDK_spacingDz
    global IPSDK_spacingOx
    global IPSDK_spacingOy
    global IPSDK_spacingOz
    
    # create and initialize channel from array
    a = createChannelFromNumpyArray(ipsdkImage.array)

    b = a.getAsMultiROIInArea( 0, 0, 0, a.getXSize()-1,  a.getYSize()-1 , a.getZSize()-1 , None, None)
  
    if b != None:
        b.setTitle(dragonFlyImageName+"_multiROI")
        b.setOrigin(orsVect(IPSDK_spacingOx,IPSDK_spacingOy,IPSDK_spacingOz))
        
        if (ipsdkImage.hasGeometricCalibration()):
            calib = ipsdkImage.getGeometricCalibration()	        
            b.setXSpacing(calib.getXScale())
            b.setYSpacing(calib.getYScale())
            b.setZSpacing(calib.getZScale())
        else:
            b.setXSpacing(IPSDK_spacingDx)
            b.setYSpacing(IPSDK_spacingDy)
            b.setZSpacing(IPSDK_spacingDz)

    b.publish()    
    a.deleteObject()
    return None