# ------------------------------------------------------------------------------
# Installation script for IPSDK library python wrapping to Digisens software
# ------------------------------------------------------------------------------

# ------------------------------------------------------------------------------
# function allowing to update ipsdk module
# ------------------------------------------------------------------------------

def updateIPSDKForDigisens(ipsdk_base_module):
    """function allowing to update ipsdk module
    """

    try:
        import digisens as ds
    except:
        return

    # updates ipsdk base module
    import sys
    this_module = sys.modules[__name__]
    ipsdk_base_module.__dict__["fromDigisens"] = this_module.__dict__["fromDigisens"]
    ipsdk_base_module.__dict__["toDigisens"]   = this_module.__dict__["toDigisens"]
        
# ------------------------------------------------------------------------------
# function allowing to import an image from Digisens Scene to IPSDK
# ------------------------------------------------------------------------------

def fromDigisens( volume, bDeepCopy=False ):
    """
    function allowing to pass an image from Digisens Scene to IPSDK

    Parameters
    ----------
    volume: str or ds.dvolume
        If a string, the uniq name of an object already loded in Digisens scene. Usually
        the basename of a loaded volume, without its .vol extension.
        Otherwise, shall be the object from the Digisens scene graph.

    Returns
    -------
    out: PyIPSDK.Image
        The corresponding image, with calibration (voxel size).
    """

    import digisens as ds
    import PyIPSDK

    
    # retrieve associated image
    if type(volume) is str:
        try:
            volume = ds.scene()[volume]
        except:
            raise IndexError("Image " + volume + " has not been found.")

    if type(volume) is not ds.dvolume:
        raise ArgumentError("Argument 'volume' shall be a string or a ds.dvolume.")

    # NOTE: we don't force a deep copy anymore, because IPDSK does know how
    #       to handle non-contiguous ndarray
    #       eg to 32x32x32 when in GigaVoxel mode.
    # NOTE: currently returns a U16 (or less) version of the data, even
    #       when the original might be in F32...
    #
    # Other options are: region=ds.dregion and fill=float
    Img = PyIPSDK.fromArray( volume.proxy.toArray(deepCopy=bDeepCopy) )

    # Internal Digisens units are always mm, even if the interface display
    # them as um, inch, etc.
    # The parametrization order is :
    #createGeometricCalibration3d(xScale, yScale, zScale,       # Scale
    #                             xOrigin, yOrigin, zOrigin,    # Origin
    #                             qr, qi, qj, qk)               # Orientation
    calib = PyIPSDK.createGeometricCalibration3d( volume.volume.scale.x,
                                                  volume.volume.scale.y,
                                                  volume.volume.scale.z,
                                                  'mn',
                                                  0, 0, 0, 0, 0, 0, 0) # Fill these values with the image calibration
    Img.setGeometricCalibration(calib)
    return Img

# ------------------------------------------------------------------------------
# function allowing PyIPSDK export an image from IPSDK to Digisens Scene
# ------------------------------------------------------------------------------

def toDigisens(ipsdk_image, volume_path = None, semantic = None):
    """
    function allowing to export an image from IPSDK to Digisens Scene

    The result volume is displayed in the DigiXCT scene tree.

    Parameters
    ----------
    ipsdk_image: IPSDK.Image
        The source of the data.

    volume_name: str, optional
        The uniq name of an object already loded in Digisens scene. Usually
        the basename of a loaded volume, without its .vol extension.

    semantic: int
        Wether or not to interpolate data. Allowed values:
        - ds.proxy.DENSITY  (interpolated, the default when None is used)
        - ds.proxy.LABEL    (not interpolated)
        - ds.proxy.MASK     (idem)
    """

    import digisens as ds
    import PyIPSDK

    # We always need a name. If none given, the scene will make it uniq
    # automatically...
    if not volume_path:
        volume_path = 'ipdsk'
    if '.vol' not in volume_path:
        volume_path += '.vol'

    # Manage default semantic = DENSITY
    if semantic is None:
        semantic = ds.proxy.DENSITY

    # Make a new volume, by cloning from a NumPy 3D array
#    vol = ds.dvolume.fromArray(volume_path, model=ipsdk_image.array, semantic=semantic)
    vol = ds.dvolume.fromArray(volume_path, array=ipsdk_image.array, semantic=semantic)

    if  ipsdk_image.hasGeometricCalibration():
        # For Digisens, we always shall have millimeters
        # QUESTION: Shall we handle units here too ? How ?
        # Problem: IPSDK does not handle position or rotation ... so they are lost!
        calib = ipsdk_image.getGeometricCalibration()
        vol.volume.scale = ds.vec3f( calib.getXScale(),
                                     calib.getYScale(),
                                     calib.getZScale() )

    return vol