#!/usr/bin/env python

# ------------------------------------------------------------------------------
# Installation script for IPSDK library python wrapping
#
# This script allows to install IPSDK library python wrappers (PyIPSDK)
# for your python executable
#
# By default script install PyIPSDK into site package subdirectory of
# python installation
#
# User can specify a custom destination directory as input argument
#
# Note that this custom destination directory should be in path to be used
# within python
# ------------------------------------------------------------------------------

import os
import sys
import site
import shutil

# retrieve current directory
curDirectory = os.path.dirname(os.path.realpath(__file__))

# retrieve ipsdk version informations
ipsdkPythonPath = os.path.dirname(curDirectory)
sys.path.insert(0, ipsdkPythonPath)
from PyIPSDKUtils import getAvailablePythonVersions
from PyIPSDKUtils import getIPSDKVersionStr
from PyIPSDKUtils import getIPSDKRootDir
sys.path.remove(ipsdkPythonPath)

# check expected python version
availablePythonVersions = getAvailablePythonVersions()
systemVersionInfo = sys.version_info
bSystemVersionSupported = False
for curAvailablePythonVersion in availablePythonVersions:
    
    if systemVersionInfo.major == curAvailablePythonVersion[0] and systemVersionInfo.minor == curAvailablePythonVersion[1]:
        bSystemVersionSupported = True

if bSystemVersionSupported == False:
    msg = "Python version " + str(systemVersionInfo.major) + "." + str(systemVersionInfo.minor) + " is not supported by IPSDK library.\n"
    msg += "Supported versions are " + str(availablePythonVersions)
    raise RuntimeError(msg)
    
# retrieve site packages folder
if len(sys.argv) > 2:
    print("Invalid number of arguments, script takes only one argument")
    exit(-1)
if len(sys.argv) == 2:
    installDir = sys.argv[1]
else:
    installDir = site.getsitepackages()[0]
    if len(site.getsitepackages()) > 1:
        installDir = site.getsitepackages()[1]
print("Installing IPSDK library into python directory " + installDir)

# creation of IPSDK python package directory if needed
ipsdkPackageDir = os.path.join(installDir, "PyIPSDK")
if not os.path.exists(ipsdkPackageDir):
    try:
        os.makedirs(ipsdkPackageDir)
    except:
        raise RuntimeError("Can't create PyIPSDK python package sub directory " + ipsdkPackageDir + ", check for administrative privilege")

# copy of __init__.py file
inputInitFile = os.path.join(curDirectory, "__init__.pyipsdk")
outputInitFile = os.path.join(ipsdkPackageDir, "__init__.py")
try:
    shutil.copyfile(inputInitFile, outputInitFile)
except:
    raise RuntimeError("Can't create PyIPSDK python package file " + outputInitFile + ", check for administrative privilege")

# creation of IPSDK version file
ipsdkRootDir = os.path.normpath(getIPSDKRootDir())
ipsdkRootDir = ipsdkRootDir.replace('\\', '/')
outputVersionFile = "IPSDK_" + getIPSDKVersionStr() + ".py"
outputVersionFile = os.path.join(ipsdkPackageDir, outputVersionFile)
import codecs
f = codecs.open(outputVersionFile, mode='w', encoding="utf-8")
f.write("#!/usr/bin/env python\n")
f.write("# coding: " + "utf-8" + "\n")
f.write("import sys\n")
f.write("\n")
f.write("# function allowing to retrieve IPSDK installation root path\n")
f.write("def getIPSDKRootPath():\n")
f.write("    \"function allowing to retrieve IPSDK root path\"\n")
f.write("\n")
if systemVersionInfo.major == 2:
    f.write(u"    return (u\"" + ipsdkRootDir.decode(sys.getfilesystemencoding()) + u"\").encode(sys.getfilesystemencoding())\n")
if systemVersionInfo.major == 3:
    f.write("    return u\"" + ipsdkRootDir + "\"\n")
f.close()

# try to enable autocompletion for spyder editor
try:
    ipsdkPythonToolkitPath = os.path.join(ipsdkPythonPath, "toolkit")
    if os.path.exists(ipsdkPythonToolkitPath):
        sys.path.insert(0, ipsdkPythonToolkitPath)
        from spyder_utils import isInstalledSpyder, active_autocompletion
        if isInstalledSpyder() == True:
            active_autocompletion()
        sys.path.remove(ipsdkPythonToolkitPath)
except:
    print("Warning : Auto-completion activation for spyder has failed. Some packages may be broken.")
    print("This step will be ignored.")

# -----------------------------------------------------------------
# Create the file allowing the IPSDK Explorer binary to find python
pythonExePath = ""

# Find the python executable root directory
directoryDepth = 3
if os.name == 'posix':
    directoryDepth = 4

pythonRootPath = ipsdkPackageDir
for i in range(0,directoryDepth):
    pythonRootPath = os.path.abspath(os.path.join(pythonRootPath, os.pardir))

pythonConfigurationFileForExplorerDirPath = ""
if os.name == 'nt':
    pythonExePath = os.path.join(pythonRootPath, "python.exe")
    # Replace \ by /
    pythonExePath_split = pythonExePath.split("\\")
    sep = '/'
    pythonExePath = sep.join(pythonExePath_split)
    pythonConfigurationFileForExplorerDirPath = os.path.join(os.getenv('APPDATA'), 'ReactivIP/Explorer')
elif os.name == 'posix':
    pythonExePath = os.path.join(pythonRootPath, "bin/python")
    pythonConfigurationFileForExplorerDirPath = os.path.join(os.getenv("HOME"), '.local/ReactivIP/Explorer')

# Path to the configuration file
if not os.path.exists(pythonConfigurationFileForExplorerDirPath):
    os.makedirs(pythonConfigurationFileForExplorerDirPath)
pythonConfigurationFileForExplorer = os.path.join(pythonConfigurationFileForExplorerDirPath, 'pythonPath.txt')

# Write the path in to the configuration file
if os.path.exists(pythonConfigurationFileForExplorer):
    print("The file {} already existed and has been rewritten".format(pythonConfigurationFileForExplorer))
file = open(pythonConfigurationFileForExplorer, "w")
file.write(pythonExePath)
file.close()
print("Explorer configured")