#!/usr/bin/env python
# coding: utf-8
import sys
import os
import platform

# add python output directory to path
ipsdkRootDir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
ipsdkRootBinPath = os.path.join(ipsdkRootDir, "Output")
if os.path.exists(ipsdkRootBinPath) == False:
    ipsdkRootBinPath = os.path.join(ipsdkRootDir, "bin")
versionInfoPath = os.path.join(ipsdkRootBinPath, "python")

# import ipsdk version function
sys.path.insert(0, versionInfoPath)
from IPSDKVersion import getIPSDKInfo
sys.path.remove(versionInfoPath)

# ------------------------------------------------------------------------------
def getProjectConfig():
    "retrieve project configuration"
    
    # check for environment variable
    if "IPSDK_PYTHON_PROJECT_ROOT_PATH" in os.environ:

        # retrieve project root directory
        projectRootDir = os.environ["IPSDK_PYTHON_PROJECT_ROOT_PATH"]
        
        # retrieve project version file path
        projectRootBinPath = os.path.join(projectRootDir, "Output")
        if os.path.exists(projectRootBinPath) == False:
            projectRootBinPath = os.path.join(projectRootDir, "bin")
        versionInfoPath = os.path.join(projectRootBinPath, "python")
        
        # import getProjectInfo function
        sys.path.insert(0, versionInfoPath)
        from ProjectVersion import getProjectInfo
        sys.path.remove(versionInfoPath)
        
        # retrieve project configuration
        projectConfig = getProjectInfo()
        projectConfig["IsUserProject"] = True
        projectConfig["ProjectRootDir"] = projectRootDir
        if projectConfig["IPSDKInstallPath"] == "":
            projectConfig["IPSDKInstallPath"] = projectRootDir

        return projectConfig
        
    else: # case of non user project
    
        projectConfig = getIPSDKInfo()
        projectConfig["IsUserProject"] = False
        projectConfig["ProjectRootDir"] = ipsdkRootDir
        projectConfig["Name"] = "IPSDK"
        projectConfig["IPSDKInstallPath"] = ipsdkRootDir

        return projectConfig

# ------------------------------------------------------------------------------
def isUserProject():
    "Check whether project is part of an user project"
    
    return getProjectConfig()["IsUserProject"]
    
# ------------------------------------------------------------------------------
def getIPSDKRootDir():
    "retrieve ipsdk root directory"
    
    return ipsdkRootDir
    
# ------------------------------------------------------------------------------
def getProjectRootDir():
    "retrieve project root directory"
    
    return getProjectConfig()["ProjectRootDir"]
    
# ------------------------------------------------------------------------------
def getIPSDKPythonRootDir():
    "retrieve ipsdk python root directory"
    
    return os.path.join(getIPSDKRootDir(), "python")
    
# ------------------------------------------------------------------------------
def getProjectPythonRootDir():
    "retrieve project python root directory"
    
    return os.path.join(getProjectRootDir(), "python")
    
# ------------------------------------------------------------------------------
def getPyIPSDKDir():
    "retrieve PyIPSDK directory"
    
    return os.path.join(getIPSDKPythonRootDir(), "PyIPSDK")
    
# ------------------------------------------------------------------------------
def getProjectName():
    "retrieve project name"
    
    return getProjectConfig()["Name"]
    
# ------------------------------------------------------------------------------
def isIPSDKDebugVersion():
    "retrieve flag indicating wether we are using ipsdk debug version"
    
    if os.getenv("IPSDK_DEBUG_MODE", "False") == "True":
        return True
    else:
        return False

# ------------------------------------------------------------------------------
def getIPSDKVersionStr():
    "retrieve ipsdk version string"
    
    return getIPSDKInfo()["Version"]
    
# ------------------------------------------------------------------------------
def getProjectVersionStr():
    "retrieve project version string"
    
    return getProjectConfig()["Version"]
    
# ------------------------------------------------------------------------------
def getIPSDKVersionStatus():
    "retrieve ipsdk version status"
    
    return getIPSDKInfo()["VersionStatus"]
    
# ------------------------------------------------------------------------------
def getProjectVersionStatus():
    "retrieve project version status"
    
    return getProjectConfig()["VersionStatus"]
    
# ------------------------------------------------------------------------------
def getIPSDKRevision():
    "retrieve ipsdk revision"
    
    return getIPSDKInfo()["Revision"]
    
# ------------------------------------------------------------------------------
def getProjectRevision():
    "retrieve project revision"
    
    return getProjectConfig()["Revision"]
    
# ------------------------------------------------------------------------------
def getIPSDKFullVersionStr():
    "retrieve full ipsdk version string"
    
    fullIPSDKVersion = getIPSDKVersionStr()
    fullIPSDKVersion += " ["
    fullIPSDKVersion += getIPSDKVersionStatus()
    fullIPSDKVersion += " revision "
    fullIPSDKVersion += getIPSDKRevision()
    fullIPSDKVersion += "]"
    
    return fullIPSDKVersion
    
# ------------------------------------------------------------------------------
def getProjectFullVersionStr():
    "retrieve full project version string"
    
    fullProjectVersion = getProjectVersionStr()
    fullProjectVersion += " ["
    fullProjectVersion += getProjectVersionStatus()
    if len(getProjectRevision()) != 0 and getProjectRevision() != "Unversioned" :
        fullProjectVersion += " revision "
        fullProjectVersion += getProjectRevision()
    fullProjectVersion += "]"
    
    return fullProjectVersion
    
# ------------------------------------------------------------------------------
def getPlatformName():
    "retrieve platform name"
    
    return getIPSDKInfo()["PlatformName"]
    
# ------------------------------------------------------------------------------
def getIPSDKInstallPath():
    "retrieve IPSDK installation path"
    
    return getProjectConfig()["IPSDKInstallPath"]
    
# ------------------------------------------------------------------------------
def getIPSDKBinDebugSubDir():
    "retrieve ipsdk binary debug sub directory"
    
    return "Debug_" + getPlatformName()
    
# ------------------------------------------------------------------------------
def getIPSDKBinReleaseSubDir():
    "retrieve ipsdk binary release sub directory"
    
    return "Release_" + getPlatformName()
    
# ------------------------------------------------------------------------------
def getIPSDKBinSubDir():
    "retrieve ipsdk used binary sub directory"
    
    # check for debug version
    if isIPSDKDebugVersion() == True:
        return os.path.join(ipsdkRootBinPath, getIPSDKBinDebugSubDir())
    else:
        return os.path.join(ipsdkRootBinPath, getIPSDKBinReleaseSubDir())
        
# ------------------------------------------------------------------------------
def getProjectBinSubDir():
    "retrieve project used binary sub directory"
    
    # retrieve binary root path
    projectRootBinPath = getProjectRootDir()
    if os.path.exists(projectRootBinPath + "/Output") == True:
        projectRootBinPath += "/Output"
    else:
        projectRootBinPath += "/bin"
    
    # check for debug version
    if isIPSDKDebugVersion() == True:
        return os.path.join(projectRootBinPath, getIPSDKBinDebugSubDir())
    else:
        return os.path.join(projectRootBinPath, getIPSDKBinReleaseSubDir())
        
# ------------------------------------------------------------------------------
def getIPSDKBinDir():
    "retrieve ipsdk used binary directory"
    
    return os.path.join(getIPSDKRootDir(), getIPSDKBinSubDir())
    
# ------------------------------------------------------------------------------
def getProjectBinDir():
    "retrieve project used binary directory"
    
    return os.path.join(getProjectRootDir(), getProjectBinSubDir())
    
# ------------------------------------------------------------------------------
def getIPSDKLibName(baseLibName):
    "retrieve full ipsdk library name associated to a base library name"
    
    # retrieve ipsdk library suffix
    if isIPSDKDebugVersion() == True:
        ipsdkBinarySuffix = getIPSDKInfo()["BinarySuffixDebug"]
    else:
        ipsdkBinarySuffix = getIPSDKInfo()["BinarySuffixRelease"]
    
    # retrieve library name
    fullLibName = baseLibName + ipsdkBinarySuffix
    
    return fullLibName
    
# ------------------------------------------------------------------------------
def getProjectLibName(baseLibName):
    "retrieve full project library name associated to a base library name"
    
    # retrieve project library suffix
    if isIPSDKDebugVersion() == True:
        ipsdkBinarySuffix = getProjectConfig()["BinarySuffixDebug"]
    else:
        ipsdkBinarySuffix = getProjectConfig()["BinarySuffixRelease"]
    
    # retrieve library name
    fullLibName = baseLibName + ipsdkBinarySuffix
    
    return fullLibName
    
# ------------------------------------------------------------------------------
def getAvailablePythonVersions():
    "retrieve available python versions for ipsdk library"
    
    return getProjectConfig()["AvailablePythonVersions"]
    
# ------------------------------------------------------------------------------
def getIPSDKUnitTestRootDir():
    "retrieve project python unit tests root directory"
    
    return os.path.join(getIPSDKPythonRootDir(), "UnitTest")

# ------------------------------------------------------------------------------
def getIPSDKUnitTestToolsDir():
    "retrieve project python unit tests tools directory"
    
    return os.path.join(getIPSDKUnitTestRootDir(), "PyIPSDKUnitTestTools")
    
# ------------------------------------------------------------------------------
def getProjectUnitTestRootDir():
    "retrieve project python unit tests root directory"
    
    return os.path.join(getProjectPythonRootDir(), "UnitTest")
    
# ------------------------------------------------------------------------------
def getProjectUnitTestModuleDir(moduleName):
    "retrieve project python unit tests module directory"
    
    return os.path.join(getProjectUnitTestRootDir(), "UT_" + moduleName)
    
    

# ------------------------------------------------------------------------------
def loadIPSDKDll():
    """
    Load IPSDK DLLs, that can not be found by simply adding 
    the directory to the PATH environment variable since python 3.8
    This is necessary only for windows
    In case of python 3.7 or Linux, the function does nothing
    """

    # Retrieve the python version
    pythonVersionSplit = platform.python_version().split('.')
    
    if os.name != "nt" or (pythonVersionSplit[0] == '3' and pythonVersionSplit[1] == '7'):
        return []

    # Retrieve the binary path
    IPSDKConfig = getProjectConfig()
    
    # Find the path to the dlls to load
    binPath = getProjectBinSubDir()
    dllList = [file for file in os.listdir(binPath) if file.endswith(".dll")]
    
    # Load each dll
    loadedDllList = []
    for curFile in dllList:
        try:
            curDll = os.add_dll_directory(binPath)
            loadedDllList.append(curDll)
        except Exception as e:
            print("[ERROR] Could not load the library", curDll, ":", e)
    return loadedDllList
    
# ------------------------------------------------------------------------------
def unloadIPSDKDll(loadedDllList):
    """
    Unload IPSDK DLLs loaded by loadIPSDKDll() and contained in loadedDllList
    """
    for curDll in loadedDllList:
        curDll.close()
    
# ------------------------------------------------------------------------------
def importPyIPSDK():
    "function allowing to import and initialize PyIPSDK Module"
    
    # retrieve current python version
    pythonMajorVersion = sys.version_info.major
    pythonMinorVersion = sys.version_info.minor
    pythonVersionSubDir = "Python" + str(pythonMajorVersion) + "_" + str(pythonMinorVersion)
    
    # add ipsdk python wrapping to path
    ipsdkPythonDir = getIPSDKPythonRootDir()
    if not ipsdkPythonDir in sys.path:
        sys.path.insert(1, ipsdkPythonDir)
    projectBinaryDir = getProjectBinDir()
    if not projectBinaryDir in sys.path:
        sys.path.insert(1, projectBinaryDir)
    if os.name == "nt":
        os.environ["PATH"] = os.environ["PATH"] + ";"  + projectBinaryDir
    projectPythonBinaryDir = os.path.join(projectBinaryDir, pythonVersionSubDir)
    if not projectPythonBinaryDir in sys.path:
        sys.path.insert(1, projectPythonBinaryDir)

    # import base elements from IPSDK library
    pyIPSDKPath = getPyIPSDKDir()
    sys.path.insert(0, pyIPSDKPath)
    try:
        ipsdkBaseModule = __import__("PyIPSDKBase")
    except Exception as e:
        print("Import PyIPSDKBase fails :", e)
    sys.path.remove(pyIPSDKPath)
    
    # check wether user asked for delay library initialization
    bDelayedInit = False
    if os.getenv("IPSDK_DELAYED_INIT", "False") == "True":
        bDelayedInit = True
        
    # check wether user asked for network capabilities inhibition
    bNetworkCapabilitiesInhibited = False
    if os.getenv("IPSDK_NETWORK_CAPABILITIES_INHIBITION", "False") == "True":
        bNetworkCapabilitiesInhibited = True
        
    # initialization of IPSDK core library if needed
    libraryInitializer = ipsdkBaseModule.getLibraryInitializerInstance()
    if libraryInitializer.isInit() == False and bDelayedInit == False:
        print("Initialization of IPSDK library version " + getIPSDKFullVersionStr())
        userConfiguration = ipsdkBaseModule.createLibraryUserConfiguration()
        userConfiguration.setProjectRootDirectory(getIPSDKRootDir())
        userConfiguration.setNetworkCapabilitiesInhibited(bNetworkCapabilitiesInhibited);
        #bRes = libraryInitializer.init(userConfiguration)
        bRes = ipsdkBaseModule.initLibraryInitializer(userConfiguration, libraryInitializer)
        if bRes.getResult() == ipsdkBaseModule.eLibInitStatus.eLIS_Failed:
            raise RuntimeError("Failed to initialize IPSDK libray\n" + bRes.getMsg())

        # Activate GPU support
        bActivateGpuSupport = False
        if bActivateGpuSupport:
            ipsdkBaseModule.activateGpuSupport()
    
    # check for existence of toolkit directory
    toolkitDir = os.path.join(ipsdkPythonDir, "toolkit")
    if os.path.exists(toolkitDir):
    
        # add toolkit directory for optional features
        sys.path.insert(0, toolkitDir)
    
        # try to enable autocompletion for spyder editor
        from spyder_utils import active_autocompletion, is_disabled_autocompletion
        if is_disabled_autocompletion() == False:
            active_autocompletion()
        
        # add support for Avizo if needed
        # Avizo is a trademark of VSG, Visualization Sciences Group.
        from avizoBridge import updateIPSDKForAvizo
        updateIPSDKForAvizo(ipsdkBaseModule)
        
        # add support for DragonFly if needed
        # DragonFly is a trademark of ORS, Object Research Systems.
        from dragonFlyBridge import updateIPSDKForDragonFly
        updateIPSDKForDragonFly(ipsdkBaseModule)

        # add support for Digisens if needed
        from digisensBridge import updateIPSDKForDigisens
        updateIPSDKForDigisens(ipsdkBaseModule)
    
        # restore configuration without installation directory
        sys.path.remove(toolkitDir)
    
    return ipsdkBaseModule
    
# ------------------------------------------------------------------------------
def startUnitTest(moduleName, bDebugMode):

    # import unit test monitor
    unitTestToolsPath = getIPSDKUnitTestToolsDir()
    sys.path.insert(0, unitTestToolsPath)
    from UnitTestCfg import UnitTestCfg
    from UnitTestMonitor import UnitTestMonitor
    sys.path.remove(unitTestToolsPath)
    
    # add ipsdk python root path to system path
    ipsdkPythonDir = getIPSDKPythonRootDir()
    if not ipsdkPythonDir in sys.path:
        sys.path.insert(1, ipsdkPythonDir)
    
    # define unit tests used configuration
    unitTestCfg = UnitTestCfg()
    unitTestCfg.moduleName = moduleName
    unitTestCfg.bDebugMode = bDebugMode
    unitTestCfg.projectRootDir = getProjectRootDir()
    unitTestCfg.unitTestResultDir = ""
    unitTestCfg.unitTestLogResourceDir = ""
    unitTestCfg.bDisplayReport = True
    
    # start tests
    unitTestMonitor = UnitTestMonitor(unitTestCfg)
    