import functions_Doc
import docGenUtils

import sys
import os
import glob
import shutil
import re

# -------------------------------------------------------------------------------------------------------------------- #
def updateImgLinksInCore(content):
    """
    Parse each line of the content and change the links to the images by adding the prefix IPSDKCore
    e.g. <img src="illustration.png" /> becomes <img src="IPSDKCore_illustration.png" />

    the content is nor directly modified but a new string is returned

    This function is implemented to avoid mistakes between Core and IPL resources. The
    resource files from IPSDKCore are renamed when they are copied to the Essential directory

    WARNING : this function must be called only for IPSDKCore content
    """
    outStr = ""
    for line in content.splitlines():
        idx = line.find("<img")
        if idx == -1:
            outStr += line + '\n'
        else:
            # Add the content before "<img" to the new line
            newLine = line[:idx]

            # Remove the content before "<img"
            line = line[idx:]

            # Find "src="
            idxSrc = line.find("src=")
            if idxSrc != -1:
                newLine += line[:idxSrc+5] + "IPSDKCore_" + line[idxSrc+5:]
            outStr += newLine + '\n'

    return outStr

# -------------------------------------------------------------------------------------------------------------------- #
def generateMainPage(rootDocDir, outDir, bDevDoc):
    """
    Generates the main page
    """

    seeFullDocStr = ""
    if bDevDoc:
        seeFullDocStr = '<span class="LinkToFullDoc"><a href="../IPSDKIPL/html/modules.html">See full documentation</a></span>'

    header = docGenUtils.generateHTMLHeader(rootDocDir, "IPSDK documentation")
    footer = docGenUtils.generateHTMLFooter()

    content = '<div class="header">\n'+\
              '    <div class="headertitle">\n'+\
              '        <div class="title">IPSDK documentation '+seeFullDocStr+'</div>\n'+\
              '    </div>\n'+\
              '</div>\n'+\
              '<div class="contents">\n'+\
              '     <div class="textblock">\n'+\
              '         <h1>Welcome</h1>\n'+\
              '         <p>This is the documentation of the <b>IPSDK</b> image processing library.</p>\n'+\
              '         <p>You can access to any category with the list below.</p>\n'+\
              '         <details open>\n'+\
              '             <summary class=level0>Interface functionalities</summary>\n'+\
              '             <p class=level1><a href="machinelearning.html">Machine Learning</a></p>\n'+\
              '             <p class=level1><a href="viewer3d.html">3D Viewer</a></p>\n'+\
              '         </details>\n'+\
              '         <details open>\n'+\
              '             <summary class=level0>Image processing</summary>\n'+\
              '             <p class=level1><a href="algorithms.html">Algorithms</a></p>\n'+\
              '             <p class=level1><a href="measures.html">Measures</a></p>\n'
    content += '             <p class=level1><a href="concepts.html">Concepts</a></p>\n'

    content += '         </details>\n'+\
              '     </div>\n'

    docGenUtils.writeHTML (os.path.join(outDir, "index.html"), header + content + footer)

# -------------------------------------------------------------------------------------------------------------------- #
def generateMachineLearningPages(rootDocDir, outDir, patterns, pathDict, bDevDoc):
    """
    Generates the page leading to the Machine learning sections : smart segmentation, smart classification, ...
    """

    seeFullDocStr = ""
    if bDevDoc:
        seeFullDocStr = '<span class="LinkToFullDoc"><a href="../IPSDKCore/html/modules.html">See full documentation</a></span>'

    # Generate the summary page
    header = docGenUtils.generateHTMLHeader(rootDocDir, "IPSDK documentation")
    footer = docGenUtils.generateHTMLFooter()

    content = '<div class="header">\n'+\
              '    <div class="headertitle">\n'+\
              '        <div class="title">IPSDK Interface documentation '+seeFullDocStr+'</div>\n'+\
              '    </div>\n'+\
              '</div>\n'+\
              '<div class="contents">\n'+\
              '     <div class="textblock">\n'+\
              '         <h1>Machine learning</h1>\n'+\
              '         <p>This is the documentation of the <b>IPSDK</b> machine Learning functionalities.</p>\n'+\
              '         <p>The modules available in IPSDK are:</p>\n'+\
              '         <ul>\n'+\
              '             <li>\n'+\
              '                 <p><a href="smartsegmentation.html">Smart Segmentation</a><br/>\n'+\
              '                    Assign a class to each pixel</p>\n'+\
              '             </li>\n'+\
              '             <li>\n'+\
              '                 <p><a href="smartclassification.html">Smart Classification</a><br/>\n'+\
              '                    Assign a class to each object</p>\n'+\
              '             </li>\n'+\
              '             <li>\n'+\
              '                 <p><a href="superpixel_smartsegmentation.html">Super-Pixel Smart Segmentation</a><br/>\n'+\
              '                    Assign a class to each super-pixel</p>\n'+\
              '             </li>\n'+\
              '         </ul>\n'+\
              '     </div>\n'

    docGenUtils.writeHTML (os.path.join(outDir, "machinelearning.html"), header + content + footer)

    # Now generate the simplified documentations for each module according to the input pattern dictionary
    simplifyPage(rootDocDir, pathDict[patterns[0]], outDir + "/smartsegmentation.html", "Smart Segmentation", "Interface", bDevDoc, True)
    simplifyPage(rootDocDir, pathDict[patterns[1]], outDir + "/smartclassification.html", "Smart Classification", "Interface", bDevDoc, True)
    simplifyPage(rootDocDir, pathDict[patterns[2]], outDir + "/superpixel_smartsegmentation.html", "Super-Pixel Smart Segmentation", "Interface", bDevDoc, True)


# -------------------------------------------------------------------------------------------------------------------- #
def generateViewerPages(rootDocDir, outDir, patterns, pathDict, bDevDoc):
    """
    Generates the page leading to the Viewer sections : Viewer 3D, Movie Maker, ...
    """

    seeFullDocStr = ""
    if bDevDoc:
        seeFullDocStr = '<span class="LinkToFullDoc"><a href="../IPSDKCore/html/modules.html">See full documentation</a></span>'

    # Generate the summary page
    header = docGenUtils.generateHTMLHeader(rootDocDir, "IPSDK documentation")
    footer = docGenUtils.generateHTMLFooter()

    content = '<div class="header">\n'+\
              '    <div class="headertitle">\n'+\
              '        <div class="title">IPSDK Interface documentation '+seeFullDocStr+'</div>\n'+\
              '    </div>\n'+\
              '</div>\n'+\
              '<div class="contents">\n'+\
              '     <div class="textblock">\n'+\
              '         <h1>Machine learning</h1>\n'+\
              '         <p>This is the documentation of the <b>IPSDK</b> viewer functionalities.</p>\n'+\
              '         <p>The modules available in IPSDK are:</p>\n'+\
              '         <ul>\n'+\
              '             <li>\n'+\
              '                 <p><a href="viewerdesc.html">Viewer interface</a><br/>\n'+\
              '                    This section describes the viewer interface</p>\n'+\
              '             </li>\n'+\
              '             <li>\n'+\
              '                 <p><a href="moviemaker.html">Movie maker functionalities</a><br/>\n'+\
              '                    This sections explains how to use the movie maker</p>\n'+\
              '             </li>\n'+\
              '         </ul>\n'+\
              '     </div>\n'

    docGenUtils.writeHTML (os.path.join(outDir, "viewer3d.html"), header + content + footer)

    # Now generate the simplified documentations for each module according to the input pattern dictionary
    simplifyPage(rootDocDir, pathDict[patterns[3]], outDir + "/viewerdesc.html", "3D Viewer", "Interface", bDevDoc, True)
    simplifyPage(rootDocDir, pathDict[patterns[4]], outDir + "/moviemaker.html", "Movie maker", "Interface", bDevDoc, True)

# -------------------------------------------------------------------------------------------------------------------- #
def simplifyPage(rootDocDir, inPath, outPath, title, headerTitleStr, bDevDoc, bForceCore = False):
    """
    Generates a new page without the header content
    """

    if inPath == "":
        print("No source page found to simplify " + title)
        return

    baseName_inPath = os.path.basename(inPath)
    subDir = "IPSDKIPL"
    if bForceCore or "Core_" in outPath:
        subDir = "IPSDKCore"
    relativeInPath = "../"+subDir+"/html/"+baseName_inPath

    header = docGenUtils.generateHTMLHeader(rootDocDir, title)
    footer = docGenUtils.generateHTMLFooter()
    content = docGenUtils.removeHeaderContent(inPath)
    content = docGenUtils.removeRelativeLinks(content)
    content = docGenUtils.removeGroupHeaders(content)

    seeFullDocStr = ""
    if bDevDoc:
        seeFullDocStr = '<span class="LinkToFullDoc"><a href="'+relativeInPath+'">See full documentation</a></span>'

    content = '<div class="header">\n' + \
              '    <div class="headertitle">' + \
              '        <div class="title">IPSDK ' + headerTitleStr + ' documentation '+seeFullDocStr+'</div>\n' + \
              '    </div>\n' + \
              '</div>\n' +\
              '<div class="contents">' + content

    if subDir == "IPSDKCore":
        content = updateImgLinksInCore(content)

    docGenUtils.writeHTML(outPath, header + content + footer)

# -------------------------------------------------------------------------------------------------------------------- #
def generateAlgosPage(rootDocDir, ddict, outDir, bDevDoc):
    """
    Generates the page containing the list of algorithm
    """

    seeFullDocStr = ""
    if bDevDoc:
        seeFullDocStr = '<span class="LinkToFullDoc"><a href="../IPSDKIPL/html/modules.html">See full documentation</a></span>'

    header = docGenUtils.generateHTMLHeader(rootDocDir, "IPSDK algorithms")
    footer = docGenUtils.generateHTMLFooter()

    # Parse all the modules from the input dictionary
    content = '<div class="header">\n'+\
              '    <div class="headertitle">'+\
              '        <div class="title">IPSDK algorithm documentation '+seeFullDocStr+'</div>\n'+\
              '    </div>\n'+\
              '</div>\n'+ \
              '<div class="contents">\n' + \
              '     <div class="textblock">\n' + \
              '         <h1>IPSDK Algorithm modules</h1>\n' + \
              '         <p>This page lists all the image processing algorithms available in <b>IPSDK</b>.</p>\n'

    modulesDict = ddict["AllModules"]["content"]
    for keyModule in modulesDict:
        curModuleDict = modulesDict[keyModule]

        content += '         <details close>\n' + \
                   '             <summary class=level0>'+curModuleDict["title"]+' algorithms</summary>\n'

        for keyAlgo in curModuleDict["content"]:
            curAlgoDict = curModuleDict["content"][keyAlgo]

            content += '             <p class=level1><a href="'+curAlgoDict["doc"]+'">'+curAlgoDict["title"]+'</a></p>\n'
        content += '         </details>\n'
    content += '    </div>\n'

    docGenUtils.writeHTML (os.path.join(outDir, "algorithms.html"), header + content + footer)

# -------------------------------------------------------------------------------------------------------------------- #
def generateMeasuresCategoryContent(ddict, tabPrefix="", level=0):
    """
    Generates the content for a measure category
    """
    content = ""

    # First end recursion test : ddict does have "content"
    bRecursionCall = "content" in ddict
    if bRecursionCall:
        content += tabPrefix + '<details close>\n' + \
                   tabPrefix + '    <summary class=level'+str(level)+'>' + ddict["title"] + ' measures</summary>\n'

        # Possible recursion on the content
        # The recursion only appears if the content does not have "doc" and "title" keys
        # because the keys will only be the name of sub-categories
        ddictContent = ddict["content"]
        # Second recursion test
        if "doc" in ddictContent:
            content += tabPrefix+tabPrefix+\
                       '<p class=level'+str(level+1)+'><a href="'+ddictContent["doc"]+'">'+ddictContent["title"]+'</a></p>\n'
        else:
            # Recursion
            for contentKey in ddictContent:
                content += generateMeasuresCategoryContent(ddictContent[contentKey], tabPrefix + "    ", level+1)
        content += tabPrefix + '</details>\n'
    else:
        content += tabPrefix+\
                   '<p class=level'+str(level)+'><a href="'+ddict["doc"]+'">'+ddict["title"]+'</a></p>\n'

    return content

# -------------------------------------------------------------------------------------------------------------------- #
def generateMeasuresPage(rootDocDir, ddict, outDir, bDevDoc):
    """
    Generates the page containing the list of measures
    """

    seeFullDocStr = ""
    if bDevDoc:
        seeFullDocStr = '<span class="LinkToFullDoc"><a href="../IPSDKIPL/html/modules.html">See full documentation</a></span>'

    header = docGenUtils.generateHTMLHeader(rootDocDir, "IPSDK measures")
    footer = docGenUtils.generateHTMLFooter()

    # Parse all the modules from the input dictionary
    content = '<div class="header">\n'+\
              '    <div class="headertitle">'+\
              '        <div class="title">IPSDK documentation '+seeFullDocStr+'</div>\n'+\
              '    </div>\n'+\
              '</div>\n'+ \
              '<div class="contents">\n' + \
              '     <div class="textblock">\n' + \
              '         <h1>IPSDK Measure modules</h1>\n' + \
              '         <p>This page lists all the image processing measures available in <b>IPSDK</b>.</p>\n'

    modulesDict = ddict["AllMeasures"]["content"]
    for keyModule in modulesDict:
        content += generateMeasuresCategoryContent(modulesDict[keyModule], "            ")

    content += '    </div>\n'

    docGenUtils.writeHTML (os.path.join(outDir, "measures.html"), header + content + footer)

# -------------------------------------------------------------------------------------------------------------------- #
def generateConceptMainPage(rootDocDir, outDir, patterns, pathDict, bDevDoc):
    """
    Generates the page containing the list of concepts
    """
    header = docGenUtils.generateHTMLHeader(rootDocDir, "Concepts")
    footer = docGenUtils.generateHTMLFooter()

    # Build the list of titles
    vTitles = ["Geometrical conventions", "Data item", "Image geometry", "Basic image manipulation",
               "Manipulating sub-parts of images", "Numpy array brigde", "Tiff image files", "Raw image files",
               "Image formula", "Measure formula", "Neighborhood", "Kernels", "Morphological structuring elements",
               "Border policy", "Measure description"]

    vNewUrls = []
    for i in range(15):
        pageBaseName = os.path.basename(pathDict[patterns[i + 5]])
        vNewUrls.append("IPSDKCore_"+pageBaseName)

    seeFullDocStr = ""
    if bDevDoc:
        seeFullDocStr = '<span class="LinkToFullDoc"><a href="../IPSDKCore/html/modules.html">See full documentation</a></span>'

    # Parse all the modules from the input dictionary
    content = '<div class="header">\n'+\
              '    <div class="headertitle">'+\
              '        <div class="title">IPSDK Concepts '+seeFullDocStr+'</div>\n'+\
              '    </div>\n'+\
              '</div>\n'+ \
              '<div class="contents">\n' + \
              '     <div class="textblock">\n' + \
              '         <h1>IPSDK Concepts</h1>\n' + \
              '         <p>This page lists all the concepts used in <b>IPSDK</b>.</p>\n'+\
              '         <p class=level0><a href="'+vNewUrls[0]+'">Geometrical conventions</a></p>\n' + \
              '         <details open>\n'+\
              '             <summary class=level0>Basic Data</summary>\n'+\
              '             <p class=level1><a href="'+vNewUrls[1]+'">Data Item</a></p>\n'+ \
              '             <details open>\n' + \
              '                 <summary class=level1>Image geometry</summary>\n' + \
              '                 <p class=level2><a href="'+vNewUrls[2]+'">Image geometry</a></p>\n' + \
              '                 <p class=level2><a href="'+vNewUrls[3]+'">Basic image manipulation</a></p>\n' + \
              '                 <p class=level2><a href="'+vNewUrls[4]+'">Manipulating sub-parts of images</a></p>\n' + \
              '                 <p class=level2><a href="'+vNewUrls[5]+'">Numpy array brigde</a></p>\n' + \
              '                 <details open>\n' + \
              '                     <summary class=level2>Read and write images</summary>\n' + \
              '                     <p class=level3><a href="'+vNewUrls[6]+'">Read and write TIFF image files</a></p>\n' + \
              '                     <p class=level3><a href="'+vNewUrls[7]+'">Read and write RAW image files</a></p>\n' + \
              '                 </details>\n' + \
              '             </details>\n' + \
              '         </details>\n'+\
              '         <details open>\n' + \
              '             <summary class=level0>Image processing</summary>\n' + \
              '             <p class=level1><a href="'+vNewUrls[8]+'">Image formula</a></p>\n' + \
              '             <p class=level1><a href="'+vNewUrls[9]+'">Measure formula</a></p>\n' + \
              '             <p class=level1><a href="'+vNewUrls[10]+'">Neighborhood</a></p>\n' + \
              '             <p class=level1><a href="'+vNewUrls[11]+'">Kernels</a></p>\n' + \
              '             <p class=level1><a href="'+vNewUrls[12]+'">Morphological structuring elements</a></p>\n' + \
              '             <p class=level1><a href="'+vNewUrls[13]+'">Border policy</a></p>\n' + \
              '             <p class=level1><a href="'+vNewUrls[14]+'">Measure description</a></p>\n' + \
              '         </details>\n' + \
              '    </div>\n'

    docGenUtils.writeHTML (os.path.join(outDir, "concepts.html"), header + content + footer)

    # Simplify the existing pages
    for i in range(15):
        simplifyPage(rootDocDir, pathDict[patterns[i+5]], outDir + "/" + vNewUrls[i], vTitles[i], "Concepts", bDevDoc)

# -------------------------------------------------------------------------------------------------------------------- #
def simplify(rootSrcDir, rootDocDir, outDir, bDevDoc = True):
    """
    Main call to generate the simplified documentation
    """

    os.makedirs(outDir, exist_ok=True)

    docGenUtils.copyFiles("gif", rootDocDir+"/IPSDKCore/html", outDir)
    docGenUtils.copyFiles("png", rootDocDir+"/IPSDKCore/html", outDir)
    docGenUtils.copyFiles("png", rootDocDir+"/IPSDKIPL/html", outDir)
    docGenUtils.copyFiles("css", rootDocDir+"/IPSDKIPL/html", outDir)
    docGenUtils.copyFile(rootSrcDir+"/data/res/Common/images/CompanyLogo.png", outDir)
    # docGenUtils.copyFile(rootSrcDir+"/doc/Images/Main/web-link.png", outDir)

    # ddict = docGenUtils.generateFakeDict()
    # ddict = functions_Doc.createAllDoc(rootDocDir)

    generateMainPage(rootDocDir, outDir, bDevDoc)

    patterns = ['id="IPSDKPythonSmartSegmentation_Module_Grp"',
                'id="IPSDKPythonSmartClassification_Module_Grp"',
                'id="IPSDKPythonSmartSegmentationSuperPixels_Module_Grp"',
                'id="IPSDKPythonViewer3d_Module_Grp"',
                'id="IPSDKPythonMovieMaker_Module_Grp"',
                '<div class="title">Geometrical conventions<',
                '<div class="title">Data item<',
                '<div class="title">Image geometry concepts in IPSDK<',
                '<div class="title">Basic image manipulation<',
                '<div class="title">Manipulating sub-parts of images<',
                '<div class="title">Numpy array brigde<',
                '<div class="title">Tiff image files<',
                '<div class="title">Raw image files<',
                '<div class="title">Image Formula<',
                '<div class="title">Measure Formula<',
                '<div class="title">Neighborhood<',
                '<div class="title">Kernels<',
                '<div class="title">Morphological structuring elements<',
                '<div class="title">Border policy<',
                '<div class="title">Measure description<']
    pathDict = docGenUtils.findPages(rootDocDir+"/IPSDKCore/html/", patterns)
    # print(pathDict)
    # print(os.path.basename(pathDict[patterns[0]]))

    generateMachineLearningPages(rootDocDir, outDir, patterns, pathDict, bDevDoc)
    generateViewerPages(rootDocDir, outDir, patterns, pathDict, bDevDoc)

    generateConceptMainPage(rootDocDir, outDir, patterns, pathDict, bDevDoc)

    ddict = functions_Doc.createAllDoc(rootDocDir,True)

    generateAlgosPage(rootDocDir, ddict, outDir, bDevDoc)
    generateMeasuresPage(rootDocDir, ddict, outDir, bDevDoc)
    
if __name__ == "__main__":
    # simplify("D:/dev/IPSDK_dev/", "D:/dev/DocSimplification/doc4100/public/", "D:/dev/DocSimplification/doc4100/public/Essential", True)
    # simplify("D:/dev/IPSDK_dev/", "D:/dev/IPSDK_dev/Output/doc/public/", "D:/dev/IPSDK_dev/Output/doc/public/Essential", True)
    simplify("D:/dev/IPSDK_dev/", "D:/IPSDKExplorer_4_0_0_4/doc - Copie/public/", "D:/IPSDKExplorer_4_0_0_4/doc - Copie/public/Essential", True)
    # simplify("D:/dev/IPSDK_dev/", "D:/IPSDKExplorer_4_0_0_4/doc/public/", "D:/IPSDKExplorer_4_0_0_4/doc/public/Essential", True)
    # simplify("D:/dev/IPSDK_dev/src/IPSDKIPL", "D:/dev/IPSDK_dev/IPSDK_Nightly_Tmp/Output/doc/public/", "D:/dev/IPSDK_dev/IPSDK_Nightly_Tmp/Output/doc/public/Essential", True)
    # simplify("C:/dev/IPSDK/dev/IPSDK_dev/src/IPSDKIPL", "C:/Program Files/ReactivIP/IPSDK_ToolKit_3_2_2_1/doc/public/", "C:/dev/IPSDK/dev/IPSDK_dev/Output/doc/public/Essential")
