import sys
import os

from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore
from PyQt5.QtCore import pyqtSignal, Qt, QCoreApplication

import PyIPSDK
import PyIPSDK.IPSDKIPLUtility as util
import PyIPSDK.IPSDKIPLIntensityTransform as itrans
import PyIPSDK.IPSDKIPLShapeAnalysis as shapeanalysis
import PyIPSDK.IPSDKIPLColor as colorIP
import PyIPSDK.IPSDKIPLAdvancedMorphology as advmorpho
import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr

import WidgetTypes
import UsefullVariables as vrb
import UsefullWidgets as wgt
import UsefullTexts as txt
import UsefullFunctions as fct
import DatabaseFunction as Dfct
import DrawFunctions as DWfct

import xml.etree.ElementTree as xmlet
import XMLtoInfoSet as xml2IS

# from sklearn.externals import joblib
import joblib

import numpy as np
import time
import traceback

from WidgetLabelImage import WidgetLabelImage
from ImageViewer import WidgetImage
from ShapeAnalysisWidget import MeasureInfoSetWidget
from ShapeAnalysisDisplayer import measureIterator,measureIteratorByType

from moduleRandomForest.FeaturesSuperPixelsWidget import FeaturesSuperPixelsWidget

from GroupMenu import GroupBoxContrast
from WidgetContrast import WidgetContrast

import random
# random.seed(1)
import webbrowser

from sklearn.ensemble import RandomForestClassifier

from LutsWidget import lutJet

import PyIPSDK.IPSDKUI as ui

class MainWidget(qt.QWidget):

    def __init__(self):
        qt.QWidget.__init__(self)

        self.moduleType = 'Super Pixel Classification'
        self.graphicElements = None
        self.currentLabel = None
        self.currentXmlElement = None

        self.modelDimension = "2D"

        self.model = None
        self.modelChanged = False

        self.posX = None
        self.posY = None
        self.posZ = None

        self.currentMode = "Pen"
        self.liveUpdate = True

        self.groupBoxProcessing = wgt.GroupBoxProcessing()
        self.groupBoxProcessing.buttonStop.setFixedWidth(0)

        self.xmlElement = xmlet.Element('IPSDK')
        allElements = Dfct.SubElement(self.xmlElement, 'AllElements')
        featuresElement = Dfct.SubElement(self.xmlElement, 'Features')

        self.previousSuperPixelsElement = None

        self.featuresWidget = FeaturesSuperPixelsWidget(parent=self, xmlElement=featuresElement)

        self.widgetLabelImage = WidgetLabelImage(xmlElement=allElements, module=self)
        self.widgetImage = WidgetImage(standAlone=True, module=self, parent=self)
        self.imageViewer = self.widgetImage.imageViewer
        self.settingsWidget = SettingsWidget(parent=self)
        self.labelScrollArea = self.settingsWidget.labelSelectionGroupBox.labelScrollArea

        self.graphicElements = GraphicElements(parent=self)
        self.selectionRectangle = None
        self.currentPointSelected = None
        self.refPosX = None
        self.refPosY = None
        self.drawing = False

        self.groupContrast = GroupBoxContrast(parentWidget=self)
        self.groupContrast.buttonAuto.setToolTip(txt.dictToolTips["MinMaxContrast"])
        self.widgetContrast = WidgetContrast()
        self.widgetContrast.setValues(50, 50)

        self.settingsWidget.listWidgetFeatures.fillListWidget()

        self.layout = qt.QGridLayout()
        self.layout.setContentsMargins(5, 5, 5, 5)
        self.layout.setSizeConstraint(1)

        self.layout.addWidget(self.groupContrast, 0, 1)
        self.layout.addWidget(self.widgetLabelImage, 1, 0)
        self.layout.addWidget(self.widgetImage, 1, 1)
        self.layout.addWidget(self.groupBoxProcessing, 1, 1)
        self.layout.addWidget(self.settingsWidget, 0, 2,2,1)

        self.groupBoxProcessing.setVisible(False)

        self.setLayout(self.layout)

        self.widgetImage.SignalRefreshPreview.connect(self.computeModel)
        self.imageViewer.signalNewImage.connect(self.addLabelsFromUrls)
        self.imageViewer.signalEmitMouse.connect(self.applySignalImageViewer)

        self.widgetLabelImage.signalDoubleClickLabel.connect(self.changeCurrentXmlElement)
        self.widgetLabelImage.signalLabelDeletePosition.connect(self.labelDeleted)

        # self.widgetImage.lutInformations.initLut(lutJet["Array"],vertical=False)
        #
        self.labelScrollArea.SignalDeleteLabel.connect(self.colorLabelDeleted)
        self.labelScrollArea.SignalAddLabel.connect(self.changeLut)
        self.labelScrollArea.SignalColorChanged.connect(self.changeLut)

        self.settingsWidget.buttonSaveModel.clicked.connect(self.saveModel)

        self.settingsWidget.pushButtonValidateParamSuperPixels.clicked.connect(self.computeAllSuperPixels)

        self.groupContrast.buttonBrigthness.clicked.connect(self.showContrastWindow)
        self.widgetContrast.signalValueChanged.connect(self.widgetImage.actualizeScene)
        self.settingsWidget.drawGroupBox.buttonBack.clicked.connect(self.getPreviousSuperPixelsElement)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.resize(1200 * vrb.ratio, 1200 * vrb.ratio)
        self.setMinimumSize(600 * vrb.ratio, 600 * vrb.ratio)

    def resizeEvent(self, event):

        heightGroupContrast = 50 * vrb.ratio
        self.groupContrast.setFixedHeight(heightGroupContrast)
        self.widgetLabelImage.setFixedSize(200*vrb.ratio,self.height() - heightGroupContrast-25*vrb.ratio)
        self.widgetImage.setFixedSize(max(0,self.width()-470*vrb.ratio),self.height()-heightGroupContrast-25*vrb.ratio)
        self.settingsWidget.setFixedSize(250*vrb.ratio,self.height())

    def closeEvent(self, a0: QtGui.QCloseEvent):

        try:
            vrb.mainWindow.currentModuleMachineLearning = None
        except:
            pass

    def clearDraw(self):

        allSuperPixelsElement = Dfct.SubElement(self.currentXmlElement, "AllSuperPixels")
        Dfct.clearElement(allSuperPixelsElement)

        self.updateSuperPixelsElement()

        self.computeModel()

        self.actualizeNumberOfSuperPixels()


    def applySignalImageViewer(self,ddict, imageViewer):

        if self.currentMode in ["Pen","Eraser"]:
            self.updateSuperPixelsModel(ddict)
        if self.currentMode == "Selection":
            self.drawSelectionRectangle(ddict)

    def updateSuperPixelsModel(self,ddict):

        if self.currentXmlElement is not None:

            if self.currentMode == "Pen":
                pixmap = QtGui.QPixmap(vrb.folderImages + '/Pencil.png')
                pixmap = pixmap.scaled(40, 40, aspectRatioMode=Qt.KeepAspectRatio, transformMode=Qt.SmoothTransformation)
                self.imageViewer.viewport().setCursor(QtGui.QCursor(pixmap))
            elif self.currentMode == "Eraser":
                pixmap = QtGui.QPixmap(vrb.folderImages + '/Eraser_Cursor.png')
                pixmap = pixmap.scaled(40, 40, aspectRatioMode=Qt.KeepAspectRatio, transformMode=Qt.SmoothTransformation)
                self.imageViewer.viewport().setCursor(QtGui.QCursor(pixmap))

            if (ddict["Event"] == "Move" and self.imageViewer.mousePressed and self.imageViewer.currentButtonPressed == 1) or (ddict["Event"] == "MousePressed" and ddict["Button"] == 1):

                self.modelChanged = True

                if self.posX is None or self.posY is None:

                    if self.modelDimension == "2D":
                        refPosX = int(ddict["Position"][0])
                        refPosY = int(ddict["Position"][1])
                        if self.currentMode == "Pen":
                            self.addSuperPixel(refPosX, refPosY)
                        elif self.currentMode == "Eraser":
                            self.removeSuperPixel(refPosX, refPosY)
                    else:
                        if self.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonX.isChecked():
                            refPosX = self.widgetImage.imageViewerStandAlone.sliderAxis.sliderX.slider.value()
                            refPosY = int(ddict["Position"][0])
                            refPosZ = int(ddict["Position"][1])
                        elif self.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonY.isChecked():
                            refPosX = int(ddict["Position"][0])
                            refPosY = self.widgetImage.imageViewerStandAlone.sliderAxis.sliderY.slider.value()
                            refPosZ = int(ddict["Position"][1])
                        elif self.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonZ.isChecked():
                            refPosX = int(ddict["Position"][0])
                            refPosY = int(ddict["Position"][1])
                            refPosZ = self.widgetImage.imageViewerStandAlone.sliderAxis.sliderZ.slider.value()
                        if self.currentMode == "Pen":
                            self.addSuperPixel(refPosX, refPosY,posZ=refPosZ)
                        elif self.currentMode == "Eraser":
                            self.removeSuperPixel(refPosX, refPosY,posZ=refPosZ)

                else:

                    if self.modelDimension == "2D":
                        refPosX = int(ddict["Position"][0])
                        refPosY = int(ddict["Position"][1])
                        nbStep = max(round(abs(self.posX - refPosX)), round(abs(self.posY - refPosY)))
                        nbStep = max(nbStep, 1)
                        nbStep *= 5
                        if nbStep > 0:
                            for step in range(nbStep + 1):
                                posX = round(refPosX + step * (self.posX - refPosX) / nbStep)
                                posY = round(refPosY + step * (self.posY - refPosY) / nbStep)

                                if self.currentMode == "Pen":
                                    self.addSuperPixel(posX, posY)
                                elif self.currentMode == "Eraser":
                                    self.removeSuperPixel(posX, posY)
                    else:
                        if self.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonX.isChecked():
                            refPosX = self.widgetImage.imageViewerStandAlone.sliderAxis.sliderX.slider.value()
                            refPosY = int(ddict["Position"][0])
                            refPosZ = int(ddict["Position"][1])
                        elif self.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonY.isChecked():
                            refPosX = int(ddict["Position"][0])
                            refPosY = self.widgetImage.imageViewerStandAlone.sliderAxis.sliderY.slider.value()
                            refPosZ = int(ddict["Position"][1])
                        elif self.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonZ.isChecked():
                            refPosX = int(ddict["Position"][0])
                            refPosY = int(ddict["Position"][1])
                            refPosZ = self.widgetImage.imageViewerStandAlone.sliderAxis.sliderZ.slider.value()
                        nbStep = max(round(abs(self.posX - refPosX)), round(abs(self.posY - refPosY)),round(abs(self.posZ - refPosZ)))
                        nbStep = max(nbStep, 1)
                        nbStep *= 5
                        if nbStep > 0:
                            for step in range(nbStep + 1):
                                posX = round(refPosX + step * (self.posX - refPosX) / nbStep)
                                posY = round(refPosY + step * (self.posY - refPosY) / nbStep)
                                posZ = round(refPosZ + step * (self.posZ - refPosZ) / nbStep)

                                if self.currentMode == "Pen":
                                    self.addSuperPixel(posX, posY,posZ=posZ)
                                elif self.currentMode == "Eraser":
                                    self.removeSuperPixel(posX, posY,posZ=posZ)
                
                self.actualizeNumberOfSuperPixels()

                if ddict["Event"] == "Move":
                    self.imageViewer.getRoiImage(changeRoiImage=False)

                self.posX = refPosX
                self.posY = refPosY
                if self.modelDimension == "3D":
                    self.posZ = refPosZ

            if ddict["Event"] == "Release":

                self.posX = None
                self.posY = None

                if self.liveUpdate:
                    if self.modelChanged:
                        self.modelChanged = False
                        self.computeModel()

                self.updateSuperPixelsElement()

                #self.imageViewer.getRoiImage(changeRoiImage=False)

    def addSuperPixel(self,posX,posY,posZ = None):

        allSuperPixelsElement = Dfct.SubElement(self.currentXmlElement, "AllSuperPixels")
        currentNum = self.labelScrollArea.getCurrentNum()

        try:
            if posZ is None:
                labelValue = self.currentLabel.imageSuperPixels.array[posY,posX]
            else:
                labelValue = self.currentLabel.imageSuperPixels.array[posZ,posY,posX]

            newSuperPixel = True
            for child in allSuperPixelsElement:
                if Dfct.childText(child,"LabelValue") == str(labelValue) and newSuperPixel:
                    newSuperPixel = False
                    Dfct.SubElement(child,"X").text = str(posX)
                    Dfct.SubElement(child,"Y").text = str(posY)
                    if posZ is not None:
                        Dfct.SubElement(child,"Z").text = str(posZ)
                    Dfct.SubElement(child,"Class").text = str(currentNum)

            if newSuperPixel:
                superPixelElement = xmlet.SubElement(allSuperPixelsElement,"SuperPixel")
                Dfct.SubElement(superPixelElement, "LabelValue").text = str(labelValue)
                Dfct.SubElement(superPixelElement, "X").text = str(posX)
                Dfct.SubElement(superPixelElement, "Y").text = str(posY)
                if posZ is not None:
                    Dfct.SubElement(superPixelElement, "Z").text = str(posZ)
                Dfct.SubElement(superPixelElement, "Class").text = str(currentNum)
        except:
            traceback.print_exc(file=sys.stderr)

    def addModifiedSuperPixel(self,posX,posY,num,allSuperPixelsElement,imageSuperPixels,posZ = None):

        # allSuperPixelsElement = Dfct.SubElement(self.currentXmlElement, "AllSuperPixels")

        try:
            if posZ is None:
                labelValue = imageSuperPixels.array[posY,posX]
            else:
                labelValue = imageSuperPixels.array[posZ,posY,posX]

            newSuperPixel = True
            for child in allSuperPixelsElement:
                if Dfct.childText(child,"LabelValue") == str(labelValue) and newSuperPixel:
                    newSuperPixel = False
                    Dfct.SubElement(child,"X").text = str(posX)
                    Dfct.SubElement(child,"Y").text = str(posY)
                    if posZ is not None:
                        Dfct.SubElement(child, "Z").text = str(posZ)
                    Dfct.SubElement(child,"Class").text = str(num)

            if newSuperPixel:
                superPixelElement = xmlet.SubElement(allSuperPixelsElement,"SuperPixel")
                Dfct.SubElement(superPixelElement, "LabelValue").text = str(labelValue)
                Dfct.SubElement(superPixelElement, "X").text = str(posX)
                Dfct.SubElement(superPixelElement, "Y").text = str(posY)
                if posZ is None:
                    Dfct.SubElement(superPixelElement, "Z").text = str(posZ)
                Dfct.SubElement(superPixelElement, "Class").text = str(num)
        except:
            traceback.print_exc(file=sys.stderr)

    def removeSuperPixel(self,posX,posY,posZ = None):

        allSuperPixelsElement = Dfct.SubElement(self.currentXmlElement, "AllSuperPixels")
        try:
            if posZ is None:
                labelValue = self.currentLabel.imageSuperPixels.array[posY,posX]
            else:
                labelValue = self.currentLabel.imageSuperPixels.array[posZ,posY,posX]
            foundSuperPixel = False
            for child in allSuperPixelsElement:
                if Dfct.childText(child, "LabelValue") == str(labelValue) and foundSuperPixel==False:
                    Dfct.removeElement(allSuperPixelsElement,child)
                    foundSuperPixel = True
        except:
            pass

    def drawSelectionRectangle(self,ddict):

        self.imageViewer.viewport().setCursor(Qt.ArrowCursor)
        # currentNum = self.labelScrollArea.getCurrentNum()

        if ddict["Button"] == 1 and ddict["Event"] == "MousePressed":

            self.selectionRectangle = xmlet.Element('Selection')
            self.refPosX = ddict["Position"][0]
            self.refPosY = ddict["Position"][1]
            DWfct.addPointToPolygon(self.selectionRectangle, QtCore.QPointF(self.refPosX, self.refPosY),allowSamePoint = True)
            DWfct.addPointToPolygon(self.selectionRectangle, QtCore.QPointF(self.refPosX, self.refPosY),allowSamePoint = True)
            DWfct.addPointToPolygon(self.selectionRectangle, QtCore.QPointF(self.refPosX, self.refPosY),allowSamePoint = True)
            modif,point = DWfct.addPointToPolygon(self.selectionRectangle, QtCore.QPointF(self.refPosX, self.refPosY),returnPointElement=True,allowSamePoint = True)
            self.currentPointSelected = point

            self.drawing = True
            self.imageViewer.getRoiImage(changeRoiImage=False)

        if ddict["Button"] == 0 and ddict["Event"] == "Move":

            if self.drawing and self.selectionRectangle is not None:
                if self.currentPointSelected is not None:
                    # Moving this point
                    pointXElement = Dfct.SubElement(self.currentPointSelected, "PointX")
                    pointXElement.text = str(ddict["Position"][0])
                    pointYElement = Dfct.SubElement(self.currentPointSelected, "PointY")
                    pointYElement.text = str(ddict["Position"][1])

                    # In case of the selection, the polygon is a rectangle, so moving a point moves 2 other points
                    numPointSelected = int(Dfct.childText(self.currentPointSelected, 'Num'))
                    for point in Dfct.SubElement(self.selectionRectangle, 'AllPoints'):
                        if int(Dfct.childText(point, 'Num')) == (numPointSelected + 1) % 4:
                            if numPointSelected % 2 == 1:
                                pointXElement = Dfct.SubElement(point, "PointX")
                                pointXElement.text = str(ddict["Position"][0])
                            else:
                                pointYElement = Dfct.SubElement(point, "PointY")
                                pointYElement.text = str(ddict["Position"][1])
                        if int(Dfct.childText(point, 'Num')) == (numPointSelected - 1) % 4:
                            if numPointSelected % 2 == 0:
                                pointXElement = Dfct.SubElement(point, "PointX")
                                pointXElement.text = str(ddict["Position"][0])
                            else:
                                pointYElement = Dfct.SubElement(point, "PointY")
                                pointYElement.text = str(ddict["Position"][1])
                    self.imageViewer.getRoiImage(changeRoiImage=False)

        if ddict["Button"] == 1 and ddict["Event"] == "Release":

            minX, minY, sizeX, sizeY = DWfct.boundingBoxPolygon(self.selectionRectangle)

            labelFound = []

            for i in range(sizeX):
                for j in range(sizeY):
                    if self.modelDimension == "2D":
                        posX = minX + i
                        posY = minY + j
                        labelValue = self.currentLabel.imageSuperPixels.array[posY,posX]
                    else:
                        if self.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonX.isChecked():
                            posX = self.widgetImage.imageViewerStandAlone.sliderAxis.sliderX.slider.value()
                            posY = minX + i
                            posZ = minY + j
                        elif self.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonY.isChecked():
                            posX = minX + i
                            posY = self.widgetImage.imageViewerStandAlone.sliderAxis.sliderY.slider.value()
                            posZ = minY + j
                        elif self.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonZ.isChecked():
                            posX = minX + i
                            posY = minY + j
                            posZ = self.widgetImage.imageViewerStandAlone.sliderAxis.sliderZ.slider.value()
                        labelValue = self.currentLabel.imageSuperPixels.array[posZ,posY,posX]
                    if labelValue not in labelFound:
                        if self.modelDimension == "2D":
                            self.addSuperPixel(posX,posY)
                        else:
                            self.addSuperPixel(posX, posY,posZ=posZ)
                        labelFound.append(labelValue)

            self.currentPointSelected = None
            self.refPosX = None
            self.refPosY = None
            self.drawing = False
            self.selectionRectangle = None

            self.updateSuperPixelsElement()

            self.actualizeNumberOfSuperPixels()

            if self.liveUpdate:
                self.computeModel()

            self.imageViewer.getRoiImage(changeRoiImage=False)

    def actualizeNumberOfSuperPixels(self):

        nbLabels = self.settingsWidget.labelSelectionGroupBox.labelScrollArea.layout.count()

        for numLabel in range(nbLabels):

            labelItem = self.settingsWidget.labelSelectionGroupBox.labelScrollArea.layout.itemAt(numLabel)
            if labelItem is not None:
                labelWidget = labelItem.widget()
                nbPixels = 0
                for i in range(self.widgetLabelImage.layout.count()):
                    item = self.widgetLabelImage.layout.itemAt(i)
                    if item is not None:
                        label = item.widget()
                        allSuperPixelsElement = Dfct.SubElement(label.xmlElement, "AllSuperPixels")
                        for child in allSuperPixelsElement:
                            if Dfct.childText(child,"Class") == str(numLabel):
                                nbPixels+=1
                labelWidget.labelCount.setText(str(nbPixels))

    def changeLut(self):

        self.customLutDraw = self.labelScrollArea.getCustomLutDraw()
        self.customLutOverlay = self.labelScrollArea.getCustomLutOverlay()

        self.imageViewer.getRoiImage()

    def computeAllSuperPixels(self):

        for i in range(self.widgetLabelImage.layout.count()):
            item = self.widgetLabelImage.layout.itemAt(i)
            if item is not None:
                label = item.widget()
                self.computeSuperPixels(label)

                self.adjustSuperPixelsLabeling(label)

        self.updateSettingsSuperPixelsElement()

        self.changeCurrentXmlElement(self.currentLabel)

    def adjustSuperPixelsLabeling(self,label):

        allSuperPixelsElement = Dfct.SubElement(label.xmlElement,"AllSuperPixels")
        imageSuperPixels = label.imageSuperPixels

        elementCopy = Dfct.copyXmlElement(allSuperPixelsElement)
        Dfct.clearElement(allSuperPixelsElement)
        for child in elementCopy:
            num = int(Dfct.childText(child,"Class"))
            posX = int(Dfct.childText(child,"X"))
            posY = int(Dfct.childText(child,"Y"))
            if self.modelDimension == "2D":
                self.addModifiedSuperPixel(posX,posY,num,allSuperPixelsElement,imageSuperPixels)
            else:
                posZ = int(Dfct.childText(child, "Z"))
                self.addModifiedSuperPixel(posX,posY,num,allSuperPixelsElement,imageSuperPixels,posZ=posZ)


        self.updateSuperPixelsElement()

        self.actualizeNumberOfSuperPixels()
        
    def computeSuperPixels(self,label):

        pass

        try:
            size = int(self.settingsWidget.lineEditSize.text())
        except:
            size = 10
        try:
            compactness = float(self.settingsWidget.lineEditCompactness.text())
        except:
            compactness = 0.5

        nbIter = 3

        try:
            sizeRatio = int(self.settingsWidget.lineEditSizeRatio.text())
        except:
            sizeRatio = 0.5

        try:
            if label.image.getSizeZ() == 1:
                imageSuperPixels = advmorpho.superPixels2dImg(label.image,size,compactness,nbIter,sizeRatio,PyIPSDK.eSPT_Size)
            else:
                imageSuperPixels = advmorpho.superVoxels3dImg(label.image,size,compactness,nbIter,sizeRatio,PyIPSDK.eSPT_Size)
        except:
            traceback.print_exc(file=sys.stderr)
            imageSuperPixels = None

        label.imageSuperPixels = imageSuperPixels
        if imageSuperPixels is not None:
            if label.image.getSizeZ() == 1:
                label.nbSuperPixels = glbmsr.statsMsr2d(imageSuperPixels).max
            else:
                label.nbSuperPixels = glbmsr.statsMsr3d(imageSuperPixels).max
        else:
            label.nbSuperPixels = 0
        self.widgetImage.setImageSuperPixels(imageSuperPixels)

        self.computeFeatures(label)

        self.imageViewer.getRoiImage(changeRoiImage=False)

        self.settingsWidget.pushButtonValidateParamSuperPixels.setEnabled(False)

    def computeAllFeatures(self):

        try:
            for i in range(self.widgetLabelImage.layout.count()):
                item = self.widgetLabelImage.layout.itemAt(i)
                if item is not None:
                    label = item.widget()
                    self.computeFeatures(label)

            self.computeModel()
        except:
            pass

    def computeFeatures(self,label):

        if label.imageSuperPixels is not None:
            featuresElement = Dfct.SubElement(self.xmlElement,"Features")

            listFeatures = []
            if self.modelDimension == "2D":
                inMeasureInfoSet = PyIPSDK.createMeasureInfoSet2d()
            else:
                inMeasureInfoSet = PyIPSDK.createMeasureInfoSet3d()

            if Dfct.childText(featuresElement,"NbPixels") == "True":
                if self.modelDimension == "2D":
                    PyIPSDK.createMeasureInfo(inMeasureInfoSet, "NbPixels2dMsr")
                    listFeatures.append(["NbPixels2dMsr", "Nb Pixels"])
                else:
                    PyIPSDK.createMeasureInfo(inMeasureInfoSet, "NbPixels3dMsr")
                    listFeatures.append(["NbPixels3dMsr", "Nb Pixels"])
            if Dfct.childText(featuresElement,"SizeX") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "BoundingBoxSizeXMsr")
                listFeatures.append(["BoundingBoxSizeXMsr","Size X"])
            if Dfct.childText(featuresElement,"SizeY") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "BoundingBoxSizeYMsr")
                listFeatures.append(["BoundingBoxSizeYMsr","Size Y"])
            if self.modelDimension == "3D" and Dfct.childText(featuresElement, "SizeZ") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "BoundingBoxSizeZMsr")
                listFeatures.append(["BoundingBoxSizeZMsr", "Size Z"])
            if Dfct.childText(featuresElement,"Minimum") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "MinMsr")
                listFeatures.append(["MinMsr","Minimum"])
            if Dfct.childText(featuresElement,"Maximum") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "MaxMsr")
                listFeatures.append(["MaxMsr","Maximum"])
            if Dfct.childText(featuresElement,"Mean") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "MeanMsr")
                listFeatures.append(["MeanMsr","Mean"])
            if Dfct.childText(featuresElement,"Median") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "MedianMsr")
                listFeatures.append(["MedianMsr","Median"])
            if Dfct.childText(featuresElement,"Variance") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "VarianceMsr")
                listFeatures.append(["VarianceMsr","Variance"])
            if Dfct.childText(featuresElement,"Energy") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "EnergyMsr")
                listFeatures.append(["EnergyMsr","Energy"])
            if Dfct.childText(featuresElement,"Entropy") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "EntropyMsr")
                listFeatures.append(["EntropyMsr","Entropy"])
            if Dfct.childText(featuresElement,"Kurtosis") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "KurtosisMsr")
                listFeatures.append(["KurtosisMsr","Kurtosis"])
            if Dfct.childText(featuresElement,"Skewness") == "True":
                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "SkewnessMsr")
                listFeatures.append(["SkewnessMsr","Skewness"])
            if Dfct.childText(featuresElement,"Histogram") == "True":

                if Dfct.childText(featuresElement, "NbPixels") != "True":
                    if self.modelDimension == "2D":
                        PyIPSDK.createMeasureInfo(inMeasureInfoSet, "NbPixels2dMsr")
                    else:
                        PyIPSDK.createMeasureInfo(inMeasureInfoSet, "NbPixels3dMsr")

                minValue = float(Dfct.childText(featuresElement,"HistogramMin"))
                maxValue = float(Dfct.childText(featuresElement,"HistogramMax"))
                nbClassesValue = int(Dfct.childText(featuresElement,"HistogramNbClasses"))

                PyIPSDK.createMeasureInfo(inMeasureInfoSet, "HistogramMsr", shapeanalysis.createHistogramMsrParamsNbClasses(nbClassesValue,minValue,maxValue))
                listFeatures.append(["HistogramMsr","Histogram"])

            if self.modelDimension == "2D":
                analysis = shapeanalysis.labelAnalysis2d(label.image, label.imageSuperPixels, inMeasureInfoSet)
            else:
                analysis = shapeanalysis.labelAnalysis3d(label.image, label.imageSuperPixels, inMeasureInfoSet)

            nbColorPlan = label.image.getSizeC()

            outListFeatures = []
            features = []
            offset = 0

            for feature in listFeatures:
                if feature[0] in ["NbPixels2dMsr","NbPixels3dMsr","BoundingBoxSizeXMsr","BoundingBoxSizeYMsr","BoundingBoxSizeZMsr"]:
                    values = analysis.getMeasure(feature[0]).getMeasureResult().getColl(0)
                    features.append(values)
                    outListFeatures.append(feature[1])
                    offset+=1
                elif feature[0] != "HistogramMsr":
                    for c in range(nbColorPlan):
                        values = analysis.getMeasure(feature[0]).getMeasureResult().getColl(c)
                        features.append(values)
                        outListFeatures.append(feature[1])
                        offset+=1

                else:
                    nbClasses = len(PyIPSDK.toPyDict(analysis.getMeasure("HistogramMsr").getMeasureResult().getColl(0)[1])["Frequencies"])
                    if self.modelDimension == "2D":
                        valuesNbPixels = analysis.getMeasure("NbPixels2dMsr").getMeasureResult().getColl(0)
                    else:
                        valuesNbPixels = analysis.getMeasure("NbPixels3dMsr").getMeasureResult().getColl(0)

                    for _ in range(nbClasses * nbColorPlan):
                        features.append([0])
                        outListFeatures.append("Histogram")
                    for c in range(nbColorPlan):
                        valuesHistogram = analysis.getMeasure("HistogramMsr").getMeasureResult().getColl(c)
                        for numLabel in range(len(valuesHistogram)):
                            valueHisto = valuesHistogram[numLabel]
                            nbPixels = valuesNbPixels[numLabel]
                            if valueHisto is not None:
                                frequencies = PyIPSDK.toPyDict(valueHisto)["Frequencies"]
                                if len(frequencies)>0:
                                    for clss in range(nbClasses):
                                        features[offset + clss + c * nbClasses].append(frequencies[clss] / nbPixels)

            features = np.asarray(features)

            label.features = features
            label.listFeatures = outListFeatures

        self.settingsWidget.buttonSaveModel.setEnabled(True)

    def showContrastWindow(self):
        self.widgetContrast.window().setWindowModality(Qt.ApplicationModal)
        self.widgetContrast.show()
        self.widgetContrast.widgetSettings.applySliderValues()

    def updateLabelClasses(self):

        labelClassesElement = Dfct.SubElement(self.xmlElement,"LabelClasses")
        numberLabelElement = Dfct.SubElement(labelClassesElement,"NumberLabels")
        nbLabels = self.settingsWidget.labelSelectionGroupBox.labelScrollArea.layout.count()
        numberLabelElement.text = str(nbLabels)
        for numLabel in range(nbLabels):
            labelItem = self.settingsWidget.labelSelectionGroupBox.labelScrollArea.layout.itemAt(numLabel)
            if labelItem is not None:
                labelWidget = labelItem.widget()
                labelElement = Dfct.SubElement(labelClassesElement,"Label_"+str(numLabel))
                colorElement = Dfct.SubElement(labelElement,"Color")
                colorElement.text = str(labelWidget.color[0]) + "," + str(labelWidget.color[1]) + "," + str(labelWidget.color[2])
                nameElement = Dfct.SubElement(labelElement,"Name")
                nameElement.text = Dfct.convertTextToAscii(labelWidget.editableLabelName.label.text())
                nameElement = Dfct.SubElement(labelElement, "Name").set('ASCII', str(True))
                valueElement = Dfct.SubElement(labelElement,"Value")
                valueElement.text = str(labelWidget.value)

    def saveModel(self):

        modelName = self.settingsWidget.lineEditModelName.text()
        if modelName == "":
            modelName = self.settingsWidget.lineEditModelName.placeholderText()
        filePath = vrb.folderSuperPixelSegmentation + "/" + modelName
        if os.path.exists(filePath):
            fct.clearDirectory(filePath)
        else:
            os.makedirs(filePath)

        self.updateLabelClasses()

        self.updateSettingsSuperPixelsElement()

        Dfct.saveXmlElement(self.xmlElement, filePath + "/Settings.mho")

        for i in range(self.widgetLabelImage.layout.count()):
            item = self.widgetLabelImage.layout.itemAt(i)
            if item is not None:
                label = item.widget()

                name = label.labelName.text()
                folderImageName = filePath+"/"+name
                if os.path.exists(folderImageName):
                    fct.clearDirectory(folderImageName)
                else:
                    os.makedirs(folderImageName)

                PyIPSDK.saveTiffImageFile(folderImageName+"/Image.tif",label.image)

        if self.model is not None:
            joblib.dump(self.model, filePath + "/Model")

        try:
            # vrb.mainWindow.actualizeFunctionMachineLearning()
            vrb.mainWindow.updateCurrentFunction()
        except:
            pass
        self.settingsWidget.buttonSaveModel.setEnabled(False)

    def updateSettingsSuperPixelsElement(self):

        settingsSuperPixelsElement = Dfct.SubElement(self.xmlElement,"SettingsSuperPixels")

        Dfct.SubElement(settingsSuperPixelsElement,"ModelDimension").text = self.modelDimension

        if self.settingsWidget.lineEditSize.text() != "":
            Dfct.SubElement(settingsSuperPixelsElement,"Size").text = self.settingsWidget.lineEditSize.text()
        else:
            Dfct.SubElement(settingsSuperPixelsElement,"Size").text = "10"
        if self.settingsWidget.lineEditCompactness.text() != "":
            Dfct.SubElement(settingsSuperPixelsElement,"Compactness").text = self.settingsWidget.lineEditCompactness.text()
        else:
            Dfct.SubElement(settingsSuperPixelsElement,"Compactness").text = "0.5"
        if self.settingsWidget.lineEditSizeRatio.text() != "":
            Dfct.SubElement(settingsSuperPixelsElement,"SizeRatio").text = self.settingsWidget.lineEditSizeRatio.text()
        else:
            Dfct.SubElement(settingsSuperPixelsElement,"SizeRatio").text = "0.5"

    def loadModel(self,modelName):

        self.settingsWidget.lineEditModelName.setPlaceholderText(modelName)

        folder = vrb.folderSuperPixelSegmentation + "/" + modelName

        filePath = os.path.join(folder + "/Settings.mho")
        file = xmlet.parse(filePath)
        self.xmlElement = file.getroot()

        settingsSuperPixelsElement = Dfct.SubElement(self.xmlElement, "SettingsSuperPixels")

        try:
            if Dfct.childText(settingsSuperPixelsElement,"ModelDimension") is not None and Dfct.childText(settingsSuperPixelsElement,"ModelDimension") != "":
                self.modelDimension = Dfct.childText(settingsSuperPixelsElement,"ModelDimension")
        except:
            self.modelDimension = "2D"
        try:
            self.settingsWidget.lineEditSize.setText(Dfct.childText(settingsSuperPixelsElement,"Size"))
        except:
            self.settingsWidget.lineEditSize.setText("10")
        try:
            self.settingsWidget.lineEditCompactness.setText(Dfct.childText(settingsSuperPixelsElement,"Compactness"))
        except:
            self.settingsWidget.lineEditCompactness.setText("0.5")
        try:
            self.settingsWidget.lineEditSizeRatio.setText(Dfct.childText(settingsSuperPixelsElement,"SizeRatio"))
        except:
            self.settingsWidget.lineEditSizeRatio.setText("0.5")

        try:
            labelClassesElement = Dfct.SubElement(self.xmlElement, "LabelClasses")
            self.loadClassesElement(labelClassesElement)
        except:
            numberLabelElement = Dfct.SubElement(self.xmlElement, "NumberLabels")
            nbLabels = int(numberLabelElement.text)
            for numLabel in range(2, nbLabels):
                self.settingsWidget.labelSelectionGroupBox.labelScrollArea.addLabelWithName()

        featuresElement = Dfct.SubElement(self.xmlElement, 'Features')
        self.featuresWidget.xmlElement = featuresElement
        self.featuresWidget.loadXmlElement()

        allElements = Dfct.SubElement(self.xmlElement, 'AllElements')
        self.widgetLabelImage.xmlElement = allElements

        allElements = Dfct.SubElement(self.xmlElement, 'AllElements')
        for child in allElements:
            if child.tag == "Image":
                nameImage = Dfct.childText(child, "Name")
                if os.path.exists(folder + "/" + nameImage):
                    image = PyIPSDK.loadTiffImageFile(folder + "/" + nameImage + "/Image.tif")

                    label, name = self.widgetLabelImage.addNewImage(nameImage, image, xmlElement=child)

                    self.changeCurrentXmlElement(label, evaluate=False)

        try:
            self.model = joblib.load(folder+"/Model")

        except:
            traceback.print_exc(file=sys.stderr)
            self.model = None

        self.computeModel()
        self.actualizeNumberOfSuperPixels()

    def loadClassesElement(self,labelClassesElement,labelMin = 2):

        self.labelScrollArea.clearScrollArea()

        numberLabelElement = Dfct.SubElement(labelClassesElement, "NumberLabels")
        nbLabels = int(numberLabelElement.text)
        for numLabel in range(nbLabels):
            labelElement = Dfct.SubElement(labelClassesElement, "Label_" + str(numLabel))
            colorText = Dfct.childText(labelElement, "Color").split(",")
            color = [int(colorText[0]), int(colorText[1]), int(colorText[2])]
            name = Dfct.childText(labelElement, "Name")
            value = Dfct.childText(labelElement,"Value")

            self.settingsWidget.labelSelectionGroupBox.labelScrollArea.addLabelWithName(color=color, name=name,value = value)

    def updateSuperPixelsElement(self):

        if len(vrb.smartSegmentationSuperPixelsElements) == 0:
            if self.previousSuperPixelsElement is not None:
                superPixelsElements = Dfct.copyXmlElement(self.previousSuperPixelsElement)
            else:
                superPixelsElements = Dfct.copyXmlElement(Dfct.SubElement(self.currentXmlElement,"AllSuperPixels"))
            vrb.smartSegmentationSuperPixelsElements.append(superPixelsElements)
        else:
            if len(vrb.smartSegmentationSuperPixelsElements) == vrb.smartSegmentationSuperPixelsNumberPreviousElements:
                vrb.smartSegmentationSuperPixelsElements.remove(vrb.smartSegmentationSuperPixelsElements[0])
            if self.previousSuperPixelsElement is not None:
                superPixelsElements = Dfct.copyXmlElement(self.previousSuperPixelsElement)
                vrb.smartSegmentationSuperPixelsElements.append(superPixelsElements)
                # vrb.smartSegmentationSuperPixelsElements.append(self.previousSuperPixelsElement)

        self.previousSuperPixelsElement = Dfct.copyXmlElement(Dfct.SubElement(self.currentXmlElement,"AllSuperPixels"))

        self.settingsWidget.drawGroupBox.buttonBack.setEnabled(True)

    def getPreviousSuperPixelsElement(self):

        try:
            if len(vrb.smartSegmentationSuperPixelsElements) != 0:
                superPixelsElement = vrb.smartSegmentationSuperPixelsElements[len(vrb.smartSegmentationSuperPixelsElements) - 1]

                vrb.smartSegmentationSuperPixelsElements.remove(superPixelsElement)

                allSuperPixelsElement = Dfct.SubElement(self.currentXmlElement,"AllSuperPixels")
                Dfct.clearElement(allSuperPixelsElement)
                # Dfct.copySubElement(allSuperPixelsElement,superPixelsElement)
                Dfct.copySubElement(self.currentXmlElement,superPixelsElement)

                self.previousSuperPixelsElement = Dfct.copyXmlElement(superPixelsElement)

            self.settingsWidget.drawGroupBox.buttonBack.setEnabled(len(vrb.smartSegmentationSuperPixelsElements) >= 1)

            self.actualizeNumberOfSuperPixels()
            self.imageViewer.getRoiImage(changeRoiImage=False)

            if self.liveUpdate:
                self.computeModel()

        except:
            traceback.print_exc(file=sys.stderr)


    def addLabelsFromUrls(self, urls):

        for i in range(len(urls)):
            try:
                url = str(urls[i].toLocalFile())
            except:
                url = urls[i]
            urlSplit = url.split('.')
            extension = urlSplit[len(urlSplit) - 1]
            if extension in vrb.extensionImages:
                try:
                    image, name = self.urlToImage(url)
                    self.addImage(image, name)
                except:
                    traceback.print_exc(file=sys.stderr)
                    pass

    def urlToImage(self, url):

        name = os.path.basename(url)
        self.groupBoxProcessing.setText('Adding image\n' + name)
        self.toggleGroupBoxProcessing(True)
        image = fct.readImage(url, name=name)

        return image,name

    def addImage(self,image,name):

        self.groupBoxProcessing.setText('Adding image\n' + name)
        self.toggleGroupBoxProcessing(True)
        self.featuresMessage = True

        if image.getBufferType() in [PyIPSDK.eImageBufferType.eIBT_Label8,PyIPSDK.eImageBufferType.eIBT_Label16,PyIPSDK.eImageBufferType.eIBT_Label32,PyIPSDK.eImageBufferType.eIBT_Binary]:
            self.messageBox = wgt.MessageBox("You can't add this image\n This module doesn't accept image with type Label or Binary", '', buttons=[qt.QMessageBox.Ok], windowTitle="Error image")
            self.messageBox.exec()

        else:
            isFirstImage = self.isFirstImage()
            if isFirstImage:
                if image.getBufferType() == PyIPSDK.eIBT_Int8:
                    self.featuresWidget.lineEditMin.setText("-128")
                    self.featuresWidget.lineEditMax.setText("127")
                elif image.getBufferType() == PyIPSDK.eIBT_UInt16:
                    self.featuresWidget.lineEditMin.setText("0")
                    self.featuresWidget.lineEditMax.setText("65535")
                elif image.getBufferType() == PyIPSDK.eIBT_Int16:
                    self.featuresWidget.lineEditMin.setText("-32768")
                    self.featuresWidget.lineEditMax.setText("32767")
                else:
                    self.featuresWidget.lineEditMin.setText("0")
                    self.featuresWidget.lineEditMax.setText("255")
                self.featuresWidget.createXmlElement(None)

                if image.getSizeZ() == 1:
                    self.featuresWidget.checkBoxSizeZ.setVisible(False)
                    self.featuresWidget.resize(300*vrb.ratio,530*vrb.ratio)
                    self.modelDimension = "2D"
                else:
                    self.featuresWidget.checkBoxSizeZ.setVisible(True)
                    self.featuresWidget.resize(300*vrb.ratio,560*vrb.ratio)
                    self.modelDimension = "3D"


            item = self.widgetLabelImage.layout.itemAt(0)
            if item is None:
                label, name = self.widgetLabelImage.addNewImage(name, image,allowDuplicate=True)
                self.changeCurrentXmlElement(label)
            else:
                imageRef = item.widget().image
                if ((imageRef.getSizeZ() == 1 and image.getSizeZ() == 1) or (imageRef.getSizeZ() > 1 and image.getSizeZ() > 1)) and (imageRef.getSizeC() == image.getSizeC()):
                    label, name = self.widgetLabelImage.addNewImage(name, image,allowDuplicate=True)
                    self.changeCurrentXmlElement(label)
                else:
                    self.messageBox = wgt.MessageBox("You can't add this image\n All images must have the same geometry (2D or 3D and grey or color)", '', buttons=[qt.QMessageBox.Ok], windowTitle="Error image")
                    self.messageBox.exec()

        self.toggleGroupBoxProcessing(False)
        self.settingsWidget.buttonSaveModel.setEnabled(True)

    def isFirstImage(self):

        isFirstImage = True
        for i in range(self.widgetLabelImage.layout.count()):
            item = self.widgetLabelImage.layout.itemAt(i)
            if item is not None:
                isFirstImage = False

        return isFirstImage

    def changeCurrentXmlElement(self, label, evaluate=True):

        self.currentXmlElement = label.xmlElement
        self.currentLabel = label

        self.settingsWidget.drawGroupBox.buttonBack.setEnabled(False)
        self.previousSuperPixelsElement = None
        vrb.smartSegmentationSuperPixelsElements = []

        if self.currentLabel.imageSuperPixels is None:
            self.computeSuperPixels(self.currentLabel)

        allSuperPixelsElement = Dfct.SubElement(self.currentXmlElement, "AllSuperPixels")
        firstElement = Dfct.copyXmlElement(allSuperPixelsElement)
        vrb.smartSegmentationSuperPixelsElements.append(firstElement)

        if label is not None:
            self.setWindowTitle('IPSDK Explorer - ' + str(Dfct.childText(label.xmlElement, "Name")))
        else:
            self.setWindowTitle('IPSDK Explorer')

        # self.currentLabel.setStyleSelected(True)
        self.widgetLabelImage.updateStyleSheet()

        self.imageViewer.setImageOverlay(None)
        self.imageViewer.setImageProbabilities(None)
        self.widgetImage.setImage(self.currentLabel)
        self.widgetImage.setImageSuperPixels(self.currentLabel.imageSuperPixels)

        self.groupContrast.loadContrast()

        if evaluate:
            self.applyModel()

        self.imageViewer.getRoiImage(changeRoiImage=False)

    def toggleGroupBoxProcessing(self,showValue):

        if showValue:
            self.groupBoxProcessing.setVisible(True)
            qt.QApplication.processEvents()
            qt.QApplication.processEvents()
        else:
            self.groupBoxProcessing.setVisible(False)

    def computeModel(self):

        try:

            trainingData = None

            for i in range(self.widgetLabelImage.layout.count()):
                item = self.widgetLabelImage.layout.itemAt(i)
                if item is not None:
                    label = item.widget()
                    allSuperPixelsElement = Dfct.SubElement(label.xmlElement,"AllSuperPixels")
                    for child in allSuperPixelsElement:
                        labelValue = int(Dfct.childText(child,"LabelValue"))
                        classValue = [int(Dfct.childText(child,"Class"))]

                        data = np.concatenate((label.features[:, labelValue], classValue))
                        if trainingData is None:
                            trainingData = data
                        else:
                            trainingData = np.vstack((trainingData, data))

            #trainingData = fct.correctArrayValues(trainingData)

            try:
                nbTrees = int(self.settingsWidget.lineEditTreeNumber.text())
            except:
                nbTrees = 100

            self.model = RandomForestClassifier(n_estimators=nbTrees)
            self.model.fit(trainingData[:, :-1], trainingData[:, -1])

            self.settingsWidget.listWidgetFeatures.fillListWidget()

            self.settingsWidget.buttonSaveModel.setEnabled(True)

            self.applyModel()

        except:
            self.model = None
            self.settingsWidget.listWidgetFeatures.fillListWidget()
            self.imageViewer.getRoiImage(changeRoiImage=False)
            # traceback.print_exc(file=sys.stderr)

    def applyModel(self):

        try:

            if self.model is not None:

                dataToPredict = fct.correctArrayValues(self.currentLabel.features.T)
                vectorPrediction = self.model.predict(dataToPredict)
                vectorPredictionProba = self.model.predict_proba(dataToPredict)

                self.currentLabel.resultVector = vectorPrediction
                self.currentLabel.probabilitiesVector = vectorPredictionProba

                self.imageViewer.getRoiImage(changeRoiImage=False)

                self.widgetImage.sliderOpacity.setVisible(True)
                self.widgetImage.groupBoxProbabilities.setVisible(True)

        except:

            pass

    def colorLabelDeleted(self,num):

        allElements = Dfct.SubElement(self.xmlElement, 'AllElements')
        for elementImage in allElements:
            if elementImage.tag == "Image":
                allSuperPixelsElement = Dfct.SubElement(elementImage,"AllSuperPixels")
                childToRemove = []
                for child in allSuperPixelsElement:
                    if Dfct.childText(child,"Class") == str(num):
                        childToRemove.append(child)
                for child in childToRemove:
                    Dfct.removeElement(allSuperPixelsElement,child)
                for child in allSuperPixelsElement:
                    if Dfct.childText(child, "Class") is not None:
                        numElement = int(Dfct.childText(child, "Class"))
                        if numElement > num:
                            Dfct.SubElement(child,"Class").text = str(numElement-1)

        self.computeModel()
        self.imageViewer.getRoiImage(changeRoiImage=False)

    def labelDeleted(self, position):

        self.model = None

        self.setWindowTitle('IPSDK Explorer')
        currentPosition = int(Dfct.childText(self.currentXmlElement, "Position"))

        if currentPosition == -1:
            item = self.widgetLabelImage.layout.itemAt(position)
            if item is not None:
                self.changeCurrentXmlElement(item.widget())
            else:
                if position > 0:
                    item = self.widgetLabelImage.layout.itemAt(position - 1)
                    if item is not None:
                        self.changeCurrentXmlElement(item.widget())
                    else:
                        self.resetScene()
                else:
                    self.resetScene()

        self.model = None
        self.computeModel()
        self.settingsWidget.buttonSaveModel.setEnabled(True)

    def resetScene(self):

        self.model = None
        self.currentLabel = None
        self.currentXmlElement = None
        self.imageViewer.scene.clear()
        self.imageViewer.scene.clearSelection()
        self.imageViewer.scene.clearFocus()
        self.widgetImage.setImage(None)
        self.imageViewer.setImageOverlay(None)
        # self.imageViewer.setImageProbabilities(None)
        self.image = None
        self.widgetImage.sliderOpacity.setVisible(False)
        self.actualizeNumberOfSuperPixels()


class SettingsWidget(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.parent = parent

        self.pushButtonDocumentation = wgt.PushButtonImage(margins=1, filename=vrb.folderImages + '/Interrogation.png')
        self.pushButtonDocumentation.setStyleSheet("background-color : transparent; border :0px")
        self.pushButtonDocumentation.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.labelSelectionGroupBox = LabelSelectionGroupBox(parent=self.parent)
        self.drawGroupBox = DrawGroupBox(parent=self.parent)

        self.labelFeatures = qt.QLabel("Features")
        font = QtGui.QFont()
        font.setBold(True)
        font.setPointSize(10)
        self.labelFeatures.setFont(font)
        self.labelFeatures.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.labelFeatures.setFixedWidth(70 * vrb.ratio)

        self.emptyLabel0 = qt.QLabel()
        self.emptyLabel0.setFixedHeight(25 * vrb.ratio)

        self.groupBoxSuperPixelsParam = qt.QGroupBox()
        self.layoutSuperPixelsParam = qt.QGridLayout()
        self.labelSize = qt.QLabel("Size")
        self.lineEditSize = qt.QLineEdit()
        self.lineEditSize.setPlaceholderText("10")
        self.lineEditSize.setFixedSize(40*vrb.ratio,20*vrb.ratio)
        self.lineEditSize.setAlignment(Qt.AlignRight)
        self.labelCompactness = qt.QLabel("Compactness")
        self.lineEditCompactness = qt.QLineEdit()
        self.lineEditCompactness.setPlaceholderText("0.5")
        self.lineEditCompactness.setFixedSize(40*vrb.ratio,20*vrb.ratio)
        self.lineEditCompactness.setAlignment(Qt.AlignRight)
        self.labelSizeRatio = qt.QLabel("Size Ratio")
        self.lineEditSizeRatio = qt.QLineEdit()
        self.lineEditSizeRatio.setPlaceholderText("0.5")
        self.lineEditSizeRatio.setFixedSize(40*vrb.ratio,20*vrb.ratio)
        self.lineEditSizeRatio.setAlignment(Qt.AlignRight)

        self.pushButtonValidateParamSuperPixels = wgt.PushButtonImage(margins=1, filename=vrb.folderImages + '/Validate.png')
        self.pushButtonValidateParamSuperPixels.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.layoutSuperPixelsParam.addWidget(self.labelSize,0,0)
        self.layoutSuperPixelsParam.addWidget(self.lineEditSize,0,1,Qt.AlignLeft)
        self.layoutSuperPixelsParam.addWidget(self.labelCompactness,1,0)
        self.layoutSuperPixelsParam.addWidget(self.lineEditCompactness,1,1,Qt.AlignLeft)
        self.layoutSuperPixelsParam.addWidget(self.labelSizeRatio,2,0)
        self.layoutSuperPixelsParam.addWidget(self.lineEditSizeRatio,2,1,Qt.AlignLeft)
        self.layoutSuperPixelsParam.addWidget(self.pushButtonValidateParamSuperPixels,4,1,Qt.AlignRight)

        self.groupBoxSuperPixelsParam.setLayout(self.layoutSuperPixelsParam)
        self.groupBoxSuperPixelsParam.setTitle("Super pixels parameters")
        self.groupBoxSuperPixelsParam.setStyleSheet(f'QGroupBox:title {{font-size: {vrb.fontSizeMachineLearning*vrb.ratio}px; left: 20px;padding-left: 10px;padding-right: 10px;padding-top: {-10*vrb.ratio}px; color:rgb(52, 152, 219)}}  QGroupBox {{font-weight: bold; border: 1px solid gray; margin-top: {10*vrb.ratio}px }}')
        self.layoutSuperPixelsParam.setContentsMargins(10*vrb.ratio, 10*vrb.ratio, 10*vrb.ratio, 10*vrb.ratio)
        fontSPParams = QtGui.QFont()
        fontSPParams.setBold(False)
        fontSPParams.setPixelSize(vrb.fontSizeMachineLearning * vrb.ratio)

        # sets a font to the children of a widget
        for widget in self.groupBoxSuperPixelsParam.findChildren(qt.QWidget):
            widget.setFont(fontSPParams)

        self.buttonSettings = wgt.PushButtonImage(vrb.folderImages + "/Edit_3.png", margins=0)
        self.buttonSettings.setToolTip("Modify features")
        self.buttonSettings.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.labelTreeNumber = qt.QLabel("Number of trees")
        self.labelTreeNumber.setFont(font)
        self.labelTreeNumber.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.labelTreeNumber.setFixedWidth(130 * vrb.ratio)
        self.lineEditTreeNumber = wgt.NumberLineEdit()
        self.lineEditTreeNumber.setPlaceholderText("100")
        self.lineEditTreeNumber.setFixedWidth(50 * vrb.ratio)

        self.groupBoxTreeNumber = qt.QGroupBox()
        self.layoutTreeNumber = qt.QGridLayout()
        self.layoutTreeNumber.addWidget(self.labelTreeNumber, 0, 0)
        self.layoutTreeNumber.addWidget(self.lineEditTreeNumber, 0, 1, Qt.AlignLeft)
        self.groupBoxTreeNumber.setLayout(self.layoutTreeNumber)
        self.layoutTreeNumber.setContentsMargins(0, 0, 0, 0)
        self.layoutTreeNumber.setSizeConstraint(1)

        self.labelLiveUpdate = qt.QLabel("Live update")
        self.labelLiveUpdate.setFont(font)
        self.labelLiveUpdate.setAlignment(Qt.AlignCenter)
        self.labelLiveUpdate.setFixedWidth(90 * vrb.ratio)
        self.labelLiveUpdate.setStyleSheet('QLabel {color: rgb(52, 152, 219);background-color: rgb(70,70,70);}')

        self.buttonLiveUpdate = wgt.PushButtonDoubleImage(vrb.folderImages + "/Pause.png", vrb.folderImages + "/Lecture.png")
        self.buttonLiveUpdate.setToolTip("Stop live update")
        self.buttonLiveUpdate.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.buttonRefresh = wgt.PushButtonImage(vrb.folderImages + "/Refresh_White.png", margins=5)
        self.buttonRefresh.setToolTip("Evaluate model")
        self.buttonRefresh.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.emptyLabel1 = qt.QLabel()
        self.emptyLabel1.setFixedHeight(25 * vrb.ratio)

        self.groupBoxFeatures = qt.QGroupBox()
        self.layoutFeatures = qt.QGridLayout()
        self.layoutFeatures.addWidget(self.labelFeatures,0,0)
        self.layoutFeatures.addWidget(self.buttonSettings,0,1,Qt.AlignLeft)

        self.layoutFeatures.addWidget(self.labelTreeNumber,1,0)
        self.layoutFeatures.addWidget(self.lineEditTreeNumber,1,1,Qt.AlignLeft)

        self.groupBoxFeatures.setLayout(self.layoutFeatures)
        self.groupBoxFeatures.setFixedHeight(80*vrb.ratio)
        self.layoutFeatures.setContentsMargins(20, 0, 0, 0)
        self.layoutFeatures.setSizeConstraint(1)

        self.emptyLabel2 = qt.QLabel()
        self.emptyLabel2.setFixedHeight(25 * vrb.ratio)

        self.groupBoxLiveUpdate = qt.QGroupBox()
        self.layoutLiveUpdate = qt.QGridLayout()
        self.layoutLiveUpdate.addWidget(self.labelLiveUpdate, 0, 0)
        self.layoutLiveUpdate.addWidget(self.buttonLiveUpdate, 0, 1)
        self.layoutLiveUpdate.addWidget(self.buttonRefresh, 0, 2)
        self.layoutLiveUpdate.setAlignment(Qt.AlignLeft)
        self.buttonRefresh.setVisible(False)
        self.groupBoxLiveUpdate.setLayout(self.layoutLiveUpdate)
        self.groupBoxLiveUpdate.setFixedHeight(30 * vrb.ratio)
        self.layoutLiveUpdate.setContentsMargins(20, 0, 0, 0)
        self.layoutLiveUpdate.setSizeConstraint(1)

        self.emptyLabel3 = qt.QLabel()
        self.emptyLabel3.setFixedHeight(25 * vrb.ratio)

        self.groupBoxLabelListFeatures = qt.QGroupBox()
        self.groupBoxLabelListFeatures.setFixedHeight(25 * vrb.ratio)
        layoutLabelListFeatures = qt.QGridLayout()
        self.groupBoxLabelListFeatures.setLayout(layoutLabelListFeatures)
        layoutLabelListFeatures.setContentsMargins(0, 0, 0, 0)
        layoutLabelListFeatures.setSizeConstraint(1)
        self.groupBoxLabelListFeatures.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.labelListOfFeatures = qt.QLabel("List of features (sorted by importance)")
        font = QtGui.QFont()
        font.setPointSize(9)
        self.labelListOfFeatures.setFont(font)
        self.labelListOfFeatures.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.labelListOfFeatures.setFixedHeight(25 * vrb.ratio)

        layoutLabelListFeatures.addWidget(self.labelListOfFeatures, 0, 0)

        self.groupBoxRemoveFeatures = qt.QGroupBox()
        self.groupBoxRemoveFeatures.setFixedHeight(25 * vrb.ratio)
        layoutRemoveFeatures = qt.QGridLayout()
        self.groupBoxRemoveFeatures.setLayout(layoutRemoveFeatures)
        layoutRemoveFeatures.setContentsMargins(0, 0, 0, 0)
        layoutRemoveFeatures.setSizeConstraint(1)
        self.groupBoxRemoveFeatures.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.labelRemoveFeatures = qt.QLabel("Remove features below")
        font = QtGui.QFont()
        font.setPointSize(9)
        self.labelRemoveFeatures.setFont(font)
        # self.labelRemoveFeatures.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.lineEditPercentage = wgt.NumberLineEdit(constraint='Real')
        self.lineEditPercentage.setText("0")
        self.lineEditPercentage.setPlaceholderText("0")
        self.lineEditPercentage.setFixedWidth(35 * vrb.ratio)
        self.labelPercentage = qt.QLabel("%")
        self.labelPercentage.setFont(font)
        # self.labelPercentage.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.buttonRemoveFeatures = wgt.PushButtonImage(vrb.folderImages + "/Trash2.png", margins=5)
        self.buttonRemoveFeatures.setToolTip("Remove Features")
        self.buttonRemoveFeatures.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        layoutRemoveFeatures.addWidget(self.labelRemoveFeatures, 0, 0)
        layoutRemoveFeatures.addWidget(self.lineEditPercentage, 0, 1)
        layoutRemoveFeatures.addWidget(self.labelPercentage, 0, 2)
        layoutRemoveFeatures.addWidget(self.buttonRemoveFeatures, 0, 3)

        self.groupSaveModel = qt.QGroupBox()
        self.groupSaveModel.setFixedHeight(25 * vrb.ratio)
        layoutSaveModel = qt.QGridLayout()
        self.groupSaveModel.setLayout(layoutSaveModel)
        layoutSaveModel.setContentsMargins(0, 0, 0, 0)
        layoutSaveModel.setSizeConstraint(1)
        self.groupSaveModel.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.labelModelName = qt.QLabel("Model name")
        font = QtGui.QFont()
        font.setPointSize(9)
        self.labelModelName.setFont(font)
        self.labelModelName.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.lineEditModelName = qt.QLineEdit()
        self.lineEditModelName.setFixedWidth(140 * vrb.ratio)
        modelName = fct.getModelName(vrb.folderShapeClassification)
        self.lineEditModelName.setPlaceholderText(modelName)
        self.buttonSaveModel = wgt.PushButtonImage(vrb.folderImages + "/Save.png", margins=2)
        self.buttonSaveModel.setToolTip("Save model")
        self.buttonSaveModel.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        layoutSaveModel.addWidget(self.labelModelName, 0, 0)
        layoutSaveModel.addWidget(self.lineEditModelName, 0, 1)
        layoutSaveModel.addWidget(self.buttonSaveModel, 0, 2)

        self.groupBoxListWidgetFeatures = qt.QGroupBox()
        self.groupBoxListWidgetFeatures.setFixedHeight(25 * vrb.ratio)
        layoutListWidgetFeatures = qt.QGridLayout()
        self.groupBoxListWidgetFeatures.setLayout(layoutListWidgetFeatures)
        layoutListWidgetFeatures.setContentsMargins(0, 0, 0, 0)
        layoutListWidgetFeatures.setSizeConstraint(1)
        self.groupBoxListWidgetFeatures.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.listWidgetFeatures = ListWidgetFeatures(parent=self.parent)

        self.labelFeaturesName = wgt.LabelClickable("Name    ▼")
        self.labelFeaturesName.sort = True
        self.labelFeaturesName.setAlignment(Qt.AlignLeft)
        self.labelFeaturesName.setFixedWidth(150*vrb.ratio)
        self.labelFeaturesPercentage = wgt.LabelClickable()
        self.labelFeaturesPercentage.sort = True
        self.labelFeaturesPercentage.setText("Percentage")
        self.labelFeaturesPercentage.setAlignment(Qt.AlignLeft)
        self.currentLabelSort = self.labelFeaturesName
        layoutListWidgetFeatures.addWidget(self.labelFeaturesName,0,0)
        layoutListWidgetFeatures.addWidget(self.labelFeaturesPercentage,0,1)
        layoutListWidgetFeatures.addWidget(self.listWidgetFeatures,1,0,1,2)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.pushButtonDocumentation, 0, 0, Qt.AlignRight)
        self.layout.addWidget(self.labelSelectionGroupBox, 1, 0)
        self.layout.addWidget(self.drawGroupBox, 2, 0)
        self.layout.addWidget(self.emptyLabel0, 3, 0)
        self.layout.addWidget(self.groupBoxSuperPixelsParam, 4, 0)
        self.layout.addWidget(self.emptyLabel1, 5, 0)
        self.layout.addWidget(self.groupBoxFeatures, 6, 0)
        self.layout.addWidget(self.emptyLabel2, 7, 0)
        self.layout.addWidget(self.groupBoxLiveUpdate, 8, 0)
        self.layout.addWidget(self.emptyLabel3, 9, 0)
        self.layout.addWidget(self.groupBoxLabelListFeatures, 10, 0)
        self.layout.addWidget(self.groupBoxListWidgetFeatures, 10, 0)
        self.layout.addWidget(self.groupBoxRemoveFeatures,12,0,Qt.AlignTop)
        self.layout.addWidget(self.groupSaveModel,13,0,Qt.AlignBottom)

        self.layout.setContentsMargins(5*vrb.ratio, 5*vrb.ratio, 5*vrb.ratio, 35*vrb.ratio)
        self.layout.setVerticalSpacing(0)
        self.layout.setSizeConstraint(1)

        self.setLayout(self.layout)

        self.pushButtonDocumentation.clicked.connect(self.showDocumentation)
        self.buttonSettings.clicked.connect(self.displayFeatures)
        self.buttonLiveUpdate.clicked.connect(self.changeLiveUpdate)
        self.buttonRefresh.clicked.connect(self.refreshModel)
        self.lineEditTreeNumber.textChanged.connect(self.changeNumberTree)
        self.lineEditTreeNumber.returnPressed.connect(self.parent.computeModel)
        self.buttonRemoveFeatures.clicked.connect(self.removeLowFeatures)
        self.labelFeaturesName.clicked.connect(self.sortByName)
        self.labelFeaturesPercentage.clicked.connect(self.sortByPercentage)
        self.lineEditModelName.textChanged.connect(self.allowSaveButton)

        self.lineEditSize.textChanged.connect(self.changeButtonSuperPixels)
        self.lineEditCompactness.textChanged.connect(self.changeButtonSuperPixels)
        self.lineEditSizeRatio.textChanged.connect(self.changeButtonSuperPixels)

    def resizeEvent(self, a0: QtGui.QResizeEvent):

        totalHeight = 0

        totalHeight += 30*vrb.ratio + 25 *vrb.ratio #size button Interrogation + margins

        drawGroupBoxHeight = 40*vrb.ratio
        self.drawGroupBox.setFixedSize(max(0, self.width() - 10 * vrb.ratio), drawGroupBoxHeight)
        totalHeight += drawGroupBoxHeight

        superPixelsGroupBoxHeight = 120*vrb.ratio
        self.groupBoxSuperPixelsParam.setFixedSize(max(0, self.width() - 10 * vrb.ratio), superPixelsGroupBoxHeight)
        totalHeight += superPixelsGroupBoxHeight

        groupBoxFeaturesHeight = 80*vrb.ratio
        self.groupBoxFeatures.setFixedSize(max(0, self.width() - 10 * vrb.ratio), groupBoxFeaturesHeight)
        totalHeight += groupBoxFeaturesHeight

        groupBoxLiveUpdateHeight = 40*vrb.ratio
        self.groupBoxLiveUpdate.setFixedSize(max(0, self.width() - 10 * vrb.ratio), groupBoxLiveUpdateHeight)
        totalHeight += groupBoxLiveUpdateHeight

        groupBoxLabelListFeaturesHeight = 25*vrb.ratio
        self.groupBoxLabelListFeatures.setFixedSize(max(0, self.width() - 10 * vrb.ratio), groupBoxLabelListFeaturesHeight)
        totalHeight += groupBoxLabelListFeaturesHeight

        groupBoxRemoveFeaturesHeight = 25*vrb.ratio
        self.groupBoxRemoveFeatures.setFixedSize(max(0, self.width() - 10 * vrb.ratio), groupBoxRemoveFeaturesHeight)
        totalHeight += groupBoxRemoveFeaturesHeight

        minSizeLabelSelectionGroupBox = 200*vrb.ratio
        minSizegroupBoxLabelListFeatures = 200*vrb.ratio
        sizeEmptyLabel = (self.height() - totalHeight - minSizeLabelSelectionGroupBox - minSizegroupBoxLabelListFeatures)/3
        sizeEmptyLabel = max(0,sizeEmptyLabel)
        sizeEmptyLabel = min(10*vrb.ratio,sizeEmptyLabel)

        self.emptyLabel1.setFixedHeight(sizeEmptyLabel)
        self.emptyLabel2.setFixedHeight(sizeEmptyLabel)
        self.emptyLabel3.setFixedHeight(sizeEmptyLabel)
        totalHeight += 3*sizeEmptyLabel
        #
        totalHeight += 80*vrb.ratio # spacing between widgets

        # heightLabelSelectionGroupBox = (self.height()-totalHeight)*3/5
        heightLabelSelectionGroupBox = (self.height()-totalHeight)
        heightLabelSelectionGroupBox = min(heightLabelSelectionGroupBox,250*vrb.ratio)

        self.labelSelectionGroupBox.setFixedSize(max(0,self.width()-10*vrb.ratio),heightLabelSelectionGroupBox)
        totalHeight += heightLabelSelectionGroupBox

        self.groupBoxListWidgetFeatures.setFixedSize(max(0,self.width()-10*vrb.ratio),self.height()-totalHeight + 40*vrb.ratio)

        self.labelSelectionGroupBox.setFixedSize(max(0, self.width() - 10 * vrb.ratio), 230 * vrb.ratio)
        self.drawGroupBox.setFixedSize(max(0, self.width() - 10 * vrb.ratio), 40 * vrb.ratio)

    def changeButtonSuperPixels(self):

        self.pushButtonValidateParamSuperPixels.setEnabled(True)

    def sortByName(self):

        if self.currentLabelSort == self.labelFeaturesName:
            self.labelFeaturesName.sort = not(self.labelFeaturesName.sort)

        self.currentLabelSort = self.labelFeaturesName

        if self.labelFeaturesName.sort:
            self.labelFeaturesName.setText("Name    ▼")
        else:
            self.labelFeaturesName.setText("Name    ▲")
        self.labelFeaturesPercentage.setText("Percentage")
        self.listWidgetFeatures.fillListWidget()

    def sortByPercentage(self):

        if self.currentLabelSort == self.labelFeaturesPercentage:
            self.labelFeaturesPercentage.sort = not (self.labelFeaturesPercentage.sort)

        self.currentLabelSort = self.labelFeaturesPercentage

        if self.labelFeaturesPercentage.sort:
            self.labelFeaturesPercentage.setText("Percentage    ▼")
        else:
            self.labelFeaturesPercentage.setText("Percentage    ▲")
        self.labelFeaturesName.setText("Name")
        self.listWidgetFeatures.fillListWidget()

    def removeLowFeatures(self):

        try:
            try:
                ponderationRef = float(self.lineEditPercentage.text())
            except:
                ponderationRef = 0

            listFeatures = self.listWidgetFeatures.listFeatures
            listPercentage = self.listWidgetFeatures.listPercentage

            for i in range(len(listFeatures)):
                if listPercentage[i]*100<=ponderationRef:
                    featuresToRemove = listFeatures[i]

                    if featuresToRemove == "Nb Pixels":
                        self.parent.featuresWidget.checkBoxNbPixels.setChecked(False)
                    if featuresToRemove == "Size X":
                        self.parent.featuresWidget.checkBoxSizeX.setChecked(False)
                    if featuresToRemove == "Size Y":
                        self.parent.featuresWidget.checkBoxSizeY.setChecked(False)
                    if featuresToRemove == "Size Z":
                        self.parent.featuresWidget.checkBoxSizeZ.setChecked(False)
                    if featuresToRemove == "Minimum":
                        self.parent.featuresWidget.checkBoxMinimum.setChecked(False)
                    if featuresToRemove == "Maximum":
                        self.parent.featuresWidget.checkBoxMaximum.setChecked(False)
                    if featuresToRemove == "Mean":
                        self.parent.featuresWidget.checkBoxMean.setChecked(False)
                    if featuresToRemove == "Median":
                        self.parent.featuresWidget.checkBoxMedian.setChecked(False)
                    if featuresToRemove == "Variance":
                        self.parent.featuresWidget.checkBoxVariance.setChecked(False)
                    if featuresToRemove == "Energy":
                        self.parent.featuresWidget.checkBoxEnergy.setChecked(False)
                    if featuresToRemove == "Entropy":
                        self.parent.featuresWidget.checkBoxEntropy.setChecked(False)
                    if featuresToRemove == "Kurtosis":
                        self.parent.featuresWidget.checkBoxKurtosis.setChecked(False)
                    if featuresToRemove == "Skewness":
                        self.parent.featuresWidget.checkBoxSkewness.setChecked(False)
                    if featuresToRemove == "Histogram":
                        self.parent.featuresWidget.checkBoxHistogram.setChecked(False)

            self.parent.featuresWidget.createXmlElement(None)

        except:
            pass


    def showDocumentation(self):

        webbrowser.open_new_tab(vrb.folderDocEssential + "/superpixel_smartsegmentation.html")

    def allowSaveButton(self):

        try:
            self.buttonSaveModel.setEnabled(True)
            self.labelSelectionGroupBox.updateSaveSettingsButton()
        except:
            pass

    def changeNumberTree(self):

        self.parent.model = None

    def refreshModel(self):

        self.parent.computeModel()

    def displayFeatures(self):

        fct.showWidget(self.parent.featuresWidget)

    def changeLiveUpdate(self):

        self.buttonLiveUpdate.changeActivation()
        if self.buttonLiveUpdate.activate is False:
            self.parent.computeModel()
            self.parent.liveUpdate = True
            self.labelLiveUpdate.setStyleSheet('QLabel {color: rgb(52, 152, 219); background-color: rgb(70,70,70);}')
            self.buttonRefresh.setVisible(False)
        else:
            self.parent.liveUpdate = False
            self.labelLiveUpdate.setStyleSheet('QLabel {color: rgb(52, 152, 219); background-color: transparent;}')
            self.buttonRefresh.setVisible(True)


class ListWidgetFeatures(qt.QListWidget):

    def __init__(self,parent=None):
        qt.QListWidget.__init__(self)

        self.parent = parent
        self.listFeatures = []
        self.listPercentage = []

    def fillListWidget(self):

        settingsWidget = self.parent.settingsWidget

        try:

            self.clear()
            label = self.parent.currentLabel
            model = self.parent.model

            featuresElement = Dfct.SubElement(self.parent.xmlElement, "Features")
            listFeatures = []
            listPercentage = []

            if Dfct.childText(featuresElement, "NbPixels") == "True":
                listFeatures.append("Nb Pixels")
            if Dfct.childText(featuresElement, "SizeX") == "True":
                listFeatures.append( "Size X")
            if Dfct.childText(featuresElement, "SizeY") == "True":
                listFeatures.append("Size Y")
            if Dfct.childText(featuresElement, "SizeZ") == "True" and self.parent.modelDimension == "3D":
                listFeatures.append("Size Z")
            if Dfct.childText(featuresElement, "Minimum") == "True":
                listFeatures.append("Minimum")
            if Dfct.childText(featuresElement, "Maximum") == "True":
                listFeatures.append("Maximum")
            if Dfct.childText(featuresElement, "Mean") == "True":
                listFeatures.append( "Mean")
            if Dfct.childText(featuresElement, "Median") == "True":
                listFeatures.append("Median")
            if Dfct.childText(featuresElement, "Variance") == "True":
                listFeatures.append("Variance")
            if Dfct.childText(featuresElement, "Energy") == "True":
                listFeatures.append("Energy")
            if Dfct.childText(featuresElement, "Entropy") == "True":
                listFeatures.append("Entropy")
            if Dfct.childText(featuresElement, "Kurtosis") == "True":
                listFeatures.append("Kurtosis")
            if Dfct.childText(featuresElement, "Skewness") == "True":
                listFeatures.append("Skewness")
            if Dfct.childText(featuresElement, "Histogram") == "True":
                listFeatures.append("Histogram")

            for _ in listFeatures:
                listPercentage.append(0)

            if label is not None and len(label.listFeatures) != 0 and model is not None:

                featuresImportance = model.feature_importances_
                for i in range(len(listFeatures)):
                    for j in range(len(label.listFeatures)):
                        if label.listFeatures[j] == listFeatures[i]:
                            listPercentage[i] += featuresImportance[j]

            if settingsWidget.currentLabelSort == settingsWidget.labelFeaturesName:
                ind = np.argsort(listFeatures)
                if settingsWidget.labelFeaturesName.sort == False:
                    ind = ind[::-1]
            else:
                if model is not None:
                    ind = np.argsort(listPercentage)
                    if settingsWidget.labelFeaturesPercentage.sort == False:
                        ind = ind[::-1]
                else:
                    ind = np.argsort(listFeatures)

            for index in ind:

                newItem = qt.QListWidgetItem()
                newItem.setText(listFeatures[index])

                if model is not None:
                    widget = qt.QLabel(fct.numberCalibration(listPercentage[index] * 100) + " %")
                else:
                    widget = qt.QLabel("0 %")
                widget.setAlignment(Qt.AlignRight)

                self.addItem(newItem)
                self.setItemWidget(newItem, widget)

            self.listFeatures = listFeatures
            self.listPercentage = listPercentage

        except:
            traceback.print_exc(file=sys.stderr)

class DrawGroupBox(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.parent = parent

        self.buttonPen = wgt.PushButtonDoubleImage(vrb.folderImages + "/Pen.png", vrb.folderImages + "/Pen_Pressed.png")
        self.buttonPen.setToolTip("Draw on the image")
        self.buttonPen.setFixedSize(25*vrb.ratio, 25*vrb.ratio)
        self.buttonPen.mode = "Pen"

        self.buttonEraser = wgt.PushButtonDoubleImage(vrb.folderImages + "/Eraser.png", vrb.folderImages + "/Eraser_Pressed.png")
        self.buttonEraser.setToolTip("Erase on the image")
        self.buttonEraser.setFixedSize(25*vrb.ratio, 25*vrb.ratio)
        self.buttonEraser.mode = "Eraser"

        self.buttonSelection = wgt.PushButtonDoubleImage(vrb.folderImages + "/Zone.png",vrb.folderImages + "/Zone_Pressed.png")
        self.buttonSelection.setToolTip("Draw a rectangle to select superpixels")
        self.buttonSelection.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonSelection.mode = "Selection"

        self.buttonBack = wgt.PushButtonImage(vrb.folderImages + "/Back.png")
        self.buttonBack.setFixedSize(25* vrb.ratio,25* vrb.ratio)
        self.buttonBack.setEnabled(False)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.buttonPen, 0, 0)
        self.layout.addWidget(self.buttonEraser, 0, 1)
        self.layout.addWidget(self.buttonSelection, 0, 2)
        self.layout.addWidget(self.buttonBack, 0, 3,Qt.AlignLeft)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setHorizontalSpacing(5)
        self.layout.setContentsMargins(10*vrb.ratio, 5*vrb.ratio, 10*vrb.ratio, 5*vrb.ratio)

        self.buttonPen.clicked.connect(self.changeMode)
        self.buttonEraser.clicked.connect(self.changeMode)
        self.buttonSelection.clicked.connect(self.changeMode)

        self.setMode("Pen")

    def changeMode(self):

        sender = self.sender()
        for button in [self.buttonPen, self.buttonEraser,self.buttonSelection]:
            button.setActivation(button == sender)
            if button == sender:
                self.parent.currentMode = button.mode

    def setMode(self, mode):

        for button in [self.buttonPen, self.buttonEraser,self.buttonSelection]:
            button.setActivation(button.mode == mode)
            self.parent.currentMode = mode

class LabelSelectionGroupBox(qt.QGroupBox):

    def __init__(self,parent=None):
        qt.QGroupBox.__init__(self)

        self.parent = parent

        # self.labelSelection = qt.QLabel("Class selection")
        self.labelSelection = qt.QLabel("Settings")
        font = QtGui.QFont()
        font.setBold(True)
        font.setPointSize(10)

        self.labelSelection.setFont(font)
        # self.labelSelection.setStyleSheet('QLabel {color: rgb(6, 115, 186);}')
        self.labelSelection.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.labelSelection.setFixedWidth(80*vrb.ratio)

        self.comboBoxLoadingSettings = qt.QComboBox()
        self.comboBoxLoadingSettings.setFixedSize(70*vrb.ratio,25*vrb.ratio)

        self.buttonLoadSettings = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=2)
        self.buttonLoadSettings.setToolTip("Load settings")
        self.buttonLoadSettings.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.buttonDeleteSettings = wgt.PushButtonImage(vrb.folderImages + "/Delete2.png", margins=2)
        self.buttonDeleteSettings.setToolTip("Delete settings")
        self.buttonDeleteSettings.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.groupLabelSelection = qt.QGroupBox()
        #self.groupLabelSelection.setFixedHeight(25 * vrb.ratio)
        layoutLabelSelection = qt.QGridLayout()
        self.groupLabelSelection.setLayout(layoutLabelSelection)
        layoutLabelSelection.setContentsMargins(0, 0, 0, 0)
        layoutLabelSelection.setSizeConstraint(1)
        self.groupLabelSelection.setStyleSheet('QGroupBox {border: 0px transparent; }')

        layoutLabelSelection.addWidget(self.labelSelection,0,0,Qt.AlignLeft)
        layoutLabelSelection.addWidget(self.comboBoxLoadingSettings,0,1,Qt.AlignRight)
        layoutLabelSelection.addWidget(self.buttonLoadSettings,0,2)
        layoutLabelSelection.addWidget(self.buttonDeleteSettings,0,3)

        self.labelScrollArea = LabelScrollArea(parent = self.parent)

        self.buttonAdd = wgt.PushButtonImage(vrb.folderImages + "/Add_4.png", margins=0)
        # self.buttonAdd = wgt.PushButtonImage(vrb.folderImages + "/Pen.png", margins=5)
        self.buttonAdd.setToolTip("Add a new label")
        self.buttonAdd.setFixedSize(25*vrb.ratio,25*vrb.ratio)

        self.lineEditSaveSettings = qt.QLineEdit()
        self.lineEditSaveSettings.setPlaceholderText("Save settings")
        self.lineEditSaveSettings.setFixedSize(120*vrb.ratioDimension,25*vrb.ratio)

        self.buttonSaveSettings = wgt.PushButtonImage(vrb.folderImages + "/Save.png", margins=2)
        self.buttonSaveSettings.setToolTip("Save settings")
        self.buttonSaveSettings.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.groupLabelSelection, 0, 0,1,3)
        self.layout.addWidget(self.labelScrollArea, 1, 0,1,3)
        self.layout.addWidget(self.buttonAdd, 2, 0, Qt.AlignLeft)
        self.layout.addWidget(self.lineEditSaveSettings, 2, 1, Qt.AlignRight)
        self.layout.addWidget(self.buttonSaveSettings, 2, 2)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setHorizontalSpacing(5)
        self.layout.setContentsMargins(20, 5, 20, 5)

        self.buttonAdd.clicked.connect(self.labelScrollArea.addLabel)
        self.lineEditSaveSettings.textChanged.connect(self.updateSaveSettingsButton)
        self.buttonSaveSettings.clicked.connect(self.saveSettings)
        self.buttonLoadSettings.clicked.connect(self.loadSettings)
        self.buttonDeleteSettings.clicked.connect(self.deleteSettings)

        self.updateSaveSettingsButton()
        self.updateComboboxSettings()

        # self.setTitle("Label selection")
        # self.setStyleSheet('QGroupBox:title {left: 8px ;padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {font: bold; margin-top: 12 px;font-size:12px}')

    def resizeEvent(self, a0: QtGui.QResizeEvent):

        self.labelScrollArea.setFixedSize(self.width()-20*vrb.ratio,self.height()-90*vrb.ratio)

    def updateSaveSettingsButton(self):

        self.buttonSaveSettings.setEnabled(self.lineEditSaveSettings.text()!="")

    def updateComboboxSettings(self):

        self.comboBoxLoadingSettings.clear()
        for fileName in os.listdir(vrb.folderSettingsRF):
            fileSettings = xmlet.parse(vrb.folderSettingsRF + "/" + fileName)
            xmlElement = fileSettings.getroot()
            self.comboBoxLoadingSettings.addItem(fileName[:-4],xmlElement)

    def saveSettings(self):

        self.parent.updateLabelClasses()

        labelClassesElement = Dfct.SubElement(self.parent.xmlElement, "LabelClasses")
        nameSettings = self.lineEditSaveSettings.text()

        found = False
        for fileName in os.listdir(vrb.folderSettingsRF):

            if nameSettings + ".mho" == fileName:
                found = True
                text = 'The settings "'+nameSettings+'" already exists.\nDo you want to replace it ?'
                self.messageBox = wgt.MessageBox(text, '', buttons=[qt.QMessageBox.Yes,qt.QMessageBox.No],windowTitle="Settings already exists", icon=qt.QMessageBox.Warning)
                res = self.messageBox.exec()
                if res == qt.QMessageBox.Yes:
                    Dfct.saveXmlElement(labelClassesElement,vrb.folderSettingsRF + "/" + nameSettings + ".mho")
                    self.buttonSaveSettings.setEnabled(False)

        if found == False:
            Dfct.saveXmlElement(labelClassesElement, vrb.folderSettingsRF + "/" + nameSettings + ".mho")
            self.buttonSaveSettings.setEnabled(False)

        self.updateComboboxSettings()

    def loadSettings(self):

        self.labelScrollArea.clearScrollArea()

        xmlElement = self.comboBoxLoadingSettings.currentData()

        if xmlElement is not None:
            try:
                self.parent.loadClassesElement(xmlElement,labelMin = 0)
            except:
                traceback.print_exc(file=sys.stderr)

        self.parent.actualizeNumberOfSuperPixels()
        self.parent.settingsWidget.buttonSaveModel.setEnabled(True)

        self.lineEditSaveSettings.setText(self.comboBoxLoadingSettings.currentText())

        self.parent.computeModel()

    def deleteSettings(self):

        nameSettings = self.comboBoxLoadingSettings.currentText()

        self.messageBox = wgt.MessageBox('Are you sure you want to delete the settings "'+nameSettings+'" ?', '', buttons=[qt.QMessageBox.Yes, qt.QMessageBox.No], windowTitle="Delete settings", icon=qt.QMessageBox.Warning)
        res = self.messageBox.exec()
        if res == qt.QMessageBox.Yes and os.path.exists(vrb.folderSettingsRF + "/" + nameSettings + ".mho"):
            try:
                os.remove(vrb.folderSettingsRF + "/" + nameSettings + ".mho")
            except:
                pass
            self.updateComboboxSettings()


class LabelScrollArea(qt.QScrollArea):

    SignalDeleteLabel = pyqtSignal(int)
    SignalColorChanged = pyqtSignal()
    SignalAddLabel = pyqtSignal()

    def __init__(self,parent=None):
        qt.QScrollArea.__init__(self)

        self.parent = parent

        self.selectedLabel = None

        self.centralWidget = qt.QGroupBox()

        self.layout = qt.QVBoxLayout()
        self.layout.setAlignment(Qt.AlignTop)

        self.centralWidget.setLayout(self.layout)

        self.setWidget(self.centralWidget)

        self.layout.setContentsMargins(0, 0, 0, 0)

        self.setStyleSheet('QScrollArea {border: 1px solid gray; }')

        self.addLabel()
        self.addLabel()

    def resizeEvent(self, a0: QtGui.QResizeEvent):

        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                label = item.widget()
                label.setFixedSize(self.width() - 20*vrb.ratio, 35 * vrb.ratio)

        self.centralWidget = qt.QWidget()
        self.centralWidget.setFixedWidth(self.width()-20*vrb.ratio)
        self.centralWidget.setLayout(self.layout)
        self.setWidget(self.centralWidget)

    def addLabel(self):

        self.addLabelWithName()
        self.allowSaveButton()

    def addLabelWithName(self,color=None,name=None,value=None):

        labelNum = self.layout.count()
        label = GroupBoxColor(labelNum + 1, labelNum,color=color,name=name,parent=self.parent,value=value)
        label.setFixedSize(self.width() - 20*vrb.ratio, 35 * vrb.ratio)
        label.SignalSelection.connect(self.selectLabel)
        label.SignalDeleteVariable.connect(self.deleteLabel)
        label.SignalColorChanged.connect(self.emitSignalColorChanged)
        label.editableLabelName.lineEdit.textChanged.connect(self.allowSaveButton)

        self.layout.addWidget(label)

        self.centralWidget = qt.QWidget()
        self.centralWidget.setFixedWidth(self.width() - 20*vrb.ratio)
        self.centralWidget.setLayout(self.layout)
        self.setWidget(self.centralWidget)

        self.verticalScrollBar().setValue(self.verticalScrollBar().maximum())

        self.SignalAddLabel.emit()

        if labelNum == 0:
            self.selectLabel(label)

    def selectLabel(self, label):

        try:
            if self.parent.currentMode == "Eraser":
                self.parent.settingsWidget.drawGroupBox.setMode("Pen")
        except:
            pass

        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                if item.widget() == label:
                    label.selectLabel()
                    self.selectedLabel = label
                    # self.parent.boxPencil.radioButtonDraw.setChecked(True)
                else:
                    item.widget().unselectLabel()

    def selectLabelByNum(self,numLabel):

        try:
            if self.parent.currentMode == "Eraser":
                self.parent.settingsWidget.drawGroupBox.setMode("Pen")
        except:
            pass

        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                if num == numLabel:
                    item.widget().selectLabel()
                    self.selectedLabel = item.widget()
                    # self.parent.boxPencil.radioButtonDraw.setChecked(True)
                else:
                    item.widget().unselectLabel()

    def deleteLabel(self, label):

        numDeleted = None

        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                if item.widget() == label:
                    item.widget().deleteLater()
                    self.layout.removeItem(item)
                    numDeleted = num

        if numDeleted is not None:
            for num in range(self.layout.count()):
                item = self.layout.itemAt(num)
                if item is not None:
                    if item.widget().value > numDeleted:
                        item.widget().value -=1
            self.SignalDeleteLabel.emit(numDeleted)

        self.allowSaveButton()

    def clearScrollArea(self):

        while self.layout.count() != 0:
            item = self.layout.itemAt(0)
            item.widget().deleteLater()
            self.layout.removeItem(item)

    def getCurrentNum(self):

        currentNum = 0
        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                if item.widget() == self.selectedLabel:
                    currentNum = num

        return currentNum

    def getCurrentColor(self):

        color = [255,255,255]
        try:
            for num in range(self.layout.count()):
                item = self.layout.itemAt(num)
                if item is not None:
                    if item.widget() == self.selectedLabel:
                        color = self.selectedLabel.color
        except:
            traceback.print_exc(file=sys.stderr)

        return color

    def getCustomLutDraw(self):

        outputLut = {}
        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                color = item.widget().color
                for c in range(3):
                    if c not in outputLut:
                        outputLut[c] = [0]
                    outputLut[c].append(color[c])

        return outputLut

    def getCustomLutOverlay(self):

        outputLut = {}
        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                widget = item.widget()
                color = widget.color
                value = widget.value
                for c in range(3):
                    if c not in outputLut:
                        outputLut[c] = []
                    while len(outputLut[c])<value:
                        outputLut[c].append(0)
                    if len(outputLut[c])>value:
                        outputLut[c][value] = color[c]*0.7
                    else:
                        outputLut[c].append(color[c]*0.7)

        return outputLut

    def emitSignalColorChanged(self):

        self.SignalColorChanged.emit()
        self.allowSaveButton()

    def allowSaveButton(self):

        try:
            self.parent.settingsWidget.buttonSaveModel.setEnabled(True)
            self.parent.settingsWidget.labelSelectionGroupBox.updateSaveSettingsButton()
        except:
            pass

class GroupBoxColor(qt.QGroupBox):

    SignalSelection = pyqtSignal(qt.QGroupBox)
    SignalDeleteVariable = pyqtSignal(qt.QGroupBox)
    SignalColorChanged = pyqtSignal()

    def __init__(self, num=1, colorNum=1,color=None,name = None,parent=None,value=None):
        qt.QGroupBox.__init__(self)

        self.parent=parent

        self.setStyleSheet('QGroupBox {border: 0px transparent; }')

        if color is None:
            try:
                self.color = vrb.dictColor[colorNum]
            except:
                self.color = [random.randint(vrb.randomColorMin, vrb.randomColorMax),random.randint(vrb.randomColorMin, vrb.randomColorMax),random.randint(vrb.randomColorMin, vrb.randomColorMax)]
        else:
            self.color = color

        self.buttonViewLabel = wgt.PushButtonDoubleImage(vrb.folderImages + "/eye_close.png", vrb.folderImages + "/eye_open.png")
        self.buttonViewLabel.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)
        self.buttonViewLabel.changeActivation()

        if value is None:
            self.value = num-1
        else:
            self.value = int(value)

        if name is None:
            if num == 1:
                self.editableLabelName = wgt.EditableLabel("Background")
            else:
                self.editableLabelName = wgt.EditableLabel("Class "+str(num-1))
        else:
            self.editableLabelName = wgt.EditableLabel(name)

        font = QtGui.QFont()
        font.setPixelSize(vrb.fontSizeMachineLearning*vrb.ratio)

        self.editableLabelName.label.setFixedSize(65*vrb.ratio,20*vrb.ratio)
        self.editableLabelName.label.setFont(font)
        self.editableLabelName.lineEdit.setFixedSize(65*vrb.ratio,20*vrb.ratio)
        self.editableLabelName.lineEdit.setFont(font)

        self.labelColor = qt.QLabel()
        self.labelColor.setFixedSize(20*vrb.ratio, 20*vrb.ratio)
        self.changeButtonColor()

        self.labelCount = qt.QLabel('0')
        self.labelCount.setFixedSize(40 * vrb.ratio, 20 * vrb.ratio)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.buttonViewLabel, 0, 0)
        self.layout.addWidget(self.editableLabelName, 0, 1)
        self.layout.addWidget(self.labelColor, 0, 2 ,Qt.AlignLeft)
        self.layout.addWidget(self.labelCount, 0, 3, Qt.AlignCenter)

        if num > 2:
            self.deleteLabel = wgt.LabelDelete()
            self.deleteLabel.setFixedSize(20* vrb.ratio, 20* vrb.ratio)
            self.deleteLabel.setToolTip(txt.dictToolTips["DeleteLabel"])
            self.layout.addWidget(self.deleteLabel, 0, 3, QtCore.Qt.AlignRight)
            self.deleteLabel.clicked.connect(self.emitSignalDeleteVariable)
        else:
            self.emptyLabel = qt.QLabel()
            self.emptyLabel.setFixedSize(20* vrb.ratio, 20* vrb.ratio)
            self.layout.addWidget(self.emptyLabel, 0, 3, QtCore.Qt.AlignRight)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setHorizontalSpacing(5)
        self.layout.setContentsMargins(5, 0, 5, 0)

        self.buttonViewLabel.clicked.connect(self.viewLabel)
        self.labelColor.mouseDoubleClickEvent = self.getColor
        self.editableLabelName.mouseDoubleClickEvent = self.changeName
        self.labelColor.mousePressEvent = self.emitSelection
        self.editableLabelName.mousePressEvent = self.emitSelection

    def viewLabel(self):

        self.buttonViewLabel.changeActivation()
        if self.parent is not None:
            self.parent.widgetImage.actualizeScene()

    def getColor(self, event):

        color = qt.QColorDialog.getColor()

        if color.isValid():
            self.color = [color.red(), color.green(), color.blue()]
            self.changeButtonColor()

    def changeName(self, event):

        self.editableLabelName.switchVisibleWidget()

    def changeButtonColor(self):

        self.labelColor.setStyleSheet('QLabel {background-color: rgb(' + str(self.color[0]) + ',' + str(self.color[1]) + ',' + str(self.color[2]) + ');}')
        self.SignalColorChanged.emit()

    def emitSelection(self, event):

        self.SignalSelection.emit(self)

    def emitSignalDeleteVariable(self):

        self.SignalDeleteVariable.emit(self)

    def unselectLabel(self):

        font = QtGui.QFont()
        font.setPixelSize(vrb.fontSizeMachineLearning*vrb.ratio)
        font.setBold(False)
        self.editableLabelName.label.setFont(font)
        self.labelCount.setFont(font)
        self.editableLabelName.label.setStyleSheet('QLabel {background-color: transparent;}')

    def selectLabel(self):

        font = QtGui.QFont()
        font.setPixelSize(vrb.fontSizeMachineLearning*vrb.ratio)
        font.setBold(True)
        self.editableLabelName.label.setFont(font)
        self.labelCount.setFont(font)
        self.editableLabelName.label.setStyleSheet('QLabel {background-color: rgb(70,70,70);}')

class GraphicElements(QtCore.QObject):
    SignalComputePreview = pyqtSignal()

    def __init__(self, parent=None):
        super(GraphicElements, self).__init__()

        self.parent = parent
        self.imageViewer = parent.imageViewer

        # self.imageViewer.signalEmitMouse.connect(self.imageViewer.getRoiImage)

    def applyGraphicElement(self, imageViewer):

        if self.parent.selectionRectangle is not None:
            DWfct.addPolygonToScene(imageViewer, self.parent.selectionRectangle, color=[255, 255, 255], showPoints=True, lineSize=2, closed="True")
            DWfct.addPolygonToScene(imageViewer, self.parent.selectionRectangle, color=[6, 115, 186], dashLines=True, showPoints=True, lineSize=2, closed="True", pointMode="circle", pointSize=1)

if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook

    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    foo = MainWidget()

    # foo.loadModel("Model 1")

    # foo.loadModel("Model 11")

    foo.resize(1200, 1000)

    foo.show()

    app.exec_()
