import sys
import os

from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore
from PyQt5.QtCore import pyqtSignal, Qt, QCoreApplication

import PyIPSDK
import PyIPSDK.IPSDKIPLUtility as util
import PyIPSDK.IPSDKIPLIntensityTransform as itrans
import PyIPSDK.IPSDKIPLShapeAnalysis as shapeanalysis
import PyIPSDK.IPSDKIPLColor as colorIP

import WidgetTypes
import UsefullVariables as vrb
import UsefullWidgets as wgt
import UsefullTexts as txt
import UsefullFunctions as fct
import DatabaseFunction as Dfct

import xml.etree.ElementTree as xmlet
import XMLtoInfoSet as xml2IS

# from sklearn.externals import joblib
import joblib

import numpy as np
import time
import traceback

from WidgetLabelImage import WidgetLabelImage
from ImageViewer import WidgetImage
from ShapeAnalysisWidget import MeasureInfoSetWidget
from ShapeAnalysisDisplayer import measureIterator,measureIteratorByType

import random
# random.seed(1)
import webbrowser

from sklearn.ensemble import RandomForestClassifier

from LutsWidget import lutJet

class MainWidget(qt.QWidget):

    def __init__(self):
        qt.QWidget.__init__(self)

        self.layout = qt.QGridLayout()

        self.moduleType = 'Label Classification'
        self.graphicElements = None
        self.currentLabel = None
        self.currentXmlElement = None

        self.model = None

        self.posX = None
        self.posY = None

        self.currentMode = "Pen"
        self.liveUpdate = True

        self.groupBoxProcessing = wgt.GroupBoxProcessing()
        self.groupBoxProcessing.buttonStop.setFixedWidth(0)

        self.xmlElement = xmlet.Element('IPSDK')
        allElements = Dfct.SubElement(self.xmlElement, 'AllElements')

        self.widgetLabelImage = WidgetLabelImage(xmlElement=allElements, module=self)
        self.widgetImage = WidgetImage(standAlone=True, module=self, parent=self)
        self.imageViewer = self.widgetImage.imageViewer
        self.settingsWidget = SettingsWidget(parent=self)
        self.labelScrollArea = self.settingsWidget.labelSelectionGroupBox.labelScrollArea
        self.graphicElements = GraphicElements(parent=self)

        self.layout.addWidget(self.widgetLabelImage, 0, 0)
        self.layout.addWidget(self.widgetImage, 0, 1)
        self.layout.addWidget(self.groupBoxProcessing, 0, 1)
        self.layout.addWidget(self.settingsWidget, 0, 2)

        self.groupBoxProcessing.setVisible(False)

        self.layout.setContentsMargins(5, 5, 5, 5)
        self.layout.setSizeConstraint(1)

        self.setLayout(self.layout)

        self.imageViewer.signalNewImage.connect(self.addLabelsFromUrls)
        self.imageViewer.signalEmitMouse.connect(self.graphicElements.applySignalImageViewer)

        self.widgetLabelImage.signalDoubleClickLabel.connect(self.changeCurrentXmlElement)
        self.widgetLabelImage.signalLabelDeletePosition.connect(self.labelDeleted)

        self.widgetImage.lutInformations.initLut(lutJet["Array"],vertical=False)

        self.labelScrollArea.SignalDeleteLabel.connect(self.stampLabelDeleted)
        self.labelScrollArea.SignalAddLabel.connect(self.changeLut)
        self.labelScrollArea.SignalColorChanged.connect(self.changeLut)

        self.settingsWidget.measuresInfoSet.SignalInfoSetChanged.connect(self.reComputeFeatures)

        self.settingsWidget.buttonViewFeatures.clicked.connect(self.viewFeatures)
        self.settingsWidget.buttonSaveModel.clicked.connect(self.saveModel)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.resize(1200 * vrb.ratio, 900 * vrb.ratio)
        self.setMinimumSize(600 * vrb.ratio, 600 * vrb.ratio)

    def resizeEvent(self, event):

        self.widgetLabelImage.setFixedSize(200*vrb.ratio,self.height())
        self.widgetImage.setFixedSize(max(0,self.width()-470*vrb.ratio),self.height())
        self.settingsWidget.setFixedSize(250*vrb.ratio,self.height())

    def closeEvent(self, a0: QtGui.QCloseEvent):

        try:
            vrb.mainWindow.currentModuleMachineLearning = None
        except:
            pass

    def saveModel(self):

        modelName = self.settingsWidget.lineEditModelName.text()
        if modelName == "":
            modelName = self.settingsWidget.lineEditModelName.placeholderText()
        filePath = vrb.folderShapeClassification + "/" + modelName
        if os.path.exists(filePath):
            fct.clearDirectory(filePath)
        else:
            os.makedirs(filePath)

        labelClassesElement = Dfct.SubElement(self.xmlElement,"LabelClasses")
        numberLabelElement = Dfct.SubElement(labelClassesElement,"NumberLabels")
        nbLabel = self.settingsWidget.labelSelectionGroupBox.labelScrollArea.layout.count()
        numberLabelElement.text = str(nbLabel)
        for numLabel in range(nbLabel):
            labelItem = self.settingsWidget.labelSelectionGroupBox.labelScrollArea.layout.itemAt(numLabel)
            if labelItem is not None:
                labelWidget = labelItem.widget()
                labelElement = Dfct.SubElement(labelClassesElement,"Label_"+str(numLabel))
                colorElement = Dfct.SubElement(labelElement,"Color")
                colorElement.text = str(labelWidget.color[0]) + "," + str(labelWidget.color[1]) + "," + str(labelWidget.color[2])
                nameElement = Dfct.SubElement(labelElement,"Name")
                nameElement.text = labelWidget.editableLabelName.label.text()

        Dfct.saveXmlElement(self.xmlElement,filePath+"/Settings.mho")
        Dfct.saveXmlElement(self.settingsWidget.measuresInfoSet.comboBoxInfoSet.currentData()[0], filePath + "/Model_Infoset.mho")

        for i in range(self.widgetLabelImage.layout.count()):
            item = self.widgetLabelImage.layout.itemAt(i)
            if item is not None:
                label = item.widget()

                name = label.labelName.text()
                folderImageName = filePath+"/"+name
                if os.path.exists(folderImageName):
                    fct.clearDirectory(folderImageName)
                else:
                    os.makedirs(folderImageName)

                PyIPSDK.saveTiffImageFile(folderImageName+"/Image.tif",label.image)
                if label.imageGrey is not None:
                    PyIPSDK.saveTiffImageFile(folderImageName + "/ImageGrey.tif", label.imageGrey)

        if self.model is not None:
            joblib.dump(self.model, filePath + "/Model")

        try:
            # vrb.mainWindow.actualizeFunctionMachineLearning()
            vrb.mainWindow.updateCurrentFunction()
        except:
            pass
        self.settingsWidget.buttonSaveModel.setEnabled(False)

    def loadModel(self,modelName):

        self.settingsWidget.lineEditModelName.setPlaceholderText(modelName)

        folder = vrb.folderShapeClassification + "/" + modelName

        filePath = os.path.join(folder+"/Settings.mho")
        file = xmlet.parse(filePath)
        self.xmlElement = file.getroot()

        labelClassesElement = Dfct.SubElement(self.xmlElement, "LabelClasses")
        numberLabelElement = Dfct.SubElement(labelClassesElement, "NumberLabels")
        nbLabel = int(numberLabelElement.text)
        for numLabel in range(nbLabel):
            labelElement = Dfct.SubElement(labelClassesElement, "Label_"+str(numLabel))
            colorText = Dfct.childText(labelElement, "Color").split(",")
            color = [int(colorText[0]),int(colorText[1]),int(colorText[2])]
            name = Dfct.childText(labelElement, "Name")

            if numLabel <2:
                labelItem = self.settingsWidget.labelSelectionGroupBox.labelScrollArea.layout.itemAt(numLabel)
                if labelItem is not None:
                    labelWidget = labelItem.widget()
                    labelWidget.color = color
                    labelWidget.changeButtonColor()
                    labelWidget.editableLabelName.label.setText(name)

            else:
                self.settingsWidget.labelSelectionGroupBox.labelScrollArea.addLabelWithName(color=color,name=name)
        modelInfosetFile = xmlet.parse(vrb.folderShapeClassification + "/" + modelName + "/" + "Model_Infoset.mho")
        modelInfosetElement = modelInfosetFile.getroot()
        self.settingsWidget.measuresInfoSet.dimension = modelInfosetElement.get('Dimension')

        self.settingsWidget.measuresInfoSet.additionalFolder = folder
        self.settingsWidget.measuresInfoSet.fillComboBox()

        allElements = Dfct.SubElement(self.xmlElement, 'AllElements')
        self.widgetLabelImage.xmlElement = allElements
        # self.widgetLabelImage = WidgetLabelImage(xmlElement=allElements, module=self)

        try:
            self.model = joblib.load(folder+"/Model")
        except:
            self.model = None

        listImages = []
        allElements = Dfct.SubElement(self.xmlElement, 'AllElements')
        for child in allElements:
            if child.tag == "Image":
                nameImage = Dfct.childText(child,"Name")
                listImages.append(nameImage)

        for nameImage in listImages:

            if os.path.exists(folder + "/" + nameImage):

                image = PyIPSDK.loadTiffImageFile(folder + "/" + nameImage + "/Image.tif")
                label, name = self.widgetLabelImage.addNewImage(nameImage, image, xmlElement=child)

                self.changeCurrentXmlElement(label)

                try:
                    imageGrey = PyIPSDK.loadTiffImageFile(folder + "/" + nameImage + "/imageGrey.tif")
                    self.addImage(imageGrey, nameImage)
                except:
                    pass

                self.changeCurrentXmlElement(label)
        #
        # self.featuresWidget.actualizeComboBoxLoading()

    def loadModelRef(self,name = ""):

        if name != "":
            self.settingsWidget.lineEditModelName.setPlaceholderText(name)


        self.featuresWidget.actualizeComboBoxLoading()

    def viewFeatures(self):

        if self.settingsWidget.buttonViewFeatures.activate == True:
            self.settingsWidget.buttonViewFeatures.changeActivation()
            self.changeCurrentXmlElement(self.currentLabel)
            self.widgetImage.sliderOpacity.slider.setValue(50)
            self.displayLut(False)

    def displayLut(self,boolValue):

        if boolValue:
            index = self.currentLabel.currentFeaturesIndex
            measure = self.currentLabel.features[index]
            minMeasure,maxMeasure = min(measure[1:]),max(measure[1:])
            self.widgetImage.lutInformations.labelMin.label.setText(fct.numberCalibration(minMeasure))
            self.widgetImage.lutInformations.labelMax.setText(fct.numberCalibration(maxMeasure))

        self.widgetImage.lutInformations.setVisible(boolValue)

    def evaluateModel(self,changeModel = True):

        try:
            start = time.time()

            self.viewFeatures()
            self.computeFeatures(messageFeatures=False)

            trainingData = None

            for i in range(self.widgetLabelImage.layout.count()):
                item = self.widgetLabelImage.layout.itemAt(i)
                if item is not None:

                    label = item.widget()

                    graphicElement = Dfct.SubElement(label.xmlElement,"GraphicElement")
                    for child in graphicElement:
                        labelValue = int(Dfct.childText(child, "LabelValue"))
                        classValue = [int(Dfct.childText(child, "Class"))]
                        data = np.concatenate((label.features[:, labelValue], classValue))
                        if trainingData is None:
                            trainingData = data
                        else:
                            trainingData = np.vstack((trainingData, data))

            trainingData = fct.correctArrayValues(trainingData)

            if self.model is None or changeModel:
                try:
                    nbTrees = int(self.settingsWidget.lineEditTreeNumber.text())
                except:
                    nbTrees = 100
                self.model = self.computeModel(trainingData, nbTrees)

            if self.model is not None:
                self.settingsWidget.listWidgetFeatures.fillListWidget()
                dataToPredict = fct.correctArrayValues(self.currentLabel.features.T)
                vectorPrediction = self.model.predict(dataToPredict)

                vectorPredictionProba = self.model.predict_proba(dataToPredict)

                vectorPrediction[0] = 0

                self.currentLabel.vectorPrediction = vectorPrediction

                # lut = PyIPSDK.createIntensityLUT(0, 1, vectPred)
                # outImage = itrans.lutTransform2dImg(self.currentLabel.image, lut)

                self.imageViewer.getRoiImage()

            # print("End apply model", time.time() - start)

        except:
            pass
            # traceback.print_exc(file=sys.stderr)

        self.settingsWidget.buttonSaveModel.setEnabled(True)

    def computeModel(self, data, nTrees):

        model = RandomForestClassifier(n_estimators=nTrees)
        model.fit(data[:, :-1], data[:, -1])

        self.settingsWidget.buttonSaveModel.setEnabled(True)

        return model

    def reComputeFeatures(self):

        self.computeFeatures(reCompute = True)
        self.evaluateModel()

    def computeFeatures(self,reCompute = False,messageFeatures = True):

        colorList = ["Red", "Green", "Blue"]

        if messageFeatures:
            self.groupBoxProcessing.setText('Computing features')
            self.toggleGroupBoxProcessing(True)

        try:
            xmlElement = self.settingsWidget.measuresInfoSet.comboBoxInfoSet.currentData()[0]

            if self.currentLabel is not None:
                if self.currentLabel.image.getSizeZ() == 1:
                    dimension = "2D"
                else:
                    dimension = "3D"

                for i in range(self.widgetLabelImage.layout.count()):
                    item = self.widgetLabelImage.layout.itemAt(i)
                    if item is not None:
                        label = item.widget()
                        if reCompute or label.features is None:
                            if label.imageGrey is not None:
                                imageGrey = label.imageGrey
                            else:
                                imageGrey = label.image

                            measures = xmlInfosetToMeasure(xmlElement, imageGrey, label.image, dimension=dimension)
                            features = []
                            listNamesFeatures = []
                            allMeasures = measureIterator(xmlElement)
                            for callName,userName in allMeasures:
                                try:
                                    valuesInfosetColl = measures.getMeasure(callName).getMeasureResult()
                                    nbColorPlan = valuesInfosetColl.getNbColorPlans()

                                    for colorPlan in range(nbColorPlan):
                                        features.append(valuesInfosetColl.getColl(colorPlan))
                                        if nbColorPlan == 1:
                                            listNamesFeatures.append(userName)
                                        elif nbColorPlan == 3:
                                            listNamesFeatures.append(userName + " (" + colorList[colorPlan] + ")")
                                        else:
                                            listNamesFeatures.append(userName + " (Channel "+str(colorPlan)+")")
                                except:
                                    pass


                            features = np.asarray(features)
                            label.features = features
                            label.listNamesFeatures = listNamesFeatures
        except:
            traceback.print_exc(file=sys.stderr)

        self.settingsWidget.listWidgetFeatures.fillListWidget()

        if messageFeatures:
            self.toggleGroupBoxProcessing(False)

    def resizeEvent(self, event):

        widgetLabelImageWidth = 200*vrb.ratio
        settingsWidgetWidth = 250*vrb.ratio

        self.widgetLabelImage.setFixedSize(widgetLabelImageWidth, self.height())
        self.settingsWidget.setFixedSize(settingsWidgetWidth, self.height())
        self.widgetImage.setFixedSize(max(0, self.width()-(widgetLabelImageWidth + settingsWidgetWidth + 20*vrb.ratio)), self.height())

    def countLabel(self):

        if self.currentXmlElement is not None:
            count = [0 for i in range(self.labelScrollArea.layout.count())]

            for child in self.currentXmlElement:
                if child.tag == 'GraphicElement':
                    for subChild in child:
                        nClass = int(Dfct.childText(subChild, 'Class'))
                        count[nClass - 1] += 1

            for num in range(self.labelScrollArea.layout.count()):
                item = self.labelScrollArea.layout.itemAt(num)
                if item is not None:
                    label = item.widget()
                    label.labelCount.setText(' ' + str(count[num]) + ' ')

        else:
            for num in range(self.labelScrollArea.layout.count()):
                item = self.labelScrollArea.layout.itemAt(num)
                if item is not None:
                    label = item.widget()
                    label.labelCount.setText(' 0 ')

    def stampLabelDeleted(self, num):
        # self.resetStates()

        for allElements in self.xmlElement:
            for image in allElements:
                for graphicElement in image:
                    if graphicElement.tag == 'GraphicElement':

                        allDeleted = False
                        while allDeleted is False:
                            allDeleted = True
                            for child in graphicElement:
                                oldClass = int(Dfct.childText(child, 'Class'))
                                if oldClass == num + 1:
                                    Dfct.removeElement(graphicElement, child)
                                    allDeleted = False

                        for child in graphicElement:
                            oldClass = int(Dfct.childText(child, 'Class'))
                            if oldClass > num + 1:
                                for subChild in child:
                                    if subChild.tag == 'Class':
                                        subChild.text = str(oldClass - 1)

        self.imageViewer.getRoiImage()
        self.countLabel()

        #     if self.currentImage.outImage is not None:
        #         self.process()

    def changeLut(self):

        #self.customLutOverlay = self.labelScrollArea.getCustomLutOverlay()
        self.imageViewer.getRoiImage()

    def addLabelsFromUrls(self, urls):

        for i in range(len(urls)):
            try:
                url = str(urls[i].toLocalFile())
            except:
                url = urls[i]
            urlSplit = url.split('.')
            extension = urlSplit[len(urlSplit) - 1]
            if extension in vrb.extensionImages:
                try:
                    image, name = self.urlToImage(url)
                    self.addImage(image, name)
                except:
                    traceback.print_exc(file=sys.stderr)
                    pass

    def urlToImage(self, url):

        name = os.path.basename(url)
        self.groupBoxProcessing.setText('Adding image\n' + name)
        self.toggleGroupBoxProcessing(True)
        image = fct.readImage(url, name=name)

        return image,name

    def addImage(self, image,name):

        self.groupBoxProcessing.setText('Adding image\n' + name)
        self.toggleGroupBoxProcessing(True)

        # name = os.path.basename(url)
        # image = fct.readImage(url, name=name)

        item = self.widgetLabelImage.layout.itemAt(0)
        if image.getBufferType() == PyIPSDK.eIBT_Label16:
            if item is None:
                #label, name = self.widgetLabelImage.addNewImage(name, image)
                label, name = self.widgetLabelImage.addNewImage(name, image, allowDuplicate=True)
                if image.getSizeZ() == 1:
                    self.settingsWidget.measuresInfoSet.dimension = "2D"
                else:
                    self.settingsWidget.measuresInfoSet.dimension = "3D"
                self.settingsWidget.measuresInfoSet.fillComboBox()
                self.changeCurrentXmlElement(label)
            else:
                imageRef = item.widget().image
                if (imageRef.getSizeZ() == 1 and image.getSizeZ() == 1) or (imageRef.getSizeZ() > 1 and image.getSizeZ() > 1):
                    # label, name = self.widgetLabelImage.addNewImage(name, image)
                    label, name = self.widgetLabelImage.addNewImage(name, image, allowDuplicate=True)
                    self.changeCurrentXmlElement(label)
                else:
                    self.messageBox = wgt.MessageBox("You can't add this image\n All images must have the same geometry (2D or 3D)", '', buttons=[qt.QMessageBox.Ok], windowTitle="Error image")
                    self.messageBox.exec()
        else:
            if item is None:
                self.messageBox = wgt.MessageBox("You need to add a label image first", '', buttons=[qt.QMessageBox.Ok], windowTitle="Error image")
                self.messageBox.exec()
            else:
                if (self.currentLabel.image.getSizeX() == image.getSizeX() and self.currentLabel.image.getSizeY() == image.getSizeY() and self.currentLabel.image.getSizeZ() == image.getSizeZ()):
                    #Case Color
                    self.currentLabel.imageGrey = image
                    if image.getSizeC()>1:
                        self.currentLabel.customContextMenu.addAction('Channel selector', self.currentLabel.modifyColorChannels)
                        self.currentLabel.widgetMultiSpectral = wgt.WidgetMultiSpectral(image=image, parent=self.currentLabel)
                        self.currentLabel.widgetMultiSpectral.SignalMultiSpectralChanged.connect(self.currentLabel.labelImageMiniature.changeImage)
                    self.currentLabel.valueMinGrey,self.currentLabel.valueMaxGrey = fct.getMinMaxValue(image)
                    self.changeCurrentXmlElement(self.currentLabel)
                    self.computeFeatures(reCompute=True)

                    image = util.copyImg(self.currentLabel.image)
                    imageGrey = util.copyImg(self.currentLabel.imageGrey)

                    self.currentLabel.labelImageMiniature.initImage(image,imageGrey)

                else:
                    self.messageBox = wgt.MessageBox("You can't add this image\n Label image and grey image must have same dimensions", '', buttons=[qt.QMessageBox.Ok], windowTitle="Error image")
                    self.messageBox.exec()

        self.settingsWidget.buttonSaveModel.setEnabled(True)

        self.toggleGroupBoxProcessing(False)
        self.settingsWidget.buttonSaveModel.setEnabled(True)

        #self.computeFeatures(reCompute=True)

    def changeCurrentXmlElement(self, label):

        # try:
        #     self.currentLabel.setStyleSelected(False)
        # except:
        #     pass
        self.currentXmlElement = label.xmlElement
        self.currentLabel = label

        if label is not None:
            self.setWindowTitle('IPSDK Explorer - ' + str(Dfct.childText(label.xmlElement, "Name")))
        else:
            self.setWindowTitle('IPSDK Explorer')

        # self.currentLabel.setStyleSelected(True)
        self.widgetLabelImage.updateStyleSheet()

        if self.currentLabel.imageGrey is None:
            self.widgetImage.changeMode('classic',acutalize=False)
            self.widgetImage.setImageOverlay(None)
        else:
            self.widgetImage.changeMode('overlay',acutalize=False)
            self.widgetImage.setImageOverlay(self.currentLabel.imageGrey)

        self.widgetImage.setImage(self.currentLabel)

        image = util.copyImg(self.currentLabel.image)
        if self.currentLabel.imageGrey is not None:
            imageGrey = util.copyImg(self.currentLabel.imageGrey)
        else:
            imageGrey = None

        self.currentLabel.labelImageMiniature.initImage(image, imageGrey)

        self.countLabel()
        self.evaluateModel(changeModel=False)

    def labelDeleted(self, position):

        self.model = None

        self.setWindowTitle('IPSDK Explorer')
        currentPosition = int(Dfct.childText(self.currentXmlElement, "Position"))

        if currentPosition == -1:
            item = self.widgetLabelImage.layout.itemAt(position)
            if item is not None:
                self.changeCurrentXmlElement(item.widget())
            else:
                if position > 0:
                    item = self.widgetLabelImage.layout.itemAt(position - 1)
                    if item is not None:
                        self.changeCurrentXmlElement(item.widget())
                    else:
                        self.resetScene()
                else:
                    self.resetScene()

        self.model = None
        if self.liveUpdate:
            self.evaluateModel()
        self.settingsWidget.buttonSaveModel.setEnabled(True)

    def toggleGroupBoxProcessing(self,showValue):

        if showValue:
            self.groupBoxProcessing.setVisible(True)
            qt.QApplication.processEvents()
            qt.QApplication.processEvents()
        else:
            self.groupBoxProcessing.setVisible(False)

    def resetScene(self):

        self.currentLabel = None
        self.currentXmlElement = None
        self.imageViewer.scene.clear()
        self.imageViewer.scene.clearSelection()
        self.imageViewer.scene.clearFocus()
        self.widgetImage.setImage(None)
        self.imageViewer.setImageOverlay(None)
        self.imageViewer.setImageProbabilities(None)
        self.image = None
        self.widgetImage.sliderOpacity.setVisible(False)
        self.countLabel()


class GraphicElements(QtCore.QObject):
    SignalComputePreview = pyqtSignal()

    def __init__(self, parent=None):
        super(GraphicElements, self).__init__()

        self.parent = parent
        self.imageViewer = parent.imageViewer

        #self.nbLabel = 0

        self.imageViewer.signalEmitMouse.connect(self.actualizeSceneElement)

    def actualizeSceneElement(self, ddict, imageViewer):

        modelChanged = False
        try:
            if self.parent.currentLabel is not None:
                if self.imageViewer.image is not None and ddict["Button"] == 1 and ddict["Event"] == "MousePressed":
                    value = None
                    if self.parent.currentLabel.image.getSizeZ() == 1:
                        value = self.parent.currentLabel.image.array[int(ddict["Position"][1]), int(ddict["Position"][0])]
                    else:
                        if self.parent.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonX.isChecked():
                            positionSlider = self.parent.widgetImage.imageViewerStandAlone.sliderAxis.sliderX.slider.value()
                            value = self.parent.currentLabel.image.array[int(ddict["Position"][1]), int(ddict["Position"][0]),positionSlider]
                        if self.parent.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonY.isChecked():
                            positionSlider = self.parent.widgetImage.imageViewerStandAlone.sliderAxis.sliderY.slider.value()
                            value = self.parent.currentLabel.image.array[int(ddict["Position"][1]),positionSlider, int(ddict["Position"][0])]
                        if self.parent.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonZ.isChecked():
                            positionSlider = self.parent.widgetImage.imageViewerStandAlone.sliderAxis.sliderZ.slider.value()
                            value = self.parent.currentLabel.image.array[positionSlider,int(ddict["Position"][1]), int(ddict["Position"][0])]

                    if self.parent.currentMode == 'Pen':
                        if value > 0 and self.parent.labelScrollArea.getCurrentNum() is not None:
                            if self.parent.labelScrollArea.getCurrentNum() > 0:
                                self.addLabelToXml(ddict, value)
                                modelChanged = True
                    else:
                        labelValue = value
                        for child in self.parent.currentXmlElement:
                            if child.tag == 'GraphicElement':
                                for subChild in child:
                                    if int(Dfct.childText(subChild, 'LabelValue')) == labelValue:
                                        Dfct.removeElement(child, subChild)
                                        modelChanged = True

                    self.imageViewer.getRoiImage()
                    self.parent.countLabel()
                    if self.parent.liveUpdate and modelChanged:
                        self.parent.evaluateModel()
                    # self.parent.update()

                    self.parent.settingsWidget.buttonSaveModel.setEnabled(True)
        except:
            traceback.print_exc(file=sys.stderr)


    def addLabelToXml(self, ddict, value):

        if self.parent.currentLabel.image.getSizeZ() == 1:
            posX = int(ddict["Position"][0])
            posY = int(ddict["Position"][1])
            posZ = 0
        else:
            if self.parent.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonX.isChecked():
                posX = self.parent.widgetImage.imageViewerStandAlone.sliderAxis.sliderX.slider.value()
                posY = int(ddict["Position"][0])
                posZ = int(ddict["Position"][1])
            if self.parent.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonY.isChecked():
                posX = int(ddict["Position"][0])
                posY = self.parent.widgetImage.imageViewerStandAlone.sliderAxis.sliderY.slider.value()
                posZ = int(ddict["Position"][1])
            if self.parent.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonZ.isChecked():
                posX = int(ddict["Position"][0])
                posY = int(ddict["Position"][1])
                posZ = self.parent.widgetImage.imageViewerStandAlone.sliderAxis.sliderZ.slider.value()

        labelAlreadyMarked = False

        graphicElement = Dfct.SubElement(self.parent.currentXmlElement,"GraphicElement")

        for child in graphicElement:
            if int(Dfct.childText(child, 'LabelValue')) == value:
                labelAlreadyMarked = True
                for subChild in child:
                    if subChild.tag == 'PosX':
                        subChild.text = str(posX)
                    if subChild.tag == 'PosY':
                        subChild.text = str(posY)
                    if subChild.tag == 'PosZ':
                        subChild.text = str(posZ)
                    if subChild.tag == 'Class':
                        subChild.text = str(self.parent.labelScrollArea.getCurrentNum())

        if labelAlreadyMarked is False:

            #self.nbLabel += 1
            numLabel = 1
            while Dfct.containsTag(graphicElement,'Label'+str(numLabel)):
                numLabel+=1
            label = Dfct.SubElement(graphicElement, 'Label' + str(numLabel))

            Dfct.SubElement(label, 'PosX').text = str(posX)
            Dfct.SubElement(label, 'PosY').text = str(posY)
            Dfct.SubElement(label, 'PosZ').text = str(posZ)

            labelClass = Dfct.SubElement(label, 'Class')
            labelClass.text = str(self.parent.labelScrollArea.getCurrentNum())

            labelValue = Dfct.SubElement(label, "LabelValue")
            labelValue.text = str(value)

    def applySignalImageViewer(self,ddict, imageViewer):

        if self.parent.currentXmlElement is not None:
            for child in self.parent.currentXmlElement:
                if child.tag == 'GraphicElement':
                    for subChild in child:
                        posX = None
                        posY = None
                        if self.parent.currentLabel.image.getSizeZ() == 1:
                            posX = float(Dfct.childText(subChild, 'PosX'))
                            posY = float(Dfct.childText(subChild, 'PosY'))
                        else:
                            if self.parent.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonX.isChecked() and int(Dfct.childText(subChild, 'PosX'))==self.parent.widgetImage.imageViewerStandAlone.sliderAxis.sliderX.slider.value():
                                posX = float(Dfct.childText(subChild, 'PosY'))
                                posY = float(Dfct.childText(subChild, 'PosZ'))
                            if self.parent.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonY.isChecked() and int(Dfct.childText(subChild, 'PosY'))==self.parent.widgetImage.imageViewerStandAlone.sliderAxis.sliderY.slider.value():
                                posX = float(Dfct.childText(subChild, 'PosX'))
                                posY = float(Dfct.childText(subChild, 'PosZ'))
                            if self.parent.widgetImage.imageViewerStandAlone.sliderAxis.radioButtonZ.isChecked() and int(Dfct.childText(subChild, 'PosZ'))==self.parent.widgetImage.imageViewerStandAlone.sliderAxis.sliderZ.slider.value():
                                posX = float(Dfct.childText(subChild, 'PosX'))
                                posY = float(Dfct.childText(subChild, 'PosY'))

                        if posX is not None and posY is not None:
                            labelClass = Dfct.childText(subChild, 'Class')
                            addLabelToScene(imageViewer, posX, posY, labelClass)

class ListWidgetFeatures(qt.QListWidget):

    def __init__(self,parent=None):
        qt.QListWidget.__init__(self)

        self.parent = parent
        self.itemSelectionChanged.connect(self.loadCurrentItem)
        self.itemClicked.connect(self.itemClickedFunction)

    def fillListWidget(self):

        try:

            self.clear()
            label = self.parent.currentLabel
            model = self.parent.model

            if label is not None and label.listNamesFeatures is not None:

                if model is not None:

                    featuresImportance = model.feature_importances_
                    ind = np.argsort(featuresImportance)
                    ind = ind[::-1]

                else:
                    ind = range(len(label.listNamesFeatures))

                for index in ind:

                    newItem = qt.QListWidgetItem()
                    newItem.setText(label.listNamesFeatures[index])
                    newItem.index = index

                    if model is not None:
                        widget = qt.QLabel(fct.numberCalibration(featuresImportance[index] * 100) + " %")
                    else:
                        widget = qt.QLabel("0 %")
                    widget.setAlignment(Qt.AlignRight)

                    self.addItem(newItem)
                    self.setItemWidget(newItem, widget)

        except:
            traceback.print_exc(file=sys.stderr)

    def itemClickedFunction(self):

        if self.parent.settingsWidget.buttonViewFeatures.activate == False:
            self.parent.widgetImage.sliderOpacity.slider.setValue(100)
        self.parent.settingsWidget.buttonViewFeatures.setActivation(True)
        self.loadCurrentItem()

    def loadCurrentItem(self):

        try:
            label = self.parent.currentLabel
            label.currentFeaturesIndex = self.currentItem().index

            self.parent.imageViewer.getRoiImage()
            self.parent.displayLut(True)

        except:
            traceback.print_exc(file=sys.stderr)


class SettingsWidget(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.parent = parent

        self.pushButtonDocumentation = wgt.PushButtonImage(margins=1, filename=vrb.folderImages + '/Interrogation.png')
        self.pushButtonDocumentation.setStyleSheet("background-color : transparent; border :0px")
        self.pushButtonDocumentation.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.labelSelectionGroupBox = LabelSelectionGroupBox(parent=self.parent)
        self.drawGroupBox = DrawGroupBox(parent=self.parent)

        self.labelFeatures = qt.QLabel("Features")
        font = QtGui.QFont()
        font.setBold(True)
        font.setPointSize(10)
        self.labelFeatures.setFont(font)
        self.labelFeatures.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.labelFeatures.setFixedWidth(70 * vrb.ratio)

        self.measuresInfoSet = MeasureInfoSetWidget(None,type=WidgetTypes.InputType.MEASURE_INFO_SET,hasConstraint=True)

        self.groupBoxInfoSet = qt.QGroupBox()
        self.layoutInfoSet = qt.QGridLayout()
        self.layoutInfoSet.addWidget(self.labelFeatures, 0, 0)
        self.layoutInfoSet.addWidget(self.measuresInfoSet, 0, 1, Qt.AlignLeft)
        self.groupBoxInfoSet.setLayout(self.layoutInfoSet)
        self.layoutInfoSet.setContentsMargins(0, 0, 0, 0)
        self.layoutInfoSet.setSizeConstraint(1)

        self.buttonSettings = wgt.PushButtonImage(vrb.folderImages + "/Edit_3.png", margins=0)
        self.buttonSettings.setToolTip("Modify features")
        self.buttonSettings.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.labelTreeNumber = qt.QLabel("Number of trees")
        self.labelTreeNumber.setFont(font)
        self.labelTreeNumber.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.labelTreeNumber.setFixedWidth(130 * vrb.ratio)
        self.lineEditTreeNumber = wgt.NumberLineEdit()
        self.lineEditTreeNumber.setPlaceholderText("100")
        self.lineEditTreeNumber.setFixedWidth(50 * vrb.ratio)

        self.groupBoxTreeNumber = qt.QGroupBox()
        self.layoutTreeNumber = qt.QGridLayout()
        self.layoutTreeNumber.addWidget(self.labelTreeNumber, 0, 0)
        self.layoutTreeNumber.addWidget(self.lineEditTreeNumber, 0, 1, Qt.AlignLeft)
        self.groupBoxTreeNumber.setLayout(self.layoutTreeNumber)
        self.layoutTreeNumber.setContentsMargins(0, 0, 0, 0)
        self.layoutTreeNumber.setSizeConstraint(1)

        self.labelLiveUpdate = qt.QLabel("Live update")
        self.labelLiveUpdate.setFont(font)
        self.labelLiveUpdate.setAlignment(Qt.AlignCenter)
        self.labelLiveUpdate.setFixedWidth(90 * vrb.ratio)
        self.labelLiveUpdate.setStyleSheet('QLabel {color: rgb(52, 152, 219);background-color: rgb(70,70,70);}')

        self.buttonLiveUpdate = wgt.PushButtonDoubleImage(vrb.folderImages + "/Pause.png", vrb.folderImages + "/Lecture.png")
        self.buttonLiveUpdate.setToolTip("Stop live update")
        self.buttonLiveUpdate.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.buttonRefresh = wgt.PushButtonImage(vrb.folderImages + "/Refresh_White.png", margins=5)
        self.buttonRefresh.setToolTip("Evaluate model")
        self.buttonRefresh.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.emptyLabel1 = qt.QLabel()
        self.emptyLabel1.setFixedHeight(25 * vrb.ratio)

        self.groupBoxFeatures = qt.QGroupBox()
        self.layoutFeatures = qt.QGridLayout()
        self.layoutFeatures.addWidget(self.groupBoxInfoSet, 0, 0)
        self.layoutFeatures.addWidget(self.groupBoxTreeNumber,1,0)
        self.groupBoxFeatures.setLayout(self.layoutFeatures)
        self.layoutFeatures.setContentsMargins(20, 0, 0, 0)
        self.layoutFeatures.setSizeConstraint(1)

        self.emptyLabel2 = qt.QLabel()
        self.emptyLabel2.setFixedHeight(25 * vrb.ratio)

        self.groupBoxLiveUpdate = qt.QGroupBox()
        self.layoutLiveUpdate = qt.QGridLayout()
        self.layoutLiveUpdate.addWidget(self.labelLiveUpdate, 0, 0)
        self.layoutLiveUpdate.addWidget(self.buttonLiveUpdate, 0, 1)
        self.layoutLiveUpdate.addWidget(self.buttonRefresh, 0, 2)
        self.layoutLiveUpdate.setAlignment(Qt.AlignLeft)
        self.buttonRefresh.setVisible(False)
        self.groupBoxLiveUpdate.setLayout(self.layoutLiveUpdate)
        self.groupBoxLiveUpdate.setFixedHeight(30 * vrb.ratio)
        self.layoutLiveUpdate.setContentsMargins(20, 0, 0, 0)
        self.layoutLiveUpdate.setSizeConstraint(1)

        self.emptyLabel3 = qt.QLabel()
        self.emptyLabel3.setFixedHeight(25 * vrb.ratio)

        self.groupBoxLabelListFeatures = qt.QGroupBox()
        self.groupBoxLabelListFeatures.setFixedHeight(25 * vrb.ratio)
        layoutLabelListFeatures = qt.QGridLayout()
        self.groupBoxLabelListFeatures.setLayout(layoutLabelListFeatures)
        layoutLabelListFeatures.setContentsMargins(0, 0, 0, 0)
        layoutLabelListFeatures.setSizeConstraint(1)
        self.groupBoxLabelListFeatures.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.labelListOfFeatures = qt.QLabel("List of features (sorted by importance)")
        font = QtGui.QFont()
        font.setPointSize(9)
        self.labelListOfFeatures.setFont(font)
        self.labelListOfFeatures.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.labelListOfFeatures.setFixedHeight(25 * vrb.ratio)

        self.buttonViewFeatures = wgt.PushButtonDoubleImage(vrb.folderImages + "/eye_close.png", vrb.folderImages + "/eye_open.png")
        self.buttonViewFeatures.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        layoutLabelListFeatures.addWidget(self.labelListOfFeatures, 0, 0)
        layoutLabelListFeatures.addWidget(self.buttonViewFeatures, 0, 1)

        self.groupBoxRemoveFeatures = qt.QGroupBox()
        self.groupBoxRemoveFeatures.setFixedHeight(25 * vrb.ratio)
        layoutRemoveFeatures = qt.QGridLayout()
        self.groupBoxRemoveFeatures.setLayout(layoutRemoveFeatures)
        layoutRemoveFeatures.setContentsMargins(0, 0, 0, 0)
        layoutRemoveFeatures.setSizeConstraint(1)
        self.groupBoxRemoveFeatures.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.labelRemoveFeatures = qt.QLabel("Remove features below")
        font = QtGui.QFont()
        font.setPointSize(9)
        self.labelRemoveFeatures.setFont(font)
        # self.labelRemoveFeatures.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.lineEditPercentage = wgt.NumberLineEdit(constraint='Real')
        self.lineEditPercentage.setText("0")
        self.lineEditPercentage.setPlaceholderText("0")
        self.lineEditPercentage.setFixedWidth(35 * vrb.ratio)
        self.labelPercentage = qt.QLabel("%")
        self.labelPercentage.setFont(font)
        # self.labelPercentage.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.buttonRemoveFeatures = wgt.PushButtonImage(vrb.folderImages + "/Trash2.png", margins=5)
        self.buttonRemoveFeatures.setToolTip("Remove Features")
        self.buttonRemoveFeatures.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        layoutRemoveFeatures.addWidget(self.labelRemoveFeatures, 0, 0)
        layoutRemoveFeatures.addWidget(self.lineEditPercentage, 0, 1)
        layoutRemoveFeatures.addWidget(self.labelPercentage, 0, 2)
        layoutRemoveFeatures.addWidget(self.buttonRemoveFeatures, 0, 3)

        self.groupSaveModel = qt.QGroupBox()
        self.groupSaveModel.setFixedHeight(25 * vrb.ratio)
        layoutSaveModel = qt.QGridLayout()
        self.groupSaveModel.setLayout(layoutSaveModel)
        layoutSaveModel.setContentsMargins(0, 0, 0, 0)
        layoutSaveModel.setSizeConstraint(1)
        self.groupSaveModel.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.labelModelName = qt.QLabel("Model name")
        font = QtGui.QFont()
        font.setPointSize(9)
        self.labelModelName.setFont(font)
        self.labelModelName.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.lineEditModelName = qt.QLineEdit()
        self.lineEditModelName.setFixedWidth(140 * vrb.ratio)
        modelName = fct.getModelName(vrb.folderShapeClassification)
        self.lineEditModelName.setPlaceholderText(modelName)
        self.buttonSaveModel = wgt.PushButtonImage(vrb.folderImages + "/Save.png", margins=2)
        self.buttonSaveModel.setToolTip("Save model")
        self.buttonSaveModel.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        layoutSaveModel.addWidget(self.labelModelName, 0, 0)
        layoutSaveModel.addWidget(self.lineEditModelName, 0, 1)
        layoutSaveModel.addWidget(self.buttonSaveModel, 0, 2)

        self.listWidgetFeatures = ListWidgetFeatures(parent=self.parent)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.pushButtonDocumentation, 0, 0, Qt.AlignRight)
        self.layout.addWidget(self.labelSelectionGroupBox, 1, 0)
        self.layout.addWidget(self.drawGroupBox, 2, 0)
        self.layout.addWidget(self.emptyLabel1, 3, 0)
        self.layout.addWidget(self.groupBoxFeatures, 4, 0)
        self.layout.addWidget(self.emptyLabel2, 5, 0)
        self.layout.addWidget(self.groupBoxLiveUpdate, 6, 0)
        self.layout.addWidget(self.emptyLabel3, 7, 0)
        self.layout.addWidget(self.groupBoxLabelListFeatures, 8, 0)
        self.layout.addWidget(self.listWidgetFeatures, 9, 0)
        self.layout.addWidget(self.groupBoxRemoveFeatures,10,0,Qt.AlignTop)
        self.layout.addWidget(self.groupSaveModel,11,0,Qt.AlignBottom)

        self.layout.setContentsMargins(5*vrb.ratio, 5*vrb.ratio, 5*vrb.ratio, 35*vrb.ratio)
        self.layout.setSizeConstraint(1)

        self.setLayout(self.layout)

        self.pushButtonDocumentation.clicked.connect(self.showDocumentation)
        self.buttonLiveUpdate.clicked.connect(self.changeLiveUpdate)
        self.buttonRefresh.clicked.connect(self.refreshModel)
        self.lineEditTreeNumber.textChanged.connect(self.changeNumberTree)
        self.lineEditTreeNumber.returnPressed.connect(self.parent.evaluateModel)
        self.buttonRemoveFeatures.clicked.connect(self.removeLowFeatures)
        self.lineEditModelName.textChanged.connect(self.allowSaveButton)

    def resizeEvent(self, a0: QtGui.QResizeEvent):

        totalHeight = 0

        totalHeight += 30 * vrb.ratio  # size button Interrogation

        drawGroupBoxHeight = 40*vrb.ratio
        self.drawGroupBox.setFixedSize(max(0, self.width() - 10 * vrb.ratio), drawGroupBoxHeight)
        totalHeight += drawGroupBoxHeight

        groupBoxFeaturesHeight = 80*vrb.ratio
        self.groupBoxFeatures.setFixedSize(max(0, self.width() - 10 * vrb.ratio), groupBoxFeaturesHeight)
        totalHeight += groupBoxFeaturesHeight

        groupBoxLiveUpdateHeight = 40*vrb.ratio
        self.groupBoxLiveUpdate.setFixedSize(max(0, self.width() - 10 * vrb.ratio), groupBoxLiveUpdateHeight)
        totalHeight += groupBoxLiveUpdateHeight

        groupBoxLabelListFeaturesHeight = 25*vrb.ratio
        self.groupBoxLabelListFeatures.setFixedSize(max(0, self.width() - 10 * vrb.ratio), groupBoxLabelListFeaturesHeight)
        totalHeight += groupBoxLabelListFeaturesHeight

        groupBoxRemoveFeaturesHeight = 25*vrb.ratio
        self.groupBoxRemoveFeatures.setFixedSize(max(0, self.width() - 10 * vrb.ratio), groupBoxRemoveFeaturesHeight)
        totalHeight += groupBoxRemoveFeaturesHeight

        groupBoxLabelListFeaturesHeight = 25*vrb.ratio
        self.groupBoxLiveUpdate.setFixedSize(max(0, self.width() - 10 * vrb.ratio), groupBoxLabelListFeaturesHeight)
        totalHeight += groupBoxLabelListFeaturesHeight

        minSizeLabelSelectionGroupBox = 200*vrb.ratio
        minSizegroupBoxLabelListFeatures = 200*vrb.ratio
        sizeEmptyLabel = (self.height() - totalHeight - minSizeLabelSelectionGroupBox - minSizegroupBoxLabelListFeatures)/3
        sizeEmptyLabel = max(0,sizeEmptyLabel)
        sizeEmptyLabel = min(25*vrb.ratio,sizeEmptyLabel)

        self.emptyLabel1.setFixedHeight(sizeEmptyLabel)
        self.emptyLabel2.setFixedHeight(sizeEmptyLabel)
        self.emptyLabel3.setFixedHeight(sizeEmptyLabel)
        totalHeight += 3*sizeEmptyLabel
        #
        totalHeight += 80*vrb.ratio # spacing between widgets

        heightLabelSelectionGroupBox = (self.height()-totalHeight)*3/5
        heightLabelSelectionGroupBox = min(heightLabelSelectionGroupBox,250*vrb.ratio)

        self.labelSelectionGroupBox.setFixedSize(max(0,self.width()-10*vrb.ratio),heightLabelSelectionGroupBox)
        totalHeight += heightLabelSelectionGroupBox

        self.listWidgetFeatures.setFixedSize(max(0,self.width()-10*vrb.ratio),self.height()-totalHeight)

        self.labelSelectionGroupBox.setFixedSize(max(0, self.width() - 10 * vrb.ratio), 230 * vrb.ratio)
        self.drawGroupBox.setFixedSize(max(0, self.width() - 10 * vrb.ratio), 40 * vrb.ratio)

    def showDocumentation(self):

        webbrowser.open_new_tab(vrb.folderDocEssential + "/smartclassification.html")

        # try:
        #     file = xmlet.parse(vrb.folderInformation + "/MachineLearningDoc.mho")
        #     xmlElement = file.getroot()
        #     pageDoc = Dfct.childText(xmlElement,"SmartClassification")
        #     webbrowser.open_new_tab(vrb.folderDocCore + "/" + pageDoc)
        # except:
        #     pass

    def allowSaveButton(self):

        try:
            self.buttonSaveModel.setEnabled(True)
            self.labelSelectionGroupBox.updateSaveSettingsButton()
        except:
            pass

    def removeLowFeatures(self):

        try:
            try:
                ponderationRef = float(self.lineEditPercentage.text())
            except:
                ponderationRef = 0
            model = self.parent.model
            featuresImportance = model.feature_importances_
            label = self.parent.currentLabel
            listNamesFeatures = label.listNamesFeatures

            xmlElement = self.measuresInfoSet.comboBoxInfoSet.currentData()[0]

            allMeasuresIterator = Dfct.childIterator(xmlElement,"Measure")
            allMeasuresElements = []
            for element in allMeasuresIterator:
                allMeasuresElements.append(element)

            measureToRemove = []
            measureToNotRemove = []

            for i in range(len(listNamesFeatures)):
                name = listNamesFeatures[i]
                name = name.split(" (Red)")[0]
                name = name.split(" (Green)")[0]
                name = name.split(" (Blue)")[0]
                name = name.split(" (Channel ")[0]
                if featuresImportance[i]*100<=ponderationRef:
                    if name not in measureToRemove:
                        measureToRemove.append(name)
                else:
                    if name not in measureToNotRemove:
                        measureToNotRemove.append(name)

            for measureName in measureToRemove:
                if measureName not in measureToNotRemove:
                    for element in allMeasuresElements:
                        if Dfct.childText(element,"UserName") == measureName:
                            element.set('CheckState', '0')

            xmlElement = Dfct.adjustCheckStateInfoset(xmlElement,returnElement=True)

            self.measuresInfoSet.validateInfoSetSelector(xmlElement)
            #self.measuresInfoSet.validateInfoSetSelector(xmlElement,self.measuresInfoSet.comboBoxInfoSet.currentText())

        except:
            pass
            # traceback.print_exc(file=sys.stderr)

    def changeNumberTree(self):

        self.parent.model = None

    def refreshModel(self):

        self.parent.evaluateModel()

    def changeLiveUpdate(self):

        self.buttonLiveUpdate.changeActivation()
        if self.buttonLiveUpdate.activate is False:
            self.parent.evaluateModel()
            self.parent.liveUpdate = True
            self.labelLiveUpdate.setStyleSheet('QLabel {color: rgb(52, 152, 219); background-color: rgb(70,70,70);}')
            self.buttonRefresh.setVisible(False)
        else:
            self.parent.liveUpdate = False
            self.labelLiveUpdate.setStyleSheet('QLabel {color: rgb(52, 152, 219); background-color: transparent;}')
            self.buttonRefresh.setVisible(True)


class DrawGroupBox(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.parent = parent

        self.buttonPen = wgt.PushButtonDoubleImage(vrb.folderImages + "/Pen.png", vrb.folderImages + "/Pen_Pressed.png")
        self.buttonPen.setToolTip("Draw on the image")
        self.buttonPen.setFixedSize(25*vrb.ratio, 25*vrb.ratio)
        self.buttonPen.mode = "Pen"

        self.buttonEraser = wgt.PushButtonDoubleImage(vrb.folderImages + "/Eraser.png", vrb.folderImages + "/Eraser_Pressed.png")
        self.buttonEraser.setToolTip("Erase on the image")
        self.buttonEraser.setFixedSize(25*vrb.ratio, 25*vrb.ratio)
        self.buttonEraser.mode = "Eraser"


        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.buttonPen, 0, 0)
        self.layout.addWidget(self.buttonEraser, 0, 1,Qt.AlignLeft)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setHorizontalSpacing(5)
        self.layout.setContentsMargins(20, 5, 20, 5)

        self.buttonPen.clicked.connect(self.changeMode)
        self.buttonEraser.clicked.connect(self.changeMode)

        self.setMode("Pen")

    def changeMode(self):

        sender = self.sender()
        for button in [self.buttonPen, self.buttonEraser]:
            button.setActivation(button == sender)
            if button == sender:
                self.parent.currentMode = button.mode

    def setMode(self, mode):

        for button in [self.buttonPen, self.buttonEraser]:
            button.setActivation(button.mode == mode)
            self.parent.currentMode = mode


class LabelSelectionGroupBox(qt.QGroupBox):

    def __init__(self,parent=None):
        qt.QGroupBox.__init__(self)

        self.parent = parent

        self.labelSelection = qt.QLabel("Class selection")
        font = QtGui.QFont()
        font.setBold(True)
        font.setPointSize(10)
        self.labelSelection.setFont(font)
        self.labelSelection.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.labelSelection.setFixedWidth(110*vrb.ratio)

        self.labelScrollArea = LabelScrollArea(parent=self.parent)

        self.buttonAdd = wgt.PushButtonImage(vrb.folderImages + "/Add_4.png", margins=0)
        self.buttonAdd.setToolTip("Add a new class")
        self.buttonAdd.setFixedSize(25*vrb.ratio, 25*vrb.ratio)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelSelection, 0, 0)
        self.layout.addWidget(self.labelScrollArea, 1, 0)
        self.layout.addWidget(self.buttonAdd, 2, 0, Qt.AlignLeft)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setHorizontalSpacing(5)
        self.layout.setContentsMargins(20, 5, 20, 5)

        self.buttonAdd.clicked.connect(self.labelScrollArea.addLabel)

    def resizeEvent(self, a0: QtGui.QResizeEvent):

        self.labelScrollArea.setFixedSize(self.width()-20*vrb.ratio, self.height()-70*vrb.ratio)


class LabelScrollArea(qt.QScrollArea):

    SignalDeleteLabel = pyqtSignal(int)
    SignalColorChanged = pyqtSignal()
    SignalAddLabel = pyqtSignal()

    def __init__(self,parent):
        qt.QScrollArea.__init__(self)

        self.parent = parent

        self.selectedLabel = None

        self.centralWidget = qt.QGroupBox()

        self.layout = qt.QVBoxLayout()
        self.layout.setAlignment(Qt.AlignTop)

        self.centralWidget.setLayout(self.layout)

        self.setWidget(self.centralWidget)

        self.layout.setContentsMargins(0, 0, 0, 0)

        self.setStyleSheet('QScrollArea {border: 1px solid gray; }')

        self.addLabel()
        self.addLabel()

    def resizeEvent(self, a0: QtGui.QResizeEvent):

        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                label = item.widget()
                label.setFixedSize(self.width() - 20, 35 * vrb.ratio)

        self.centralWidget = qt.QWidget()
        self.centralWidget.setFixedWidth(self.width()-20)
        self.centralWidget.setLayout(self.layout)
        self.setWidget(self.centralWidget)

    def addLabel(self):

        self.addLabelWithName()

    def addLabelWithName(self,color=None,name=None):

        labelNum = self.layout.count()
        label = GroupBoxColor(labelNum + 1, labelNum,color=color,name=name)
        label.setFixedSize(self.width() - 20*vrb.ratio, 35 * vrb.ratio)
        label.SignalSelection.connect(self.selectLabel)
        label.SignalDeleteVariable.connect(self.deleteLabel)
        label.SignalColorChanged.connect(self.emitSignalColorChanged)

        self.layout.addWidget(label)

        self.centralWidget = qt.QWidget()
        self.centralWidget.setFixedWidth(self.width() - 20*vrb.ratio)
        self.centralWidget.setLayout(self.layout)
        self.setWidget(self.centralWidget)

        self.verticalScrollBar().setValue(self.verticalScrollBar().maximum())

        self.SignalAddLabel.emit()

        if labelNum == 0:
            self.selectLabel(label)

    def selectLabel(self, label):

        try:
            if self.parent.currentMode == "Eraser":
                self.parent.settingsWidget.drawGroupBox.setMode("Pen")
        except:
            pass

        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                if item.widget() == label:
                    label.selectLabel()
                    self.selectedLabel = label
                    # self.parent.boxPencil.radioButtonDraw.setChecked(True)
                else:
                    item.widget().unselectLabel()

    def selectLabelByNum(self,numLabel):

        try:
            self.parent.settingsWidget.drawGroupBox.setMode("Pen")
        except:
            pass

        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                if num == numLabel:
                    item.widget().selectLabel()
                    self.selectedLabel = item.widget()
                    # self.parent.boxPencil.radioButtonDraw.setChecked(True)
                else:
                    item.widget().unselectLabel()

    def deleteLabel(self, label):

        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                if item.widget() == label:
                    item.widget().deleteLater()
                    self.layout.removeItem(item)
                    self.SignalDeleteLabel.emit(num)

    def getCurrentNum(self):

        currentNum = 0
        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                if item.widget() == self.selectedLabel:
                    currentNum = num + 1

        return currentNum

    def emitSignalColorChanged(self):

        self.SignalColorChanged.emit()

    def getCustomLutOverlay(self):

        outputLut = {}
        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                color = item.widget().color
                for c in range(3):
                    if c not in outputLut:
                        outputLut[c] = [0]
                    outputLut[c].append(color[c]*0.7)

        return outputLut


class GroupBoxColor(qt.QGroupBox):

    SignalSelection = pyqtSignal(qt.QGroupBox)
    SignalDeleteVariable = pyqtSignal(qt.QGroupBox)
    SignalColorChanged = pyqtSignal()

    def __init__(self, num=1, colorNum=1,color=None,name = None):
        qt.QGroupBox.__init__(self)

        self.setStyleSheet('QGroupBox {border: 0px transparent; }')

        if color is None:
            try:
                if colorNum not in [0,1]:
                    self.color = vrb.dictColor[colorNum]
                elif colorNum == 0:
                    self.color = [20, 200, 190]
                elif colorNum == 1:
                    self.color = [200, 182, 41]
            except:
                self.color = [random.randint(vrb.randomColorMin, vrb.randomColorMax),random.randint(vrb.randomColorMin, vrb.randomColorMax),random.randint(vrb.randomColorMin, vrb.randomColorMax)]
        else:
            self.color = color

        if name is None:
            self.editableLabelName = wgt.EditableLabel("Class "+str(num))
        else:
            self.editableLabelName = wgt.EditableLabel(name)

        font = QtGui.QFont()
        font.setPixelSize(vrb.fontSizeMachineLearning*vrb.ratio)

        self.editableLabelName.label.setFixedSize(75*vrb.ratio,20*vrb.ratio)
        self.editableLabelName.label.setFont(font)
        self.editableLabelName.lineEdit.setFixedSize(75*vrb.ratio,20*vrb.ratio)
        self.editableLabelName.lineEdit.setFont(font)

        self.labelColor = qt.QLabel()
        self.labelColor.setFixedSize(20*vrb.ratio, 20*vrb.ratio)
        self.changeButtonColor()

        self.labelCount = qt.QLabel('0')
        self.labelCount.setFixedSize(40 * vrb.ratio, 20 * vrb.ratio)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.editableLabelName, 0, 0)
        self.layout.addWidget(self.labelColor, 0, 1 ,Qt.AlignLeft)
        self.layout.addWidget(self.labelCount, 0, 2, Qt.AlignCenter)

        if num > 2:
            self.deleteLabel = wgt.LabelDelete()
            self.deleteLabel.setFixedSize(20* vrb.ratio, 20* vrb.ratio)
            self.deleteLabel.setToolTip(txt.dictToolTips["DeleteLabel"])
            self.layout.addWidget(self.deleteLabel, 0, 3, QtCore.Qt.AlignRight)
            self.deleteLabel.clicked.connect(self.emitSignalDeleteVariable)
        else:
            self.emptyLabel = qt.QLabel()
            self.emptyLabel.setFixedSize(20* vrb.ratio, 20* vrb.ratio)
            self.layout.addWidget(self.emptyLabel, 0, 3, QtCore.Qt.AlignRight)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setHorizontalSpacing(5)
        self.layout.setContentsMargins(5, 0, 5, 0)

        self.labelColor.mouseDoubleClickEvent = self.getColor
        self.editableLabelName.mouseDoubleClickEvent = self.changeName

        self.labelColor.mousePressEvent = self.emitSelection
        self.editableLabelName.mousePressEvent = self.emitSelection

    def getColor(self, event):

        color = qt.QColorDialog.getColor()

        if color.isValid():
            self.color = [color.red(), color.green(), color.blue()]
            self.changeButtonColor()

    def changeName(self, event):

        self.editableLabelName.switchVisibleWidget()

    def changeButtonColor(self):

        self.labelColor.setStyleSheet('QLabel {background-color: rgb(' + str(self.color[0]) + ',' + str(self.color[1]) + ',' + str(self.color[2]) + ');}')
        self.SignalColorChanged.emit()

    def emitSelection(self, event):

        self.SignalSelection.emit(self)

    def emitSignalDeleteVariable(self):

        self.SignalDeleteVariable.emit(self)

    def unselectLabel(self):

        font = QtGui.QFont()
        font.setPixelSize(vrb.fontSizeMachineLearning*vrb.ratio)
        font.setBold(False)
        self.editableLabelName.label.setFont(font)
        self.labelCount.setFont(font)
        self.editableLabelName.label.setStyleSheet('QLabel {background-color: transparent;}')

    def selectLabel(self):

        font = QtGui.QFont()
        font.setPixelSize(vrb.fontSizeMachineLearning*vrb.ratio)
        font.setBold(True)
        self.editableLabelName.label.setFont(font)
        self.labelCount.setFont(font)
        self.editableLabelName.label.setStyleSheet('QLabel {background-color: rgb(70,70,70);}')


def addLabelToScene(imageViewer, posX, posY, labelClass):

    relativePoint = imageViewer.realPointToScene(float(posX), float(posY))

    colorCircle = QtGui.QColor(50, 50, 50)
    colorText = QtGui.QColor(255, 255, 255)

    circleRect = QtCore.QRectF(QtCore.QPointF(relativePoint[0] - 15, relativePoint[1] - 15), QtCore.QPointF(relativePoint[0] + 15, relativePoint[1] + 15))
    pen = QtGui.QPen(colorCircle, 1)
    circle = imageViewer.scene.addEllipse(circleRect, pen)
    circle.setBrush(QtGui.QBrush(colorCircle, style=QtCore.Qt.SolidPattern))
    circle.setOpacity(0.7)

    text = imageViewer.scene.addText(labelClass)
    font = QtGui.QFont()
    font.setPointSize(12)
    font.setBold(True)
    text.setFont(font)
    text.setDefaultTextColor(colorText)

    text.setPos(relativePoint[0] - text.boundingRect().width() / 2, relativePoint[1] - text.boundingRect().height() / 2)


def xmlInfosetToMeasure(xmlElement,greyImage, labelImage, dimension="3D",intensity = False):

    text = ""
    if dimension == "2D":
        text += "calibration = PyIPSDK.createGeometricCalibration2d(1,1,'px')\n"
        if intensity == False:
            text += xml2IS.xmlToInfoset(xmlElement, 'inMeasureInfoSet2d', '2D', 'calibration', addBarycenter=False)
        else:
            text += xml2IS.xmlToInfosetIntensity(xmlElement, 'inMeasureInfoSet2d', '2D', 'calibration')
        text += 'outMeasure = shapeanalysis.labelAnalysis2d(greyImage,labelImage,inMeasureInfoSet2d)\n'
    else:
        text += "calibration = PyIPSDK.createGeometricCalibration3d(1,1,1,'px')\n"
        if intensity == False:
            text += xml2IS.xmlToInfoset(xmlElement, 'inMeasureInfoSet3d', '3D', 'calibration', addBarycenter=False)
        else:
            text += xml2IS.xmlToInfosetIntensity(xmlElement, 'inMeasureInfoSet3d', '3D', 'calibration')
        text += 'outMeasure = shapeanalysis.labelAnalysis3d(greyImage,labelImage,inMeasureInfoSet3d)\n'

    _locals = locals()
    exec(text, globals(), _locals)
    outMeasure = _locals['outMeasure']

    return outMeasure


if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook

    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    foo = MainWidget()

    # foo.loadModel("Model 11")

    foo.resize(1200, 900)

    foo.show()

    app.exec_()

