import sys,os
import traceback

from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore
from PyQt5.QtCore import pyqtSignal, Qt, QCoreApplication

import PyIPSDK
import PyIPSDK.IPSDKIPLUtility as util
import PyIPSDK.IPSDKIPLBasicMorphology as morpho
import PyIPSDK.IPSDKIPLFiltering as filtering
import PyIPSDK.IPSDKIPLStats as stats

import numpy as np

import xml.etree.ElementTree as xmlet
import DatabaseFunction as Dfct

import UsefullWidgets as wgt
import UsefullTexts as txt
import UsefullVariables as vrb
import UsefullFunctions as fct
import moduleRandomForest.UsefullMachineLearning as mlFct

from Sliders import SimpleHorizontalSliderLabel

class FeaturesWidget(qt.QWidget):

    def __init__(self, parent=None,xmlElement=None):
        qt.QWidget.__init__(self)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.parent = parent
        self.xmlElement = xmlElement
        # self.xmlElement = xmlet.Element('Features')

        self.groupBoxLoading = GroupBoxLoading()

        self.tableFeatures = TableFeatures(parent=self)

        self.buttonAdd = wgt.PushButtonImage(vrb.folderImages + "/Add_4.png", margins=0)
        self.buttonAdd.setToolTip("Add a new size")
        self.buttonAdd.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        font = QtGui.QFont()
        font.setPixelSize(vrb.fontSizeMachineLearning * vrb.ratio)

        self.groupBoxParameters = GroupBoxParameters()

        self.groupBoxValidate = GroupBoxValidate()

        self.layout = qt.QGridLayout()

        # self.layout.addWidget(self.groupBox,0,0)
        self.layout.addWidget(self.groupBoxLoading,0,0,1,2)
        self.layout.addWidget(self.tableFeatures,1,0,Qt.AlignTop | Qt.AlignLeft)
        self.layout.addWidget(self.buttonAdd,1,1,Qt.AlignTop | Qt.AlignLeft)
        self.layout.addWidget(self.groupBoxParameters,2,0,Qt.AlignTop | Qt.AlignLeft)
        self.layout.addWidget(self.groupBoxValidate,3,0,1,2)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(10, 10, 10, 0)

        self.groupBoxLoading.buttonValidate.clicked.connect(self.loadFromModel)
        self.buttonAdd.clicked.connect(self.tableFeatures.addNewSize)
        self.groupBoxValidate.buttonValidate.clicked.connect(self.createXmlElement)

        self.setWindowTitle("Features")

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def createXmlElement(self,event,close=True):

        allSizes = Dfct.SubElement(self.xmlElement,"All_Sizes")
        text = ""
        for col in range(1,self.tableFeatures.columnCount()):
            text+= self.tableFeatures.item(0,col).text() + ","
        if text != "":
            text = text[:-1]
        allSizes.text = text

        for row in range(1,self.tableFeatures.rowCount()):
            nameFunction = self.tableFeatures.item(row,0).text()
            if nameFunction in self.tableFeatures.listFilters or nameFunction in self.tableFeatures.listMorphology:
                nameFunction = nameFunction.replace(" ", "_")
                functionElement = Dfct.SubElement(self.xmlElement, nameFunction)
                text = ""
                for col in range(1, self.tableFeatures.columnCount()):
                    try:
                        if self.tableFeatures.item(row, col).checked:
                            text += "True,"
                        else:
                            text+="False,"
                    except:
                        text+="False,"
                if text != "":
                    text = text[:-1]
                functionElement.text = text

        dimensionElement = Dfct.SubElement(self.xmlElement, "Dimension")
        if self.groupBoxParameters.buttonElement3D.isChecked():
            dimensionElement.text = "3D"
        else:
            dimensionElement.text = "2D"

        multiResElement = Dfct.SubElement(self.xmlElement, "MultiRes")
        text = ""
        for button in self.groupBoxParameters.resolutionButtons:
            if button.isChecked():
                text += str(button.size) + ","
        if text!="":
            text = text[:-1]
        multiResElement.text = text

        if close:
            self.close()

        try:
            self.parent.model = None
            self.parent.modelIPSDK = None
            self.parent.featuresMessage = True
            self.parent.evaluateModel()
        except:
            pass

    def loadXmlElement(self,xmlElement,save=True):

        if save:
            self.xmlElement = xmlElement

        self.tableFeatures.loadXmlElement(xmlElement)

        dimension = Dfct.childText(xmlElement, "Dimension")
        if dimension == "3D":
            self.groupBoxParameters.buttonElement3D.setChecked(True)
        else:
            self.groupBoxParameters.buttonElement2D.setChecked(True)

        multiResElement = Dfct.childText(xmlElement, "MultiRes")
        if multiResElement is not None:
            multiRes = multiResElement.split(",")
            for res in multiRes:
                for button in self.groupBoxParameters.resolutionButtons:
                    if str(button.size) == res:
                        button.setChecked(True)

    def actualizeComboBoxLoading(self):

        self.groupBoxLoading.comboBoxLoad.clear()
        for modelName in sorted(os.listdir(vrb.folderPixelClassification),key=str.casefold):
            filePath = vrb.folderPixelClassification + '/' + modelName
            if os.path.isdir(filePath):

                try:
                    settingsPath = os.path.join(filePath + "/Settings.mho")
                    file = xmlet.parse(settingsPath)
                    xmlElement = file.getroot()
                    featuresElement = Dfct.SubElement(xmlElement, 'Features')
                    self.groupBoxLoading.comboBoxLoad.addItem(modelName,featuresElement)
                except:
                    pass

    def loadFromModel(self):

        try:
            featuresElement = self.groupBoxLoading.comboBoxLoad.currentData()
            self.loadXmlElement(featuresElement,save=False)
        except:
            pass

class GroupBoxLoading(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.labelLoad = qt.QLabel("Load from model")
        self.labelLoad.setFixedWidth(80*vrb.ratio)
        self.comboBoxLoad = qt.QComboBox()
        self.comboBoxLoad.setFixedSize(120*vrb.ratio,25*vrb.ratio)
        self.buttonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png")
        self.buttonValidate.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)
        self.buttonValidate.setToolTip("Load features from this model")

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelLoad,0,0)
        self.layout.addWidget(self.comboBoxLoad,0,1)
        self.layout.addWidget(self.buttonValidate,0,2,Qt.AlignLeft)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(0, 0, 0, 0)
        self.layout.setHorizontalSpacing(15*vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

class GroupBoxValidate(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.buttonCancel = wgt.PushButtonImage(vrb.folderImages + "/Back.png")
        self.buttonCancel.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.buttonCancel.setToolTip(txt.dictToolTips["Cancel"])
        self.buttonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png")
        self.buttonValidate.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.buttonValidate.setToolTip(txt.dictToolTips["Apply"])

        self.layout = qt.QGridLayout()

        # self.layout.addWidget(self.buttonCancel,0,0,Qt.AlignRight)
        # self.layout.addWidget(self.buttonValidate,0,1)
        self.layout.addWidget(self.buttonValidate,0,0,Qt.AlignRight)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(0, 0, 0, 0)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

class GroupBoxParameters(qt.QGroupBox):

    def __init__(self):
        qt.QGroupBox.__init__(self)

        groupBoxResolution = qt.QGroupBox()
        layoutGroupBoxResolution = qt.QHBoxLayout()
        self.labelResolution = qt.QLabel("Multiple resolution")
        layoutGroupBoxResolution.addWidget(self.labelResolution)
        self.resolutionButtons = []
        for size in [1,2,4,8]:
            button = qt.QCheckBox(str(size))
            button.size = size
            if size == 1:
                button.setChecked(True)
                button.setEnabled(False)
            self.resolutionButtons.append(button)
            layoutGroupBoxResolution.addWidget(button)
        groupBoxResolution.setLayout(layoutGroupBoxResolution)
        groupBoxResolution.setStyleSheet('QGroupBox {border: 0px transparent;}')
        layoutGroupBoxResolution.setContentsMargins(0,0,0,0)
        layoutGroupBoxResolution.setSpacing(10)

        groupBoxElement = qt.QGroupBox()
        layoutGroupBoxElement = qt.QHBoxLayout()
        self.labelElement = qt.QLabel("Structural element")
        layoutGroupBoxElement.addWidget(self.labelElement)
        self.buttonElement2D = qt.QRadioButton("2D")
        self.buttonElement2D.setChecked(True)
        layoutGroupBoxElement.addWidget(self.buttonElement2D)
        self.buttonElement3D = qt.QRadioButton("3D")
        layoutGroupBoxElement.addWidget(self.buttonElement3D,Qt.AlignLeft)

        # self.labelElement.setFont(font)
        # self.buttonElement3D.setFont(font)
        # self.buttonElement2D.setFont(font)
        # self.pushButtonDocumentation = wgt.PushButtonImage(margins=1, filename=vrb.folderImages + '/Interrogation.png')
        # self.pushButtonDocumentation.setStyleSheet("background-color : transparent; border :0px")
        # self.pushButtonDocumentation.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)
        # layoutGroupBoxElement.addWidget(self.pushButtonDocumentation,Qt.AlignRight)

        groupBoxElement.setLayout(layoutGroupBoxElement)
        groupBoxElement.setStyleSheet('QGroupBox {border: 0px transparent;}')
        layoutGroupBoxElement.setContentsMargins(0,0,0,0)
        layoutGroupBoxElement.setSpacing(10)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(groupBoxElement, 0, 0)
        self.layout.addWidget(groupBoxResolution, 1, 0)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        # self.layout.setVerticalSpacing(20)
        self.layout.setContentsMargins(10, 10, 10, 20)

        self.setTitle("Parameters")
        self.setStyleSheet(f'QGroupBox:title {{left: 8px ;padding-left: 10px;padding-right: 10px; padding-top: {-10*vrb.ratio}px; color:rgb(6, 115, 186)}}  QGroupBox {{font: bold; margin-top: {10*vrb.ratio} px;font-size:{12*vrb.ratio}px;border: 1px solid gray;}}')
        self.setFixedWidth(320*vrb.ratio)

        font = QtGui.QFont()
        font.setBold(False)
        # sets a font to the children of a widget
        for widget in self.findChildren(qt.QWidget):
            widget.setFont(font)

class IconDelegate(qt.QStyledItemDelegate):
    def initStyleOption(self, option, index):
        super(IconDelegate, self).initStyleOption(option, index)
        if option.features & qt.QStyleOptionViewItem.HasDecoration:
            s = option.decorationSize
            s.setWidth(option.rect.width())
            option.decorationSize = s

class TableFeatures(qt.QTableWidget):

    def __init__(self,parent=None):
        qt.QTableWidget.__init__(self)

        self.parent = parent

        delegate = IconDelegate(self)
        self.setItemDelegate(delegate)

        self.horizontalHeader().setContextMenuPolicy(Qt.CustomContextMenu)
        self.horizontalHeader().customContextMenuRequested.connect(self.showHeaderMenu)

        self.horizontalHeader().setSectionResizeMode(qt.QHeaderView.Fixed)
        self.verticalHeader().setSectionResizeMode(qt.QHeaderView.Fixed)

        self.listFilters = ["Gaussian", "Mean", "Laplacian Difference", "High Pass", "Variance"]
        self.listMorphology = ["Erosion", "Dilation", "Opening", "Closing"]

        # self.loadXmlElement(None)

        self.horizontalHeader().sectionClicked.connect(self.cellHorizontalHeaderClicked)
        self.verticalHeader().sectionClicked.connect(self.cellVerticalHeaderClicked)
        self.itemPressed.connect(self.cellClick)

    def loadXmlElement(self,xmlElement):

        self.clear()

        nbRow = len(self.listFilters) + len(self.listMorphology) + 3
        if xmlElement is None:
            nbCol = 6
            allSizes = ["1","2","3","4","5"]
        else:
            try:
                allSizes = Dfct.childText(xmlElement,"All_Sizes")
                allSizes = allSizes.split(",")
                nbCol = len(allSizes)+1
            except:
                nbCol = 6
                allSizes = ["1", "2", "3", "4", "5"]

        self.setRowCount(nbRow)
        for i in range(self.rowCount()):
            self.setRowHeight(i,30*vrb.ratio)

        self.setColumnCount(nbCol)
        for i in range(self.columnCount()):
            if i == 0:
                self.setColumnWidth(i,120*vrb.ratio)
                self.setHorizontalHeaderItem(i, qt.QTableWidgetItem(''))
            else:
                self.setColumnWidth(i, 30 * vrb.ratio)
                headerItem = qt.QTableWidgetItem('▼')
                self.setHorizontalHeaderItem(i, headerItem)

        for i in range(self.rowCount()):
            self.setVerticalHeaderItem(i, qt.QTableWidgetItem(''))

        cell = qt.QTableWidgetItem()
        cell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
        cell.setFlags(cell.flags() & ~Qt.ItemIsEditable)
        self.setItem(0, 0, cell)

        for i in range(1,self.columnCount()):
            cell = qt.QTableWidgetItem(allSizes[i-1])
            cell.setTextAlignment(Qt.AlignCenter)
            cell.setBackground(QtGui.QColor(220, 220, 255))
            cell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
            #cell.setFlags(cell.flags() & ~Qt.ItemIsEditable)
            self.setItem(0, i, cell)

        numRow = 1
        self.fillRowTitle("Filters",numRow)
        numRow+=1
        for name in self.listFilters:
            self.fillRowFeature(name,numRow,xmlElement)
            numRow+=1
        self.fillRowTitle("Morphology",numRow)
        numRow+=1
        for name in self.listMorphology:
            self.fillRowFeature(name,numRow,xmlElement)
            numRow+=1

        self.resizeTable()

    def resizeTable(self):

        minWidth = 290

        width = max(120*vrb.ratio + (self.columnCount()-1)*40*vrb.ratio + 20*vrb.ratio,minWidth*vrb.ratio)
        height = self.rowCount()*30*vrb.ratio + 15*vrb.ratio
        self.setFixedSize(width,height)

        self.parent.setFixedSize(width+50*vrb.ratio,height+260*vrb.ratio)

    def showHeaderMenu(self, point ):

        column = self.horizontalHeader().logicalIndexAt(point.x())

        if column > 1:
            menu = qt.QMenu(self)
            removeAction = menu.addAction('Remove column')
            removeAction.triggered.connect(lambda: self.removeSelectedColumn(column))
            menu.popup(self.horizontalHeader().mapToGlobal(point))

    def contextMenuEvent(self, event):

        row = self.rowAt(event.pos().y())
        col = self.columnAt(event.pos().x())
        cell = self.item(row, col)

        try:
            cell.checked

            pixmap = QtGui.QPixmap(vrb.folderImages + "/Validate.png")
            icon = QtGui.QIcon(pixmap)
            cell.setIcon(icon)
            cell.checked = True

            self.menu = qt.QMenu(self)
            previewAction = qt.QAction('Preview', self)
            previewAction.row = row
            previewAction.col = col
            previewAction.triggered.connect(lambda: self.previewSlot(event))
            self.menu.addAction(previewAction)
            # add other required actions
            self.menu.popup(QtGui.QCursor.pos())
        except:
            pass

    def removeSelectedColumn(self,col):

        self.removeColumn(col)
        self.resizeTable()

    def previewSlot(self,event):

        try:

            moduleWidget = self.parent.parent

            row = self.sender().row
            nameFunction = self.item(row,0).text()
            col = self.sender().col
            size = int(self.item(0,col).text())

            currentImage = moduleWidget.currentLabel.image
            image = mlFct.featureFromName(currentImage,nameFunction,size,1,"2D")

            try:
                image.valueMin
            except:
                valueMin, valueMax = fct.getMinMaxValue(image)
                image.valueMin = valueMin
                image.valueMax = valueMax

            if moduleWidget.settingsWidget.buttonViewFeatures.activate == False:
                moduleWidget.widgetImage.sliderOpacity.slider.setValue(0)
            moduleWidget.settingsWidget.buttonViewFeatures.setActivation(True)
            moduleWidget.widgetImage.setImage(image)

        except:
            traceback.print_exc(file=sys.stderr)

    def addNewSize(self):

        self.setColumnCount(self.columnCount()+1)
        numCol = self.columnCount()-1
        self.setColumnWidth(numCol,30*vrb.ratio)
        self.setHorizontalHeaderItem(numCol, qt.QTableWidgetItem('▼'))

        maxSize = 0
        for i in range(self.columnCount()):
            cell = self.item(0, i)
            try:
                maxSize = max(int(cell.text()),maxSize)
            except:
                pass

        cell = qt.QTableWidgetItem(str(maxSize+1))
        cell.setTextAlignment(Qt.AlignCenter)
        cell.setBackground(QtGui.QColor(220, 220, 255))
        cell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
        self.setItem(0, numCol, cell)

        self.resizeTable()

        for i in range(1,self.rowCount()):
            cellRef = self.item(i,1)
            try:
                cellRef.checked
                cell = qt.QTableWidgetItem()
                cell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
                cell.setFlags(cell.flags() & ~Qt.ItemIsEditable)
                cell.checked = False
            except:
                cell = qt.QTableWidgetItem()
                cell.setBackground(QtGui.QColor(127, 127, 127))
                cell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
                cell.setFlags(cell.flags() & ~Qt.ItemIsEditable)
            self.setItem(i, numCol, cell)

    def cellClick(self,cellItem):

        try:
            if cellItem.checked == False:
                pixmap = QtGui.QPixmap(vrb.folderImages + "/Validate.png")
                icon = QtGui.QIcon(pixmap)
                cellItem.setIcon(icon)
                cellItem.checked = True
            else:
                cell = qt.QTableWidgetItem()
                cell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
                cell.setFlags(cell.flags() & ~Qt.ItemIsEditable)
                cell.checked = False
                self.setItem(cellItem.row(),cellItem.column(), cell)
        except:
            pass

    def cellHorizontalHeaderClicked(self,numCol):

        allChecked = True
        for i in range(self.rowCount()):
            try:
                cell = self.item(i, numCol)
                if cell.checked == False:
                    allChecked = False
            except:
                pass
        for i in range(self.rowCount()):
                try:
                    cell = self.item(i, numCol)
                    cell.checked
                    if allChecked:
                        newCell = qt.QTableWidgetItem()
                        newCell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
                        newCell.setFlags(cell.flags() & ~Qt.ItemIsEditable)
                        newCell.checked = False
                        self.setItem(i, numCol, newCell)
                    else:
                        pixmap = QtGui.QPixmap(vrb.folderImages + "/Validate.png")
                        icon = QtGui.QIcon(pixmap)
                        cell.setIcon(icon)
                        cell.checked = True
                except:
                    pass

    def cellVerticalHeaderClicked(self,numRow):

        allChecked = True
        for i in range(self.columnCount()):
            try:
                cell = self.item(numRow, i)
                if cell.checked == False:
                    allChecked = False
            except:
                pass
        for i in range(self.columnCount()):
                try:
                    cell = self.item(numRow, i)
                    cell.checked
                    if allChecked:
                        newCell = qt.QTableWidgetItem()
                        newCell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
                        newCell.setFlags(cell.flags() & ~Qt.ItemIsEditable)
                        newCell.checked = False
                        self.setItem(numRow, i, newCell)
                    else:
                        pixmap = QtGui.QPixmap(vrb.folderImages + "/Validate.png")
                        icon = QtGui.QIcon(pixmap)
                        cell.setIcon(icon)
                        cell.checked = True
                except:
                    pass

    def fillRowTitle(self,text,num):

        cell = qt.QTableWidgetItem(text)
        cell.setTextAlignment(Qt.AlignCenter)
        cell.setBackground(QtGui.QColor(127, 127, 127))
        cell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
        cell.setFlags(cell.flags() & ~Qt.ItemIsEditable)
        self.setItem(num, 0, cell)
        for i in range(1,self.columnCount()):
            cell =  self.item(num,i)
            if cell is None:
                cell = qt.QTableWidgetItem()
            cell.setBackground(QtGui.QColor(127, 127, 127))
            cell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
            cell.setFlags(cell.flags() & ~Qt.ItemIsEditable)
            self.setItem(num, i, cell)

    def fillRowFeature(self,text,num,xmlElement):

        if xmlElement is None:
            values = []
            for i in range(self.columnCount()-1):
                values.append(False)
        else:
            elementText = Dfct.childText(xmlElement,text.replace(" ","_"))
            if elementText is not None:
                values = elementText.split(",")
                for i in range(len(values)):
                    if values[i] == "False":
                        values[i] = False
                    else:
                        values[i] = True
            elif text == "Dilation":
                elementText = Dfct.childText(xmlElement, "Dilatation")
                if elementText is not None:
                    values = elementText.split(",")
                    for i in range(len(values)):
                        if values[i] == "False":
                            values[i] = False
                        else:
                            values[i] = True
            else:
                values = []
                for i in range(self.columnCount() - 1):
                    values.append(False)

        cell = qt.QTableWidgetItem(text)
        cell.setTextAlignment(Qt.AlignCenter)
        cell.setBackground(QtGui.QColor(220, 220, 255))
        cell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
        cell.setFlags(cell.flags() & ~Qt.ItemIsEditable)
        self.setItem(num, 0, cell)
        for i in range(1,self.columnCount()):
            cell = self.item(num,i)
            if cell is None:
                cell = qt.QTableWidgetItem()
            cell.setFlags(cell.flags() & ~Qt.ItemIsSelectable)
            cell.setFlags(cell.flags() & ~Qt.ItemIsEditable)

            if values[i-1]:
                pixmap = QtGui.QPixmap(vrb.folderImages + "/Validate.png")
                icon = QtGui.QIcon(pixmap)
                cell.setIcon(icon)
                cell.checked = True
            else:
                cell.checked = False
            self.setItem(num, i, cell)
        # self.setVerticalHeaderItem(num, qt.QTableWidgetItem('▶'))
        cell = qt.QTableWidgetItem()
        pixmap = QtGui.QPixmap(vrb.folderImages + "/TriangleRight.png")
        icon = QtGui.QIcon(pixmap)
        cell.setIcon(icon)
        self.setVerticalHeaderItem(num, cell)

class ListWidgetFeatures(qt.QListWidget):

    def __init__(self,parent=None):
        qt.QListWidget.__init__(self)

        self.parent = parent
        self.itemSelectionChanged.connect(self.loadCurrentItem)
        self.itemClicked.connect(self.itemClickedFunction)

        # self.setFixedHeight(100*vrb.ratio)

    def fillListWidget(self):

        settingsWidget = self.parent.settingsWidget

        try:

            settingsWidget.currentLabelSort

            try:
                self.clear()
                label = self.parent.currentLabel
                model = self.parent.model

                if model is None or settingsWidget.currentLabelSort == settingsWidget.labelFeaturesName:

                    if model is not None:
                        featuresImportance = model.feature_importances_
                    else:
                        featuresImportance = []

                    ind = np.argsort(label.listNamesFeatures)

                    vectorStart = ["Original image","Original image (Blue)","Original image (Green)","Original image (Red)"]
                    for i in range(len(label.listNamesFeatures)):
                        vectorStart.append("Original image (channel "+str(i)+")")

                    if settingsWidget.labelFeaturesName.sort:

                        for featureText in vectorStart:
                            for i in range(len(label.listNamesFeatures)):
                                if label.listNamesFeatures[i] == featureText:
                                    newItem = qt.QListWidgetItem()
                                    newItem.setText(label.listNamesFeatures[i])
                                    newItem.image = label.listFeatures[i]
                                    self.addItem(newItem)

                                    if len(featuresImportance) != 0:
                                        widget = qt.QLabel(fct.numberCalibration(featuresImportance[i] * 100) + " %")
                                        widget.setAlignment(Qt.AlignRight)
                                        self.addItem(newItem)
                                        self.setItemWidget(newItem, widget)

                        for i in ind:
                            if label.listNamesFeatures[i] not in vectorStart:
                                newItem = qt.QListWidgetItem()
                                newItem.setText(label.listNamesFeatures[i])
                                newItem.image = label.listFeatures[i]
                                self.addItem(newItem)

                                if len(featuresImportance) != 0:
                                    widget = qt.QLabel(fct.numberCalibration(featuresImportance[i] * 100) + " %")
                                    widget.setAlignment(Qt.AlignRight)
                                    self.addItem(newItem)
                                    self.setItemWidget(newItem, widget)

                    else:
                        ind = ind[::-1]
                        for i in ind:
                            if label.listNamesFeatures[i] not in vectorStart:
                                newItem = qt.QListWidgetItem()
                                newItem.setText(label.listNamesFeatures[i])
                                newItem.image = label.listFeatures[i]
                                self.addItem(newItem)

                                if len(featuresImportance) != 0:
                                    widget = qt.QLabel(fct.numberCalibration(featuresImportance[i] * 100) + " %")
                                    widget.setAlignment(Qt.AlignRight)
                                    self.addItem(newItem)
                                    self.setItemWidget(newItem, widget)

                        for featureText in vectorStart:
                            for i in range(len(label.listNamesFeatures)):
                                if label.listNamesFeatures[i] == featureText:
                                    newItem = qt.QListWidgetItem()
                                    newItem.setText(label.listNamesFeatures[i])
                                    newItem.image = label.listFeatures[i]
                                    self.addItem(newItem)

                                    if len(featuresImportance) != 0:
                                        widget = qt.QLabel(fct.numberCalibration(featuresImportance[i] * 100) + " %")
                                        widget.setAlignment(Qt.AlignRight)
                                        self.addItem(newItem)
                                        self.setItemWidget(newItem, widget)

                else:
                    featuresImportance = model.feature_importances_
                    ind = np.argsort(featuresImportance)
                    if settingsWidget.labelFeaturesPercentage.sort:
                        ind = ind[::-1]

                    for i in ind:
                        newItem = qt.QListWidgetItem()
                        newItem.setText(label.listNamesFeatures[i])
                        newItem.image = label.listFeatures[i]

                        widget = qt.QLabel(fct.numberCalibration(featuresImportance[i] * 100) + " %")
                        widget.setAlignment(Qt.AlignRight)
                        self.addItem(newItem)
                        self.setItemWidget(newItem, widget)

            except:
                traceback.print_exc(file=sys.stderr)


        except:

            try:
                self.clear()
                label = self.parent.currentLabel
                model = self.parent.model

                if model is None:

                    for i in range(len(label.listNamesFeatures)):
                        newItem = qt.QListWidgetItem()
                        newItem.setText(label.listNamesFeatures[i])
                        newItem.image = label.listFeatures[i]
                        self.addItem(newItem)
                else:
                    featuresImportance = model.feature_importances_
                    ind = np.argsort(featuresImportance)
                    ind = ind[::-1]

                    for i in ind:

                        newItem = qt.QListWidgetItem()
                        newItem.setText(label.listNamesFeatures[i])
                        newItem.image = label.listFeatures[i]

                        widget = qt.QLabel(fct.numberCalibration(featuresImportance[i]*100)+" %")
                        widget.setAlignment(Qt.AlignRight)
                        self.addItem(newItem)
                        self.setItemWidget(newItem, widget)

            except:
                traceback.print_exc(file=sys.stderr)

    def itemClickedFunction(self):

        if self.parent.settingsWidget.buttonViewFeatures.activate == False:
            self.parent.widgetImage.sliderOpacity.slider.setValue(0)
        self.parent.settingsWidget.buttonViewFeatures.setActivation(True)
        self.loadCurrentItem()

    def loadCurrentItem(self):

        if self.parent.settingsWidget.buttonViewFeatures.activate:

            image = self.currentItem().image

            try:
                image.valueMin
            except:
                valueMin, valueMax = fct.getMinMaxValue(image)
                image.valueMin = valueMin
                image.valueMax = valueMax

            self.parent.widgetImage.setImage(image)


if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook


    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    foo = FeaturesWidget()

    # foo.setImage(img)

    #foo.resize(1200, 900)

    foo.show()

    app.exec_()
