import sys
import os
import traceback

import PyQt5.QtWidgets as qt
from PyQt5 import QtCore, QtGui
from PyQt5.QtCore import pyqtSignal, Qt, QCoreApplication

import xml.etree.ElementTree as xmlet
import DatabaseFunction as Dfct
import ScriptFunction as Sfct

import UsefullWidgets as wgt

import UsefullVariables as vrb
import UsefullTexts as txt
import UsefullFunctions as fct
# import ElementIDManager as idManager
from ScriptFunction import ScriptWindow
from LutsWidget import GroupBoxLut

import PyIPSDK
import PyIPSDK.IPSDKIPLColor as colorIP
import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr
import PyIPSDK.IPSDKIPLUtility as util
import PyIPSDK.IPSDKIPLFiltering as filtering

import numpy as np

import napari

import warnings
# Ignorer tous les warnings provenant de Napari
warnings.filterwarnings("ignore", module="napari")

import time


try:
    import gc
except:
    pass

import uuid

class WidgetLabelImage(qt.QScrollArea):
    signalLabelDeletePosition = pyqtSignal(int)
    signalDoubleClickLabel = pyqtSignal(qt.QGroupBox)
    SignalNameHasChanged = pyqtSignal()

    def __init__(self, xmlElement=None, mainWindow=None,module="Classic"):
        qt.QScrollArea.__init__(self)

        self.module = module
        try:
            self.moduleType = module.moduleType
        except:
            self.moduleType = "Classic"

        self.rubberBand = qt.QRubberBand(qt.QRubberBand.Rectangle, self)
        self.origin = None

        self.setHorizontalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)
        self.xmlElement = xmlElement
        self.mainWindow = mainWindow

        self.centralWidget = qt.QGroupBox()

        self.layout = qt.QVBoxLayout()
        self.layout.setAlignment(QtCore.Qt.AlignTop)

        self.centralWidget.setLayout(self.layout)
        self.centralWidget.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.setWidget(self.centralWidget)

        self.miniatureVisible = True

        self.layout.setContentsMargins(5, 5, 20, 5)

    # def keyPressEvent(self, event):
    #     key = event.key()
    #     if key == 77:
    #         self.toggleMiniature()

    def resizeEvent(self, event):

        for num in range(self.layout.count()):

            item = self.layout.itemAt(num)
            if item is not None:
                labelImage = item.widget()
                if labelImage.objectType == "Image" and self.miniatureVisible:
                    labelImage.setFixedSize(max(1,self.width() - 22*vrb.ratio), 6 * labelImage.labelHeight + 4*vrb.ratio)
                else:
                    labelImage.setFixedSize(max(1,self.width() - 22*vrb.ratio), labelImage.labelHeight + 4*vrb.ratio)

        self.centralWidget2 = qt.QWidget()
        self.centralWidget2.setLayout(self.layout)
        self.setWidget(self.centralWidget2)

    def mousePressEvent(self, event):

        self.origin = event.pos()
        if not self.rubberBand:
            self.rubberBand = qt.QRubberBand(qt.QRubberBand.Rectangle, self)
        self.rubberBand.setGeometry(QtCore.QRect(self.origin, QtCore.QSize()))
        self.rubberBand.show()

        modifierPressed = qt.QApplication.keyboardModifiers()
        if event.button() == QtCore.Qt.LeftButton:
            try:
                if (modifierPressed & Qt.ControlModifier) != Qt.ControlModifier:
                    self.unselectAll()
            except:
                pass

        self.updateStyleSheet()

    def mouseMoveEvent(self, event):

        try:
            rectangle = QtCore.QRect(self.origin, event.pos()).normalized()
            self.rubberBand.setGeometry(rectangle)
            self.detectSelectedOjects(rectangle)
        except:
            traceback.print_exc(file=sys.stderr)

        self.updateStyleSheet()


    def mouseReleaseEvent(self, event):

        self.rubberBand.hide()
        self.updateStyleSheet()

    def addNewImage(self, name, image, functionXmlElement=None,xmlElement = None,allowDuplicate=False,id=None):

        try:
            # try:
            #     gc.collect()
            # except:
            #     pass

            if id is None:
                id = str(uuid.uuid4()).replace('-','')

            vrb.hasImage = True
            vrb.sessionIsSaved = False

            if xmlElement is None:
                xmlElementLabel = None
                label = None

                if name is None:
                    name = ""

                for extension in vrb.extensionImages:
                    name = name.replace('.' + extension, '')

                if name == '' or name is None or name == 'None':
                    # Default name
                    # defaultStart = 'OutputImage_'
                    defaultStart = 'Image_'
                    maxi = -1
                    for xmlElem in self.xmlElement:
                        elemName = Dfct.childText(xmlElem, 'Name')
                        if defaultStart in elemName:
                            elemName = elemName.split(defaultStart)
                            try:
                                maxi = max(maxi, int(elemName[1]))
                            except:
                                pass
                    name = defaultStart + str(maxi + 1)

                if allowDuplicate:
                    existAlready=False
                    i = 1
                    newName = name
                    for num in range(self.layout.count()):
                        # Find if the label already exists
                        item = self.layout.itemAt(num)
                        if item is not None:
                            labelImage = item.widget()
                            itemXmlElement = labelImage.xmlElement
                            if itemXmlElement.tag == "Image":
                                if Dfct.childText(itemXmlElement, 'Name') == name:
                                    existAlready = True
                    while existAlready:
                        existAlready = False
                        i+=1
                        newName = name + " (" + str(i) + ")"
                        for num in range(self.layout.count()):
                            # Find if the label already exists
                            item = self.layout.itemAt(num)
                            if item is not None:
                                labelImage = item.widget()
                                itemXmlElement = labelImage.xmlElement
                                if Dfct.childText(itemXmlElement, 'Name') == newName:
                                    existAlready = True
                    name = newName
                else:

                    for num in range(self.layout.count()):
                        # Find if the label already exists
                        item = self.layout.itemAt(num)
                        if item is not None:
                            labelImage = item.widget()
                            itemXmlElement = labelImage.xmlElement
                            if itemXmlElement.tag == "Image":
                                if Dfct.childText(itemXmlElement, 'Name') == name:
                                    xmlElementLabel = itemXmlElement
                                    label = labelImage

                image.name = name

                xmlElementNewLabel = xmlet.SubElement(self.xmlElement, "Image")

                # Dfct.SubElement(xmlElementNewLabel, "Name").text = name
                Dfct.SubElement(xmlElementNewLabel, "Name").text = Dfct.convertTextToAscii(name)
                Dfct.SubElement(xmlElementNewLabel, 'Name').set('ASCII', str(True))

                # Dfct.SubElement(xmlElementNewLabel, "ElementID").text = str(idManager.maxElementID(self.xmlElement) + 1)
                Dfct.SubElement(xmlElementNewLabel, "ElementID").text = id
                # Dfct.SubElement(xmlElementNewLabel, "ReferenceID").text = id
                Dfct.SubElement(xmlElementNewLabel, "OjectType").text = "Image"
                Dfct.SubElement(xmlElementNewLabel, "X").text = str(image.getSizeX())
                Dfct.SubElement(xmlElementNewLabel, "Y").text = str(image.getSizeY())
                Dfct.SubElement(xmlElementNewLabel, "Z").text = str(image.getSizeZ())
                Dfct.SubElement(xmlElementNewLabel, "C").text = str(image.getSizeC())
                Dfct.SubElement(xmlElementNewLabel, "T").text = str(image.getSizeT())
                if image.getSizeC() > 1:
                    Dfct.SubElement(xmlElementNewLabel, "ColorType").text = str(image.getColorGeometryType())
                else:
                    Dfct.SubElement(xmlElementNewLabel, "ColorType").text = "Grey"
                Dfct.SubElement(xmlElementNewLabel, "BufferType").text = str(image.getBufferType())
                Dfct.SubElement(xmlElementNewLabel, "DiskImage").text = str(image.isDiskImage())

                if image.hasGeometricCalibration():
                    calibration = image.getGeometricCalibration()
                else:
                    if image.getVolumeGeometryType() == PyIPSDK.eVGT_2d:
                        calibration = PyIPSDK.createGeometricCalibration2d(1, 1, "px")
                    else:
                        calibration = PyIPSDK.createGeometricCalibration3d(1, 1, 1, "px")

                originX = calibration.getXOriginUncalibrated()
                originY = calibration.getYOriginUncalibrated()
                originZ = calibration.getZOriginUncalibrated()

                Dfct.SubElement(xmlElementNewLabel, "OriginX").text = str(originX)
                Dfct.SubElement(xmlElementNewLabel, "OriginY").text = str(originY)
                Dfct.SubElement(xmlElementNewLabel, "OriginZ").text = str(originZ)

                if self.moduleType == 'Label Classification':
                    Dfct.SubElement(xmlElementNewLabel, 'GraphicElement')

                if functionXmlElement is not None:
                    xmlElementNewLabel.append(functionXmlElement)

                if xmlElementLabel is None:
                    Dfct.SubElement(xmlElementNewLabel, "Position").text = str(self.layout.count())
                else:
                    # if the label already exists, we delete it before creating a new one.
                    Dfct.SubElement(xmlElementNewLabel, "Position").text = Dfct.SubElement(xmlElementLabel, "Position").text

                    self.deleteLabel(label, actualize=False)

            else:
                xmlElementNewLabel = xmlElement

            # create the label

            label = LabelImage(xmlElement=xmlElementNewLabel, parent=self, image=image)
            label.id = id
            label.signalLabelDelete.connect(self.deleteLabel)
            label.signalDoubleClickLabel.connect(self.emitDoubleClickLabel)
            label.SignalNameHasChanged.connect(self.emitSignalNameHasChanged)
            self.layout.insertWidget(int(Dfct.SubElement(xmlElementNewLabel, "Position").text), label)


            self.resizeEvent(None)
            self.SignalNameHasChanged.emit()

            # self.ensureWidgetVisible(label)

            if vrb.mainWindow is not None:
                vrb.mainWindow.displayThumbailsAndRefresh()

            self.verticalScrollBar().setValue(self.verticalScrollBar().maximum())

            outputDict = {'Name': name, 'Type': 'Image'}

            vrb.dictElements[id] = [outputDict, image, functionXmlElement, label]

            return label, name

        except:

            traceback.print_exc(file=sys.stderr)

            try:
                Dfct.removeElement(self.xmlElement,xmlElementNewLabel)
            except:
                pass

    def addNewMesh(self, name, mesh, functionXmlElement=None,xmlElement = None,allowDuplicate=False,id=None):

        try:

            # try:
            #     gc.collect()
            # except:
            #     pass

            if id is None:
                id = str(uuid.uuid4()).replace('-','')

            vrb.hasImage = True
            vrb.sessionIsSaved = False

            if xmlElement is None:
                xmlElementLabel = None
                label = None

                if name is None:
                    name = ""

                for extension in vrb.extensionMesh:
                    name = name.replace('.' + extension, '')

                if name == '' or name is None or name == 'None':
                    # Default name
                    # defaultStart = 'OutputImage_'
                    defaultStart = 'Mesh_'
                    maxi = -1
                    for xmlElem in self.xmlElement:
                        elemName = Dfct.childText(xmlElem, 'Name')
                        if defaultStart in elemName:
                            elemName = elemName.split(defaultStart)
                            try:
                                maxi = max(maxi, int(elemName[1]))
                            except:
                                pass
                    name = defaultStart + str(maxi + 1)

                if allowDuplicate:
                    existAlready = False
                    i = 1
                    newName = name
                    for num in range(self.layout.count()):
                        # Find if the label already exists
                        item = self.layout.itemAt(num)
                        if item is not None:
                            labelMesh = item.widget()
                            itemXmlElement = labelMesh.xmlElement
                            if itemXmlElement.tag == "Image":
                                if Dfct.childText(itemXmlElement, 'Name') == name:
                                    existAlready = True
                    while existAlready:
                        existAlready = False
                        i += 1
                        newName = name + " (" + str(i) + ")"
                        for num in range(self.layout.count()):
                            # Find if the label already exists
                            item = self.layout.itemAt(num)
                            if item is not None:
                                labelMesh = item.widget()
                                itemXmlElement = labelMesh.xmlElement
                                if Dfct.childText(itemXmlElement, 'Name') == newName:
                                    existAlready = True
                    name = newName
                else:

                    for num in range(self.layout.count()):
                        # Find if the label already exists
                        item = self.layout.itemAt(num)
                        if item is not None:
                            labelMesh = item.widget()
                            itemXmlElement = labelMesh.xmlElement
                            if itemXmlElement.tag == "Image":
                                if Dfct.childText(itemXmlElement, 'Name') == name:
                                    xmlElementLabel = itemXmlElement
                                    label = labelMesh

                mesh.name = name

                xmlElementNewLabel = xmlet.SubElement(self.xmlElement, "Image")

                # Dfct.SubElement(xmlElementNewLabel, "Name").text = name
                Dfct.SubElement(xmlElementNewLabel, "Name").text = Dfct.convertTextToAscii(name)
                Dfct.SubElement(xmlElementNewLabel, 'Name').set('ASCII', str(True))

                # Dfct.SubElement(xmlElementNewLabel, "ElementID").text = str(idManager.maxElementID(self.xmlElement) + 1)
                Dfct.SubElement(xmlElementNewLabel, "ElementID").text = id
                # Dfct.SubElement(xmlElementNewLabel, "ReferenceID").text = id

                Dfct.SubElement(xmlElementNewLabel, "ObjectType").text = "Mesh"

                if functionXmlElement is not None:
                    xmlElementNewLabel.append(functionXmlElement)

                if xmlElementLabel is None:
                    Dfct.SubElement(xmlElementNewLabel, "Position").text = str(self.layout.count())
                else:
                    # if the label already exists, we delete it before creating a new one.
                    Dfct.SubElement(xmlElementNewLabel, "Position").text = Dfct.SubElement(xmlElementLabel, "Position").text

                    self.deleteLabel(label, actualize=False)

            else:
                xmlElementNewLabel = xmlElement

            # create the label

            label = LabelMesh(xmlElement=xmlElementNewLabel, parent=self, mesh=mesh)
            label.id = id
            label.signalLabelDelete.connect(self.deleteLabel)
            label.signalDoubleClickLabel.connect(self.emitDoubleClickLabel)
            label.SignalNameHasChanged.connect(self.emitSignalNameHasChanged)
            self.layout.insertWidget(int(Dfct.SubElement(xmlElementNewLabel, "Position").text), label)

            self.resizeEvent(None)
            self.SignalNameHasChanged.emit()

            # self.ensureWidgetVisible(label)

            self.verticalScrollBar().setValue(self.verticalScrollBar().maximum())

            outputDict = {'Name': name, 'Type': 'Mesh'}

            vrb.dictElements[id] = [outputDict, mesh, functionXmlElement, label]

            return label, name

        except:

            traceback.print_exc(file=sys.stderr)

            try:
                Dfct.removeElement(self.xmlElement, xmlElementNewLabel)
            except:
                pass

    def updateStyleSheet(self):

        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                labelImage = item.widget()
                try:
                    if self.moduleType == "Classic":
                        if labelImage == vrb.mainWindow.currentLabel:
                            qFont = QtGui.QFont()
                            qFont.setBold(True)
                            labelImage.labelName.setFont(qFont)
                            if labelImage.objectType == "Mesh":
                                if labelImage.isSelected:
                                    labelImage.setStyleSheet("QGroupBox {background-color: rgb(150,150,35);border: 3px solid rgb(6, 115, 186); }")
                                else:
                                    labelImage.setStyleSheet("QGroupBox {background-color: rgb(80,80,20);border: 3px solid rgb(6, 115, 186); }")
                            else:
                                if labelImage.isSelected:
                                    labelImage.setStyleSheet("QGroupBox {background-color: rgb(98,98,98);border: 3px solid rgb(6, 115, 186); }")
                                else:
                                    labelImage.setStyleSheet("QGroupBox {border: 3px solid rgb(6, 115, 186); }")
                        else:
                            qFont = QtGui.QFont()
                            qFont.setBold(False)
                            labelImage.labelName.setFont(qFont)
                            if labelImage.objectType == "Mesh":
                                if labelImage.isSelected:
                                    labelImage.setStyleSheet("QGroupBox {background-color: rgb(150,150,35);border: 1px solid white; }")
                                else:
                                    labelImage.setStyleSheet("QGroupBox {background-color: rgb(80,80,20);border: 1px solid white; }")
                            else:
                                if labelImage.isSelected:
                                    labelImage.setStyleSheet("QGroupBox {background-color: rgb(98,98,98);border: 1px solid white; }")
                                else:
                                    labelImage.setStyleSheet("QGroupBox {border: 1px solid white; }")
                    else:
                        if labelImage==self.module.currentLabel:
                            qFont = QtGui.QFont()
                            qFont.setBold(True)
                            labelImage.labelName.setFont(qFont)
                            if labelImage.isSelected:
                                labelImage.setStyleSheet("QGroupBox {background-color: rgb(98,98,98);border: 3px solid rgb(6, 115, 186); }")
                            else:
                                labelImage.setStyleSheet("QGroupBox {border: 3px solid rgb(6, 115, 186); }")
                        else:
                            qFont = QtGui.QFont()
                            qFont.setBold(False)
                            labelImage.labelName.setFont(qFont)
                            if labelImage.isSelected:
                                labelImage.setStyleSheet("QGroupBox {background-color: rgb(98,98,98);border: 1px solid white; }")
                            else:
                                labelImage.setStyleSheet("QGroupBox {border: 1px solid white; }")
                except:
                    traceback.print_exc(file=sys.stderr)

    def unselectAll(self):

        for i in range(self.layout.count()):
            item = self.layout.itemAt(i)
            if item is not None:
                labelImage = item.widget()
                try:
                    labelImage.setSelected(False)
                except:
                    pass

    def detectSelectedOjects(self,rectangle):

        for i in range(self.layout.count()):
            item = self.layout.itemAt(i)
            if item is not None:
                labelImage = item.widget()

                positionRef = labelImage.pos()
                position = QtCore.QPoint(positionRef.x(),positionRef.y()-self.verticalScrollBar().value())
                rectangleLabelImage = QtCore.QRect(position, labelImage.size())

                labelImage.setSelected(rectangleLabelImage.intersects(rectangle))

    def deleteLabel(self, widget, actualize=True):
        valueBefore = self.verticalScrollBar().value()

        # try:
        #     if widget.linkedLayers == []:
        #         print("layers = ", vrb.mainWindow.viewerNapari.layers)
        #         vrb.mainWindow.viewerNapari.layers.remove(widget.layer)
        #     else:
        #         for layer in widget.linkedLayers:
        #             vrb.mainWindow.viewerNapari.layers.remove(layer)
        # except:
        #     traceback.print_exc(file=sys.stderr)


        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                if item.widget() == widget:

                    if widget.objectType == "Image":
                        widget.image.clear()

                        del widget.image
                        try:
                            gc.collect()
                        except:
                            pass
                        widget.image = None

                        if widget.orthosliceX is not None:
                            vrb.mainWindow.viewerNapari.layers.remove(widget.orthosliceX)
                            vrb.mainWindow.viewerNapari.layers.remove(widget.orthosliceY)
                            vrb.mainWindow.viewerNapari.layers.remove(widget.orthosliceZ)
                            widget.orthosliceX = None
                            widget.orthosliceY = None
                            widget.orthosliceZ = None

                    elif widget.objectType == "Mesh":
                        widget.mesh.clear()

                        del widget.mesh
                        try:
                            gc.collect()
                        except:
                            pass
                        widget.mesh = None

                    self.layout.removeItem(item)

                    if len(widget.linkedLayers) == 0 and widget.layer in vrb.mainWindow.viewerNapari.layers:
                        vrb.mainWindow.viewerNapari.layers.remove(widget.layer)
                        widget.layer = None

                    else:
                        for layer in widget.linkedLayers:
                            vrb.mainWindow.viewerNapari.layers.remove(layer)
                        widget.linkedLayers = None

        self.centralWidget = qt.QWidget()
        self.centralWidget.setLayout(self.layout)
        self.setWidget(self.centralWidget)

        widgetPosition = int(Dfct.childText(widget.xmlElement, "Position"))
        Dfct.SubElement(widget.xmlElement, "Position").text = '-1'

        if actualize:
            self.actualizePosition()
            self.signalLabelDeletePosition.emit(widgetPosition)

        # idManager.actualizeAllElements(self.xmlElement)

        self.verticalScrollBar().setValue(valueBefore)

    def clearAll(self):

        while self.layout.count() != 0:
            item = self.layout.itemAt(0)
            if item is not None:
                self.deleteLabel(item.widget())

    def actualizePosition(self):

        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                widget = item.widget()
                position = Dfct.SubElement(widget.xmlElement, "Position")
                position.text = str(num)

    def emitDoubleClickLabel(self, label):
        self.signalDoubleClickLabel.emit(label)

    def emitSignalNameHasChanged(self):
        self.SignalNameHasChanged.emit()

    def toggleMiniature(self):
        self.miniatureVisible = not self.miniatureVisible
        for num in range(self.layout.count()):
            item = self.layout.itemAt(num)
            if item is not None:
                item.widget().setVisibleMiniature(self.miniatureVisible)
        self.resizeEvent(None)

    def numIDToImage(self, numID):
        for i in range(len(self.layout)):
            item = self.layout.itemAt(i)
            if item is not None:
                widget = item.widget()
                if Dfct.childText(widget.xmlElement, 'ElementID') == numID:
                    value = widget.image
                    if value is None:
                        value = widget.mesh
                    return value


class LabelImage(qt.QGroupBox):
    signalLabelDelete = pyqtSignal(qt.QGroupBox)
    signalDoubleClickLabel = pyqtSignal(qt.QGroupBox)
    SignalNameHasChanged = pyqtSignal()

    def __init__(self, xmlElement=None, parent=None, image=None):
        qt.QGroupBox.__init__(self)

        self.objectType = "Image"

        self.hasNapari = False
        self.layer = None
        self.visual = None
        self.linkedLayers = []

        self.parent = parent
        self.xmlElement = xmlElement
        self.name = Dfct.childText(self.xmlElement, "Name")
        self.image = image
        self.mesh = None
        # if image.getBufferType() not in [PyIPSDK.eIBT_Label8,
        #                                    PyIPSDK.eIBT_Label16,
        #                                    PyIPSDK.eIBT_Label32,
        #                                     PyIPSDK.eIBT_Binary]:
        #     self.imageNapari = util.copyImg(image)
        # else:
        #     self.imageNapari = image
        self.imageNapari = image

        self.orthosliceX = None
        self.orthosliceY = None
        self.orthosliceZ = None

        self.thresholdValues = []

        self.labelHeight = 20*vrb.ratio
        self.isSelected = False

        self.colorVector = [0,1,2]

        self.groupBoxLut = None
        self.widgetMultiSpectral = None

        self.initImage()

        self.customContextMenu = wgt.CustomContextMenu(self)
        if self.parent.moduleType == "Classic":
            self.customContextMenu.addAction('Properties', self.showProperties)
            self.customContextMenu.addAction('Save as...', self.saveImage)
            if vrb.isRunTime == False:
                self.customContextMenu.addAction('Generate Function Script', self.generateFunctionScript)
                self.customContextMenu.addAction('Generate Complete Script', self.generateCompleteScript)
            if image.getSizeC() == 1:
                self.customContextMenu.addAction('Change image LUT', self.changeImageLut)
            if vrb.isDigisens:
                self.customContextMenu.addAction('Add To Digisens', self.addToDigisens)
            if vrb.isORS:
                self.customContextMenu.addAction('Add To Dragonfly', self.addToDragonfly)
            if vrb.isRunTime == False:
                self.customContextMenu.addAction('Add To Machine Learning training', self.addToMachineLearningTraining)

        if image.getSizeC() > 1:
            self.customContextMenu.addAction('Channel selector',self.modifyColorChannels)

        if image.isDiskImage():
            self.customContextMenu.addAction('Load as a memory image',self.loadAsMemoryImage)

        self.scriptWindow = ScriptWindow(mainWindow=self.parent.mainWindow, xmlElement=self.xmlElement)
        # self.propertiesWindow = wgt.PropertiesWindow(self.xmlElement, self.image)
        self.propertiesWindow = wgt.PropertiesWindow(self.xmlElement)

        if self.parent.moduleType == 'Pixel Classification':
            self.customContextMenu.addAction('Import image draw', self.importImageDraw)
            self.customContextMenu.addAction('Clear draw', self.clearDraw)
            self.customContextMenu.addAction('Export result to Explorer', self.exportResultSmartSegmentation)
            # self.imageFeatures = None
            self.dictFeatures = {}
            self.listFeatures = []
            self.listNamesFeatures = []
            self.dimension = ""
            self.imageDraw = None
            self.imageProbabilities = None
            self.imageLabel = None
        if self.parent.moduleType == 'Label Classification':
            self.features = None
            self.listNamesFeatures = []
            self.currentFeaturesIndex = 0
            self.imageGrey = None
            self.vectorPrediction = None
            self.valueMinGrey = 0
            self.valueMaxGrey = 255
        if self.parent.moduleType == "Super Pixel Classification":
            self.customContextMenu.addAction('Clear draw', self.clearSuperPixelsDraw)
            self.customContextMenu.addAction('Export result to Explorer', self.exportResultSuperPixelSegmentation)
            self.imageSuperPixels = None
            self.features = []
            self.listFeatures = []
            self.resultVector = []
            self.probabilitiesVector = []

        self.contextMenuSelected = wgt.CustomContextMenuReference(self)
        self.contextMenuSelected.addAction('Delete', self.deleteSelectedLabel)
        if self.parent.moduleType == "Classic":
            self.contextMenuSelected.addAction('Create multi channels image', self.mergeChannels)
            self.contextMenuSelected.addAction('Create sequence image', self.createSequenceImage)
            self.contextMenuSelected.addAction('Z stack focus', self.applyZStackFocus)

        self.editableLabelName = wgt.EditableLabel('')
        self.labelName = self.editableLabelName.label

        self.labelEmpty = qt.QLabel("")

        self.labelIconImageDisk = wgt.LabelImage(vrb.folderImages + '/Disk_Image.png')
        self.labelIconImageDisk.setFixedSize(10*vrb.ratio,10*vrb.ratio)

        self.labelName.setContentsMargins(5, 0, 0, 0)
        if Dfct.childText(self.xmlElement, "Name") is not None:
            self.labelName.setText(Dfct.childText(self.xmlElement, "Name"))
        self.labelDelete = wgt.LabelDelete()
        self.labelDelete.setFixedSize(self.labelHeight, self.labelHeight)

        if self.image.isDiskImage() == False:
            self.labelIconImageDisk.setVisible(False)
        else:
            # self.labelName.setStyleSheet("background-color: rgb(50,50,80)")
            self.labelName.setStyleSheet("background-color: rgb(35,61,140)")
            self.labelEmpty.setStyleSheet("background-color: rgb(35,61,140)")
            self.labelDelete.style = "background-color: rgb(35,61,140); color: white"
        self.labelDelete.leaveEvent(None)

        self.labelImageMiniature = wgt.LabelPythonImage(image=self.image, parent=self)

        self.groupBoxMiniature = qt.QGroupBox()
        self.groupBoxMiniature.setFixedHeight(5 * self.labelHeight)
        self.layoutMiniature = qt.QGridLayout()
        self.layoutMiniature.addWidget(self.labelImageMiniature, 0, 0, Qt.AlignCenter)
        self.layoutMiniature.setContentsMargins(0, 0, 0, 0)
        self.layoutMiniature.setSizeConstraint(1)
        self.groupBoxMiniature.setLayout(self.layoutMiniature)
        # self.groupBoxMiniature.setStyleSheet("QGroupBox {background:white; border: 0px transparent;}")
        self.groupBoxMiniature.setStyleSheet("QGroupBox {background:white; border: 1px white;}")

        self.buttonEye = wgt.PushButtonDoubleImage(vrb.folderImages + "/eye_open.png", vrb.folderImages + "/eye_close.png")
        self.buttonEye.setFixedSize(self.labelHeight, self.labelHeight)
        self.buttonEye.setActivation(True)
        self.buttonEye.setVisible(vrb.mainWindow.widgetImage.modeViewer != "2D")

        self.buttonOrthoslices = wgt.PushButtonDoubleImage(vrb.folderImages + "/orthoslice_on_2.png",
                                                   vrb.folderImages + "/orthoslice_off_2.png")
        self.buttonOrthoslices.setFixedSize(self.labelHeight, self.labelHeight)
        self.buttonOrthoslices.setActivation(True)
        self.buttonOrthoslices.setVisible(vrb.mainWindow.widgetImage.modeViewer != "2D")

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelEmpty, 0, 0)
        self.layout.addWidget(self.editableLabelName, 0, 0)
        self.layout.addWidget(self.labelIconImageDisk, 0, 1)
        self.layout.addWidget(self.buttonEye, 0, 1, Qt.AlignRight)
        self.layout.addWidget(self.buttonOrthoslices, 0, 2, Qt.AlignRight)
        self.layout.addWidget(self.labelDelete, 0, 3, Qt.AlignRight)
        self.layout.addWidget(self.groupBoxMiniature, 1, 0, 3, 4)

        self.setLayout(self.layout)

        self.layout.setVerticalSpacing(0)
        self.layout.setContentsMargins(0, 0, 0, 0)
        self.layout.setSizeConstraint(1)

        # self.buttonEye.setVisible(vrb.mainWindow.widgetImage.modeViewer != "2D")

        # self.setStyleSheet('QGroupBox {border: 1px solid gray; }')

        #eye button
        self.buttonEye.clicked.connect(self.changeButtonEyeActivation)
        self.buttonOrthoslices.clicked.connect(self.changeButtonOrthosliceActivation)
        # delete
        self.labelDelete.mousePressEvent = self.showConfirmWidget
        self.customContextMenu.signalDelete.connect(self.showConfirmWidget)
        # double click
        self.labelName.mouseDoubleClickEvent = self.emitDoubleClickLabel
        self.labelImageMiniature.mouseDoubleClickEvent = self.emitDoubleClickLabel
        self.mouseDoubleClickEvent = self.emitDoubleClickLabel
        # right click
        self.labelName.setContextMenuPolicy(Qt.CustomContextMenu)
        self.labelName.customContextMenuRequested.connect(self.showMenu)
        # self.labelName.customContextMenuRequested.connect(self.customContextMenu.showMenu)
        self.labelImageMiniature.setContextMenuPolicy(Qt.CustomContextMenu)
        self.labelImageMiniature.customContextMenuRequested.connect(self.showMenu)
        # self.labelImageMiniature.customContextMenuRequested.connect(self.customContextMenu.showMenu)
        # text changed
        self.editableLabelName.SignalLabelTextChanged.connect(self.nameHasChanged)
        if self.groupBoxLut is not None:
            self.groupBoxLut.SignalLutChanged.connect(self.labelImageMiniature.changeImage)
        if self.widgetMultiSpectral is not None:
            self.widgetMultiSpectral.SignalMultiSpectralChanged.connect(self.labelImageMiniature.changeImage)

        self.setStyleSheet("QGroupBox {border: 1px solid white; }")

    def resizeEvent(self, event):

        self.labelEmpty.setFixedSize(max(1, self.width()), self.labelHeight)
        self.editableLabelName.setFixedSize(max(1, self.width() - 2*self.labelHeight - 15*vrb.ratio), self.labelHeight)
        height = 5 * self.labelHeight
        width = min(self.image.getSizeX() * height / self.image.getSizeY(), self.width())
        height = self.image.getSizeY() * width / self.image.getSizeX()
        self.labelImageMiniature.setFixedSize(max(1, width), height)
        # self.groupBoxMiniature.setFixedHeight( 6 * self.labelHeight )
        # self.labelImageMiniature.setFixedSize(max(0, self.width()), 5 * self.labelHeight)

    def addToNapari(self):

        if self.hasNapari == False:

            if self.image.getBufferType() not in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32]:
                if self.image.getBufferType() == PyIPSDK.eIBT_Binary:
                    self.imageNapari = util.convertImg(self.image,PyIPSDK.eIBT_UInt8)
                else:
                    # self.imageNapari = util.copyImg(self.image)
                    self.imageNapari = self.image

                if self.imageNapari.getSizeC() != 3:
                    # print("dask test!!!!")
                    # from dask import delayed
                    #
                    # lazy_load = delayed(self.imageNapari.array)
                    # reader = lazy_load()  # doesn't actually read the file
                    # ipsdkImg = reader.compute()
                    # vrb.mainWindow.viewerNapari.add_image(ipsdkImg,name=self.name)
                    # vrb.mainWindow.viewerNapari.add_image(self.imageNapari.array,name=self.name)

                    vrb.mainWindow.viewerNapari.add_image(self.imageNapari.array,name=self.id + "_")
                else:
                    self.imageNapari = fct.splitColorImage(self.image)
                    lutRed = []
                    lutGreen = []
                    lutBlue = []
                    for i in range(255):
                        lutRed.append([i / 255, 0, 0, 1])
                        lutGreen.append([0, i / 255, 0, 1])
                        lutBlue.append([0, 0, i / 255, 1])
                    colormapRed = napari.utils.Colormap(lutRed)
                    colormapGreen = napari.utils.Colormap(lutGreen)
                    colormapBlue = napari.utils.Colormap(lutBlue)

                    vrb.mainWindow.viewerNapari.add_image(self.imageNapari[0].array, name=self.id + "_Red", colormap=colormapRed)
                    vrb.mainWindow.viewerNapari.add_image(self.imageNapari[1].array, name=self.id + "_Green", colormap=colormapGreen)
                    vrb.mainWindow.viewerNapari.add_image(self.imageNapari[2].array, name=self.id + "_Blue", colormap=colormapBlue)

                    vrb.mainWindow.viewerNapari.layers.link_layers([self.id + "_Red", self.id + "_Green", self.id + "_Blue"])
                    numLayer = len(vrb.mainWindow.viewerNapari.layers)
                    self.linkedLayers = [vrb.mainWindow.viewerNapari.layers[numLayer - 3], vrb.mainWindow.viewerNapari.layers[numLayer - 2], vrb.mainWindow.viewerNapari.layers[numLayer - 1]]

            else:

                lut = self.groupBoxLut.comboBoxLut.currentData()
                currentLut = self.groupBoxLut.currentLut

                # pour assurer le fonctionnement dans napari 0.4 et 0.5
                if napari.__version__.startswith('0.4'):

                    if len(currentLut[0]) > 1000:
                        color_dict = None
                    else:
                        color_dict = {}
                        for i in range(1, len(currentLut[0])):
                            # Normaliser les couleurs et créer une couleur RGBA
                            color = np.array([
                                currentLut[0][i] / 255,  # R
                                currentLut[1][i] / 255,  # G
                                currentLut[2][i] / 255,  # B
                                1.0
                            ])

                        color_dict[int(i)] = color  # Associer une valeur de label à une couleur

                    vrb.mainWindow.viewerNapari.add_labels(self.imageNapari.array, color=color_dict, name=self.id+"_", cache=True)


                elif napari.__version__.startswith('0.5'):

                    colorArray = []
                    nbLabels = min(len(currentLut[0]),500) # if the number of label is too large, the number of label is blocked at 500
                    for i in range(0, nbLabels):
                        # Normalise the colors and create a RGBA color
                        color = np.array([
                            currentLut[0][i] / 255,  # R
                            currentLut[1][i] / 255,  # G
                            currentLut[2][i] / 255,  # B
                            1.0
                        ])
                        colorArray.append(color)  # Associer une valeur de label à une couleur

                    colormap = napari.utils.CyclicLabelColormap(name=lut["Name"],background_color=0,colors=colorArray)

                    # vrb.mainWindow.viewerNapari.add_labels(self.imageNapari.array, colormap=colormap, name=self.name, cache=True)

                vrb.mainWindow.viewerNapari.add_labels(self.imageNapari.array, colormap=colormap, name=self.id+"_", cache=True)

            numLayer = len(vrb.mainWindow.viewerNapari.layers)
            self.layer = vrb.mainWindow.viewerNapari.layers[numLayer-1]


            if self.image.getBufferType() in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32]:
                # self.layer.opacity = 1
                # self.layer.gamma = 1
                # self.layer.interpolation = "linear"
                # self.layer.depiction = "volume"
                # self.layer.rendering = "attenuated_mip"
                # self.layer.blending = "additive"
                # self.layer.attenuation = 0.5
                # self.layer.contrast_limits = [0,255]
                # self.layer.colormap=None
                # self.layer.opacity = 1
                # self.layer.depiction = "volume"
                self.layer.opacity = 1
                self.layer.rendering = "iso_categorical"
                self.layer.blending = "translucent"

                self.layer.class_keymap.clear() #deletes the default interactions with the labels layers

                if napari.__version__.startswith('0.5'):
                    self.layer.iso_gradient_mode = 'smooth'

            else:
                self.layer.opacity = 1
                self.layer.interpolation = "linear"
                self.layer.depiction = "volume"
                self.layer.rendering = "attenuated_mip"

                if self.image.getSizeC() > 1:
                    self.layer.blending = "additive"
                else:
                	self.layer.blending = "translucent"
                self.layer.attenuation = 0.05
                self.layer.contrast_limits = [self.valueMinRef,self.valueMaxRef]


            if self.image.hasGeometricCalibration():
                calib = self.image.getGeometricCalibration()
                voxsizeX = calib.getXScale()
                voxsizeY = calib.getYScale()
                voxsizeZ = calib.getZScale()
                originX = calib.getXOriginUncalibrated()
                originY = calib.getYOriginUncalibrated()
                originZ = calib.getZOriginUncalibrated()
            else:
                calib = PyIPSDK.createGeometricCalibration3d(1, 1, 1, 'px')
                self.image.setGeometricCalibration(calib)
                voxsizeX = 1
                voxsizeY = 1
                voxsizeZ = 1
                originX = calib.getXOriginUncalibrated()
                originY = calib.getYOriginUncalibrated()
                originZ = calib.getZOriginUncalibrated()

            #scaling factors to apply to the new volume created
            # minVoxelSize = min(voxsizeX,voxsizeY,voxsizeZ)
            # factorZ = voxsizeZ / minVoxelSize
            # factorY = voxsizeY / minVoxelSize
            # factorX = voxsizeX / minVoxelSize

            # translate the volume to the origin position
            if self.image.getSizeZ() <= 1:
                # self.layer.translate = [originY * voxsizeY, originX * voxsizeX]
                self.layer.translate = [originY * voxsizeY, originX * voxsizeX]
                self.layer.scale = [voxsizeY, voxsizeX]
            else:
                # self.layer.translate = [originZ*voxsizeZ, originY*voxsizeY, originX*voxsizeX]
                self.layer.translate = [originZ*voxsizeZ, originY*voxsizeY, originX*voxsizeX]
                self.layer.scale = [voxsizeZ, voxsizeY, voxsizeX]

            # self.viewerNapari.layers[self.currentLabel.numLayer].contrast_limits = [minI, maxI]
            # self.layer.attenuation = self.groupBoxRight3dViewer.imageSettings.sliderAttenuation.slider.value()

            self.thresholdValues.append(self.valueMinRef)
            self.thresholdValues.append(self.valueMaxRef)

            # modification de la fonction updateColorMap pour éviter le bug de colormap sur les images binaires
            # avec ajout du paramètre "label"
            # bug : lorsqu'on importait 2 images binaires, la première s'affichait bien en bleu, mais lorsqu'on
            # affichait la deuxième elle s'affichait avec la lut binary grey

            #vrb.mainWindow.viewer3dSettingsWidget.updateColorMap(label=self)

            # SLOW ?
            self.hasNapari = True

            visible = not self.buttonEye.activate
            if self.layer.visible != visible:
                self.layer.visible = visible

            vrb.mainWindow.viewer3dSettingsWidget.loadSettings(label=self)
            # vrb.mainWindow.viewer3dSettingsWidget.imageSettings.updateWidgets()

            # SLOW
            # print(self.layer.visible)
            # if self.layer.visible is not True:
            #     self.layer.visible = True

        # vrb.mainWindow.viewerNapari.camera.perspective = 60


    def addOrthosliceToNapari(self):
        try:
            startX_param = {
                'position': (self.image.getSizeX()/2, self.image.getSizeX()/2, self.image.getSizeX()/2),
                'normal': (0, 0, 1),
                'thickness': 1.0,
            }
            startY_param = {
                'position': (self.image.getSizeY()/2, self.image.getSizeY()/2, self.image.getSizeY()/2),
                'normal': (0, 1, 0),
                'thickness': 1.0,
            }
            startZ_param = {
                'position': (self.image.getSizeZ()/2, self.image.getSizeZ()/2, self.image.getSizeZ()/2),
                'normal': (1, 0, 0),
                'thickness': 1.0,
            }


            if self.image.getBufferType() in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32]:
                # creates new images as planes to represent the orthoslices
                self.orthosliceX = vrb.mainWindow.viewerNapari.add_labels(self.imageNapari.array,name=self.id + "_orthosliceX", blending='translucent', rendering='translucent',
                                                    depiction='plane', plane=startX_param)
                self.orthosliceY = vrb.mainWindow.viewerNapari.add_labels(self.imageNapari.array,name=self.id + "_orthosliceY", blending='translucent', rendering='translucent',
                                                    depiction='plane', plane=startY_param)
                self.orthosliceZ = vrb.mainWindow.viewerNapari.add_labels(self.imageNapari.array,name=self.id + "_orthosliceZ", blending='translucent', rendering='translucent',
                                                    depiction='plane', plane=startZ_param)

            # elif self.image.getBufferType() == PyIPSDK.eIBT_Binary:
            #
            #     # img = util.convertImg(self.image, PyIPSDK.eIBT_Int8)
            #     self.addToNapari()
            #
            #     # creates new images as planes to represent the orthoslices
            #     self.orthosliceX = vrb.mainWindow.viewerNapari.add_image(self.imageNapari.array, blending='translucent',
            #                                                              rendering='translucent', attenuation=0.1,
            #                                                              depiction='plane', plane=startX_param)
            #     self.orthosliceY = vrb.mainWindow.viewerNapari.add_image(self.imageNapari.array, blending='translucent',
            #                                                              rendering='translucent', attenuation=0.1,
            #                                                              depiction='plane', plane=startY_param)
            #     self.orthosliceZ = vrb.mainWindow.viewerNapari.add_image(self.imageNapari.array, blending='translucent',
            #                                                              rendering='translucent', attenuation=0.1,
            #                                                              depiction='plane', plane=startZ_param)
            else:

                # creates new images as planes to represent the orthoslices
                self.orthosliceX = vrb.mainWindow.viewerNapari.add_image(self.image.array,name=self.id + "_orthosliceX", blending='translucent',rendering='translucent', attenuation=0.1,
                                                                         depiction='plane', opacity=1, plane=startX_param)
                self.orthosliceY = vrb.mainWindow.viewerNapari.add_image(self.image.array,name=self.id + "_orthosliceY", blending='translucent',
                                                                         rendering='translucent', attenuation=0.1,
                                                                         depiction='plane', opacity=1, plane=startY_param)
                self.orthosliceZ = vrb.mainWindow.viewerNapari.add_image(self.image.array,name=self.id + "_orthosliceZ", blending='translucent',
                                                                         rendering='translucent', attenuation=0.1,
                                                                         depiction='plane', opacity=1, plane=startZ_param)

            #vrb.mainWindow.viewer3dSettingsWidget.orthoslicesSettings.sliderThickness.slider.setRange(1*100, self.image.getSizeX()/3*100)
            vrb.mainWindow.viewer3dSettingsWidget.orthoslicesSettings.sliderThickness.slider.setRange(1, self.image.getSizeX()/3)

            # if self.image.getBufferType() not in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32]:
            #     self.orthosliceX.blending = 'translucent'

            if self.image.hasGeometricCalibration():
                calib = self.image.getGeometricCalibration()
                voxsizeX = calib.getXScale()
                voxsizeY = calib.getYScale()
                voxsizeZ = calib.getZScale()
                originX = calib.getXOriginUncalibrated()
                originY = calib.getYOriginUncalibrated()
                originZ = calib.getZOriginUncalibrated()
            else:
                calib = PyIPSDK.createGeometricCalibration3d(1, 1, 1, 'px')
                self.image.setGeometricCalibration(calib)
                voxsizeX = 1
                voxsizeY = 1
                voxsizeZ = 1
                originX = calib.getXOriginUncalibrated()
                originY = calib.getYOriginUncalibrated()
                originZ = calib.getZOriginUncalibrated()

            # scaling factors to apply to the new volume created
            factorZ = voxsizeZ / voxsizeX
            factorY = voxsizeY / voxsizeX
            factorX = voxsizeX / voxsizeX

            # translate the volume to the origin position
            self.orthosliceX.translate = [originZ * factorX, originY * factorY, originX * factorZ]
            self.orthosliceY.translate = [originZ * factorX, originY * factorY, originX * factorZ]
            self.orthosliceZ.translate = [originZ * factorX, originY * factorY, originX * factorZ]
            self.orthosliceX.scale = [factorZ, factorY, factorX]
            self.orthosliceY.scale = [factorZ, factorY, factorX]
            self.orthosliceZ.scale = [factorZ, factorY, factorX]

            # vrb.mainWindow.viewer3dSettingsWidget.updateOrthosliceColorMap()

            vrb.mainWindow.viewer3dSettingsWidget.reorderLayers()

            vrb.mainWindow.viewer3dSettingsWidget.loadSettings(label=self)
        except:
            traceback.print_exc(file=sys.stderr)


    def changeButtonEyeActivation(self):
        self.buttonEye.changeActivation()
        self.changeVisibleLabel()

    def setButtonEyeActivation(self,visible):
        self.buttonEye.setActivation(not visible)
        self.changeVisibleLabel()

    def changeButtonOrthosliceActivation(self):
        self.buttonOrthoslices.changeActivation()
        self.changeVisibleOrthoslices()

    def setButtonOrthosliceActivation(self, visible):
        self.buttonOrthoslices.setActivation(not visible)
        self.changeVisibleOrthoslices()

    def changeVisibleLabel(self):

        if self.buttonEye.activate == False:
            self.addToNapari()
            if self.layer.visible is not True:
                self.layer.visible = True
        else:
            if self.hasNapari:
                if self.layer.visible is not False:
                    self.layer.visible = False

                    # # if the orthoslice's mode is on, no need to hide the orthoslices at the same time as the volume
                    # if self.buttonOrthoslices.activate:
                    #     vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible=False
                    #     vrb.mainWindow.viewer3dSettingsWidget.yPlane.visible=False
                    #     vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible=False


    def changeVisibleOrthoslices(self):

        if self.objectType == "Image" and not self.buttonOrthoslices.activate:
            if self.orthosliceX is None:
                self.addOrthosliceToNapari()
            self.orthosliceX.visible = True
            self.orthosliceY.visible = True
            self.orthosliceZ.visible = True

            if self == vrb.mainWindow.currentLabel:

                if vrb.mainWindow.viewer3dSettingsWidget.xPlane is not None:
                    vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible = False
                    vrb.mainWindow.viewer3dSettingsWidget.yPlane.visible = False
                    vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible = False

                # vrb.mainWindow.viewer3dSettingsWidget.spoilerOrthoslicesSettings.setVisible(True)

                # activate the button eyes so that the orthoslices appears in the viewer
                vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.setActivation(False)
                vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.setActivation(False)
                vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.setActivation(False)

        else:
            if self.orthosliceX is not None:
                self.orthosliceX.visible = False
                self.orthosliceY.visible = False
                self.orthosliceZ.visible = False

            # vrb.mainWindow.viewer3dSettingsWidget.spoilerOrthoslicesSettings.setVisible(False)

            vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.setActivation(True)
            vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.setActivation(True)
            vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.setActivation(True)
        vrb.mainWindow.viewer3dSettingsWidget.loadSettings()

    def mergeChannels(self):

        labelsSelected = self.getLabelsSelected()

        if len(labelsSelected) > 1:
            verifSize = True
            refSizeX,refSizeY,refSizeZ,refBuffer = labelsSelected[0].image.getSizeX(),labelsSelected[0].image.getSizeY(),labelsSelected[0].image.getSizeZ(),labelsSelected[0].image.getBufferType()
            for labelImage in labelsSelected:
                if labelImage.image.getSizeX() != refSizeX or labelImage.image.getSizeY() != refSizeY or labelImage.image.getSizeZ() != refSizeZ:
                    verifSize = False
            if verifSize:

                vrb.mainWindow.groupBoxProcessing.setText('Processing...\nPlease wait.')
                vrb.mainWindow.toggleGroupBoxProcessing(True)
                vrb.mainWindow.groupBoxProcessing.buttonStop.setVisible(False)
                vrb.mainWindow.groupBoxProcessing.progressBar.setVisible(False)

                sizeC = 0
                for labelImage in labelsSelected:
                    sizeC += labelImage.image.getSizeC()

                colorGeometry = PyIPSDK.ColorGeometry()
                colorGeometry.initUser(sizeC)
                volumeGeometry = PyIPSDK.VolumeGeometry()
                if refSizeZ == 1:
                    volumeGeometry.init2d()
                else:
                    volumeGeometry.init3d(refSizeZ)
                temporalGeometry = PyIPSDK.TemporalGeometry()
                temporalGeometry.initSingle()
                geometry = PyIPSDK.geometry(refBuffer, refSizeX, refSizeY, volumeGeometry, colorGeometry, temporalGeometry)
                outImage = PyIPSDK.createImage(geometry)
                util.eraseImg(outImage, 0)
                idx = 0
                for labelImage in labelsSelected:
                    for c in range(labelImage.image.getSizeC()):
                        if refSizeZ == 1:
                            plan = PyIPSDK.extractPlan(0, idx, 0, outImage)
                            newPlan = PyIPSDK.extractPlan(0, c, 0, labelImage.image)
                            newPlan = util.convertImg(newPlan, refBuffer)
                            util.copyImg(newPlan, plan)
                        else:
                            volume = PyIPSDK.extractVolume(idx, 0, outImage)
                            newVolume = PyIPSDK.extractVolume(c, 0, labelImage.image)
                            newVolume = util.convertImg(newVolume, refBuffer)
                            util.copyImg(newVolume, volume)
                        idx+=1

                functionXmlElement = xmlet.Element('FunctionCall')
                Dfct.SubElement(functionXmlElement, 'Name').text = 'MergeChannelsImage'

                label, name = self.parent.addNewImage("", outImage, functionXmlElement=functionXmlElement)
                Dfct.SubElement(label.xmlElement, 'AssociatedID').text = Dfct.childText(label.xmlElement, 'ElementID')
                vrb.mainWindow.changeCurrentXmlElement(label)

                vrb.mainWindow.updateCurrentFunction()
                vrb.mainWindow.toggleGroupBoxProcessing(False)

            else:
                self.messageBox = wgt.MessageBox("All images must have the same size along X, Y, and Z axis", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                self.messageBox.exec()

        else:
            self.messageBox = wgt.MessageBox("Please select at least 2 images", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
            self.messageBox.exec()

    def createSequenceImage(self):

        try:
            labelsSelected = self.getLabelsSelected()
            if len(labelsSelected) > 1:
                verifSize = True
                refSizeX, refSizeY, refSizeC, refSizeZ, refBuffer = labelsSelected[0].image.getSizeX(), labelsSelected[0].image.getSizeY(), labelsSelected[0].image.getSizeC(), \
                                                                    labelsSelected[0].image.getSizeZ(), labelsSelected[0].image.getBufferType()
                refGeometry = labelsSelected[0].image.getGeometry()
                for labelImage in labelsSelected:
                    if labelImage.image.getSizeX() != refSizeX or labelImage.image.getSizeY() != refSizeY or labelImage.image.getSizeC() != refSizeC or labelImage.image.getSizeZ() != refSizeZ:
                        verifSize = False
                if verifSize:
                    vrb.mainWindow.groupBoxProcessing.setText('Processing...\nPlease wait.')
                    vrb.mainWindow.toggleGroupBoxProcessing(True)
                    vrb.mainWindow.groupBoxProcessing.buttonStop.setVisible(False)
                    vrb.mainWindow.groupBoxProcessing.progressBar.setVisible(False)

                    sizeT = len(labelsSelected)
                    temporalGeometry = PyIPSDK.TemporalGeometry()
                    temporalGeometry.initSequence(sizeT)

                    colorGeometry = refGeometry.getColorGeometry()
                    volumeGeometry = refGeometry.getVolumeGeometry()

                    geometry = PyIPSDK.geometry(refBuffer, refSizeX, refSizeY, volumeGeometry, colorGeometry, temporalGeometry)
                    outImage = PyIPSDK.createImage(geometry)
                    util.eraseImg(outImage, 0)
                    idx = 0
                    for labelImage in labelsSelected:
                        if refSizeC == 1 and refSizeZ == 1:
                            outSubImage = PyIPSDK.extractPlan(0, 0, idx, outImage)
                        elif refSizeC > 1 and refSizeZ == 1:
                            outSubImage = PyIPSDK.extractColor(0, idx, outImage)
                        elif refSizeC == 1 and refSizeZ > 1:
                            outSubImage = PyIPSDK.extractVolume(0, idx, outImage)
                        else:
                            outSubImage = PyIPSDK.extractColorVolume(idx, outImage)
                        subImage = util.convertImg(labelImage.image, refBuffer)
                        util.copyImg(subImage, outSubImage)
                        idx+=1

                    functionXmlElement = xmlet.Element('FunctionCall')
                    Dfct.SubElement(functionXmlElement, 'Name').text = 'CreateSequenceImage'

                    label, name = self.parent.addNewImage("", outImage, functionXmlElement=functionXmlElement)
                    Dfct.SubElement(label.xmlElement, 'AssociatedID').text = Dfct.childText(label.xmlElement, 'ElementID')
                    vrb.mainWindow.changeCurrentXmlElement(label)

                    vrb.mainWindow.updateCurrentFunction()
                    vrb.mainWindow.toggleGroupBoxProcessing(False)

                else:
                    self.messageBox = wgt.MessageBox("All images must have the same size along X, Y, Z and C axis", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                    self.messageBox.exec()

            else:
                self.messageBox = wgt.MessageBox("Please select at least 2 images", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                self.messageBox.exec()

        except:
            vrb.mainWindow.toggleGroupBoxProcessing(False)
            traceback.print_exc(file=sys.stderr)

    def applyZStackFocus(self):

        try:
            labelsSelected = self.getLabelsSelected()

            if len(labelsSelected) > 1:
                verifSize = True
                refSizeX,refSizeY,refSizeC,refSizeT,refBuffer = labelsSelected[0].image.getSizeX(),labelsSelected[0].image.getSizeY(),labelsSelected[0].image.getSizeC(),\
                                                                labelsSelected[0].image.getSizeT(),labelsSelected[0].image.getBufferType()
                for labelImage in labelsSelected:
                    if labelImage.image.getSizeX() != refSizeX or labelImage.image.getSizeY() != refSizeY or labelImage.image.getSizeC() != refSizeC or labelImage.image.getSizeT() != refSizeT:
                        verifSize = False
                if verifSize:
                    vrb.mainWindow.groupBoxProcessing.setText('Processing...\nPlease wait.')
                    vrb.mainWindow.toggleGroupBoxProcessing(True)
                    vrb.mainWindow.groupBoxProcessing.buttonStop.setVisible(False)
                    vrb.mainWindow.groupBoxProcessing.progressBar.setVisible(False)

                    sizeZ = len(labelsSelected)

                    colorGeometryType = labelsSelected[0].image.getColorGeometryType()
                    colorGeometry = PyIPSDK.ColorGeometry()
                    if colorGeometryType == PyIPSDK.eCGT_User:
                        colorGeometry.initUser(labelsSelected[0].image.getSizeC())
                    else:
                        colorGeometry.init(colorGeometryType)
                    volumeGeometry = PyIPSDK.VolumeGeometry()
                    volumeGeometry.init3d(sizeZ)
                    temporalGeometry = PyIPSDK.TemporalGeometry()
                    if refSizeT == 1:
                        temporalGeometry.initSingle()
                    else:
                        temporalGeometry.initSequence(refSizeT)
                    geometry = PyIPSDK.geometry(refBuffer, refSizeX, refSizeY, volumeGeometry, colorGeometry, temporalGeometry)
                    outImage = PyIPSDK.createImage(geometry)
                    util.eraseImg(outImage, 0)
                    idx = 0
                    for labelImage in labelsSelected:
                        if refSizeC == 1:
                            plan = PyIPSDK.extractPlan(idx, 0, 0, outImage)
                            newPlan = PyIPSDK.extractPlan(0, 0, 0, labelImage.image)
                            newPlan = util.convertImg(newPlan, refBuffer)
                            util.copyImg(newPlan, plan)
                        else:
                            plan = PyIPSDK.extractColor(idx, 0, outImage)
                            newPlan = PyIPSDK.extractColor(0, 0, labelImage.image)
                            newPlan = util.convertImg(newPlan, refBuffer)
                            util.copyImg(newPlan, plan)
                        idx+=1

                    try:
                        file = xmlet.parse(vrb.folderInformation + "/Settings.mho")
                        self.settingsElement = file.getroot()

                        methodNum = Dfct.SubElement(self.settingsElement, "ZStackMethod").text
                        if methodNum == "0":
                            method = '"Ponderation"'
                        else:
                            method = '"Maximum"'
                        contrastSize = int(Dfct.SubElement(self.settingsElement, "ZStackContrastSize").text)
                        meanSmoothing = int(Dfct.SubElement(self.settingsElement, "ZStackMeanSmoothing").text)
                        zoomFactor = float(Dfct.SubElement(self.settingsElement, "ZStackZoomFactor").text)

                    except:
                        method = '"Ponderation"'
                        contrastSize = 10
                        meanSmoothing = 1
                        zoomFactor = 0.25

                    if method =="Ponderation":
                        resultImage = filtering.zStackFocusPonderationImg(outImage, contrastSize, meanSmoothing, zoomFactor)
                    else:
                        resultImage = filtering.zStackFocusMaximumImg(outImage, contrastSize, meanSmoothing, zoomFactor)

                    outputDict = {}
                    outputDict["Name"] = ""
                    outputDict["Type"] = "Image"

                    functionXmlElement = xmlet.Element('FunctionCall')
                    Dfct.SubElement(functionXmlElement, 'Name').text = 'InteractiveZStack'
                    paramsNode = Dfct.SubElement(functionXmlElement, 'Parameters')
                    paramNode0 = Dfct.SubElement(paramsNode, 'Parameter_0')
                    Dfct.SubElement(paramNode0, 'Value').text = str(method)
                    paramNode1 = Dfct.SubElement(paramsNode, 'Parameter_1')
                    Dfct.SubElement(paramNode1, 'Value').text = str(contrastSize)
                    paramNode2 = Dfct.SubElement(paramsNode, 'Parameter_2')
                    Dfct.SubElement(paramNode2, 'Value').text = str(meanSmoothing)
                    paramNode3 = Dfct.SubElement(paramsNode, 'Parameter_3')
                    Dfct.SubElement(paramNode3, 'Value').text = str(zoomFactor)

                    label, realOutputName = vrb.mainWindow.addResult(resultImage, outputDict, functionXmlElement)

                    xmlElement = label.xmlElement
                    Dfct.SubElement(xmlElement, 'AssociatedID').text = Dfct.childText(label.xmlElement, 'ElementID')

                    vrb.mainWindow.updateCurrentFunction()
                    vrb.mainWindow.toggleGroupBoxProcessing(False)

                else:
                    self.messageBox = wgt.MessageBox("All images must have the same size along X Y, and C axis", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                    self.messageBox.exec()
            else:
                self.messageBox = wgt.MessageBox("Please select at least 2 images", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                self.messageBox.exec()

        except:
            vrb.mainWindow.toggleGroupBoxProcessing(False)
            traceback.print_exc(file=sys.stderr)

    def showMenu(self,position):

        if self.isSelected:
            labelsSelected = self.getLabelsSelected()
            if len(labelsSelected) > 1:
                self.contextMenuSelected.showMenu(position)
            else:
                self.customContextMenu.showMenu(position)
        else:
            self.customContextMenu.showMenu(position)

    def getLabelsSelected(self):

        labelsSelected = []

        layout = self.parent.layout
        for i in range(layout.count()):
            item = layout.itemAt(i)
            if item is not None:
                labelImage = item.widget()
                try:
                    if labelImage.isSelected:
                        labelsSelected.append(labelImage)
                except:
                    pass

        return labelsSelected

    def deleteSelectedLabel(self):

        labelsToDelete = self.getLabelsSelected()

        if len(labelsToDelete) == 1:
            self.showConfirmWidget(None)
        elif len(labelsToDelete) != 0:
            messageBox = wgt.MessageBox('Are you sure you want to delete these ' + str(len(labelsToDelete)) + ' images', '', buttons=[qt.QMessageBox.Yes, qt.QMessageBox.No],icon=qt.QMessageBox.Warning)
            res = messageBox.exec()
            if res == qt.QMessageBox.Yes:
                vrb.mainWindow.toggleGroupBoxProcessing(True)
                vrb.mainWindow.groupBoxProcessing.setText('Removing images\n')
                vrb.mainWindow.groupBoxProcessing.buttonStop.setVisible(False)
                vrb.mainWindow.groupBoxProcessing.progressBar.setVisible(False)
                for labelImage in labelsToDelete:
                    labelImage.emitSignalLabelDelete()
                vrb.mainWindow.toggleGroupBoxProcessing(False)

    def mousePressEvent(self, event):

        modifierPressed = qt.QApplication.keyboardModifiers()

        if event.button() == QtCore.Qt.LeftButton:
            try:
                if (modifierPressed & Qt.ControlModifier) == Qt.ControlModifier:
                    self.changeSelected()
                else:
                    self.parent.unselectAll()
            except:
                self.setSelected(False)

        self.parent.origin =self.parent.mapFromGlobal(self.mapToGlobal(event.pos()))
        if not self.parent.rubberBand:
            self.parent.rubberBand = qt.QRubberBand(qt.QRubberBand.Rectangle, self.parent)
        self.parent.rubberBand.setGeometry(QtCore.QRect(self.parent.origin, QtCore.QSize()))
        self.parent.rubberBand.show()

        self.parent.updateStyleSheet()

    def mouseMoveEvent(self, event):

        try:
            eventPos = self.parent.mapFromGlobal(self.mapToGlobal(event.pos()))
            rectangle = QtCore.QRect(self.parent.origin, eventPos).normalized()

            self.parent.rubberBand.setGeometry(rectangle)
            self.parent.detectSelectedOjects(rectangle)

        except:
            traceback.print_exc(file=sys.stderr)

        self.parent.updateStyleSheet()

    def mouseReleaseEvent(self, event):

        self.parent.rubberBand.hide()
        self.parent.updateStyleSheet()

    def changeSelected(self):

        self.setSelected(not self.isSelected)

    def setSelected(self,boolValue):

        self.isSelected = boolValue
        # self.parent.updateStyleSheet()

    def modifyColorChannels(self):

        self.widgetMultiSpectral.updateComboboxNames()
        self.widgetMultiSpectral.show()

    def loadAsMemoryImage(self):

        vrb.mainWindow.toggleGroupBoxProcessing(True)
        vrb.mainWindow.groupBoxProcessing.setText('Loading image in memory\n')
        vrb.mainWindow.groupBoxProcessing.buttonStop.setVisible(False)
        vrb.mainWindow.groupBoxProcessing.progressBar.setVisible(False)
        qt.QApplication.processEvents()

        try:
            imageMemory = PyIPSDK.createImage(self.image)
            util.copyImg(self.image,imageMemory)
            self.image = imageMemory

            self.labelIconImageDisk.setVisible(False)
            self.labelEmpty.setStyleSheet("background-color: transparent")
            self.labelName.setStyleSheet("background-color: transparent")
            self.labelDelete.style = "background-color: solid grey; color: white"
            self.labelDelete.leaveEvent(None)

            vrb.mainWindow.widgetImage.setImage(self, changeRoi=False)

            # vrb.mainWindow.widgetImage.updateImageViewer()

            for imageViewerStandAlone in vrb.mainWindow.widgetImage.listImageViewerStandAlone:
                imageViewerStandAlone.sliderAxis.radioButtonX.setEnabled(True)
                imageViewerStandAlone.sliderAxis.radioButtonY.setEnabled(True)

        except Exception as e:
            traceback.print_exc(file=sys.stderr)
            messageBox = wgt.MessageBox("Error loading image in memory\n" + str(e), '', buttons=[qt.QMessageBox.Ok],
                                        icon=qt.QMessageBox.Warning, windowTitle="Error")
            messageBox.exec()

        vrb.mainWindow.toggleGroupBoxProcessing(False)

    def exportResultSmartSegmentation(self):

        try:
            vrb.mainWindow.moduleRF_Segmentation.changeCurrentXmlElement(self)
            vrb.mainWindow.exportResultSmartSegmentation()
        except:
            traceback.print_exc(file=sys.stderr)

    def exportResultSuperPixelSegmentation(self):

        try:
            vrb.mainWindow.moduleRF_SuperPixels.changeCurrentXmlElement(self)
            vrb.mainWindow.exportResultSuperPixelSegmentation()
        except:
            traceback.print_exc(file=sys.stderr)

    def importImageDraw(self):

        vrb.widgetImportImageLabel = wgt.WidgetImportImageLabel()

        self.parent.module.updateLabelClasses()

        vrb.widgetImportImageLabel.show()
        vrb.widgetImportImageLabel.settingsClasses = self.parent.module.xmlElement
        vrb.widgetImportImageLabel.parentLabel = self

        vrb.widgetImportImageLabel.loadComboBox()

        # try:
        #     filename = qt.QFileDialog.getOpenFileName(self, "Select your image", "C:/", "Image" + " (*.tif *.tiff)")
        #     if filename[0] != [] and filename[0] != '' and filename[0] != None:
        #         image = PyIPSDK.loadTiffImageFile(filename[0])
        #         if image.getBufferType() in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16]:
        #
        #             self.parent.module.widgetImportImageLabel.show()
        #
        #             # self.imageDraw = image
        #             # self.parent.module.actualizeImageDraw()
        #             # self.parent.module.evaluateModel()
        # except:
        #     traceback.print_exc(file=sys.stderr)


    def clearDraw(self):

        util.eraseImg(self.imageDraw, 0)
        self.parent.module.actualizeImageDraw()

        self.parent.module.model = None
        self.parent.module.imageViewer.setImageOverlay(None)
        self.parent.module.imageViewer.setImageProbabilities(None)
        self.parent.module.evaluateModel()
        self.parent.module.imageViewer.getRoiImage()

    def clearSuperPixelsDraw(self):

        self.parent.module.clearDraw()

    def addToMachineLearningTraining(self):

        image = util.copyImg(self.image)

        if vrb.mainWindow.currentModuleMachineLearning is None:

            if vrb.mainWindow.machineLearningSettingsWidget.groupBoxModules.radioButtonPixelClassification.isChecked():
                vrb.mainWindow.resetModuleRFSegmentation()
                vrb.mainWindow.machineLearningSettingsWidget.loading(image=image,name=self.name)
                fct.showWidget(vrb.mainWindow.machineLearningSettingsWidget)
            elif vrb.mainWindow.machineLearningSettingsWidget.groupBoxModules.radioButtonShapeClassification.isChecked():
                vrb.mainWindow.resetModuleRFClassification()
                vrb.mainWindow.machineLearningSettingsWidget.loading(image=image, name=self.name)
                fct.showWidget(vrb.mainWindow.machineLearningSettingsWidget)
            elif vrb.mainWindow.machineLearningSettingsWidget.groupBoxModules.radioButtonSuperPixelSegmentation.isChecked():
                vrb.mainWindow.resetModuleRFSuperPixels()
                vrb.mainWindow.machineLearningSettingsWidget.loading(image=image, name=self.name)
                fct.showWidget(vrb.mainWindow.machineLearningSettingsWidget)
        else:
            if vrb.mainWindow.machineLearningSettingsWidget.groupBoxModules.radioButtonPixelClassification.isChecked() or \
                    vrb.mainWindow.machineLearningSettingsWidget.groupBoxModules.radioButtonSuperPixelSegmentation.isChecked():
                vrb.mainWindow.currentModuleMachineLearning.addImage(image, self.name)
            elif vrb.mainWindow.machineLearningSettingsWidget.groupBoxModules.radioButtonShapeClassification.isChecked():
                if vrb.mainWindow.widgetImage.imageOverlayLabel is not None and vrb.mainWindow.widgetImage.imageOverlayLabel.image is not None \
                        and image.getBufferType not in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32]:
                    imageOverlay = util.copyImg(vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.currentData().image)
                    nameOverlay = vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.currentText()
                    vrb.mainWindow.currentModuleMachineLearning.addImage(imageOverlay, nameOverlay)
                    vrb.mainWindow.currentModuleMachineLearning.addImage(image, self.name)
                else:
                    vrb.mainWindow.currentModuleMachineLearning.addImage(image, self.name)

            fct.showWidget(vrb.mainWindow.currentModuleMachineLearning)

    def initImage(self,compute=False):

        if self.image.isDiskImage():
            if self.image.getSizeZ() == 1:
                # imageForInit = self.image
                imageForInit = fct.zoomDiskAndNormalImage(self.image,min(1,1000/self.image.getSizeX()),min(1,1000/self.image.getSizeY()))
            else:
                imageForInit = self.image.loadPlan(int(self.image.getSizeZ()/2),0,0,self.image.getSizeX(),self.image.getSizeY(),0,0,1,1)
        else:
            imageForInit = self.image

        self.valueMinRef, self.valueMaxRef = fct.getMinMaxValue(self.image,compute=compute)
        self.valueMinMask = None
        self.valueMin = self.valueMinRef
        self.valueMax = self.valueMaxRef

        if imageForInit.getSizeC() == 1:
            imageGrey = imageForInit
        elif imageForInit.getColorGeometryType() == PyIPSDK.eCGT_Rgb:
            imageGrey = colorIP.lightnessImg(imageForInit)
        else:
            imageGrey = fct.customLightness(imageForInit)

        self.frequencies = fct.getImageFrequencies(imageGrey,self.valueMin,self.valueMax)

        if self.image.getSizeZ() == 1:
            self.multiRes = fct.getMultiResolutionImage(self.image)

        if imageForInit.getSizeC() == 1:
            # self.groupBoxLut = GroupBoxLut(labelImage=imageForInit, parent=self)
            self.groupBoxLut = GroupBoxLut(labelImage=self.image, parent=self)
            try:
                if imageForInit.minIsBlack() == False and imageForInit.getBufferType() not in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32]:
                    self.groupBoxLut.comboBoxLut.setCurrentText("Inverted")
            except:
                traceback.print_exc(file=sys.stderr)

            try:
                self.groupBoxLut.comboBoxLut.setCurrentText(self.image.lut)
            except:
                pass
            try:
                self.groupBoxLut.checkBoxBackground.setChecked(not self.image.background)
            except:
                pass
            self.wdgetMultiSpectral = None
        else:
            self.groupBoxLut = None
            self.widgetMultiSpectral = wgt.WidgetMultiSpectral(image=imageForInit,parent=self)

    def changeImageLut(self):

        self.groupBoxLut.show()

    def saveImage(self):

        try:
            if self.image != None:
                defaultFolder = Dfct.childText(vrb.userPathElement, "ImportImages")
                if defaultFolder is None or defaultFolder == "" or os.path.exists(defaultFolder) == False:
                    defaultFolder = os.path.dirname(vrb.folderExplorer) + "/images"
                    if not os.path.exists(defaultFolder):
                        defaultFolder = os.path.dirname(vrb.folderExplorer) + "/data/Explorer/images"

            # filename = qt.QFileDialog.getSaveFileName(self, "Save as", vrb.folderDefault, "TIF " + 'Image' + " (*.tif)")
            filename = qt.QFileDialog.getSaveFileName(self, "Save image as", defaultFolder + "/", "TIF Image (*.tif *.tiff);;RAW Image(*.raw)")
            if filename[0] is not None and filename[0] != '':
                self.parent.mainWindow.groupBoxProcessing.setText('Saving...\nPlease wait.')
                self.parent.mainWindow.toggleGroupBoxProcessing(True)
                nameSplit = filename[0].split(".")
                extension = nameSplit[len(nameSplit) - 1]
                if extension in ["tif", "TIF", "Tif", "tiff", "TIFF", "Tiff"]:
                    PyIPSDK.saveTiffImageFile(filename[0], self.image)
                elif extension in ["raw", "RAW", "Raw"]:
                    PyIPSDK.saveRawImageFile(filename[0], self.image)
                self.parent.mainWindow.toggleGroupBoxProcessing(False)
                Dfct.SubElement(vrb.userPathElement, "ImportImages").text = os.path.dirname(filename[0])
                Dfct.saveXmlElement(vrb.userPathElement, vrb.folderInformation + "/UserPath.mho")

        except Exception as e:
            self.toggleGroupBoxProcessing(False)
            messageBox = wgt.MessageBox(str(e), '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error while saving image")
            messageBox.exec()
            traceback.print_exc(file=sys.stderr)

    def saveImageMultislices(self):

        defaultFolder = Dfct.childText(vrb.userPathElement, "SaveImageMultiSlices")
        if defaultFolder is None or defaultFolder == "" or os.path.exists(defaultFolder) == False:
            defaultFolder = os.path.dirname(vrb.folderExplorer) + "/images"
            if not os.path.exists(defaultFolder):
                defaultFolder = os.path.dirname(vrb.folderExplorer) + "/data/Explorer/images"

        filename = qt.QFileDialog.getExistingDirectory(self, "Select the folder to save your image", defaultFolder)

        folderImage = os.path.join(filename, self.name)
        if not os.path.exists(folderImage):
            os.makedirs(folderImage)

        for z in range(self.image.getSizeZ()):
            if self.image.getSizeC() == 1:
                plan = PyIPSDK.extractPlan(z, 0, 0, self.image)
            else:
                plan = PyIPSDK.extractColor(z, 0, self.image)
            plan = util.copyImg(plan)

            length = int(np.log10(self.image.getSizeZ()-1))+1

            PyIPSDK.saveTiffImageFile(folderImage + "/Slice_"+fct.writeWithZeros(z,length)+".tif", plan)

        if filename != "" and filename is not None:
            Dfct.SubElement(vrb.userPathElement, "SaveImageMultiSlices").text = filename
            Dfct.saveXmlElement(vrb.userPathElement, vrb.folderInformation + "/UserPath.mho")


    def showConfirmWidget(self, event=None):

        if event is None or event.button() == QtCore.Qt.LeftButton:

            modifierPressed = qt.QApplication.keyboardModifiers()

            try:
                # if keyboard.is_pressed("Ctrl"):
                if (modifierPressed & Qt.ControlModifier) == Qt.ControlModifier:
                    self.emitSignalLabelDelete()
                else:
                    messageBox = wgt.MessageBox('Are you sure you want to delete the image: ' + str(Dfct.childText(self.xmlElement, "Name")), '', buttons=[qt.QMessageBox.Yes, qt.QMessageBox.No], icon=qt.QMessageBox.Warning)
                    res = messageBox.exec()
                    if res == qt.QMessageBox.Yes:
                        self.emitSignalLabelDelete()
            except:
                messageBox = wgt.MessageBox('Are you sure you want to delete the image: ' + str(Dfct.childText(self.xmlElement, "Name")), '', buttons=[qt.QMessageBox.Yes, qt.QMessageBox.No],icon=qt.QMessageBox.Warning)
                res = messageBox.exec()
                if res == qt.QMessageBox.Yes:
                    self.emitSignalLabelDelete()

    def showProperties(self):
        if self.image is not None:
            self.propertiesWindow.displayProperties()

    def emitSignalLabelDelete(self):
        self.signalLabelDelete.emit(self)

    def emitDoubleClickLabel(self, event):
        self.signalDoubleClickLabel.emit(self)

    def setImage(self, image, visible=True):
        self.image = image
        self.labelImageMiniature.image = image
        self.setVisibleMiniature(visible)

    def setVisibleMiniature(self, visible):
        self.groupBoxMiniature.setVisible(visible)

    def nameHasChanged(self):

        newName = self.labelName.text()

        Dfct.SubElement(self.xmlElement, "Name").text = Dfct.convertTextToAscii(newName)
        Dfct.SubElement(self.xmlElement, 'Name').set('ASCII', str(True))
        self.name = newName

        # Dfct.SubElement(self.xmlElement, 'Name').text = self.labelName.text()

        self.SignalNameHasChanged.emit()

    def generateFunctionScript(self):
        script = Sfct.functionScript(self.xmlElement, self.parent.xmlElement)
        self.scriptWindow.setText(script)
        self.scriptWindow.pushButtonEditMacro.setVisible(False)
        self.scriptWindow.close()
        self.scriptWindow.show()

    def generateCompleteScript(self):
        script = Sfct.completeScript(self.xmlElement, self.parent.xmlElement)
        self.scriptWindow.setText(script)
        self.scriptWindow.pushButtonEditMacro.setVisible(True)
        self.scriptWindow.close()
        self.scriptWindow.show()

    # def setStyleSelected(self, isSelected):
    #     if not isSelected:
    #         qFont = QtGui.QFont()
    #         qFont.setBold(False)
    #         self.labelName.setFont(qFont)
    #         # self.setStyleSheet("QGroupBox {border: 1px white; }")
    #         self.setStyleSheet("QGroupBox {border: 1px solid white; }")
    #
    #     else:
    #         qFont = QtGui.QFont()
    #         qFont.setBold(True)
    #         self.labelName.setFont(qFont)
    #         # self.setStyleSheet("QGroupBox {border: 3px solid blue; }")
    #         self.setStyleSheet("QGroupBox {border: 3px solid rgb(6, 115, 186); }")

    def addToDragonfly(self):

        try:

            import PyIPSDK

            if self.image.getBufferType() not in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16, PyIPSDK.eIBT_Label32, PyIPSDK.eIBT_Binary]:

                from ORSModel import createChannelFromNumpyArray
                import PyIPSDK

                # create and initialize channel from array
                dragonFlyChannel = createChannelFromNumpyArray(self.image.array)
                if Dfct.SubElement(self.xmlElement, 'Name').text != "":
                    dragonFlyChannel.setTitle(Dfct.SubElement(self.xmlElement, 'Name').text)
                    if (self.image.hasGeometricCalibration()):
                        calib = self.image.getGeometricCalibration()
                        dragonFlyChannel.setXSpacing(calib.getXScale())
                        dragonFlyChannel.setYSpacing(calib.getYScale())
                        dragonFlyChannel.setZSpacing(calib.getZScale())
                dragonFlyChannel.publish()

            else:

                from ORSModel import createChannelFromNumpyArray
                from ORSModel import orsVect
                import PyIPSDK
                import PyIPSDK.IPSDKIPLUtility as util

                imageGrey = util.convertImg(self.image, PyIPSDK.eIBT_UInt16)

                dragonFlyChannel = createChannelFromNumpyArray(imageGrey.array)
                multiROI = dragonFlyChannel.getAsMultiROIInArea(0, 0, 0, dragonFlyChannel.getXSize() - 1, dragonFlyChannel.getYSize() - 1, dragonFlyChannel.getZSize() - 1, None, None)
                if multiROI != None:
                    multiROI.setTitle(Dfct.SubElement(self.xmlElement, 'Name').text)
                    multiROI.setOrigin(orsVect(0, 0, 0))
                    if (self.image.hasGeometricCalibration()):
                        calib = self.image.getGeometricCalibration()
                        multiROI.setXSpacing(calib.getXScale())
                        multiROI.setYSpacing(calib.getYScale())
                        multiROI.setZSpacing(calib.getZScale())

                multiROI.publish()
                dragonFlyChannel.deleteObject()

        except Exception as e:
            print(e)

    def addToDigisens(self):

        try:
            import digisens as ds
            import PyIPSDK
            import numpy

            semantic = 'DENSITY'
            volumeType = 'UINT8'
            shaderName = 'porosity'
            shader2DName = 'porosity'
            # shaderName possibility : 'metallic' 'plaster' porosity' 'label' 'AIP' 'minIP' 'MIP' 'normals' 'translucent' 'transparent'
            # shader2DName possibility : 'label' 'porosity' 'AIP' 'minIP' 'MIP'

            bufferType = self.image.getBufferType()
            if bufferType == PyIPSDK.eIBT_UInt8:
                volumeType = 'UINT8'
            elif bufferType == PyIPSDK.eIBT_Int8:
                volumeType = 'INT8'
            elif bufferType == PyIPSDK.eIBT_UInt16:
                volumeType = 'UINT16'
            elif bufferType == PyIPSDK.eIBT_Int16:
                volumeType = 'INT16'
            elif bufferType == PyIPSDK.eIBT_UInt32:
                volumeType = 'UINT32'
            elif bufferType == PyIPSDK.eIBT_Int32:
                volumeType = 'INT32'
            elif bufferType == PyIPSDK.eIBT_Real32:
                volumeType = 'FLOAT32'
            elif bufferType == PyIPSDK.eIBT_Binary:
                volumeType = 'UINT8'
                semantic = 'MASK'
            elif bufferType == PyIPSDK.eIBT_Label8:
                volumeType = 'UINT8'
                semantic = 'LABEL'
                shaderName = 'porosity'
                shader2DName = 'porosity'
            elif bufferType == PyIPSDK.eIBT_Label16:
                volumeType = 'UINT16'
                semantic = 'LABEL'
                shaderName = 'porosity'
                shader2DName = 'porosity'
            elif bufferType == PyIPSDK.eIBT_Label32:
                volumeType = 'UINT32'
                semantic = 'LABEL'
                shaderName = 'porosity'
                shader2DName = 'porosity'

            try:
                vol = ds.dvolume.fromArray(Dfct.SubElement(self.xmlElement, 'Name').text + ".vol", self.image.array, model=vrb.model, type=volumeType, semantic=semantic)
            except:
                vol = ds.dvolume.fromArray(Dfct.SubElement(self.xmlElement, 'Name').text + ".vol", self.image.array, model=None, type=volumeType, semantic=semantic)

            vol.setColors(numpy.asarray(self.groupBoxLut.randomLut))
            vol.shaderName = shaderName
            vol.shader2DName = shader2DName

            if self.image.hasGeometricCalibration() and not model:
                # For Digisens, we always shall have millimeters
                # QUESTION: Shall we handle units here too ? How ?
                # Problem: IPSDK does not handle position or rotation ... so they are lost in this case!
                try:
                    calib = self.image.getGeometricCalibration()
                    vol.volume.scale = ds.vec3f(calib.getXScale(), calib.getYScale(), calib.getZScale())
                except:
                    pass

            return vol

        except:
            pass


class LabelMesh(qt.QGroupBox):
    signalLabelDelete = pyqtSignal(qt.QGroupBox)
    signalDoubleClickLabel = pyqtSignal(qt.QGroupBox)
    SignalNameHasChanged = pyqtSignal()

    def __init__(self, xmlElement=None, parent=None, mesh=None):
        qt.QGroupBox.__init__(self)

        self.orthosliceX = None
        self.orthosliceY = None
        self.orthosliceZ = None
        self.nbLabels = 0

        self.hasNapari = False
        self.layer = None
        self.visual = None
        self.linkedLayers = []

        self.surface = None
        self.startValues = None
        self.endValues = None
        self.nbFaces = None
        self.nbFacesSimplify = None
        self.meshCenter = None
        self.ratio = 0
        self.minFaces = 1000

        self.objectType = "Mesh"

        self.parent = parent
        self.xmlElement = xmlElement
        self.name = Dfct.childText(self.xmlElement, "Name")
        self.mesh = mesh
        self.image = None

        self.labelHeight = 20 * vrb.ratio
        self.isSelected = False

        self.customContextMenu = wgt.CustomContextMenu(self)

        # self.customContextMenu.addAction('Properties', self.showProperties)
        self.customContextMenu.addAction('Generate Function Script', self.generateFunctionScript)
        self.customContextMenu.addAction('Generate Complete Script', self.generateCompleteScript)
        self.customContextMenu.addAction('Save as Bin or Xml', self.saveMeshAsXmlOrBin)
        self.customContextMenu.addAction('Save as STL', self.saveMeshAsStl)

        self.editableLabelName = wgt.EditableLabel('')
        self.labelName = self.editableLabelName.label

        self.labelEmpty = qt.QLabel("")

        self.labelIconMesh = wgt.LabelImage(vrb.folderImages + '/Mesh.png')
        self.labelIconMesh.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.labelName.setContentsMargins(5, 0, 0, 0)
        if Dfct.childText(self.xmlElement, "Name") is not None:
            self.labelName.setText(Dfct.childText(self.xmlElement, "Name"))
        self.labelDelete = wgt.LabelDelete()
        self.labelDelete.setFixedSize(self.labelHeight, self.labelHeight)

        # self.labelName.setStyleSheet("background-color: rgb(50,50,80)")
        # self.labelName.setStyleSheet("background-color: rgb(80,80,20)")
        # self.labelEmpty.setStyleSheet("background-color: rgb(80,80,20)")
        # self.labelDelete.style = "background-color: rgb(80,80,20); color: white"
        self.labelDelete.leaveEvent(None)

        self.scriptWindow = ScriptWindow(mainWindow=self.parent.mainWindow, xmlElement=self.xmlElement)
        # self.propertiesWindow = wgt.PropertiesWindow(self.xmlElement)


        self.buttonEye = wgt.PushButtonDoubleImage(vrb.folderImages + "/eye_open.png", vrb.folderImages + "/eye_close.png")
        self.buttonEye.setFixedSize(self.labelHeight, self.labelHeight)
        self.buttonEye.setVisible(True)
        self.buttonEye.setActivation(True)
        self.buttonEye.setVisible(vrb.mainWindow.widgetImage.modeViewer != "2D")

        self.contextMenuSelected = wgt.CustomContextMenuReference(self)
        self.contextMenuSelected.addAction('Delete', self.deleteSelectedLabel)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelEmpty, 0, 0)
        self.layout.addWidget(self.editableLabelName, 0, 0)
        self.layout.addWidget(self.buttonEye, 0, 1, Qt.AlignRight)
        self.layout.addWidget(self.labelIconMesh, 0, 2)

        self.layout.addWidget(self.labelDelete, 0, 3, Qt.AlignRight)

        self.setLayout(self.layout)

        self.layout.setVerticalSpacing(0)
        self.layout.setContentsMargins(0, 0, 0, 0)
        self.layout.setSizeConstraint(1)

        # self.buttonEye.setVisible(vrb.mainWindow.widgetImage.modeViewer != "2D")

        # self.setStyleSheet('QGroupBox {border: 1px solid gray; }')

        # eye button
        self.buttonEye.clicked.connect(self.changeButtonEyeActivation)
        # delete
        self.labelDelete.mousePressEvent = self.showConfirmWidget
        self.customContextMenu.signalDelete.connect(self.showConfirmWidget)
        # double click
        self.labelName.mouseDoubleClickEvent = self.emitDoubleClickLabel
        self.mouseDoubleClickEvent = self.emitDoubleClickLabel
        # right click
        self.labelName.setContextMenuPolicy(Qt.CustomContextMenu)
        self.labelName.customContextMenuRequested.connect(self.showMenu)
        # self.labelName.customContextMenuRequested.connect(self.customContextMenu.showMenu)
        # self.labelImageMiniature.customContextMenuRequested.connect(self.customContextMenu.showMenu)
        # text changed
        self.editableLabelName.SignalLabelTextChanged.connect(self.nameHasChanged)

        self.setStyleSheet("QGroupBox {border: 1px solid white;background-color: rgb(80,80,20) }")

    def resizeEvent(self, event):

        self.labelEmpty.setFixedSize(max(1, self.width()), self.labelHeight)
        self.editableLabelName.setFixedSize(max(1, self.width() - 2 * self.labelHeight - 15 * vrb.ratio), self.labelHeight)

    def changeButtonEyeActivation(self):
        self.buttonEye.changeActivation()
        self.changeVisibleLabel()

    def setButtonEyeActivation(self, visible):
        self.buttonEye.setActivation(not visible)
        self.changeVisibleLabel()

    def changeVisibleLabel(self):

        if self.buttonEye.activate == False:
            self.addToNapari()
            if self.layer.visible is not True:
                self.layer.visible = True
        else:
            if self.hasNapari:
                if self.layer.visible is not False:
                    self.layer.visible = False

    def addToNapari(self):

        if self.hasNapari == False:

            vertices, faces, values, nbFaces = fct.createSurfaceForNapari(self.mesh,ratio=0)


            self.nbFaces = nbFaces
            self.nbFacesSimplify = len(faces)

            self.startValues = np.min(vertices, axis=0)
            self.endValues = np.max(vertices, axis=0)

            sizeX = round(self.endValues[2]) - int(self.startValues[2]) - 1
            sizeY = round(self.endValues[1]) - int(self.startValues[1]) - 1
            sizeZ = round(self.endValues[0]) - int(self.startValues[0]) - 1
            self.meshCenter = [sizeX / 2, sizeY / 2, sizeZ / 2]

            self.surface = (vertices, faces, values)

            self.nbLabels = np.max(values)

            if self.nbLabels > 1:
                colormap = fct.createLabelLutNapari(self.nbLabels)
                vrb.mainWindow.viewerNapari.add_surface(self.surface, name=self.id+"_",colormap=colormap)
            else:
                vrb.mainWindow.viewerNapari.add_surface(self.surface, name=self.id+"_")

            numLayer = len(vrb.mainWindow.viewerNapari.layers)
            self.layer = vrb.mainWindow.viewerNapari.layers[numLayer-1]

            try:
                scaleX = self.mesh.calibration.getXScale()
                scaleY = self.mesh.calibration.getYScale()
                scaleZ = self.mesh.calibration.getZScale()
                originX = self.mesh.calibration.getXOriginUncalibrated()
                originY = self.mesh.calibration.getYOriginUncalibrated()
                originZ = self.mesh.calibration.getZOriginUncalibrated()

                # self.layer.translate = [originZ, originY, originX]
                self.layer.translate = [originZ*scaleZ, originY*scaleY, originX*scaleX]
            except:
                pass

            try:
                calibration = self.mesh.calibration
                xScale = calibration.getXScale()
                yScale = calibration.getYScale()
                zScale = calibration.getZScale()
                if xScale != 1 or yScale != 1 or zScale != 1:
                    minScale = min(zScale, yScale, xScale)
                    self.layer.scale = [zScale/minScale, yScale/minScale, xScale/minScale]
                else:
                    vrb.mainWindow.groupMenu.groupCalibration.applyCalibration()
            except:
                vrb.mainWindow.groupMenu.groupCalibration.applyCalibration()

            self.visual = vrb.mainWindow.viewerNapariQt.layer_to_visual[self.layer]

            self.layer.opacity = 1
            self.layer.gamma = 1
            self.layer.blending = "translucent"
            self.layer.shading = "flat"
            self.layer.wireframe = {'visible': False, 'color': (1.,0,0,1.), 'width': 1}


            # self.layer.scale = [4, 1, 1]

            # SLOW ?
            self.hasNapari = True

            # print("Load settings")
            # vrb.mainWindow.viewer3dSettingsWidget.loadSettings(label=self)
            vrb.mainWindow.cameraChanged()

            visible = not self.buttonEye.activate
            if self.layer.visible != visible:
                self.layer.visible = visible

            self.layer.refresh()

    def showConfirmWidget(self, event=None):

        modifierPressed = qt.QApplication.keyboardModifiers()

        if event is None or event.button() == QtCore.Qt.LeftButton:
            try:
                if (modifierPressed & Qt.ControlModifier) == Qt.ControlModifier:
                    self.emitSignalLabelDelete()
                else:
                    messageBox = wgt.MessageBox('Are you sure you want to delete the mesh: ' + str(Dfct.childText(self.xmlElement, "Name")), '', buttons=[qt.QMessageBox.Yes, qt.QMessageBox.No], icon=qt.QMessageBox.Warning)
                    res = messageBox.exec()
                    if res == qt.QMessageBox.Yes:
                        self.emitSignalLabelDelete()
            except:
                messageBox = wgt.MessageBox('Are you sure you want to delete the mesh: ' + str(Dfct.childText(self.xmlElement, "Name")), '', buttons=[qt.QMessageBox.Yes, qt.QMessageBox.No],icon=qt.QMessageBox.Warning)
                res = messageBox.exec()
                if res == qt.QMessageBox.Yes:
                    self.emitSignalLabelDelete()

    def emitSignalLabelDelete(self):

        self.signalLabelDelete.emit(self)

    def emitDoubleClickLabel(self, event):

        self.signalDoubleClickLabel.emit(self)

    def showMenu(self,position):

        self.customContextMenu.showMenu(position)

    def nameHasChanged(self):

        newName = self.labelName.text()

        Dfct.SubElement(self.xmlElement, "Name").text = Dfct.convertTextToAscii(newName)
        Dfct.SubElement(self.xmlElement, 'Name').set('ASCII', str(True))
        self.name = newName

        # Dfct.SubElement(self.xmlElement, 'Name').text = self.labelName.text()

        self.SignalNameHasChanged.emit()

    # def showProperties(self):
    #
    #     if self.mesh is not None:
    #         self.propertiesWindow.displayProperties()

    def generateFunctionScript(self):

        script = Sfct.functionScript(self.xmlElement, self.parent.xmlElement)
        self.scriptWindow.setText(script)
        self.scriptWindow.pushButtonEditMacro.setVisible(False)
        self.scriptWindow.close()
        self.scriptWindow.show()

    def generateCompleteScript(self):

        script = Sfct.completeScript(self.xmlElement, self.parent.xmlElement)
        self.scriptWindow.setText(script)
        self.scriptWindow.pushButtonEditMacro.setVisible(True)
        self.scriptWindow.close()
        self.scriptWindow.show()

    def saveMeshAsXmlOrBin(self):

        defaultFolder = Dfct.childText(vrb.userPathElement, "SaveStl")
        if defaultFolder is None or defaultFolder == "" or os.path.exists(defaultFolder) == False:
            defaultFolder = ""

        filename = qt.QFileDialog.getSaveFileName(self, "Save shape as", defaultFolder + "/", "Bin (*.bin);;Xml (*.xml)")

        if filename[0] != None and filename[0] != '':
            try:
                vrb.mainWindow.groupBoxProcessing.setText('Saving shapes...\nPlease wait.')
                vrb.mainWindow.groupBoxProcessing.buttonStop.setVisible(False)
                vrb.mainWindow.groupBoxProcessing.progressBar.setVisible(False)
                vrb.mainWindow.toggleGroupBoxProcessing(True)
                qt.QApplication.processEvents()

                if filename[1] == 'Bin (*.bin)':
                    PyIPSDK.writeToBinaryFile(filename[0], self.mesh)
                if filename[1] == 'Xml (*.xml)':
                    PyIPSDK.writeToXmlFile(filename[0], self.mesh)

                vrb.mainWindow.toggleGroupBoxProcessing(False)
                Dfct.SubElement(vrb.userPathElement, "SaveStl").text = os.path.dirname(filename[0])
                Dfct.saveXmlElement(vrb.userPathElement, vrb.folderInformation + "/UserPath.mho")
            except:
                import traceback
                traceback.print_exc(file=sys.stderr)

    def saveMeshAsStl(self):

        defaultFolder = Dfct.childText(vrb.userPathElement, "SaveStl")
        if defaultFolder is None or defaultFolder == "" or os.path.exists(defaultFolder) == False:
            defaultFolder = ""

        filename = qt.QFileDialog.getSaveFileName(self, "Save shape as", defaultFolder + "/", "Binary (*.stl);;ASCII (*.stl)")

        if filename[0] != None and filename[0] != '':
            try:
                vrb.mainWindow.groupBoxProcessing.setText('Saving shapes...\nPlease wait.')
                vrb.mainWindow.groupBoxProcessing.buttonStop.setVisible(False)
                vrb.mainWindow.groupBoxProcessing.progressBar.setVisible(False)
                vrb.mainWindow.toggleGroupBoxProcessing(True)
                qt.QApplication.processEvents()
                if filename[1] == 'Binary (*.stl)':
                    PyIPSDK.saveToStLBinary(filename[0], self.mesh.getColl(), True)
                elif filename[1] == 'ASCII (*.stl)':
                    basename = os.path.basename(filename[0])
                    basename = basename.split(".")[0]
                    PyIPSDK.saveToStLAscii(filename[0], self.mesh.getColl(), True, basename)
                vrb.mainWindow.toggleGroupBoxProcessing(False)
                Dfct.SubElement(vrb.userPathElement, "SaveStl").text = os.path.dirname(filename[0])
                Dfct.saveXmlElement(vrb.userPathElement, vrb.folderInformation + "/UserPath.mho")
            except:
                import traceback
                traceback.print_exc(file=sys.stderr)

    def showMenu(self,position):

        if self.isSelected:
            labelsSelected = self.getLabelsSelected()
            if len(labelsSelected) > 1:
                self.contextMenuSelected.showMenu(position)
            else:
                self.customContextMenu.showMenu(position)
        else:
            self.customContextMenu.showMenu(position)

    def getLabelsSelected(self):

        labelsSelected = []

        layout = self.parent.layout
        for i in range(layout.count()):
            item = layout.itemAt(i)
            if item is not None:
                labelImage = item.widget()
                try:
                    if labelImage.isSelected:
                        labelsSelected.append(labelImage)
                except:
                    pass

        return labelsSelected

    def deleteSelectedLabel(self):

        labelsToDelete = self.getLabelsSelected()

        if len(labelsToDelete) == 1:
            self.showConfirmWidget(None)
        elif len(labelsToDelete) != 0:
            messageBox = wgt.MessageBox('Are you sure you want to delete these ' + str(len(labelsToDelete)) + ' images', '', buttons=[qt.QMessageBox.Yes, qt.QMessageBox.No],icon=qt.QMessageBox.Warning)
            res = messageBox.exec()
            if res == qt.QMessageBox.Yes:
                vrb.mainWindow.toggleGroupBoxProcessing(True)
                vrb.mainWindow.groupBoxProcessing.setText('Removing images\n')
                vrb.mainWindow.groupBoxProcessing.buttonStop.setVisible(False)
                vrb.mainWindow.groupBoxProcessing.progressBar.setVisible(False)
                for labelImage in labelsToDelete:
                    labelImage.emitSignalLabelDelete()
                vrb.mainWindow.toggleGroupBoxProcessing(False)

    def mousePressEvent(self, event):

        modifierPressed = qt.QApplication.keyboardModifiers()

        if event.button() == QtCore.Qt.LeftButton:
            try:
                if (modifierPressed & Qt.ControlModifier) == Qt.ControlModifier:
                    self.changeSelected()
                else:
                    self.parent.unselectAll()
            except:
                self.setSelected(False)

        self.parent.origin =self.parent.mapFromGlobal(self.mapToGlobal(event.pos()))
        if not self.parent.rubberBand:
            self.parent.rubberBand = qt.QRubberBand(qt.QRubberBand.Rectangle, self.parent)
        self.parent.rubberBand.setGeometry(QtCore.QRect(self.parent.origin, QtCore.QSize()))
        self.parent.rubberBand.show()

        self.parent.updateStyleSheet()

    def mouseMoveEvent(self, event):

        try:
            eventPos = self.parent.mapFromGlobal(self.mapToGlobal(event.pos()))
            rectangle = QtCore.QRect(self.parent.origin, eventPos).normalized()

            self.parent.rubberBand.setGeometry(rectangle)
            self.parent.detectSelectedOjects(rectangle)

        except:
            traceback.print_exc(file=sys.stderr)

        self.parent.updateStyleSheet()

    def mouseReleaseEvent(self, event):

        self.parent.rubberBand.hide()
        self.parent.updateStyleSheet()

    def changeSelected(self):

        self.setSelected(not self.isSelected)

    def setSelected(self,boolValue):

        self.isSelected = boolValue
        # self.parent.updateStyleSheet()



if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    foo = WidgetLabelImage()
    foo.show()

    app.exec_()
