import PyQt5.QtWidgets as qt
from PyQt5 import QtCore
from PyQt5 import QtGui
from PyQt5.QtCore import pyqtSignal,QPointF,Qt,QCoreApplication

import UsefullWidgets as wgt
import UsefullVariables as vrb
import DatabaseFunction as Dfct
import UsefullTexts as txt
import UsefullFunctions as fct

class GraphicsView(qt.QGraphicsView):

    def __init__(self, scene, parent=None):
        qt.QGraphicsView.__init__(self)

        self.image = None
        self.zoomFactor = 1
        self.mousePressed = False

        self.realStartPointX = 0
        self.realStartPointY = 0
        self.refPosX = None
        self.refPosY = None

        self.setHorizontalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)
        self.setVerticalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)

        self.scene = scene
        self.setScene(scene)
        self.setBackgroundBrush(QtCore.Qt.black)

        self.setContentsMargins(0,0,0,0)

class WidgetSettings(qt.QWidget):

    signalValueChanged = pyqtSignal()

    def __init__(self):
        qt.QWidget.__init__(self)

        self.modif = True

        self.margins = 10
        self.QColor = QtGui.QColor(255, 255, 255)
        self.penLine = QtGui.QPen(self.QColor, 2)

        self.scene = qt.QGraphicsScene(self)
        self.graphicsView = GraphicsView(self.scene)

        self.groupBoxEmpty = qt.QGroupBox()
        self.groupBoxEmpty2 = qt.QGroupBox()
        self.labelBrightness = qt.QLabel("Brightness")
        self.labelBrightness.setAlignment(QtCore.Qt.AlignRight | QtCore.Qt.AlignVCenter)
        self.labelBrightness.setContentsMargins(5, 5, 5, 5)

        self.labelContrast = qt.QLabel("Contrast")
        self.labelContrast.setAlignment(QtCore.Qt.AlignBottom | QtCore.Qt.AlignRight)
        self.labelContrast.setContentsMargins(5,5,5,5)
        #self.labelContrast.setStyleSheet('QLabel {background-color: red;}')

        self.lineEditBrightness = wgt.CustomLineEdit(mask=vrb.maskNumbers)

        self.lineEditContrast = wgt.CustomLineEdit(mask=vrb.maskNumbers)

        self.sliderBrightness = qt.QSlider(QtCore.Qt.Vertical)
        self.sliderBrightness.setInvertedAppearance(True)
        self.sliderBrightness.setMinimum(0)
        self.sliderBrightness.setMaximum(100)

        self.sliderContrast = qt.QSlider(QtCore.Qt.Horizontal)
        self.sliderContrast.setMinimum(0)
        self.sliderContrast.setMaximum(100)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelContrast,1,0,QtCore.Qt.AlignCenter)
        self.layout.addWidget(self.lineEditContrast,2,0,QtCore.Qt.AlignCenter)

        self.layout.addWidget(self.labelBrightness,0,1)#,QtCore.Qt.AlignRight)
        self.layout.addWidget(self.graphicsView,1,1)
        self.layout.addWidget(self.sliderContrast,2,1)

        self.layout.addWidget(self.lineEditBrightness,0,2)
        self.layout.addWidget(self.sliderBrightness,1,2,QtCore.Qt.AlignCenter)

        #self.layout.setSizeConstraint(1)
        self.layout.setSpacing(0)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(self.margins, self.margins, self.margins, self.margins)

        self.setFixedSize(358*vrb.ratio,319*vrb.ratio)

        self.sliderBrightness.valueChanged.connect(self.changeLineEditBrightness)
        self.sliderContrast.valueChanged.connect(self.changeLineEditContrast)
        self.lineEditContrast.SignalTextEditedVerif.connect(self.changeSliderContrast)
        self.lineEditBrightness.SignalTextEditedVerif.connect(self.changeSliderBrightness)

    def resizeEvent(self,event):

        width = self.width()- 2*self.margins
        height = self.height()- 2*self.margins

        self.labelContrast.setFixedSize(width*18/100,height*85/100)
        self.lineEditContrast.setFixedSize(width*10/100,height*5/100)

        self.labelBrightness.setFixedSize(width * 65 / 100, height * 8 / 100)
        self.graphicsView.setFixedSize(self.width()*65/100,self.height()*73/100)
        self.sliderContrast.setFixedSize(self.width()*65/100 + 6,self.height()*5/100)

        self.lineEditBrightness.setFixedSize(width * 10 / 100, height * 5 / 100)
        self.sliderBrightness.setFixedSize(self.width()*5/100,self.height()*73/100 + 6*vrb.ratio)

    def changeLineEditBrightness(self):

        self.lineEditBrightness.setText(str(self.sliderBrightness.value()))
        self.applySliderValues()

    def changeLineEditContrast(self):

        self.lineEditContrast.setText(str(self.sliderContrast.value()))
        self.applySliderValues()

    def changeSliderContrast(self):

        text = self.lineEditContrast.text()
        if text != '':
            self.sliderContrast.setValue(int(text))

    def changeSliderBrightness(self):

        text = self.lineEditBrightness.text()
        if text != '':
            self.sliderBrightness.setValue(int(text))

    def mousePressEvent(self, event):

        posX = event.pos().x() - self.graphicsView.pos().x()
        posY = event.pos().y() - self.graphicsView.pos().y()

        if posX >= 0 and posX <= self.graphicsView.width() and posY >=0 and posY <= self.graphicsView.height():

            self.clearScene()
            self.plotLines(posX,posY)

            valueContrast = round(posX*100/self.graphicsView.width())
            self.modif = False
            self.sliderContrast.setValue(valueContrast)
            self.modif = True
            valueBrightness = round(posY*100/self.graphicsView.height())
            self.sliderBrightness.setValue(valueBrightness)

    def plotLines(self,posX,posY):

        self.clearScene()
        posX = min(posX,self.graphicsView.width()-3)
        posX = max(1,posX)
        posY = min(posY,self.graphicsView.height()-3)
        posY = max(1,posY)
        self.graphicsView.scene.addLine(posX, 0, posX, self.graphicsView.height(), self.penLine)
        self.graphicsView.scene.addLine(0, posY, self.graphicsView.width(), posY, self.penLine)

    def applySliderValues(self):

        if self.modif:
            posX = self.sliderContrast.value() * self.graphicsView.width() / 100
            posY = self.sliderBrightness.value() * self.graphicsView.width() / 100
            self.plotLines(posX,posY)
            self.signalValueChanged.emit()

    def clearScene(self):

        self.scene.clear()
        self.scene.clearSelection()
        self.scene.clearFocus()
        self.scene.setSceneRect(0, 0, self.graphicsView.width(), self.graphicsView.height())

    def setValues(self,contrastValue,brightnessValue):

        self.modif = False
        self.sliderContrast.setValue(contrastValue)
        self.modif = True
        self.sliderBrightness.setValue(brightnessValue)


class WidgetContrast(qt.QWidget):

    signalValueChanged = pyqtSignal()

    def __init__(self,xmlElement = None):
        qt.QWidget.__init__(self)

        self.xmlElement = xmlElement

        self.margins = 10

        self.widgetSettings = WidgetSettings()

        self.pushButtonReinit = qt.QPushButton("Reinit")

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.widgetSettings, 0, 0, QtCore.Qt.AlignCenter)
        self.layout.addWidget(self.pushButtonReinit, 1, 0, QtCore.Qt.AlignLeft)

        self.layout.setSpacing(0)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(self.margins, self.margins, self.margins, self.margins)

        self.setFixedSize(378*vrb.ratio, 360*vrb.ratio)

        self.pushButtonReinit.clicked.connect(self.reinitValues)
        self.widgetSettings.signalValueChanged.connect(self.emitSignalValueChanged)

        self.setWindowTitle("Contrast and brightness")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def setValues(self,contrastValue, brightnessValue):

        self.widgetSettings.setValues(contrastValue,brightnessValue)

    def reinitValues(self):

        self.setValues(50,50)

    def emitSignalValueChanged(self):

        self.signalValueChanged.emit()

    def createXmlElement(self):

        if self.xmlElement != None:

            Dfct.clearElement(self.xmlElement)

            contrastElement = Dfct.SubElement(self.xmlElement,"Contrast")
            contrastElement.text = str(self.widgetSettings.sliderContrast.value())
            brightnessElement = Dfct.SubElement(self.xmlElement, "Brightness")
            brightnessElement.text = str(self.widgetSettings.sliderBrightness.value())

    def loadXmlElement(self,element):

        self.xmlElement = element
        for child in element:
            if child.tag == "Contrast":
                self.widgetSettings.sliderContrast.setValue(int(child.text))
            if child.tag == "Brightness":
                self.widgetSettings.sliderBrightness.setValue(int(child.text))


if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    foo = WidgetContrast()
    foo.show()
    foo.setValues(50,50)

    app.exec_()