import os
import sys

from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore
from PyQt5.QtCore import pyqtSignal, QPointF, Qt, QCoreApplication

import PyIPSDK
import PyIPSDK.IPSDKIPLUtility as util
import PyIPSDK.IPSDKIPLIntensityTransform as itrans
import PyIPSDK.IPSDKIPLGeometricTransform as gtrans

import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr
import PyIPSDK.IPSDKIPLArithmetic as arithm
import PyIPSDK.IPSDKIPLBinarization as bin
import PyIPSDK.IPSDKIPLColor as colorIP
import PyIPSDK.IPSDKIPLBasicMorphology as morpho
import PyIPSDK.IPSDKIPLLogical as logic

import PyIPSDK.IPSDKVariables as IPSDKVrb

import numpy
import xml.etree.ElementTree as xmlet

import UsefullTexts as txt

import UsefullVariables as vrb
import UsefullWidgets as wgt
import UsefullFunctions as fct
import DatabaseFunction as Dfct

import DrawFunctions as DWfct

import time
import traceback

from Sliders import SimpleHorizontalSliderLabel
from RangeSlider import RangeSlider
# from RangeSliderIDS import RangeSlider
from ImageViewer import ImageViewer,ImageViewerWithScrollBar,WidgetInfo

import numpy
from PIL import Image, ImageQt
from PyQt5.QtCore import Qt, QRectF
from PyQt5.QtGui import QPixmap
from PyQt5.QtWidgets import QWidget, QGraphicsScene, QGraphicsView, QPushButton, QVBoxLayout, QApplication, QMessageBox


import sys
import platform
# if platform.system() == 'Windows':
#     import win32event

try:
    import cv2
except:
    pass

try:
    from vimba import *
except:
    pass

import time

class WidgetCameraToDisplay(qt.QWidget):

    def __init__(self):
        qt.QWidget.__init__(self)

        try:
            file = xmlet.parse(vrb.folderInformation + "/CameraSelected.mho")
            self.xmlElement = file.getroot()
        except:
            self.xmlElement = xmlet.Element('Camera')

        self.labelCamera = qt.QLabel("Select the cameras to display")

        self.checkBoxCamera1 = qt.QCheckBox("Camera 1")
        self.checkBoxCamera1.setChecked(True)

        self.checkBoxCamera2 = qt.QCheckBox("Camera 2")
        self.checkBoxCamera2.setChecked(True)

        self.pushButtonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=2)
        self.pushButtonValidate.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelCamera, 0, 0, 1, 2)
        self.layout.addWidget(self.checkBoxCamera1, 1, 0)
        self.layout.addWidget(self.checkBoxCamera2, 1, 1)
        self.layout.addWidget(self.pushButtonValidate, 2, 0, 1, 2, Qt.AlignRight)

        self.setLayout(self.layout)
        self.layout.setContentsMargins(5 * vrb.ratio, 5 * vrb.ratio, 5 * vrb.ratio, 15 * vrb.ratio)
        self.layout.setSizeConstraint(1)

        self.pushButtonValidate.clicked.connect(self.displayCameras)

        self.loadXmlElement()

        self.setWindowTitle("Select camera")

        self.setFixedSize(150 * vrb.ratio, 90 * vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def displayCameras(self):

        self.close()
        if self.checkBoxCamera1.isChecked():
            try:
                fct.showWidget(vrb.widgetCamera[0])
            except:
                pass
        if self.checkBoxCamera2.isChecked():
            try:
                fct.showWidget(vrb.widgetCamera[1])
            except:
                pass

        fct.displayCamera()

    def saveXmlElement(self):

        Dfct.SubElement(self.xmlElement, "Camera1").text = str(self.checkBoxCamera1.isChecked())
        Dfct.SubElement(self.xmlElement, "Camera2").text = str(self.checkBoxCamera2.isChecked())

        Dfct.saveXmlElement(self.xmlElement, vrb.folderInformation + "/CameraSelected.mho")

    def loadXmlElement(self):

        self.checkBoxCamera1.setChecked(Dfct.childText(self.xmlElement,"Camera1")!="False")
        self.checkBoxCamera2.setChecked(Dfct.childText(self.xmlElement,"Camera2")!="False")

class WidgetCamera(qt.QWidget):

    def __init__(self, parent=None,cameraNum=0):
        qt.QWidget.__init__(self)

        self.cameraNum = cameraNum

        try:
            file = xmlet.parse(vrb.folderInformation + "/Camera_"+str(self.cameraNum)+".mho")
            self.xmlElement = file.getroot()
        except:
            self.xmlElement = xmlet.Element('Camera')

        self.parent = parent

        # self.update = True
        self.histogramFirstTime = True

        # try:
        #     self.setWindowFlag(Qt.WindowStaysOnTopHint)
        # except:
        #     self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.groupBoxSettings = GroupBoxSettings(parent = self)

        self.scene = qt.QGraphicsScene()
        self.imageViewerWithScrollArea = ImageViewerWithScrollBar(self.scene, widgetImage=self, parent=self)
        self.imageViewer = self.imageViewerWithScrollArea.imageViewer

        self.graphicElements = GraphicElements(widgetImage=self)

        self.loadXmlElement()

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.imageViewerWithScrollArea,0,0,Qt.AlignCenter)
        self.layout.addWidget(self.groupBoxSettings,0,1)

        self.setLayout(self.layout)
        self.layout.setContentsMargins(5* vrb.ratio, 5* vrb.ratio, 5* vrb.ratio, 15* vrb.ratio)
        self.layout.setSizeConstraint(1)

        self.setWindowTitle("Camera")

        self.groupBoxSettings.pushButtonLecure.clicked.connect(self.changeLecture)
        self.groupBoxSettings.pushButtonExport.clicked.connect(self.exportImage)
        self.groupBoxSettings.pushButtonExportAll.clicked.connect(self.exportAllImages)
        self.imageViewer.signalEmitMouse.connect(self.graphicElements.selection)

        self.resize(1000*vrb.ratio,1000*vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        # self.displayImage()

    def resizeEvent(self,event):

        self.imageViewerWithScrollArea.setFixedSize(max(1,self.width() * 100 / 100 - 290* vrb.ratio), max(1, self.height() * 100 / 100 - 50*vrb.ratio))
        self.groupBoxSettings.setFixedSize(270* vrb.ratio, max(1, self.height() * 100 / 100 - 20*vrb.ratio))

    def exportImage(self):

        try:
            outImage = self.transformImage(self.image)
            vrb.mainWindow.addImageFromCamera(outImage,"Capture_"+str(vrb.captureNum))
            vrb.captureNum+=1
        except:
            traceback.print_exc(file=sys.stderr)

    def exportAllImages(self):

        for widget in vrb.widgetCamera:
            try:
                outImage = widget.transformImage(widget.image)
                vrb.mainWindow.addImageFromCamera(outImage,"Capture_"+str(vrb.captureNum))
                vrb.captureNum+=1
            except:
                traceback.print_exc(file=sys.stderr)


    def openSettings(self):

        self.widgetSettings.show()

    def changeLecture(self):

        self.groupBoxSettings.pushButtonLecure.changeActivation()
        if not self.groupBoxSettings.pushButtonLecure.activate:
            # self.update = True
            # self.display()
            vrb.updateCamera = True
            fct.displayCamera()

    def frame_handler(self,cam, frame):

        # cam.wait_for_frames(100)
        cam.queue_frame(frame)
        if self.cameraNum == 0:
            IPSDKVrb.frame0 = frame
        if self.cameraNum == 1:
            IPSDKVrb.frame1 = frame

        # time.sleep(0.01)

    # def display(self):
    #
    #     while self.update:
    #         if not self.isVisible():
    #             break
    #         self.displayImage()
    #         if not self.update:
    #             self.update = True
    #         else:
    #             break
    #
    # def displayImage(self):
    #
    #     with Vimba.get_instance() as vimba:
    #         cams = vimba.get_all_cameras()
    #
    #         with cams[self.cameraNum] as cam:
    #
    #             try:
    #                 self.verifyCameraSettings(cam)
    #             except:
    #                 traceback.print_exc(file=sys.stderr)
    #                 self.groupBoxSettings.pushButtonLecure.setActivation(True)
    #
    #             try:
    #                 cam.start_streaming(self.frame_handler)
    #             except:
    #                 pass
    #
    #             while self.update:
    #                 if not self.isVisible():
    #                     cam.stop_streaming()
    #                     break
    #                 if not self.groupBoxSettings.pushButtonLecure.activate:
    #                     self.createCameraImage(cam)
    #                 else:
    #                     break


    def verifyCameraSettings(self,cam):

        try:
            self.applyCameraSettings(cam)
        except:
            cam.stop_streaming()
            self.applyCameraSettings(cam)
            cam.start_streaming(self.frame_handler)

    def applyCameraSettings(self,cam):

        if self.groupBoxSettings.radioButtonBinning1.isChecked():
            value = 1
        if self.groupBoxSettings.radioButtonBinning2.isChecked():
            value = 2
        if self.groupBoxSettings.radioButtonBinning4.isChecked():
            value = 4
        if self.groupBoxSettings.radioButtonBinning8.isChecked():
            value = 8

        sizeX = self.groupBoxSettings.lineEditSizeX.text()
        if sizeX == "":
            sizeX = self.groupBoxSettings.lineEditSizeX.placeholderText()
        sizeY = self.groupBoxSettings.lineEditSizeY.text()
        if sizeY == "":
            sizeY = self.groupBoxSettings.lineEditSizeY.placeholderText()

        offsetX = int(self.groupBoxSettings.lineEditOffsetX.text())
        offsetY = int(self.groupBoxSettings.lineEditOffsetY.text())
        sizeX = max(2, int(sizeX))
        sizeY = max(2, int(sizeY))

        try:
            cam.BinningVertical.set(value)
            cam.BinningHorizontal.set(value)
        except:
            cam.BinningHorizontal.set(value)
            cam.BinningVertical.set(value)

        if self.groupBoxSettings.checkBoxCropImage.isChecked():
            try:
                cam.OffsetX.set(offsetX)
                cam.OffsetY.set(offsetY)
                cam.Width.set(sizeX)
                cam.Height.set(sizeY)
            except:
                cam.Width.set(sizeX)
                cam.Height.set(sizeY)
                cam.OffsetX.set(offsetX)
                cam.OffsetY.set(offsetY)
        else:
            cam.OffsetX.set(0)
            cam.OffsetY.set(0)
            cam.Width.set(self.groupBoxSettings.resolutionX / value)
            cam.Height.set(self.groupBoxSettings.resolutionY / value)

    def createCameraImage(self,cam):

        # cam.start_streaming(self.frame_handler)

        if self.cameraNum == 0:
            frame = IPSDKVrb.frame0
        if self.cameraNum == 1:
            frame = IPSDKVrb.frame1

        if self.groupBoxSettings.sliderGain.buttonAuto.isChecked():
            if str(cam.GainAuto.get()) == "Off":
                cam.GainAuto.set("Continuous")
            self.groupBoxSettings.sliderGain.slider.setValue(cam.Gain.get()*100)
        if not self.groupBoxSettings.sliderGain.buttonAuto.isChecked():
            if str(cam.GainAuto.get()) == "Continuous":
                cam.GainAuto.set("Off")
            gain = float(self.groupBoxSettings.sliderGain.slider.value())
            cam.Gain.set(gain/100)

        if self.groupBoxSettings.sliderExposureTime.buttonAuto.isChecked():
            if str(cam.ExposureAuto.get()) == "Off":
                cam.ExposureAuto.set("Continuous")
            self.groupBoxSettings.sliderExposureTime.slider.setValue(cam.ExposureTime.get()*20)
        if not self.groupBoxSettings.sliderExposureTime.buttonAuto.isChecked():
            if str(cam.ExposureAuto.get()) == "Continuous":
                cam.ExposureAuto.set("Off")
            exposureTime = float(self.groupBoxSettings.sliderExposureTime.slider.value())
            cam.ExposureTime.set(exposureTime/20)

        if frame is not None:
            frame.convert_pixel_format(PixelFormat.Mono8)
            openCvImage = frame.as_opencv_image()

            image = openCvImage[:, :, 0]
            self.image = PyIPSDK.fromArray(image)

            image = fct.getMultiResolutionImage(self.image)

            self.applyHistogram(self.image)

            self.imageViewer.setImage(image)

            self.imageViewer.getRoiImage()

            qt.QApplication.processEvents()

        # time.sleep(0.001)
        # cam.stop_streaming()

    def applyHistogram(self,image):

        histogramMsrParams = PyIPSDK.createHistoMsrParamsWithBinWidth(0, 255, 1)
        histogramMsr2dResult = glbmsr.histogramMsr2d(image, histogramMsrParams)
        self.groupBoxSettings.rangeSlider.groupSlider.getHistogram(histogramMsr2dResult.frequencies, contour=self.groupBoxSettings.checkBoxContour.isChecked(), color=[200, 200, 200])

        if self.histogramFirstTime:
            self.groupBoxSettings.rangeSlider.setValues(self.groupBoxSettings.rangeSlider.currentValueMin, self.groupBoxSettings.rangeSlider.currentValueMax)
            # self.groupBoxSettings.rangeSlider.groupLabelValue.labelMinValue.setText("0")
            self.histogramFirstTime = False

    def transformImage(self,image,imageOverlay = None,imageDraw = None,imageProbabilities = None,imageThreshold = None, imageSuperPixels = None):

        outImage = fct.applyImageContrast(image, self.groupBoxSettings.rangeSlider.currentValueMin, self.groupBoxSettings.rangeSlider.currentValueMax)

        return outImage

    def closeEvent(self, a0: QtGui.QCloseEvent) -> None:

        self.createXmlElement()
        self.groupBoxSettings.pushButtonLecure.setActivation(True)

    def createXmlElement(self,save=True):

        self.groupBoxSettings.createXmlElement(self.xmlElement)
        if save:
            Dfct.saveXmlElement(self.xmlElement,vrb.folderInformation + "/Camera_"+str(self.cameraNum)+".mho")

    def loadXmlElement(self):

        self.groupBoxSettings.loadXmlElement(self.xmlElement)

class GroupBoxSettings(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.resolutionX = vrb.cameraMaxSizeX
        self.resolutionY = vrb.cameraMaxSizeY

        self.parent = parent
        self.verify = True

        self.groupBoxLoadSettings = wgt.GroupBoxLoadSettings()

        ###Acquisition
        self.pushButtonLecure = wgt.PushButtonDoubleImage(vrb.folderImages + "/Pause.png", vrb.folderImages + "/Lecture.png")
        self.pushButtonLecure.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.pushButtonLecure.setToolTip(txt.dictToolTips["CameraLecture"])
        self.pushButtonLecure.setActivation(False)

        self.pushButtonExport = wgt.PushButtonImage(vrb.folderImages + "/Camera.png", margins=2)
        self.pushButtonExport.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.pushButtonExport.setToolTip(txt.dictToolTips["Camera"])

        self.pushButtonExportAll = wgt.PushButtonImage(vrb.folderImages + "/Camera_All.png", margins=2)
        self.pushButtonExportAll.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.pushButtonExportAll.setToolTip(txt.dictToolTips["Camera_All"])
        if vrb.nbCameras < 2:
            self.pushButtonExportAll.setVisible(False)

        groupBoxAcquisition = qt.QGroupBox()
        layoutAcquisition = qt.QHBoxLayout()
        layoutAcquisition.addWidget(self.pushButtonLecure)
        layoutAcquisition.addWidget(self.pushButtonExport)
        layoutAcquisition.addWidget(self.pushButtonExportAll)
        groupBoxAcquisition.setLayout(layoutAcquisition)
        groupBoxAcquisition.setFixedSize(250*vrb.ratio,80*vrb.ratio)

        groupBoxAcquisition.setTitle("Acquisition")
        groupBoxAcquisition.setStyleSheet('QGroupBox:title {left: '+ str(int(20*vrb.ratio)) + 'px;padding-left: '+ str(int(10*vrb.ratio)) + 'px;padding-right: '+ str(int(10*vrb.ratio)) +
                                          'px;padding-top: -'+ str(int(10*vrb.ratio)) + 'px; color:rgb(52, 152, 219)}  QGroupBox {font-size: '+ str(int(10*vrb.ratio)) + 'px;font: bold;border: '+
                                          str(int(1*vrb.ratio)) + 'px solid gray; margin-top: '+ str(int(10*vrb.ratio)) + ' px }')

        #ROI
        self.labelBinning = qt.QLabel("Binning")
        self.labelBinning.setFixedSize(50*vrb.ratio,20*vrb.ratio)
        self.radioButtonBinning1 = qt.QRadioButton("1")
        self.radioButtonBinning1.setChecked(True)
        self.radioButtonBinning2 = qt.QRadioButton("2")
        self.radioButtonBinning4 = qt.QRadioButton("4")
        self.radioButtonBinning8 = qt.QRadioButton("8")
        self.buttonGroup = qt.QButtonGroup()
        self.buttonGroup.addButton(self.radioButtonBinning1, 0)
        self.buttonGroup.addButton(self.radioButtonBinning2, 1)
        self.buttonGroup.addButton(self.radioButtonBinning4, 2)
        self.buttonGroup.addButton(self.radioButtonBinning8, 3)
        groupBoxBinning = qt.QGroupBox()
        layoutBinning = qt.QHBoxLayout()
        layoutBinning.addWidget(self.labelBinning)
        layoutBinning.addWidget(self.radioButtonBinning1)
        layoutBinning.addWidget(self.radioButtonBinning2)
        layoutBinning.addWidget(self.radioButtonBinning4)
        layoutBinning.addWidget(self.radioButtonBinning8)
        layoutBinning.setContentsMargins(0,0,0,0)
        groupBoxBinning.setLayout(layoutBinning)
        groupBoxBinning.setStyleSheet('QGroupBox {border: 0px transparent}')
        groupBoxBinning.setFixedSize(180*vrb.ratio,25*vrb.ratio)

        self.labelOffsetX = qt.QLabel("Offset X")
        self.labelOffsetX.setFixedSize(50*vrb.ratio,20*vrb.ratio)
        self.lineEditOffsetX = wgt.NumberLineEdit()
        self.lineEditOffsetX.setPlaceholderText("0")
        self.lineEditOffsetX.setText("0")
        self.lineEditOffsetX.setFixedSize(30*vrb.ratio,20*vrb.ratio)
        self.labelOffsetY = qt.QLabel("Offset Y")
        self.labelOffsetY.setFixedSize(50*vrb.ratio,20*vrb.ratio)
        self.lineEditOffsetY = wgt.NumberLineEdit()
        self.lineEditOffsetY.setPlaceholderText("0")
        self.lineEditOffsetY.setText("0")
        self.lineEditOffsetY.setFixedSize(30*vrb.ratio,20*vrb.ratio)
        self.labelSizeX = qt.QLabel("Size X")
        self.labelSizeX.setFixedSize(50*vrb.ratio,20*vrb.ratio)
        self.lineEditSizeX = wgt.NumberLineEdit()
        self.lineEditSizeX.setPlaceholderText(str(self.resolutionX))
        self.lineEditSizeX.setText(str(self.resolutionX))
        self.lineEditSizeX.setFixedSize(30*vrb.ratio,20*vrb.ratio)
        self.labelSizeY = qt.QLabel("Size Y")
        self.labelSizeY.setFixedSize(50*vrb.ratio,20*vrb.ratio)
        self.lineEditSizeY = wgt.NumberLineEdit()
        self.lineEditSizeY.setPlaceholderText(str(self.resolutionY))
        self.lineEditSizeY.setText(str(self.resolutionY))
        self.lineEditSizeY.setFixedSize(30*vrb.ratio,20*vrb.ratio)
        self.checkBoxCropImage = qt.QCheckBox("Crop image")
        self.checkBoxCropImage.setFixedSize(65*vrb.ratio,25*vrb.ratio)
        self.checkBoxCropImage.setChecked(True)
        self.buttonFull = qt.QPushButton("Full")
        self.buttonFull.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.pushButtonRefresh = wgt.PushButtonImage(vrb.folderImages + "/Refresh_White.png", margins=2)
        self.pushButtonRefresh.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.pushButtonRefresh.setToolTip(txt.dictToolTips["Camera_All"])
        groupBoxCoordinate = qt.QGroupBox()
        layoutCoordinate = qt.QGridLayout()
        layoutCoordinate.addWidget(self.labelOffsetX,0,0)
        layoutCoordinate.addWidget(self.lineEditOffsetX,0,1)
        layoutCoordinate.addWidget(self.labelOffsetY,0,2)
        layoutCoordinate.addWidget(self.lineEditOffsetY,0,3)
        layoutCoordinate.addWidget(self.labelSizeX,1,0)
        layoutCoordinate.addWidget(self.lineEditSizeX,1,1)
        layoutCoordinate.addWidget(self.labelSizeY,1,2)
        layoutCoordinate.addWidget(self.lineEditSizeY,1,3)
        layoutCoordinate.addWidget(self.checkBoxCropImage,2,0)
        layoutCoordinate.addWidget(self.buttonFull,2,2)
        layoutCoordinate.addWidget(self.pushButtonRefresh,2,3)
        layoutCoordinate.setContentsMargins(0,0,0,0)
        groupBoxCoordinate.setLayout(layoutCoordinate)
        groupBoxCoordinate.setStyleSheet('QGroupBox {border: 0px transparent}')
        groupBoxCoordinate.setFixedSize(180*vrb.ratio,140*vrb.ratio)

        groupBoxRoi = qt.QGroupBox()
        layoutRoi = qt.QGridLayout()
        layoutRoi.addWidget(groupBoxBinning,0,0)
        layoutRoi.addWidget(groupBoxCoordinate,1,0)
        groupBoxRoi.setLayout(layoutRoi)
        groupBoxRoi.setFixedSize(250*vrb.ratio,190*vrb.ratio)

        groupBoxRoi.setTitle("ROI")
        groupBoxRoi.setStyleSheet('QGroupBox:title {left: '+ str(int(20*vrb.ratio)) + 'px;padding-left: '+ str(int(10*vrb.ratio)) + 'px;padding-right: '+ str(int(10*vrb.ratio)) +
                                          'px;padding-top: -'+ str(int(10*vrb.ratio)) + 'px; color:rgb(52, 152, 219)}  QGroupBox {font-size: '+ str(int(10*vrb.ratio)) + 'px;font: bold;border: '+
                                          str(int(1*vrb.ratio)) + 'px solid gray; margin-top: '+ str(int(10*vrb.ratio)) + ' px }')

        ###Brightness
        self.sliderGain = SimpleHorizontalSliderLabel(label="Gain (db)", titleSize=110 * vrb.ratio,widthLabel=30*vrb.ratio, minimum=0, maximum=2400, defaultValue=0, ratio=100,auto=True)
        self.sliderGain.setFixedWidth(240 * vrb.ratio)
        self.sliderGain.setStyleSheet('QGroupBox {border: 0px transparent}')
        self.sliderGain.buttonAuto.setCheckable(True)

        self.sliderExposureTime = SimpleHorizontalSliderLabel(label="ExposureTime (us)", titleSize=110 * vrb.ratio,widthLabel=30*vrb.ratio, minimum=3240, maximum=10000000, defaultValue=3240, ratio=20,auto=True,logarithmique = True, logarithmiqueCoeff=8)
        self.sliderExposureTime.setFixedWidth(240 * vrb.ratio)
        self.sliderExposureTime.setStyleSheet('QGroupBox {border: 0px transparent}')
        self.sliderExposureTime.buttonAuto.setCheckable(True)

        groupBoxBrightness = qt.QGroupBox()
        layoutBrightness = qt.QGridLayout()
        layoutBrightness.addWidget(self.sliderGain,0,0)
        layoutBrightness.addWidget(self.sliderExposureTime,1,0)
        layoutBrightness.setContentsMargins(5*vrb.ratio,5*vrb.ratio,5*vrb.ratio,5*vrb.ratio)
        groupBoxBrightness.setLayout(layoutBrightness)
        groupBoxBrightness.setFixedSize(250*vrb.ratio,100*vrb.ratio)

        groupBoxBrightness.setTitle("Brightness")
        groupBoxBrightness.setStyleSheet('QGroupBox:title {left: '+ str(int(20*vrb.ratio)) + 'px;padding-left: '+ str(int(10*vrb.ratio)) + 'px;padding-right: '+ str(int(10*vrb.ratio)) +
                                          'px;padding-top: -'+ str(int(10*vrb.ratio)) + 'px; color:rgb(52, 152, 219)}  QGroupBox {font-size: '+ str(int(10*vrb.ratio)) + 'px;font: bold;border: '+
                                          str(int(1*vrb.ratio)) + 'px solid gray; margin-top: '+ str(int(10*vrb.ratio)) + ' px }')
        ###Histogram
        fontSize = 10
        self.rangeSlider = RangeSlider(fontSize=fontSize,customResize=[15,85])
        self.rangeSlider.currentValueMin = 0
        self.rangeSlider.currentValueMax = 255

        self.checkBoxContour = qt.QCheckBox("Contour")
        font = QtGui.QFont()
        font.setPointSize(fontSize)
        self.checkBoxContour.setFont(font)

        groupBoxHistogram = qt.QGroupBox()
        layoutHistogram = qt.QGridLayout()
        layoutHistogram.addWidget(self.checkBoxContour,0,0)
        layoutHistogram.addWidget(self.rangeSlider,1,0)
        self.rangeSlider.setFixedHeight(150*vrb.ratio)
        groupBoxHistogram.setLayout(layoutHistogram)

        groupBoxHistogram.setTitle("Histogram")
        groupBoxHistogram.setStyleSheet('QGroupBox:title {left: '+ str(int(20*vrb.ratio)) + 'px;padding-left: '+ str(int(10*vrb.ratio)) + 'px;padding-right: '+ str(int(10*vrb.ratio)) +
                                          'px;padding-top: -'+ str(int(10*vrb.ratio)) + 'px; color:rgb(52, 152, 219)}  QGroupBox {font-size: '+ str(int(10*vrb.ratio)) + 'px;font: bold;border: '+
                                          str(int(1*vrb.ratio)) + 'px solid gray; margin-top: '+ str(int(10*vrb.ratio)) + ' px }')
        groupBoxHistogram.setFixedSize(250*vrb.ratio,230*vrb.ratio)

        emptyLabel = qt.QLabel()
        emptyLabel.setFixedHeight(5*vrb.ratio)

        self.pushButtonSynchronize = wgt.PushButtonImage(vrb.folderImages + "/Synchronize.png", margins=2)
        self.pushButtonSynchronize.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.pushButtonSynchronize.setToolTip(txt.dictToolTips["Synchronize"])
        self.pushButtonSynchronize.setVisible(False)

        self.lineEditSaveSettings = qt.QLineEdit()
        self.lineEditSaveSettings.setPlaceholderText("Save settings")
        self.lineEditSaveSettings.setFixedSize(120*vrb.ratioDimension,25*vrb.ratio)

        self.buttonSaveSettings = wgt.PushButtonImage(vrb.folderImages + "/Save.png", margins=2)
        self.buttonSaveSettings.setToolTip("Save settings")
        self.buttonSaveSettings.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.buttonExportSettings = wgt.PushButtonImage(vrb.folderImages + "/Export.png", margins=2)
        self.buttonExportSettings.setToolTip("Export settings")
        self.buttonExportSettings.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        groupBoxSaveSettings = qt.QGroupBox()
        layoutSaveSettings = qt.QGridLayout()
        layoutSaveSettings.addWidget(self.lineEditSaveSettings,0,0)
        layoutSaveSettings.addWidget(self.buttonSaveSettings,0,1)
        layoutSaveSettings.addWidget(self.buttonExportSettings,0,2)
        layoutSaveSettings.setAlignment(Qt.AlignRight)
        layoutSaveSettings.setContentsMargins(0*vrb.ratio,0*vrb.ratio,0*vrb.ratio,0*vrb.ratio)
        groupBoxSaveSettings.setLayout(layoutSaveSettings)
        groupBoxSaveSettings.setFixedSize(250*vrb.ratio,40*vrb.ratio)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.groupBoxLoadSettings,0,0)
        self.layout.addWidget(groupBoxAcquisition,1,0)
        self.layout.addWidget(groupBoxRoi,2,0)
        self.layout.addWidget(groupBoxBrightness,3,0)
        self.layout.addWidget(groupBoxHistogram,4,0)
        self.layout.addWidget(emptyLabel,5,0)
        self.layout.addWidget(self.pushButtonSynchronize,5,0, Qt.AlignRight)
        self.layout.addWidget(groupBoxSaveSettings,6,0, Qt.AlignTop)
        self.layout.setAlignment(Qt.AlignTop)
        self.layout.setContentsMargins(10*vrb.ratio,30*vrb.ratio,10*vrb.ratio,10*vrb.ratio)

        self.buttonGroup.buttonClicked.connect(self.changeResolution)
        self.buttonFull.clicked.connect(self.applyFullResolution)
        self.lineEditOffsetX.returnPressed.connect(self.updateValue)
        self.lineEditOffsetY.returnPressed.connect(self.updateValue)
        self.lineEditSizeX.returnPressed.connect(self.updateValue)
        self.lineEditSizeY.returnPressed.connect(self.updateValue)
        self.pushButtonRefresh.clicked.connect(self.refreshCamera)
        self.checkBoxCropImage.stateChanged.connect(self.updateValue)
        self.sliderExposureTime.buttonAuto.clicked.connect(self.applyAutoExposureTime)
        self.sliderGain.buttonAuto.clicked.connect(self.applyAutoGain)
        self.buttonSaveSettings.clicked.connect(self.saveSettings)
        self.buttonExportSettings.clicked.connect(self.exportSettings)
        self.lineEditOffsetX.textChanged.connect(self.allowSaveButton)
        self.lineEditOffsetY.textChanged.connect(self.allowSaveButton)
        self.lineEditSizeX.textChanged.connect(self.allowSaveButton)
        self.lineEditSizeY.textChanged.connect(self.allowSaveButton)
        self.groupBoxLoadSettings.buttonLoadSettings.clicked.connect(self.loadSettings)
        self.groupBoxLoadSettings.buttonDeleteSettings.clicked.connect(self.deleteSettings)

        self.setLayout(self.layout)

        self.setFixedSize(270*vrb.ratio,1100*vrb.ratio)

        self.updateComboboxSettings()

        # self.updateValue()

    def createXmlElement(self,element):

        elementROI = Dfct.SubElement(element,"ROI")
        if self.radioButtonBinning1.isChecked():
            Dfct.SubElement(elementROI, "Binning").text = "1"
        if self.radioButtonBinning2.isChecked():
            Dfct.SubElement(elementROI, "Binning").text = "2"
        if self.radioButtonBinning4.isChecked():
            Dfct.SubElement(elementROI, "Binning").text = "4"
        if self.radioButtonBinning8.isChecked():
            Dfct.SubElement(elementROI, "Binning").text = "8"
        Dfct.SubElement(elementROI,"OffsetX").text = fct.lineEditValue(self.lineEditOffsetX)
        Dfct.SubElement(elementROI,"OffsetY").text = fct.lineEditValue(self.lineEditOffsetY)
        Dfct.SubElement(elementROI,"SizeX").text = fct.lineEditValue(self.lineEditSizeX)
        Dfct.SubElement(elementROI,"SizeY").text = fct.lineEditValue(self.lineEditSizeY)
        Dfct.SubElement(elementROI,"CropImage").text = str(self.checkBoxCropImage.isChecked())

        elementBrightness = Dfct.SubElement(element, "Brightness")
        Dfct.SubElement(elementBrightness, "Gain").text = str(self.sliderGain.slider.value())
        Dfct.SubElement(elementBrightness, "AutoGain").text = str(self.sliderGain.buttonAuto.isChecked())
        Dfct.SubElement(elementBrightness, "ExposureTime").text = str(self.sliderExposureTime.slider.value())
        Dfct.SubElement(elementBrightness, "AutoExposureTime").text = str(self.sliderExposureTime.buttonAuto.isChecked())

        elementHistogram = Dfct.SubElement(element, "Histogram")
        Dfct.SubElement(elementHistogram, "Contour").text = str(self.checkBoxContour.isChecked())
        Dfct.SubElement(elementHistogram, "MinValue").text = str(self.rangeSlider.currentValueMin)
        Dfct.SubElement(elementHistogram, "MaxValue").text = str(self.rangeSlider.currentValueMax)

    def loadXmlElement(self,element):

        self.verify = False

        elementROI = Dfct.SubElement(element,"ROI")
        if Dfct.SubElement(elementROI, "Binning").text == "1":
            self.radioButtonBinning1.setChecked(True)
        if Dfct.SubElement(elementROI, "Binning").text == "2":
            self.radioButtonBinning2.setChecked(True)
        if Dfct.SubElement(elementROI, "Binning").text == "4":
            self.radioButtonBinning4.setChecked(True)
        if Dfct.SubElement(elementROI, "Binning").text == "8":
            self.radioButtonBinning8.setChecked(True)
        fct.setLineEditValue(self.lineEditOffsetX,Dfct.SubElement(elementROI,"OffsetX").text)
        fct.setLineEditValue(self.lineEditOffsetY,Dfct.SubElement(elementROI,"OffsetY").text)
        fct.setLineEditValue(self.lineEditSizeX,Dfct.SubElement(elementROI,"SizeX").text)
        fct.setLineEditValue(self.lineEditSizeY,Dfct.SubElement(elementROI,"SizeY").text)
        self.checkBoxCropImage.setChecked(not Dfct.SubElement(elementROI,"CropImage").text == "False")

        elementBrightness = Dfct.SubElement(element, "Brightness")
        if Dfct.SubElement(elementBrightness, "Gain").text is not None:
            self.sliderGain.slider.setValue(float(Dfct.SubElement(elementBrightness, "Gain").text))
        self.sliderGain.buttonAuto.setChecked(Dfct.SubElement(elementBrightness,"AutoGain").text == "True")
        if Dfct.SubElement(elementBrightness, "ExposureTime").text is not None:
            self.sliderExposureTime.slider.setValue(float(Dfct.SubElement(elementBrightness, "ExposureTime").text))
        self.sliderExposureTime.buttonAuto.setChecked(Dfct.SubElement(elementBrightness,"AutoExposureTime").text == "True")

        self.applyAutoGain()
        self.applyAutoExposureTime()

        elementHistogram = Dfct.SubElement(element, "Histogram")
        self.checkBoxContour.setChecked(Dfct.SubElement(elementHistogram, "Contour").text == "True")
        if Dfct.SubElement(elementHistogram, "MinValue").text is not None:
            valueMin = int(Dfct.SubElement(elementHistogram, "MinValue").text)
        else:
            valueMin = 0
        if Dfct.SubElement(elementHistogram, "MaxValue").text is not None:
            valueMax = int(Dfct.SubElement(elementHistogram, "MaxValue").text)
        else:
            valueMax = 255
        self.rangeSlider.currentValueMin = valueMin
        self.rangeSlider.currentValueMax = valueMax
        self.rangeSlider.setValues(valueMin,valueMax)

        self.verify = True
        # self.updateValue()

    def allowSaveButton(self):

        self.buttonSaveSettings.setEnabled(True)

    def loadSettings(self):

        xmlElement = self.groupBoxLoadSettings.comboBoxLoadingSettings.currentData()
        self.loadXmlElement(xmlElement)
        self.lineEditSaveSettings.setText(self.groupBoxLoadSettings.comboBoxLoadingSettings.currentText())
        self.updateValue()

    def deleteSettings(self):

        nameSettings = self.groupBoxLoadSettings.comboBoxLoadingSettings.currentText()

        self.messageBox = wgt.MessageBox('Are you sure you want to delete the settings "'+nameSettings+'" ?', '', buttons=[qt.QMessageBox.Yes, qt.QMessageBox.No], windowTitle="Delete settings", icon=qt.QMessageBox.Warning)
        res = self.messageBox.exec()
        if res == qt.QMessageBox.Yes and os.path.exists(vrb.folderSettingsCamera + "/" + nameSettings + ".mho"):
            try:
                os.remove(vrb.folderSettingsCamera + "/" + nameSettings + ".mho")
            except:
                pass
            self.updateComboboxSettings()

    def saveSettings(self):

        self.updateValue()
        self.createXmlElement(self.parent.xmlElement)

        element = self.parent.xmlElement
        nameSettings = self.lineEditSaveSettings.text()

        found = False
        for fileName in os.listdir(vrb.folderSettingsCamera):

            if nameSettings + ".mho" == fileName:
                found = True
                text = 'The settings "'+nameSettings+'" already exists.\nDo you want to replace it ?'
                self.messageBox = wgt.MessageBox(text, '', buttons=[qt.QMessageBox.Yes,qt.QMessageBox.No],windowTitle="Settings already exists", icon=qt.QMessageBox.Warning)
                res = self.messageBox.exec()
                if res == qt.QMessageBox.Yes:
                    Dfct.saveXmlElement(element,vrb.folderSettingsCamera + "/" + nameSettings + ".mho")
                    self.buttonSaveSettings.setEnabled(False)

        if found == False:
            Dfct.saveXmlElement(element, vrb.folderSettingsCamera + "/" + nameSettings + ".mho")
            self.buttonSaveSettings.setEnabled(False)

        self.updateComboboxSettings()

    def exportSettings(self):

        try:
            try:
                file = xmlet.parse(vrb.folderInformation + "/folderExportCameraSettings.mho")
                folderExportCameraSettingsElement = file.getroot()
            except:
                folderExportCameraSettingsElement = xmlet.Element('folderExportCameraSettings')

            path = Dfct.childText(folderExportCameraSettingsElement, "Path")
            if path is not None:
                defaultFolder = path
            else:
                defaultFolder = "C:/"

            filename = qt.QFileDialog.getSaveFileName(self, "Save settings", defaultFolder + "/", "mho file (*.mho)")

            if filename[0] != '' and filename[0] != None:
                Dfct.SubElement(folderExportCameraSettingsElement, "Path").text = os.path.dirname(filename[0])
                Dfct.saveXmlElement(folderExportCameraSettingsElement, vrb.folderInformation + "/folderExportCameraSettings.mho")

                self.updateValue()
                self.createXmlElement(self.parent.xmlElement)
                element = self.parent.xmlElement
                Dfct.saveXmlElement(element,filename[0])
        except:
            traceback.print_exc(file=sys.stderr)

    def updateComboboxSettings(self):

        try:
            for cameraNum in range(vrb.nbCameras):
                vrb.widgetCamera[cameraNum].groupBoxSettings.updateCurrentComboBoxSettings()
        except:
            self.updateCurrentComboBoxSettings()

    def updateCurrentComboBoxSettings(self):

        self.groupBoxLoadSettings.comboBoxLoadingSettings.clear()
        for fileName in os.listdir(vrb.folderSettingsCamera):
            fileSettings = xmlet.parse(vrb.folderSettingsCamera + "/" + fileName)
            xmlElement = fileSettings.getroot()
            self.groupBoxLoadSettings.comboBoxLoadingSettings.addItem(fileName[:-4],xmlElement)

    def applyAutoGain(self):

        self.sliderGain.slider.setEnabled(not self.sliderGain.buttonAuto.isChecked())

    def applyAutoExposureTime(self):

        self.sliderExposureTime.slider.setEnabled(not self.sliderExposureTime.buttonAuto.isChecked())

    def changeResolution(self):

        self.verify = False

        if self.radioButtonBinning1.isChecked():
            value= 1
        if self.radioButtonBinning2.isChecked():
            value = 2
        if self.radioButtonBinning4.isChecked():
            value= 4
        if self.radioButtonBinning8.isChecked():
            value= 8

        self.lineEditSizeX.setPlaceholderText(str(int(self.resolutionX/value)))
        self.lineEditSizeY.setPlaceholderText(str(int(self.resolutionY/value)))

        self.verify = True
        self.updateValue()

    def applyFullResolution(self):

        self.verify = False

        if self.radioButtonBinning1.isChecked():
            value= 1
        if self.radioButtonBinning2.isChecked():
            value = 2
        if self.radioButtonBinning4.isChecked():
            value= 4
        if self.radioButtonBinning8.isChecked():
            value= 8

        self.lineEditOffsetX.setText("0")
        self.lineEditOffsetY.setText("0")
        self.lineEditSizeX.setText(str(int(self.resolutionX/value)))
        self.lineEditSizeY.setText(str(int(self.resolutionY/value)))

        self.verify = True
        self.updateValue()

    def verifyValue(self):

        if self.radioButtonBinning1.isChecked():
            value= 1
        if self.radioButtonBinning2.isChecked():
            value = 2
        if self.radioButtonBinning4.isChecked():
            value= 4
        if self.radioButtonBinning8.isChecked():
            value= 8

        try:
            if self.lineEditOffsetX.text() != "":
                offsetX = int(self.lineEditOffsetX.text())
                offsetX = int(offsetX / 2) * 2
            else:
                offsetX = 0
            if self.lineEditSizeX.text() != "":
                sizeX = int(self.lineEditSizeX.text())
            else:
                sizeX = int(self.resolutionX/value)
            if self.lineEditOffsetY.text() != "":
                offsetY = int(self.lineEditOffsetY.text())
                offsetY = int(offsetY / 2) * 2
            else:
                offsetY = 0
            if self.lineEditSizeY.text() != "":
                sizeY = int(self.lineEditSizeY.text())
            else:
                sizeY = int(self.resolutionY/value)

            sizeX = int(sizeX/8)*8
            sizeY = int(sizeY/2)*2

            sizeX = min(sizeX,int(self.resolutionX/value)-offsetX)
            sizeX = max(sizeX,0)
            sizeY = min(sizeY, int(self.resolutionY / value) - offsetY)
            sizeY = max(sizeY, 0)

            self.lineEditOffsetX.setText(str(offsetX))
            self.lineEditOffsetY.setText(str(offsetY))
            self.lineEditSizeX.setText(str(sizeX))
            self.lineEditSizeY.setText(str(sizeY))

            Dfct.clearElement(self.parent.graphicElements.roiSelection)
            allPoints = Dfct.SubElement(self.parent.graphicElements.roiSelection,"AllPoints")
            if len(allPoints) == 0:
                DWfct.addPointToPolygon(self.parent.graphicElements.roiSelection, QtCore.QPointF(offsetX, offsetY))
                DWfct.addPointToPolygon(self.parent.graphicElements.roiSelection, QtCore.QPointF(offsetX + sizeX, offsetY))
                DWfct.addPointToPolygon(self.parent.graphicElements.roiSelection, QtCore.QPointF(offsetX + sizeX, offsetY + sizeY))
                DWfct.addPointToPolygon(self.parent.graphicElements.roiSelection, QtCore.QPointF(offsetX, offsetY + sizeY))

        except:
            traceback.print_exc(file=sys.stderr)

    def updateValue(self):

        if self.verify:
            self.verifyValue()
            vrb.updateCamera = False
            # if self.parent is not None:
            #     self.parent.update = False

        self.allowSaveButton()

    def refreshCamera(self):

        self.verify = True
        self.updateValue()

class GraphicElements(QtCore.QObject):

    def __init__(self,widgetImage = None):
        super(GraphicElements, self).__init__()

        self.widgetImage = widgetImage
        self.imageViewer = widgetImage.imageViewer
        self.xmlElement = widgetImage.xmlElement

        self.roiSelection = Dfct.SubElement(self.xmlElement, "RoiSelection")

        self.distMinSelection = 10

        self.currentPointSelected = None
        self.currentPolygonSelected = None

        self.drawing = False
        self.hasMoving = False

        self.selectionRectangle = None
        self.refPosX = None
        self.refPosY = None

    def applyGraphicElement(self,imageViewer):

        try:
            if self.widgetImage.groupBoxSettings.checkBoxCropImage.isChecked() == False:
                DWfct.addPolygonToScene(imageViewer, self.roiSelection, closed="True", color=[float(vrb.polygonColor[0]), float(vrb.polygonColor[1]),
                                                                                          float(vrb.polygonColor[2])], showPoints=True, lineSize=2, pointMode="circle", pointSize=6)
        except:
            traceback.print_exc(file=sys.stderr)

    def selection(self, ddict):

        distMoveRef = self.distMinSelection

        if ddict["Button"] == 1 and ddict["Event"] == "MousePressed":

            self.resetCurrentPolygonSelected()

            point, distMove = DWfct.selectPointPolygon(self.imageViewer, QtCore.QPointF(ddict["Position"][0], ddict["Position"][1]), self.roiSelection, distMin=self.distMinSelection)
            if distMove < distMoveRef:
                self.currentPointSelected = point
                self.currentPolygonSelected = self.roiSelection
                distMoveRef = distMove

            if self.currentPointSelected is None:
                polygon, distMove = DWfct.selectPolygon(self.imageViewer, QtCore.QPointF(ddict["Position"][0], ddict["Position"][1]), self.roiSelection, distMin=self.distMinSelection)
                if distMove < distMoveRef:
                    self.currentPolygonSelected = self.roiSelection

            if self.currentPolygonSelected != None:
                self.refPosX = ddict["Position"][0]
                self.refPosY = ddict["Position"][1]
                selectionElement = Dfct.SubElement(self.currentPolygonSelected, "Selection")
                selectionElement.text = "True"
            else:
                self.imageViewer.standAlone = True
                self.imageViewer.viewport().setCursor(Qt.ClosedHandCursor)

            self.imageViewer.getRoiImage(changeRoiImage=False)

        if ddict["Button"] == 0 and ddict["Event"] == "Move":

            if self.currentPointSelected is not None:

                # self.widgetImage.groupBoxSettings.verify = False

                # Moving this point
                pointXElement = Dfct.SubElement(self.currentPointSelected, "PointX")
                pointXElement.text = str(ddict["Position"][0])
                pointYElement = Dfct.SubElement(self.currentPointSelected, "PointY")
                pointYElement.text = str(ddict["Position"][1])

                if self.currentPolygonSelected == self.roiSelection:
                    # In case of the roiSelection, the polygon is a rectangle, so moving a point moves 2 other points
                    numPointSelected = int(Dfct.childText(self.currentPointSelected, 'Num'))
                    for point in Dfct.SubElement(self.roiSelection, 'AllPoints'):
                        if int(Dfct.childText(point, 'Num')) == (numPointSelected + 1) % 4:
                            if numPointSelected % 2 == 1:
                                pointXElement = Dfct.SubElement(point, "PointX")
                                pointXElement.text = str(ddict["Position"][0])
                            else:
                                pointYElement = Dfct.SubElement(point, "PointY")
                                pointYElement.text = str(ddict["Position"][1])
                        if int(Dfct.childText(point, 'Num')) == (numPointSelected - 1) % 4:
                            if numPointSelected % 2 == 0:
                                pointXElement = Dfct.SubElement(point, "PointX")
                                pointXElement.text = str(ddict["Position"][0])
                            else:
                                pointYElement = Dfct.SubElement(point, "PointY")
                                pointYElement.text = str(ddict["Position"][1])

                for point in Dfct.SubElement(self.roiSelection, 'AllPoints'):
                    if int(Dfct.childText(point, 'Num')) == 0:
                        offsetX = int(float(Dfct.SubElement(point, "PointX").text))
                        offsetY = int(float(Dfct.SubElement(point, "PointY").text))
                for point in Dfct.SubElement(self.roiSelection, 'AllPoints'):
                    if int(Dfct.childText(point, 'Num')) == 2:
                        sizeX = int(float(Dfct.SubElement(point, "PointX").text))-offsetX
                        sizeY = int(float(Dfct.SubElement(point, "PointY").text))-offsetY

                self.widgetImage.groupBoxSettings.lineEditOffsetX.setText(str(int(offsetX)))
                self.widgetImage.groupBoxSettings.lineEditOffsetY.setText(str(int(offsetY)))
                self.widgetImage.groupBoxSettings.lineEditSizeX.setText(str(int(sizeX)))
                self.widgetImage.groupBoxSettings.lineEditSizeY.setText(str(int(sizeY)))

                self.widgetImage.groupBoxSettings.verify = True
                # self.widgetImage.groupBoxSettings.verifyValue()
                self.imageViewer.getRoiImage(changeRoiImage=False)

            else:
                # self.widgetImage.groupBoxSettings.verify = False
                # Not moving a point
                if self.currentPolygonSelected is not None and self.refPosX is not None and self.refPosY is not None:
                    # moving a rectangle
                    deltaX = ddict["Position"][0] - self.refPosX
                    deltaY = ddict["Position"][1] - self.refPosY
                    allPointsElement = Dfct.SubElement(self.currentPolygonSelected, "AllPoints")
                    for child in allPointsElement:
                        if child.tag == "Point":
                            pointXElement = Dfct.SubElement(child, "PointX")
                            deltaX = min(deltaX, self.imageViewer.image[0].getSizeX() - float(pointXElement.text))
                            deltaX = max(deltaX, -float(pointXElement.text))
                            pointYElement = Dfct.SubElement(child, "PointY")
                            deltaY = min(deltaY, self.imageViewer.image[0].getSizeY() - float(pointYElement.text))
                            deltaY = max(deltaY, -float(pointYElement.text))

                    self.widgetImage.groupBoxSettings.lineEditOffsetX.setText(str(int(self.widgetImage.groupBoxSettings.lineEditOffsetX.text())+int(deltaX)))
                    self.widgetImage.groupBoxSettings.lineEditOffsetY.setText(str(int(self.widgetImage.groupBoxSettings.lineEditOffsetY.text())+int(deltaY)))

                    self.refPosX = ddict["Position"][0]
                    self.refPosY = ddict["Position"][1]

                    self.widgetImage.groupBoxSettings.verify = True
                    self.widgetImage.groupBoxSettings.verifyValue()
                    self.imageViewer.getRoiImage(changeRoiImage=False)

        if ddict["Button"] == 1 and ddict["Event"] == "Release":
            self.widgetImage.groupBoxSettings.verifyValue()
            self.currentPointSelected = None
            #self.currentPolygonSelected = None
            self.refPosX = None
            self.refPosY = None
            self.imageViewer.standAlone = False


    def resetCurrentPolygonSelected(self):
        if self.currentPolygonSelected != None:
            selectionElement = Dfct.SubElement(self.currentPolygonSelected, "Selection")
            selectionElement.text = "False"
            self.currentPolygonSelected = None

if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook


    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    # foo = WidgetCameraToDisplay()
    # foo = WidgetTwoCameras()
    # foo = GroupBoxSettings()

    foo = WidgetCamera(cameraNum=0)
    foo.show()
    # foo.display()

    # foo2 = WidgetCamera(cameraNum=1)
    # foo2.show()
    # foo2.display()

    vrb.widgetCamera.append(foo)
    # vrb.widgetCamera.append(foo2)

    fct.displayCamera()

    # try:
    #     foo.show()
    #     foo.display()
    # except:
    #     foo.display()

    app.exec_()