import os, sys
import time
import traceback
import ctypes

import numpy as np

import xml.etree.ElementTree as xmlet
from PyQt5.QtCore import pyqtSignal, QPointF, Qt, QCoreApplication
from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore
from PyQt5.QtGui import QPixmap

import UsefullVariables as vrb
import UsefullWidgets as wgt
import UsefullFunctions as fct
import DatabaseFunction as Dfct
import UsefullTexts as txt

from HistogramWidget import HistogramWidget
from ImageViewer import WidgetImage, ImageViewer,ImageViewerWithScrollBar,WidgetInfo, customBlending
from Sliders import SimpleHorizontalSliderLabel
# from GraphicElements import GraphicElements
# from SelectorWidget import SelectorWidget, GroupBoxText
from Sliders import SimpleHorizontalSliderLabel

from RangeSlider import RangeSlider
import LutsWidget as luts
from LutsWidget import GroupBoxLut, getLutArray, lutFromVector

import PyIPSDK
import PyIPSDK.IPSDKIPLBasicMorphology as morpho
import PyIPSDK.IPSDKIPLAdvancedMorphology as advmorpho
import PyIPSDK.IPSDKIPLColor as color
import PyIPSDK.IPSDKIPLUtility as util
import PyIPSDK.IPSDKIPLFiltering as filtering
import PyIPSDK.IPSDKIPLBinarization as bin
import PyIPSDK.IPSDKIPLArithmetic as arithm
import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr
import PyIPSDK.IPSDKIPLGeometricTransform as gtrans
import PyIPSDK.IPSDKIPLShapeAnalysis as shapeanalysis
import PyIPSDK.IPSDKIPLShapeSegmentation as shapesegmentation
import PyIPSDK.IPSDKIPLIntensityTransform as itrans
import PyIPSDK.IPSDKIPLLogical as logic
import PyIPSDK.IPSDKIPLFeatureDetection as fd
import PyIPSDK.IPSDKIPLStats as stats
import PyIPSDK.IPSDKIPLClassification as classif
import PyIPSDK.IPSDKIPLRegistration as registration
import PyIPSDK.IPSDKIPLMachineLearning as ml

import napari
from napari.utils.geometry import rotation_matrix_from_vectors_3d
from napari_threedee.manipulators.base_manipulator import BaseManipulator
from napari_threedee.utils.napari_utils import data_to_world_normal, world_to_data_normal

from napari.qt.threading import thread_worker
os.environ['NAPARI_ASYNC'] = '1'

import vispy
from vispy import scene, visuals, gloo

from vispy.scene.visuals import Text
from vispy import keys


# Allows to create spoiler widgets
class SpoilerWidget(qt.QGroupBox):

    def __init__(self,widget,title = "",size = 200*vrb.ratio):
        qt.QGroupBox.__init__(self)

        self.widget = widget
        self.title = title
        self.size = size

        self.doubleLabelHeader = wgt.DoubleLabelSpoiler(self.title)
        self.doubleLabelHeader.labelCollapse.setText('▲')

        self.margins = 5
        self.spacing = 0

        # self.layout = qt.QGridLayout()
        self.layout = qt.QVBoxLayout()

        self.layout.setAlignment(Qt.AlignTop)
        self.setLayout(self.layout)
        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(self.margins, self.margins, self.margins, self.margins)
        self.layout.setSpacing(self.spacing)

        self.layout.addWidget(self.doubleLabelHeader)
        self.layout.addWidget(self.widget)

        self.doubleLabelHeader.label.mousePressEvent = self.expand
        self.doubleLabelHeader.labelCollapse.mousePressEvent = self.expand

        self.setStyleSheet("QGroupBox {border: 0px transparent; }")
        # self.setFixedHeight(self.doubleLabelHeader.height())

    def expand(self, forceEvent=None):

        try:
            if forceEvent == 'Close':
                self.widget.setVisible(False)
            elif forceEvent == 'Open':
                self.widget.setVisible(True)
            else:
                self.widget.setVisible(not(self.widget.isVisible()))

            # self.isExpanded = self.widget.isVisible()

            # if self.isExpanded:
            #     self.setFixedHeight(self.size)
            # else:
            #     self.setFixedHeight(30*vrb.ratio)

            self.doubleLabelHeader.labelCollapse.setText(('▼', '▲')[self.widget.isVisible()])

            self.changeHeight()

        except:
            traceback.print_exc(file=sys.stderr)

    def changeHeight(self,visible = None):
        if visible is None:
            visible = self.widget.isVisible()
        if visible:
            # self.setFixedHeight(self.doubleLabelHeader.height() + 2 * self.margins + self.spacing + self.widget.height())
            self.setFixedHeight(self.doubleLabelHeader.height() + 2 * self.margins + self.spacing + self.widget.height() - 10*vrb.ratio)
        else:
            self.setFixedHeight(self.doubleLabelHeader.height() + 2 * self.margins)

class Viewer3dSettingsScrollArea(qt.QScrollArea):

    def __init__(self):
        qt.QScrollArea.__init__(self)

        self.setWidgetResizable(True)
        # self.setHorizontalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)

        self.centralWidget = Viewer3dSettingsWidget()

        self.setWidget(self.centralWidget)

class Viewer3dSettingsWidget(qt.QWidget):

    def __init__(self, parent=None):
        qt.QWidget.__init__(self)

        self.loadingCamera = False

        self.keyboardLanguage = self.get_keyboard_language()

        if self.keyboardLanguage == "azerty":
            keyboardKeys = {
                keys.UP: (+1, 1), keys.DOWN: (-1, 1),
                keys.RIGHT: (+1, 2), keys.LEFT: (-1, 2),
                #
                'Z': (+1, 1), 'S': (-1, 1),
                'D': (+1, 2), 'Q': (-1, 2),
                'R': (+1, 3), 'F': (-1, 3),
                #
                'I': (+1, 4), 'K': (-1, 4),
                'L': (+1, 5), 'J': (-1, 5),
                'E': (+1, 6), 'A': (-1, 6),
                #
                keys.SPACE: (0, 1, 2, 3),  # 0 means brake, apply to translation
                # keys.ALT: (+5, 1),  # Turbo
            }
        else:
            keyboardKeys = {
                keys.UP: (+1, 1), keys.DOWN: (-1, 1),
                keys.RIGHT: (+1, 2), keys.LEFT: (-1, 2),
                #
                'W': (+1, 1), 'S': (-1, 1),
                'D': (+1, 2), 'A': (-1, 2),
                'R': (+1, 3), 'F': (-1, 3),
                #
                'I': (+1, 4), 'K': (-1, 4),
                'L': (+1, 5), 'J': (-1, 5),
                'E': (+1, 6), 'Q': (-1, 6),
                #
                keys.SPACE: (0, 1, 2, 3),  # 0 means brake, apply to translation
                # keys.ALT: (+5, 1),  # Turbo
            }


        if napari.__version__.startswith("0.4"):
            vrb.mainWindow.viewerNapariQt.camera._3D_turnCamera = scene.cameras.ArcballCamera(fov=0)
            vrb.mainWindow.viewerNapariQt.camera._3D_flyCamera = scene.cameras.fly.FlyCamera(fov=60)
            quaternion = vispy.util.quaternion.Quaternion(w=1, x=0, y=0, z=0, normalize=True)
            vrb.mainWindow.viewerNapariQt.camera._3D_flyCamera._quaternion = quaternion

            vrb.mainWindow.viewerNapariQt.camera._3D_flyCamera._keymap = keyboardKeys

            self.axis = scene.visuals.XYZAxis(parent=vrb.mainWindow.viewerNapariQt.view.scene, width=10)
            self.xLabel = Text('x', parent=vrb.mainWindow.viewerNapariQt.view.scene, color='red')
            self.yLabel = Text('y', parent=vrb.mainWindow.viewerNapariQt.view.scene, color='green')
            self.zLabel = Text('z', parent=vrb.mainWindow.viewerNapariQt.view.scene, color='blue')

        elif napari.__version__.startswith("0.5"):
            vrb.mainWindow.viewerNapariQt.canvas.camera._3D_turnCamera = scene.cameras.ArcballCamera(fov=0)
            vrb.mainWindow.viewerNapariQt.canvas.camera._3D_flyCamera = scene.cameras.fly.FlyCamera(fov=60)
            quaternion = vispy.util.quaternion.Quaternion(w=1, x=0, y=0, z=0, normalize=True)
            vrb.mainWindow.viewerNapariQt.canvas.camera._3D_flyCamera._quaternion = quaternion

            vrb.mainWindow.viewerNapariQt.canvas.camera._3D_flyCamera._keymap = keyboardKeys

            self.axis = scene.visuals.XYZAxis(parent=vrb.mainWindow.viewerNapariQt.canvas.view.scene, width=10)
            self.xLabel = Text('x', parent=vrb.mainWindow.viewerNapariQt.canvas.view.scene, color='red')
            self.yLabel = Text('y', parent=vrb.mainWindow.viewerNapariQt.canvas.view.scene, color='green')
            self.zLabel = Text('z', parent=vrb.mainWindow.viewerNapariQt.canvas.view.scene, color='blue')

        else:
            print("ERROR : Explorer needs napari 0.4.* or 0.5.* version, be sure to get the right versions.")
            return 1

        self.axis.visible = False
        self.xLabel.visible = False
        self.yLabel.visible = False
        self.zLabel.visible = False

        #Planes for 4 views mode
        self.zPlane = None
        self.yPlane = None
        self.xPlane = None

        #Roi selection for 4 views mode
        self.roi3D = None

        self.loadingSettings = False

        self.currentMinCur = None
        self.currentMaxCur = None

        self.visualizationTimer_start = 0
        self.visualizationTimer_end = sys.maxsize

        self.labelOpacity = qt.QLabel("Opacity")
        self.sliderOpacity = SimpleHorizontalSliderLabel(label="", minimum=0, maximum=100, defaultValue=100, ratio=100,
                                                       titleSize=0 * vrb.ratio, widthLabel=15*vrb.ratio)
        self.sliderOpacity.setStyleSheet('QGroupBox {border: 0px transparent; }')
        self.sliderOpacity.setFixedWidth(100 * vrb.ratio)
        self.sliderOpacity.setContentsMargins(0, 5, 0, 5)

        self.groupBoxOpacity = qt.QGroupBox()
        self.layoutOpacity = qt.QGridLayout()
        self.layoutOpacity.addWidget(self.labelOpacity, 0, 0)
        self.layoutOpacity.addWidget(self.sliderOpacity, 0, 1)
        self.groupBoxOpacity.setLayout(self.layoutOpacity)

        # parameters for grey volumes
        self.imageSettings = ImageSettings(parent=self)
        self.spoilerImageSettings = SpoilerWidget(self.imageSettings, title="3D settings")
        self.spoilerImageSettings.setContentsMargins(0,5*vrb.ratio,0,5*vrb.ratio)
        self.imageSettings.updateWidgets()
        self.spoilerImageSettings.changeHeight(visible=True)
        # self.imageSettings.updateWidgets()
        # self.spoilerImageSettings.expand()

        # parameters for clipping
        self.clippingSettings = ClippingSettings(parent=self)
        self.spoilerClippingSettings = SpoilerWidget(self.clippingSettings, title="Clipping settings")
        self.spoilerClippingSettings.setContentsMargins(0,5,0,5)
        self.spoilerClippingSettings.changeHeight(visible=True)

        # self.thresholdSettings = ThresholdSettings()
        # self.spoilerThresholdLeveling = SpoilerWidget(self.thresholdSettings, title="Threshold leveling")
        # self.spoilerThresholdLeveling.setContentsMargins(0, 0, 0, 0)
        # self.spoilerThresholdLeveling.changeHeight(visible=True)

        self.meshSimplificationSettings = MeshSimplificationSettings()
        self.spoilerMeshSimplificationSettings = SpoilerWidget(self.meshSimplificationSettings, title="Mesh simplification")
        self.spoilerMeshSimplificationSettings.setContentsMargins(0, 0, 0, 0)
        self.spoilerMeshSimplificationSettings.changeHeight(visible=True)
        self.spoilerMeshSimplificationSettings.setVisible(False)

        self.orthoslicesSettings = OrthoslicesSettings(parent=self)
        self.spoilerOrthoslicesSettings = SpoilerWidget(self.orthoslicesSettings, title="Orthoslices settings")
        self.spoilerOrthoslicesSettings.setContentsMargins(0, 0, 0, 0)
        self.spoilerOrthoslicesSettings.changeHeight(visible=True)
        self.spoilerOrthoslicesSettings.setVisible(False)
        self.spoilerOrthoslicesSettings.setMaximumHeight(100*vrb.ratio)

        self.viewerSettings = ViewerSettings(parent=self)
        self.spoilerViewerSettings = SpoilerWidget(self.viewerSettings, title="Viewer settings")
        self.spoilerViewerSettings.setContentsMargins(0, 0, 0, 0)
        self.spoilerViewerSettings.changeHeight(visible=True)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.groupBoxOpacity, 0, 0)
        self.layout.addWidget(self.spoilerImageSettings, 1, 0)
        self.layout.addWidget(self.spoilerClippingSettings, 2, 0)
        # self.layout.addWidget(self.spoilerThresholdLeveling, 3, 0)
        self.layout.addWidget(self.spoilerMeshSimplificationSettings, 3, 0)
        self.layout.addWidget(self.spoilerOrthoslicesSettings, 4, 0)
        self.layout.addWidget(self.spoilerViewerSettings, 5, 0)
        self.setLayout(self.layout)
        self.layout.setAlignment(Qt.AlignTop)
        self.setFixedWidth(190*vrb.ratio)


        #############    SIGNALS    ##############

        self.sliderOpacity.slider.valueChanged.connect(self.sliderOpacityChanged)

        #CLIPPING SETTINGS SIGNALS
        self.clippingSettings.sliderClipX.signalValueChanged.connect(self.updateClipping)
        self.clippingSettings.sliderClipY.signalValueChanged.connect(self.updateClipping)
        self.clippingSettings.sliderClipZ.signalValueChanged.connect(self.updateClipping)
        self.clippingSettings.buttonReset.clicked.connect(self.resetCliping)
        self.clippingSettings.buttonCrop.clicked.connect(self.applyClipping)
        self.clippingSettings.buttonBBox.clicked.connect(self.displayBBox)

        #VIEWER SETTINGS SIGNALS
        self.viewerSettings.sliderFov.slider.valueChanged.connect(self.sliderFovChanged)
        self.viewerSettings.checkBoxAxis.stateChanged.connect(self.changeDisplayAxis)
        self.viewerSettings.radioButtonCameraNormal.clicked.connect(self.changeCameraMode)
        self.viewerSettings.radioButtonCameraFly.clicked.connect(self.changeCameraMode)

        #IMAGE SETTINGS SIGNALS
        self.imageSettings.sliderGamma.slider.valueChanged.connect(self.sliderGammaChanged)
        self.imageSettings.sliderIsoThreshold.slider.valueChanged.connect(self.sliderIsoThresholdChanged)
        self.imageSettings.sliderAttenuation.slider.valueChanged.connect(self.sliderAttenuationChanged)
        self.imageSettings.comboBoxBlending.currentIndexChanged.connect(self.updateVisualizationFromSignal)
        self.imageSettings.comboBoxInterpolation.currentIndexChanged.connect(self.updateVisualizationFromSignal)
        self.imageSettings.comboBoxRendering.currentIndexChanged.connect(self.updateVisualizationFromSignal)
        self.imageSettings.comboBoxLabelsRendering.currentIndexChanged.connect(self.updateVisualizationFromSignal)
        self.imageSettings.comboBoxShading.currentIndexChanged.connect(self.updateVisualizationFromSignal)
        self.imageSettings.checkBoxWireframe.stateChanged.connect(self.updateVisualizationFromSignal)
        self.imageSettings.buttonColorWireframe.clicked.connect(self.updateVisualizationFromSignal)
        self.imageSettings.buttonReset.clicked.connect(self.reset3DParameters)

        self.meshSimplificationSettings.buttonApplySimplification.clicked.connect(self.applyMeshSimplification)

        #ORTHOSLICES AND XYZ PLANES SIGNALS
        # vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.sliderMoved.connect(self.sliderPlaneXPositionChanged)
        # vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.sliderMoved.connect(self.sliderPlaneYPositionChanged)
        # vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.sliderMoved.connect(self.sliderPlaneZPositionChanged)
        # self.orthoslicesSettings.sliderX.slider.valueChanged.connect(self.sliderPlaneXPositionChanged)
        # self.orthoslicesSettings.sliderY.slider.valueChanged.connect(self.sliderPlaneYPositionChanged)
        # self.orthoslicesSettings.sliderZ.slider.valueChanged.connect(self.sliderPlaneZPositionChanged)
        vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.sliderMoved.connect(self.sliderPlaneXPositionChanged)
        vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.sliderMoved.connect(self.sliderPlaneXPositionChanged)
        vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.sliderMoved.connect(self.sliderPlaneXPositionChanged)
        self.orthoslicesSettings.sliderX.slider.valueChanged.connect(self.sliderPlaneXPositionChanged)
        self.orthoslicesSettings.sliderY.slider.valueChanged.connect(self.sliderPlaneXPositionChanged)
        self.orthoslicesSettings.sliderZ.slider.valueChanged.connect(self.sliderPlaneXPositionChanged)


        # ORTHOSLICES SETTINGS SIGNALS

        vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.clicked.connect(self.showAxisPlane)
        vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.clicked.connect(self.showAxisPlane)
        vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.clicked.connect(self.showAxisPlane)
        self.orthoslicesSettings.buttonEyeX.clicked.connect(self.showAxisPlane)
        self.orthoslicesSettings.buttonEyeY.clicked.connect(self.showAxisPlane)
        self.orthoslicesSettings.buttonEyeZ.clicked.connect(self.showAxisPlane)
        self.orthoslicesSettings.sliderOpacity.slider.valueChanged.connect(self.sliderOrthosliceOpacityChanged)
        self.orthoslicesSettings.sliderThickness.slider.valueChanged.connect(self.sliderOrthosliceThicknessChanged)
        self.orthoslicesSettings.comboBoxBlending.currentIndexChanged.connect(self.updateVisualizationFromSignal)
        self.orthoslicesSettings.comboBoxRendering.currentIndexChanged.connect(self.updateVisualizationFromSignal)
        self.orthoslicesSettings.comboBoxLabelsRendering.currentIndexChanged.connect(self.updateVisualizationFromSignal)
        # self.orthoslicesSettings.comboBoxColormap.currentIndexChanged.connect(self.updateVisualizationFromSignal)
        self.orthoslicesSettings.comboBoxColormap.currentIndexChanged.connect(self.updateOrthosliceColorMap)
        self.orthoslicesSettings.sliderAttenuation.slider.valueChanged.connect(self.sliderAttenuationOrthoslicesChanged)

        self.orthoslicesSettings.buttonReset.clicked.connect(self.resetOrthoslicesParameters)
        ########################################## SIGNALS

        vrb.mainWindow.viewerNapariQt.canvas.events.mouse_double_click.connect(lambda event: self.on_mouse_double_click(vrb.mainWindow.viewerNapari, event))


############################################################
#                   ORTHOSLICES METHODS                    #
############################################################

    # # creates the 3 orthoslices if not created, and updates the layers if the orthoslices already exists
    # def updateOrthoslices(self,mode='4Views'):
    #     print("update ortho")
    #     currentLabel = vrb.mainWindow.currentLabel
    #
    #     if mode=='4Views':
    #         sliderX = vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX
    #         sliderY = vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY
    #         sliderZ = vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ
    #     # elif mode=='3D':
    #     else:
    #         sliderX = self.orthoslicesSettings.sliderX
    #         sliderY = self.orthoslicesSettings.sliderY
    #         sliderZ = self.orthoslicesSettings.sliderZ
    #
    #     print("get x y z")
    #     x = float(sliderX.slider.value())
    #     y = float(sliderY.slider.value())
    #     z = float(sliderZ.slider.value())
    #
    #     currentLabel.orthosliceX.plane.position = (x,x,x)
    #     currentLabel.orthosliceY.plane.position = (y,y,y)
    #     currentLabel.orthosliceZ.plane.position = (z,z,z)


    def get_keyboard_language(self):
        """
        Gets the keyboard language in use by the current
        active window process.
        """
        if os.name == "nt":
            languages_qwerty = {
                '0x409': "English - United States",
                '0x809': "English - United Kingdom",
                '0cc09': "English - Australia",
                '0x1009': "English - Canada",
                '0x2409': "English - Caribbean",
                '0x3c09': "English - Hong Kong SAR",
                '0x4009': "English - India",
                '0x3809': "English - Indonesia",
                '0x1809': "English - Ireland",
                '0x2009': "English - Jamaica",
                '0x4409': "English - Malaysia",
                '0x1409': "English - New Zealand",
                '0x3409': "English - Philippines",
                '0x4809': "English - Singapore",
                '0x1c09': "English - South Africa",
                '0x425': "Estonian",
                '0x413': "Dutch - Netherlands",
                '0x813': "Dutch - Belgium",
                '0cc0a': "Spanish - Spain (Modern Sort)",
                '0c40a': "Spanish - Spain (Traditional Sort)",
                '0x2c0a': "Spanish - Argentina",
                '0xe40a': "Spanish - Latin America",
                '0c80a': "Spanish - Mexico",
                '0x4c0a': "Spanish - Nicaragua",
                '0x180a': "Spanish - Panama",
                '0x3c0a': "Spanish - Paraguay",
                '0x280a': "Spanish - Peru",
                '0x500a': "Spanish - Puerto Rico",
                '0x540a': "Spanish - United States",
                '0x380a': "Spanish - Uruguay",
                '0x200a': "Spanish - Venezuela",
                '0x410': "Italian - Italy",
                '0x810': "Italian - Switzerland",
                '0x414': "Norwegian (Bokmål)",
                '0x814': "Norwegian (Nynorsk)",
                '0c41d': "Swedish",
                '0c81d': "Swedish - Finland",
                '0x405': "Czech",
                '0c40b': "Finnish",
                '0x406': "Danish",
                '0c40f': "Icelandic",
                '0x407': "German - Germany",
                '0cc07': "German - Austria",
                '0x1407': "German - Liechtenstein",
                '0x1007': "German - Luxembourg",
                '0x807': "German - Switzerland",
                '0x438': "Faroese",
                '0x408': "Greek",
                '0x426': "Latvian", '0x427': "Lithuanian",
                '0c43a': "Maltese",
                '0x415': "Polish",
                '0x416': "Portuguese - Brazil",
                '0x418': "Romanian",
                '0c41b': "Slovak",
                '0c41f': "Turkish"
            }

            languages_azerty = {
                '0x40c': "French - France",
                '0c80c': "French - Belgium",
                '0x2c0c': "French - Cameroon",
                '0cc0c': "French - Canada",
                '0x240c': "French - Democratic Rep. of Congo",
                '0x300c': "French - Cote d'Ivoire",
                '0x3c0c': "French - Haiti",
                '0x140c': "French - Luxembourg",
                '0x340c': "French - Mali",
                '0x180c': "French - Monaco",
                '0x380c': "French - Morocco",
                '0xe40c': "French - North Africa",
                '0x200c': "French - Reunion",
                '0x280c': "French - Senegal",
                '0x100c': "French - Switzerland",
                '0x1c0c': "French - West Indies",
                '0x401': "Arabic - Saudi Arabia",
                '0x1401': "Arabic - Algeria",
                '0x3c01': "Arabic - Bahrain",
                '0cc01': "Arabic - Egypt",
                '0x801': "Arabic - Iraq",
                '0x2c01': "Arabic - Jordan",
                '0x3401': "Arabic - Kuwait",
                '0x3001': "Arabic - Lebanon",
                '0x1001': "Arabic - Libya",
                '0x1801': "Arabic - Morocco",
                '0x2001': "Arabic - Oman",
                '0x4001': "Arabic - Qatar",
                '0x2801': "Arabic - Syria",
                '0x1c01': "Arabic - Tunisia",
                '0x3801': "Arabic - U.A.E.",
                '0x2401': "Arabic - Yemen",
                '0c45f': "Tamazight (Arabic)",
                '0c42a': "Vietnamese",
                '0x816': "Portuguese - Portugal"
            }

            user32 = ctypes.WinDLL('user32', use_last_error=True)

            # Get the current active window handle
            handle = user32.GetForegroundWindow()

            # Get the thread id from that window handle
            threadid = user32.GetWindowThreadProcessId(handle, 0)

            # Get the keyboard layout id from the threadid
            layout_id = user32.GetKeyboardLayout(threadid)

            # Extract the keyboard language id from the keyboard layout id
            language_id = layout_id & (2 ** 16 - 1)

            # Convert the keyboard language id from decimal to hexadecimal
            language_id_hex = hex(language_id)

            # Check if the hex value is in the dictionary.
            if language_id_hex in languages_azerty.keys():
                return "azerty"
            elif language_id_hex in languages_qwerty.keys():
                return "qwerty"
            else:
                return "qwerty"
        else:
            from subprocess import check_output
            cmd_output = check_output(["setxkbmap", "-query"]).decode("utf-8")
            layoutStr = cmd_output.split(" ")[-1][:-1]

            azerty_layouts = ["fr", "be", "ca", "cd", "ml", "ma", "sn"]
            if layoutStr in azerty_layouts:
                return "azerty"
            else:
                return "qwerty"



    def resetOrthoslicesParameters(self):
        ''' Resets orthoslices parameters to default when
            clicking the reset button in orthoslices settings area.
        '''

        currentLabel = vrb.mainWindow.currentLabel
        if currentLabel is not None:
            if currentLabel.orthosliceX is not None:
                self.reorderLayers()
                self.orthoslicesSettings.updateMode()

                vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.setActivation(
                    not (currentLabel.orthosliceX.visible))
                vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.setActivation(
                    not (currentLabel.orthosliceY.visible))
                vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.setActivation(
                    not (currentLabel.orthosliceZ.visible))

                # self.orthoslicesSettings.sliderX.slider.blockSignals(True)
                # self.orthoslicesSettings.sliderY.slider.blockSignals(True)
                # self.orthoslicesSettings.sliderZ.slider.blockSignals(True)
                # vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.blockSignals(True)
                # vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.blockSignals(True)
                # vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.blockSignals(True)

                self.orthoslicesSettings.sliderX.slider.setValue(int(currentLabel.image.getSizeX()/2))
                self.orthoslicesSettings.sliderY.slider.setValue(int(currentLabel.image.getSizeY()/2))
                self.orthoslicesSettings.sliderZ.slider.setValue(int(currentLabel.image.getSizeZ()/2))

                vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.setValue(int(currentLabel.image.getSizeX()/2))
                vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.setValue(int(currentLabel.image.getSizeY()/2))
                vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.setValue(int(currentLabel.image.getSizeZ()/2))

                currentLabel.orthosliceX.plane.position = (int(currentLabel.image.getSizeX()/2),int(currentLabel.image.getSizeX()/2),int(currentLabel.image.getSizeX()/2))
                currentLabel.orthosliceY.plane.position = (int(currentLabel.image.getSizeY()/2),int(currentLabel.image.getSizeY()/2),int(currentLabel.image.getSizeY()/2))
                currentLabel.orthosliceZ.plane.position = (int(currentLabel.image.getSizeZ()/2),int(currentLabel.image.getSizeZ()/2),int(currentLabel.image.getSizeZ()/2))

                # self.orthoslicesSettings.sliderX.slider.blockSignals(False)
                # self.orthoslicesSettings.sliderY.slider.blockSignals(False)
                # self.orthoslicesSettings.sliderZ.slider.blockSignals(False)
                # vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.blockSignals(
                #     False)
                # vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.blockSignals(
                #     False)
                # vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.blockSignals(
                #     False)

                # self.orthoslicesSettings.comboBoxBlending.blockSignals(True)
                # self.orthoslicesSettings.comboBoxRendering.blockSignals(True)
                # self.orthoslicesSettings.comboBoxColormap.blockSignals(True)
                self.orthoslicesSettings.sliderOpacity.blockSignals(True)
                self.orthoslicesSettings.sliderAttenuation.blockSignals(True)
                self.orthoslicesSettings.sliderThickness.blockSignals(True)

                self.orthoslicesSettings.comboBoxBlending.setCurrentText('translucent')

                if currentLabel.image.getBufferType() not in [PyIPSDK.eIBT_Label8, PyIPSDK.eIBT_Label16, PyIPSDK.eIBT_Label32]:

                    self.orthoslicesSettings.comboBoxRendering.setCurrentText('translucent')
                    self.orthoslicesSettings.sliderAttenuation.slider.setValue(0.05*100)

                    if currentLabel.image.getBufferType() == PyIPSDK.eIBT_Binary:
                        self.orthoslicesSettings.comboBoxColormap.setCurrentText("Binary")
                    else:
                        self.orthoslicesSettings.comboBoxColormap.setCurrentText("Classic")
                else:
                    self.orthoslicesSettings.comboBoxLabelsRendering.setCurrentText("translucent")
                    self.orthoslicesSettings.comboBoxColormap.setCurrentText("Random")

                self.orthoslicesSettings.sliderThickness.slider.setValue(0)
                self.orthoslicesSettings.sliderOpacity.slider.setValue(100)

                # self.orthoslicesSettings.comboBoxBlending.blockSignals(False)
                # self.orthoslicesSettings.comboBoxRendering.blockSignals(False)
                # self.orthoslicesSettings.comboBoxColormap.blockSignals(False)
                self.orthoslicesSettings.sliderOpacity.blockSignals(False)
                self.orthoslicesSettings.sliderAttenuation.blockSignals(False)
                self.orthoslicesSettings.sliderThickness.blockSignals(False)

                # self.updateOrthosliceColorMap()

                # currentLabel.orthosliceX.refresh()
                # currentLabel.orthosliceY.refresh()
                # currentLabel.orthosliceZ.refresh()

##################### ORTHOSLICES METHODS #################



############################################################
#                   VIEW PLANES METHODS                    #
############################################################

    def showAxisPlane(self):
        '''Shows or hides the colored planes in 4 views mode
           when using the eyes buttons'''

        widget = self.sender()
        activate = widget.activate
        widget.setActivation(not activate)

        if vrb.mainWindow.currentLabel is not None:
            curPlane = None
            if not vrb.mainWindow.currentLabel.buttonOrthoslices.activate: # orthoslice mode is on

                if widget == vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye:
                    self.orthoslicesSettings.buttonEyeX.setActivation(widget.activate)
                    curPlane = vrb.mainWindow.currentLabel.orthosliceX
                elif widget == self.orthoslicesSettings.buttonEyeX:
                    vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.setActivation(not activate)
                    curPlane = vrb.mainWindow.currentLabel.orthosliceX
                elif widget == vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye:
                    self.orthoslicesSettings.buttonEyeY.setActivation(widget.activate)
                    curPlane = vrb.mainWindow.currentLabel.orthosliceY
                elif widget == self.orthoslicesSettings.buttonEyeY:
                    vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.setActivation(not activate)
                    curPlane = vrb.mainWindow.currentLabel.orthosliceY
                elif widget == vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye:
                    self.orthoslicesSettings.buttonEyeZ.setActivation(widget.activate)
                    curPlane = vrb.mainWindow.currentLabel.orthosliceZ
                elif widget == self.orthoslicesSettings.buttonEyeZ:
                    vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.setActivation(widget.activate)
                    curPlane = vrb.mainWindow.currentLabel.orthosliceZ

            else: # updates visibility of colored planes
                if widget == vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye:
                    self.orthoslicesSettings.buttonEyeX.setActivation(widget.activate)
                    curPlane = self.xPlane
                elif widget == vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye:
                    self.orthoslicesSettings.buttonEyeY.setActivation(widget.activate)
                    curPlane = self.yPlane
                elif widget == vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye:
                    self.orthoslicesSettings.buttonEyeZ.setActivation(widget.activate)
                    curPlane = self.zPlane

            if curPlane is not None:
                curPlane.visible = activate

##################### VIEW PLANES METHODS  #############



############################################################
#               MESH SIMPLIFICATION METHODS                #
############################################################

    def applyMeshSimplification(self):
        '''Changes the surface data in Napari with simplified data,
           and changes the informations displayed about the number
           of triangles between original and simplified mesh.
        '''

        currentLabel = vrb.mainWindow.currentLabel

        if currentLabel is not None and currentLabel.layer is not None and currentLabel.objectType == "Mesh":

            try:
                if self.meshSimplificationSettings.lineEditReduction.text() != "":
                    ratio = float(self.meshSimplificationSettings.lineEditReduction.text())
                    ratio = max(0,ratio)
                    ratio = min(1,ratio)
                else:
                    ratio = float(self.meshSimplificationSettings.lineEditReduction.placeholderText())
                if self.meshSimplificationSettings.lineEditMinimumFaces.text() != "":
                    minFaces = float(self.meshSimplificationSettings.lineEditMinimumFaces.text())
                else:
                    minFaces = float(self.meshSimplificationSettings.lineEditMinimumFaces.placeholderText())

                vertices, faces, values, nbFaces = fct.createSurfaceForNapari(currentLabel.mesh,ratio=ratio,minFaces=minFaces)

                currentLabel.surface = (vertices, faces, values)

                currentLabel.ratio = ratio
                currentLabel.minFaces = minFaces

                currentLabel.layer.data = currentLabel.surface

                # currentLabel.nbFaces = nbFaces
                currentLabel.nbFacesSimplify = len(faces)

                self.meshSimplificationSettings.labelMeshNbFaces.setText("Original mesh: " + str(currentLabel.nbFaces) + " faces")
                self.meshSimplificationSettings.labelMeshSimplifyNbFaces.setText("Simplified mesh : " + str(currentLabel.nbFacesSimplify) + " faces")

            except:
                traceback.print_exc(file=sys.stderr)

##################### MESH SIMPLIFICATION METHODS #############



############################################################
#               CLIPPING RELATED METHODS                   #
############################################################

    # resets the clipping (updated for calibration)
    def resetCliping(self):
        '''Resets the clipping of volumes or meshes.

            update : takes the calibration in account for volumes.
        '''
        currentLabel = vrb.mainWindow.currentLabel
        if currentLabel is not None:
            currentLayer = currentLabel.layer
            if currentLayer is not None:
                if currentLabel.objectType == "Image":
                    startX = 0
                    startY = 0
                    startZ = 0
                    endX = currentLabel.image.getSizeX() - 1
                    endY = currentLabel.image.getSizeY() - 1
                    endZ = currentLabel.image.getSizeZ() - 1

                    # retrieve the calibration
                    if currentLabel.image.hasGeometricCalibration():
                        
                        calibration = currentLabel.image.getGeometricCalibration()
                    else:
                        
                        calibration = PyIPSDK.createGeometricCalibration(1, 1, 1, "px")
                        currentLabel.image.setGeometricCalibration(calibration)

                    # retrieve the origin of the calibration (for cropped volumes)
                    originX = calibration.getXOriginUncalibrated()
                    originY = calibration.getYOriginUncalibrated()
                    originZ = calibration.getZOriginUncalibrated()


                elif currentLabel.objectType == "Mesh":

                    startX = int(currentLabel.startValues[2])-1
                    startY = int(currentLabel.startValues[1])-1
                    startZ = int(currentLabel.startValues[0])-1
                    endX = round(currentLabel.endValues[2])
                    endY = round(currentLabel.endValues[1])
                    endZ = round(currentLabel.endValues[0])

                    try:
                        # retrieve the origin of the calibration (for cropped volumes)
                        originX = currentLabel.mesh.calibration.getXOriginUncalibrated()
                        originY = currentLabel.mesh.calibration.getYOriginUncalibrated()
                        originZ = currentLabel.mesh.calibration.getZOriginUncalibrated()
                    except:
                        # retrieve the origin of the calibration (for cropped volumes)
                        originX = 0
                        originY = 0
                        originZ = 0

                if len(currentLayer.scale) != 3:
                    scaleX = currentLayer.scale[3]
                    scaleY = currentLayer.scale[2]
                    scaleZ = currentLayer.scale[1]
                else:
                    scaleX = currentLayer.scale[2]
                    scaleY = currentLayer.scale[1]
                    scaleZ = currentLayer.scale[0]

                planeX_start_position = ((startX+originX)*scaleZ, (startX+originX)*scaleY, (startX+originX)*scaleX)
                planeY_start_position = ((startY+originY)*scaleZ, (startY+originY)*scaleY, (startY+originY)*scaleX)
                planeZ_start_position = ((startZ+originZ)*scaleZ, (startZ+originZ)*scaleY, (startZ+originZ)*scaleX)
                planeX_end_position = ((endX+originX)*scaleZ, (endX+originX)*scaleY, (endX+originX)*scaleX)
                planeY_end_position = ((endY+originY)*scaleZ, (endY+originY)*scaleY, (endY+originY)*scaleX)
                planeZ_end_position = ((endZ+originZ)*scaleZ, (endZ+originZ)*scaleY, (endZ+originZ)*scaleX)

                # update clipping planes positions with origin and a scale for the calibration
                planeX_start = {
                    'position': planeX_start_position,
                    'normal': (0, 0, 1),
                    'enabled': True
                }
                planeX_end = {
                    'position': planeX_end_position,
                    'normal': (0, 0, -1),
                    'enabled': True
                }
                planeY_start = {
                    'position': planeY_start_position,
                    'normal': (0, 1, 0),
                    'enabled': True
                }
                planeY_end = {
                    'position': planeY_end_position,
                    'normal': (0, -1, 0),
                    'enabled': True
                }
                planeZ_start = {
                    'position': planeZ_start_position,
                    'normal': (1, 0, 0),
                    'enabled': True
                }
                planeZ_end = {
                    'position': planeZ_end_position,
                    'normal': (-1, 0, 0),
                    'enabled': True
                }

                if fct.compareExperimentalClippingPlanes(currentLayer.experimental_clipping_planes,
                                                         [planeX_start, planeX_end, planeY_start, planeY_end,
                                                          planeZ_start, planeZ_end]) == False:
                    currentLayer.experimental_clipping_planes = [planeX_start, planeX_end, planeY_start,
                                                                       planeY_end, planeZ_start, planeZ_end]
                    if currentLabel.objectType == "Mesh":
                        currentLayer.refresh()



                self.clippingSettings.sliderClipX.blockSignals(True)
                self.clippingSettings.sliderClipY.blockSignals(True)
                self.clippingSettings.sliderClipZ.blockSignals(True)
                self.clippingSettings.sliderClipX.setMinMaxValues(startX, endX)
                self.clippingSettings.sliderClipY.setMinMaxValues(startY, endY)
                self.clippingSettings.sliderClipZ.setMinMaxValues(startZ, endZ)
                self.clippingSettings.sliderClipX.setValues(startX, endX)
                self.clippingSettings.sliderClipY.setValues(startY, endY)
                self.clippingSettings.sliderClipZ.setValues(startZ, endZ)
                self.clippingSettings.sliderClipX.blockSignals(False)
                self.clippingSettings.sliderClipY.blockSignals(False)
                self.clippingSettings.sliderClipZ.blockSignals(False)





    # updates the clipping of a volume (updated for calibration)
    def updateClipping(self, valueMin,valueMax):
        ''' Changes the clipping of volumes or meshes when moving
            the clipping sliders.

            Update : Takes the calibration in account for volumes.
        '''
        try:
            currentLabel = vrb.mainWindow.currentLabel

            if currentLabel is not None:

                if (currentLabel.objectType == "Image" and currentLabel.image.getSizeZ() > 1) or currentLabel.objectType == "Mesh":

                    currentLayer = currentLabel.layer
                    # if currentLabel.objectType == "Image" and currentLabel.layer is not None or currentLabel.objectType == "Mesh":
                    if currentLayer is not None:

                        # gets min and max values of clipping sliders
                        startX = self.clippingSettings.sliderClipX.currentValueMin
                        endX = self.clippingSettings.sliderClipX.currentValueMax
                        startY = self.clippingSettings.sliderClipY.currentValueMin
                        endY = self.clippingSettings.sliderClipY.currentValueMax
                        startZ = self.clippingSettings.sliderClipZ.currentValueMin
                        endZ = self.clippingSettings.sliderClipZ.currentValueMax


                        # Retrieve origin if label type is an image, and sets the origin to (0,0,0) for meshes
                        if currentLabel.objectType == "Image":
                            # retrieve calibration
                            if currentLabel.image.hasGeometricCalibration():
                                calibration = currentLabel.image.getGeometricCalibration()
                            else:
                                calibration = PyIPSDK.createGeometricCalibration(1, 1, 1, "px")
                                currentLabel.image.setGeometricCalibration(calibration)

                            # retrieve origin of volume
                            originX = calibration.getXOriginUncalibrated()
                            originY = calibration.getYOriginUncalibrated()
                            originZ = calibration.getZOriginUncalibrated()

                        elif currentLabel.objectType == "Mesh":

                            try:
                                # retrieve origin of volume
                                originX = currentLabel.mesh.calibration.getXOriginUncalibrated()
                                originY = currentLabel.mesh.calibration.getYOriginUncalibrated()
                                originZ = currentLabel.mesh.calibration.getZOriginUncalibrated()
                            except:
                                originX = 0
                                originY = 0
                                originZ = 0

                        if len(currentLayer.scale) != 3:
                            scaleX = currentLayer.scale[3]
                            scaleY = currentLayer.scale[2]
                            scaleZ = currentLayer.scale[1]
                        else:
                            scaleX = currentLayer.scale[2]
                            scaleY = currentLayer.scale[1]
                            scaleZ = currentLayer.scale[0]

                        planeX_start_position = ((startX + originX) * scaleZ,
                                                 (startX + originX) * scaleY,
                                                 (startX + originX) * scaleX)
                        planeY_start_position = ((startY + originY) * scaleZ,
                                                 (startY + originY) * scaleY,
                                                 (startY + originY) * scaleX)
                        planeZ_start_position = ((startZ + originZ) * scaleZ,
                                                 (startZ + originZ) * scaleY,
                                                 (startZ + originZ) * scaleX)
                        planeX_end_position = ((endX + originX) * scaleZ,
                                               (endX + originX) * scaleY,
                                               (endX + originX) * scaleX)
                        planeY_end_position = ((endY + originY) * scaleZ,
                                               (endY + originY) * scaleY,
                                               (endY + originY) * scaleX)
                        planeZ_end_position = ((endZ + originZ) * scaleZ,
                                               (endZ + originZ) * scaleY,
                                               (endZ + originZ) * scaleX)


                       # updates clipping planes positions with origin and scale
                        planeX_start = {
                            'position': planeX_start_position,
                            'normal': (0, 0, 1),
                            'enabled': True
                        }

                        planeX_end = {
                            'position': planeX_end_position,
                            'normal': (0, 0, -1),
                            'enabled': True
                        }

                        planeY_start = {
                            'position': planeY_start_position,
                            'normal': (0, 1, 0),
                            'enabled': True
                        }

                        planeY_end = {
                            'position': planeY_end_position,
                            'normal': (0, -1, 0),
                            'enabled': True
                        }

                        planeZ_start = {
                            'position': planeZ_start_position,
                            'normal': (1, 0, 0),
                            'enabled': True
                        }

                        planeZ_end = {
                            'position': planeZ_end_position,
                            'normal': (-1, 0, 0),
                            'enabled': True
                        }

                        if fct.compareExperimentalClippingPlanes(currentLayer.experimental_clipping_planes, [planeX_start, planeX_end, planeY_start, planeY_end, planeZ_start, planeZ_end]) == False:
                            currentLayer.experimental_clipping_planes = [planeX_start, planeX_end, planeY_start, planeY_end, planeZ_start, planeZ_end]
                            # currentLayer.refresh()
                            if currentLabel.objectType == "Mesh":
                                currentLayer.refresh()

        except:
            traceback.print_exc(file=sys.stderr)


    # creates a new volume using the clipping given
    def applyClipping(self):
        ''' Creates a new volume using the clipping given.

            Update : Takes the calibration and origins of
             volume in account.

            Restrictions : Only on volumes, not on meshes.
        '''

        try:
            currentLabel = vrb.mainWindow.currentLabel
            if currentLabel is not None:
                vrb.mainWindow.toggleGroupBoxProcessing(True)
                curLayer = currentLabel.layer
                if currentLabel.objectType == "Image" and curLayer is not None:
                    img = vrb.mainWindow.currentLabel.image
                    if len(curLayer.experimental_clipping_planes) != 0:
                        # x = int(curLayer.experimental_clipping_planes[0].position[0])/curLayer.scale[2]
                        # y = int(curLayer.experimental_clipping_planes[2].position[0])/curLayer.scale[1]
                        # z = int(curLayer.experimental_clipping_planes[4].position[0])/curLayer.scale[0]
                        # x = int(curLayer.experimental_clipping_planes[0].position[0])
                        # y = int(curLayer.experimental_clipping_planes[2].position[0])
                        # z = int(curLayer.experimental_clipping_planes[4].position[0])

                        # sizeX = int(curLayer.experimental_clipping_planes[1].position[0] -
                        #            curLayer.experimental_clipping_planes[0].position[0])
                        # sizeY = int(curLayer.experimental_clipping_planes[3].position[0] -
                        #            curLayer.experimental_clipping_planes[2].position[0])
                        # sizeZ = int(curLayer.experimental_clipping_planes[5].position[0] -
                        #            curLayer.experimental_clipping_planes[4].position[0])

                        x = int(self.clippingSettings.sliderClipX.currentValueMin)
                        y = int(self.clippingSettings.sliderClipY.currentValueMin)
                        z = int(self.clippingSettings.sliderClipZ.currentValueMin)

                        sizeX = int(self.clippingSettings.sliderClipX.currentValueMax)-x
                        sizeY = int(self.clippingSettings.sliderClipY.currentValueMax)-y
                        sizeZ = int(self.clippingSettings.sliderClipZ.currentValueMax)-z

                        roiImg = util.getROI3dImg(img, x, y, z, sizeX, sizeY, sizeZ)

                        output = roiImg

                        outputDict = {}
                        outputDict["Name"] = ""
                        outputDict["Type"] = "Image"

                        functionXmlElement = xmlet.Element('FunctionCall')
                        Dfct.SubElement(functionXmlElement, 'Name').text = 'applyClipping'
                        paramsNode = Dfct.SubElement(functionXmlElement, 'Parameters')

                        label, realOutputName = vrb.mainWindow.addResult(output, outputDict, functionXmlElement)

                    vrb.mainWindow.toggleGroupBoxProcessing(False)
        except:
            traceback.print_exc(file=sys.stderr)


    def displayBBox(self):
        """ Displays the boundind box of a volume or mesh.

            Helps to visualize the borders of an object when clipping.
        """

        self.clippingSettings.buttonBBox.changeActivation()

        if vrb.mainWindow.currentLabel is not None:
            if self.clippingSettings.buttonBBox.activate:
                vrb.mainWindow.currentLabel.layer.bounding_box.visible = True
                if not vrb.mainWindow.currentLabel.objectType == "Mesh" and not vrb.mainWindow.currentLabel.buttonOrthoslices.activate:
                    vrb.mainWindow.currentLabel.orthosliceX.bounding_box.visible = True
            else:
                vrb.mainWindow.currentLabel.layer.bounding_box.visible = False
                if not vrb.mainWindow.currentLabel.objectType == "Mesh" and not vrb.mainWindow.currentLabel.buttonOrthoslices.activate:
                    vrb.mainWindow.currentLabel.orthosliceX.bounding_box.visible = False

##################### CLIPPING RELATED METHODS #############


############################################################
#                  3D PARAMETERS METHODS                   #
############################################################

    def reset3DParameters(self):
        ''' Resets all 3D parameters to default value when
            clicking the reset button.
        '''

        currentLabel = vrb.mainWindow.currentLabel
        if currentLabel is not None:
            currentLayer = currentLabel.layer
            if currentLayer is not None:

                self.sliderOpacity.blockSignals(True)
                self.imageSettings.sliderGamma.blockSignals(True)
                self.imageSettings.sliderAttenuation.blockSignals(True)
                self.imageSettings.sliderIsoThreshold.blockSignals(True)
                self.imageSettings.comboBoxRendering.blockSignals(True)
                self.imageSettings.comboBoxBlending.blockSignals(True)
                self.imageSettings.comboBoxShading.blockSignals(True)
                self.imageSettings.comboBoxInterpolation.blockSignals(True)

                self.sliderOpacity.slider.setValue(100)
                self.imageSettings.sliderGamma.slider.setValue(10)

                if currentLabel.objectType == "Image":

                    if currentLabel.image.getBufferType() in [PyIPSDK.eIBT_Label8, PyIPSDK.eIBT_Label16, PyIPSDK.eIBT_Label32]:
                        self.imageSettings.comboBoxBlending.setCurrentIndex(self.imageSettings.comboBoxBlending.findText('translucent'))
                        self.imageSettings.comboBoxLabelsRendering.setCurrentIndex(self.imageSettings.comboBoxLabelsRendering.findText('iso_categorical'))
                    else:
                        self.imageSettings.comboBoxInterpolation.setCurrentIndex(self.imageSettings.comboBoxInterpolation.findText("linear"))
                        self.imageSettings.comboBoxRendering.setCurrentIndex(self.imageSettings.comboBoxRendering.findText("attenuated_mip"))
                        self.imageSettings.comboBoxBlending.setCurrentIndex(self.imageSettings.comboBoxBlending.findText('translucent'))
                        self.imageSettings.sliderAttenuation.slider.setValue(int(0.05*100))


                        # currentLayer.contrast_limits = [currentLabel.valueMinRef, currentLabel.valueMaxRef]

                        if currentLabel.image.getBufferType() == PyIPSDK.eIBT_Real32:
                            valueMin = float(currentLabel.valueMinRef)
                            valueMax = float(currentLabel.valueMaxRef)
                        else:
                            valueMin = int(currentLabel.valueMinRef)
                            valueMax = int(currentLabel.valueMaxRef)

                        vrb.mainWindow.groupMenu.groupContrast.lineEditMin.setText(fct.numberCalibration(valueMin))
                        vrb.mainWindow.groupMenu.groupContrast.lineEditMax.setText(fct.numberCalibration(valueMax))

                        vrb.mainWindow.groupMenu.groupContrast.rangeSlider.setValues(valueMin, valueMax)

                else:
                    self.imageSettings.checkBoxWireframe.setChecked(False)
                    self.imageSettings.buttonColorWireframe.setColor([1,0,0])
                    self.imageSettings.comboBoxShading.setCurrentIndex(self.imageSettings.comboBoxShading.findText('flat'))
                    self.imageSettings.comboBoxBlending.setCurrentIndex(self.imageSettings.comboBoxBlending.findText('translucent'))

            self.sliderOpacity.blockSignals(False)
            self.imageSettings.sliderGamma.blockSignals(False)
            self.imageSettings.sliderAttenuation.blockSignals(False)
            self.imageSettings.sliderIsoThreshold.blockSignals(False)
            self.imageSettings.comboBoxRendering.blockSignals(False)
            self.imageSettings.comboBoxBlending.blockSignals(False)
            self.imageSettings.comboBoxShading.blockSignals(False)
            self.imageSettings.comboBoxInterpolation.blockSignals(False)

            self.imageSettings.updateWidgets()
            # currentLayer.refresh()


############################################################
#                  GLOBAL METHODS                          #
############################################################

    def on_mouse_double_click(self, viewer, event):
        '''A function that detect a double click inside the
           napari viewer.

           If in 4views mode the double click brings to the 3d mode.
        '''
        if vrb.mainWindow.widgetImage.modeViewer == "4Views":
            if event.button == 1:  # left click button
                vrb.mainWindow.widgetImage.changeModeViewer("3D")
                vrb.mainWindow.groupMenu.groupViewer.button3dMode.setActivation(True)
                vrb.mainWindow.groupMenu.groupViewer.button4ViewsMode.setActivation(False)


    def updateVisualizationFromSignal(self):
        self.updateVisualization()

    def updateVisualization(self,label=None):
        ''' Called after a loading of all settings,
            and when parameters are changed through UI.

            Sets the new parameters with widgets values and updates the view.
        '''
        try:
            if label is None:
                currentLabel = vrb.mainWindow.currentLabel
            else:
                currentLabel = label

            if currentLabel is not None:
                if currentLabel.hasNapari:
                    currentLayer = currentLabel.layer

                    if currentLayer.opacity != float(self.sliderOpacity.slider.value() / 100):
                        currentLayer.opacity = float(self.sliderOpacity.slider.value() / 100)
                    if currentLayer.blending != self.imageSettings.comboBoxBlending.currentText():
                        currentLayer.blending = self.imageSettings.comboBoxBlending.currentText()

                    if currentLabel.objectType == "Image" and currentLayer.visible:

                        # vrb.mainWindow.viewerNapari.camera.perspective = self.viewerSettings.sliderFov.slider.value()

                        if currentLabel.image.getBufferType() not in [PyIPSDK.eIBT_Label8, PyIPSDK.eIBT_Label16,
                                                                      PyIPSDK.eIBT_Label32]:

                            if currentLayer.gamma != float(self.imageSettings.sliderGamma.slider.value() / 10):
                                currentLayer.gamma = float(self.imageSettings.sliderGamma.slider.value() / 10)

                            if currentLayer.interpolation3d != self.imageSettings.comboBoxInterpolation.currentText():
                                currentLayer.interpolation3d = self.imageSettings.comboBoxInterpolation.currentText()

                            if currentLayer.rendering != self.imageSettings.comboBoxRendering.currentText():
                                currentLayer.rendering = self.imageSettings.comboBoxRendering.currentText()

                            if currentLayer.attenuation != float(
                                    self.imageSettings.sliderAttenuation.slider.value() / 100):
                                currentLayer.attenuation = float(
                                    self.imageSettings.sliderAttenuation.slider.value() / 100)
                            if currentLabel.image.getBufferType() not in [PyIPSDK.eIBT_Binary, PyIPSDK.eIBT_Label8,
                                                                          PyIPSDK.eIBT_Label16, PyIPSDK.eIBT_Label32]:
                                min_intensity = float(vrb.mainWindow.groupMenu.groupContrast.lineEditMin.text())
                                max_intensity = float(vrb.mainWindow.groupMenu.groupContrast.lineEditMax.text())
                                # if currentLayer.contrast_limits[0] != min_intensity or currentLayer.contrast_limits[1] != max_intensity or min_intensity == max_intensity:
                                if currentLayer.contrast_limits[0] != min_intensity \
                                        or currentLayer.contrast_limits[1] != max_intensity \
                                        or min_intensity < max_intensity:
                                    # or min_intensity != max_intensity \
                                    currentLayer.contrast_limits = [currentLabel.valueMin, currentLabel.valueMax]

                                if currentLayer.iso_threshold != float(
                                        self.imageSettings.sliderIsoThreshold.slider.value()):
                                    currentLayer.iso_threshold = float(self.imageSettings.sliderIsoThreshold.slider.value())

                        else:
                            if currentLayer.rendering != self.imageSettings.comboBoxLabelsRendering.currentText():
                                currentLayer.rendering = self.imageSettings.comboBoxLabelsRendering.currentText()

                        if currentLabel.image.getSizeC() != 3 and currentLayer.colormap.name != currentLabel.groupBoxLut.comboBoxLut.currentText():
                            self.updateColorMap(label=currentLabel)

                    if currentLabel.objectType == "Mesh":
                        if currentLayer.shading != self.imageSettings.comboBoxShading.currentText():
                            currentLayer.shading = self.imageSettings.comboBoxShading.currentText()


                        if currentLayer.wireframe.visible != self.imageSettings.checkBoxWireframe.isChecked():
                            currentLayer.wireframe.visible = self.imageSettings.checkBoxWireframe.isChecked()
                        currentLayer.wireframe.color = (self.imageSettings.buttonColorWireframe.color[0] / 255,
                                                             self.imageSettings.buttonColorWireframe.color[1] / 255,
                                                             self.imageSettings.buttonColorWireframe.color[2] / 255,
                                                             1.)

                if currentLabel.objectType == "Image":
                    #dont need "hasNapari" verification
                    if currentLabel.orthosliceX is not None and currentLabel.orthosliceY is not None and currentLabel.orthosliceZ is not None:
                        if currentLabel.orthosliceX.colormap.name != self.orthoslicesSettings.comboBoxColormap.currentText():
                            # idx = self.orthoslicesSettings.comboBoxColormap.findText(currentLabel.orthosliceX.colormap.name)
                            # if idx == -1:
                            #     idx = 1
                            # self.orthoslicesSettings.comboBoxColormap.setCurrentIndex(idx)
                            self.updateOrthosliceColorMap() #updates the orthoslices colormap
                        if currentLabel.orthosliceX.blending != self.orthoslicesSettings.comboBoxBlending.currentText():
                            currentLabel.orthosliceX.blending = self.orthoslicesSettings.comboBoxBlending.currentText()
                            currentLabel.orthosliceY.blending = self.orthoslicesSettings.comboBoxBlending.currentText()
                            currentLabel.orthosliceZ.blending = self.orthoslicesSettings.comboBoxBlending.currentText()

                            self.reorderLayers()

                        if currentLabel.image.getBufferType() not in [PyIPSDK.eIBT_Label8, PyIPSDK.eIBT_Label16, PyIPSDK.eIBT_Label32]:
                            if currentLabel.orthosliceX.rendering != self.orthoslicesSettings.comboBoxRendering.currentText():
                                currentLabel.orthosliceX.rendering = self.orthoslicesSettings.comboBoxRendering.currentText()
                                currentLabel.orthosliceY.rendering = self.orthoslicesSettings.comboBoxRendering.currentText()
                                currentLabel.orthosliceZ.rendering = self.orthoslicesSettings.comboBoxRendering.currentText()

                            if currentLabel.orthosliceX.attenuation != self.orthoslicesSettings.sliderAttenuation.slider.value():
                                currentLabel.orthosliceX.attenuation = self.orthoslicesSettings.sliderAttenuation.slider.value() / 100
                                currentLabel.orthosliceY.attenuation = self.orthoslicesSettings.sliderAttenuation.slider.value() / 100
                                currentLabel.orthosliceZ.attenuation = self.orthoslicesSettings.sliderAttenuation.slider.value() / 100
                        else:
                            if currentLabel.orthosliceX.rendering != self.orthoslicesSettings.comboBoxLabelsRendering.currentText():
                                currentLabel.orthosliceX.rendering = self.orthoslicesSettings.comboBoxLabelsRendering.currentText()
                                currentLabel.orthosliceY.rendering = self.orthoslicesSettings.comboBoxLabelsRendering.currentText()
                                currentLabel.orthosliceZ.rendering = self.orthoslicesSettings.comboBoxLabelsRendering.currentText()

                        if currentLabel.orthosliceX.plane.thickness != self.orthoslicesSettings.sliderThickness.slider.value():
                            # currentLabel.orthosliceX.plane.thickness = self.orthoslicesSettings.sliderThickness.slider.value()/100
                            # currentLabel.orthosliceY.plane.thickness = self.orthoslicesSettings.sliderThickness.slider.value()/100
                            # currentLabel.orthosliceZ.plane.thickness = self.orthoslicesSettings.sliderThickness.slider.value()/100
                            currentLabel.orthosliceX.plane.thickness = self.orthoslicesSettings.sliderThickness.slider.value()
                            currentLabel.orthosliceY.plane.thickness = self.orthoslicesSettings.sliderThickness.slider.value()
                            currentLabel.orthosliceZ.plane.thickness = self.orthoslicesSettings.sliderThickness.slider.value()


                        vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.blockSignals(True)
                        vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.blockSignals(True)
                        vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.blockSignals(True)

                        if currentLabel.orthosliceX.plane.position[0] != vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.value():
                            vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.setValue(
                                currentLabel.orthosliceX.plane.position[0])
                        if currentLabel.orthosliceY.plane.position[0] != vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.value():
                            vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.setValue(
                                currentLabel.orthosliceY.plane.position[0])
                        if currentLabel.orthosliceZ.plane.position[0] != vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.value():
                            vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.setValue(
                                currentLabel.orthosliceZ.plane.position[0])

                        vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.blockSignals(False)
                        vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.blockSignals(False)
                        vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.blockSignals(False)

        except:
            traceback.print_exc(file=sys.stderr)



    def updateOrthosliceColorMap(self):
        ''' Updates the colormap of orthoslices layers when
            choosing a colormap in the colormap combobox.

            Changes are done by converting the LUT selected defined
            in Explorer into a napari.utils.Colormap.
        '''
        currentLabel = vrb.mainWindow.currentLabel
        if currentLabel is not None:
            try:
                orthosliceX = currentLabel.orthosliceX
                orthosliceY = currentLabel.orthosliceY
                orthosliceZ = currentLabel.orthosliceZ

                if orthosliceX is not None and currentLabel.image.getBufferType() not in [PyIPSDK.eIBT_Label8,
                                                                              PyIPSDK.eIBT_Label16, PyIPSDK.eIBT_Label32]:
                    # Get the current LUT
                    lut = self.orthoslicesSettings.comboBoxColormap.currentData()
                    colormap = None
                    if lut is not None:
                        name = lut['Name']
                        if currentLabel.image.getBufferType() != PyIPSDK.eIBT_Binary:
                            array = lut["Array"]
                            image = fct.lutImageFromArray(array)
                            currentLut = lutFromVector(image)
                            lutTabColor = []
                            for i in range(len(currentLut[0])):
                                lutTabColor.append([currentLut[0][i]/255, currentLut[1][i]/255, currentLut[2][i]/255, 1.0])

                            colormap = napari.utils.Colormap(colors=lutTabColor, name=name)

                        # create blue and grey lut for binary images
                        else:
                            if self.orthoslicesSettings.comboBoxColormap.currentText() == "Binary":
                                colormap = napari.utils.Colormap([[0,0,0,1],[0,0,1,1]], name="Binary")
                            elif self.orthoslicesSettings.comboBoxColormap.currentText() == "BinaryGrey":
                                colormap = napari.utils.Colormap([[0,0,0,1],[1,1,1,1]], name="BinaryGrey")

                    vrb.mainWindow.currentLabel.orthosliceX.colormap = colormap
                    vrb.mainWindow.currentLabel.orthosliceY.colormap = colormap
                    vrb.mainWindow.currentLabel.orthosliceZ.colormap = colormap

                else:
                    max_val = vrb.mainWindow.currentLabel.valueMaxRef
                    lut = vrb.mainWindow.viewer3dSettingsWidget.orthoslicesSettings.comboBoxColormap.currentData()

                    if lut is not None:
                        array = lut["Array"]
                        name = lut["Name"]
                        image = fct.lutImageFromArray(array)

                        if name != "Random" and currentLabel.orthosliceX.colormap.name != 'label_colormap':
                            currentLut = lutFromVector(image)
                            labels = np.arange(1, max_val + 1)

                            # Interpolation pour chaque canal (R, G, B)
                            r = np.interp(labels, np.linspace(1, max_val, num=256), currentLut[0])
                            g = np.interp(labels, np.linspace(1, max_val, num=256), currentLut[1])
                            b = np.interp(labels, np.linspace(1, max_val, num=256), currentLut[2])

                            # Création du dictionnaire de couleurs
                            color_dict = {label: (r[i] / 255, g[i] / 255, b[i] / 255, 1.0) for i, label in enumerate(labels)}

                        else:
                            currentLut = vrb.mainWindow.currentLabel.groupBoxLut.currentLut

                            if napari.__version__.startswith("0.4"):
                                color_dict = {}
                                for i in range(1, len(currentLut[0])):

                                    # Normaliser les couleurs et créer une couleur RGBA
                                    color = np.asarray([
                                        currentLut[0][i] / 255,  # R
                                        currentLut[1][i] / 255,  # G
                                        currentLut[2][i] / 255,  # B
                                        1.0
                                    ])
                                    color_dict[int(i)] = color  # Associer une valeur de label à une couleur

                            elif napari.__version__.startswith('0.5'):
                                colorArray = []
                                nbLabels = min(len(currentLut[0]), 500)
                                for i in range(0, nbLabels):
                                    # Normaliser les couleurs et créer une couleur RGBA
                                    color = np.array([
                                        float(currentLut[0][i] / 255),  # R
                                        float(currentLut[1][i] / 255),  # G
                                        float(currentLut[2][i] / 255),  # B
                                        float(1.0)
                                    ])
                                    colorArray.append(color)  # Associer une valeur de label à une couleur

                                # utilisation d'une colormap cyclique pour éviter des problèmes de lags
                                color_dict = napari.utils.CyclicLabelColormap(name=lut["Name"], background_value=0,
                                                                            colors=colorArray)


                    #apply colormap
                    if napari.__version__.startswith("0.4"):
                        color_dict[0] = 'black'
                        orthosliceX.color = color_dict
                        orthosliceY.color = color_dict
                        orthosliceZ.color = color_dict
                    elif napari.__version__.startswith("0.5"):
                        if orthosliceX.colormap != color_dict:
                            orthosliceX.colormap = color_dict
                            orthosliceY.colormap = color_dict
                            orthosliceZ.colormap = color_dict
                    else:
                        print("ERROR : Explorer needs napari 0.4.* or 0.5.* to work, check your version of napari.")
                        return 1
            except:
                print("fail updating colormap")
                traceback.print_exc(file=sys.stderr)


    def updateColorMap(self, label=None):
        ''' Updates the colormap of orthoslices layers when
            choosing a colormap in the colormap combobox.

            Changes are done by converting the LUT selected defined
            in Explorer into a napari.utils.Colormap.
        '''
        try:
            if label is None:
                currentLabel = vrb.mainWindow.currentLabel
            else:
                currentLabel = label
            # currentLabel = vrb.mainWindow.currentLabel
            # if currentLabel is not None and currentLabel.objectType == "Image" and currentLabel.image.getBufferType() not in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32]:
            if currentLabel.image.getSizeC() != 3:
                if currentLabel is not None and currentLabel.image.getBufferType() not in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32]:
                    currentLayer = currentLabel.layer
                    min_val = currentLabel.valueMinRef
                    max_val = currentLabel.valueMaxRef

                    # Get the current LUT
                    if currentLabel.image.getBufferType() != PyIPSDK.eIBT_Binary:
                        currentLut = currentLabel.groupBoxLut.currentLut

                        lutTabColor = []
                        for i in range(len(currentLut[0])):
                            # for i in range(255):

                            value = min_val + (max_val - min_val) * i / 255

                            if value <= min_val or value > max_val:
                                opacity = 0
                            else:
                                opacity = 1

                            lutTabColor.append(
                                [currentLut[0][i] / 255, currentLut[1][i] / 255, currentLut[2][i] / 255, opacity])

                        colormap = napari.utils.Colormap(lutTabColor, name=currentLabel.groupBoxLut.comboBoxLut.currentText())


                    # create blue and grey lut for binary images
                    else:
                        # currentLut = [[], [], []]
                        if currentLabel.groupBoxLut.comboBoxLut.currentText() == "Binary":
                            # for i in range(255):
                            #     currentLut[0].append(0)
                            #     currentLut[1].append(0)
                            #     currentLut[2].append(i)
                            colormap = napari.utils.Colormap([[0,0,0,1],[0,0,1,1]], name="Binary")
                            # currentLut = [[0,0],[0,0],[0,1]]
                        elif currentLabel.groupBoxLut.comboBoxLut.currentText() == "Binary Grey":
                            # for i in range(255):
                            #     currentLut[0].append(i)
                            #     currentLut[1].append(i)
                            #     currentLut[2].append(i)
                            # currentLut = [[0,1], [0,1], [0,1]]
                            colormap = napari.utils.Colormap([[0, 0, 0, 1], [1, 1, 1, 1]], name="BinaryGrey")

                    # if fct.compareColorMapNapari(currentLayer.colormap, colormap) == False:
                    #     currentLayer.colormap = colormap

                else: # for Label layers
                    currentLayer = currentLabel.layer
                    max_val = currentLabel.valueMaxRef

                    if currentLabel.image.getBufferType() != PyIPSDK.eIBT_Binary:
                        currentLut = currentLabel.groupBoxLut.currentLut

                    lut = currentLabel.groupBoxLut.comboBoxLut.currentData()
                    if lut is not None and lut["Name"] != "Random":

                        array = lut["Array"]
                        image = fct.lutImageFromArray(array)
                        currentLut = lutFromVector(image)

                        labels = np.arange(1, max_val + 1)

                        # Interpolation pour chaque canal (R, G, B)
                        r = np.interp(labels, np.linspace(1, max_val, num=256), currentLut[0])
                        g = np.interp(labels, np.linspace(1, max_val, num=256), currentLut[1])
                        b = np.interp(labels, np.linspace(1, max_val, num=256), currentLut[2])

                        # Création du dictionnaire de couleurs
                        colormap = {label: (r[i] / 255, g[i] / 255, b[i] / 255, 1.0) for i, label in enumerate(labels)}

                    else:
                        start = time.time()

                        lut = currentLabel.groupBoxLut.comboBoxLut.currentData()
                        currentLut = currentLabel.groupBoxLut.currentLut

                        # pour assurer le fonctionnement dans napari 0.4 et 0.5
                        if napari.__version__.startswith('0.4'):

                            colormap = {}
                            # création d'un dictionnaire de couleur cyclique pour la colormap des labels randoms
                            for i in range(1,len(currentLut[0])):
                                if i == 500:
                                    # Récupérer la première couleur de manière cyclique
                                    color = np.array([
                                        currentLut[0][0] / 255,  # R
                                        currentLut[1][0] / 255,  # G
                                        currentLut[2][0] / 255,  # B
                                        1.0
                                    ])
                                else:
                                    # Normaliser les couleurs et créer une couleur RGBA
                                    color = np.array([
                                        currentLut[0][i % len(currentLut[0])] / 255,  # R
                                        currentLut[1][i % len(currentLut[1])] / 255,  # G
                                        currentLut[2][i % len(currentLut[2])] / 255,  # B
                                        1.0
                                    ])

                                colormap[int(i)] = color  # Associer une valeur de label à une couleur


                        elif napari.__version__.startswith('0.5'):
                            colorArray = []
                            nbLabels = len(currentLut[0])
                            if nbLabels > 500:
                                nbLabels = 500
                            for i in range(0, nbLabels):
                                # Normaliser les couleurs et créer une couleur RGBA
                                color = np.array([
                                    currentLut[0][i] / 255,  # R
                                    currentLut[1][i] / 255,  # G
                                    currentLut[2][i] / 255,  # B
                                    1.0
                                ])
                                colorArray.append(color)  # Associer une valeur de label à une couleur

                            # utilisation d'une colormap cyclique pour éviter des problèmes de lags
                            colormap = napari.utils.CyclicLabelColormap(name=lut["Name"], background_value=0,colors=colorArray)


                # pour assurer le fonctionnement dans napari 0.4 et 0.5
                if napari.__version__.startswith('0.4'):
                    if currentLayer.color != colormap:
                        currentLayer.color = colormap
                elif napari.__version__.startswith('0.5'):
                    if currentLayer.colormap != colormap:
                        currentLayer.colormap = colormap
        except:
            traceback.print_exc(file=sys.stderr)


#########################################
#           UPDATE FUNCTIONS            #
#########################################

    def opacityUpdate(opacity):
        ''' Updates the opacity of a layer when moving
            the opacity slider.
        '''
        if vrb.mainWindow.currentLabel is not None:
            currentLayer = vrb.mainWindow.currentLabel.layer
            if currentLayer is not None:
                if currentLayer.opacity != opacity:
                    currentLayer.opacity = opacity
                    currentLayer.blending = currentLayer.blending

    @thread_worker(connect={'yielded': opacityUpdate})
    def sliderOpacityChanged(self, opacity):
        ''' Connects to opacity update function using
            a thread worker.

            Yields the slider opacity value.
        '''
        if vrb.mainWindow.currentLabel is not None and self.loadingSettings == False:
            yield self.sliderOpacity.slider.value()/100

    def attenuationUpdate(attenuation):
        ''' Updates the attenuation of a layer when moving
            the attenuation slider.
        '''
        if vrb.mainWindow.currentLabel is not None and vrb.mainWindow.currentLabel.objectType == "Image":
            currentLayer = vrb.mainWindow.currentLabel.layer
            if currentLayer is not None:
                if currentLayer.attenuation != attenuation:
                    currentLayer.attenuation = attenuation

    @thread_worker(connect={'yielded': attenuationUpdate})
    def sliderAttenuationChanged(self, attenuation):
        ''' Connects to attenuation update function using
            a thread worker.

            Yields the slider attenuation value.
        '''
        if vrb.mainWindow.currentLabel is not None and self.loadingSettings == False:
            yield self.imageSettings.sliderAttenuation.slider.value()/100

    def gammaUpdate(gamma):
        ''' Updates the opacity of a layer when moving
            the opacity slider.
        '''
        if vrb.mainWindow.currentLabel is not None and vrb.mainWindow.currentLabel.objectType == "Image":
            currentLayer = vrb.mainWindow.currentLabel.layer
            if currentLayer is not None:
                if currentLayer.gamma != gamma:
                    currentLayer.gamma = gamma

    @thread_worker(connect={'yielded': gammaUpdate})
    def sliderGammaChanged(self, gamma):
        ''' Connects to gamma update function using a
            thread worker.

            Yields the slider gamma value.
        '''
        if vrb.mainWindow.currentLabel is not None and self.loadingSettings == False:
            yield self.imageSettings.sliderGamma.slider.value()/10

    def isoThresholdUpdate(isoThreshold):
        ''' Updates the iso threshold of a layer when moving
            the iso threshold slider.
        '''
        if vrb.mainWindow.currentLabel is not None and vrb.mainWindow.currentLabel.objectType == "Image":
            currentLayer = vrb.mainWindow.currentLabel.layer
            if currentLayer is not None and not isinstance(currentLayer, napari.layers.Labels):
                if currentLayer.iso_threshold != isoThreshold:
                    currentLayer.iso_threshold = isoThreshold

    @thread_worker(connect={'yielded': isoThresholdUpdate})
    def sliderIsoThresholdChanged(self, isoThreshold):
        ''' Connects to iso threshold update function
            using a thread worker.

            Yields the slider of iso threshold value.
        '''
        if vrb.mainWindow.currentLabel is not None and self.loadingSettings == False:
            yield self.imageSettings.sliderIsoThreshold.slider.value()

    def intensitiesUpdate(intensities):
        ''' Updates the opacity of a layer when moving
            the opacity slider.
        '''
        if vrb.mainWindow.currentLabel is not None and vrb.mainWindow.currentLabel.objectType == "Image":
            currentLayer = vrb.mainWindow.currentLabel.layer
            if currentLayer is not None:
                if currentLayer.contrast_limits[0] != intensities[0] or currentLayer.contrast_limits[1] != intensities[1]:
                    currentLayer.contrast_limits = [intensities[0], intensities[1]]

                    if vrb.mainWindow.currentLabel.orthosliceX is not None:
                        vrb.mainWindow.currentLabel.orthosliceX.contrast_limits = [intensities[0], intensities[1]]
                        vrb.mainWindow.currentLabel.orthosliceY.contrast_limits = [intensities[0], intensities[1]]
                        vrb.mainWindow.currentLabel.orthosliceZ.contrast_limits = [intensities[0], intensities[1]]

    @thread_worker(connect={'yielded': intensitiesUpdate})
    def sliderIntensitiesChanged(self, intensities):
        ''' Connects to intensities update function
            using a thread worker.

            Yields the slider intensities values.
        '''
        if vrb.mainWindow.currentLabel is not None and vrb.mainWindow.currentLabel.objectType == "Image" and self.loadingSettings == False:
            if vrb.mainWindow.currentLabel.image.getBufferType() not in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32, PyIPSDK.eIBT_Binary]:
                min_intensity = int(float(vrb.mainWindow.groupMenu.groupContrast.lineEditMin.text()))
                max_intensity = int(float(vrb.mainWindow.groupMenu.groupContrast.lineEditMax.text()))
                if min_intensity != max_intensity:
                    intensities = []
                    intensities.append(min_intensity)
                    intensities.append(max_intensity)
                    yield intensities

    def sliderFovChanged(self):
        ''' Updates the FOV when using the FOV
            slider in viewer settings.

            FOV=Field of View (default=60)
        '''
        if self.loadingCamera == False:

            try:
                if vrb.mainWindow.viewerNapari.camera.perspective != self.viewerSettings.sliderFov.slider.value():
                    vrb.mainWindow.viewerNapari.camera.perspective = self.viewerSettings.sliderFov.slider.value()
                    print("fov = ",vrb.mainWindow.viewerNapari.camera.zoom )
                    pass
            except:
                traceback.print_exc(file=sys.stderr)

    def cameraCenterChanged(self):
        ''' Allows to change the camera center by
            entering X,Y and Z of the camera center
            position.
        '''
        if self.loadingCamera == False:

            try:
                if self.viewerSettings.labelWithLineEditX.lineEdit.text() != "":
                    centerX = float(self.viewerSettings.labelWithLineEditX.lineEdit.text())
                else:
                    centerX = float(self.viewerSettings.labelWithLineEditX.lineEdit.placeholderText())
                if self.viewerSettings.labelWithLineEditY.lineEdit.text() != "":
                    centerY = float(self.viewerSettings.labelWithLineEditY.lineEdit.text())
                else:
                    centerY = float(self.viewerSettings.labelWithLineEditY.lineEdit.placeholderText())
                if self.viewerSettings.labelWithLineEditZ.lineEdit.text() != "":
                    centerZ = float(self.viewerSettings.labelWithLineEditZ.lineEdit.text())
                else:
                    centerZ = float(self.viewerSettings.labelWithLineEditZ.lineEdit.placeholderText())

                if vrb.mainWindow.viewerNapari.camera.center != [centerX,centerY,centerZ]:
                    vrb.mainWindow.viewerNapari.camera.center = [centerX,centerY,centerZ]
            except:
                traceback.print_exc(file=sys.stderr)

    def cameraAnglesChanged(self):
        ''' Allows to change the camera angle by
            entering ro,theta and phi angles of
            the desired angle.

            observation: Mainly used inside the moviemaker.
        '''
        if self.loadingCamera == False:

            try:
                if self.viewerSettings.labelWithLineEditAlpha.lineEdit.text() != "":
                    angleAlpha = float(self.viewerSettings.labelWithLineEditAlpha.lineEdit.text())
                else:
                    angleAlpha = float(self.viewerSettings.labelWithLineEditAlpha.lineEdit.placeholderText())
                if self.viewerSettings.labelWithLineEditBeta.lineEdit.text() != "":
                    angleBeta = float(self.viewerSettings.labelWithLineEditBeta.lineEdit.text())
                else:
                    angleBeta = float(self.viewerSettings.labelWithLineEditBeta.lineEdit.placeholderText())
                if self.viewerSettings.labelWithLineEditGamma.lineEdit.text() != "":
                    angleGamma = float(self.viewerSettings.labelWithLineEditGamma.lineEdit.text())
                else:
                    angleGamma = float(self.viewerSettings.labelWithLineEditGamma.lineEdit.placeholderText())

                if vrb.mainWindow.viewerNapari.camera.angles != (angleAlpha,angleBeta,angleGamma):
                    vrb.mainWindow.viewerNapari.camera.angles = (angleAlpha,angleBeta,angleGamma)
            except:
                traceback.print_exc(file=sys.stderr)

    def cameraZoomChanged(self):
        ''' Allows to change the camera zoom by
            entering a zoom percentage of the zoom.
        '''
        if self.loadingCamera == False:

            try:
                if self.viewerSettings.lineEditZoom.text() != "":
                    vrb.mainWindow.viewerNapari.camera.zoom = float(self.viewerSettings.lineEditZoom.text())
                else:
                    vrb.mainWindow.viewerNapari.camera.zoom = float(self.viewerSettings.lineEditZoom.placeholderText())
            except:
                traceback.print_exc(file=sys.stderr)

        # creates the 3 orthoslices if not created, and updates the layers if the orthoslices already exists


    def attenuationOrthoslicesUpdate(attenuation):
        ''' Updates the opacity of a layer when moving
            the opacity slider.
        '''
        if vrb.mainWindow.currentLabel.orthosliceX is not None and vrb.mainWindow.currentLabel.objectType == "Image":
            curLabel = vrb.mainWindow.currentLabel
            if curLabel.orthosliceX is not None:
                if curLabel.orthosliceX.attenuation != attenuation:
                    curLabel.orthosliceX.attenuation = attenuation
                if curLabel.orthosliceY.attenuation != attenuation:
                    curLabel.orthosliceY.attenuation = attenuation
                if curLabel.orthosliceZ.attenuation != attenuation:
                    curLabel.orthosliceZ.attenuation = attenuation

    @thread_worker(connect={'yielded': attenuationOrthoslicesUpdate})
    def sliderAttenuationOrthoslicesChanged(self, attenuation):
        ''' Connects to orthoslice attenuation update function
            using a thread worker.

            Yields the slider of orthoslice's attenuation value.
        '''
        if vrb.mainWindow.currentLabel.orthosliceX is not None:
            yield self.orthoslicesSettings.sliderAttenuation.slider.value()/100

    #updates orthoslices positions
    def updateOrthoslices(position):
        ''' Updates the orthoslices positions when moving
            the axis sliders.
        '''
        currentLabel = vrb.mainWindow.currentLabel
        if currentLabel is not None:
            # sets plane position
            if currentLabel.orthosliceX.plane.position[2]!=position[0]:
                currentLabel.orthosliceX.plane.position = (position[0], position[0], position[0])
            if currentLabel.orthosliceY.plane.position[1]!=position[1]:
                currentLabel.orthosliceY.plane.position = (position[1], position[1], position[1])
            if currentLabel.orthosliceZ.plane.position[0]!=position[2]:
                currentLabel.orthosliceZ.plane.position = (position[2], position[2], position[2])

            # currentLabel.orthosliceX.reset_contrast_limits()
            # currentLabel.orthosliceY.reset_contrast_limits()
            # currentLabel.orthosliceZ.reset_contrast_limits()


    def planeXPositionUpdate(position):
        ''' Updates the plane position on X axis when moving
            the x axis slider.

            Moving the x plane hides any other visible plane.
        '''
        if vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.activate:
            vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.setActivation(False)
        if not vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.activate or \
                vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.activate:
            vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.setActivation(True)
            vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.setActivation(True)

        # if not vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible:
        if not vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.activate:
            vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible = True
            vrb.mainWindow.viewer3dSettingsWidget.xPlane.opacity = 0.3
        if vrb.mainWindow.viewer3dSettingsWidget.yPlane.visible:
            vrb.mainWindow.viewer3dSettingsWidget.yPlane.visible = False
        if vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible:
            vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible = False

        vrb.mainWindow.viewer3dSettingsWidget.xPlane.translate = [0, 0, position]

    ### For Plane X (not orthoslice)
    @thread_worker(connect={'yielded': planeXPositionUpdate})
    def planeXChanged(self, pos):
        yield vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.value() * self.xPlane.scale[2]


    def planeYPositionUpdate(position):
        ''' Updates the plane position on Y axis when moving
            the y axis slider.

            Moving the y plane hides any other visible plane.
        '''
        if vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.activate:
            vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.setActivation(False)
        if not vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.activate or \
                vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.activate:
            vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.setActivation(True)
            vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.setActivation(True)

        # if not vrb.mainWindow.viewer3dSettingsWidget.yPlane.visible:
        vrb.mainWindow.viewer3dSettingsWidget.yPlane.visible = True
        vrb.mainWindow.viewer3dSettingsWidget.yPlane.opacity = 0.3
        if vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible:
            vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible = False
        if vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible:
            vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible = False

        vrb.mainWindow.viewer3dSettingsWidget.yPlane.translate = [0, position, 0]

    ### For Plane Y (not orthoslice)
    @thread_worker(connect={'yielded': planeYPositionUpdate})
    def planeYChanged(self, pos):
        yield vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.value() * self.xPlane.scale[1]


    def planeZPositionUpdate(position):
        ''' Updates the plane position on Y axis when moving
            the y axis slider.

            Moving the y plane hides any other visible plane.
        '''
        if vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.activate:
            vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.setActivation(False)
        if not vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.activate or \
                vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.activate:
            vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.setActivation(True)
            vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.setActivation(True)

        # if not vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible:
        vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible = True
        vrb.mainWindow.viewer3dSettingsWidget.zPlane.opacity = 0.3
        if vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible:
            vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible = False
        if vrb.mainWindow.viewer3dSettingsWidget.yPlane.visible:
            vrb.mainWindow.viewer3dSettingsWidget.yPlane.visible = False

        vrb.mainWindow.viewer3dSettingsWidget.zPlane.translate = [position, 0, 0]

    ### For Plane Z (not orthoslice)
    @thread_worker(connect={'yielded': planeZPositionUpdate})
    def planeZChanged(self, pos):
        yield vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.value() * self.xPlane.scale[0]


    @thread_worker(connect={'yielded': updateOrthoslices})
    def orthoslicesChanged(self, pos):
        if vrb.mainWindow.widgetImage.modeViewer == '4Views':
            sliderX = vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX
            sliderY = vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY
            sliderZ = vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ
        # elif mode=='3D':
        else:
            sliderX = self.orthoslicesSettings.sliderX
            sliderY = self.orthoslicesSettings.sliderY
            sliderZ = self.orthoslicesSettings.sliderZ

        yield tuple((sliderX.slider.value(), sliderY.slider.value(), sliderZ.slider.value()))


    # @thread_worker(connect={'yielded': updateOrthoslices})
    # def orthoslicesChanged(self, pos):
    #     currentLabel = vrb.mainWindow.currentLabel
    #     if vrb.mainWindow.widgetImage.modeViewer == '4Views':
    #         sliderX = vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX
    #         sliderY = vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY
    #         sliderZ = vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ
    #     # elif mode=='3D':
    #     else:
    #         sliderX = self.orthoslicesSettings.sliderX
    #         sliderY = self.orthoslicesSettings.sliderY
    #         sliderZ = self.orthoslicesSettings.sliderZ
    #
    #     yield tuple((sliderX.slider.value(), sliderY.slider.value(), sliderZ.slider.value()))


    # def sliderPlanesPositionChanged(self):
    #     ''' Triggers every time the x slider (in 4 views),
    #         or the x slider in orthoslices parameter, is moved.
    #
    #         If orthoslices are activated it connects to the
    #         orthoslices update function, if not it connects
    #         to plane X position update function.
    #     '''
    #     if not vrb.mainWindow.currentLabel.buttonOrthoslices.activate:
    #         self.orthoslicesChanged((0,0,0))
    #     else:
    #         if vrb.mainWindow.widgetImage.modeViewer == "4Views":#check if in 4views mode
    #             self.planesChanged((0,0,0))

    def sliderPlaneXPositionChanged(self):
        ''' Triggers every time the x slider (in 4 views),
            or the x slider in orthoslices parameter, is moved.

            If orthoslices are activated it connects to the
            orthoslices update function, if not it connects
            to plane X position update function.
        '''
        if vrb.mainWindow.currentLabel is not None:
            if not vrb.mainWindow.currentLabel.buttonOrthoslices.activate:
                self.orthoslicesChanged((0,0,0))
            else:
                if vrb.mainWindow.widgetImage.modeViewer == "4Views":#check if in 4views mode
                    if self.sender()==vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider:
                        self.planeXChanged((0,0,0))
                    if self.sender()==vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider:
                        self.planeYChanged((0,0,0))
                    if self.sender()==vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider:
                        self.planeZChanged((0,0,0))

    def planeYPositionUpdate(position):
        ''' Updates the plane position on Y axis when moving
            the y axis slider.

            Moving the y plane hides any other visible plane.
        '''
        if vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.activate:
            vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.setActivation(False)
        vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.setActivation(True)
        vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.setActivation(True)
        vrb.mainWindow.viewer3dSettingsWidget.yPlane.translate = [0, position, 0]
        vrb.mainWindow.viewer3dSettingsWidget.yPlane.visible = True
        vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible = False
        vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible = False
        vrb.mainWindow.viewer3dSettingsWidget.yPlane.opacity = 0.3


    @thread_worker(connect={'yielded': planeYPositionUpdate})
    def sliderPlaneYPositionChanged(self, position):
        if vrb.mainWindow.currentLabel is not None:
            if not vrb.mainWindow.currentLabel.buttonOrthoslices.activate:
                self.updateOrthoslices(mode=vrb.mainWindow.widgetImage.modeViewer)
            else:
                if vrb.mainWindow.widgetImage.modeViewer == "4Views":
                    yield vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.value()*self.xPlane.scale[1]

    def planeZPositionUpdate(position):
        ''' Updates the plane position on Z axis when moving
            the z axis slider.

            Moving the z plane hides any other visible plane.
        '''
        if vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.activate:
            vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.setActivation(False)
            vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.setActivation(True)
            vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.setActivation(True)
        if not vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible:
            vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible = True
            vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible = False
            vrb.mainWindow.viewer3dSettingsWidget.yPlane.visible = False

        # vrb.mainWindow.viewer3dSettingsWidget.zPlane.opacity = 0.3
        vrb.mainWindow.viewer3dSettingsWidget.zPlane.translate = [position, 0, 0]


    @thread_worker(connect={'yielded': planeZPositionUpdate})
    def sliderPlaneZPositionChanged(self, position):
        if vrb.mainWindow.currentLabel is not None:
            if not vrb.mainWindow.currentLabel.buttonOrthoslices.activate:
                self.updateOrthoslices(mode=vrb.mainWindow.widgetImage.modeViewer)
            else:
                if vrb.mainWindow.widgetImage.modeViewer == "4Views":
                    yield vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.value()*self.xPlane.scale[0]


    def orthosliceOpacityUpdate(opacity):
        ''' Updates the orthoslices opacity when moving
            the orthoslices opacity slider.
        '''
        if vrb.mainWindow.currentLabel is not None:
            if vrb.mainWindow.currentLabel.orthosliceX.opacity != opacity:
                vrb.mainWindow.currentLabel.orthosliceX.opacity = opacity
                vrb.mainWindow.currentLabel.orthosliceY.opacity = opacity
                vrb.mainWindow.currentLabel.orthosliceZ.opacity = opacity

    @thread_worker(connect={'yielded': orthosliceOpacityUpdate})
    def sliderOrthosliceOpacityChanged(self, opacity):
        if vrb.mainWindow.currentLabel is not None:
            if not vrb.mainWindow.currentLabel.buttonOrthoslices.activate:
                yield self.orthoslicesSettings.sliderOpacity.slider.value()/100

    def orthosliceThicknessUpdate(thickness):
        ''' Updates the orthoslices thickness parameter
            when moving the thickness slider.
        '''
        if vrb.mainWindow.currentLabel is not None:
            orthosliceX = vrb.mainWindow.currentLabel.orthosliceX
            orthosliceY = vrb.mainWindow.currentLabel.orthosliceY
            orthosliceZ = vrb.mainWindow.currentLabel.orthosliceZ

            if orthosliceX.plane.thickness != thickness:
                orthosliceX.plane.thickness = thickness
                orthosliceY.plane.thickness = thickness
                orthosliceZ.plane.thickness = thickness


    @thread_worker(connect={'yielded': orthosliceThicknessUpdate})
    def sliderOrthosliceThicknessChanged(self, thickness):
        if vrb.mainWindow.currentLabel is not None:
            if not vrb.mainWindow.currentLabel.buttonOrthoslices.activate:
                # yield self.orthoslicesSettings.sliderThickness.slider.value()/100
                yield self.orthoslicesSettings.sliderThickness.slider.value()


    def changeDisplayAxis(self):

        try:
            currentLabel = vrb.mainWindow.currentLabel
            if currentLabel is not None:

                # sizeX = currentLabel.image.getSizeX()
                # sizeY = currentLabel.image.getSizeY()
                # sizeZ = currentLabel.image.getSizeZ()
                # maxsize = max(sizeX, sizeY, sizeZ) / 2

                if currentLabel.objectType == "Image":
                    sizeX = currentLabel.image.getSizeX()
                    sizeY = currentLabel.image.getSizeY()
                    sizeZ = currentLabel.image.getSizeZ()
                elif currentLabel.objectType == "Mesh":
                    sizeX = round(currentLabel.endValues[0]) - int(currentLabel.startValues[0]) + 1
                    sizeY = round(currentLabel.endValues[1]) - int(currentLabel.startValues[1]) + 1
                    sizeZ = round(currentLabel.endValues[2]) - int(currentLabel.startValues[2]) + 1

                maxsize = max(sizeX, sizeY, sizeZ) / 2

                transform = vispy.visuals.transforms.MatrixTransform()
                # transform.scale((maxsize, maxsize, maxsize))
                transform.scale((maxsize, maxsize, maxsize))
                transform.translate((-(maxsize * (5 / 100)), -(maxsize * (5 / 100)), -(maxsize * (5 / 100))))
                self.axis.transform = transform

                # we get the number of layers in the viewer to give the right order to the axis display
                # so that it's not disappearing in a volume
                orderNum = len(vrb.mainWindow.viewerNapari.layers)

                self.xLabel.order = orderNum + 1
                self.xLabel.font_size = 10 * vrb.ratio
                self.xLabel.pos = (maxsize + 2, -(maxsize * (5 / 100)), -(maxsize * (5 / 100)))

                self.yLabel.order = orderNum + 2
                self.yLabel.font_size = 10 * vrb.ratio
                self.yLabel.pos = (-(maxsize * (5 / 100)), maxsize + 2, -(maxsize * (5 / 100)))

                self.zLabel.order = orderNum + 3
                self.zLabel.font_size = 10 * vrb.ratio
                self.zLabel.pos = (-(maxsize * (5 / 100)), -(maxsize * (5 / 100)), maxsize + 2)

            self.axis.visible = self.viewerSettings.checkBoxAxis.isChecked()
            # self.xLabel.visible = self.viewerSettings.checkBoxAxis.isChecked()
            # self.yLabel.visible = self.viewerSettings.checkBoxAxis.isChecked()
            # self.zLabel.visible = self.viewerSettings.checkBoxAxis.isChecked()

        except:
            traceback.print_exc(file=sys.stderr)

    def changeCameraMode(self):

        self.viewerSettings.showHelpButton()

        if self.viewerSettings.radioButtonCameraFly.isChecked():
            # center = vrb.mainWindow.viewerNapari.camera.center
            # view_direction = vrb.mainWindow.viewerNapari.camera.view_direction
            # distance = vrb.mainWindow.viewerNapari.camera.zoom

            currentLabel = vrb.mainWindow.currentLabel
            if currentLabel is not None:
                if currentLabel.objectType == "Image":
                    sizeX = currentLabel.image.getSizeX()
                    sizeY = currentLabel.image.getSizeY()
                    sizeZ = currentLabel.image.getSizeZ()
                elif currentLabel.objectType == "Mesh":
                    sizeX = round(currentLabel.endValues[0]) - int(currentLabel.startValues[0]) + 1
                    sizeY = round(currentLabel.endValues[1]) - int(currentLabel.startValues[1]) + 1
                    sizeZ = round(currentLabel.endValues[2]) - int(currentLabel.startValues[2]) + 1

                if currentLabel.layer is not None:
                    calib = currentLabel.layer.scale
                else:
                    calib = [1,1,1]

                # get the maximal size of the volume to decide of the distance of the camera center
                maxSize = max(sizeX, sizeY, sizeZ)

                if napari.__version__.startswith('0.4'):
                    vrb.mainWindow.viewerNapariQt.camera._3D_camera = vrb.mainWindow.viewerNapariQt.camera._3D_flyCamera
                    vrb.mainWindow.viewerNapariQt.camera._view.camera = vrb.mainWindow.viewerNapariQt.camera._3D_flyCamera
                elif napari.__version__.startswith('0.5'):
                    vrb.mainWindow.viewerNapariQt.canvas.camera._3D_camera = vrb.mainWindow.viewerNapariQt.canvas.camera._3D_flyCamera
                    vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera = vrb.mainWindow.viewerNapariQt.canvas.camera._3D_flyCamera

                # Create a rotation for the fly camera to stare at the volume
                q = vispy.util.quaternion.Quaternion.create_from_euler_angles(180, 180, 0, degrees=True)

                if napari.__version__.startswith('0.4'):
                    vrb.mainWindow.viewerNapariQt.camera._view.camera.rotation1 = q
                    vrb.mainWindow.viewerNapariQt.camera._view.camera.auto_roll = False
                    # camera center is placed on the Z axis and at sizeX/2 and sizeY/2
                    vrb.mainWindow.viewerNapariQt.camera._view.camera.center = [(sizeX / 2)*calib[2], (sizeY / 2)*calib[1], (-maxSize)*calib[0]]

                elif napari.__version__.startswith('0.5'):
                    vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera.rotation1 = q
                    vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera.auto_roll = False
                    # camera center is placed on the Z axis and at sizeX/2 and sizeY/2
                    vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera.center = [(sizeX / 2)*calib[2], (sizeY / 2)*calib[1], (-maxSize)*calib[0]]

        else:
            if napari.__version__.startswith('0.4'):
                vrb.mainWindow.viewerNapariQt.camera._3D_camera = vrb.mainWindow.viewerNapariQt.camera._3D_turnCamera
                vrb.mainWindow.viewerNapariQt.camera._view.camera = vrb.mainWindow.viewerNapariQt.camera._3D_turnCamera
            elif napari.__version__.startswith('0.5'):
                vrb.mainWindow.viewerNapariQt.canvas.camera._3D_camera = vrb.mainWindow.viewerNapariQt.canvas.camera._3D_turnCamera
                vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera = vrb.mainWindow.viewerNapariQt.canvas.camera._3D_turnCamera

            # resets the view when going back to turn camera
            vrb.mainWindow.resetView()



    # create the colored planes layers used inside the 4 views mode
    def create4ViewsPlanes(self):
        currentLabel = vrb.mainWindow.currentLabel
        currentLayer = vrb.mainWindow.currentLabel.layer
        if currentLabel is not None and currentLayer is not None:
            if currentLabel.objectType == "Image":
                # get calibration infos
                if currentLabel.image.hasGeometricCalibration():
                    calibration = currentLabel.image.getGeometricCalibration()
                    originX = calibration.getXOriginUncalibrated()
                    originY = calibration.getYOriginUncalibrated()
                    originZ = calibration.getZOriginUncalibrated()

                else:
                    calibration = PyIPSDK.createGeometricCalibration(1,1,1,"px")
                    currentLabel.image.setGeometricCalibration(calibration)
                    originX = calibration.getXOrigin()
                    originY = calibration.getYOrigin()
                    originZ = calibration.getZOrigin()

                # creates the rectangles shapes for the planes XYZ taking the origin in account
                curImage = currentLabel.image
                planZ = [[originZ,originY,originX],
                         [originZ,(curImage.getSizeY()-1)+originY,originX],
                         [originZ,(curImage.getSizeY()-1)+originY,(curImage.getSizeX()-1)+originX],
                         [originZ,originY,(curImage.getSizeX()-1)+originX]]
                planY = [[(curImage.getSizeZ()-1)+originZ,originY,originX],
                         [originZ,originY,originX],
                         [originZ,originY,(curImage.getSizeX()-1)+originX],
                         [(curImage.getSizeZ()-1)+originZ,originY,(curImage.getSizeX()-1)+originX]]
                planX = [[originZ,originY,originX],
                         [originZ,(curImage.getSizeY()-1)+originY,originX],
                         [(curImage.getSizeZ()-1)+originZ,(curImage.getSizeY()-1)+originY,originX],
                         [(curImage.getSizeZ()-1)+originZ,originY,originX]]


                # create the shapes layers of XYZ planes
                if self.zPlane is None or self.yPlane is None or self.xPlane is None:
                    self.zPlane = vrb.mainWindow.viewerNapari.add_shapes(data=planZ,
                                                                        # blending='additive',
                                                                        blending='additive',
                                                                        shape_type='rectangle',
                                                                        face_color=[1.0, 0, 1.0],
                                                                        edge_color=[1.0, 0, 1.0],
                                                                        opacity=0,
                                                                        visible=True
                                                                        )
                    self.yPlane = vrb.mainWindow.viewerNapari.add_shapes(data=planY,
                                                                        blending='additive',
                                                                        shape_type='rectangle',
                                                                        face_color=[0, 1.0, 1.0],
                                                                        edge_color=[0, 1.0, 1.0],
                                                                        opacity=0,
                                                                        visible=True
                                                                        )
                    self.xPlane = vrb.mainWindow.viewerNapari.add_shapes(data=planX,
                                                                        blending='additive',
                                                                        shape_type='rectangle',
                                                                        face_color=[1.0, 1.0, 0],
                                                                        edge_color=[1.0, 1.0, 0],
                                                                        opacity=0,
                                                                        visible=True
                                                                        )
                else: # update the planes layers data
                    self.zPlane.data = planZ
                    self.yPlane.data = planY
                    self.xPlane.data = planX
                    self.zPlane.opacity = 0
                    self.yPlane.opacity = 0
                    self.xPlane.opacity = 0


                if len(currentLayer.scale) == 4:
                    self.xPlane.scale = [currentLayer.scale[1],currentLayer.scale[2],currentLayer.scale[3]]
                    self.yPlane.scale = [currentLayer.scale[1],currentLayer.scale[2],currentLayer.scale[3]]
                    self.zPlane.scale = [currentLayer.scale[1],currentLayer.scale[2],currentLayer.scale[3]]
                else:
                    self.xPlane.scale = currentLayer.scale
                    self.yPlane.scale = currentLayer.scale
                    self.zPlane.scale = currentLayer.scale

                vrb.mainWindow.updateNapariLayers()
                # reindexing the layers so that the planes layers are always above the others layers to blend correctly
                # get current index of the z plane layer
                idZ = vrb.mainWindow.viewerNapari.layers.index(self.zPlane)
                # get current layer index
                currentLayerImgIdx = vrb.mainWindow.viewerNapari.layers.index(vrb.mainWindow.currentLabel.layer)
                # put the current layer before the planeZ layer in the layers list

                if currentLayerImgIdx > idZ:
                    vrb.mainWindow.viewerNapari.layers.move(currentLayerImgIdx, idZ)


    # reorders the layers so that the orthoslices blend correctly with other layers
    def reorderLayers(self):
        currentLabel = vrb.mainWindow.currentLabel
        if currentLabel is not None:
            currentLayer = currentLabel.layer

            if currentLayer is not None and currentLabel.orthosliceZ is not None:

                # get current layer index
                currentLayerImgIdx = vrb.mainWindow.viewerNapari.layers.index(currentLayer)

                if currentLabel.orthosliceZ.blending == "additive":
                    # get index of x orthoslice of current layer
                    idOrthoX = vrb.mainWindow.viewerNapari.layers.index(currentLabel.orthosliceX)

                    # put the current layer before the planeZ layer in the layers list
                    if currentLayerImgIdx > idOrthoX:
                        # moves the currentLayer before the index idOrthoZ+1
                        vrb.mainWindow.viewerNapari.layers.move(currentLayerImgIdx, idOrthoX)

                else:
                    # get index of z orthoslice of current layer
                    idOrthoZ = vrb.mainWindow.viewerNapari.layers.index(currentLabel.orthosliceZ)

                    # put the current layer before the planeZ layer in the layers list
                    if currentLayerImgIdx < idOrthoZ:
                        # moves the currentLayer before the index idOrthoZ+1
                        vrb.mainWindow.viewerNapari.layers.move(currentLayerImgIdx, idOrthoZ + 1)

                if self.roi3D is not None:
                    # get roi3D layer index
                    roi3DIdx = vrb.mainWindow.viewerNapari.layers.index(self.roi3D)

                    if currentLayerImgIdx > roi3DIdx:
                        vrb.mainWindow.viewerNapari.layers.move(roi3DIdx, currentLayerImgIdx+1)

    def loadImageSettings(self, label):
        ######## LOADING OF VOLUMES & MESH SETTINGS #########

        currentLayer = label.layer
        labelTypes = [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16, PyIPSDK.eIBT_Label32]

        self.sliderOpacity.blockSignals(True)
        self.imageSettings.sliderGamma.blockSignals(True)
        self.imageSettings.sliderAttenuation.blockSignals(True)
        self.imageSettings.sliderIsoThreshold.blockSignals(True)
        self.imageSettings.comboBoxRendering.blockSignals(True)
        self.imageSettings.comboBoxLabelsRendering.blockSignals(True)
        self.imageSettings.comboBoxBlending.blockSignals(True)

        if self.sliderOpacity.slider.value() != int(currentLayer.opacity * 100):
            self.sliderOpacity.slider.setValue(int(currentLayer.opacity * 100))
        if self.imageSettings.comboBoxBlending.currentText() != currentLayer.blending:
            self.imageSettings.comboBoxBlending.setCurrentIndex(self.imageSettings.comboBoxBlending.findText(currentLayer.blending))

        # Loading of image settings
        start = time.time()
        self.updateColorMap()

        if label.image.getBufferType() not in labelTypes:

            if self.imageSettings.sliderGamma.slider.value() != int(currentLayer.gamma * 10):
                self.imageSettings.sliderGamma.slider.setValue(int(currentLayer.gamma * 10))
            if self.imageSettings.sliderAttenuation.slider.value() != int(currentLayer.attenuation * 100):
                self.imageSettings.sliderAttenuation.slider.setValue(int(currentLayer.attenuation * 100))

            #CONTRASTS
            if label.image.getBufferType() == PyIPSDK.eIBT_Real32:
                valueMin = float(currentLayer.contrast_limits[0])
                valueMax = float(currentLayer.contrast_limits[1])
            else:
                valueMin = int(currentLayer.contrast_limits[0])
                valueMax = int(currentLayer.contrast_limits[1])

            vrb.mainWindow.groupMenu.groupContrast.lineEditMin.setText(fct.numberCalibration(valueMin))
            vrb.mainWindow.groupMenu.groupContrast.lineEditMax.setText(fct.numberCalibration(valueMax))
            vrb.mainWindow.groupMenu.groupContrast.rangeSlider.setValues(valueMin, valueMax)
            vrb.mainWindow.groupMenu.groupContrast.rangeSlider.setValues(valueMin, valueMax)

            if self.imageSettings.sliderIsoThreshold.slider.maximum() != currentLayer.contrast_limits_range[1] or \
             self.imageSettings.sliderIsoThreshold.slider.minimum() != currentLayer.contrast_limits_range[0]:
                self.imageSettings.sliderIsoThreshold.slider.setRange(currentLayer.contrast_limits_range[0],currentLayer.contrast_limits_range[1])
            if self.imageSettings.sliderIsoThreshold.slider.value() != int(currentLayer.iso_threshold):
                self.imageSettings.sliderIsoThreshold.slider.setValue(int(currentLayer.iso_threshold))
            if self.imageSettings.comboBoxRendering.currentText() != currentLayer.rendering:
                self.imageSettings.comboBoxRendering.setCurrentIndex(self.imageSettings.comboBoxRendering.findText(currentLayer.rendering))

        else:
            if self.imageSettings.comboBoxLabelsRendering.currentText() != currentLayer.rendering:
                self.imageSettings.comboBoxLabelsRendering.setCurrentIndex(self.imageSettings.comboBoxLabelsRendering.findText(currentLayer.rendering))

        self.sliderOpacity.blockSignals(False)
        self.imageSettings.sliderGamma.blockSignals(False)
        self.imageSettings.sliderAttenuation.blockSignals(False)
        self.imageSettings.sliderIsoThreshold.blockSignals(False)
        self.imageSettings.comboBoxRendering.blockSignals(False)
        self.imageSettings.comboBoxLabelsRendering.blockSignals(False)
        self.imageSettings.comboBoxBlending.blockSignals(False)


    def loadMeshSettings(self, label):
        self.imageSettings.comboBoxShading.blockSignals(True)

        layer = label.layer

        self.imageSettings.checkBoxWireframe.setChecked(layer.wireframe.visible)
        self.imageSettings.buttonColorWireframe.setColor([layer.wireframe.color[0] * 255,
                                                          layer.wireframe.color[1] * 255,
                                                          layer.wireframe.color[2] * 255])
        self.imageSettings.comboBoxShading.setCurrentIndex(self.imageSettings.comboBoxShading.findText(layer.shading))
        self.meshSimplificationSettings.labelMeshNbFaces.setText("Original mesh: " + str(label.nbFaces) + " faces")
        self.meshSimplificationSettings.labelMeshSimplifyNbFaces.setText("Simplified mesh : " + str(label.nbFacesSimplify) + " faces")

        self.imageSettings.comboBoxShading.blockSignals(False)


    def loadClippingSettings(self, label):
        layer = label.layer
        applyClipping = True

        if layer is not None:
            self.clippingSettings.buttonBBox.setActivation(layer.bounding_box.visible)
            self.clippingSettings.buttonCrop.setVisible(label.objectType == "Image")  # sets the crop button to enabled only if the current object is an image

            if label.objectType == "Image":
                if label.image.getSizeZ() <= 1:
                    applyClipping = False
                self.spoilerClippingSettings.setVisible(label.image.getSizeZ() > 1)

                startX = 0
                startY = 0
                startZ = 0
                endX = label.image.getSizeX() - 1
                endY = label.image.getSizeY() - 1
                endZ = label.image.getSizeZ() - 1

            elif label.objectType == "Mesh":
                startX = int(label.startValues[2]) - 1
                startY = int(label.startValues[1]) - 1
                startZ = int(label.startValues[0]) - 1
                endX = round(label.endValues[2])
                endY = round(label.endValues[1])
                endZ = round(label.endValues[0])


            if len(layer.scale) != 3:
                scaleX = layer.scale[3]
                scaleY = layer.scale[2]
                scaleZ = layer.scale[1]
            else:
                scaleX = layer.scale[2]
                scaleY = layer.scale[1]
                scaleZ = layer.scale[0]

            if applyClipping:
                self.clippingSettings.sliderClipX.blockSignals(True)
                self.clippingSettings.sliderClipY.blockSignals(True)
                self.clippingSettings.sliderClipZ.blockSignals(True)

                if len(layer.experimental_clipping_planes) != 0:
                    clipX_start = int(layer.experimental_clipping_planes[0].position[0])
                    clipX_end = int(layer.experimental_clipping_planes[1].position[0])
                    clipY_start = int(layer.experimental_clipping_planes[2].position[0])
                    clipY_end = int(layer.experimental_clipping_planes[3].position[0])
                    clipZ_start = int(layer.experimental_clipping_planes[4].position[0])
                    clipZ_end = int(layer.experimental_clipping_planes[5].position[0])

                    self.clippingSettings.sliderClipX.setMinMaxValues(startX * layer.scale[2],
                                                                      endX * layer.scale[2])
                    self.clippingSettings.sliderClipY.setMinMaxValues(startY * layer.scale[1],
                                                                      endY * layer.scale[1])

                    # if (currentLabel.objectType == "Image" and currentLabel.image.getSizeZ() > 1) or currentLabel.objectType == "Mesh":
                    self.clippingSettings.sliderClipZ.setMinMaxValues(startZ * layer.scale[0],
                                                                      endZ * layer.scale[0])
                    self.clippingSettings.sliderClipZ.setVisible(True)
                    self.clippingSettings.labelClipZ.setVisible(True)
                    # else:
                    #     self.clippingSettings.sliderClipZ.setMinMaxValues(0, 1)
                    #     self.clippingSettings.sliderClipZ.setVisible(False)
                    #     self.clippingSettings.labelClipZ.setVisible(False)

                    if label.objectType == "Image":
                        if label.image.hasGeometricCalibration():
                            calibration = label.image.getGeometricCalibration()
                            # retrieve the origin of the calibration (for cropped volumes)
                            originX = calibration.getXOriginUncalibrated()
                            originY = calibration.getYOriginUncalibrated()
                            originZ = calibration.getZOriginUncalibrated()
                        else:
                            originX = 0
                            originY = 0
                            originZ = 0
                    else:
                        try:
                            originX = label.mesh.calibration.getXOriginUncalibrated()
                            originY = label.mesh.calibration.getYOriginUncalibrated()
                            originZ = label.mesh.calibration.getZOriginUncalibrated()
                        except:
                            originX=0
                            originY=0
                            originZ=0

                    self.clippingSettings.sliderClipX.setValues(clipX_start-originX, clipX_end-originX)
                    self.clippingSettings.sliderClipY.setValues(clipY_start-originY, clipY_end-originY)
                    self.clippingSettings.sliderClipZ.setValues(clipZ_start-originZ, clipZ_end-originZ)

                else:
                    self.clippingSettings.sliderClipX.setMinMaxValues(startX * layer.scale[0],
                                                                      endX * layer.scale[0])
                    self.clippingSettings.sliderClipY.setMinMaxValues(startY * layer.scale[1],
                                                                      endY * layer.scale[1])
                    self.clippingSettings.sliderClipZ.setMinMaxValues(startZ * layer.scale[2],
                                                                      endZ * layer.scale[2])
                    # self.clippingSettings.sliderClipX.setMinMaxValues(startX,
                    #                                                   endX)
                    # self.clippingSettings.sliderClipY.setMinMaxValues(startY,
                    #                                                   endY)
                    # self.clippingSettings.sliderClipZ.setMinMaxValues(startZ,
                    #                                                   endZ)

                    self.clippingSettings.sliderClipZ.setVisible(True)
                    self.clippingSettings.labelClipZ.setVisible(True)
                    # else:
                    #     self.clippingSettings.sliderClipZ.setMinMaxValues(0, 0)
                    #     self.clippingSettings.sliderClipZ.setVisible(False)
                    #     self.clippingSettings.labelClipZ.setVisible(False)

                    self.clippingSettings.sliderClipX.resetValues()
                    self.clippingSettings.sliderClipY.resetValues()
                    self.clippingSettings.sliderClipZ.resetValues()

                self.clippingSettings.sliderClipX.blockSignals(False)
                self.clippingSettings.sliderClipY.blockSignals(False)
                self.clippingSettings.sliderClipZ.blockSignals(False)


    def loadOrthoslicesSettings(self, label):
        layer = label.layer
        labelTypes = [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32]

        if vrb.mainWindow.currentLabel is not None:
            if label is not None:  # don't check if hasNapari for the orthoslices
                if label.objectType == "Image" and label.image.getSizeZ() > 1:
                    self.spoilerOrthoslicesSettings.setVisible(not label.buttonOrthoslices.activate)
                    self.reorderLayers()
                    self.orthoslicesSettings.updateMode()

                    if vrb.mainWindow.currentLabel.groupBoxLut.comboBoxLut.count() != self.orthoslicesSettings.comboBoxColormap.count():
                        if label.image.getBufferType() != PyIPSDK.eIBT_Binary:
                            self.orthoslicesSettings.addCustomLuts()

                    # show orthoslices or planes
                    if not label.buttonOrthoslices.activate:  # if orthoslices on

                        if label.image.getBufferType() in labelTypes:
                            self.orthoslicesSettings.comboBoxColormap.setCurrentIndex(1)
                        else:
                            self.orthoslicesSettings.comboBoxColormap.setCurrentIndex(0)

                        if vrb.mainWindow.viewer3dSettingsWidget.xPlane is not None and vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible:
                            # hide the colored planes
                            vrb.mainWindow.viewer3dSettingsWidget.xPlane.visible = False
                            vrb.mainWindow.viewer3dSettingsWidget.yPlane.visible = False
                            vrb.mainWindow.viewer3dSettingsWidget.zPlane.visible = False

                        # activate the button eyes so that the orthoslices appears in the viewer
                        # vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.setActivation(False)
                        # vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.setActivation(False)
                        # vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.setActivation(False)

                        #if vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.activate: #activate the buttons if they're not activated
                        vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.buttonEye.setActivation(
                            not (label.orthosliceX.visible))
                        vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.buttonEye.setActivation(
                            not (label.orthosliceY.visible))
                        vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.buttonEye.setActivation(
                            not (label.orthosliceZ.visible))

                        if self.orthoslicesSettings.sliderX.slider.maximum() != label.image.getSizeX() - 1:
                            self.orthoslicesSettings.sliderX.slider.setRange(0, label.image.getSizeX() - 1)
                        if self.orthoslicesSettings.sliderY.slider.maximum() != label.image.getSizeY() - 1:
                            self.orthoslicesSettings.sliderY.slider.setRange(0, label.image.getSizeY() - 1)
                        if self.orthoslicesSettings.sliderZ.slider.maximum() != label.image.getSizeZ() - 1:
                            self.orthoslicesSettings.sliderZ.slider.setRange(0, label.image.getSizeZ() - 1)

                        # block signals to avoid warnings
                        # self.orthoslicesSettings.sliderX.slider.blockSignals(True)
                        # self.orthoslicesSettings.sliderY.slider.blockSignals(True)
                        # self.orthoslicesSettings.sliderZ.slider.blockSignals(True)
                        # vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.blockSignals(True)
                        # vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.blockSignals(True)
                        # vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.blockSignals(True)

                        if self.orthoslicesSettings.sliderX.slider.value() != int(
                                label.orthosliceX.plane.position[0]):
                            self.orthoslicesSettings.sliderX.slider.setValue(
                                int(round(label.orthosliceX.plane.position[0])))
                        if self.orthoslicesSettings.sliderY.slider.value() != int(
                                label.orthosliceY.plane.position[0]):
                            self.orthoslicesSettings.sliderY.slider.setValue(
                                int(round(label.orthosliceY.plane.position[0])))
                        if self.orthoslicesSettings.sliderZ.slider.value() != int(
                                label.orthosliceZ.plane.position[0]):
                            self.orthoslicesSettings.sliderZ.slider.setValue(
                                int(round(label.orthosliceZ.plane.position[0])))

                        if vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.value != int(
                                label.orthosliceX.plane.position[0]):
                            vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.setValue(
                                int(label.orthosliceX.plane.position[0]))
                        if vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.value != int(
                                label.orthosliceY.plane.position[0]):
                            vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.setValue(
                                int(label.orthosliceY.plane.position[0]))
                        if vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.value != int(
                                label.orthosliceZ.plane.position[0]):
                            vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.setValue(
                                int(label.orthosliceZ.plane.position[0]))

                        # self.orthoslicesSettings.sliderX.slider.blockSignals(False)
                        # self.orthoslicesSettings.sliderY.slider.blockSignals(False)
                        # self.orthoslicesSettings.sliderZ.slider.blockSignals(False)
                        # vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.blockSignals(
                        #     False)
                        # vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.blockSignals(
                        #     False)
                        # vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.blockSignals(
                        #     False)
                        #
                        # self.orthoslicesSettings.sliderX.slider.setValue(int(round(currentLabel.orthosliceX.plane.position[0])))
                        # self.orthoslicesSettings.sliderY.slider.setValue(int(round(currentLabel.orthosliceY.plane.position[0])))
                        # self.orthoslicesSettings.sliderZ.slider.setValue(int(round(currentLabel.orthosliceZ.plane.position[0])))

                        # vrb.mainWindow.widgetImage.imageViewerStandAloneX.sliderAxis.sliderX.slider.setValue(
                        #     int(currentLabel.orthosliceX.plane.position[0]))
                        # vrb.mainWindow.widgetImage.imageViewerStandAloneY.sliderAxis.sliderY.slider.setValue(
                        #     int(currentLabel.orthosliceY.plane.position[0]))
                        # vrb.mainWindow.widgetImage.imageViewerStandAloneZ.sliderAxis.sliderZ.slider.setValue(
                        #     int(currentLabel.orthosliceZ.plane.position[0]))

                        if label.image.getBufferType() not in labelTypes:

                            if label.image.getBufferType() == PyIPSDK.eIBT_Binary:
                                if label.orthosliceX.contrast_limits != (0,1):
                                    label.orthosliceX.contrast_limits = (0,1)
                                    label.orthosliceY.contrast_limits = (0,1)
                                    label.orthosliceZ.contrast_limits = (0,1)

                            if self.orthoslicesSettings.comboBoxRendering.currentText() != label.orthosliceX.rendering:
                                self.orthoslicesSettings.comboBoxRendering.setCurrentIndex(self.orthoslicesSettings.comboBoxRendering.findText(label.orthosliceX.rendering))

                            if self.orthoslicesSettings.sliderAttenuation.slider.value() != int(label.orthosliceX.attenuation * 100):
                                self.orthoslicesSettings.sliderAttenuation.slider.setValue(
                                    int(label.orthosliceX.attenuation * 100))

                        else:
                            if self.orthoslicesSettings.comboBoxLabelsRendering.currentText() != label.orthosliceX.rendering:
                                self.orthoslicesSettings.comboBoxLabelsRendering.setCurrentIndex(
                                    self.orthoslicesSettings.comboBoxLabelsRendering.findText(
                                        label.orthosliceX.rendering))

                        if self.orthoslicesSettings.comboBoxBlending.currentText() != label.orthosliceX.blending:
                            self.orthoslicesSettings.comboBoxBlending.setCurrentIndex(
                                self.orthoslicesSettings.comboBoxBlending.findText(label.orthosliceX.blending))

                        start = time.time()
                        #if self.orthoslicesSettings.comboBoxColormap.currentText() != label.orthosliceX.colormap.name:
                            #idx = self.orthoslicesSettings.comboBoxColormap.findText(label.orthosliceX.colormap.name)
                            #if idx == -1:
                                #idx = 1
                            #self.orthoslicesSettings.comboBoxColormap.setCurrentIndex(idx)
                            #self.updateOrthosliceColorMap()

                        # self.orthoslicesSettings.sliderThickness.slider.setValue(currentLabel.orthosliceX.plane.thickness * 100)

                        if self.orthoslicesSettings.sliderThickness.slider.value() != int(
                                label.orthosliceX.plane.thickness):
                            self.orthoslicesSettings.sliderThickness.slider.setValue(
                                int(label.orthosliceX.plane.thickness))
                        if self.orthoslicesSettings.sliderOpacity.slider.value() != int(
                                label.orthosliceX.opacity * 100):
                            self.orthoslicesSettings.sliderOpacity.slider.setValue(
                                int(label.orthosliceX.opacity * 100))


                    # else:  # if checkbox orthoslice not checked
                    #     # creates the planes if they dont exist
                    #     # if vrb.mainWindow.widgetImage.modeViewer == "4Views":
                    #     #     self.create4ViewsPlanes()
                    #     vrb.mainWindow.viewer3dSettingsWidget.spoilerOrthoslicesSettings.setVisible(False)  # hides the orthoslices settings on the left
                else:
                    self.spoilerOrthoslicesSettings.setVisible(False)
                    label.buttonOrthoslices.setVisible(False)

    def loadSettings(self,label=None):
        self.loadingSettings = True
        try:
            vrb.mainWindow.groupMenu.groupContrast.allowContrastChanging = False
        except:
            pass
        if vrb.mainWindow.widgetImage.modeViewer != "2D":

            # vrb.mainWindow.viewerNapari.camera.perspective = float(self.viewerSettings.sliderFov.slider.value())

            try:
                if label is None:
                    currentLabel = vrb.mainWindow.currentLabel
                else:
                    currentLabel = label

                if currentLabel is not None and currentLabel.hasNapari:

                    self.imageSettings.changeMode(currentLabel.objectType)

                    #creates 4views planes if a volume is already imported and hides or show them
                    if vrb.mainWindow.widgetImage.modeViewer == "4Views":
                        self.create4ViewsPlanes()

                    if currentLabel.objectType == 'Image':

                        if currentLabel.image.getSizeZ() <= 1:
                            self.spoilerClippingSettings.setVisible(False)
                        self.spoilerMeshSimplificationSettings.setVisible(False)
                        self.loadImageSettings(currentLabel)

                    elif currentLabel.objectType == 'Mesh':

                        self.spoilerMeshSimplificationSettings.setVisible(True)
                        self.loadMeshSettings(currentLabel)

                    self.imageSettings.updateWidgets()
                    self.loadClippingSettings(currentLabel)

                    np.seterr(divide='ignore', invalid='ignore')  # used to ignore warnings from vispy
                    #removes the ROI when selecting a new volume
                    if self.roi3D is not None:
                        self.roi3D.data = [[0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0]]
                    np.seterr(divide='warn', invalid='warn')  # used to ignore warnings from vispy
                else:
                    self.loadingSettings = False

                if currentLabel is not None:
                    if currentLabel.orthosliceX is not None:
                        self.loadOrthoslicesSettings(currentLabel)


                self.loadingSettings = False
                self.updateVisualization(currentLabel)

            except:
                traceback.print_exc(file=sys.stderr)
                self.loadingSettings = False

        try:
            vrb.mainWindow.groupMenu.groupContrast.allowContrastChanging = True
        except:
            pass

class ThresholdSettings(qt.QGroupBox):
    def __init__(self, parent=None):
        qt.QWidget.__init__(self)

        self.parent = parent

        # For window leveling in 3D viewer
        self.thresholdSlider = RangeSlider()
        self.thresholdSlider.setFixedSize(160 * vrb.ratio, 30 * vrb.ratio)

        self.buttonReset = wgt.PushButtonImage(vrb.folderImages + "/Refresh.png", margins=2)
        self.buttonReset.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        # self.buttonApplyThreshold = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=2)
        # self.buttonApplyThreshold.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.layoutThreshold = qt.QGridLayout()
        # self.layoutThreshold.addWidget(self.thresholdSlider, 0, 0, 1, 2, Qt.AlignLeft)
        self.layoutThreshold.addWidget(self.thresholdSlider, 0, 0, Qt.AlignLeft)
        self.layoutThreshold.addWidget(self.buttonReset, 1, 0, Qt.AlignRight)
        # self.layoutThreshold.addWidget(self.buttonApplyThreshold, 1, 1, Qt.AlignRight)
        self.setLayout(self.layoutThreshold)
        self.layoutThreshold.setAlignment(Qt.AlignTop)
        self.layoutThreshold.setSizeConstraint(1)
        self.layoutThreshold.setContentsMargins(10 * vrb.ratio, 10 * vrb.ratio, 10 * vrb.ratio, 10 * vrb.ratio)
        self.layoutThreshold.setVerticalSpacing(10 * vrb.ratio)

        self.setFixedHeight(85*vrb.ratio)

class MeshSimplificationSettings(qt.QGroupBox):
    def __init__(self, parent=None):
        qt.QWidget.__init__(self)

        self.parent = parent

        self.labelReduction = qt.QLabel("Reduction factor")
        self.labelReduction.setFixedSize(120*vrb.ratio,20*vrb.ratio)
        self.lineEditReduction = qt.QLineEdit()
        self.lineEditReduction.setAlignment(Qt.AlignRight)
        self.lineEditReduction.setFixedSize(40*vrb.ratio,20*vrb.ratio)
        self.lineEditReduction.setPlaceholderText("0.9")

        self.labelMinimumFaces = qt.QLabel("Min number of faces")
        self.labelMinimumFaces.setFixedSize(120*vrb.ratio,20*vrb.ratio)
        self.lineEditMinimumFaces = qt.QLineEdit()
        self.lineEditMinimumFaces.setAlignment(Qt.AlignRight)
        self.lineEditMinimumFaces.setFixedSize(40*vrb.ratio,20*vrb.ratio)
        self.lineEditMinimumFaces.setPlaceholderText("1000")

        self.labelMeshNbFaces = qt.QLabel()
        self.labelMeshNbFaces.setFixedSize(180*vrb.ratio,20*vrb.ratio)
        self.labelMeshSimplifyNbFaces = qt.QLabel()
        self.labelMeshSimplifyNbFaces.setFixedSize(180*vrb.ratio,20*vrb.ratio)

        self.buttonApplySimplification = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=2)
        self.buttonApplySimplification.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelReduction, 0, 0)
        self.layout.addWidget(self.lineEditReduction, 0, 1)
        self.layout.addWidget(self.labelMinimumFaces, 1, 0)
        self.layout.addWidget(self.lineEditMinimumFaces, 1, 1)
        self.layout.addWidget(self.labelMeshNbFaces, 2, 0)
        self.layout.addWidget(self.labelMeshSimplifyNbFaces, 3, 0)
        self.layout.addWidget(self.buttonApplySimplification, 4, 0)
        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(10 * vrb.ratio, 10 * vrb.ratio, 10 * vrb.ratio, 10 * vrb.ratio)
        self.layout.setVerticalSpacing(10 * vrb.ratio)

        self.setFixedHeight(175*vrb.ratio)


class BackgroundImage(scene.Image):
    def draw(self):
        super().draw()
        gloo.clear(color=False, depth=True)


class ViewerSettings(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QWidget.__init__(self)

        self.parent = parent

        # Creation of background image for Napari viewer
        self.view_bg = vrb.mainWindow.viewerNapariQt.canvas.central_widget.add_view(camera=scene.PanZoomCamera())
        self.view_bg.order = float("-inf")
        imgGradient = PyIPSDK.createImage(PyIPSDK.eImageBufferType.eIBT_UInt8, 500, 500)
        self.imageGradient = arithm.formula2dImg("sqrt((Cx-x)*(Cx-x)+(Cy-y)*(Cy-y))", InOptImg1=imgGradient)
        self.imageGradient = itrans.invert2dImg(self.imageGradient)
        # structuringElement = PyIPSDK.circularSEXYInfo(50)
        # self.imageGradient = morpho.dilate2dImg(self.imageGradient, structuringElement)
        self.background_image = BackgroundImage(self.imageGradient.array, parent=self.view_bg.scene)
        self.background_image.cmap = 'grays'
        self.view_bg.camera.set_range(margin=0)

        # Creation of colorbar viewport for Napari viewer
        # canvas_size = vrb.mainWindow.viewerNapariQt.canvas.size
        # colorbar_image = PyIPSDK.loadTiffImageFile("C:/Users/ae/Desktop/colorbar.tif")
        # colorbar_array = (np.transpose(colorbar_image.array, (1,2,0))/255).astype(np.float32)
        # print("colorbar shape = ", colorbar_array.shape)
        # self.view_colorbar = vrb.mainWindow.viewerNapariQt.canvas.central_widget.add_view(camera=scene.PanZoomCamera())
        # self.view_colorbar.camera.aspect = 1  # Préserver l'aspect
        # # self.colorbar_image = BackgroundImage(colorbar_array, parent=self.view_colorbar.scene)
        # self.colorbar_image = scene.Image(colorbar_array, parent=self.view_colorbar.scene)
        # # self.colorbar_image.cmap = None  # Pas de colormap, car l'image contient déjà les couleurs
        # self.view_colorbar.camera.set_range(x=(0,colorbar_array.shape[0]), y=(0,colorbar_array.shape[1]))
        # self.colorbar_image.transform.translate = (2, 2, 0)
        # # self.view_colorbar.camera.zoom(0)  # Ajuster le zoom pour contrôler la taille de la colorbar
        # # self.view_colorbar.camera.set_range(0,)
        #
        # # self.view_colorbar.size = (10, 5)

        # print("canvas size = ", vrb.mainWindow.viewerNapariQt.canvas.size)


        # Désactiver les interactions pour le viewport de fond
        self.view_bg.interactive = False
        self.view_bg.events.mouse_blocked = True
        self.view_bg.order = float("-inf")  # S'assurer qu'il reste en arrière-plan

        # # Désactiver les interactions pour le viewport de la colorbar
        # self.view_colorbar.interactive = False
        # self.view_colorbar.events.mouse_blocked = True
        # self.view_colorbar.order = float("inf")  # S'assurer qu'il reste visible mais non interactif

        # Activer les interactions pour le viewer Napari
        vrb.mainWindow.viewerNapari.camera.interactive = True


        self.labelFov = qt.QLabel("FOV")
        self.labelFov.setFixedSize(70*vrb.ratio,20*vrb.ratio)
        self.sliderFov = SimpleHorizontalSliderLabel(label="", minimum=0, maximum=100, defaultValue=60, ratio=1,
                                                       titleSize=0 * vrb.ratio, widthLabel=10*vrb.ratio)
        self.sliderFov.setFixedSize(85*vrb.ratio,20*vrb.ratio)
        self.sliderFov.setStyleSheet('QGroupBox {border: 0px transparent; }')
        # self.sliderFov.setFixedWidth(75*vrb.ratio)
        self.sliderFov.setContentsMargins(0,2,0,2)

        self.radioButtonCameraNormal = qt.QRadioButton("Normal")
        self.radioButtonCameraNormal.setFixedSize(50*vrb.ratio,20*vrb.ratio)
        self.radioButtonCameraNormal.setChecked(True)
        self.radioButtonCameraFly = qt.QRadioButton("Fly")
        self.radioButtonCameraFly.setFixedSize(50*vrb.ratio,20*vrb.ratio)

        self.buttonCameraHelp = wgt.PushButtonImage(vrb.folderImages + "/Interrogation.png", margins=0)
        self.buttonCameraHelp.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)
        self.buttonCameraHelp.setStyleSheet("background-color : transparent; border :0px")
        self.buttonCameraHelp.setVisible(False)

        self.cameraHelpWindow = CameraHelpWindow(parent=self.parent)

        self.checkBoxAxis = qt.QCheckBox("Show axis")
        self.checkBoxAxis.setFixedSize(90*vrb.ratio,20*vrb.ratio)

        # self.labelBackgroundColor = qt.QLabel("Background color")
        self.labelBackgroundColor = qt.QLabel("Center")
        self.labelBackgroundColor.setFixedSize(40 * vrb.ratio, 20 * vrb.ratio)
        self.buttonBackgroundColor = qt.QPushButton("")
        self.buttonBackgroundColor.setFixedSize(30 * vrb.ratio, 20 * vrb.ratio)
        self.buttonBackgroundColor.setStyleSheet('QPushButton {background-color: rgb(240,240,240);}')

        self.labelBackgroundColor2 = qt.QLabel("Borders")
        self.labelBackgroundColor2.setFixedSize(40 * vrb.ratio, 20 * vrb.ratio)
        self.buttonBackgroundColor2 = qt.QPushButton("")
        self.buttonBackgroundColor2.setFixedSize(30 * vrb.ratio,20 * vrb.ratio)
        self.buttonBackgroundColor2.setStyleSheet('QPushButton {background-color: rgb(0,0,0);}')

        self.groupboxBackgroundColor = qt.QGroupBox()
        self.layoutBackgroundColor = qt.QGridLayout()
        self.layoutBackgroundColor.addWidget(self.labelBackgroundColor, 0, 0)
        self.layoutBackgroundColor.addWidget(self.labelBackgroundColor2, 0, 1)
        self.layoutBackgroundColor.addWidget(self.buttonBackgroundColor, 1, 0)
        self.layoutBackgroundColor.addWidget(self.buttonBackgroundColor2, 1, 1)
        self.layoutBackgroundColor.setAlignment(Qt.AlignLeft)
        self.groupboxBackgroundColor.setLayout(self.layoutBackgroundColor)
        self.groupboxBackgroundColor.setTitle("Background color")
        self.groupboxBackgroundColor.setStyleSheet(f'QGroupBox:title {{left: 8px ;padding-left: 10px;padding-right: 10px; padding-top: {-10*vrb.ratio}px; color:rgb(6, 115, 186)}}  QGroupBox {{font: bold; margin-top: {10*vrb.ratio} px;font-size:{12*vrb.ratio}px;border: 1px solid gray;}}')

        self.groupboxCameraMode = qt.QGroupBox()
        self.layoutCameraMode = qt.QGridLayout()
        self.layoutCameraMode.addWidget(self.radioButtonCameraNormal, 0, 0)
        self.layoutCameraMode.addWidget(self.radioButtonCameraFly, 0, 1, Qt.AlignCenter)
        self.layoutCameraMode.addWidget(self.buttonCameraHelp, 0, 2)
        self.layoutCameraMode.setAlignment(Qt.AlignLeft)
        self.groupboxCameraMode.setLayout(self.layoutCameraMode)
        self.groupboxCameraMode.setTitle("Camera mode")
        self.groupboxCameraMode.setStyleSheet(f'QGroupBox:title {{left: 8px ;padding-left: 10px;padding-right: 10px; padding-top: {-10*vrb.ratio}px; color:rgb(6, 115, 186)}}  QGroupBox {{font: bold; margin-top: {10*vrb.ratio}px;font-size:{12*vrb.ratio}px;border: 1px solid gray;}}')


        font = QtGui.QFont()
        font.setBold(False)
        # sets a font to the children of a widget
        for widget in self.groupboxBackgroundColor.findChildren(qt.QWidget):
            widget.setFont(font)
        for widget in self.groupboxCameraMode.findChildren(qt.QWidget):
            widget.setFont(font)

        self.layoutSettings = qt.QGridLayout()
        self.layoutSettings.addWidget(self.labelFov, 0, 0)
        self.layoutSettings.addWidget(self.sliderFov, 0, 1,1,2)
        self.layoutSettings.addWidget(self.groupboxCameraMode, 1, 0, 1, 3)
        self.layoutSettings.addWidget(self.groupboxBackgroundColor, 2, 0, 1, 3)
        self.layoutSettings.addWidget(self.checkBoxAxis, 3,0)
        self.setLayout(self.layoutSettings)

        self.layoutSettings.setSizeConstraint(1)
        self.layoutSettings.setContentsMargins(10 * vrb.ratio, 10 * vrb.ratio, 10 * vrb.ratio, 10 * vrb.ratio)
        self.layoutSettings.setVerticalSpacing(10 * vrb.ratio)

        self.buttonCameraHelp.clicked.connect(self.cameraHelpWindow.show)
        self.buttonBackgroundColor.clicked.connect(self.getColor)
        self.buttonBackgroundColor2.clicked.connect(self.getColor)

        self.setFixedHeight(200*vrb.ratio)


    def getColor(self):

        color = qt.QColorDialog.getColor()

        if (color.isValid()):

            if self.buttonBackgroundColor == self.sender():
                self.buttonBackgroundColor.setStyleSheet(
                    'QPushButton {background-color: rgb(' + str(color.red()) + ',' + str(
                        color.green()) + ',' + str(color.blue()) + ');}')

                palette = self.buttonBackgroundColor2.palette()
                background_color = palette.color(self.buttonBackgroundColor2.backgroundRole())

                color_center = color
                color_corner = background_color

            else:
                self.buttonBackgroundColor2.setStyleSheet(
                    'QPushButton {background-color: rgb(' + str(color.red()) + ',' + str(
                        color.green()) + ',' + str(color.blue()) + ');}')

                palette = self.buttonBackgroundColor.palette()
                background_color = palette.color(self.buttonBackgroundColor.backgroundRole())

                color_center = background_color
                color_corner = color

            # vrb.mainWindow.viewerNapariQt.canvas.background_color_override = (color.red()/255,color.green()/255,color.blue()/255,0.2)

            # background_image = BackgroundImage(self.imageGradient.array, parent=self.view_bg.scene)
            # if color.red()==0 and color.green()==0 and color.blue()==0:
            #     cmap = vispy.color.colormap.Colormap(
            #         [(0, 0, 0), (0, 0, 0)],
            #         controls=None, interpolation='linear')
            #     self.background_image.cmap = cmap
            # else:

            cmap = vispy.color.colormap.Colormap([(color_corner.red()/255,color_corner.green()/255,color_corner.blue()/255),
                                                  (color_center.red()/255, color_center.green()/255,color_center.blue()/255)],
                                                  interpolation='linear')
            self.background_image.cmap = cmap

            self.view_bg.camera.set_range(margin=0)


    def showHelpButton(self):

        if self.radioButtonCameraFly.isChecked():
            self.buttonCameraHelp.setVisible(True)
        else:
            self.buttonCameraHelp.setVisible(False)


class CameraHelpWindow(qt.QWidget):
    def __init__(self, parent=None):
        qt.QWidget.__init__(self)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.parent = parent

        self.keyboardLanguage = parent.get_keyboard_language()

        if self.keyboardLanguage == "azerty":
            self.pixmapPath = vrb.folderImages + "/cameraHelp_azerty.png"
        else:
            self.pixmapPath = vrb.folderImages + "/cameraHelp_qwerty.png"

        self.pixmap = QPixmap(self.pixmapPath)
        self.helpImgLabel = qt.QLabel()

        self.helpImgLabel.setPixmap(self.pixmap)
        self.helpImgLabel.setScaledContents(True)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.helpImgLabel, 0, 0)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(35, 25, 35, 25)
        self.layout.setHorizontalSpacing(10 * vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        # self.setFixedSize(280*vrb.ratio,250*vrb.ratio)

        self.setWindowTitle("Fly mode help")



class ImageSettings(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QWidget.__init__(self)

        self.mode = "Image"

        self.parent = parent

        self.labelGamma = qt.QLabel("Gamma")
        self.labelGamma.setFixedSize(50*vrb.ratio,20*vrb.ratio)
        self.sliderGamma = SimpleHorizontalSliderLabel(label="", minimum=2, maximum=50, defaultValue=10, ratio=10,
                                                   titleSize=0 * vrb.ratio, widthLabel=10*vrb.ratio)
        self.sliderGamma.setStyleSheet('QGroupBox {border: 0px transparent; }')
        self.sliderGamma.setFixedSize(85*vrb.ratio,20*vrb.ratio)
        # self.sliderGamma.setAlignment(Qt.AlignLeft)
        self.sliderGamma.setContentsMargins(0,2,0,2)

        self.labelBlending = qt.QLabel("Blending")
        self.labelBlending.setFixedSize(70*vrb.ratio,20*vrb.ratio)
        self.comboBoxBlending = qt.QComboBox()
        self.comboBoxBlending.setFixedSize(75*vrb.ratio,20*vrb.ratio)
        self.comboBoxBlending.addItem("translucent")
        self.comboBoxBlending.addItem("translucent_no_depth")
        self.comboBoxBlending.addItem("additive")
        self.comboBoxBlending.addItem("minimum")
        self.comboBoxBlending.addItem("opaque")
        self.comboBoxBlending.setCurrentIndex(0)

        self.labelInterpolation = qt.QLabel("Interpolation")
        self.labelInterpolation.setFixedSize(60*vrb.ratio,20*vrb.ratio)
        self.comboBoxInterpolation = qt.QComboBox()
        self.comboBoxInterpolation.setFixedSize(75*vrb.ratio,20*vrb.ratio)
        self.comboBoxInterpolation.addItem("bicubic")
        self.comboBoxInterpolation.addItem("linear")
        self.comboBoxInterpolation.addItem("kaiser")
        self.comboBoxInterpolation.addItem("nearest")
        # self.comboBoxInterpolation.addItem("spline36")
        self.comboBoxInterpolation.setCurrentIndex(1)

        self.labelRendering = qt.QLabel("Rendering")
        self.labelRendering.setFixedHeight(30*vrb.ratio)
        self.comboBoxRendering = qt.QComboBox()
        self.comboBoxRendering.setFixedSize(75*vrb.ratio,20*vrb.ratio)
        self.comboBoxRendering.addItem("translucent")
        self.comboBoxRendering.addItem("additive")
        self.comboBoxRendering.addItem("iso")
        self.comboBoxRendering.addItem("mip")
        self.comboBoxRendering.addItem("minip")
        self.comboBoxRendering.addItem("attenuated_mip")
        self.comboBoxRendering.addItem("average")
        idx = self.comboBoxRendering.findText("attenuated_mip")
        self.comboBoxRendering.setCurrentIndex(idx)

        self.comboBoxLabelsRendering = qt.QComboBox()
        self.comboBoxLabelsRendering.setFixedSize(75*vrb.ratio,20*vrb.ratio)
        self.comboBoxLabelsRendering.addItem("iso_categorical")
        self.comboBoxLabelsRendering.addItem("translucent")
        idx = self.comboBoxLabelsRendering.findText("iso_categorical")
        self.comboBoxLabelsRendering.setCurrentIndex(idx)
        self.comboBoxLabelsRendering.setVisible(False)

        # visible if rendering == iso
        self.labelIsoThreshold = qt.QLabel("Iso value")
        self.labelIsoThreshold.setFixedSize(45*vrb.ratio,20*vrb.ratio)
        self.sliderIsoThreshold = SimpleHorizontalSliderLabel(label=" ", widthLabel=10*vrb.ratio, minimum=0, maximum=100, defaultValue=0, ratio=1, titleSize=1 * vrb.ratio)
        self.sliderIsoThreshold.setStyleSheet('QGroupBox {border: 0px transparent; }')
        self.sliderIsoThreshold.setFixedSize(85 * vrb.ratio,20*vrb.ratio)
        # self.labelIsoThreshold.
        self.sliderIsoThreshold.setContentsMargins(0, 2, 0, 2)
        self.labelIsoThreshold.setVisible(False)
        self.sliderIsoThreshold.setVisible(False)

        # visible if rendering == attenuated_mip
        self.labelAttenuation = qt.QLabel("Attenuation")
        self.labelAttenuation.setFixedSize(50*vrb.ratio,20*vrb.ratio)
        self.sliderAttenuation = SimpleHorizontalSliderLabel(label="", minimum=0, maximum=60, defaultValue=5,ratio=100, titleSize=0 * vrb.ratio,logarithmique = True, logarithmiqueCoeff=12,precision=5,widthLabel=10*vrb.ratio)
        # self.sliderAttenuation = SimpleHorizontalSliderLabel(label="", minimum=0, maximum=150, defaultValue=5,ratio=100, titleSize=0 * vrb.ratio,logarithmique = True, logarithmiqueCoeff=8,precision=4,widthLabel=35)
        self.sliderAttenuation.setStyleSheet('QGroupBox {border: 0px transparent; }')
        self.sliderAttenuation.setFixedSize(85 * vrb.ratio,20*vrb.ratio)
        self.sliderAttenuation.setContentsMargins(0, 2, 0, 2)

        self.labelShading = qt.QLabel("Shading")
        self.labelShading.setFixedSize(70*vrb.ratio,20*vrb.ratio)
        self.comboBoxShading = qt.QComboBox()
        self.comboBoxShading.setFixedSize(75*vrb.ratio,20*vrb.ratio)
        self.comboBoxShading.addItem("flat")
        self.comboBoxShading.addItem("smooth")
        self.comboBoxShading.addItem("none")
        self.labelShading.setVisible(False)
        self.comboBoxShading.setVisible(False)

        self.checkBoxWireframe = qt.QCheckBox("Wireframe")
        self.checkBoxWireframe.setFixedSize(70*vrb.ratio,20*vrb.ratio)
        self.buttonColorWireframe = wgt.PushButtonColor(nameSettings="Color_Wireframe")
        self.checkBoxWireframe.setVisible(False)
        self.buttonColorWireframe.setVisible(False)

        # self.buttonReset = qt.QPushButton("Reset")
        self.buttonReset = wgt.PushButtonImage(vrb.folderImages + "/Refresh.png", margins=2)
        self.buttonReset.setFixedSize(25*vrb.ratio, 25*vrb.ratio)


        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelGamma, 0, 0)
        self.layout.addWidget(self.sliderGamma, 0, 1, Qt.AlignLeft)
        self.layout.addWidget(self.labelBlending, 1, 0)
        self.layout.addWidget(self.comboBoxBlending, 1, 1)
        self.layout.addWidget(self.labelInterpolation, 2, 0)
        self.layout.addWidget(self.comboBoxInterpolation, 2, 1)
        self.layout.addWidget(self.labelRendering, 3, 0)
        self.layout.addWidget(self.comboBoxRendering, 3, 1)
        self.layout.addWidget(self.comboBoxLabelsRendering, 3, 1)
        self.layout.addWidget(self.labelIsoThreshold, 4, 0)
        self.layout.addWidget(self.sliderIsoThreshold, 4, 1)
        self.layout.addWidget(self.labelAttenuation, 5, 0)
        self.layout.addWidget(self.sliderAttenuation, 5, 1)
        self.layout.addWidget(self.labelShading, 6, 0)
        self.layout.addWidget(self.comboBoxShading, 6, 1)
        self.layout.addWidget(self.checkBoxWireframe, 7, 0)
        self.layout.addWidget(self.buttonColorWireframe, 7, 1, Qt.AlignRight)
        self.layout.addWidget(self.buttonReset, 8, 0, Qt.AlignLeft)
        self.setLayout(self.layout)

        self.labelAttenuation.setVisible(True)
        self.sliderAttenuation.setVisible(True)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(10*vrb.ratio,10*vrb.ratio,0*vrb.ratio,10*vrb.ratio)
        self.layout.setVerticalSpacing(10*vrb.ratio)
        self.layout.setAlignment(Qt.AlignTop)

        # self.comboBoxDepiction.currentIndexChanged.connect(self.updateWidgets)
        self.comboBoxRendering.currentIndexChanged.connect(self.updateWidgets)

    def changeMode(self,mode = "Image"):

        self.mode = mode

        if self.mode == "Image":
            if vrb.mainWindow.currentLabel is not None:
                if vrb.mainWindow.currentLabel.image is not None:
	                if vrb.mainWindow.currentLabel.image.getBufferType() in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16, PyIPSDK.eIBT_Label32]:
	                    self.labelGamma.setVisible(False)
	                    self.sliderGamma.setVisible(False)
	                    self.labelInterpolation.setVisible(False)
	                    self.comboBoxInterpolation.setVisible(False)
	                    self.labelRendering.setVisible(True)
	                    self.comboBoxRendering.setVisible(False)
	                    self.comboBoxLabelsRendering.setVisible(True)
	                    self.labelShading.setVisible(False)
	                    self.comboBoxShading.setVisible(False)
	                    self.checkBoxWireframe.setVisible(False)
	                    self.buttonColorWireframe.setVisible(False)
	                    self.sliderAttenuation.setVisible(False)
	                else:
	                    self.labelGamma.setVisible(True)
	                    self.sliderGamma.setVisible(True)
	                    self.labelInterpolation.setVisible(True)
	                    self.comboBoxInterpolation.setVisible(True)
	                    # self.labelDepiction.setVisible(True)
	                    # self.comboBoxDepiction.setVisible(True)
	                    self.labelRendering.setVisible(True)
	                    self.comboBoxRendering.setVisible(True)
	                    self.comboBoxLabelsRendering.setVisible(False)
	                    self.labelShading.setVisible(False)
	                    self.comboBoxShading.setVisible(False)
	                    self.checkBoxWireframe.setVisible(False)
	                    self.buttonColorWireframe.setVisible(False)
	                    self.sliderAttenuation.setVisible(True)


        elif self.mode == "Mesh":
            self.labelGamma.setVisible(False)
            self.sliderGamma.setVisible(False)
            self.labelInterpolation.setVisible(False)
            self.comboBoxInterpolation.setVisible(False)
            # self.labelDepiction.setVisible(False)
            # self.comboBoxDepiction.setVisible(False)
            self.labelRendering.setVisible(False)
            self.comboBoxRendering.setVisible(False)
            self.comboBoxLabelsRendering.setVisible(False)
            self.labelShading.setVisible(True)
            self.comboBoxShading.setVisible(True)
            self.checkBoxWireframe.setVisible(True)
            self.buttonColorWireframe.setVisible(True)

        self.layout.setVerticalSpacing(10*vrb.ratio)

        self.updateWidgets()

    def updateWidgets(self):

        self.labelAttenuation.setVisible(False)
        self.sliderAttenuation.setVisible(False)
        self.labelIsoThreshold.setVisible(False)
        self.sliderIsoThreshold.setVisible(False)
        if self.mode == "Image":
            currentLabel = vrb.mainWindow.currentLabel
            if currentLabel is not None and currentLabel.image.getBufferType() not in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32]:
                height = 165 * vrb.ratio
                if self.comboBoxRendering.currentText() == "attenuated_mip":
                    self.labelAttenuation.setVisible(True)
                    self.sliderAttenuation.setVisible(True)
                    height += 30*vrb.ratio
                if self.comboBoxRendering.currentText() == "iso":
                    self.labelIsoThreshold.setVisible(True)
                    self.sliderIsoThreshold.setVisible(True)
                    height += 30*vrb.ratio
            else:
                height = 110*vrb.ratio
        elif self.mode == "Mesh":
            height = 110*vrb.ratio

        self.setFixedHeight(height)
        self.parent.spoilerImageSettings.changeHeight()

class ClippingSettings(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QWidget.__init__(self)

        self.parent = parent

        self.labelClipX = qt.QLabel("X")
        self.labelClipX.setFixedWidth(25*vrb.ratio)
        self.sliderClipX = RangeSlider(style="Reduced",labelSize=25*vrb.ratio)
        self.sliderClipX.setFixedSize(130 * vrb.ratio, 20 * vrb.ratio)
        self.sliderClipX.setContentsMargins(0, 2, 0, 2)

        self.labelClipY = qt.QLabel("Y")
        self.labelClipY.setFixedWidth(25*vrb.ratio)

        self.sliderClipY = RangeSlider(style="Reduced",labelSize=25*vrb.ratio)
        self.sliderClipY.setFixedSize(130 * vrb.ratio, 20 * vrb.ratio)
        self.sliderClipY.setContentsMargins(0, 2, 0, 2)

        self.labelClipZ = qt.QLabel("Z")
        self.labelClipZ.setFixedWidth(25*vrb.ratio)
        self.sliderClipZ = RangeSlider(style="Reduced",labelSize=25*vrb.ratio)
        self.sliderClipZ.setFixedSize(130 * vrb.ratio, 20 * vrb.ratio)
        self.sliderClipZ.setContentsMargins(0, 2, 0, 2)

        self.buttonReset = wgt.PushButtonImage(vrb.folderImages + "/Refresh.png", margins=2)
        self.buttonReset.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)
        self.buttonReset.setToolTip(txt.dictToolTips["ResetClipping"])

        self.buttonCrop = wgt.PushButtonImage(vrb.folderImages + "/Scissors.png", margins=2)
        self.buttonCrop.setFixedSize(25*vrb.ratio, 25*vrb.ratio)
        self.buttonCrop.setToolTip(txt.dictToolTips["Crop"])

        self.buttonBBox = wgt.PushButtonDoubleImage(vrb.folderImages + "/Bbox.png", vrb.folderImages + "/Bbox_Pressed.png")
        self.buttonBBox.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.layoutClippingSettings = qt.QGridLayout()
        self.layoutClippingSettings.addWidget(self.labelClipX, 0, 0)
        self.layoutClippingSettings.addWidget(self.sliderClipX, 0, 1, Qt.AlignLeft)
        self.layoutClippingSettings.addWidget(self.labelClipY, 1, 0)
        self.layoutClippingSettings.addWidget(self.sliderClipY, 1, 1, Qt.AlignLeft)
        self.layoutClippingSettings.addWidget(self.labelClipZ, 2, 0)
        self.layoutClippingSettings.addWidget(self.sliderClipZ, 2, 1, Qt.AlignLeft)
        self.layoutClippingSettings.addWidget(self.buttonReset, 3, 0)
        self.layoutClippingSettings.addWidget(self.buttonCrop, 3, 1, Qt.AlignHCenter)
        self.layoutClippingSettings.addWidget(self.buttonBBox, 3, 2, Qt.AlignRight)
        self.setLayout(self.layoutClippingSettings)

        self.layoutClippingSettings.setContentsMargins(10*vrb.ratio,10*vrb.ratio,10*vrb.ratio,10*vrb.ratio)
        self.layoutClippingSettings.setVerticalSpacing(10*vrb.ratio)

        self.setFixedHeight(135*vrb.ratio)



class OrthoslicesSettings(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QWidget.__init__(self)

        self.parent = parent

        self.buttonEyeX = wgt.PushButtonDoubleImage(vrb.folderImages + "/eye_open.png", vrb.folderImages + "/eye_close.png")
        self.buttonEyeX.setFixedSize(15*vrb.ratio, 15*vrb.ratio)
        self.buttonEyeX.setActivation(False)

        self.buttonEyeY = wgt.PushButtonDoubleImage(vrb.folderImages + "/eye_open.png", vrb.folderImages + "/eye_close.png")
        self.buttonEyeY.setFixedSize(15*vrb.ratio, 15*vrb.ratio)
        self.buttonEyeY.setActivation(False)

        self.buttonEyeZ = wgt.PushButtonDoubleImage(vrb.folderImages + "/eye_open.png", vrb.folderImages + "/eye_close.png")
        self.buttonEyeZ.setFixedSize(15*vrb.ratio, 15*vrb.ratio)
        self.buttonEyeZ.setActivation(False)

        self.sliderX = SimpleHorizontalSliderLabel(label="x= ", minimum=0, maximum=1, defaultValue=0, ratio=1,
                                                   titleSize=15*vrb.ratio)
        self.sliderX.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.sliderY = SimpleHorizontalSliderLabel(label="y= ", minimum=0, maximum=1, defaultValue=0, ratio=1,
                                                   titleSize=15*vrb.ratio)
        self.sliderY.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.sliderZ = SimpleHorizontalSliderLabel(label="z= ", minimum=0, maximum=1, defaultValue=0, ratio=1,
                                                   titleSize=15*vrb.ratio)
        self.sliderZ.setStyleSheet('QGroupBox {border: 0px transparent; }')

        # self.sliderThickness = SimpleHorizontalSliderLabel(label="Thickness ", minimum=100, maximum=5000,
        #                                                    defaultValue=10, ratio=100,
        #                                                    titleSize=60 * vrb.ratio)
        self.sliderThickness = SimpleHorizontalSliderLabel(label="Thickness ", minimum=1, maximum=50,
                                                           defaultValue=1, ratio=1,
                                                           titleSize=60 * vrb.ratio)
        self.sliderThickness.setStyleSheet('QGroupBox {border: 0px transparent; }')


        self.groupBoxSlicesSelection = qt.QGroupBox()
        self.layoutSlicesSelection = qt.QGridLayout()
        self.layoutSlicesSelection.addWidget(self.sliderX, 0, 0)
        self.layoutSlicesSelection.addWidget(self.buttonEyeX, 0, 1)
        self.layoutSlicesSelection.addWidget(self.sliderY, 1, 0)
        self.layoutSlicesSelection.addWidget(self.buttonEyeY, 1, 1)
        self.layoutSlicesSelection.addWidget(self.sliderZ, 2, 0)
        self.layoutSlicesSelection.addWidget(self.buttonEyeZ, 2, 1)
        self.groupBoxSlicesSelection.setLayout(self.layoutSlicesSelection)
        self.groupBoxSlicesSelection.setStyleSheet("QGroupBox{ border: 1px solid;}")

        self.sliderOpacity = SimpleHorizontalSliderLabel(label="Opacity ", minimum=0, maximum=100, defaultValue=100, ratio=100,
                                                       titleSize=60 * vrb.ratio)
        self.sliderOpacity.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.labelBlending = qt.QLabel("Blending")
        self.labelBlending.setFixedSize(70 * vrb.ratio, 20 * vrb.ratio)
        self.comboBoxBlending = qt.QComboBox()
        self.comboBoxBlending.setFixedSize(75 * vrb.ratio, 20 * vrb.ratio)
        self.comboBoxBlending.addItem("translucent")
        self.comboBoxBlending.addItem("additive")
        self.comboBoxBlending.setCurrentIndex(0)

        self.labelRendering = qt.QLabel("Rendering")
        self.labelRendering.setFixedSize(70 * vrb.ratio, 20 * vrb.ratio)
        self.comboBoxRendering = qt.QComboBox()
        self.comboBoxRendering.setFixedSize(75 * vrb.ratio, 20 * vrb.ratio)
        self.comboBoxRendering.addItem("translucent")
        self.comboBoxRendering.addItem("additive")
        self.comboBoxRendering.addItem("mip")
        self.comboBoxRendering.addItem("minip")
        self.comboBoxRendering.addItem("attenuated_mip")
        self.comboBoxRendering.addItem("average")
        self.comboBoxRendering.setCurrentIndex(0)

        self.comboBoxLabelsRendering = qt.QComboBox()
        self.comboBoxLabelsRendering.setFixedSize(75 * vrb.ratio, 20 * vrb.ratio)
        self.comboBoxLabelsRendering.addItem("iso_categorical")
        self.comboBoxLabelsRendering.addItem("translucent")
        self.comboBoxLabelsRendering.setCurrentIndex(1)
        self.comboBoxLabelsRendering.setVisible(False)


        self.labelAttenuation = qt.QLabel("Attenuation")
        self.labelAttenuation.setFixedSize(50 * vrb.ratio, 20 * vrb.ratio)
        self.sliderAttenuation = SimpleHorizontalSliderLabel(label="", minimum=0, maximum=60, defaultValue=5, ratio=100,
                                                             titleSize=0, logarithmique=True,
                                                             logarithmiqueCoeff=12, precision=5, widthLabel=30)
        self.sliderAttenuation.setStyleSheet('QGroupBox {border: 0px transparent; }')
        self.sliderAttenuation.setFixedSize(85 * vrb.ratio, 20 * vrb.ratio)
        self.sliderAttenuation.setContentsMargins(0, 2, 0, 2)

        self.labelColormap = qt.QLabel("Colormap")
        self.comboBoxColormap = qt.QComboBox()
        self.comboBoxColormap.setFixedSize(75 * vrb.ratio, 20 * vrb.ratio)

        # self.comboBoxColormap.addItem("Classic", luts.lutClassic)
        # self.comboBoxColormap.addItem("Random", luts.lutRandom)
        # self.comboBoxColormap.addItem("Inverted", luts.lutInverted)
        # self.comboBoxColormap.addItem("Magma", luts.lutMagma)
        # self.comboBoxColormap.addItem("Proba", luts.lutProba)
        # self.comboBoxColormap.addItem("Jet", luts.lutJet)
        # self.comboBoxColormap.addItem("Extrema", luts.lutExtrema)
        # self.comboBoxColormap.addItem("Bones", luts.lutBones)
        # # self.addCustomLuts()
        # self.comboBoxColormap.setCurrentIndex(1)

        # self.addCustomLuts()

        # self.buttonReset = qt.QPushButton("Reset")
        self.buttonReset = wgt.PushButtonImage(vrb.folderImages + "/Refresh.png", margins=2)
        self.buttonReset.setFixedSize(25*vrb.ratio, 25*vrb.ratio)

        self.layoutOrthoslicesSettings = qt.QGridLayout()
        self.layoutOrthoslicesSettings.addWidget(self.groupBoxSlicesSelection, 0, 0, 1, 2)
        self.layoutOrthoslicesSettings.addWidget(self.sliderThickness, 1, 0, 1, 2)
        self.layoutOrthoslicesSettings.addWidget(self.sliderOpacity, 2, 0, 1, 2)
        self.layoutOrthoslicesSettings.addWidget(self.labelRendering, 3, 0)
        self.layoutOrthoslicesSettings.addWidget(self.comboBoxRendering, 3, 1)
        self.layoutOrthoslicesSettings.addWidget(self.comboBoxLabelsRendering, 3, 1)
        self.layoutOrthoslicesSettings.addWidget(self.labelAttenuation, 4, 0)
        self.layoutOrthoslicesSettings.addWidget(self.sliderAttenuation, 4, 1)
        self.layoutOrthoslicesSettings.addWidget(self.labelBlending, 5, 0)
        self.layoutOrthoslicesSettings.addWidget(self.comboBoxBlending, 5, 1)
        self.layoutOrthoslicesSettings.addWidget(self.labelColormap, 6, 0)
        self.layoutOrthoslicesSettings.addWidget(self.comboBoxColormap, 6, 1)
        self.layoutOrthoslicesSettings.addWidget(self.buttonReset, 7, 0, Qt.AlignLeft)
        self.setLayout(self.layoutOrthoslicesSettings)


        # self.setMaximumHeight(550*vrb.ratio)

        self.layoutOrthoslicesSettings.setSizeConstraint(1)
        self.layoutOrthoslicesSettings.setContentsMargins(10 * vrb.ratio, 10 * vrb.ratio, 0 * vrb.ratio, 10 * vrb.ratio)
        self.layoutOrthoslicesSettings.setVerticalSpacing(10 * vrb.ratio)
        self.layoutOrthoslicesSettings.setAlignment(Qt.AlignTop)

        self.setFixedHeight(150 * vrb.ratio)

        self.comboBoxRendering.currentIndexChanged.connect(self.updateWidgets)


    def updateMode(self):

        if vrb.mainWindow.widgetImage.modeViewer=='4Views':
            self.groupBoxSlicesSelection.setVisible(False)

        elif vrb.mainWindow.widgetImage.modeViewer=='3D':
            self.groupBoxSlicesSelection.setVisible(True)

        if vrb.mainWindow.currentLabel is not None:
            if vrb.mainWindow.currentLabel.image.getBufferType() in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,
                                                                     PyIPSDK.eIBT_Label32]:
                self.comboBoxRendering.setVisible(False)
                self.sliderAttenuation.setVisible(False)
                self.comboBoxLabelsRendering.setVisible(True)
            else:
                self.comboBoxRendering.setVisible(True)
                self.sliderAttenuation.setVisible(True)
                self.comboBoxLabelsRendering.setVisible(False)

            if vrb.mainWindow.currentLabel.image.getBufferType() == PyIPSDK.eIBT_Binary:
                # while self.comboBoxColormap.count() != 0:
                #     self.comboBoxColormap.removeItem(self.comboBoxColormap.count()-1)
                if self.comboBoxColormap.count() == 0 or self.comboBoxColormap.count() > 2:
                    self.comboBoxColormap.clear()
                    self.comboBoxColormap.addItem("Binary", luts.lutBinary)
                    self.comboBoxColormap.addItem("BinaryGrey", luts.lutBinaryGrey)
            else:
                if self.comboBoxColormap.count() <= 2:
                    # while self.comboBoxColormap.count() != 0:
                    #     self.comboBoxColormap.removeItem(self.comboBoxColormap.count()-1)
                    self.comboBoxColormap.clear()

                    self.comboBoxColormap.addItem("Classic", luts.lutClassic)
                    self.comboBoxColormap.addItem("Random", luts.lutRandom)
                    self.comboBoxColormap.addItem("Inverted", luts.lutInverted)
                    self.comboBoxColormap.addItem("Magma", luts.lutMagma)
                    self.comboBoxColormap.addItem("Proba", luts.lutProba)
                    self.comboBoxColormap.addItem("Jet", luts.lutJet)
                    self.comboBoxColormap.addItem("Extrema", luts.lutExtrema)
                    self.comboBoxColormap.addItem("Bones", luts.lutBones)

                self.addCustomLuts()

                # self.comboBoxColormap.setCurrentIndex(0)

        self.updateWidgets()


    def updateWidgets(self):

        self.labelAttenuation.setVisible(False)
        self.sliderAttenuation.setVisible(False)

        if vrb.mainWindow.widgetImage.modeViewer=='4Views':
            height = 180 * vrb.ratio
            
        elif vrb.mainWindow.widgetImage.modeViewer=='3D':
            height = 260*vrb.ratio
        if self.comboBoxRendering.isVisible() and self.comboBoxRendering.currentText() == "attenuated_mip":
            self.labelAttenuation.setVisible(True)
            self.sliderAttenuation.setVisible(True)
            height += 30*vrb.ratio

        self.setFixedHeight(height)
        self.parent.spoilerOrthoslicesSettings.changeHeight()


    def addCustomLuts(self):

        try:
            currentIndex = self.comboBoxColormap.currentIndex()

            while self.comboBoxColormap.count() > 8:
                self.comboBoxColormap.removeItem(self.comboBoxColormap.count()-1)

            try:
                file = xmlet.parse(vrb.folderInformation + "/UserLut.mho")
                xmlElement = file.getroot()

                for child in xmlElement:
                    name = child.tag.replace("___"," ")

                    lut = {}
                    lut['Name'] = name
                    lut['Type'] = "Interp"
                    lut["Array"] = {}

                    nbSeeds = len(child)
                    maxi = 0
                    for i in range(nbSeeds):
                        childSeed = Dfct.SubElement(child, "Seed_" + str(i))
                        try:
                            maxi = max(maxi,float(Dfct.childText(childSeed, "Position")))
                        except:
                            pass

                    for i in range(nbSeeds):
                        try:
                            childSeed = Dfct.SubElement(child, "Seed_" + str(i))
                            position = maxi - float(Dfct.childText(childSeed, "Position"))
                            color = Dfct.childText(childSeed, "Color")
                            color = color.split(",")
                            for j in range(len(color)):
                                color[j] = float(color[j])

                            lut["Array"][position] = color
                        except:
                            pass

                    self.comboBoxColormap.addItem(lut['Name'], lut)

            except:
                pass

            if currentIndex < self.comboBoxColormap.count():
                self.comboBoxColormap.setCurrentIndex(currentIndex)
        except:
            traceback.print_exc(file=sys.stderr)



class VolumeManipulator(BaseManipulator):
    """A manipulator for moving and orienting an image layer or a plane."""

    def __init__(self, viewer, layer=None, mode="volume", rotator_axes=None, translator_axes=None, enabled=True):

        self.mode = mode
        self.rotator_axes = rotator_axes
        self.translator_axes = translator_axes

        if self.mode == "volume":
            super().__init__(viewer, layer, rotator_axes="xyz", translator_axes="xyz", enabled=enabled)

        elif self.mode=="plane":
            super().__init__(viewer, layer, rotator_axes=rotator_axes, translator_axes=translator_axes, enabled=enabled)


    def set_layers(self, layers: napari.layers.Image):
        super().set_layers(layers)

    def _connect_events(self):
        if self.mode == "plane":
            self.layer.plane.events.position.connect(self._update_transform)
            self.layer.plane.events.normal.connect(self._update_transform)
            self.layer.events.visible.connect(self._on_visibility_change)
            self.layer.events.depiction.connect(self._on_depiction_change)
            self._viewer.layers.events.removed.connect(self._disable_and_remove)
            self._viewer.camera.events.connect(self._update_transform)

    def _disconnect_events(self):
        if self.mode == "plane":
            self.layer.plane.events.position.disconnect(self._update_transform)
            self.layer.plane.events.normal.disconnect(self._update_transform)
            self._viewer.camera.events.disconnect(self._update_transform)

    def _update_transform(self):
        if self.mode == "plane":
            # ensure the manipulator is clamped to the layer extent
            self._backend.clamp_to_layer_bbox = True
            # get the new transformation data
            self._initialize_transform()
        else:
            # ensure the manipulator is clamped to the layer extent
            self._backend.clamp_to_layer_bbox = False
            # get the new transformation data
            self._initialize_transform()
        # redraw
        self._backend._on_transformation_changed()


    def _initialize_transform(self):

        if self.mode == 'plane':
            origin_world = self.layer.data_to_world(self.layer.plane.position)
            self.origin = np.array(origin_world)
            plane_normal_data = self.layer.plane.normal
            plane_normal_world = data_to_world_normal(vector=plane_normal_data, layer=self.layer)
            manipulator_normal = plane_normal_world

        # flip the manipulator so it's always visible
        if np.dot(manipulator_normal, self._viewer.camera.up_direction) < 0:
            manipulator_normal *= -1
        self.rotation_matrix = rotation_matrix_from_vectors_3d([1, 0, 0], manipulator_normal)


    def _pre_drag(self):
        """Préparer les actions avant le début du drag."""
        self.layer.translate = self.origin
        self.layer.rotate = self.rotation_matrix

    def _while_dragging_translator(self):
        if self.mode == 'volume':
            self.layer.translate = self.origin
        else:
            with self.layer.plane.events.position.blocker(self._update_transform):
                self.layer.plane.position = self.layer.world_to_data(self.origin)

    def _while_dragging_rotator(self):
        if self.mode == 'volume':
            self.layer.rotate = self.rotation_matrix
        else:
            with self.layer.plane.events.normal.blocker(self._update_transform):
                z_vector_data = world_to_data_normal(vector=self.z_vector, layer=self.layer)
                self.layer.plane.normal = z_vector_data

    def _post_drag(self):
        """Actions après la fin du drag."""
        if self.mode == 'volume':
            self.layer.translate = self.origin
            self.layer.rotate = self.rotation_matrix


    # def _on_depiction_change(self):
    #     if self.layer.depiction == 'plane':
    #         self.enabled = True
    #     else:
    #         self.enabled = True


if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook

    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    # foo = Viewer3dSettingsWidget()
    foo = Viewer3dSettingsScrollArea()

    widget = qt.QWidget()
    layout = qt.QGridLayout()
    layout.addWidget(foo)
    widget.setLayout(layout)


    style = fct.getStyleSheet()
    # app.setStyleSheet(style)
    widget.setStyleSheet(style)

    widget.show()

    app.exec_()



