import os
import sys, traceback

from PyQt5 import QtCore, QtWidgets as qt
from PyQt5 import QtGui
from PyQt5.QtCore import pyqtSignal, Qt, QCoreApplication

import UsefullVariables as vrb
import UsefullTexts as txt
import DatabaseFunction as Dfct
import UsefullFunctions as fct
import DrawFunctions as DWfct
import UsefullWidgets as wgt

import PyIPSDK
import PyIPSDK.IPSDKIPLUtility as util
import PyIPSDK.IPSDKIPLGeometricTransform as gtrans
import PyIPSDK.IPSDKIPLBinarization as bin
import PyIPSDK.IPSDKIPLIntensityTransform as itrans
import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr
import PyIPSDK.IPSDKIPLLogical as logic

import PyIPSDK.IPSDKUI as ui

try:
    from ImageViewer import customBlending
except:
    pass

import xml.etree.ElementTree as xmlet

import numpy as np
import shutil
import re

try:
    import matplotlib
    matplotlib.use('QT5Agg')
    import matplotlib.pylab as plt
    from matplotlib.backends.qt_compat import QtCore, QtWidgets#, is_pyqt5
    from matplotlib.backends.backend_qt5agg import FigureCanvas, NavigationToolbar2QT as NavigationToolbar
    from matplotlib.figure import Figure
except:
    pass

class DoubleLabelSpoiler(qt.QGroupBox):
    def __init__(self, title, initial=False,favorite = None):
        qt.QGroupBox.__init__(self)

        self.margins = 0
        self.sizeWidget = 25*vrb.ratio

        self.label = qt.QLabel(title)
        self.label.setFixedHeight(self.sizeWidget)
        self.label.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
        self.label.setContentsMargins(5,0,0,0)
        self.labelCollapse = qt.QLabel(("▼", "▲")[initial])
        self.labelCollapse.setFixedSize(25*vrb.ratio,self.sizeWidget)
        self.labelCollapse.setAlignment(Qt.AlignRight | Qt.AlignVCenter)
        self.labelCollapse.setContentsMargins(0,0,5,0)

        if favorite is not None:
            self.buttonFavorite = PushButtonDoubleImage(vrb.folderImages + "/Star_Disabled.png", vrb.folderImages + "/Star_Enabled.png")
            self.buttonFavorite.setStyleSheet("background-color : transparent; border :0px")
            self.buttonFavorite.setFixedSize(12*vrb.ratio,12*vrb.ratio)

        font = QtGui.QFont()
        font.setPixelSize(10*vrb.ratio)
        self.label.setFont(font)
        self.labelCollapse.setFont(font)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.label, 0, 0)
        if favorite is not None:
            self.layout.addWidget(self.buttonFavorite, 0, 1, Qt.AlignRight)
            self.layout.addWidget(self.labelCollapse, 0, 2)
        else:
            self.layout.addWidget(self.labelCollapse, 0, 1, Qt.AlignRight)

        self.setLayout(self.layout)
        self.layout.setContentsMargins(self.margins,self.margins,self.margins,self.margins)
        self.layout.setHorizontalSpacing(0)

        # self.setStyleSheet("QGroupBox {border: 0px transparent; }")
        self.setFixedHeight(self.sizeWidget)
        self.setStyleSheet("QGroupBox {border: 1px solid gray; }")


class NavigationToolbarWidget(NavigationToolbar):

    def __init__(self, plotWidget,widget):
        super().__init__(plotWidget,widget)

        self.connectToolBar()
        self.setToolBarColor()

    def connectToolBar(self):
        for i in range(self.layout().count()):
            item = self.layout().itemAt(i)
            if item is not None:
                widget = item.widget()
                if type(widget) == type(qt.QToolButton()):
                    widget.clicked.connect(self.setToolBarColor)

    def setToolBarColor(self):
        ### Customize toolbar buttons
        # pathButtons = ["back_large","filesave_large","forward_large","home_large","move_large","qt4_editor_options_large","subplots_large","zoom_to_rect_large"]
        pathButtons = ["home_large", "back_large", "forward_large", "move_large", "zoom_to_rect_large", "subplots_large", "qt4_editor_options_large", "filesave_large"]
        numButton = 0

        for i in range(self.layout().count()):
            item = self.layout().itemAt(i)
            if item is not None:
                widget = item.widget()
                if type(widget) == type(qt.QToolButton()):
                    # if numButton == 4:
                    #     widget.setChecked(False)
                    #     widget.setCheckable(False)
                    pixmap = QtGui.QPixmap(vrb.folderImages + "/" + pathButtons[numButton] + ".png")
                    sizeX, sizeY = 30, 30
                    pixmap = pixmap.scaled(sizeX, sizeY, QtCore.Qt.IgnoreAspectRatio, QtCore.Qt.SmoothTransformation)
                    icon = QtGui.QIcon(pixmap)
                    widget.setIcon(icon)

                    widget.setIconSize(QtCore.QSize(sizeX, sizeY))
                    numButton += 1

class Measure:
    def __init__(self):
        self.value = 0
        self.subType = None

class EditableLabel(qt.QWidget):
    SignalLabelTextChanged = pyqtSignal()

    def __init__(self, text):
        super().__init__()

        self.label = qt.QLabel(text)
        self.lineEdit = qt.QLineEdit(text)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.label, 0, 0)
        self.layout.addWidget(self.lineEdit, 0, 0)
        self.lineEdit.setVisible(False)

        self.setLayout(self.layout)
        self.lineEdit.returnPressed.connect(self.endOfEdit)
        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(0, 0, 0, 0)

    def switchVisibleWidget(self):
        if self.label.isVisible():
            self.lineEdit.setText(self.label.text())
            self.label.setVisible(False)
            self.lineEdit.setVisible(True)
            self.lineEdit.setFocus()
            self.lineEdit.focusOutEvent = self.endOfEdit
        else:
            self.label.setText(self.lineEdit.text().replace('\n', ''))
            self.label.setVisible(True)
            self.lineEdit.setVisible(False)
            self.lineEdit.focusOutEvent = None

    def endOfEdit(self, event=None):
        self.switchVisibleWidget()
        self.SignalLabelTextChanged.emit()

    # def resizeEvent(self, event):
    #     self.label.setFixedSize(self.width() * 100 / 100, self.height() * 100 / 100)
    #     self.lineEdit.setFixedSize(self.width() * 100 / 100, self.height() * 100 / 100)

class SwitchingLabel(qt.QStackedWidget):
    SignalWidgetSwitched = pyqtSignal(int)

    def __init__(self, label1, label2):
        super(SwitchingLabel, self).__init__()
        self.state = 0
        self.label1 = label1
        self.label2 = label2

        self.addWidget(label1)
        self.addWidget(label2)

        self.setState(self.state)

        self.setContentsMargins(0, 0, 0, 0)

        self.label1.mousePressEvent = self.switchWidget
        self.label2.mousePressEvent = self.switchWidget

    def setState(self, newState):
        self.state = newState
        self.setCurrentWidget((self.label1, self.label2)[self.state])

    def switchWidget(self, event: QtGui.QMouseEvent) -> None:
        if event.button() == QtCore.Qt.LeftButton:
            self.setState((self.state + 1) % 2)
            self.SignalWidgetSwitched.emit(self.state)

    def clickedWidget(self):
        self.setState(1-self.state)
        self.SignalWidgetSwitched.emit(self.state)

class ScrollableLabel(qt.QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)

        # Create a QLabel
        self.label = qt.QLabel()
        # self.label.setText("This is a very long text. " * 20)  # Long text
        self.label.setWordWrap(True)  # Ensure the text wraps

        # Create a QScrollArea
        self.scrollArea = qt.QScrollArea()
        self.scrollArea.setWidget(self.label)
        self.scrollArea.setWidgetResizable(True)

        # Create a layout and add the scroll area to it
        layout = qt.QVBoxLayout()
        layout.addWidget(self.scrollArea)

        self.setLayout(layout)
        self.setWindowTitle("Scrollable QLabel Example")
        self.setGeometry(100, 100, 400, 300)

class LabelWithLineEdit(qt.QGroupBox):

    def __init__(self,sizeLabel = 20,textLabel = "",textLineEdit = "",typeValue = "Number"):

        qt.QGroupBox.__init__(self)

        self.sizeLabel = sizeLabel

        self.layout = qt.QHBoxLayout()

        self.label = qt.QLabel(textLabel)
        self.label.setFixedSize(sizeLabel*vrb.ratio,20*vrb.ratio)
        if typeValue == "Number":
            self.lineEdit = NumberLineEdit()
        else:
            self.lineEdit = qt.QLineEdit()
        self.lineEdit.setPlaceholderText(textLineEdit)

        self.layout.addWidget(self.label)
        self.layout.addWidget(self.lineEdit)

        self.setLayout(self.layout)
        self.layout.setContentsMargins(0, 0, 0, 0)
        self.setStyleSheet("QGroupBox {border: 0px transparent;}")

    def resizeEvent(self, event):

        self.lineEdit.setFixedSize(self.width()-(self.sizeLabel+5)*vrb.ratio,20*vrb.ratio)

class RadioButtonWithLineEdit(qt.QGroupBox):

    def __init__(self,sizeLabel = 20,textLabel = "",textLineEdit = "",typeValue = "Number"):

        qt.QGroupBox.__init__(self)

        self.sizeLabel = sizeLabel

        self.layout = qt.QHBoxLayout()

        self.radioButton = qt.QRadioButton(textLabel)
        self.radioButton.setFixedSize(sizeLabel*vrb.ratio,20*vrb.ratio)
        if typeValue == "Number":
            self.lineEdit = NumberLineEdit()
        else:
            self.lineEdit = qt.QLineEdit()
        self.lineEdit.setPlaceholderText(textLineEdit)

        self.layout.addWidget(self.radioButton)
        self.layout.addWidget(self.lineEdit)

        self.setLayout(self.layout)
        self.layout.setContentsMargins(0, 0, 0, 0)
        self.setStyleSheet("QGroupBox {border: 0px transparent;}")

    def resizeEvent(self, event):

        self.lineEdit.setFixedSize(self.width()-(self.sizeLabel+5)*vrb.ratio,20*vrb.ratio)

class MacroLineEdit(qt.QLineEdit):
    SignalTextEditedVerif = pyqtSignal()

    def __init__(self, **kwargs):
        qt.QLineEdit.__init__(self)

        self.textEdited.connect(self.verifNameCharacter)

    def verifNameCharacter(self):

        newText = ''
        for character in self.text():

            if newText != '' or character in vrb.firstCharacterMacro:
                newText+=character
        if newText != self.text():
            self.setText(newText)
        self.SignalTextEditedVerif.emit()

class CustomLineEdit(qt.QLineEdit):
    SignalTextEditedVerif = pyqtSignal()

    def __init__(self, **kwargs):
        qt.QLineEdit.__init__(self)

        if 'mask' in kwargs:
            self.mask = kwargs.pop('mask')
        else:
            self.mask = None
        if 'align' in kwargs:
            self.align = kwargs.pop('align')
        else:
            self.align = QtCore.Qt.AlignRight

        self.setAlignment(self.align)
        self.textEdited.connect(self.verifNameCharacter)

    def verifNameCharacter(self):

        newText = ''
        for character in self.text():
            if character in self.mask:
                newText += character
        if newText != self.text():
            self.setText(newText)
        self.SignalTextEditedVerif.emit()


class NumberLineEdit(qt.QLineEdit):
    SignalTextEditedVerif = pyqtSignal()

    def __init__(self, **kwargs):
        qt.QLineEdit.__init__(self)

        if 'align' in kwargs:
            self.align = kwargs.pop('align')
        else:
            self.align = QtCore.Qt.AlignRight

        if 'constraint' in kwargs:
            self.constraint = kwargs.pop('constraint')
        else:
            self.constraint = "Default"

        self.mask = vrb.maskNumbers

        self.setAlignment(self.align)
        self.textEdited.connect(self.verifNameCharacter)

    def verifNameCharacter(self):

        try:
            if self.constraint.lower() != "any" :
                newText = ''
                if self.constraint == "Natural":
                    for character in self.text():
                        if character in self.mask:
                            newText += character
                elif self.constraint == "Relative":
                    for i in range(len(self.text())):
                        if (self.text()[i] in self.mask) or (i == 0 and self.text()[i] == '-'):
                            newText += self.text()[i]
                elif self.constraint == "Real" or self.constraint == "Default":
                    nbPoints = 0
                    for i, character in enumerate(self.text(), 0):
                        if character in self.mask or (i == 0 and self.text()[i] == '-'):
                            newText += character
                        if character == '.' and nbPoints == 0 and newText != '' and newText != '-':
                            newText += character
                            nbPoints += 1
                if newText != self.text():
                    self.setText(newText)
                self.SignalTextEditedVerif.emit()
            else:
                self.SignalTextEditedVerif.emit()

        except:
            self.SignalTextEditedVerif.emit()


class DecimalLineEdit(qt.QLineEdit):
    SignalTextEditedVerif = pyqtSignal()

    def __init__(self, **kwargs):
        qt.QLineEdit.__init__(self)

        if 'align' in kwargs:
            self.align = kwargs.pop('align')
        else:
            self.align = QtCore.Qt.AlignRight

        self.mask = vrb.maskNumbers

        self.setAlignment(self.align)
        self.textEdited.connect(self.verifNameCharacter)

    def verifNameCharacter(self):

        newText = ''
        nbPoints = 0
        for character in self.text():
            if character in self.mask:
                newText += character
            if character == '.' and nbPoints == 0 and newText != '':
                newText += character
                nbPoints += 1
        self.setText(newText)
        self.SignalTextEditedVerif.emit()


class RelativeLineEdit(qt.QLineEdit):
    SignalTextEditedVerif = pyqtSignal()

    def __init__(self, **kwargs):
        qt.QLineEdit.__init__(self)

        if 'align' in kwargs:
            self.align = kwargs.pop('align')
        else:
            self.align = QtCore.Qt.AlignRight

        self.mask = vrb.maskNumbers

        self.setAlignment(self.align)
        self.textEdited.connect(self.verifNameCharacter)

    def verifNameCharacter(self):

        newText = ''
        for i in range(len(self.text())):
            if (self.text()[i] in self.mask) or (i == 0 and self.text()[i] == '-'):
                newText += self.text()[i]
        self.setText(newText)
        self.SignalTextEditedVerif.emit()

class LabelClickable(qt.QLabel):
    clicked = pyqtSignal()

    def __init__(self,text = ""):
        qt.QLabel.__init__(self)

        self.setText(text)

    def mousePressEvent(self, ev: QtGui.QMouseEvent) -> None:
        self.clicked.emit()

class LabelHoverable(qt.QLabel):
    clicked = pyqtSignal()

    def __init__(self, text, style, styleHover):
        qt.QLabel.__init__(self)

        self.style = style
        self.styleHover = styleHover

        self.setText(text)
        self.setAlignment(QtCore.Qt.AlignCenter)
        font = QtGui.QFont()
        font.setPixelSize(15*vrb.ratio)
        self.setFont(font)
        self.setContentsMargins(0, 0, 0, 5)

    def enterEvent(self, event):
        if self.isEnabled():
            self.setStyleSheet("LabelHoverable { " + self.styleHover + "}")

    def leaveEvent(self, event):
        if self.isEnabled():
            self.setStyleSheet("LabelHoverable { " + self.style + "}")

    def mousePressEvent(self, ev: QtGui.QMouseEvent) -> None:
        self.clicked.emit()


class LabelDelete(LabelHoverable):
    def __init__(self, text='x'):
        super(LabelDelete, self).__init__(text, "background-color: solid grey; color: white", "background-color: rgb(255,0,0); color: white")
        self.setStyleSheet("background-color: solid grey; color: white")
        self.setFixedWidth(20*vrb.ratio)

class LabelAdd(LabelHoverable):
    def __init__(self, text='+'):
        super(LabelAdd, self).__init__(text, "background-color: solid grey; color: white", "background-color: rgb(80,80,80); color: white")
        self.setFixedWidth(20*vrb.ratio)

class LabelEdit(LabelHoverable):
    def __init__(self, text='e'):
        super(LabelEdit, self).__init__(text, "background-color: solid grey; color: white", "background-color: rgb(80,80,80); color: white")
        self.setFixedWidth(20*vrb.ratio)

class LabelImage(qt.QLabel):

    def __init__(self, filename=None, image=None,pixmap = None):
        qt.QLabel.__init__(self)

        self.filename = filename
        self.image = image
        self.pixmap = pixmap
        self.smooth = True

    def resizeEvent(self, event):

        if self.image is None and self.pixmap is None:
            if self.filename != None:
                pixmap = QtGui.QPixmap(self.filename)
                if self.smooth:
                    self.setPixmap(pixmap.scaled(self.width(), self.height(), QtCore.Qt.IgnoreAspectRatio, QtCore.Qt.SmoothTransformation))
                else:
                    self.setPixmap(pixmap.scaled(self.width(), self.height()))
        elif self.filename is None:
            if self.pixmap is None:
                if self.image.shape[2] == 1:
                    qimg = QtGui.QImage(self.image, self.image.shape[1], self.image.shape[0], self.image.shape[1], QtGui.QImage.Format_Grayscale8)
                if self.image.shape[2] == 3:
                    qimg = QtGui.QImage(self.image, self.image.shape[1], self.image.shape[0], 3 * self.image.shape[1], QtGui.QImage.Format_RGB888)
                pixmap = QtGui.QPixmap.fromImage(qimg)
            else:
                pixmap = self.pixmap
            if self.smooth:
                self.setPixmap(pixmap.scaled(self.width(), self.height(), QtCore.Qt.IgnoreAspectRatio, QtCore.Qt.SmoothTransformation))
            else:
                self.setPixmap(pixmap.scaled(self.width(), self.height()))


# class LabelImageListWidget(qt.QGroupBox):
#
#     def __init__(self, filename=None,size = 15*vrb.ratio):
#         qt.QGroupBox.__init__(self)
#
#         # self.emptyLabel = qt.QLabel("")
#         self.labelImage = LabelImage(filename=filename)
#         # self.labelImage.setFixedSize(size,size)
#
#         self.layout = qt.QGridLayout()
#         # self.layout.addWidget(self.emptyLabel)
#         self.layout.addWidget(self.labelImage,0,0)
#         self.setLayout(self.layout)
#
#         # self.layout.setSizeConstraint(1)
#         # self.layout.setContentsMargins(10*vrb.ratio,10*vrb.ratio,10*vrb.ratio,10*vrb.ratio)



class LabelLut(LabelImage):

    def __init__(self, filename=None, image=None):
        LabelImage.__init__(self)

        self.setMouseTracking(True)

        self.valueMin = 0
        self.valueMax = 255

    def mouseMoveEvent(self, event):
        ratio = event.pos().y() / (self.height() - 1)

        qt.QToolTip.showText(QtGui.QCursor.pos(), str(int(self.valueMin * ratio + self.valueMax * (1 - ratio))))

class LabelPythonImage(qt.QLabel):

    def __init__(self, image=None, parent=None):
        qt.QLabel.__init__(self)

        self.image = image
        self.imageMiniature = None
        self.imageOverlay = None

        self.parent = parent
        self.initImage(image)

    def initImage(self,image,imageOverlay = None):

        if image.getSizeX() > image.getSizeY():
            newSizeY = 200
            newSizeX = newSizeY * image.getSizeX() / image.getSizeY()
        else:
            newSizeX = 200
            newSizeY = newSizeX * image.getSizeY() / image.getSizeX()

        if image.getSizeZ() > 1:
            if image.isDiskImage():
                image = fct.loadPlanImageDisk(image,image.getSizeX(),image.getSizeY(),0,0,1,1,zValue=int(image.getSizeZ()/2))
            else:
                if image.getSizeC() == 1:
                    image = PyIPSDK.extractPlan(int(image.getSizeZ() / 2), 0, 0, image)
                else:
                    image = PyIPSDK.extractColor(int(image.getSizeZ() / 2), 0, image)
            #image = util.copyImg(image)

        interp = PyIPSDK.eZoomInterpolationMethod.eZIM_NearestNeighbour
        sizeX,sizeY = image.getSizeX(),image.getSizeY()

        image = fct.zoomDiskAndNormalImage(image, newSizeX / sizeX, newSizeY / sizeY, interp =  interp)

        # image = gtrans.zoom2dImg(image, newSizeX / sizeX, newSizeY / sizeY, interp)

        self.image = image

        if imageOverlay is not None:
            if imageOverlay.getSizeZ() > 1:
                if imageOverlay.getSizeC() == 1:
                    imageOverlay = PyIPSDK.extractPlan(int(imageOverlay.getSizeZ() / 2), 0, 0, imageOverlay)
                else:
                    imageOverlay = PyIPSDK.extractColor(int(imageOverlay.getSizeZ() / 2), 0, imageOverlay)
                imageOverlay = util.copyImg(imageOverlay)
            imageOverlay = gtrans.zoom2dImg(imageOverlay, newSizeX / sizeX, newSizeY / sizeY, interp)
            self.imageOverlay = imageOverlay

        self.imageMiniature = None
        self.changeImage()

    def changeImage(self):

        if self.parent is not None:
            self.imageMiniature = fct.imageToDisplay(self.image, self.parent)
        else:
            self.imageMiniature = self.image

        if self.imageOverlay is not None:

            if self.imageMiniature.getBufferType() != PyIPSDK.eIBT_UInt8:
                self.imageMiniature = itrans.normalizeImg(self.imageMiniature, PyIPSDK.createRange(0, 255))
                self.imageMiniature = util.convertImg(self.imageMiniature, PyIPSDK.eIBT_UInt8)

            if self.imageOverlay.getSizeC() > 1:
                imageOverlay = fct.convertToRgbImage(self.imageOverlay, self.parent)
            else:
                imageOverlay = self.imageOverlay
            refImage = bin.lightThresholdImg(self.image, 1)

            if imageOverlay.getBufferType() != PyIPSDK.eIBT_UInt8:
                imageOverlay = itrans.normalizeImg(imageOverlay, PyIPSDK.createRange(0, 255))
                imageOverlay = util.convertImg(imageOverlay, PyIPSDK.eIBT_UInt8)

            self.imageMiniature = customBlending(self.imageMiniature,imageOverlay,0.5,refImage)

        self.resizeEvent(None)

    def resizeEvent(self, event=None):

        if self.imageMiniature is not None:

            try:

                if self.imageMiniature.getSizeC() == 1:
                    imgFormat = QtGui.QImage.Format_Grayscale8
                    qimg = QtGui.QImage(self.imageMiniature.array, self.imageMiniature.getSizeX(), self.imageMiniature.getSizeY(), self.imageMiniature.getSizeX(), imgFormat)

                if self.imageMiniature.getSizeC() == 3:
                    imgFormat = QtGui.QImage.Format_RGB888
                    qimg = QtGui.QImage(self.imageMiniature.getSizeX(), self.imageMiniature.getSizeY(), imgFormat)
                    for v in range(self.imageMiniature.getSizeY()):
                        ptr = qimg.scanLine(v)
                        ptr.setsize(qimg.bytesPerLine())
                        lineArr = np.asarray(ptr)
                        for c in range(3):
                            lineArr[c::3][:self.imageMiniature.getSizeX()] = self.imageMiniature.array[c][v][0:self.imageMiniature.getSizeX()]
                pixmap = QtGui.QPixmap.fromImage(qimg)
                self.setPixmap(pixmap.scaled(self.width(), self.height(), QtCore.Qt.IgnoreAspectRatio, QtCore.Qt.SmoothTransformation))

            except:

                traceback.print_exc(file=sys.stderr)


class PushButtonImage(qt.QPushButton):

    def __init__(self, filename, margins=5):
        qt.QPushButton.__init__(self)

        self.pixmap = QtGui.QPixmap(filename)
        self.margins = margins

        self.setStyleSheet("PushButtonImage { border:0px }")

    def resizeEvent(self, event):

        sizeX,sizeY = self.width(),self.height()

        pixmap = self.pixmap.scaled(max(0,sizeX - self.margins), max(0,sizeY - self.margins), QtCore.Qt.IgnoreAspectRatio, QtCore.Qt.SmoothTransformation)
        icon = QtGui.QIcon(pixmap)
        self.setIcon(icon)
        self.setIconSize(QtCore.QSize(max(0,sizeX - self.margins), max(0,sizeY - self.margins)))

class PushButtonDoubleImage(qt.QPushButton):

    def __init__(self, filename1,filename2, margins1=0,margins2=0):
        qt.QPushButton.__init__(self)

        self.pixmap1 = QtGui.QPixmap(filename1)
        self.pixmap2 = QtGui.QPixmap(filename2)
        self.margins1 = margins1
        self.margins2 = margins2

        self.activate = False

    def changeActivation(self):

        self.activate = not(self.activate)
        self.resizeEvent(None)

    def setActivation(self,boolValue):

        self.activate = boolValue
        self.resizeEvent(None)

    def resizeEvent(self, event):

        sizeX,sizeY = self.width(),self.height()

        if self.activate:
            margins = self.margins2
            pixmap = self.pixmap2.scaled(max(0,sizeX - margins), max(0,sizeY - margins), QtCore.Qt.IgnoreAspectRatio, QtCore.Qt.SmoothTransformation)
        else:
            margins = self.margins1
            pixmap = self.pixmap1.scaled(max(0,sizeX - margins), max(0,sizeY - margins), QtCore.Qt.IgnoreAspectRatio, QtCore.Qt.SmoothTransformation)

        icon = QtGui.QIcon(pixmap)
        self.setIcon(icon)
        self.setIconSize(QtCore.QSize(sizeX - margins, sizeY - margins))

class PushButtonColor(qt.QPushButton):

    signalChange = pyqtSignal()

    def __init__(self,nameSettings=None):
        qt.QPushButton.__init__(self)

        self.nameSettings = nameSettings

        if self.nameSettings is not None:
            try:
                file = xmlet.parse(vrb.folderInformation + "/"+self.nameSettings + ".mho")
                self.colorElement = file.getroot()
            except:
                self.colorElement = xmlet.Element('Settings')

        try:
            if self.colorElement.text is not None and self.colorElement.text != "":
                colorSplit = self.colorElement.text.split(",")
                self.color = [int(colorSplit[0]),int(colorSplit[1]),int(colorSplit[2])]
            else:
                self.color = [255, 0, 0]
        except:
            self.color = [255,0,0]

        self.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.setToolTip("Change the color")

        self.setColor(self.color)

        self.clicked.connect(self.getColor)


    def getColor(self):

        color = qt.QColorDialog.getColor()
        if (color.isValid()):

            self.color = [color.red(),color.green(),color.blue()]
            self.setStyleSheet('QPushButton {background-color: rgb('+str(color.red())+','+str(color.green())+','+str(color.blue()) +');}')
            self.colorElement.text = str(color.red())+","+str(color.green())+","+str(color.blue())
            if self.nameSettings is not None:
                Dfct.saveXmlElement(self.colorElement, vrb.folderInformation + "/"+self.nameSettings + ".mho")

        self.signalChange.emit()

    def setColor(self,color):

        self.color = color
        self.setStyleSheet('QPushButton {background-color: rgb(' + str(color[0]) + ',' + str(color[1]) + ',' + str(color[2]) + ');}')
        self.colorElement.text = str(color[0]) + "," + str(color[1]) + "," + str(color[2])
        if self.nameSettings is not None:
            Dfct.saveXmlElement(self.colorElement, vrb.folderInformation + "/" + self.nameSettings + ".mho")


class GenericTableDisplayer(qt.QWidget):
    def __init__(self, ipsdkObject):

        self.ipsdkObject = ipsdkObject
        self.tableView = qt.QTableWidget()

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.tableView, 0, 0)
        self.setLayout(self.layout)

    def initValues(self):
        self.tableView.clear()

    def stopDisplay(self):
        self.close()

    def display(self, xmlElement=None, boolShow=True):
        self.setWindowTitle(Dfct.childText(xmlElement, 'Name'))
        if boolShow:
            if self.isMaximized():
                self.showMaximized()
            else:
                self.showNormal()
            self.window().raise_()
            self.window().activateWindow()
        self.initValues()


class PopUpWindow(qt.QWidget):
    def __init__(self):
        super().__init__()

        self.margins = [10, 10, 10, 10]

        self.pushButtonClose = qt.QPushButton('Close')
        self.pushButtonClose.setMinimumSize(70, 15)

        self.labelError = qt.QLabel('')
        self.labelError.setStyleSheet("QLabel { background-color : white; color : red; }")
        self.labelError.setTextInteractionFlags(Qt.LinksAccessibleByMouse | Qt.TextSelectableByMouse)
        self.labelError.setWordWrap(True)
        self.labelError.setContentsMargins(*self.margins)

        font = QtGui.QFont("Arial", 12)
        self.labelError.setFont(font)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelError, 0, 0, 1, 2)
        # self.layout.addWidget(self.pushButtonClose, 1, 1, Qt.AlignRight)

        self.layout.setContentsMargins(*self.margins)
        self.layout.setSizeConstraint(1)
        self.setLayout(self.layout)

        # self.pushButtonClose.clicked.connect(self.close)
        self.labelError.setOpenExternalLinks(False)
        self.labelError.linkActivated.connect(self.openLink)

        self.setMinimumWidth(400*vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def setMessage(self, text):

        self.labelError.setText(text)

    def setMessageAndShow(self, text, title):
        self.setWindowTitle('Error ' + title)
        self.setMessage(text)
        fct.showWidget(self)
        # self.close()
        # self.show()

    def openLink(self,link):

        macroName = link.split(" ")[0]
        line = link.split(" ")[1]

        if macroName in vrb.dictMacros:
            functionWidget = vrb.dictMacros[macroName].widget

            if functionWidget.isInit == False:
                functionWidget.initFunction()

            functionWidget.editMacro()
            qt.QApplication.processEvents()
            functionWidget.editScriptWindow.textEditScript.placeCursorOnLine(int(line)-1)

class CustomContextMenuReference(qt.QMenu):

    def __init__(self, parent):
        super().__init__(parent)

    def showMenu(self, position):
        self.exec(self.parent().mapToGlobal(position))

class CustomContextMenu(qt.QMenu):
    signalDelete = pyqtSignal()

    def __init__(self, parent):
        super().__init__(parent)
        self.addAction('Rename', self.renameLabel)
        self.addAction('Delete', self.deleteLabel)

    def showMenu(self, position):
        self.exec(self.parent().mapToGlobal(position))

    def renameLabel(self):
        self.parent().editableLabelName.switchVisibleWidget()

    def deleteLabel(self):
        self.signalDelete.emit()

class WidgetValidate(qt.QWidget):
    signalValidate = pyqtSignal()

    def __init__(self, text):
        qt.QWidget.__init__(self)

        self.labelText = qt.QLabel(text)
        self.pushButtonConfirm = qt.QPushButton("Yes")
        self.pushButtonCancel = qt.QPushButton("No")

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelText, 0, 0, 1, 2)
        self.layout.addWidget(self.pushButtonConfirm, 1, 0)
        self.layout.addWidget(self.pushButtonCancel, 1, 1)

        self.setLayout(self.layout)

        self.setWindowTitle("Explorer")

        self.pushButtonConfirm.clicked.connect(self.emitConfirm)
        self.pushButtonCancel.clicked.connect(self.emitCancel)

    def emitConfirm(self):
        self.signalValidate.emit()
        self.close()

    def emitCancel(self):
        self.close()


class SliderShowingValue(qt.QWidget):
    SignalSliderChangedValue = pyqtSignal()

    def __init__(self, initialValue=0, orientation=Qt.Horizontal, invert=False, prefix = "",minValue = 1, maxValue = 20):
        super().__init__()
        self.invert = invert
        self.initialValue = initialValue
        self.prefix = prefix

        self.slider = qt.QSlider()
        self.slider.setOrientation(orientation)
        # self.slider.setTickPosition(qt.QSlider.TicksAbove)
        self.slider.setTickInterval(1)
        self.slider.setRange(minValue, maxValue)
        self.slider.setValue(initialValue)
        self.labelValue = qt.QLabel(self.prefix + str(self.initialValue))

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelValue, 0, 0, Qt.AlignCenter)
        self.layout.addWidget(self.slider, 1, 0)
        self.setLayout(self.layout)

        self.layout.setSpacing(0)

        self.slider.valueChanged.connect(self.sliderChanged)

    def setInitialValue(self, newInitialValue):
        self.initialValue = newInitialValue
        self.sliderChanged(None, emit=False)

    def getValue(self):
        value = self.initialValue * self.slider.value()
        if self.invert:
            value = self.slider.maximum() - value
        return value

    def getSliderPosition(self):
        return self.slider.value()

    def sliderChanged(self, event, emit=True):
        newValue = self.getValue()
        self.labelValue.setText(self.prefix + str(fct.numberCalibration(newValue,3)))
        if emit:
            self.SignalSliderChangedValue.emit()


class HistoryWidget(qt.QGroupBox):
    def __init__(self, xmlElement):
        super().__init__()
        self.setTitle('History')
        self.xmlElement = xmlElement
        self.functionCallXml = Dfct.SubElement(self.xmlElement, 'FunctionCall')

        self.labelParents = []
        # for parameterXml in parametersIterator(Dfct.SubElement(self.functionCallXml, 'Parameters')):
        #     self.labelParents.append(qt.QLabel(parameterXml.text))
        try:
            self.labelFunction = qt.QLabel('Function: ' + Dfct.childText(self.functionCallXml, 'Name'))
        except:
            self.labelFunction = qt.QLabel('Function:')

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelFunction, 0, 0)
        # for nb, label in enumerate(self.labelParents, 0):
        #     self.layout.addWidget(label, 1, nb)
        self.setLayout(self.layout)


# class PropertiesWindow(qt.QWidget):
#     def __init__(self, xmlElement):
#         qt.QWidget.__init__(self)
#
#         self.xmlElement = xmlElement
#         # self.ipsdkElement = ipsdkElement
#
#         # groupbox for general properties
#         self.groupBoxGeneral = qt.QGroupBox()
#         self.groupBoxGeneral.setStyleSheet(
#             'QGroupBox:title {left: 20px; top: -12px ;padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {border: 1px solid gray; margin-top: 12 px}')
#
#         # name of image
#         self.labelName = qt.QLabel("Name")
#         self.labelName.setFixedWidth(100*vrb.ratio)
#         self.labelNameRes = qt.QLabel()
#         # processing time
#         self.labelTime = qt.QLabel("Processing time")
#         self.labelTime.setFixedWidth(100*vrb.ratio)
#         self.labelTimeRes = qt.QLabel()
#         # function name
#         self.labelFunction = qt.QLabel("Function")
#         self.labelFunction.setFixedWidth(100*vrb.ratio)
#         self.labelFunctionRes = qt.QLabel()
#
#         if self.xmlElement.tag == "Image":
#             # geometry type
#             self.labelGeometry = qt.QLabel("Geometry")
#             self.labelGeometry.setFixedWidth(100 * vrb.ratio)
#             self.labelGeometryRes = qt.QLabel()
#             # color type
#             self.labelColor = qt.QLabel("Color")
#             self.labelColor.setFixedWidth(100 * vrb.ratio)
#             self.labelColorRes = qt.QLabel()
#             # buffer type
#             self.labelBuffer = qt.QLabel("Buffer")
#             self.labelBuffer.setFixedWidth(100 * vrb.ratio)
#             self.labelBufferRes = qt.QLabel()
#             # type between memory image or disk image
#             self.labelType = qt.QLabel("Type")
#             self.labelType.setFixedWidth(100 * vrb.ratio)
#             self.labelTypeRes = qt.QLabel()
#
#         self.layoutGeneral = qt.QGridLayout()
#
#         self.layoutGeneral.addWidget(self.labelName, 0, 0)
#         self.layoutGeneral.addWidget(self.labelNameRes, 0, 1)
#         self.layoutGeneral.addWidget(self.labelTime, 1, 0)
#         self.layoutGeneral.addWidget(self.labelTimeRes, 1, 1)
#         self.layoutGeneral.addWidget(self.labelFunction, 2, 0)
#         self.layoutGeneral.addWidget(self.labelFunctionRes, 2, 1)
#         if self.xmlElement.tag == "Image":
#             self.layoutGeneral.addWidget(self.labelGeometry, 3, 0)
#             self.layoutGeneral.addWidget(self.labelGeometryRes, 3, 1)
#             self.layoutGeneral.addWidget(self.labelColor, 4, 0)
#             self.layoutGeneral.addWidget(self.labelColorRes, 4, 1)
#             self.layoutGeneral.addWidget(self.labelBuffer, 5, 0)
#             self.layoutGeneral.addWidget(self.labelBufferRes, 5, 1)
#             self.layoutGeneral.addWidget(self.labelType, 6, 0)
#             self.layoutGeneral.addWidget(self.labelTypeRes, 6, 1)
#
#         self.layoutGeneral.setSizeConstraint(1)
#         self.layoutGeneral.setContentsMargins(10*vrb.ratio,10*vrb.ratio,10*vrb.ratio,10*vrb.ratio)
#         self.layoutGeneral.setVerticalSpacing(10*vrb.ratio)
#
#         self.groupBoxGeneral.setLayout(self.layoutGeneral)
#
#         # tab widget with 2 tabs for General and multi channels
#         self.tabwidget = qt.QTabWidget()
#
#         self.tabwidget.addTab(self.groupBoxGeneral, "Generals")
#
#         self.layout = qt.QGridLayout()
#         self.layout.addWidget(self.tabwidget, 0, 0)
#         self.setLayout(self.layout)
#
#         self.setMinimumHeight(150*vrb.ratio)
#         self.setMaximumHeight(250*vrb.ratio)
#
#         style = fct.getStyleSheet()
#         self.setStyleSheet(style)
#
#         self.setWindowTitle("Properties")

class PropertiesWindow(qt.QWidget):
    def __init__(self, xmlElement):
        qt.QWidget.__init__(self)

        self.xmlElement = xmlElement

        # groupbox for general properties
        self.groupBoxGeneral = qt.QGroupBox("General Properties")
        self.groupBoxGeneral.setStyleSheet(
            f"""
            QGroupBox {{
                border: 2px solid #0673BA;
                border-radius: 5px;
                margin-top: 15px;
                font-size: {12*vrb.ratio}px;
                color: #0673BA;
            }}
            QGroupBox::title {{
                subcontrol-origin: margin;
                subcontrol-position: top center;
                padding: 0 3px;
            }}
            """
        )

        # Labels and their responses
        self.labelName = qt.QLabel("Name:")
        self.labelName.setFixedWidth(100 * vrb.ratio)
        self.labelName.setStyleSheet("font-weight: bold;")
        self.labelNameRes = qt.QLabel()

        self.labelTime = qt.QLabel("Processing time:")
        self.labelTime.setFixedWidth(100 * vrb.ratio)
        self.labelTime.setStyleSheet("font-weight: bold;")
        self.labelTimeRes = qt.QLabel()

        self.labelFunction = qt.QLabel("Function:")
        self.labelFunction.setFixedWidth(100 * vrb.ratio)
        self.labelFunction.setStyleSheet("font-weight: bold;")
        self.labelFunctionRes = qt.QLabel()

        self.layoutGeneral = qt.QGridLayout()

        self.layoutGeneral.addWidget(self.labelName, 0, 0)
        self.layoutGeneral.addWidget(self.labelNameRes, 0, 1)
        self.layoutGeneral.addWidget(self.labelTime, 1, 0)
        self.layoutGeneral.addWidget(self.labelTimeRes, 1, 1)
        self.layoutGeneral.addWidget(self.labelFunction, 2, 0)
        self.layoutGeneral.addWidget(self.labelFunctionRes, 2, 1)

        if self.xmlElement.tag == "Image":
            self.labelGeometry = qt.QLabel("Geometry:")
            self.labelGeometry.setFixedWidth(100 * vrb.ratio)
            self.labelGeometry.setStyleSheet("font-weight: bold;")
            self.labelGeometryRes = qt.QLabel()

            self.labelColor = qt.QLabel("Color:")
            self.labelColor.setFixedWidth(100 * vrb.ratio)
            self.labelColor.setStyleSheet("font-weight: bold;")
            self.labelColorRes = qt.QLabel()

            self.labelBuffer = qt.QLabel("Buffer:")
            self.labelBuffer.setFixedWidth(100 * vrb.ratio)
            self.labelBuffer.setStyleSheet("font-weight: bold;")
            self.labelBufferRes = qt.QLabel()

            self.labelType = qt.QLabel("Type:")
            self.labelType.setFixedWidth(100 * vrb.ratio)
            self.labelType.setStyleSheet("font-weight: bold;")
            self.labelTypeRes = qt.QLabel()

            self.labelOrigin = qt.QLabel("Origin:")
            self.labelOrigin.setFixedWidth(100 * vrb.ratio)
            self.labelOrigin.setStyleSheet("font-weight: bold;")
            self.labelOriginRes = qt.QLabel()

            self.layoutGeneral.addWidget(self.labelGeometry, 3, 0)
            self.layoutGeneral.addWidget(self.labelGeometryRes, 3, 1)
            self.layoutGeneral.addWidget(self.labelColor, 4, 0)
            self.layoutGeneral.addWidget(self.labelColorRes, 4, 1)
            self.layoutGeneral.addWidget(self.labelBuffer, 5, 0)
            self.layoutGeneral.addWidget(self.labelBufferRes, 5, 1)
            self.layoutGeneral.addWidget(self.labelType, 6, 0)
            self.layoutGeneral.addWidget(self.labelTypeRes, 6, 1)
            self.layoutGeneral.addWidget(self.labelOrigin, 7, 0)
            self.layoutGeneral.addWidget(self.labelOriginRes, 7, 1)

        self.layoutGeneral.setSizeConstraint(1)
        self.layoutGeneral.setContentsMargins(15 * vrb.ratio, 15 * vrb.ratio, 15 * vrb.ratio, 15 * vrb.ratio)
        self.layoutGeneral.setVerticalSpacing(10 * vrb.ratio)

        self.groupBoxGeneral.setLayout(self.layoutGeneral)

        # tab widget with 2 tabs for General and multi channels
        self.tabwidget = qt.QTabWidget()
        self.tabwidget.addTab(self.groupBoxGeneral, "General")

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.tabwidget, 0, 0)
        self.setLayout(self.layout)

        self.setMinimumHeight(200 * vrb.ratio)
        self.setMaximumHeight(400 * vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.setWindowTitle("Properties")

    def displayProperties(self):

        try:

            self.labelNameRes.setText(Dfct.childText(self.xmlElement, 'Name'))
            try:
                execTime = float(Dfct.childText(Dfct.SubElement(self.xmlElement, 'FunctionCall'), 'ExecutionTime'))
            except:
                execTime = 0
            self.labelTimeRes.setText(str(fct.numberCalibration(execTime)) + 's')
            functionCallXml = Dfct.SubElement(self.xmlElement, 'FunctionCall')
            nameFunctionCall = Dfct.childText(functionCallXml, 'Name')
            if nameFunctionCall is not None:
                self.labelFunctionRes.setText(nameFunctionCall)

            if self.xmlElement.tag == "Image":

                imageID = Dfct.childText(self.xmlElement,"Position")

                self.image = vrb.mainWindow.widgetLabelImage.layout.itemAt(int(imageID)).widget().image

                self.groupBoxMultiSpectral = GroupBoxMultiSpectralEditor(image=self.image)

                sizeX = Dfct.childText(self.xmlElement,"X")
                sizeY = Dfct.childText(self.xmlElement,"Y")
                sizeZ = Dfct.childText(self.xmlElement,"Z")
                sizeC = Dfct.childText(self.xmlElement,"C")
                sizeT = Dfct.childText(self.xmlElement,"T")
                bufferType = Dfct.childText(self.xmlElement,"BufferType")
                diskImage = Dfct.childText(self.xmlElement,"DiskImage")
                originX = Dfct.childText(self.xmlElement,"OriginX")
                originY = Dfct.childText(self.xmlElement,"OriginY")
                originZ = Dfct.childText(self.xmlElement,"OriginZ")

                # if self.image.hasGeometricCalibration():
                #     calibration = self.image.getGeometricCalibration()
                # else:
                #     if self.image.getVolumeGeometryType() == PyIPSDK.eVGT_2d:
                #         calibration = PyIPSDK.createGeometricCalibration2d(1, 1, "px")
                #     else:
                #         calibration = PyIPSDK.createGeometricCalibration3d(1, 1, 1, "px")
                #
                # originX = calibration.getXOriginUncalibrated()
                # originY = calibration.getYOriginUncalibrated()
                # originZ = calibration.getZOriginUncalibrated()

                if sizeT == "1":
                    if sizeZ == "1":
                        self.labelGeometryRes.setText("2D ("+str(sizeX) + " x "+str(sizeY) + ")")
                        self.labelOriginRes.setText("[" + originX + "," + originY + "]")
                    else:
                        self.labelGeometryRes.setText("3D ("+str(sizeX) + " x "+str(sizeY) + " x " +str(sizeZ) + ")")
                        self.labelOriginRes.setText("[" + originX + "," + originY + "," + originZ + "]")
                else:
                    if sizeZ == "1":
                        self.labelGeometryRes.setText("2D sequence ("+str(sizeX) + " x "+str(sizeY) + " x "+str(sizeT) + ")")
                    else:
                        self.labelGeometryRes.setText("3D sequence ("+str(sizeX) + " x "+str(sizeY) + " x " +str(sizeZ) + " x " +str(sizeT) + ")")
                if sizeC == "1":
                    self.labelColorRes.setText("Grey")
                    self.tabwidget.setTabEnabled(1, False)
                elif sizeC == "3":
                    self.labelColorRes.setText("RGB")
                    if self.tabwidget.count() < 2:
                        self.tabwidget.addTab(self.groupBoxMultiSpectral, "Multi-channels")
                    self.tabwidget.setTabEnabled(1, True)
                else:
                    self.labelColorRes.setText("Multi channels ("+str(sizeC)+" channels)")
                    if self.tabwidget.count() < 2:
                        self.tabwidget.addTab(self.groupBoxMultiSpectral, "Multi-channels")
                    self.tabwidget.setTabEnabled(1, True)
                self.labelBufferRes.setText(fct.convertBufferText(bufferType))
                if diskImage == "True":
                    self.labelTypeRes.setText("Disk image")
                else:
                    self.labelTypeRes.setText("Memory image")



            fct.showWidget(self)

        except:
            traceback.print_exc(file=sys.stderr)

# class PropertiesWindow(qt.QWidget):
#     def __init__(self, xmlElement):
#         qt.QWidget.__init__(self)
#
#         self.xmlElement = xmlElement
#         # self.ipsdkElement = ipsdkElement
#
#         self.layout = qt.QGridLayout()
#
#         self.labelName = qt.QLabel("Name")
#         self.labelName.setFixedWidth(100*vrb.ratio)
#         self.labelNameRes = qt.QLabel()
#         self.labelTime = qt.QLabel("Processing time")
#         self.labelTime.setFixedWidth(100*vrb.ratio)
#         self.labelTimeRes = qt.QLabel()
#         self.labelFunction = qt.QLabel("Function")
#         self.labelFunction.setFixedWidth(100*vrb.ratio)
#         self.labelFunctionRes = qt.QLabel()
#
#         if self.xmlElement.tag == "Image":
#             self.labelGeometry = qt.QLabel("Geometry")
#             self.labelGeometry.setFixedWidth(100 * vrb.ratio)
#             self.labelGeometryRes = qt.QLabel()
#             self.labelColor = qt.QLabel("Color")
#             self.labelColor.setFixedWidth(100 * vrb.ratio)
#             self.labelColorRes = qt.QLabel()
#             self.labelBuffer = qt.QLabel("Buffer")
#             self.labelBuffer.setFixedWidth(100 * vrb.ratio)
#             self.labelBufferRes = qt.QLabel()
#             self.labelType = qt.QLabel("Type")
#             self.labelType.setFixedWidth(100 * vrb.ratio)
#             self.labelTypeRes = qt.QLabel()
#
#         self.layout.addWidget(self.labelName, 0, 0)
#         self.layout.addWidget(self.labelNameRes, 0, 1)
#         self.layout.addWidget(self.labelTime, 1, 0)
#         self.layout.addWidget(self.labelTimeRes, 1, 1)
#         self.layout.addWidget(self.labelFunction, 2, 0)
#         self.layout.addWidget(self.labelFunctionRes, 2, 1)
#         if self.xmlElement.tag == "Image":
#             self.layout.addWidget(self.labelGeometry, 3, 0)
#             self.layout.addWidget(self.labelGeometryRes, 3, 1)
#             self.layout.addWidget(self.labelColor, 4, 0)
#             self.layout.addWidget(self.labelColorRes, 4, 1)
#             self.layout.addWidget(self.labelBuffer, 5, 0)
#             self.layout.addWidget(self.labelBufferRes, 5, 1)
#             self.layout.addWidget(self.labelType, 6, 0)
#             self.layout.addWidget(self.labelTypeRes, 6, 1)
#         self.setLayout(self.layout)
#
#         self.layout.setSizeConstraint(1)
#         self.layout.setContentsMargins(10*vrb.ratio,10*vrb.ratio,10*vrb.ratio,10*vrb.ratio)
#         self.layout.setVerticalSpacing(10*vrb.ratio)
#
#         style = fct.getStyleSheet()
#         self.setStyleSheet(style)
#
#         self.setWindowTitle("Properties")
#
#     def displayProperties(self):
#
#         self.labelNameRes.setText(Dfct.childText(self.xmlElement, 'Name'))
#         try:
#             execTime = float(Dfct.childText(Dfct.SubElement(self.xmlElement, 'FunctionCall'), 'ExecutionTime'))
#         except:
#             execTime = 0
#         self.labelTimeRes.setText(str(fct.numberCalibration(execTime)) + 's')
#         functionCallXml = Dfct.SubElement(self.xmlElement, 'FunctionCall')
#         nameFunctionCall = Dfct.childText(functionCallXml, 'Name')
#         if nameFunctionCall is not None:
#             self.labelFunctionRes.setText(nameFunctionCall)
#
#         if self.xmlElement.tag == "Image":
#
#             sizeX = Dfct.childText(self.xmlElement,"X")
#             sizeY = Dfct.childText(self.xmlElement,"Y")
#             sizeZ = Dfct.childText(self.xmlElement,"Z")
#             sizeC = Dfct.childText(self.xmlElement,"C")
#             sizeT = Dfct.childText(self.xmlElement,"T")
#             bufferType = Dfct.childText(self.xmlElement,"BufferType")
#             diskImage = Dfct.childText(self.xmlElement,"DiskImage")
#
#             if sizeT == "1":
#                 if sizeZ == "1":
#                     self.labelGeometryRes.setText("2D ("+str(sizeX) + " x "+str(sizeY) + ")")
#                 else:
#                     self.labelGeometryRes.setText("3D ("+str(sizeX) + " x "+str(sizeY) + " x " +str(sizeZ) + ")")
#             else:
#                 if sizeZ == "1":
#                     self.labelGeometryRes.setText("2D sequence ("+str(sizeX) + " x "+str(sizeY) + " x "+str(sizeT) + ")")
#                 else:
#                     self.labelGeometryRes.setText("3D sequence ("+str(sizeX) + " x "+str(sizeY) + " x " +str(sizeZ) + " x " +str(sizeT) + ")")
#             if sizeC == "1":
#                 self.labelColorRes.setText("Grey")
#             elif sizeC == "3":
#                 self.labelColorRes.setText("RGB")
#             else:
#                 self.labelColorRes.setText("Multi channels ("+str(sizeC)+" channels)")
#             self.labelBufferRes.setText(fct.convertBufferText(bufferType))
#             if diskImage == "True":
#                 self.labelTypeRes.setText("Disk image")
#             else:
#                 self.labelTypeRes.setText("Memory image")
#
#         fct.showWidget(self)

class GroupBoxProcessing(qt.QGroupBox):
    def __init__(self):
        super(GroupBoxProcessing, self).__init__()

        self.innerGroupBox = qt.QGroupBox()

        self.labelLeft = qt.QLabel('Please wait...')
        self.labelRight = LabelImage(vrb.folderImages + '/sablier_blanc.png')
        # self.buttonStop = qt.QPushButton("Stop")
        self.buttonStop = PushButtonImage(vrb.folderImages + "/Delete.png", margins=2)
        self.buttonStop.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        self.buttonStop.setVisible(False)
        self.progressBar = qt.QProgressBar()
        self.progressBar.setAlignment(Qt.AlignCenter)
        self.progressBar.setRange(0, 100)
        self.progressBar.setVisible(False)

        self.layoutInner = qt.QGridLayout()
        self.layoutInner.addWidget(self.labelLeft, 0, 0)
        self.layoutInner.addWidget(self.labelRight, 0, 1)
        self.layoutInner.addWidget(self.buttonStop, 0, 2)
        self.layoutInner.addWidget(self.progressBar, 1, 0 ,1,3)
        self.innerGroupBox.setLayout(self.layoutInner)

        self.layoutOuter = qt.QGridLayout()
        self.layoutOuter.addWidget(self.innerGroupBox, 0, 0, Qt.AlignCenter)
        self.setLayout(self.layoutOuter)

        # self.setStyleSheet('GroupBoxProcessing {border: 3px solid rgb(255, 0, 0); }')
        self.innerGroupBox.setStyleSheet('QGroupBox {border: 3px solid rgb(0, 0, 220); background: black;}')
        self.labelLeft.setStyleSheet('QLabel {color:rgb(255, 255, 255)}')
        self.buttonStop.setStyleSheet('QPushButton {border: 0px; background: black;}')

        #self.innerGroupBox.setFixedSize(160, 60)
        self.labelLeft.setFixedSize(150*vrb.ratio, 55*vrb.ratio)
        self.labelRight.setFixedSize(45*vrb.ratio, 55*vrb.ratio)

    def setText(self, text):
        self.labelLeft.setText(text)

class GroupBoxSelectionDimension(qt.QGroupBox):
    def __init__(self):
        super(GroupBoxSelectionDimension, self).__init__()

        self.radioButton2D = qt.QRadioButton("2D")
        self.radioButton3D = qt.QRadioButton("3D")
        self.radioButtonBoth = qt.QRadioButton("2D/3D")

        self.buttonGroup = qt.QButtonGroup()
        self.buttonGroup.addButton(self.radioButton2D, 0)
        self.buttonGroup.addButton(self.radioButton3D, 1)
        self.buttonGroup.addButton(self.radioButtonBoth, 2)

        self.layout = qt.QGridLayout()
        self.layout.setAlignment(Qt.AlignLeft)
        self.layout.addWidget(self.radioButton2D, 0, 0)
        self.layout.addWidget(self.radioButton3D, 0, 1)
        self.layout.addWidget(self.radioButtonBoth, 0, 2)
        self.setLayout(self.layout)

        self.radioButtonBoth.setChecked(True)


class MessageBox(qt.QMessageBox):
    def __init__(self, title, description, buttons=(qt.QMessageBox.Close, qt.QMessageBox.No), defaultButton=qt.QMessageBox.No, icon=qt.QMessageBox.NoIcon,windowTitle = "Explorer"):
        # https://doc.qt.io/qt-5/qmessagebox.html
        super(MessageBox, self).__init__()

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setText(title)
        self.setInformativeText(description)
        standardButtons = buttons[0]
        for button in buttons:
            standardButtons |= button
        self.setStandardButtons(standardButtons)
        if defaultButton in buttons:
            self.setDefaultButton(defaultButton)

        self.setIcon(icon)
        self.setWindowTitle(windowTitle)
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

class WidgetDeleleAll(qt.QWidget):
    def __init__(self,widgetLabelImage = None):
        qt.QWidget.__init__(self)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        try:
            self.setWindowModality(Qt.ApplicationModal)
        except:
            pass

        self.widgetLabelImage = widgetLabelImage

        self.label = qt.QLabel("Do you want to delete those images ?")

        self.groupCheckBox = qt.QGroupBox()
        self.layoutCheckBox = qt.QVBoxLayout()
        self.groupCheckBox.setLayout(self.layoutCheckBox)

        self.buttonYes = qt.QPushButton("Yes")
        self.buttonYes.setFixedSize(40*vrb.ratio,20*vrb.ratio)
        self.buttonNo = qt.QPushButton("No")
        self.buttonNo.setFixedSize(40*vrb.ratio,20*vrb.ratio)
        self.groupBoxButtons = qt.QGroupBox()
        self.layoutButtons = qt.QGridLayout()
        self.groupBoxButtons.setLayout(self.layoutButtons)
        self.layoutButtons.addWidget(self.buttonYes,0,0)
        self.layoutButtons.addWidget(self.buttonNo,0,1)
        self.layoutButtons.setAlignment(Qt.AlignLeft)
        self.groupBoxButtons.setStyleSheet("QGroupBox {border: 0px transparent; }")

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.label, 0, 0,Qt.AlignTop)
        self.layout.addWidget(self.groupCheckBox, 1, 0,Qt.AlignCenter)
        self.layout.addWidget(self.groupBoxButtons, 2, 0, Qt.AlignBottom)
        self.setLayout(self.layout)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonNo.clicked.connect(self.cancel)

    def showWidget(self):

        if self.widgetLabelImage.layout.count() >= 1:
            while self.layoutCheckBox.count() != 0:
                item = self.layoutCheckBox.itemAt(0)
                if item is not None:
                    item.widget().deleteLater()
                    self.layoutCheckBox.removeItem(item)

            for i in range(self.widgetLabelImage.layout.count()):
                item = self.widgetLabelImage.layout.itemAt(i)
                if item is not None:
                    widget = item.widget()
                    name = Dfct.childText(widget.xmlElement, "Name")
                    checkBox = qt.QCheckBox(name)
                    checkBox.setChecked(True)
                    checkBox.widget = widget
                    self.layoutCheckBox.addWidget(checkBox)

            self.show()

    def cancel(self):

        self.close()

class FileSelector(qt.QGroupBox):
    def __init__(self,widgetLabelImage = None,name = "File"):
        qt.QGroupBox.__init__(self)

        self.labelFile = qt.QLabel(name)
        self.lineEditFile = qt.QLineEdit()
        self.buttonOpen = PushButtonImage(vrb.folderImages + "/Folder.png")
        self.buttonOpen.setFixedSize(20*vrb.ratio, 20*vrb.ratio)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelFile,0,0)
        self.layout.addWidget(self.lineEditFile,0,1)
        self.layout.addWidget(self.buttonOpen,0,2)
        self.setLayout(self.layout)

        self.layout.setContentsMargins(0,0,0,0)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonOpen.clicked.connect(self.openQDialog)

    def openQDialog(self):

        defaultFolder = Dfct.childText(vrb.userPathElement, "ImportImages")
        if defaultFolder is None or defaultFolder == "" or os.path.exists(defaultFolder) == False:
            defaultFolder = os.path.dirname(vrb.folderExplorer) + "/images"
            if not os.path.exists(defaultFolder):
                defaultFolder = os.path.dirname(vrb.folderExplorer)+"/data/Explorer/images"

        filename = qt.QFileDialog.getOpenFileName(self, "Select your image", defaultFolder)

        if filename[0] != [] and filename[0] != '' and filename[0] != None:
            self.lineEditFile.setText(filename[0])
            Dfct.SubElement(vrb.userPathElement, "ImportImages").text = os.path.dirname(filename[0])
            Dfct.saveXmlElement(vrb.userPathElement, vrb.folderInformation + "/UserPath.mho")

class FolderSelector(qt.QGroupBox):
    def __init__(self,name = "File",optional =False):
        qt.QGroupBox.__init__(self)

        self.checkBox = qt.QCheckBox()
        self.checkBox.setChecked(False)
        self.labelFile = qt.QLabel(name)
        self.lineEditFile = qt.QLineEdit()
        self.buttonOpen = PushButtonImage(vrb.folderImages + "/Folder.png")
        self.buttonOpen.setFixedSize(20*vrb.ratio, 20*vrb.ratio)

        self.layout = qt.QGridLayout()
        if optional:
            self.layout.addWidget(self.checkBox,0,0)
            self.layout.addWidget(self.labelFile, 0, 1)
            self.layout.addWidget(self.lineEditFile, 0, 2)
            self.layout.addWidget(self.buttonOpen, 0, 3)
        else:
            self.layout.addWidget(self.labelFile,0,0)
            self.layout.addWidget(self.lineEditFile,0,1)
            self.layout.addWidget(self.buttonOpen,0,2)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(0,0,0,0)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonOpen.clicked.connect(self.openQDialog)
        self.checkBox.stateChanged.connect(self.checkBoxChanged)
        if optional:
            self.checkBoxChanged()

    def openQDialog(self):

        defaultFolder = Dfct.childText(vrb.userPathElement, "ImportFolder")
        if defaultFolder is None or defaultFolder == "" or os.path.exists(defaultFolder) == False:
            defaultFolder = "C:/"

        filename = qt.QFileDialog.getExistingDirectory(self, "Select your folder", defaultFolder)

        if filename != "" and filename != None:
            self.lineEditFile.setText(filename)
            Dfct.SubElement(vrb.userPathElement, "ImportFolder").text = os.path.dirname(filename)
            Dfct.saveXmlElement(vrb.userPathElement, vrb.folderInformation + "/UserPath.mho")

    def checkBoxChanged(self):

        self.lineEditFile.setEnabled(self.checkBox.isChecked())
        self.buttonOpen.setEnabled(self.checkBox.isChecked())

class GroupBoxValidate(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.buttonValidate = PushButtonImage(vrb.folderImages + "/Validate.png", margins=2)
        self.buttonValidate.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonPreview = PushButtonImage(vrb.folderImages + "/eye_open.png", margins=2)
        self.buttonPreview.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        # self.buttonCancel = qt.QPushButton("Cancel")
        # self.buttonCancel.setFixedSize(60*vrb.ratio,20*vrb.ratio)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.buttonValidate, 0, 0)
        self.layout.addWidget(self.buttonPreview, 0, 1)
        # self.layout.addWidget(self.buttonCancel, 0, 2)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3, 0, 3, 0)
        self.layout.setAlignment(Qt.AlignLeft)
        # self.layout.setHorizontalSpacing(3)

        self.setStyleSheet("QGroupBox {border: 0px transparent; }")

        self.setFixedSize(180*vrb.ratio + 2*3*vrb.ratio + 8*vrb.ratio,36*vrb.ratio)

# class RawWidget(qt.QWidget):
#     def __init__(self,mainWindow = None):
#         qt.QWidget.__init__(self)
#
#         from ImageViewer import WidgetImage
#
#         self.mainWindow = mainWindow
#
#         try:
#             self.setWindowFlag(Qt.WindowStaysOnTopHint)
#         except:
#             self.setWindowFlags(Qt.WindowStaysOnTopHint)
#         try:
#             self.setWindowModality(Qt.ApplicationModal)
#         except:
#             pass
#
#         self.previewViewer = WidgetImage(standAlone=True)
#         self.previewViewer.resize(500*vrb.ratio,500*vrb.ratio)
#         try:
#             self.previewViewer.setWindowFlag(Qt.WindowStaysOnTopHint)
#         except:
#             self.previewViewer.setWindowFlags(Qt.WindowStaysOnTopHint)
#         try:
#             self.previewViewer.setWindowModality(Qt.ApplicationModal)
#         except:
#             pass
#
#         self.widgetDialog = FileSelector()
#
#         self.labelTitle = qt.QLabel("Define the image geometry")
#         self.labelWidth = qt.QLabel("Width (X) : ")
#         self.lineEditWidth = NumberLineEdit()
#         self.lineEditWidth.setFixedWidth(60*vrb.ratio)
#         self.labelHeight = qt.QLabel("Height (Y) : ")
#         self.lineEditHeight = NumberLineEdit()
#         self.lineEditHeight.setFixedWidth(60*vrb.ratio)
#         self.labelDepth = qt.QLabel("Depth (Z) : ")
#         self.lineEditDepth = NumberLineEdit()
#         self.lineEditDepth.setFixedWidth(60*vrb.ratio)
#         self.lineEditDepth.setPlaceholderText("3D Image only")
#         self.labelColor = qt.QLabel("Color (C) : ")
#         self.lineEditColor = NumberLineEdit()
#         self.lineEditColor.setFixedWidth(60*vrb.ratio)
#         self.lineEditColor.setPlaceholderText("Color Image only")
#         self.labelSeq = qt.QLabel("Sequence (T) : ")
#         self.lineEditSeq = NumberLineEdit()
#         self.lineEditSeq.setFixedWidth(60*vrb.ratio)
#         self.lineEditSeq.setPlaceholderText("Sequence Image only")
#         self.labelOffset = qt.QLabel("Offset : ")
#         self.lineEditOffset = NumberLineEdit()
#         self.lineEditOffset.setFixedWidth(60*vrb.ratio)
#         self.lineEditOffset.setPlaceholderText("0")
#
#         self.labelType = qt.QLabel("Type : ")
#         self.comboBoxBuffer = qt.QComboBox()
#         self.comboBoxBuffer.addItem('Int 8',"PyIPSDK.eImageBufferType.eIBT_Int8")
#         self.comboBoxBuffer.addItem('UInt 8',"PyIPSDK.eImageBufferType.eIBT_UInt8")
#         self.comboBoxBuffer.addItem('Int 16',"PyIPSDK.eImageBufferType.eIBT_Int16")
#         self.comboBoxBuffer.addItem('UInt 16',"PyIPSDK.eImageBufferType.eIBT_UInt16")
#         self.comboBoxBuffer.addItem('Int 32',"PyIPSDK.eImageBufferType.eIBT_Int32")
#         self.comboBoxBuffer.addItem('UInt 32',"PyIPSDK.eImageBufferType.eIBT_UInt32")
#         self.comboBoxBuffer.addItem('Real 32',"PyIPSDK.eImageBufferType.eIBT_Real32")
#         self.comboBoxBuffer.addItem('Binary',"PyIPSDK.eImageBufferType.eIBT_Binary")
#         self.comboBoxBuffer.addItem('Label 8',"PyIPSDK.eImageBufferType.eIBT_Label8")
#         self.comboBoxBuffer.addItem('Label 16',"PyIPSDK.eImageBufferType.eIBT_Label16")
#         self.comboBoxBuffer.addItem('Label 32',"PyIPSDK.eImageBufferType.eIBT_Label32")
#         self.comboBoxBuffer.setCurrentIndex(1)
#         self.comboBoxBuffer.setFixedWidth(60*vrb.ratio)
#
#         self.labelDiskImage = qt.QLabel("Open as a disk image")
#         self.checkBoxDiskImage = qt.QCheckBox()
#
#         self.groupValidate = GroupBoxValidate()
#         self.groupValidate.buttonValidate.setEnabled(False)
#
#         self.layout = qt.QGridLayout()
#         self.layout.addWidget(self.widgetDialog,0,0,1,2)
#         self.layout.addWidget(self.labelTitle,1,0,1,2)
#         self.layout.addWidget(self.labelWidth,2,0,1,1)
#         self.layout.addWidget(self.lineEditWidth,2,1,1,1,Qt.AlignLeft)
#         self.layout.addWidget(self.labelHeight,3,0,1,1)
#         self.layout.addWidget(self.lineEditHeight,3,1,1,1,Qt.AlignLeft)
#         self.layout.addWidget(self.labelDepth,4,0,1,1)
#         self.layout.addWidget(self.lineEditDepth,4,1,1,1,Qt.AlignLeft)
#         self.layout.addWidget(self.labelColor,5,0,1,1)
#         self.layout.addWidget(self.lineEditColor,5,1,1,1,Qt.AlignLeft)
#         self.layout.addWidget(self.labelSeq,6,0,1,1)
#         self.layout.addWidget(self.lineEditSeq,6,1,1,1,Qt.AlignLeft)
#         self.layout.addWidget(self.labelOffset, 7, 0, 1, 1)
#         self.layout.addWidget(self.lineEditOffset, 7, 1, 1, 1,Qt.AlignLeft)
#         self.layout.addWidget(self.labelType,8,0,1,1)
#         self.layout.addWidget(self.comboBoxBuffer,8,1,1,1,Qt.AlignLeft)
#         self.layout.addWidget(self.labelDiskImage,9,0,1,1,Qt.AlignLeft)
#         self.layout.addWidget(self.checkBoxDiskImage,9,1,1,1,Qt.AlignLeft)
#         self.layout.addWidget(self.groupValidate,10,0,1,2,Qt.AlignBottom)
#         self.setLayout(self.layout)
#
#         style = fct.getStyleSheet()
#         self.setStyleSheet(style)
#         self.setWindowTitle("Open raw image")
#
#         self.setFixedSize(230*vrb.ratio,300*vrb.ratio)
#
#         self.widgetDialog.lineEditFile.textChanged.connect(self.verifyImageGeometry)
#         self.lineEditWidth.textChanged.connect(self.verifyImageGeometry)
#         self.lineEditHeight.textChanged.connect(self.verifyImageGeometry)
#         self.lineEditDepth.textChanged.connect(self.verifyImageGeometry)
#         self.lineEditColor.textChanged.connect(self.verifyImageGeometry)
#         self.lineEditSeq.textChanged.connect(self.verifyImageGeometry)
#         self.lineEditOffset.textChanged.connect(self.verifyImageGeometry)
#         self.comboBoxBuffer.currentIndexChanged.connect(self.verifyImageGeometry)
#
#         self.groupValidate.buttonValidate.clicked.connect(self.validateImage)
#         self.groupValidate.buttonPreview.clicked.connect(self.previewImage)
#         # self.groupValidate.buttonCancel.clicked.connect(self.cancelFunction)
#
#     def verifyImageGeometry(self):
#
#         try:
#             numberByte = int(self.lineEditWidth.text())*int(self.lineEditHeight.text())
#             if self.lineEditDepth.text() != "":
#                 numberByte *= int(self.lineEditDepth.text())
#             if self.lineEditColor.text() != "":
#                 numberByte *= int(self.lineEditColor.text())
#             if self.lineEditSeq.text() != "":
#                 numberByte *= int(self.lineEditSeq.text())
#             if self.comboBoxBuffer.currentText() in ['Int 16','UInt 16','Label 16']:
#                 numberByte*=2
#             elif self.comboBoxBuffer.currentText() in ['Int 32','UInt 32','Real 32','Label 32']:
#                 numberByte*=4
#             if self.lineEditOffset.text() != "":
#                 offset = int(self.lineEditOffset.text())
#             else:
#                 offset = 0
#             numberByte+=offset
#             filename = self.widgetDialog.lineEditFile.text()
#             statinfo = os.stat(filename)
#             size = statinfo.st_size
#             if numberByte <= size:
#                 self.groupValidate.buttonValidate.setEnabled(True)
#             else:
#                 self.groupValidate.buttonValidate.setEnabled(False)
#         except:
#             self.groupValidate.buttonValidate.setEnabled(False)
#
#     def previewImage(self):
#
#         from LutsWidget import GroupBoxLut
#
#         self.previewViewer.close()
#
#         try:
#             sizeX = int(self.lineEditWidth.text())
#             sizeY = int(self.lineEditHeight.text())
#             buffer = self.comboBoxBuffer.currentData()
#             if self.lineEditOffset.text() != "":
#                 offset = self.lineEditOffset.text() + "\n"
#             else:
#                 offset = "0\n"
#
#             colorGeometry = "PyIPSDK.geometryGrey()"
#             if self.lineEditColor.text() != "":
#                 colorGeometry = "PyIPSDK.geometryUser("+self.lineEditColor.text()+")"
#             geometry = "PyIPSDK.geometry(" + buffer + "," + str(sizeX) + "," + str(sizeY) + ",PyIPSDK.geometry2d()," + colorGeometry + ",PyIPSDK.geometrySingle())\n"
#
#             filename = self.widgetDialog.lineEditFile.text()
#
#             text =""
#             text += "geometry = " + geometry
#             text += "offset = " + offset
#             text += "image = PyIPSDK.loadRawImageFile('" + filename + "',geometry,offset)\n"
#             _locals = locals()
#             exec(text, globals(), _locals)
#             image = _locals['image']
#
#             image.valueMin, image.valueMax = fct.getMinMaxValue(image)
#             image.valueMinMask = None
#             if image.getSizeC() == 1:
#                 image.groupBoxLut = GroupBoxLut(labelImage=image, parent=image)
#             self.previewViewer.setImage(image)
#             self.previewViewer.setWindowTitle(os.path.basename(filename))
#             self.previewViewer.show()
#         except:
#             traceback.print_exc(file=sys.stderr)
#             pass
#
#     def validateImage(self):
#
#         try:
#             sizeX = int(self.lineEditWidth.text())
#             sizeY = int(self.lineEditHeight.text())
#             buffer = self.comboBoxBuffer.currentData()
#             if self.lineEditOffset.text() != "":
#                 offset = self.lineEditOffset.text() + "\n"
#             else:
#                 offset = "0\n"
#
#             volumeGeometry = "PyIPSDK.geometry2d()"
#             if self.lineEditDepth.text() != "":
#                 volumeGeometry = "PyIPSDK.geometry3d("+self.lineEditDepth.text()+")"
#
#             colorGeometry = "PyIPSDK.geometryGrey()"
#             if self.lineEditColor.text() != "":
#                 colorGeometry = "PyIPSDK.geometryUser("+self.lineEditColor.text()+")"
#
#             temporalGeometry = "PyIPSDK.geometrySingle()"
#             if self.lineEditSeq.text() != "":
#                 temporalGeometry = "PyIPSDK.geometrySequence("+self.lineEditSeq.text()+")"
#
#             geometry = "PyIPSDK.geometry(" + buffer + "," + str(sizeX) + "," + str(sizeY) + "," + volumeGeometry + "," + colorGeometry + "," + temporalGeometry + ")\n"
#             filename = self.widgetDialog.lineEditFile.text()
#
#             self.close()
#             self.mainWindow.groupBoxProcessing.setText('Reading raw image ' + os.path.basename(filename))
#             self.mainWindow.toggleGroupBoxProcessing(True)
#
#             if self.mainWindow is not None:
#                 label,image = self.mainWindow.urlToImage(filename,geometry=geometry,offset=offset,disk=self.checkBoxDiskImage.isChecked())
#
#             self.mainWindow.toggleGroupBoxProcessing(False)
#
#         except:
#             traceback.print_exc(file=sys.stderr)
#             self.mainWindow.toggleGroupBoxProcessing(False)
#             self.close()
#
#
#     def cancelFunction(self):
#
#         self.close()

class RawWidget(qt.QWidget):
    def __init__(self,mainWindow = None):
        qt.QWidget.__init__(self)

        from ImageViewer import WidgetImage

        self.mainWindow = mainWindow

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)
        try:
            self.setWindowModality(Qt.ApplicationModal)
        except:
            pass

        self.previewViewer = WidgetImage(standAlone=True)
        self.previewViewer.resize(500*vrb.ratio,500*vrb.ratio)
        try:
            self.previewViewer.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.previewViewer.setWindowFlags(Qt.WindowStaysOnTopHint)
        try:
            self.previewViewer.setWindowModality(Qt.ApplicationModal)
        except:
            pass

        self.widgetDialog = FileSelector()

        #### GEOMETRY TYPE

        # self.labelTitle = qt.QLabel("Define the image geometry")
        self.groupBoxImageGeometry = qt.QGroupBox()
        self.labelWidth = qt.QLabel("Width (X) : ")
        self.lineEditWidth = NumberLineEdit()
        self.lineEditWidth.setFixedWidth(60*vrb.ratio)
        self.labelHeight = qt.QLabel("Height (Y) : ")
        self.lineEditHeight = NumberLineEdit()
        self.lineEditHeight.setFixedWidth(60*vrb.ratio)
        self.labelDepth = qt.QLabel("Depth (Z) : ")
        self.lineEditDepth = NumberLineEdit()
        self.lineEditDepth.setFixedWidth(60*vrb.ratio)
        self.lineEditDepth.setPlaceholderText("3D Image only")
        self.labelColor = qt.QLabel("Color (C) : ")
        self.lineEditColor = NumberLineEdit()
        self.lineEditColor.setFixedWidth(60 * vrb.ratio)
        self.lineEditColor.setPlaceholderText("Color Image only")
        self.labelSeq = qt.QLabel("Sequence (T) : ")
        self.lineEditSeq = NumberLineEdit()
        self.lineEditSeq.setFixedWidth(60 * vrb.ratio)
        self.lineEditSeq.setPlaceholderText("Sequence Image only")
        self.labelOffset = qt.QLabel("Offset : ")
        self.lineEditOffset = NumberLineEdit()
        self.lineEditOffset.setFixedWidth(60*vrb.ratio)
        self.lineEditOffset.setPlaceholderText("0")

        self.imageGeometryLayout = qt.QGridLayout()
        self.imageGeometryLayout.addWidget(self.labelWidth, 0, 0)
        self.imageGeometryLayout.addWidget(self.lineEditWidth, 0, 1)
        self.imageGeometryLayout.addWidget(self.labelHeight, 1, 0)
        self.imageGeometryLayout.addWidget(self.lineEditHeight, 1, 1)
        self.imageGeometryLayout.addWidget(self.labelDepth, 2, 0)
        self.imageGeometryLayout.addWidget(self.lineEditDepth, 2, 1)
        self.imageGeometryLayout.addWidget(self.labelColor, 3, 0)
        self.imageGeometryLayout.addWidget(self.lineEditColor, 3, 1)
        self.imageGeometryLayout.addWidget(self.labelSeq, 4, 0)
        self.imageGeometryLayout.addWidget(self.lineEditSeq, 4, 1)
        self.imageGeometryLayout.addWidget(self.labelOffset, 5, 0)
        self.imageGeometryLayout.addWidget(self.lineEditOffset, 5, 1)
        self.groupBoxImageGeometry.setLayout(self.imageGeometryLayout)
        self.groupBoxImageGeometry.setTitle("Image Geometry")
        self.groupBoxImageGeometry.setStyleSheet(
            'QGroupBox:title {left: 20px ;padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {border: 1px solid gray; margin-top: 12 px}')


        self.groupBoxImageType = qt.QGroupBox("Image Type")
        self.labelType = qt.QLabel("type : ")
        self.comboBoxBuffer = qt.QComboBox()
        self.comboBoxBuffer.addItem('Int 8',"PyIPSDK.eImageBufferType.eIBT_Int8")
        self.comboBoxBuffer.addItem('UInt 8',"PyIPSDK.eImageBufferType.eIBT_UInt8")
        self.comboBoxBuffer.addItem('Int 16',"PyIPSDK.eImageBufferType.eIBT_Int16")
        self.comboBoxBuffer.addItem('UInt 16',"PyIPSDK.eImageBufferType.eIBT_UInt16")
        self.comboBoxBuffer.addItem('Int 32',"PyIPSDK.eImageBufferType.eIBT_Int32")
        self.comboBoxBuffer.addItem('UInt 32',"PyIPSDK.eImageBufferType.eIBT_UInt32")
        self.comboBoxBuffer.addItem('Real 32',"PyIPSDK.eImageBufferType.eIBT_Real32")
        self.comboBoxBuffer.addItem('Binary',"PyIPSDK.eImageBufferType.eIBT_Binary")
        self.comboBoxBuffer.addItem('Label 8',"PyIPSDK.eImageBufferType.eIBT_Label8")
        self.comboBoxBuffer.addItem('Label 16',"PyIPSDK.eImageBufferType.eIBT_Label16")
        self.comboBoxBuffer.addItem('Label 32',"PyIPSDK.eImageBufferType.eIBT_Label32")
        self.comboBoxBuffer.setCurrentIndex(1)
        self.comboBoxBuffer.setFixedWidth(60*vrb.ratio)

        # self.radioButtonGrey = qt.QRadioButton("Grey level")
        # self.radioButtonGrey.setChecked(True)
        # self.radioButtonRgb = qt.QRadioButton("RGB")

        self.labelDiskImage = qt.QLabel("Open as a disk image")
        self.checkBoxDiskImage = qt.QCheckBox()

        self.imageTypeLayout = qt.QGridLayout()
        self.imageTypeLayout.addWidget(self.labelType, 0, 0)
        self.imageTypeLayout.addWidget(self.comboBoxBuffer, 0, 1)
        self.imageTypeLayout.addWidget(self.labelDiskImage, 1, 0)
        self.imageTypeLayout.addWidget(self.checkBoxDiskImage, 1, 1)
        self.groupBoxImageType.setLayout(self.imageTypeLayout)
        self.groupBoxImageType.setTitle("Image Type")
        self.groupBoxImageType.setStyleSheet(
            'QGroupBox:title {left: 20px ;padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {border: 1px solid gray; margin-top: 12 px}')

        self.labelPixelType = qt.QLabel("Pixel type :")
        self.comboBoxPixelType = qt.QComboBox()
        self.comboBoxPixelType.addItem("Square/Cubic")
        self.comboBoxPixelType.addItem("Rectangular")
        self.comboBoxPixelType.setCurrentIndex(0)
        self.comboBoxPixelType.setFixedWidth(80*vrb.ratio)


        self.labelUnitType = qt.QLabel("Unit :")
        self.comboBoxUnitType = qt.QComboBox()
        self.comboBoxUnitType.addItem("px")
        self.comboBoxUnitType.addItem("m")
        self.comboBoxUnitType.addItem("cm")
        self.comboBoxUnitType.addItem("mm")
        self.comboBoxUnitType.addItem("um")
        self.comboBoxUnitType.addItem("nm")
        self.comboBoxUnitType.setCurrentIndex(0)
        self.comboBoxUnitType.setFixedWidth(60*vrb.ratio)

        # self.lineEditXPixels = qt.QLineEdit()
        # self.lineEditXPixels.setAlignment(Qt.AlignRight)
        # self.lineEditYPixels = qt.QLineEdit()
        # self.lineEditYPixels.setAlignment(Qt.AlignRight)
        # self.lineEditZPixels = qt.QLineEdit()
        # self.lineEditZPixels.setAlignment(Qt.AlignRight)
        self.labelXPixel = qt.QLabel("Pixel size = ")
        self.labelYPixel = qt.QLabel("Pixel size = ")
        self.labelZPixel = qt.QLabel("Pixel size = ")
        self.labelX = qt.QLabel("X")
        self.labelY = qt.QLabel("Y")
        self.labelZ = qt.QLabel("Z")
        self.lineEditXScale = NumberLineEdit()
        self.lineEditXScale.setAlignment(Qt.AlignRight)
        self.lineEditYScale = NumberLineEdit()
        self.lineEditYScale.setAlignment(Qt.AlignRight)
        self.lineEditZScale = NumberLineEdit()
        self.lineEditZScale.setAlignment(Qt.AlignRight)
        self.labelXUnits = qt.QLabel(self.comboBoxUnitType.currentText())
        self.labelYUnits = qt.QLabel(self.comboBoxUnitType.currentText())
        self.labelZUnits = qt.QLabel(self.comboBoxUnitType.currentText())

        self.groupBoxUnits = qt.QGroupBox()
        self.layoutUnits = qt.QHBoxLayout()
        self.layoutUnits.addWidget(self.labelUnitType)
        self.layoutUnits.addWidget(self.comboBoxUnitType)
        self.groupBoxUnits.setLayout(self.layoutUnits)
        self.groupBoxUnits.setStyleSheet(
            'QGroupBox {border: 0px;}')

        # self.lineEditYPixels.setVisible(False)
        # self.lineEditZPixels.setVisible(False)
        self.labelYPixel.setVisible(False)
        self.labelZPixel.setVisible(False)
        self.labelX.setVisible(False)
        self.labelY.setVisible(False)
        self.labelZ.setVisible(False)
        self.lineEditYScale.setVisible(False)
        self.lineEditZScale.setVisible(False)
        self.labelYUnits.setVisible(False)
        self.labelZUnits.setVisible(False)

        self.groupBoxImageCalibration = qt.QGroupBox()
        self.imageCalibrationLayout = qt.QGridLayout()
        self.imageCalibrationLayout.addWidget(self.labelPixelType, 0,0)
        self.imageCalibrationLayout.addWidget(self.comboBoxPixelType, 0,1)
        # self.imageCalibrationLayout.addWidget(self.groupBoxUnits, 0,2, 1, 2, Qt.AlignRight)
        self.imageCalibrationLayout.addWidget(self.labelUnitType, 0,3, Qt.AlignRight)
        self.imageCalibrationLayout.addWidget(self.comboBoxUnitType, 0,4, Qt.AlignRight)
        self.imageCalibrationLayout.addWidget(self.labelX, 1,0)
        self.imageCalibrationLayout.addWidget(self.labelXPixel, 1,1)
        self.imageCalibrationLayout.addWidget(self.lineEditXScale, 1,3)
        self.imageCalibrationLayout.addWidget(self.labelXUnits, 1,4, Qt.AlignRight)
        self.imageCalibrationLayout.addWidget(self.labelY, 2, 0)
        self.imageCalibrationLayout.addWidget(self.labelYPixel, 2, 1)
        self.imageCalibrationLayout.addWidget(self.lineEditYScale, 2, 3)
        self.imageCalibrationLayout.addWidget(self.labelYUnits, 2,4, Qt.AlignRight)
        self.imageCalibrationLayout.addWidget(self.labelZ, 3, 0)
        self.imageCalibrationLayout.addWidget(self.labelZPixel, 3, 1)
        self.imageCalibrationLayout.addWidget(self.lineEditZScale, 3, 3)
        self.imageCalibrationLayout.addWidget(self.labelZUnits, 3,4, Qt.AlignRight)
        self.groupBoxImageCalibration.setLayout(self.imageCalibrationLayout)
        self.groupBoxImageCalibration.setTitle("Image Calibration")
        self.groupBoxImageCalibration.setStyleSheet(
            'QGroupBox:title {left: 20px ;padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {border: 1px solid gray; margin-top: 12 px}')

        self.groupValidate = GroupBoxValidate()
        self.groupValidate.buttonValidate.setEnabled(False)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.widgetDialog,0,0,1,2)
        self.layout.addWidget(self.groupBoxImageGeometry,1,0,1,2)
        self.layout.addWidget(self.groupBoxImageType,2,0,1,2)
        self.layout.addWidget(self.groupBoxImageCalibration,3,0,1,2)
        self.layout.addWidget(self.groupValidate,4,0,1,1,Qt.AlignBottom)
        self.setLayout(self.layout)

        # self.layout = qt.QGridLayout()
        # self.layout.addWidget(self.widgetDialog,0,0,1,2)
        # self.layout.addWidget(self.labelTitle,1,0,1,2)
        # self.layout.addWidget(self.labelWidth,2,0,1,1)
        # self.layout.addWidget(self.lineEditWidth,2,1,1,1,Qt.AlignLeft)
        # self.layout.addWidget(self.labelHeight,3,0,1,1)
        # self.layout.addWidget(self.lineEditHeight,3,1,1,1,Qt.AlignLeft)
        # self.layout.addWidget(self.labelDepth,4,0,1,1)
        # self.layout.addWidget(self.lineEditDepth,4,1,1,1,Qt.AlignLeft)
        # self.layout.addWidget(self.labelOffset, 5, 0, 1, 1)
        # self.layout.addWidget(self.lineEditOffset, 5, 1, 1, 1,Qt.AlignLeft)
        # self.layout.addWidget(self.labelType,6,0,1,1)
        # self.layout.addWidget(self.comboBoxBuffer,6,1,1,1,Qt.AlignLeft)
        # self.layout.addWidget(self.radioButtonGrey,7,0,1,1)
        # self.layout.addWidget(self.radioButtonRgb,7,1,1,1,Qt.AlignLeft)
        # self.layout.addWidget(self.labelDiskImage,8,0,1,1,Qt.AlignLeft)
        # self.layout.addWidget(self.checkBoxDiskImage,8,1,1,1,Qt.AlignLeft)
        # self.layout.addWidget(self.groupValidate,9,0,1,2,Qt.AlignBottom)
        # self.setLayout(self.layout)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)
        self.setWindowTitle("Open raw image")

        self.setFixedSize(325*vrb.ratio,450*vrb.ratio)

        self.widgetDialog.lineEditFile.textChanged.connect(self.verifyImageGeometry)
        self.lineEditWidth.textChanged.connect(self.verifyImageGeometry)
        self.lineEditHeight.textChanged.connect(self.verifyImageGeometry)
        self.lineEditDepth.textChanged.connect(self.verifyImageGeometry)
        self.lineEditOffset.textChanged.connect(self.verifyImageGeometry)
        self.comboBoxBuffer.currentIndexChanged.connect(self.verifyImageGeometry)
        # self.radioButtonGrey.clicked.connect(self.verifyImageGeometry)
        # self.radioButtonRgb.clicked.connect(self.verifyImageGeometry)

        self.comboBoxPixelType.currentIndexChanged.connect(self.updateCalibrationWidgets)
        self.comboBoxUnitType.currentIndexChanged.connect(self.updateCalibrationWidgets)

        self.groupValidate.buttonValidate.clicked.connect(self.validateImage)
        self.groupValidate.buttonPreview.clicked.connect(self.previewImage)
        # self.groupValidate.buttonCancel.clicked.connect(self.cancelFunction)

    def verifyImageGeometry(self):

        try:
            numberByte = int(self.lineEditWidth.text())*int(self.lineEditHeight.text())
            if self.lineEditDepth.text() != "":
                numberByte *= int(self.lineEditDepth.text())
            if self.comboBoxBuffer.currentText() in ['Int 16','UInt 16','Label 16']:
                numberByte*=2
            elif self.comboBoxBuffer.currentText() in ['Int 32','UInt 32','Real 32','Label 32']:
                numberByte*=4
            if self.lineEditColor.text() != "":
                numberByte*=3
            if self.lineEditSeq.text() != "":
                numberByte*=int(self.lineEditSeq.text())
            if self.lineEditOffset.text() != "":
                offset = int(self.lineEditOffset.text())
            else:
                offset = 0
            numberByte+=offset
            filename = self.widgetDialog.lineEditFile.text()
            statinfo = os.stat(filename)
            size = statinfo.st_size
            if numberByte <= size:
                self.groupValidate.buttonValidate.setEnabled(True)
            else:
                self.groupValidate.buttonValidate.setEnabled(False)
        except:
            self.groupValidate.buttonValidate.setEnabled(False)

    def previewImage(self):

        from LutsWidget import GroupBoxLut

        self.previewViewer.close()

        try:
            sizeX = int(self.lineEditWidth.text())
            sizeY = int(self.lineEditHeight.text())
            buffer = self.comboBoxBuffer.currentData()
            if self.lineEditOffset.text() != "":
                offset = self.lineEditOffset.text() + "\n"
            else:
                offset = "0\n"


            # if self.lineEditColor.text() == "":
            #     geometry = "PyIPSDK.geometry2d(" + buffer + "," + str(sizeX) + "," + str(sizeY) + ")\n"
            # else:
            #     geometry = "PyIPSDK.geometryRgb2d(" + buffer + "," + str(sizeX) + "," + str(sizeY) + ")\n"
            # filename = self.widgetDialog.lineEditFile.text()

            colorGeometry = "PyIPSDK.geometryGrey()"
            if self.lineEditColor.text() != "":
                colorGeometry = "PyIPSDK.geometryUser(" + self.lineEditColor.text() + ")"
            geometry = "PyIPSDK.geometry(" + buffer + "," + str(sizeX) + "," + str(
                sizeY) + ",PyIPSDK.geometry2d()," + colorGeometry + ",PyIPSDK.geometrySingle())\n"

            filename = self.widgetDialog.lineEditFile.text()


            text =""
            text += "geometry = " + geometry
            text += "offset = " + offset
            text += "image = PyIPSDK.loadRawImageFile('" + filename + "',geometry,offset)\n"
            _locals = locals()
            exec(text, globals(), _locals)
            image = _locals['image']

            image.valueMin, image.valueMax = fct.getMinMaxValue(image)
            image.valueMinMask = None
            if image.getSizeC() == 1:
                image.groupBoxLut = GroupBoxLut(labelImage=image, parent=image)
            self.previewViewer.setImage(image)
            self.previewViewer.setWindowTitle(os.path.basename(filename))
            self.previewViewer.show()
        except:
            traceback.print_exc(file=sys.stderr)
            pass

    def validateImage(self):

        try:
            sizeX = int(self.lineEditWidth.text())
            sizeY = int(self.lineEditHeight.text())
            buffer = self.comboBoxBuffer.currentData()
            if self.lineEditOffset.text() != "":
                offset = self.lineEditOffset.text() + "\n"
            else:
                offset = "0\n"

            volumeGeometry = "PyIPSDK.geometry2d()"
            if self.lineEditDepth.text() != "":
                volumeGeometry = "PyIPSDK.geometry3d(" + self.lineEditDepth.text() + ")"

            colorGeometry = "PyIPSDK.geometryGrey()"
            if self.lineEditColor.text() != "":
                colorGeometry = "PyIPSDK.geometryUser(" + self.lineEditColor.text() + ")"

            temporalGeometry = "PyIPSDK.geometrySingle()"
            if self.lineEditSeq.text() != "":
                temporalGeometry = "PyIPSDK.geometrySequence(" + self.lineEditSeq.text() + ")"

            geometry = "PyIPSDK.geometry(" + buffer + "," + str(sizeX) + "," + str(
                sizeY) + "," + volumeGeometry + "," + colorGeometry + "," + temporalGeometry + ")\n"
            filename = self.widgetDialog.lineEditFile.text()

            # if self.lineEditDepth.text() == "":
            #     if self.lineEditColor.text() == '1' or self.lineEditColor.text()=="":
            #         geometry = "PyIPSDK.geometry2d(" + buffer + "," + str(sizeX) + "," + str(sizeY) + ")\n"
            #     else:
            #         geometry = "PyIPSDK.geometryRgb2d(" + buffer + "," + str(sizeX) + "," + str(sizeY) + ")\n"
            # else:
            #     sizeZ = int(self.lineEditDepth.text())
            #     if self.lineEditColor.text() == '1' or self.lineEditColor.text()=="":
            #         geometry = "PyIPSDK.geometry3d(" + buffer + "," + str(sizeX) + "," + str(sizeY) + "," + str(sizeZ) + ")\n"
            #     else:
            #         geometry = "PyIPSDK.geometryRgb3d(" + buffer + "," + str(sizeX) + "," + str(sizeY) + "," + str(sizeZ) + ")\n"
            # filename = self.widgetDialog.lineEditFile.text()


            self.close()
            self.mainWindow.groupBoxProcessing.setText('Reading raw image ' + os.path.basename(filename))
            self.mainWindow.toggleGroupBoxProcessing(True)

            if self.mainWindow is not None:
                label,image = self.mainWindow.urlToImage(filename,geometry=geometry,offset=offset,disk=self.checkBoxDiskImage.isChecked())


            # Set the calibration to the image
            unit = self.comboBoxUnitType.currentText()
            if self.comboBoxPixelType.currentText() == "Square/Cubic":
                calibX = float(self.lineEditXScale.text())
                calibY = float(self.lineEditXScale.text())
                calibZ = float(self.lineEditXScale.text())
            else:
                calibX = float(self.lineEditXScale.text())
                calibY = float(self.lineEditYScale.text())
                calibZ = float(self.lineEditZScale.text())

            if image.getSizeZ() <= 1:
                calibration = PyIPSDK.createGeometricCalibration2d(calibX, calibY, unit)
            else:
                calibration = PyIPSDK.createGeometricCalibration3d(calibX, calibY, calibZ, unit)

            image.setGeometricCalibration(calibration)
            vrb.mainWindow.groupMenu.groupCalibration.loadImageCalibration()

            self.mainWindow.toggleGroupBoxProcessing(False)

        except:
            traceback.print_exc(file=sys.stderr)
            self.mainWindow.toggleGroupBoxProcessing(False)
            self.close()


    def updateCalibrationWidgets(self):

        if self.comboBoxPixelType.currentText() == "Square/Cubic":
            self.setFixedSize(325 * vrb.ratio, 450 * vrb.ratio)

            self.labelX.setVisible(False)
            self.labelY.setVisible(False)
            self.labelZ.setVisible(False)
            self.lineEditXScale.setVisible(True)
            self.lineEditYScale.setVisible(False)
            self.lineEditZScale.setVisible(False)
            # self.lineEditXPixels.setVisible(True)
            # self.lineEditYPixels.setVisible(False)
            # self.lineEditZPixels.setVisible(False)
            self.labelXPixel.setVisible(True)
            self.labelYPixel.setVisible(False)
            self.labelZPixel.setVisible(False)
            self.labelXUnits.setVisible(True)
            self.labelYUnits.setVisible(False)
            self.labelZUnits.setVisible(False)
        else:
            self.setFixedSize(325 * vrb.ratio, 480 * vrb.ratio)

            self.labelX.setVisible(True)
            self.labelY.setVisible(True)
            self.labelZ.setVisible(True)
            self.lineEditXScale.setVisible(True)
            self.lineEditYScale.setVisible(True)
            self.lineEditZScale.setVisible(True)
            # self.lineEditXPixels.setVisible(True)
            # self.lineEditYPixels.setVisible(True)
            # self.lineEditZPixels.setVisible(True)
            self.labelXPixel.setVisible(True)
            self.labelYPixel.setVisible(True)
            self.labelZPixel.setVisible(True)
            self.labelXUnits.setVisible(True)
            self.labelYUnits.setVisible(True)
            self.labelZUnits.setVisible(True)

        self.labelXUnits.setText(self.comboBoxUnitType.currentText())
        self.labelYUnits.setText(self.comboBoxUnitType.currentText())
        self.labelZUnits.setText(self.comboBoxUnitType.currentText())


    def cancelFunction(self):

        self.close()

class MachineLearningSettingsWidget(qt.QWidget):
    def __init__(self):
        qt.QWidget.__init__(self)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)
        try:
            self.setWindowModality(Qt.ApplicationModal)
        except:
            pass

        self.image = None
        self.name = "Image"

        self.groupBoxModules = GroupBoxModules()
        self.groupBoxModels = GroupBoxModels(parent = self)
        self.checkBoxAddImage = qt.QCheckBox("Don't add image")
        self.groupBoxValidateMachineLearning = GroupBoxValidateMachineLearning()

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.groupBoxModules, 0, 0)
        self.layout.addWidget(self.groupBoxModels, 1, 0)
        self.layout.addWidget(self.checkBoxAddImage, 2, 0)
        self.layout.addWidget(self.groupBoxValidateMachineLearning, 3, 0)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10,5,10,5)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)
        self.setWindowTitle("Machine Learning settings")

        self.setFixedSize(350 * vrb.ratio, 280 * vrb.ratio)

        self.groupBoxValidateMachineLearning.buttonValidate.clicked.connect(self.launchModel)
        self.groupBoxValidateMachineLearning.buttonCancel.clicked.connect(self.cancelLaunching)
        self.groupBoxModules.radioButtonPixelClassification.clicked.connect(self.groupBoxModels.loadingModels)
        self.groupBoxModules.radioButtonShapeClassification.clicked.connect(self.groupBoxModels.loadingModels)
        self.groupBoxModules.radioButtonSuperPixelSegmentation.clicked.connect(self.groupBoxModels.loadingModels)

    def launchModel(self):

        try:

            if self.checkBoxAddImage.isChecked():
                self.openModel()
                self.close()

            else:
                imageOverlay = None

                if self.image is not None:
                    image = util.copyImg(self.image)
                else:
                    image = None
                name = self.name
                if image is None:
                    try:
                        image = util.copyImg(vrb.mainWindow.currentLabel.image)
                        name = vrb.mainWindow.currentLabel.name
                    except:
                        image = None

                    try:
                        if vrb.mainWindow.widgetImage.imageOverlayLabel is not None and vrb.mainWindow.widgetImage.imageOverlayLabel.image is not None:
                            # imageOverlay = vrb.mainWindow.widgetImage.imageOverlayLabel.image
                            imageOverlay = util.copyImg(vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.currentData().image)
                            nameOverlay = vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.currentText()
                        else:
                            imageOverlay = None
                    except:
                        imageOverlay = None

                continueLaunching = True
                if image is not None and self.groupBoxModules.radioButtonPixelClassification.isChecked():

                    sizeImage = image.getSizeX()*image.getSizeY()*image.getSizeZ()*image.getSizeC()

                    if sizeImage > 3000000:
                        continueLaunching = False
                        nbPixels = fct.numberToScientific(sizeImage)
                        messageBox = MessageBox('Your current image contains '+str(nbPixels) + " pixels.\nTraining will be faster if you crop your image first.\n Do you want to continue with this image ?", '', buttons=[qt.QMessageBox.Yes, qt.QMessageBox.No], icon=qt.QMessageBox.Warning)
                        res = messageBox.exec()
                        if res == qt.QMessageBox.Yes:
                            continueLaunching = True

                if continueLaunching:

                    self.openModel()

                    if image is not None:
                        if self.groupBoxModules.radioButtonPixelClassification.isChecked() or self.groupBoxModules.radioButtonSuperPixelSegmentation.isChecked():
                            vrb.mainWindow.currentModuleMachineLearning.addImage(image,name)
                        elif self.groupBoxModules.radioButtonShapeClassification.isChecked():
                            if imageOverlay is None:
                                #image = util.copyImg(image)
                                vrb.mainWindow.moduleRF_Classification.addImage(image,name)
                            else:
                                #imageOverlay = util.copyImg(imageOverlay)
                                #image = util.copyImg(image)
                                vrb.mainWindow.moduleRF_Classification.addImage(imageOverlay, nameOverlay)
                                vrb.mainWindow.moduleRF_Classification.addImage(image, name)

                self.close()

        except:
            traceback.print_exc(file=sys.stderr)

    def openModel(self):

        if self.groupBoxModels.radioButtonExistingModel.isChecked():

            modelName = self.groupBoxModels.comboBoxModel.currentText()

            if self.groupBoxModules.radioButtonPixelClassification.isChecked():
                vrb.mainWindow.moduleRF_Segmentation.show()
                vrb.mainWindow.moduleRF_Segmentation.loadModel(modelName)
                vrb.mainWindow.currentModuleMachineLearning = vrb.mainWindow.moduleRF_Segmentation
            elif self.groupBoxModules.radioButtonShapeClassification.isChecked():
                vrb.mainWindow.moduleRF_Classification.show()
                vrb.mainWindow.moduleRF_Classification.loadModel(modelName)
                vrb.mainWindow.currentModuleMachineLearning = vrb.mainWindow.moduleRF_Classification
            elif self.groupBoxModules.radioButtonSuperPixelSegmentation.isChecked():
                vrb.mainWindow.moduleRF_SuperPixels.show()
                vrb.mainWindow.moduleRF_SuperPixels.loadModel(modelName)
                vrb.mainWindow.currentModuleMachineLearning = vrb.mainWindow.moduleRF_SuperPixels

        else:
            modelName = self.groupBoxModels.lineEditNewModel.text()
            if modelName == "":
                modelName = self.groupBoxModels.lineEditNewModel.placeholderText()
            if self.groupBoxModules.radioButtonPixelClassification.isChecked():
                vrb.mainWindow.moduleRF_Segmentation.show()
                vrb.mainWindow.moduleRF_Segmentation.loadModelRef(name=modelName)
                vrb.mainWindow.currentModuleMachineLearning = vrb.mainWindow.moduleRF_Segmentation
            elif self.groupBoxModules.radioButtonShapeClassification.isChecked():
                vrb.mainWindow.moduleRF_Classification.show()
                vrb.mainWindow.moduleRF_Classification.settingsWidget.lineEditModelName.setPlaceholderText(modelName)
                vrb.mainWindow.currentModuleMachineLearning = vrb.mainWindow.moduleRF_Classification
            elif self.groupBoxModules.radioButtonSuperPixelSegmentation.isChecked():
                vrb.mainWindow.moduleRF_SuperPixels.show()
                vrb.mainWindow.moduleRF_SuperPixels.settingsWidget.lineEditModelName.setPlaceholderText(modelName)
                vrb.mainWindow.currentModuleMachineLearning = vrb.mainWindow.moduleRF_SuperPixels

    def cancelLaunching(self):

        self.close()

    def loading(self,image=None,name="Image"):

        self.groupBoxModels.loadingModels()
        self.image = image
        self.name = name

class GroupBoxModules(qt.QGroupBox):

    def __init__(self):
        qt.QGroupBox.__init__(self)

        self.radioButtonPixelClassification = qt.QRadioButton("Smart segmentation")
        self.radioButtonPixelClassification.setChecked(True)
        self.radioButtonShapeClassification = qt.QRadioButton("Smart classification")
        self.radioButtonSuperPixelSegmentation = qt.QRadioButton("Super Pixel Segmentation")

        #self.radioButtonShapeClassification.setEnabled(False)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.radioButtonPixelClassification, 0, 0, QtCore.Qt.AlignLeft)
        self.layout.addWidget(self.radioButtonShapeClassification, 1, 0, QtCore.Qt.AlignLeft)
        self.layout.addWidget(self.radioButtonSuperPixelSegmentation, 2, 0, QtCore.Qt.AlignLeft)

        self.setLayout(self.layout)

        self.setTitle("Modules")
        self.setStyleSheet('QGroupBox:title {left: 20px;padding-left: 10px;padding-right: 10px;padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {font: bold;border: 1px solid gray; margin-top: 12 px }')
        self.layout.setContentsMargins(20, 10, 20, 10)

        self.setFixedHeight(90*vrb.ratio)

class GroupBoxModels(qt.QGroupBox):

    def __init__(self,parent = None):
        qt.QGroupBox.__init__(self)

        self.parent = parent

        self.radioButtonExistingModel = qt.QRadioButton("Existing model")
        self.comboBoxModel = qt.QComboBox()
        self.comboBoxModel.setFixedWidth(110*vrb.ratio)
        self.buttonDelete = PushButtonImage(vrb.folderImages + "/Trash2.png",margins=0)
        self.buttonDelete.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.radioButtonNewModel = qt.QRadioButton("New model")
        self.radioButtonNewModel.setChecked(True)
        self.lineEditNewModel = qt.QLineEdit()
        self.lineEditNewModel.setPlaceholderText("Model 1")

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.radioButtonExistingModel, 0, 0)
        self.layout.addWidget(self.comboBoxModel, 0, 1)
        self.layout.addWidget(self.buttonDelete, 0, 2, QtCore.Qt.AlignLeft)
        self.layout.addWidget(self.radioButtonNewModel, 1, 0)
        self.layout.addWidget(self.lineEditNewModel, 1, 1, QtCore.Qt.AlignLeft)

        self.setLayout(self.layout)

        self.setTitle("Models")
        self.setStyleSheet('QGroupBox:title {left: 20px;padding-left: 10px;padding-right: 10px;padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {font: bold;border: 1px solid gray; margin-top: 12 px }')
        self.layout.setContentsMargins(20, 10, 20, 10)

        self.radioButtonExistingModel.clicked.connect(self.radioButtonPressed)
        self.radioButtonNewModel.clicked.connect(self.radioButtonPressed)
        self.buttonDelete.clicked.connect(self.deleteModel)

        self.setFixedHeight(90*vrb.ratio)

        self.radioButtonPressed()

    def deleteModel(self):

        modelName = self.comboBoxModel.currentText()
        if self.parent.groupBoxModules.radioButtonPixelClassification.isChecked():
            folder = vrb.folderPixelClassification
        elif self.parent.groupBoxModules.radioButtonShapeClassification.isChecked():
            folder = vrb.folderShapeClassification
        elif self.parent.groupBoxModules.radioButtonSuperPixelSegmentation.isChecked():
            folder = vrb.folderSuperPixelSegmentation
        filePath = folder + "/" + modelName
        try:
            if os.path.exists(filePath):
                messageBox = MessageBox('Are you sure you want to delete the model: ' + modelName, '', buttons=[qt.QMessageBox.Yes, qt.QMessageBox.No], icon=qt.QMessageBox.Warning)
                res = messageBox.exec()
                if res == qt.QMessageBox.Yes:
                    shutil.rmtree(filePath)
                    self.loadingModels()
                    try:
                        # vrb.mainWindow.actualizeFunctionMachineLearning()
                        vrb.mainWindow.updateCurrentFunction()
                    except:
                        pass
        except:
            pass

    def radioButtonPressed(self):

        boolean = self.radioButtonExistingModel.isChecked()
        self.comboBoxModel.setEnabled(boolean)
        self.buttonDelete.setEnabled(boolean)
        self.lineEditNewModel.setEnabled(not boolean)

    def loadingModels(self):

        self.comboBoxModel.clear()

        if self.parent.groupBoxModules.radioButtonPixelClassification.isChecked():
            folder = vrb.folderPixelClassification
        elif self.parent.groupBoxModules.radioButtonShapeClassification.isChecked():
            folder = vrb.folderShapeClassification
        elif self.parent.groupBoxModules.radioButtonSuperPixelSegmentation.isChecked():
            folder = vrb.folderSuperPixelSegmentation

        for modelName in sorted(os.listdir(folder),key=str.casefold):
            filePath = folder + '/' + modelName
            if os.path.isdir(filePath):
                self.comboBoxModel.addItem(modelName)
        self.radioButtonExistingModel.setEnabled(self.comboBoxModel.count()!=0)

        modelName = fct.getModelName(folder)
        self.lineEditNewModel.setPlaceholderText(modelName)

class GroupBoxValidateMachineLearning(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.buttonValidate = PushButtonImage(vrb.folderImages + "/Validate.png")
        self.buttonValidate.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.buttonValidate.setToolTip("Load")

        self.buttonCancel = PushButtonImage(vrb.folderImages + "/Back.png")
        self.buttonCancel.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.buttonCancel.setToolTip("Cancel")

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.buttonValidate, 0, 0)
        self.layout.addWidget(self.buttonCancel, 0, 1,Qt.AlignLeft)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3, 0, 3, 0)
        self.layout.setHorizontalSpacing(10*vrb.ratio)

        self.setStyleSheet("QGroupBox {border: 0px transparent; }")

        self.setFixedHeight(36*vrb.ratio)


class GroupBoxMultiSpectralEditor(qt.QGroupBox):

    def __init__(self, image=None,parent=None):
        qt.QGroupBox.__init__(self)

        self.image = image
        self.parent = parent

        # groupbox holding color channels names edition
        self.groupBoxEditNames = qt.QGroupBox()
        self.groupBoxEditNames.setStyleSheet(
            'QGroupBox:title {left: 20px; top: -12px; padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {border: 1px transparent; margin-top: 12 px}')

        #  layout for groupbox edition with labels and line edits for edition
        self.editNamesLayout = qt.QGridLayout()
        self.groupBoxEditNames.setLayout(self.editNamesLayout)


        # scroll area holding groupbox for channel names edition
        self.scrollAreaChannelNames = qt.QScrollArea()
        self.scrollAreaChannelNames.setWidgetResizable(True)
        self.scrollAreaChannelNames.setWidget(self.groupBoxEditNames)

        self.buttonSaveNames = wgt.PushButtonImage(vrb.folderImages + "/Save.png")
        self.buttonSaveNames.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.buttonSaveNames.setToolTip(txt.dictToolTips["SaveChannelsNames"])
        self.buttonSaveNames.setEnabled(False)

        # groupbox holding scroll area and save button
        # self.setTitle("Channels names edition :")
        self.setStyleSheet(
            'QGroupBox:title {left: 20px; top: -12px; padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {border: 1px solid gray; margin-top: 12 px}')

        # creates line edits for edition of channels names
        self.allLineEdits = []
        for c in range(self.image.getSizeC()):
            if self.image.getChannelName(c) == "Channel "+str(c) or self.image.getChannelName(c) == '':
                label = qt.QLabel("Channel " + str(c))
                lineEdit = qt.QLineEdit()
                lineEdit.setPlaceholderText("Channel " + str(c))
            else:
                channelName = self.image.getChannelName(c)
                label = qt.QLabel("Channel " + str(c))
                lineEdit = qt.QLineEdit()
                lineEdit.setText(str(channelName))

            lineEdit.setContentsMargins(20 * vrb.ratio, 5 * vrb.ratio, 0 * vrb.ratio, 5 * vrb.ratio)
            lineEdit.textEdited.connect(self.on_text_changed)
            self.allLineEdits.append(lineEdit)
            self.groupBoxEditNames.layout().addWidget(label, c, 0)
            self.groupBoxEditNames.layout().addWidget(lineEdit, c, 1)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.scrollAreaChannelNames, 0, 0)
        self.layout.addWidget(self.buttonSaveNames, 1, 0, Qt.AlignRight)
        self.setLayout(self.layout)

        self.buttonSaveNames.clicked.connect(self.saveChannelsNames)


    # saves channels names in image and updates comboboxes
    def saveChannelsNames(self):

        self.buttonSaveNames.setEnabled(False)
        # for i in range (self.groupBoxEditNames.layout().count()):
        #     widget = self.groupBoxEditNames.layout().itemAt(i).widget()
        #     index = self.groupBoxEditNames.layout().indexOf(widget)
        #     row, column, cols, rows = self.groupBoxEditNames.layout().getItemPosition(index)
        #
        #     # only get line edits widgets
        #     if type(widget) == QtWidgets.QLineEdit:

        for i in range(len(self.allLineEdits)):
            text = self.allLineEdits[i].text()
            placeholderText = self.allLineEdits[i].placeholderText()
            if text != '':
                self.image.setChannelName(text, i)
            else:
                self.image.setChannelName(placeholderText, i)



    # enable save button when text edited in line edits
    def on_text_changed(self):
        self.buttonSaveNames.setEnabled(True)


class WidgetMultiSpectral(qt.QWidget):

    SignalMultiSpectralChanged = pyqtSignal()

    def __init__(self, image=None,parent=None):
        qt.QWidget.__init__(self)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.image = image
        self.parent = parent

        self.labellRedChannel = qt.QLabel("Red channel")
        self.labellGreenChannel = qt.QLabel("Green channel")
        self.labellBlueChannel = qt.QLabel("Blue channel")

        # groupbox holding RGB channels selections
        self.groupBoxSelection = qt.QGroupBox()
        self.groupBoxSelection.setTitle("Channels selection :")
        self.groupBoxSelection.setStyleSheet(
            'QGroupBox:title {left: 20px; top: -12px ;padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {border: 1px solid gray; margin-top: 12 px}')

        # layout for groupbox selection
        self.selectionLayout = qt.QGridLayout()
        self.selectionLayout.addWidget(self.labellRedChannel, 0, 0)
        self.selectionLayout.addWidget(self.labellGreenChannel, 1, 0)
        self.selectionLayout.addWidget(self.labellBlueChannel, 2, 0)
        self.groupBoxSelection.setLayout(self.selectionLayout)


        self.layout = qt.QGridLayout()

        self.allComboBoxes = []
        for c in range(3):
            comboBox = qt.QComboBox()
            self.fillComboBox(comboBox)
            comboBox.setCurrentIndex(c)
            self.allComboBoxes.append(comboBox)
            self.selectionLayout.addWidget(comboBox,c,1)
            comboBox.currentIndexChanged.connect(self.adjustComboBoxValues)




        # layout of channel selector window
        self.layout.addWidget(self.groupBoxSelection, 0, 0)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(35, 25, 35, 25)
        self.layout.setHorizontalSpacing(10 * vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)


        self.setFixedSize(280*vrb.ratio,150*vrb.ratio)


        self.setWindowTitle("Channel selector")

    # fills combobox for selection
    def fillComboBox(self,comboBox):

        #sizeC = max(3,self.image.getSizeC())
        sizeC = self.image.getSizeC()

        # checks if channels names were changed
        for i in range (sizeC):
            if self.image.getChannelName(i).startswith('Red') or\
               self.image.getChannelName(i).startswith('Green') or\
               self.image.getChannelName(i).startswith('Blue') or\
               self.image.getChannelName(i) != '':
                isChannelNamesEdited = False
            else:
                isChannelNamesEdited = True

        # updates names if they were edited before
        if isChannelNamesEdited:
            self.updateComboboxNames()
        else:
            for c in range(sizeC):
                comboBox.addItem("Channel "+str(c),c)
            comboBox.addItem("No data",sizeC)



    # updates the names of channels in comboboxes
    def updateComboboxNames(self):

        for comboBox in self.allComboBoxes:
            for c in range(self.image.getSizeC()):
                channelName = self.image.getChannelName(c)
                if channelName != '':
                    comboBox.setItemText(c, channelName)
                else:
                    comboBox.setItemText(c, "Channel "+str(c))

    def adjustComboBoxValues(self):

        self.parent.colorVector = []
        for comboBox in self.allComboBoxes:
            self.parent.colorVector.append(comboBox.currentIndex())
        self.SignalMultiSpectralChanged.emit()
        vrb.mainWindow.imageViewer.getRoiImage(changeRoiImage=False)
        try:
            vrb.mainWindow.currentModuleMachineLearning.imageViewer.getRoiImage(changeRoiImage=False)
        except:
            pass

class WidgetHistogramSettings(qt.QWidget):

    def __init__(self):
        qt.QWidget.__init__(self)

        self.labelMinValue = qt.QLabel("Min value")
        self.lineEditMinValue = NumberLineEdit()
        self.labelMaxValue = qt.QLabel("Max value")
        self.lineEditMaxValue = NumberLineEdit()

        self.radioButtonNbValues = qt.QRadioButton("Number of classes")
        self.radioButtonNbValues.setChecked(True)
        self.lineEditNbValues = NumberLineEdit(constraint="Natural")
        self.lineEditNbValues.setText("100")

        self.radioButtonBinWidth = qt.QRadioButton("Bin Width")
        self.lineEditBinWidth = NumberLineEdit()
        self.lineEditBinWidth.setEnabled(False)

        self.buttonValidate = PushButtonImage(vrb.folderImages + "/Validate.png")
        self.buttonValidate.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelMinValue, 0, 0)
        self.layout.addWidget(self.lineEditMinValue, 0, 1)
        self.layout.addWidget(self.labelMaxValue, 1, 0)
        self.layout.addWidget(self.lineEditMaxValue, 1, 1)
        self.layout.addWidget(self.radioButtonNbValues, 2, 0)
        self.layout.addWidget(self.lineEditNbValues, 2, 1)
        self.layout.addWidget(self.radioButtonBinWidth, 3, 0)
        self.layout.addWidget(self.lineEditBinWidth, 3, 1)
        self.layout.addWidget(self.buttonValidate, 4, 0)

        self.setLayout(self.layout)

        self.setWindowTitle("Settings histogram")

        self.radioButtonNbValues.clicked.connect(self.changeEnable)
        self.radioButtonBinWidth.clicked.connect(self.changeEnable)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def changeEnable(self):

        self.lineEditBinWidth.setEnabled(self.radioButtonBinWidth.isChecked())
        self.lineEditNbValues.setEnabled(not self.radioButtonBinWidth.isChecked())

class GroupBoxLoadSettings(qt.QGroupBox):

    def __init__(self,sizeComboBox = 70):
        qt.QWidget.__init__(self)

        self.labelSelection = qt.QLabel("Settings")
        font = QtGui.QFont()
        font.setBold(True)
        font.setPointSize(10)

        self.labelSelection.setFont(font)
        # self.labelSelection.setStyleSheet('QLabel {color: rgb(6, 115, 186);}')
        self.labelSelection.setStyleSheet('QLabel {color: rgb(52, 152, 219);}')
        self.labelSelection.setFixedWidth(80 * vrb.ratio)

        self.comboBoxLoadingSettings = qt.QComboBox()
        self.comboBoxLoadingSettings.setFixedSize(sizeComboBox * vrb.ratio, 25 * vrb.ratio)

        self.buttonLoadSettings = PushButtonImage(vrb.folderImages + "/Validate.png", margins=2)
        self.buttonLoadSettings.setToolTip("Load settings")
        self.buttonLoadSettings.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.buttonDeleteSettings = PushButtonImage(vrb.folderImages + "/Delete2.png", margins=2)
        self.buttonDeleteSettings.setToolTip("Delete settings")
        self.buttonDeleteSettings.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        # self.groupLabelSelection.setFixedHeight(25 * vrb.ratio)
        layoutLabelSelection = qt.QGridLayout()
        self.setLayout(layoutLabelSelection)
        layoutLabelSelection.setContentsMargins(0, 0, 0, 0)
        layoutLabelSelection.setSizeConstraint(1)
        self.setStyleSheet('QGroupBox {border: 0px transparent; }')

        layoutLabelSelection.addWidget(self.labelSelection,0,0,Qt.AlignLeft)
        layoutLabelSelection.addWidget(self.comboBoxLoadingSettings,0,1,Qt.AlignRight)
        layoutLabelSelection.addWidget(self.buttonLoadSettings,0,2)
        layoutLabelSelection.addWidget(self.buttonDeleteSettings,0,3)


class WidgetImportImageLabel(qt.QWidget):

    def __init__(self):
        qt.QWidget.__init__(self)

        self.parentLabel = None
        self.settingsClasses = None
        self.imageDraw = None
        self.classesValues = []
        self.listSlices = []

        self.labelTrainedImage = qt.QLabel("Trained image")
        self.labelTrainedImage.setFixedSize(80 * vrb.ratio, 20 * vrb.ratio)
        self.comboBoxNameTrainedImage = qt.QComboBox()
        self.comboBoxNameTrainedImage.setFixedSize(140 * vrb.ratio, 20 * vrb.ratio)
        self.buttonValidateTrainedImage = PushButtonImage(vrb.folderImages + "/Validate.png")
        self.buttonValidateTrainedImage.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.checkBoxSameSampling = qt.QCheckBox("Same sampling for all classes")
        self.checkBoxSameSampling.setFixedSize(160 * vrb.ratio, 20 * vrb.ratio)

        self.scrollAreaClasses = ScrollAreaClasses(parent = self)

        self.checkBoxProbilityMap = qt.QCheckBox("Probability map")
        self.checkBoxProbilityMap.setFixedSize(120 * vrb.ratio, 20 * vrb.ratio)
        self.comboBoxProbabilityMap = qt.QComboBox()
        self.comboBoxProbabilityMap.setFixedSize(110 * vrb.ratio, 20 * vrb.ratio)
        # self.buttonValidateProbabilityMap = PushButtonImage(vrb.folderImages + "/Validate.png")
        # self.buttonValidateProbabilityMap.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)
        emptyLabel = qt.QLabel()
        emptyLabel.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)
        self.labelThreshold = qt.QLabel("Threshold (%)")
        self.labelThreshold.setFixedSize(105 * vrb.ratio, 20 * vrb.ratio)
        self.lineEditThreshold = NumberLineEdit()
        self.lineEditThreshold.setText("90")
        self.lineEditThreshold.setPlaceholderText("90")
        self.lineEditThreshold.setFixedSize(30 * vrb.ratio, 20 * vrb.ratio)

        self.radioButtonAllSlices = qt.QRadioButton("All slices")
        self.radioButtonAllSlices.setFixedWidth(60 * vrb.ratio)
        self.radioButtonAllSlices.setChecked(True)
        self.radioButtonSlices = qt.QRadioButton("Slices")
        self.radioButtonSlices.setFixedWidth(60 * vrb.ratio)
        self.lineEditSlices = qt.QLineEdit()
        self.lineEditSlices.setPlaceholderText("Example : 1,3,10")
        self.lineEditSlices.setFixedSize(110 * vrb.ratio, 20 * vrb.ratio)
        self.buttonFindSlices = PushButtonImage(vrb.folderImages + "/Export.png")
        self.buttonFindSlices.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)
        self.buttonActualizeNbPixels = PushButtonImage(vrb.folderImages + "/Refresh.png")
        self.buttonActualizeNbPixels.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        groupBoxSlices = qt.QGroupBox()
        layoutSlices = qt.QHBoxLayout()
        layoutSlices.addWidget(self.radioButtonAllSlices)
        layoutSlices.addWidget(self.radioButtonSlices)
        layoutSlices.addWidget(self.lineEditSlices)
        layoutSlices.addWidget(self.buttonFindSlices)
        groupBoxSlices.setLayout(layoutSlices)
        layoutSlices.setContentsMargins(0, 0, 0, 0)
        layoutSlices.setAlignment(Qt.AlignLeft)
        groupBoxSlices.setStyleSheet('QScrollArea {border: 0px transparent; }')

        self.buttonValidate = PushButtonImage(vrb.folderImages + "/Validate.png")
        self.buttonValidate.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelTrainedImage, 0, 0)
        self.layout.addWidget(self.comboBoxNameTrainedImage, 0, 1)
        self.layout.addWidget(self.buttonValidateTrainedImage, 0, 2)
        self.layout.addWidget(self.checkBoxSameSampling, 1, 0, 1, 7, Qt.AlignLeft)
        self.layout.addWidget(self.scrollAreaClasses, 2, 0, 1, 7)
        self.layout.addWidget(self.checkBoxProbilityMap, 3, 0)
        self.layout.addWidget(self.comboBoxProbabilityMap, 3, 1)
        # self.layout.addWidget(self.buttonValidateProbabilityMap, 3, 2)
        self.layout.addWidget(emptyLabel, 3, 3)
        self.layout.addWidget(self.labelThreshold, 3, 4)
        self.layout.addWidget(self.lineEditThreshold, 3, 5, Qt.AlignLeft)
        self.layout.addWidget(groupBoxSlices, 4, 0, 1, 5)
        self.layout.addWidget(self.buttonActualizeNbPixels, 4, 6, Qt.AlignRight)
        self.layout.addWidget(self.buttonValidate, 5, 6, Qt.AlignRight)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(20 * vrb.ratio, 20 * vrb.ratio, 20 * vrb.ratio, 20 * vrb.ratio)
        self.layout.setVerticalSpacing(20 * vrb.ratio)
        self.resize(480 * vrb.ratio, 560 * vrb.ratio)

        self.setWindowTitle("Wizard to import trained image")

        self.buttonValidateTrainedImage.clicked.connect(self.importImageDraw)
        self.checkBoxSameSampling.stateChanged.connect(self.changeSameSampling)
        # self.buttonValidateProbabilityMap.clicked.connect(self.importProbabilityMap)
        self.checkBoxProbilityMap.stateChanged.connect(self.changeEnabledProbilityMap)
        self.buttonActualizeNbPixels.clicked.connect(self.actualizeNbPixels)
        self.radioButtonAllSlices.clicked.connect(self.changeSlicesEnabled)
        self.radioButtonSlices.clicked.connect(self.changeSlicesEnabled)
        self.buttonFindSlices.clicked.connect(self.findSlices)
        self.buttonValidate.clicked.connect(self.computeOutputImage)

        self.changeEnabledProbilityMap()
        self.changeSlicesEnabled()
        # self.loadComboBox()

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def loadComboBox(self):

        for num in range(vrb.mainWindow.widgetLabelImage.layout.count()):
            item = vrb.mainWindow.widgetLabelImage.layout.itemAt(num)
            if item is not None:
                label = item.widget()
                image = label.image
                if image.getSizeX() == self.parentLabel.image.getSizeX() or image.getSizeY() == self.parentLabel.image.getSizeY() or image.getSizeZ() == self.parentLabel.image.getSizeZ():
                    if image.getBufferType() in [PyIPSDK.eIBT_Label8, PyIPSDK.eIBT_Label16, PyIPSDK.eIBT_Label32, PyIPSDK.eIBT_UInt8, PyIPSDK.eIBT_UInt16, PyIPSDK.eIBT_UInt32]:
                        self.comboBoxNameTrainedImage.addItem(image.name,image)
                    if image.getBufferType() not in [PyIPSDK.eIBT_Label8, PyIPSDK.eIBT_Label16, PyIPSDK.eIBT_Label32]:
                        self.comboBoxProbabilityMap.addItem(image.name,image)

    def computeOutputImage(self):

        if len(self.scrollAreaClasses.allClasses) != 0:

            if self.checkBoxProbilityMap.isChecked() and self.comboBoxProbabilityMap.count() >0:
                imageProbabilityMap = self.comboBoxProbabilityMap.currentData()
                try:
                    valueThreshold = float(self.lineEditThreshold.text()) / 100
                except:
                    valueThreshold = float(self.lineEditThreshold.placeholderText()) / 100
                mask = bin.lightThresholdImg(imageProbabilityMap, valueThreshold)
                imageDraw = logic.maskImg(self.imageDraw, mask)
            else:
                imageDraw = self.imageDraw

            for i in range(len(self.classesValues)):
                valueLabel = self.classesValues[i][0]
                mask = bin.thresholdImg(imageDraw, valueLabel, valueLabel)
                imageDrawMasked = logic.maskImg(imageDraw, mask)
                ratioValue = float(self.scrollAreaClasses.allClasses[i].lineEditSampling.text())/100

                if ratioValue > 1:
                    self.scrollAreaClasses.allClasses[i].lineEditSampling.setText("100")
                    ratioValue = 1

                if ratioValue > 0:
                    imgReal = util.convertImg(imageDraw, PyIPSDK.eImageBufferType.eIBT_Real32)
                    imgRandom = PyIPSDK.createImage(imgReal)
                    rangeParameter = PyIPSDK.createRange(0.0, 1/ratioValue)
                    util.randomImg(rangeParameter, imgRandom)
                    maskRatio = bin.darkThresholdImg(imgRandom, 1.0)
                    #maskRatio = logic.bitwiseAndImgImg(mask,maskRatio)
                else:
                    util.eraseImg(mask,0)
                imageDrawMasked = logic.maskImg(imageDrawMasked, maskRatio)
                imageDraw = logic.maskImgImg(imageDrawMasked,imageDraw,mask)

            lutArray = []
            for i in range(255):
                lutArray.append(0)

            for classCorrespondance in self.scrollAreaClasses.allClasses:
                lutArray[classCorrespondance.num] = classCorrespondance.comboBoxClasses.currentData()

            lut = PyIPSDK.createIntensityLUT(0, 1, lutArray)
            lutImage = itrans.lutTransform2dImg(imageDraw, lut)
            lutImage = util.convertImg(lutImage,PyIPSDK.eIBT_Label8)

            if self.radioButtonAllSlices.isChecked() or self.lineEditSlices.text() == "" or self.imageDraw.getSizeZ() == 1:
                outputImage = lutImage
            else:
                outputImage = PyIPSDK.createImage(lutImage)
                util.eraseImg(outputImage,0)
                listSlices = self.lineEditSlices.text().split(",")
                for slice in listSlices:
                    z = int(slice)
                    outPlan = PyIPSDK.extractPlan(z, 0, 0, outputImage)
                    plan = PyIPSDK.extractPlan(z, 0, 0, lutImage)
                    util.copyImg(plan,outPlan)

            if self.parentLabel is not None:
                self.parentLabel.imageDraw = outputImage
                self.parentLabel.parent.module.actualizeImageDraw()
                self.parentLabel.parent.module.model = None
                self.parentLabel.parent.module.modelIPSDK = None
                self.parentLabel.parent.module.evaluateModel()

        else:
            messageBox = MessageBox("Please initialize the trained image first", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
            messageBox.exec()

    def findSlices(self):

        text = ""
        for slice in self.listSlices:
            text += str(slice) + ","
        if text != "":
            text = text [:-1]
        self.lineEditSlices.setText(text)

    def changeSlicesEnabled(self):

        self.lineEditSlices.setEnabled(self.radioButtonSlices.isChecked())
        self.buttonFindSlices.setEnabled(self.radioButtonSlices.isChecked())

    def actualizeNbPixels(self):

        try:
            if self.checkBoxProbilityMap.isChecked() and self.comboBoxProbabilityMap.count() >0:
                imageProbabilityMap = self.comboBoxProbabilityMap.currentData()
                try:
                    valueThreshold = float(self.lineEditThreshold.text())/100
                except:
                    valueThreshold = float(self.lineEditThreshold.placeholderText())/100

                mask = bin.lightThresholdImg(imageProbabilityMap,valueThreshold)

                imageDraw = logic.maskImg(self.imageDraw,mask)
            else:
                imageDraw = self.imageDraw

            if self.radioButtonSlices.isChecked() and self.lineEditSlices.text() != "" and self.imageDraw.getSizeZ() > 1:
                outputImage = PyIPSDK.createImage(imageDraw)
                util.eraseImg(outputImage, 0)
                listSlices = self.lineEditSlices.text().split(",")
                for slice in listSlices:
                    z = int(slice)
                    outPlan = PyIPSDK.extractPlan(z, 0, 0, outputImage)
                    plan = PyIPSDK.extractPlan(z, 0, 0, imageDraw)
                    util.copyImg(plan, outPlan)
                imageDraw = outputImage

            for i in range(len(self.classesValues)):
                valueLabel = self.classesValues[i][0]
                mask = bin.thresholdImg(imageDraw,valueLabel,valueLabel)
                if self.imageDraw.getSizeZ() > 1:
                    stats = glbmsr.statsMsr3d(mask)
                else:
                    stats = glbmsr.statsMsr2d(mask)
                nbPixels = int(stats.sum)

                self.scrollAreaClasses.allClasses[i].nbPixels = nbPixels
                self.scrollAreaClasses.allClasses[i].updateRatio()
            # else:
            #     for i in range(len(self.classesValues)):
            #         self.scrollAreaClasses.allClasses[i].nbPixels = self.classesValues[i][1]
            #         self.scrollAreaClasses.allClasses[i].updateRatio()

        except:
            traceback.print_exc(file=sys.stderr)

    def changeSameSampling(self):

        try:

            if self.checkBoxSameSampling.isChecked():
                refSampling = self.scrollAreaClasses.allClasses[0].lineEditSampling.text()
                if refSampling == "":
                    refSampling = self.scrollAreaClasses.allClasses[0].lineEditSampling.placeholderText()
                for i in range(len(self.scrollAreaClasses.allClasses)):
                    if i !=0:
                        self.scrollAreaClasses.allClasses[i].lineEditSampling.setText(refSampling)
                        self.scrollAreaClasses.allClasses[i].lineEditSampling.setEnabled(False)
            else:
                for i in range(len(self.scrollAreaClasses.allClasses)):
                    if i !=0:
                        self.scrollAreaClasses.allClasses[i].lineEditSampling.setEnabled(True)

        except:
            pass

    def importImageDraw(self):

        try:

            image = self.comboBoxNameTrainedImage.currentData()

            if image.getBufferType() not in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32,PyIPSDK.eIBT_UInt8,PyIPSDK.eIBT_UInt16,PyIPSDK.eIBT_UInt32]:
                messageBox = MessageBox("Trained image type must be integer and unsigned.", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                messageBox.exec()
            elif image.getSizeX() != self.parentLabel.image.getSizeX() or image.getSizeY() != self.parentLabel.image.getSizeY() or image.getSizeZ() != self.parentLabel.image.getSizeZ():
                messageBox = MessageBox("Input image and trained image must have same dimensions", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                messageBox.exec()
            else:
                try:
                    self.loadImageDraw(image)
                except:
                    pass

        except:
            pass

        # try:
        #     file = xmlet.parse(vrb.folderInformation + "/folderMachineLearningImageDraw.mho")
        #     folderMachineLearningImageDrawElement = file.getroot()
        # except:
        #     folderMachineLearningImageDrawElement = xmlet.Element('folderMachineLearningImageDraw')
        #
        # path = Dfct.childText(folderMachineLearningImageDrawElement, "Path")
        # if path is not None:
        #     defaultFolder = path
        # else:
        #     defaultFolder = "C:/"
        #
        # filename = qt.QFileDialog.getOpenFileName(self, "Select your image", defaultFolder, "Image" + " (*.tif *.tiff)")
        # if filename[0] != [] and filename[0] != '' and filename[0] != None:
        #     image = PyIPSDK.loadTiffImageFile(filename[0])
        #
        #     if image.getBufferType() not in [PyIPSDK.eIBT_Label8,PyIPSDK.eIBT_Label16,PyIPSDK.eIBT_Label32,PyIPSDK.eIBT_UInt8,PyIPSDK.eIBT_UInt16,PyIPSDK.eIBT_UInt32]:
        #         messageBox = MessageBox("Trained image type must be integer and unsigned.", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
        #         messageBox.exec()
        #     elif image.getSizeX() != self.parentLabel.image.getSizeX() or image.getSizeY() != self.parentLabel.image.getSizeY() or image.getSizeZ() != self.parentLabel.image.getSizeZ():
        #         messageBox = MessageBox("Input image and trained image must have same dimensions", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
        #         messageBox.exec()
        #     else:
        #         self.lineEditNameTrainedImage.setText(filename[0])
        #         self.loadImageDraw(image)
        #
        #     if filename[0] != [] and filename[0] != '' and filename[0] != None:
        #         Dfct.SubElement(folderMachineLearningImageDrawElement, "Path").text = filename[0]
        #         Dfct.saveXmlElement(folderMachineLearningImageDrawElement, vrb.folderInformation + "/folderMachineLearningImageDraw.mho")

    # def importProbabilityMap(self):
    #
    #     self.imageProbabilityMap = self.comboBoxProbabilityMap.currentData()

        # try:
        #     file = xmlet.parse(vrb.folderInformation + "/folderMachineLearningImageProbabilityMap.mho")
        #     folderMachineLearningImageProbabilityMapElement = file.getroot()
        # except:
        #     folderMachineLearningImageProbabilityMapElement = xmlet.Element('folderMachineLearningImageProbabilityMap')
        #
        # path = Dfct.childText(folderMachineLearningImageProbabilityMapElement, "Path")
        # if path is not None:
        #     defaultFolder = path
        # else:
        #     defaultFolder = "C:/"
        #
        # filename = qt.QFileDialog.getOpenFileName(self, "Select your image", defaultFolder, "Image" + " (*.tif *.tiff)")
        # if filename[0] != [] and filename[0] != '' and filename[0] != None:
        #     self.imageProbabilityMap = PyIPSDK.loadTiffImageFile(filename[0])
        #     self.lineEditProbabilityMap.setText(filename[0])
        #
        #     if filename[0] != [] and filename[0] != '' and filename[0] != None:
        #         Dfct.SubElement(folderMachineLearningImageProbabilityMapElement, "Path").text = filename[0]
        #         Dfct.saveXmlElement(folderMachineLearningImageProbabilityMapElement, vrb.folderInformation + "/folderMachineLearningImageProbabilityMap.mho")

    def changeEnabledProbilityMap(self):

        booleanValue = self.checkBoxProbilityMap.isChecked()
        self.comboBoxProbabilityMap.setEnabled(booleanValue)
        self.lineEditThreshold.setEnabled(booleanValue)

        self.actualizeNbPixels()

    def loadImageDraw(self,imageDraw):

        self.imageDraw = imageDraw

        self.radioButtonSlices.setVisible(self.imageDraw.getSizeZ() > 1)
        self.radioButtonAllSlices.setVisible(self.imageDraw.getSizeZ() > 1)
        self.lineEditSlices.setVisible(self.imageDraw.getSizeZ() > 1)
        self.buttonFindSlices.setVisible(self.imageDraw.getSizeZ() > 1)

        self.scrollAreaClasses.clearScrollArea()
        self.scrollAreaClasses.addLabelsTitle()

        self.listSlices = []
        sizeZ = self.imageDraw.getSizeZ()
        if sizeZ > 1:
            stats2d = glbmsr.multiSlice_statsMsr2d(self.imageDraw)
            planIndexedStats2d = PyIPSDK.toPyDict(stats2d)
            for z in range(sizeZ):
                value = planIndexedStats2d[(z,0,0)]["Max"]
                if value != 0:
                    self.listSlices.append(int(z))

            histo = glbmsr.histogramMsr3d(self.imageDraw)
        else:
            histo = glbmsr.histogramMsr2d(self.imageDraw)

        frequencies = histo.frequencies
        minValue = histo.min

        self.classesValues = []
        ind = minValue
        for f in frequencies:
            if f !=0 and ind !=0:
                self.classesValues.append([int(ind),f])
            ind+=1

        nbClasses = len(self.classesValues)

        if self.settingsClasses is not None:
            labelClassesElement = Dfct.SubElement(self.settingsClasses, "LabelClasses")
            numberLabelElement = Dfct.SubElement(labelClassesElement, "NumberLabels")
            nbLabels = int(numberLabelElement.text)

            if nbLabels>= len(self.classesValues):

                labelsList = []

                for numLabel in range(nbLabels):
                    labelElement = Dfct.SubElement(labelClassesElement, "Label_" + str(numLabel))
                    # colorText = Dfct.childText(labelElement, "Color").split(",")
                    name = Dfct.childText(labelElement, "Name")
                    value = Dfct.childText(labelElement, "Value")
                    labelsList.append([name,numLabel+1])

                for i in range(len(self.classesValues)):

                    self.scrollAreaClasses.addNewClass(self.classesValues[i][0], labelsList,self.classesValues[i][1],i)

            else:
                messageBox = MessageBox("Number of labels in the input image (" +str(nbClasses) +") cannot be higher than the number of classes in the model ("+str(nbLabels) + ").", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                messageBox.exec()


class ScrollAreaClasses(qt.QScrollArea):

    def __init__(self,parent):
        qt.QScrollArea.__init__(self)

        self.parent = parent

        self.allClasses = []

        self.centralWidget = qt.QGroupBox()

        self.layout = qt.QVBoxLayout()
        self.layout.setAlignment(Qt.AlignTop)

        self.setWidget(self.centralWidget)

        self.layout.setContentsMargins(5, 5, 5, 5)

        self.addLabelsTitle()

        self.setStyleSheet('QScrollArea {border: 1px solid gray; }')

    def addNewClass(self, num,labelsList,nbPixels,pos):

        classesCorrespondance = ClassesCorrespondance(num,labelsList,nbPixels)
        classesCorrespondance.initComboBox(pos)

        if len(self.allClasses) == 0:
            classesCorrespondance.lineEditSampling.textChanged.connect(self.parent.changeSameSampling)

        self.allClasses.append(classesCorrespondance)
        self.addNewWidget(classesCorrespondance)

    def addLabelsTitle(self):

        groupBoxTitles = qt.QGroupBox()
        layoutTitles = qt.QHBoxLayout()

        labelLabelsTitle = qt.QLabel("Labels")
        labelLabelsTitle.setFixedSize(55 * vrb.ratio, 20 * vrb.ratio)
        labelLabelsTitle.setAlignment(Qt.AlignCenter)
        labelClassesTitle = qt.QLabel("Classes")
        labelClassesTitle.setFixedSize(105 * vrb.ratio, 20 * vrb.ratio)
        labelClassesTitle.setAlignment(Qt.AlignCenter)
        labelSamplingTitle = qt.QLabel("Sampling (%)")
        labelSamplingTitle.setFixedSize(80 * vrb.ratio, 20 * vrb.ratio)
        labelSamplingTitle.setAlignment(Qt.AlignCenter)
        labelRealRatioTitle = qt.QLabel("Real ratio")
        labelRealRatioTitle.setFixedSize(105 * vrb.ratio, 20 * vrb.ratio)
        labelRealRatioTitle.setAlignment(Qt.AlignCenter)

        layoutTitles.addWidget(labelLabelsTitle)
        layoutTitles.addWidget(labelClassesTitle)
        layoutTitles.addWidget(labelSamplingTitle)
        layoutTitles.addWidget(labelRealRatioTitle)

        groupBoxTitles.setLayout(layoutTitles)
        layoutTitles.setContentsMargins(10 * vrb.ratio, 10 * vrb.ratio, 10 * vrb.ratio, 10 * vrb.ratio)
        groupBoxTitles.setStyleSheet('QScrollArea {border: 0px transparent; }')

        self.addNewWidget(groupBoxTitles)

    def addNewWidget(self, widgetToAdd):
        self.layout.addWidget(widgetToAdd)

        self.centralWidget = qt.QWidget()
        self.centralWidget.setLayout(self.layout)
        self.setWidget(self.centralWidget)

    def actualizeSize(self):
        self.centralWidget = qt.QWidget()
        self.centralWidget.setLayout(self.layout)
        self.setWidget(self.centralWidget)

    def clearScrollArea(self):

        while self.layout.count() != 0:
            item = self.layout.itemAt(0)
            item.widget().deleteLater()
            self.layout.removeItem(item)

        self.allClasses = []

class ClassesCorrespondance(qt.QGroupBox):

    def __init__(self, num, labelsList,nbPixels):
        qt.QGroupBox.__init__(self)

        self.num = num
        self.labelsList = labelsList
        self.nbPixels = nbPixels

        self.labelValue = qt.QLabel(str(num))
        self.labelValue.setFixedSize(55 * vrb.ratio, 20 * vrb.ratio)
        self.labelValue.setAlignment(Qt.AlignCenter)
        self.comboBoxClasses = qt.QComboBox()
        self.comboBoxClasses.setFixedSize(105 * vrb.ratio, 20 * vrb.ratio)
        self.lineEditSampling = qt.QLineEdit("10")
        self.lineEditSampling.setPlaceholderText("10")
        self.lineEditSampling.setFixedSize(80 * vrb.ratio, 20 * vrb.ratio)
        self.lineEditSampling.setAlignment(Qt.AlignRight)
        self.labelRatio = qt.QLabel()
        self.labelRatio.setFixedSize(105 * vrb.ratio, 20 * vrb.ratio)
        self.labelRatio.setAlignment(Qt.AlignCenter)

        self.layout = qt.QHBoxLayout()
        self.layout.addWidget(self.labelValue)
        self.layout.addWidget(self.comboBoxClasses)
        self.layout.addWidget(self.lineEditSampling)
        self.layout.addWidget(self.labelRatio)

        self.setLayout(self.layout)
        self.setLayout(self.layout)
        self.layout.setContentsMargins(10 * vrb.ratio, 10 * vrb.ratio, 10 * vrb.ratio, 10 * vrb.ratio)
        self.setStyleSheet('QScrollArea {border: 0px transparent; }')

        self.lineEditSampling.textChanged.connect(self.updateRatio)

    def updateRatio(self):

        try:
            ratioValue = float(self.lineEditSampling.text())/100
            if ratioValue > 1:
                self.lineEditSampling.setText("100")
                ratioValue = 1
        except:
            ratioValue = float(self.lineEditSampling.placeholderText())/100

        self.labelRatio.setText(str(int(self.nbPixels*ratioValue)) + " / "+str(self.nbPixels))

    def initComboBox(self,pos):

        self.comboBoxClasses.clear()
        for label in self.labelsList:
            self.comboBoxClasses.addItem(label[0],label[1])
        self.comboBoxClasses.setCurrentIndex(pos)
        self.updateRatio()



class ComboBoxImages(qt.QComboBox):
    '''Creates a combobox containing clickable images'''
    def __init__(self, image_list, margins=0, parent=None):

        super().__init__(parent)

        self.image_list = image_list
        self.margins = margins

        self.setIconSize(self.sizeHint())  # Définir la taille des icônes
        # self.setIconSize(QtCore.QSize(30*vrb.ratio, 30*vrb.ratio)) # Définir la taille des icônes
        self.initImages()

    def initImages(self):
        for image_path in self.image_list:
            pixmap = QtGui.QPixmap(image_path)
            icon = QtGui.QIcon(pixmap)
            self.addItem(icon, "")  # Ajouter chaque image comme un item avec icône

    def resizeEvent(self, event):
        sizeX, sizeY = self.width(), self.height()

        for index in range(self.count()):
            # Redimensionner les icônes en fonction de la taille actuelle de la combobox et des marges
            pixmap = QtGui.QPixmap(self.image_list[index])
            scaled_pixmap = pixmap.scaled(
                max(0, sizeY - self.margins), max(0, sizeY - self.margins),
                # max(0, sizeX - self.margins), max(0, sizeY - self.margins),
                QtCore.Qt.IgnoreAspectRatio, QtCore.Qt.SmoothTransformation
            )
            icon = QtGui.QIcon(scaled_pixmap)
            self.setItemIcon(index, icon)  # Mettre à jour l'icône de l'item
            self.setIconSize(QtCore.QSize(sizeY - self.margins, sizeY - self.margins))
            # self.setIconSize(QtCore.QSize(sizeX - self.margins, sizeY - self.margins))

    def addImageItem(self, image_path):
        self.image_list.append(image_path)
        pixmap = QtGui.QPixmap(image_path)
        icon = QtGui.QIcon(pixmap)
        self.addItem(icon, "")

class LabeledSpinBox(qt.QGroupBox):
    def __init__(self, title="Titre", minValue=0, maxValue=100, step=1, defaultValue=0, margins=10):
        qt.QGroupBox.__init__(self)

        self.setTitle(title)
        self.layout = qt.QVBoxLayout()
        self.layout.setContentsMargins(margins, margins, margins, margins)

        # palette = self.palette()
        # palette.setColor(QtGui.QPalette.WindowText, QtGui.QColor(52, 152, 219))
        # self.setPalette(palette)
        # self.setStyleSheet("QGroupBox { font-weight: bold; }")

        self.setStyleSheet('QGroupBox:title {left: 20px;padding-left: 10px;padding-right: 10px;padding-top: -12px; color:rgb(52, 152, 219)}'
                                                    '  QGroupBox {font-size: 13px;font: bold;border: 1px solid gray; margin-top: 12 px }')

        self.spinBox = qt.QSpinBox()
        self.spinBox.setRange(minValue, maxValue)
        self.spinBox.setSingleStep(step)
        self.spinBox.setValue(defaultValue)

        self.spinBox.setSizePolicy(self.sizePolicy())

        self.layout.addWidget(self.spinBox)
        self.setLayout(self.layout)

    def getValue(self):
        return self.spinBox.value()

    def setValue(self, value):
        self.spinBox.setValue(value)

    def setRange(self, minValue, maxValue):
        self.spinBox.setRange(minValue, maxValue)

class LabeledWidget(qt.QGroupBox):

    def __init__(self, widget, title="Titre",color = (52, 152, 219),margins=10*vrb.ratio,pixelSize = 10, bold = False):
        qt.QGroupBox.__init__(self)

        self.widget = widget

        self.setTitle(title)
        self.layout = qt.QVBoxLayout()
        self.layout.setContentsMargins(margins, margins, margins, margins)

        # palette = self.palette()
        # palette.setColor(QtGui.QPalette.WindowText, QtGui.QColor(52, 152, 219))
        # self.setPalette(palette)
        # self.setStyleSheet("QGroupBox { font-weight: bold; }")

        if bold:
            textBold = 'font: bold;'
        else:
            textBold = ''
        self.setStyleSheet('QGroupBox:title {left: '+str(int(20*vrb.ratio))+'px;padding-left:  '+str(int(10*vrb.ratio))+'px;padding-right:  '+str(int(10*vrb.ratio))+'px;padding-top: -'+str(int(10*vrb.ratio))+'px; color:rgb'+str(color)+'}'
                                                    '  QGroupBox {font-size: '+str(int(pixelSize*vrb.ratio))+'px;'+textBold+'border: 1px solid gray; margin-top:  '+str(int(10*vrb.ratio))+' px }')

        # self.widget.setSizePolicy(self.sizePolicy())

        self.layout.addWidget(self.widget)
        self.setLayout(self.layout)

        font = QtGui.QFont()
        font.setPixelSize(pixelSize*vrb.ratio)
        font.setBold(False)
        self.widget.setFont(font)

# class LabeledWidget(qt.QGroupBox):
#
#     def __init__(self, widget, title="Titre",color = (52, 152, 219),margins=10*vrb.ratio,pixelSize = 12, bold = False):
#         qt.QGroupBox.__init__(self)
#
#         self.widget = widget
#
#         self.setTitle(title)
#         self.layout = qt.QVBoxLayout()
#         self.layout.setContentsMargins(margins, margins, margins, margins)
#
#         # palette = self.palette()
#         # palette.setColor(QtGui.QPalette.WindowText, QtGui.QColor(52, 152, 219))
#         # self.setPalette(palette)
#         # self.setStyleSheet("QGroupBox { font-weight: bold; }")
#
#         if bold:
#             textBold = 'font: bold;'
#         else:
#             textBold = ''
#         self.setStyleSheet('QGroupBox:title {left: 20px;padding-left: 10px;padding-right: 10px;padding-top: -12px; color:rgb'+str(color)+'}'
#                                                     '  QGroupBox {font-size: '+str(pixelSize)+'px;'+textBold+'border: 1px solid gray; margin-top: 12 px }')
#
#         # self.widget.setSizePolicy(self.sizePolicy())
#
#         self.layout.addWidget(self.widget)
#         self.setLayout(self.layout)
#
#         font = QtGui.QFont()
#         font.setPixelSize(pixelSize)
#         font.setBold(False)
#         self.widget.setFont(font)

if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])


    sys._excepthook = sys.excepthook


    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    # foo = PushButtonImage(filename=vrb.folderButtons +"/Validate.jpg")
    # img = PyIPSDK.loadTiffImageFile('D:/Stage/exemple images/mandrill_256x256_UInt8.tif')
    # foo = LabelPythonImage(image = img)

    # foo = PopUpWindow()

    # button = qt.QPushButton('edit')
    # edtLabel = EditableLabel('Blabla')
    # button.clicked.connect(edtLabel.switchVisibleWidget)
    # groupBox = qt.QGroupBox()
    # groupBox.setLayout(qt.QGridLayout())
    # groupBox.layout().addWidget(button, 1, 1)
    # groupBox.layout().addWidget(edtLabel, 0, 0)
    #
    # foo = groupBox

    # foo = SliderShowingValue(5, Qt.Vertical)
    # foo.slider.setValue(600)

    # foo = EnterNameWindow()

    # foo = GroupBoxProcessing()
    # foo = RawWidget()
    # foo = FileSelector()
    # foo = MachineLearningSettingsWidget()
    # foo = WidgetMultiSpectral()

    foo = WidgetImportImageLabel()
    # image = PyIPSDK.loadTiffImageFile("D:/ExampleImages/concrete.tif")

    file = xmlet.parse("C:/Users/micro/AppData/Roaming/ReactivIP/Explorer/Machine_Learning/Pixel_Classification/Model 2/Settings.mho")
    settingsElement = file.getroot()

    foo.show()
    foo.settingsClasses = settingsElement
    # foo.loadInterface(settingsElement)

    app.exec_()


