import sys

from PyQt5.QtCore import pyqtSignal,QPointF,Qt,QCoreApplication
from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore

import time

try:
    import matplotlib
    matplotlib.use('QT5Agg')
    import matplotlib.pylab as plt
    from matplotlib.backends.qt_compat import QtCore, QtWidgets#, is_pyqt5
    from matplotlib.backends.backend_qt5agg import FigureCanvas, NavigationToolbar2QT as NavigationToolbar
    from matplotlib.figure import Figure
except:
    pass

import UsefullFunctions as fct
import UsefullWidgets as wgt
from RangeSlider import RangeSlider
from Sliders import SimpleHorizontalSliderLabel

import DatabaseFunction as Dfct

import DrawFunctions as DWfct

import PyIPSDK
import UsefullVariables as vrb

import numpy as np

import traceback

import csv
import xlwt

import xml.etree.ElementTree as xmlet

def randomPastelColor(i):
    colors = [55,194,100]
    s = 90
    l = 90
    return QtGui.QColor.fromHsl(colors[i%len(colors)], s * 255 / 100, l * 255 / 100, 100)

greenPastelColor = QtGui.QColor(66, 219, 86)
redPastelColor = QtGui.QColor(237, 82, 71)
greyColor = QtGui.QColor(220, 220, 220)

class CustomLineWidget(qt.QWidget):
    signalDelete = pyqtSignal(qt.QWidget)

    def __init__(self):
        qt.QWidget.__init__(self)

        self.xValue = []
        self.yValue = []
        self.legendX = []
        self.values = []
        self.xCoeff = 1
        self.yCoeff = 1

        self.fig = Figure()

        self.plotWidget = FigureCanvas(self.fig)
        self.toolbar = wgt.NavigationToolbarWidget(self.plotWidget, self)

        self.buttonTable = wgt.PushButtonImage(vrb.folderImages + "/tableau.png")
        self.buttonTable.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.toolbar, 0, 0)
        self.layout.addWidget(self.buttonTable, 0, 1)
        self.layout.addWidget(self.plotWidget, 1, 0, 1, 2)
        self.setLayout(self.layout)

        self.buttonTable.clicked.connect(self.displayTable)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def closeEvent(self, a0: QtGui.QCloseEvent):
        self.signalDelete.emit(self)

    def displayTable(self):

        try:
            ddict = {}
            # ddict["Index"] = range(len(self.values[0]))
            ddict["Index"] = []
            for i in range(len(self.xValue)):
                ddict["Index"].append(i)
            ddict["X coordinate"] = self.xValue
            ddict["Y coordinate"] = self.yValue
            if len(self.values) == 1:
                ddict["Intensity"] = self.values[0]
            elif len(self.values) == 3:
                ddict["Red"] = self.values[0]
                ddict["Green"] = self.values[1]
                ddict["Blue"] = self.values[2]
            else:
                for c in range(len(self.values)):
                    ddict["Channel "+str(c)] = self.values[c]

            table = TableWidgetWindow(ddict, stats=True)

            table.setWindowTitle("Table")
            # for colNum in range(3+len(self.values)):
            #     table.tableWidget.setColumnWidth(colNum, 70 * vrb.ratio)
            table.resize((122*(3+len(self.values)) + 20) * vrb.ratio, 500 * vrb.ratio)

            fct.showWidget(table)
        except:
            traceback.print_exc(file=sys.stderr)

    def figureMouseMoveEvent(self, event):

        if self.widget == vrb.mainWindow.currentLabel:

            try:
                xCoeff = self.xCoeff
                yCoeff = self.yCoeff
            except:
                xCoeff = 1
                yCoeff = 1

            try:
                minX, minY, sizeX, sizeY = DWfct.boundingBoxPolygon(self.lineElement, precise=True)
                lengthLine = np.sqrt(sizeX * xCoeff * sizeX * xCoeff + sizeY * yCoeff * sizeY * yCoeff)
                ratio = event.xdata / lengthLine

                ratio = max(0, ratio)
                ratio = min(ratio, 1)

                posX = minX + ratio * sizeX
                posY = minY + ratio * sizeY

                point = {}
                point["X"] = posX
                point["Y"] = posY

                vrb.linePoint = [point]

            except:
                vrb.linePoint = None

            vrb.mainWindow.imageViewer.getRoiImage(changeRoiImage=False)

    def figureMouseLeaveEvent(self, event):

        vrb.linePoint = None
        vrb.mainWindow.imageViewer.getRoiImage(changeRoiImage=False)


class WidgetProfile(qt.QWidget):
    signalDelete = pyqtSignal(qt.QWidget)

    def __init__(self):
        qt.QWidget.__init__(self)

        self.xLabel = "X"
        self.yLabel = "Y"

        self.value = []
        self.legend = []

        self.fig = Figure()
        self.plotWidget = FigureCanvas(self.fig)
        self.toolbar = wgt.NavigationToolbarWidget(self.plotWidget, self)
        # self.toolbar = NavigationToolbar(self.plotWidget, self)

        self.buttonTable = wgt.PushButtonImage(vrb.folderImages + "/Tableau.png")
        self.buttonTable.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)

        self.layoutLineProfile = qt.QGridLayout()
        self.layoutLineProfile.addWidget(self.toolbar, 0, 0)
        self.layoutLineProfile.addWidget(self.buttonTable, 0, 1, Qt.AlignRight)
        self.layoutLineProfile.addWidget(self.plotWidget, 1, 0, 1, 2)
        self.setLayout(self.layoutLineProfile)

        self.buttonTable.clicked.connect(self.displayTable)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def closeEvent(self, a0: QtGui.QCloseEvent):
        self.signalDelete.emit(self)

    def displayTable(self):

        values = []
        for i in range(len(self.legend)):
            values.append(float(fct.numberCalibration(self.values[i])))

        try:
            ddict = {}
            ddict[self.xLabel] = self.legend
            ddict[self.yLabel] = values

            table = TableWidgetWindow(ddict,stats=True)

            table.setWindowTitle("Table")
            table.resize(300 * vrb.ratio, 500 * vrb.ratio)
            fct.showWidget(table)

        except:
            traceback.print_exc(file=sys.stderr)

    def drawLineProfile(self, values, legend, title=None,xlabel="",yLabel=""):

        self.values = values
        self.legend = legend

        if title is not None:
            self.setWindowTitle(title)

        self.fig.clf()
        axes = self.fig.add_subplot(111)
        line = axes.plot(legend, values, color="blue")

        if xlabel is not None and xlabel != "":
            self.xLabel = xlabel
            axes.set_xlabel(xlabel)
        if yLabel is not None and yLabel != "":
            self.yLabel = yLabel
            axes.set_ylabel(yLabel)
        if title is not None:
            axes.set_title(title)

    def display(self,xmlElement=None):

        fct.showWidget(self)

    def stopDisplay(self):

        self.close()

class TableWidgetWindow(qt.QWidget):

    def __init__(self,ddict=None, stats = False, title=None,quantiles=[],quantilePonderated=False,
                 sorted=True,allowSort=True,headerHorizontal = True,displayNumberOfElement = True):
        super(TableWidgetWindow, self).__init__()

        self.sorted = sorted
        self.quantilePonderated = quantilePonderated
        self.displayNumberOfElement = displayNumberOfElement

        self.groupBoxTop = GroupBoxTop()

        self.buttonSave = self.groupBoxTop.buttonSave
        self.labelNumberLabels = self.groupBoxTop.labelNumberLabels

        if not self.displayNumberOfElement:
            self.labelNumberLabels.setVisible(False)

        if stats:
            self.usefulInfoTableWidget = StatsTableWidget(quantiles=quantiles,quantilePonderated=quantilePonderated)
        else:
            self.usefulInfoTableWidget = None
        self.tableWidget = TableWidget(ddict=ddict,parent=self,sorted=self.sorted,allowSort=allowSort,headerHorizontal=headerHorizontal)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.groupBoxTop, 0, 0)
        self.layout.addWidget(self.tableWidget, 1, 0)
        if stats:
            self.layout.addWidget(self.usefulInfoTableWidget,2,0)

        self.setLayout(self.layout)
        if title is not None:
            self.setWindowTitle(title)

        self.buttonSave.clicked.connect(self.saveFileResult)
        if stats:
            scrollbarTable = self.tableWidget.horizontalScrollBar()
            scrollbarStats = self.usefulInfoTableWidget.horizontalScrollBar()

            # Connect scrollbars
            scrollbarStats.valueChanged.connect(scrollbarTable.setValue)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def resizeEvent(self,event):

        self.tableWidget.resizeInfoTable = False

        nbColumn = self.tableWidget.columnCount()
        if nbColumn >= 1:
            sizeColumn = (self.tableWidget.width()-10*vrb.ratio)/nbColumn
            sizeColumn = max(sizeColumn,120*vrb.ratio)
            for i in range(nbColumn):
                self.tableWidget.setColumnWidth(i, sizeColumn-1)
                self.tableWidget.setColumnWidth(i, sizeColumn)

        self.tableWidget.resizeInfoTable = True

        self.tableWidget.resizeUsefulInfoColumns()

        qt.QApplication.processEvents()

    def saveFileResult(self):

        filename = qt.QFileDialog.getSaveFileName(self, "Save file as", "","(*.xls);;(*.csv)")
        if filename[1] == "(*.xls)":
            fct.convertToXlsFile(self.tableWidget,filename[0],title="Table",header=True)
        if filename[1] == "(*.csv)":
            fct.convertToCsvFile(self.tableWidget,filename[0])

    def display(self,xmlElement=None):

        if xmlElement is not None:
            self.setWindowTitle(Dfct.childText(xmlElement, 'Name'))

        fct.showWidget(self)

    def stopDisplay(self):

        self.close()

class TableWidgetWindowWithSliders(qt.QWidget):
    def __init__(self,ddict=None, stats = False, title=None,quantiles=[],quantilePonderated=False,
                 sorted=True,allowSort=True,headerHorizontal = True,displayNumberOfElement = True):
        super(TableWidgetWindowWithSliders, self).__init__()

        self.groupBoxTop = GroupBoxTop()
        self.sorted = sorted
        self.ddict = ddict
        self.displayNumberOfElement = displayNumberOfElement

        self.buttonSave = self.groupBoxTop.buttonSave

        self.labelNumberLabels = self.groupBoxTop.labelNumberLabels

        if not self.displayNumberOfElement:
            self.labelNumberLabels.setVisible(False)

        if stats:
            self.usefulInfoTableWidget = StatsTableWidget(quantiles=quantiles,quantilePonderated=quantilePonderated)
        else:
            self.usefulInfoTableWidget = None

        self.tableWidget = TableWidget(ddict=None, parent=self, sorted=self.sorted, allowSort=allowSort,
                                       headerHorizontal=headerHorizontal)

        self.layout = qt.QVBoxLayout()
        self.layout.addWidget(self.groupBoxTop)
        self.layout.addWidget(self.tableWidget)
        if stats:
            self.layout.addWidget(self.usefulInfoTableWidget)

        if "z" in ddict and ddict["z"]> 1:
            self.sliderZ = SimpleHorizontalSliderLabel(label="z", titleSize= 30*vrb.ratio, minimum=0, maximum=ddict["z"]-1, defaultValue=0)
            self.sliderZ.slider.valueChanged.connect(self.updateTableWidget)
            self.layout.addWidget(self.sliderZ)
        else:
            self.sliderZ = None

        if "c" in ddict and ddict["c"]> 1:
            self.sliderC = SimpleHorizontalSliderLabel(label="c", titleSize= 30*vrb.ratio, minimum=0, maximum=ddict["c"]-1, defaultValue=0)
            self.sliderC.slider.valueChanged.connect(self.updateTableWidget)
            self.layout.addWidget(self.sliderC)
        else:
            self.sliderC = None

        if "t" in ddict and ddict["t"]> 1:
            self.sliderT = SimpleHorizontalSliderLabel(label="t", titleSize= 30*vrb.ratio, minimum=0, maximum=ddict["t"]-1, defaultValue=0)
            self.sliderT.slider.valueChanged.connect(self.updateTableWidget)
            self.layout.addWidget(self.sliderT)
        else:
            self.sliderT = None

        self.setLayout(self.layout)
        self.layout.setSizeConstraint(1)

        if title is not None:
            self.setWindowTitle(title)

        self.buttonSave.clicked.connect(self.saveFileResult)
        if stats:
            scrollbarTable = self.tableWidget.horizontalScrollBar()
            scrollbarStats = self.usefulInfoTableWidget.horizontalScrollBar()

            # Connect scrollbars
            scrollbarStats.valueChanged.connect(scrollbarTable.setValue)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        # self.resize(300*vrb.ratio,500*vrb.ratio)

        self.updateTableWidget()
        self.tableWidget.loadDictionary(resize=True)


    def resizeEvent(self,event):

        self.tableWidget.resizeInfoTable = False

        nbColumn = self.tableWidget.columnCount()
        if nbColumn >= 1:
            sizeColumn = (self.tableWidget.width()-10*vrb.ratio)/nbColumn
            sizeColumn = max(sizeColumn,120*vrb.ratio)

            for i in range(nbColumn):
                self.tableWidget.setColumnWidth(i, sizeColumn-1)
                self.tableWidget.setColumnWidth(i, sizeColumn)

        self.tableWidget.resizeInfoTable = True

        self.tableWidget.resizeUsefulInfoColumns()

        qt.QApplication.processEvents()

    def saveFileResult(self):

        filename = qt.QFileDialog.getSaveFileName(self, "Save file as", "", "(*.xls);;(*.csv)")
        if filename[1] == "(*.xls)":
            fct.convertToXlsFile(self.tableWidget, filename[0], title="Table", header=True)
        if filename[1] == "(*.csv)":
            fct.convertToCsvFile(self.tableWidget, filename[0])

    def updateTableWidget(self):

        if self.sliderZ is not None:
            zValue = self.sliderZ.slider.value()
        else:
            zValue = 0
        if self.sliderC is not None:
            cValue = self.sliderC.slider.value()
        else:
            cValue = 0
        if self.sliderT is not None:
            tValue = self.sliderT.slider.value()
        else:
            tValue = 0

        self.tableWidget.ddict = self.ddict["("+str(zValue)+","+str(cValue)+","+str(tValue)+")"]
        self.tableWidget.loadDictionary(resize=False)

    def display(self,xmlElement=None):

        fct.showWidget(self)

    def stopDisplay(self):

        self.close()


class GroupBoxTop(qt.QGroupBox):

    def __init__(self):
        qt.QGroupBox.__init__(self)

        self.buttonSave = wgt.PushButtonImage(vrb.folderImages + "/Save.png", margins=2)
        self.buttonSave.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)

        self.labelNumberLabels = qt.QLabel()
        self.labelNumberLabels.setFixedSize(200 * vrb.ratio, 30 * vrb.ratio)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.buttonSave, 0, 0, Qt.AlignLeft)
        self.layout.addWidget(self.labelNumberLabels, 0, 1)
        self.setLayout(self.layout)

        self.layout.setContentsMargins(0, 0, 0, 0)

class StatsTableWidget(qt.QTableWidget):
    def __init__(self,quantiles=[],quantilePonderated=False):
        super(StatsTableWidget, self).__init__()

        self.horizontalHeader().setVisible(False)
        self.verticalHeader().setVisible(False)
        self.setAlternatingRowColors(True)

        self.quantiles = quantiles
        self.quantilePonderated = quantilePonderated

        self.initTable()

    def initTable(self):

        self.clear()
        if self.quantilePonderated == False:
            self.setRowCount(5+len(self.quantiles))
        else:
            self.setRowCount(5+2*len(self.quantiles))
        self.setColumnCount(1)
        self.setItem(0, 0, qt.QTableWidgetItem('Min'))
        self.setItem(1, 0, qt.QTableWidgetItem('Max'))
        self.setItem(2, 0, qt.QTableWidgetItem('Mean'))
        self.setItem(3, 0, qt.QTableWidgetItem('Deviation'))
        self.setItem(4, 0, qt.QTableWidgetItem('Sum'))

        for i in range(len(self.quantiles)):
            self.setItem(5+i, 0, qt.QTableWidgetItem('Quantile ' + str(self.quantiles[i]) + "%"))

        if self.quantilePonderated:
            for j in range(len(self.quantiles)):
                self.setItem(5+i+j+1, 0, qt.QTableWidgetItem('Quantile ponderated ' + str(self.quantiles[j]) + "%"))

        self.setMaximumHeight(self.rowHeight(0) * self.rowCount() + self.horizontalScrollBar().height())
        self.horizontalHeader().setStretchLastSection(True)

    def addRecapColumn(self, values, col):

        color=randomPastelColor(col)
        colorDarker = color.darker(120)
        col = self.columnCount()
        self.setColumnCount(col + 1)
        mini = min(values)
        maxi = max(values)
        total = sum(values)
        avg = total / len(values)
        dev = np.std(values)

        cell = qt.QTableWidgetItem(fct.numberCalibration(mini))
        cell.setTextAlignment(Qt.AlignRight)
        self.setItem(0, col, cell)
        cell = qt.QTableWidgetItem(fct.numberCalibration(maxi))
        cell.setTextAlignment(Qt.AlignRight)
        self.setItem(1, col, cell)
        cell = qt.QTableWidgetItem(fct.numberCalibration(avg))
        cell.setTextAlignment(Qt.AlignRight)
        self.setItem(2, col, cell)
        cell = qt.QTableWidgetItem(fct.numberCalibration(dev))
        cell.setTextAlignment(Qt.AlignRight)
        self.setItem(3, col, cell)
        cell = qt.QTableWidgetItem(fct.numberCalibration(total))
        cell.setTextAlignment(Qt.AlignRight)
        self.setItem(4, col, cell)
        for i in range(len(self.quantiles)):
            quantile = np.quantile(values,self.quantiles[i]/100)
            cell = qt.QTableWidgetItem(fct.numberCalibration(quantile))
            cell.setTextAlignment(Qt.AlignRight)
            self.setItem(5+i, col, cell)

        if self.quantilePonderated:
            valuesSorted = np.sort(values)
            for j in range(len(self.quantiles)):
                quantile = self.computePonderatedQuantile(valuesSorted, self.quantiles[j] / 100)
                cell = qt.QTableWidgetItem(fct.numberCalibration(quantile))
                cell.setTextAlignment(Qt.AlignRight)
                self.setItem(5 + i + j + +1, col, cell)

        nbRow = 5+len(self.quantiles)
        if self.quantilePonderated:
            nbRow += len(self.quantiles)
        for row in range(nbRow):
            if self.item(row,col) is not None:
                self.item(row, col).setBackground((colorDarker, color)[row % 2 == 0])

    def computePonderatedQuantile(self,values,quantile):

        bornQuantile = sum(values)*quantile
        sumQuantile = 0

        for val in values:
            sumQuantile += val
            if sumQuantile >= bornQuantile:
                break

        return val

class TableWidget(qt.QTableWidget):

    def __init__(self, ddict={},parent = None,sorted=True,allowSort=True,headerHorizontal = True):
        super(TableWidget, self).__init__()

        self.ddict = ddict
        self.sorted = sorted
        self.currentColumnIndex = 0
        self.order = True
        self.allowSort = allowSort
        self.headerHorizontal = headerHorizontal

        self.resizeInfoTable = True

        self.parent = parent

        self.verticalHeader().setVisible(False)
        self.setAlternatingRowColors(True)
        # self.horizontalScrollBar().setVisible(False)
        # self.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)

        self.horizontalHeader().setStretchLastSection(True)
        if self.headerHorizontal == False:
            self.horizontalHeader().setFixedHeight(5*vrb.ratio)

        #self.itemPressed.connect(self.cellClick)
        # self.horizontalHeader().sectionResized.connect(self.resizeStatsColumns)
        self.horizontalHeader().sectionResized.connect(self.resizeUsefulInfoColumns)
        self.horizontalHeader().sectionClicked.connect(self.cellHeaderClicked)

        if self.ddict != {}:
            self.loadDictionary(resize=True)

        # self.resize(400*vrb.ratio,400*vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def display(self,xmlElement=None):

        fct.showWidget(self)

    def stopDisplay(self):

        self.close()

    def resizeUsefulInfoColumns(self):

        if self.resizeInfoTable and self.parent is not None and self.parent.usefulInfoTableWidget is not None:

            for i in range(self.columnCount()):
                width = self.columnWidth(i)
                self.parent.usefulInfoTableWidget.setColumnWidth(i, width)

    def cellHeaderClicked(self,numCol):

        if self.allowSort:
            if numCol == self.currentColumnIndex:
                self.order = not self.order
            else:
                self.currentColumnIndex = numCol

            self.loadDictionary()
            self.selectColumn(numCol)
            self.resizeUsefulInfoColumns()

    def loadDictionary(self,resize=False):

        self.ddict = fct.adjustDictForMeasure(self.ddict)

        self.clear()

        if self.ddict is not None and self.ddict != {}:
            try:
                names = []
                for name in self.ddict:
                    nbShapes = len(self.ddict[name])
                    names.append(name)
                if self.allowSort:
                    if self.order:
                        prefix = " ▼"
                    else:
                        prefix = " ▲"
                else:
                    prefix = ""

                try:
                    values = self.ddict[names[self.currentColumnIndex]]
                    values = np.asarray(values)
                    ind = np.argsort(values,kind='stable')
                    # ind+=1
                except:
                    traceback.print_exc(file=sys.stderr)
                    ind = range(nbShapes)

                ind = [int(i) for i in ind]

                if self.allowSort and self.order == False:
                    ind = ind[::-1]

                if self.parent is not None and self.parent.usefulInfoTableWidget is not None:
                    self.parent.usefulInfoTableWidget.initTable()

                nbCol = 0
                nbRow = 0
                for name in self.ddict:
                    if name not in ["Ref_Barycenter_X","Ref_Barycenter_Y","Ref_Barycenter_Z"]:
                        nbCol += 1
                        nbRow = max(nbRow,len(self.ddict[name]))
                self.setColumnCount(nbCol)
                self.setRowCount(nbRow)

                col = 0
                for name in self.ddict:

                    if name not in ["Ref_Barycenter_X","Ref_Barycenter_Y","Ref_Barycenter_Z"]:

                        nameHeader = name
                        # self.setRowCount(len(self.ddict[name]))

                        if self.parent is not None and self.parent.displayNumberOfElement:
                            self.parent.labelNumberLabels.setText("Number of elements : "+str(len(self.ddict[name])))

                        if self.headerHorizontal == False:
                            nameHeader = ""
                        elif col == self.currentColumnIndex:
                            nameHeader += prefix

                        cell = qt.QTableWidgetItem(nameHeader)
                        self.setHorizontalHeaderItem(col, cell)

                        columnColor = randomPastelColor(col)
                        columnColorDarker = columnColor.darker(120)
                        cell.setBackground(columnColor)

                        row = 0
                        # valuesApprox = []
                        # for i in range(len(self.ddict[name])):
                        for i in ind:

                            try:
                                if isinstance(self.ddict[name][i], type(True)):
                                    cell = qt.QTableWidgetItem(str(self.ddict[name][i]))
                                    # valuesApprox.append(self.ddict[name][i])
                                else:
                                    cell = qt.QTableWidgetItem(fct.numberCalibration(self.ddict[name][i]))
                                    # valuesApprox.append(float(fct.numberCalibration(self.ddict[name][i])))
                            except:
                                cell = qt.QTableWidgetItem(str(self.ddict[name][i]))
                                # valuesApprox.append(self.ddict[name][i])
                            cell.setTextAlignment(Qt.AlignRight)
                            cell.setBackground((columnColor, columnColorDarker)[row % 2 == 0])
                            self.setItem(row, col, cell)
                            row += 1
                        col += 1

                    if self.parent is not None and self.parent.usefulInfoTableWidget is not None\
                            and name not in ["Index","Ref_Barycenter_X","Ref_Barycenter_Y","Ref_Barycenter_Z"] and col>1:
                        # self.parent.usefulInfoTableWidget.addRecapColumn(valuesApprox,col+2)
                        self.parent.usefulInfoTableWidget.addRecapColumn(self.ddict[name],col+2)

            except:
                pass
                # traceback.print_exc(file=sys.stderr)

        if resize:
            if self.headerHorizontal:
                sizeHeader = 22*vrb.ratio
            else:
                sizeHeader = 5*vrb.ratio
            if self.parent is None:
                self.resize(min(1000*vrb.ratio,(self.columnCount()*120+10)*vrb.ratio),min(800*vrb.ratio,(self.rowCount()*22+10)*vrb.ratio+sizeHeader))
            else:
                # if isinstance(self.parent, TableWidgetWindowWithSliders):

                if self.parent.usefulInfoTableWidget is None:
                    self.parent.resize(min(1000*vrb.ratio,(self.columnCount()*120+30)*vrb.ratio),min(800*vrb.ratio,(self.rowCount()*22+60)*vrb.ratio+sizeHeader))
                else:
                    self.parent.resize(min(1000*vrb.ratio,(self.columnCount()*120+30)*vrb.ratio),min(800*vrb.ratio,(self.rowCount()*22+60)*vrb.ratio+self.parent.usefulInfoTableWidget.height()+sizeHeader))
        #
        #     # if self.parent is None:
        #     #     self.resize()
        #     self.resizeEvent(None)

class ConfusionMatrixWidget(qt.QWidget):
    def __init__(self):
        super(ConfusionMatrixWidget, self).__init__()

        self.confusionMatrixTable = ConfusionMatrixTable()
        self.metricsConfusionMatrixTable = MetricsConfusionMatrixTable()

        self.buttonSave = wgt.PushButtonImage(vrb.folderImages + "/Save.png", margins=2)
        self.buttonSave.setFixedSize(30*vrb.ratio, 30*vrb.ratio)

        self.labelInfo = qt.QLabel()
        self.labelInfo.setFixedSize(200*vrb.ratio, 30*vrb.ratio)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.buttonSave, 0, 0, Qt.AlignLeft)
        self.layout.addWidget(self.labelInfo, 0,1, Qt.AlignCenter)
        self.layout.addWidget(self.confusionMatrixTable, 1, 0,1,2,Qt.AlignCenter)
        self.layout.addWidget(self.metricsConfusionMatrixTable, 1, 2)

        self.setLayout(self.layout)

        self.layout.setHorizontalSpacing(20*vrb.ratio)
        self.layout.setContentsMargins(10*vrb.ratio, 10*vrb.ratio, 10*vrb.ratio, 10*vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonSave.clicked.connect(self.saveFile)

        self.setWindowTitle("Confusion matrix")

    def loadMatrix(self,ddict,confusionMatrix,title=""):

        sizeCells = 75*vrb.ratio

        self.labelInfo.setText(title)

        self.confusionMatrixTable.loadMatrix(ddict,confusionMatrix,sizeCells)
        self.metricsConfusionMatrixTable.calculateMetrics(confusionMatrix)

        nbElement = len(confusionMatrix)
        self.setFixedSize(sizeCells*(nbElement+1) + 250*vrb.ratio,sizeCells*(nbElement+1)+60*vrb.ratio)

    def saveFile (self):

        try:
            try:
                file = xmlet.parse(vrb.folderInformation + "/folderSaveConfusionMatrix.mho")
                folderElement = file.getroot()
            except:
                folderElement = xmlet.Element('folderSaveConfusionMatrix')

            path = Dfct.childText(folderElement, "Path")
            if path is not None:
                defaultFolder = path
            else:
                defaultFolder = "C:/"

            filename = qt.QFileDialog.getSaveFileName(self, "Save File", defaultFolder, "CSV Files (*.csv);;Excel Files (*.xls)")

            if filename[0] != [] and filename[0] != '' and filename[0] != None:

                Dfct.SubElement(folderElement, "Path").text = filename[0]
                Dfct.saveXmlElement(folderElement, vrb.folderInformation + "/folderSaveConfusionMatrix.mho")

                if filename[0].endswith('.csv'):
                    with open(filename[0], mode='w', newline='') as file:
                        writer = csv.writer(file)

                        for row in range(self.confusionMatrixTable.rowCount()):
                            row_data = []
                            for col in range(self.confusionMatrixTable.columnCount()):
                                item = self.confusionMatrixTable.item(row, col)
                                if item is not None:
                                    row_data.append(fct.numberCalibration(item.text()))
                                else:
                                    row_data.append('')
                            row_data.append('')
                            writer.writerow(row_data)

                        for i, (name, value) in enumerate(zip(self.metricsConfusionMatrixTable.listNames,
                                                              self.metricsConfusionMatrixTable.listValues)):
                            metric_row = [''] * self.confusionMatrixTable.columnCount()  # Colonne vide
                            metric_row.append('')
                            metric_row.append(name)
                            metric_row.append(fct.numberCalibration(value))
                            writer.writerow(metric_row)

                elif filename[0].endswith('.xls'):
                    workbook = xlwt.Workbook()
                    sheet = workbook.add_sheet('Confusion Matrix')

                    for row in range(self.confusionMatrixTable.rowCount()):
                        for col in range(self.confusionMatrixTable.columnCount()):
                            item = self.confusionMatrixTable.item(row, col)
                            if item is not None:
                                sheet.write(row, col, fct.numberCalibration(item.text()))
                            else:
                                sheet.write(row, col, '')
                        sheet.write(row, self.confusionMatrixTable.columnCount(), '')

                    nb_columns = self.confusionMatrixTable.columnCount()

                    start_col = nb_columns + 1
                    for i, (name, value) in enumerate(zip(self.metricsConfusionMatrixTable.listNames,
                                                          self.metricsConfusionMatrixTable.listValues)):
                        sheet.write(i, start_col, name)
                        sheet.write(i, start_col + 1, fct.numberCalibration(value))

                    workbook.save(filename[0])

        except:
            traceback.print_exc(file=sys.stderr)

class MetricsConfusionMatrixTable(qt.QTableWidget):
    def __init__(self):
        super().__init__(8, 2)

        self.setEditTriggers(qt.QAbstractItemView.NoEditTriggers)
        self.setSelectionMode(qt.QAbstractItemView.NoSelection)

        self.setHorizontalHeaderLabels(["Metric", "Value"])

        self.verticalHeader().setVisible(False)

        self.horizontalHeader().setSectionResizeMode(qt.QHeaderView.Stretch)
        self.verticalHeader().setSectionResizeMode(qt.QHeaderView.Stretch)

        font = QtGui.QFont()
        font.setBold(True)
        self.horizontalHeader().setFont(font)

        self.listNames = ["Accuracy (%)", "Precision", "Specificity", "F1-Score",
                          "False Positive Rate (FPR)", "False Negative Rate (FNR)",
                          "True Positive Rate (TPR)", "Matthews Correlation Coefficient (MCC)"]

    def calculateMetrics(self, cm):
        num_classes = cm.shape[0]

        VP_sum = FP_sum = FN_sum = VN_sum = 0

        for i in range(num_classes):
            VP = cm[i, i]
            FP = cm[:, i].sum() - VP
            FN = cm[i, :].sum() - VP
            VN = cm.sum() - (VP + FP + FN)

            VP_sum += VP
            FP_sum += FP
            FN_sum += FN
            VN_sum += VN

        accuracy = (((VP_sum + VN_sum) / cm.sum())/num_classes)*100 if cm.sum() != 0 else 0
        precision = VP_sum / (VP_sum + FP_sum) if (VP_sum + FP_sum) != 0 else 0
        recall = VP_sum / (VP_sum + FN_sum) if (VP_sum + FN_sum) != 0 else 0
        specificity = VN_sum / (VN_sum + FP_sum) if (VN_sum + FP_sum) != 0 else 0
        f1_score = 2 * (precision * recall) / (precision + recall) if (precision + recall) != 0 else 0
        fpr = FP_sum / (FP_sum + VN_sum) if (FP_sum + VN_sum) != 0 else 0
        fnr = FN_sum / (FN_sum + VP_sum) if (FN_sum + VP_sum) != 0 else 0
        tpr = recall
        mcc = ((VP_sum * VN_sum) - (FP_sum * FN_sum)) / np.sqrt((VP_sum + FP_sum) * (VP_sum + FN_sum) * (VN_sum + FP_sum) * (VN_sum + FN_sum)) if (VP_sum + FP_sum) * (VP_sum + FN_sum) * (VN_sum + FP_sum) * (VN_sum + FN_sum) != 0 else 0

        self.listValues = [accuracy, precision, specificity, f1_score, fpr, fnr, tpr, mcc]

        for i, value in enumerate(self.listValues):
            name_item = qt.QTableWidgetItem(self.listNames[i])
            value_item = qt.QTableWidgetItem(fct.numberCalibration(value))

            value_item.setTextAlignment(Qt.AlignCenter)

            if i % 2 == 0:
                bg_color = QtGui.QColor(255, 255, 255)
            else:
                bg_color = QtGui.QColor(220, 220, 220)

            name_item.setBackground(bg_color)
            value_item.setBackground(bg_color)

            self.setItem(i, 0, name_item)
            self.setItem(i, 1, value_item)

class ConfusionMatrixTable(qt.QTableWidget):
    def __init__(self):
        super(ConfusionMatrixTable, self).__init__()

        self.setEditTriggers(qt.QAbstractItemView.NoEditTriggers)
        self.setSelectionMode(qt.QAbstractItemView.NoSelection)

        self.confusionMatrix = None

        self.horizontalHeader().setVisible(False)
        self.verticalHeader().setVisible(False)

        self.horizontalHeader().setSectionResizeMode(qt.QHeaderView.Stretch)
        self.verticalHeader().setSectionResizeMode(qt.QHeaderView.Stretch)

        self.horizontalScrollBar().setVisible(False)
        self.verticalScrollBar().setVisible(False)

        self.setWindowTitle("Confusion matrix")

        self.setContentsMargins(0,0,0,0)

        self.setFixedSize(250*vrb.ratio,350*vrb.ratio)

    def paintEvent(self, event):
        super(ConfusionMatrixTable, self).paintEvent(event)

        painter = QtGui.QPainter(self.viewport())
        pen = QtGui.QPen(QtGui.QColor(0, 0, 0), 1)  # Créer une bordure noire de 2 pixels
        painter.setPen(pen)

        for row in range(self.rowCount()):
            for col in range(self.columnCount()):
                rect = self.visualItemRect(self.item(row, col))
                painter.drawRect(rect)

        painter.end()

    def loadMatrix(self,ddict,confusionMatrix,sizeCells):

        self.clear()

        self.confusionMatrix = confusionMatrix

        nbElement = len(self.confusionMatrix)

        self.setRowCount(nbElement+1)
        self.setColumnCount(nbElement+1)

        # for row in range(nbElement+1):
        #     for col in range(nbElement+1):
        #         self.setColumnWidth(col,sizeCells)
        #         self.setRowHeight(row,sizeCells)

        self.setFixedSize(sizeCells*(nbElement+1) ,sizeCells*(nbElement+1) )

        sum_rows = np.sum(self.confusionMatrix, axis=1)
        sum_columns = np.sum(self.confusionMatrix, axis=0)
        nbPixels = np.sum(self.confusionMatrix)

        for i in range(nbElement):
            color = ddict[i]["Color"]
            name = ddict[i]["Name"]

            cell = qt.QTableWidgetItem("Actual "+name + "\n(" + str(sum_rows[i]) + ")")
            cell.setForeground(color)
            cell.setBackground(greyColor)
            self.setItem(i+1, 0, cell)

            cell = qt.QTableWidgetItem("Predicted "+name + "\n(" + str(sum_columns[i]) + ")")
            cell.setForeground(color)
            cell.setBackground(greyColor)
            self.setItem(0, i+1, cell)

            for j in range(nbElement):
                cell = qt.QTableWidgetItem(str(self.confusionMatrix[i][j]))
                if j == i:
                    cell.setBackground(greenPastelColor)
                else:
                    cell.setBackground(redPastelColor)
                cell.setTextAlignment(Qt.AlignCenter)

                self.setItem(i+1, j+1, cell)

        cell = qt.QTableWidgetItem("Total pixels = "+str(nbPixels))
        cell.setBackground(greyColor)
        self.setItem(0, 0, cell)

class HistogramWidget(qt.QWidget):

    def __init__(self, listValues=[],tamis=[], nbBins = 10, cumul = False,title = None, xLegend = None, yLegend = None,numberOfDigits = None,bornMin=None,bornMax=None,sizeText=None,spacing = 1,hasSettings=True):
        super(HistogramWidget, self).__init__()

        self.hasSettings = hasSettings

        self.listValues = listValues
        self.tamis = tamis
        self.nbBins = nbBins
        self.cumul = cumul
        self.numberOfDigits = numberOfDigits
        self.bornMin = bornMin
        self.bornMax = bornMax
        self.sizeText = sizeText

        self.spacing = spacing

        self.title = title
        self.xLegend = xLegend
        self.yLegend = yLegend

        self.fig = Figure()
        self.plotWidget = FigureCanvas(self.fig)
        self.axes = self.fig.add_subplot(111)

        self.toolbar = NavigationToolbar(self.plotWidget, self)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.toolbar, 0, 0, 1, 2)
        self.layout.addWidget(self.plotWidget, 1, 0, 1, 2)

        if self.hasSettings:
            self.labelLegendSpacing = qt.QLabel('Legend spacing')
            self.labelLegendSpacing.setFixedSize(100*vrb.ratio, 30 * vrb.ratio)
            self.labelLegendSpacing.setAlignment(Qt.AlignBottom)
            self.sliderLegendSpacing = wgt.SliderShowingValue(1)
            self.sliderLegendSpacing.setFixedSize(130*vrb.ratio ,30 * vrb.ratio)

            self.layout.addWidget(self.labelLegendSpacing, 2, 0)
            self.layout.addWidget(self.sliderLegendSpacing, 2, 1,Qt.AlignBottom | Qt.AlignLeft)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10*vrb.ratio,10*vrb.ratio,10*vrb.ratio,10*vrb.ratio)

        #self.setStyleSheet("QWidget {background-color : rgb(220, 220, 220); }")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.computeChart()

        self.sliderLegendSpacing.slider.valueChanged.connect(self.computeChart)

    def setValues(self,listValues,tamis = None, nbBins = None):

        self.listValues = listValues
        if nbBins is not None:
            self.nbBins = nbBins
            self.tamis = None
        elif tamis is not None:
            self.nbBins = None
            self.tamis = tamis
        self.computeChart()

    def computeChart(self):

        try:
            if len(self.listValues) != 0:
                self.axes.clear()

                if self.title is not None:
                    self.axes.set_title(self.title)
                if self.xLegend is not None:
                    self.axes.set_xlabel(self.xLegend)
                if self.yLegend is not None:
                    self.axes.set_ylabel(self.yLegend)

                xValues = []
                barsetValues = []
                tick_label = []
                if self.tamis is not None and len(self.tamis) != 0:

                    if self.hasSettings:
                        legendSpacing = self.sliderLegendSpacing.slider.value()
                    else:
                        legendSpacing = 1

                    for i in range(len(self.tamis)-1):
                        xValues.append(i)
                        barsetValues.append(0)
                        if i%legendSpacing == 0:
                            tick_label.append(str(round(((self.tamis[i]+self.tamis[i+1])/2)*10)/10))
                        else:
                            tick_label.append("")

                    for value in self.listValues:
                        for i in range(len(self.tamis) - 1):
                            if self.cumul == False:
                                if i < len(self.tamis) - 2:
                                    if value >= self.tamis[i] and value < self.tamis[i + 1]:
                                        barsetValues[i] += 1
                                else:
                                    if value >= self.tamis[i] and value <= self.tamis[i + 1]:
                                        barsetValues[i] += 1
                            else:
                                if i < len(self.tamis) - 2:
                                    if value < self.tamis[i + 1]:
                                        barsetValues[i] += 1
                                else:
                                    if value <= self.tamis[i + 1]:
                                        barsetValues[i] += 1
                else:
                    if self.bornMin is None:
                        mini = min(self.listValues)
                    else:
                        mini = self.bornMin
                    if self.bornMax is None:
                        maxi = max(self.listValues)
                    else:
                        maxi = self.bornMax
                    correctedListValues = []
                    for val in self.listValues:
                        if val >= mini and val <= maxi:
                            correctedListValues.append(val)
                    total = len(correctedListValues)

                    for i in range(self.nbBins):

                        xValues.append(i)
                        barsetValues.append(0)
                        if i % self.spacing == 0 or i == self.nbBins - 1:
                            tick_label.append(fct.numberCalibration(mini + i * (maxi - mini) / (self.nbBins - 1),self.numberOfDigits))
                        else:
                            tick_label.append("")

                    for value in correctedListValues:

                        if maxi != mini:
                            index = round(((value - mini) * (self.nbBins - 1)) / (maxi - mini))
                        else:
                            index = 0

                        if self.cumul == False:
                            barsetValues[index] += 1
                        else:
                            for j in range(index, self.nbBins):
                                barsetValues[j] += 1

                    if self.cumul:
                        for i in range(self.nbBins):
                            barsetValues[i] = barsetValues[i] * 100 / total

                self.axes.bar(xValues, barsetValues, tick_label=tick_label, color="blue")
                if self.sizeText is not None:
                    self.axes.tick_params(labelsize=self.sizeText)
                self.plotWidget.draw()

        except:
            traceback.print_exc(file=sys.stderr)


# class SimpleHistogramWidget(qt.QWidget):
#
#     def __init__(self, listValues=[],nbBins=10,title = None, xLegend = None, yLegend = None):
#         super(SimpleHistogramWidget, self).__init__()
#
#         self.spacing = 1
#         self.cumul = False
#
#         self.title = title
#         self.xLegend = xLegend
#         self.yLegend = yLegend
#
#         self.listValues = listValues
#         self.nbBins = nbBins
#
#         self.fig = Figure()
#         self.plotWidget = FigureCanvas(self.fig)
#         self.axes = self.fig.add_subplot(111)
#
#         self.toolbar = NavigationToolbar(self.plotWidget, self)
#         # self.toolbar.setFixedWidth(400 * vrb.ratio)
#
#         self.layout = qt.QGridLayout()
#         self.layout.addWidget(self.toolbar, 0, 0)
#         self.layout.addWidget(self.plotWidget, 1, 0)
#
#         self.setLayout(self.layout)
#
#         self.setStyleSheet("QWidget {background-color : rgb(220, 220, 220); }")
#
#         self.computeChart()
#
#     def setValues(self,listValues,nbBins):
#
#         self.listValues = listValues
#         self.nbBins = nbBins
#         self.computeChart()
#
#     def computeChart(self):
#
#         try:
#             self.axes.clear()
#
#             if self.title is not None:
#                 self.axes.set_title(self.title)
#             if self.xLegend is not None:
#                 self.axes.set_xlabel(self.xLegend)
#             if self.yLegend is not None:
#                 self.axes.set_ylabel(self.yLegend)
#
#             xValues = []
#             barsetValues = []
#             tick_label = []
#
#             mini = min(self.listValues)
#             maxi = max(self.listValues)
#             total = len(self.listValues)
#
#             for i in range(self.nbBins):
#
#                 xValues.append(i)
#                 barsetValues.append(0)
#                 if i%self.spacing == 0:
#                     tick_label.append(fct.numberCalibration(mini + i*(maxi-mini)/(self.nbBins-1)))
#                 else:
#                     tick_label.append("")
#
#             for value in self.listValues:
#
#                 if maxi != mini:
#                     index = round(((value-mini)*(self.nbBins-1))/(maxi-mini))
#                 else:
#                     index = 0
#
#                 if self.cumul == False:
#                     barsetValues[index] += 1
#                 else:
#                     for j in range(index,self.nbBins):
#                         barsetValues[j] += 1
#
#             if self.cumul:
#                 for i in range(self.nbBins):
#                     barsetValues[i] = barsetValues[i]*100/total
#
#             self.axes.bar(xValues, barsetValues, tick_label=tick_label, color="blue")
#             self.plotWidget.draw()
#
#         except:
#             pass
#             # import traceback
#             # traceback.print_exc(file=sys.stderr)

if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    ddict = {}
    ddict["x"] = [0,0,0,0,10,10,10,10,20,20,20,20]
    ddict["y"] = [0,10,20,30,0,10,20,30,0,10,20,30]
    ddict["dx"] = [5,6,7,8,9,10,11,12,13,21,14,1]
    ddict["dy"] = [5,8,17,83,90,0,11,102,111,115,5,0.5]
    # ddict["aeeeaaa"] = [0,1,2,3,4,5,6,7,8,9]
    # ddict["fffffff"] = [5,6,7,8,9,10,11,12,1,143]


    # ddict = {}
    # ddict["aaa"] = "800"
    # ddict["bbb"] = "name"

    # TableWidgetWindow(ddict=None, stats=False, title=None, quantiles=[],quantilePonderated=False, sorted=True):
    foo = TableWidgetWindow(ddict,stats=True)
    # foo = TableWidget(ddict,headerHorizontal=True)

    # listValues = [2,5,10,3,8,1,2,2,3,9,11,14,23,22]
    # tamis = [3,5,7,11,20]
    # foo = HistogramWidget(listValues,tamis)

    foo.show()

    app.exec_()