import os
import sys, traceback
import time
import webbrowser

from PyQt5.QtCore import pyqtSignal, Qt, QStringListModel,QRect,QPoint
from PyQt5 import QtGui
from PyQt5.Qsci import QsciScintilla, QsciLexerPython,QsciAPIs,QsciLexerCustom

import PyQt5.QtWidgets as qt


import UsefullVariables as vrb
import AutoCompletion as auto
import DatabaseFunction as Dfct
import UsefullFunctions as fct

import jedi
import keyword
import re
import xml.etree.ElementTree as xmlet

from PyQt5.QtCore import Qt,QTimer

class SearchDialog(qt.QFrame):
    def __init__(self, editor):
        super().__init__(editor)
        self.setFrameShape(qt.QFrame.Box)
        self.setFrameShadow(qt.QFrame.Plain)
        self.editor = editor
        self.setFixedHeight(40)

        self.searchInput = qt.QLineEdit(self)
        self.searchInput.setPlaceholderText("Rechercher...")
        self.searchInput.textChanged.connect(self.updateHighlights)

        self.findNextButton = qt.QToolButton(self)
        self.findNextButton.setArrowType(Qt.DownArrow)
        self.findPreviousButton = qt.QToolButton(self)
        self.findPreviousButton.setArrowType(Qt.UpArrow)

        self.closeLabel = qt.QLabel("X", self)
        self.closeLabel.setStyleSheet("color: #666; font-size: 14px; font-weight: lighter; padding-left: 5px;")
        self.closeLabel.setAlignment(Qt.AlignCenter)
        self.closeLabel.setCursor(Qt.PointingHandCursor)

        self.closeLabel.mousePressEvent = self.hideOnClick
        self.findNextButton.clicked.connect(self.findNext)
        self.findPreviousButton.clicked.connect(self.findPrevious)

        # Mise en page
        layout = qt.QHBoxLayout()
        layout.addWidget(self.searchInput)
        layout.addWidget(self.findPreviousButton)
        layout.addWidget(self.findNextButton)
        layout.addWidget(self.closeLabel)
        layout.setContentsMargins(5, 5, 5, 5)
        self.setLayout(layout)

    def updateHighlights(self):
        searchText = self.searchInput.text()
        if len(searchText)>=1:
            self.editor.highlightAllOccurrences(searchText)
        else:
            self.editor.clearHighlights(clear="Search")

    def findNext(self):
        text = self.searchInput.text()
        if text:
            self.editor.findNext(text)

    def findPrevious(self):
        text = self.searchInput.text()
        if text:
            self.editor.findPrevious(text)

    def hideOnClick(self, event):
        """Cache la barre de recherche quand on clique sur la croix."""
        self.hide()
        self.editor.clearHighlights(clear="Search")

    def keyPressEvent(self, event):
        if event.key() in (Qt.Key_Return, Qt.Key_Enter, Qt.Key_Down):
            self.findNext()
        elif event.key() == Qt.Key_Up:
            self.findPrevious()
        else:
            super().keyPressEvent(event)

class TextEditor(QsciScintilla):

    def __init__(self, parent=None):
        super(TextEditor, self).__init__(parent)

        self.timer = QTimer(self)
        self.timer.setInterval(500)
        self.syntaxTimer = time.time()
        self.timeBeforeActualize = 3

        self.syntaxToCheck = False
        self.textHasChanged = False

        self.ctrlPressed = False
        self.shiftPressed = False

        self.lexer = QsciLexerPython()

        self.lexer.setDefaultColor(QtGui.QColor("#d4d4d4"))
        self.lexer.setDefaultPaper(QtGui.QColor("#474747"))
        self.lexer.setDefaultFont(QtGui.QFont("Consolas", 12))
        self.lexer.setColor(QtGui.QColor("#7AA6DA"), QsciLexerPython.Keyword)

        self.setLexer(self.lexer)

        self.setMarginType(0, QsciScintilla.NumberMargin)
        self.setMarginWidth(0, "0000")
        self.setMarginsFont(QtGui.QFont("Courier", 10))

        self.setAutoIndent(True)
        self.setIndentationsUseTabs(True)
        self.setTabWidth(4)

        self.setCaretForegroundColor(QtGui.QColor("white"))
        self.SendScintilla(QsciScintilla.SCI_SETCARETSTYLE, QsciScintilla.CARETSTYLE_LINE)
        self.SendScintilla(QsciScintilla.SCI_SETCARETWIDTH, 2)
        self.setCaretLineVisible(True)
        self.setCaretLineBackgroundColor(QtGui.QColor("#5a5a5a"))

        font_strings = QtGui.QFont("Consolas")
        font_strings.setItalic(False)
        self.lexer.setFont(font_strings, QsciLexerPython.DoubleQuotedString)
        self.lexer.setFont(font_strings, QsciLexerPython.SingleQuotedString)
        self.lexer.setFont(font_strings, QsciLexerPython.TripleDoubleQuotedString)
        self.lexer.setFont(font_strings, QsciLexerPython.TripleSingleQuotedString)
        self.lexer.setFont(font_strings, QsciLexerPython.UnclosedString)
        self.lexer.setColor(QtGui.QColor("#228B22"), QsciLexerPython.DoubleQuotedString)
        self.lexer.setColor(QtGui.QColor("#228B22"), QsciLexerPython.SingleQuotedString)
        self.lexer.setColor(QtGui.QColor("#228B22"), QsciLexerPython.TripleDoubleQuotedString)
        self.lexer.setColor(QtGui.QColor("#228B22"), QsciLexerPython.TripleSingleQuotedString)
        self.lexer.setColor(QtGui.QColor("#228B22"), QsciLexerPython.UnclosedString)

        font_comments = QtGui.QFont("Consolas", 12)
        font_comments.setItalic(False)
        self.lexer.setFont(font_comments, QsciLexerPython.Comment)
        self.lexer.setColor(QtGui.QColor("#8C8C8C"), QsciLexerPython.Comment)

        font_keywords = QtGui.QFont("Consolas", 12)
        self.lexer.setFont(font_keywords, QsciLexerPython.Keyword)
        self.lexer.setColor(QtGui.QColor("#FFA500"), QsciLexerPython.Keyword)

        font_numbers = QtGui.QFont("Consolas", 12)
        self.lexer.setFont(font_numbers, QsciLexerPython.Number)
        self.lexer.setColor(QtGui.QColor("#C678DD"), QsciLexerPython.Number)

        font_functions = QtGui.QFont("Consolas", 12)
        self.lexer.setFont(font_functions, QsciLexerPython.FunctionMethodName)
        self.lexer.setFont(font_functions, QsciLexerPython.ClassName)
        self.lexer.setColor(QtGui.QColor("#57ddb5"), QsciLexerPython.FunctionMethodName)
        self.lexer.setColor(QtGui.QColor("#57ddb5"), QsciLexerPython.ClassName)

        self.errorIndicator = 1
        self.warningIndicator = 2
        self.searchIndicator = 3
        self.keywordOrangeIndicator = 4
        self.keywordGreenIndicator = 5

        self.keywordsOrange = ["True","False"]
        self.keywordsGreen = auto.modulesAndFunctionsList

        self.api = QsciAPIs(self.lexer)
        self.setAutoCompletionSource(QsciScintilla.AcsAll)
        self.setAutoCompletionThreshold(1)
        self.setAutoCompletionCaseSensitivity(False)

        self.api.prepare()

        self.searchDialog = SearchDialog(self)
        self.searchDialog.hide()

        self.searchDialog.move(self.width() - self.searchDialog.width() - 10, 10)

        self.textChanged.connect(self.connectTextChanged)
        self.timer.timeout.connect(self.timerActualization)
        self.timer.start()

        self.setupIndicators()

        self.resize(500*vrb.ratio,500*vrb.ratio)

    def timerActualization(self):

        self.syntaxToCheck = True
        self.checkSyntax(fromTimer=True)

    def highlightUnclosedStrings(self):
        for line_number in range(self.lines()):
            line_text = self.text(line_number)

            if '"' in line_text or "'" in line_text:
                quote_pos = line_text.find('"') if '"' in line_text else line_text.find("'")
                closing_quote_pos = line_text.find(line_text[quote_pos], quote_pos + 1)

                if closing_quote_pos == -1:
                    start_pos = self.positionFromLineIndex(line_number, quote_pos)
                    end_pos = self.positionFromLineIndex(line_number, len(line_text))

                    self.SendScintilla(QsciScintilla.SCI_STARTSTYLING, start_pos)
                    self.SendScintilla(QsciScintilla.SCI_SETSTYLING, end_pos - start_pos, self.unclosedStringStyle)

    def setupIndicators(self):

        self.SendScintilla(QsciScintilla.SCI_INDICSETSTYLE, self.errorIndicator, QsciScintilla.INDIC_SQUIGGLE)
        redColor = QtGui.QColor("red")
        rgbValue = self.rgbToBgr(redColor)
        self.SendScintilla(QsciScintilla.SCI_INDICSETFORE, self.errorIndicator, rgbValue)

        self.SendScintilla(QsciScintilla.SCI_INDICSETSTYLE, self.warningIndicator, QsciScintilla.INDIC_SQUIGGLE)
        blueColor = QtGui.QColor("yellow")
        rgbValue = self.rgbToBgr(blueColor)
        self.SendScintilla(QsciScintilla.SCI_INDICSETFORE, self.warningIndicator, rgbValue)

        self.SendScintilla(QsciScintilla.SCI_INDICSETSTYLE, self.searchIndicator, QsciScintilla.INDIC_ROUNDBOX)
        highlightColor = QtGui.QColor("yellow")
        rgbValue = (highlightColor.red() << 16) | (highlightColor.green() << 8) | highlightColor.blue()
        self.SendScintilla(QsciScintilla.SCI_INDICSETFORE, self.searchIndicator, rgbValue)

        self.SendScintilla(QsciScintilla.SCI_INDICSETSTYLE, self.keywordOrangeIndicator, QsciScintilla.INDIC_TEXTFORE)
        orangeColor = QtGui.QColor("#FFA500")
        rgbValue = (orangeColor.blue() << 16) | (orangeColor.green() << 8) | orangeColor.red()
        self.SendScintilla(QsciScintilla.SCI_INDICSETFORE, self.keywordOrangeIndicator, rgbValue)

        self.SendScintilla(QsciScintilla.SCI_INDICSETSTYLE, self.keywordGreenIndicator, QsciScintilla.INDIC_TEXTFORE)
        greenColor = QtGui.QColor("#3498db")
        rgbValue = (greenColor.blue() << 16) | (greenColor.green() << 8) | greenColor.red()
        self.SendScintilla(QsciScintilla.SCI_INDICSETFORE, self.keywordGreenIndicator, rgbValue)

    def highlightKeywords(self):

        start = time.time()

        self.clearHighlights(clear="Keywords")

        self.applyHighlight(self.keywordsOrange, self.keywordOrangeIndicator)
        self.applyHighlight(self.keywordsGreen, self.keywordGreenIndicator)

        # Force une restylisation complète pour synchroniser lexer et indicateurs
        self.SendScintilla(QsciScintilla.SCI_COLOURISE, 0, -1)

        # print('highlightKeywords',time.time()-start)

    def applyHighlight(self, keywords, indicator):
        self.SendScintilla(QsciScintilla.SCI_SETINDICATORCURRENT, indicator)

        fullText = self.getTextForHighlights()  # Récupère le texte complet
        for keyword in keywords:
            start = 0
            while True:
                start = fullText.find(keyword, start)
                if start == -1:
                    break

                end = start + len(keyword)

                inStringOrComment = False
                for pos in range(start, end):
                    style = self.SendScintilla(QsciScintilla.SCI_GETSTYLEAT, pos)
                    if style in [
                        QsciLexerPython.Comment,
                        QsciLexerPython.SingleQuotedString,
                        QsciLexerPython.DoubleQuotedString,
                        QsciLexerPython.TripleSingleQuotedString,
                        QsciLexerPython.TripleDoubleQuotedString
                    ]:
                        inStringOrComment = True
                        break

                if not inStringOrComment:
                    byte_start = self.positionFromLineIndex(0, start)
                    byte_length = len(keyword)
                    self.SendScintilla(QsciScintilla.SCI_INDICATORFILLRANGE, byte_start, byte_length)

                start += len(keyword)

    def getText(self):
        length = self.SendScintilla(QsciScintilla.SCI_GETTEXTLENGTH) + 1
        textBuffer = bytes(length)
        self.SendScintilla(QsciScintilla.SCI_GETTEXT, length, textBuffer)
        decodedText = textBuffer.decode('utf-8').rstrip('\x00')
        cleanText = re.sub(r'[^\x20-\x7E\t\n]', '', decodedText)

        return cleanText

    def getTextForHighlights(self):

        length = self.SendScintilla(QsciScintilla.SCI_GETTEXTLENGTH) + 1
        textBuffer = bytes(length)
        self.SendScintilla(QsciScintilla.SCI_GETTEXT, length, textBuffer)
        # cleanText = textBuffer.decode('utf-8')
        cleanText = textBuffer.decode('utf-8').rstrip('\x00')
        cleanText = cleanText.replace('\r\n', '\n').replace('\r', '\n')

        return cleanText

    def showSearchDialog(self):

        selectedText = self.selectedText().strip()  # Trim pour éviter les espaces blancs au début/fin

        if selectedText:
            self.searchDialog.searchInput.setText(selectedText)

        fct.showWidget(self.searchDialog)
        self.searchDialog.searchInput.setFocus()

        self.searchDialog.move(self.width() - self.searchDialog.width() - 10, 10)

        self.searchDialog.updateHighlights()

    def highlightAllOccurrences(self, text):

        self.clearHighlights(clear="Search")

        posRef = self.SendScintilla(QsciScintilla.SCI_GETCURRENTPOS)

        endPos = self.SendScintilla(QsciScintilla.SCI_GETTEXTLENGTH)
        pos = 0
        self.SendScintilla(QsciScintilla.SCI_GOTOPOS)

        while pos < endPos:
            self.SendScintilla(QsciScintilla.SCI_SEARCHANCHOR)

            pos = self.SendScintilla(QsciScintilla.SCI_SEARCHNEXT, 0, text.encode('utf-8'))

            if pos == -1:
                break

            self.SendScintilla(QsciScintilla.SCI_INDICATORFILLRANGE, pos, len(text))
            self.SendScintilla(QsciScintilla.SCI_GOTOPOS, pos + len(text))

        self.SendScintilla(QsciScintilla.SCI_GOTOPOS, posRef)

    def clearHighlights(self,clear = "All"):

        if clear == "Syntax":
            indicators = [self.errorIndicator,self.warningIndicator]
        elif clear == "Search":
            indicators = [self.searchIndicator]
        elif clear == "Keywords":
            indicators = [self.keywordOrangeIndicator,self.keywordGreenIndicator]
        elif clear == "All":
            [self.errorIndicator, self.warningIndicator,self.searchIndicator,self.keywordOrangeIndicator,self.keywordGreenIndicator]

        for indicator in indicators:
            self.SendScintilla(QsciScintilla.SCI_SETINDICATORCURRENT, indicator)
            self.SendScintilla(QsciScintilla.SCI_INDICATORCLEARRANGE, 0,self.SendScintilla(QsciScintilla.SCI_GETTEXTLENGTH))

    def resizeEvent(self, event):
        super(TextEditor, self).resizeEvent(event)
        self.searchDialog.move(self.width() - self.searchDialog.width() - 10, 10)

    def updateCompleter(self):

        self.api.clear()

        pythonKeywords = keyword.kwlist
        for kw in pythonKeywords:
            self.api.add(kw)

        for word in auto.modulesList:
            self.api.add(word)

        currentPos = self.getCursorPosition()
        currentLine = self.text(currentPos[0])
        cursorIndex = currentPos[1]

        wordStartPos = cursorIndex
        while wordStartPos > 0 and (currentLine[wordStartPos - 1].isalnum() or currentLine[wordStartPos - 1] == "."):
            wordStartPos -= 1
        currentWord = currentLine[wordStartPos:cursorIndex].strip()
        if currentWord.split(".")[0] in auto.functionsDict:
            for word in auto.functionsDict[currentWord.split(".")[0]]:
                self.api.add(word)

        self.api.prepare()

    def getDefinedVariables(self, script):

        definedNames = script.get_names(all_scopes=True, definitions=True)

        variables = [
            name.name for name in definedNames
            if 'def ' not in name.description and 'class ' not in name.description
        ]

        return variables

    def keyPressEvent(self, event):

        self.syntaxToCheck = False

        key = event.key()

        if key == 16777249:
            self.ctrlPressed = True
        if key == 16777248:
            self.shiftPressed = True

        if event.key() == Qt.Key_F and self.ctrlPressed:
            self.showSearchDialog()
            return

        # Capture du raccourci "Ctrl + /" pour commenter/décommenter
        if event.key() == Qt.Key_Slash and self.ctrlPressed:
            # self.syntaxToCheck = True
            self.commentOrUncomment()
            self.syntaxToCheck = True
            self.checkSyntax()
            self.highlightKeywords()
            return

        if event.key() == Qt.Key_Z and self.ctrlPressed:
            # self.syntaxToCheck = True
            if self.shiftPressed:
                self.redo()
            else:
                self.undo()
            self.syntaxToCheck = True
            self.checkSyntax()
            return

        if key == Qt.Key_Up and self.ctrlPressed and self.shiftPressed:
            # self.syntaxToCheck = True
            self.swapLineWithAbove()
            self.syntaxToCheck = True
            self.checkSyntax()
            return

        if key == Qt.Key_Down and self.ctrlPressed and self.shiftPressed:
            # self.syntaxToCheck = True
            self.swapLineWithBelow()
            self.syntaxToCheck = True
            self.checkSyntax()
            return

        if event.matches(QtGui.QKeySequence.Paste):
            self.syntaxToCheck = True
            self.handlePaste()
            return

        if event.key() in (Qt.Key_Return, Qt.Key_Enter):
            self.syntaxToCheck = True

            self.SendScintilla(QsciScintilla.SCI_BEGINUNDOACTION)
            try:
                self.setCaretLineVisible(False)
                super(TextEditor, self).keyPressEvent(event)
                self.setCaretLineVisible(True)
            finally:
                self.SendScintilla(QsciScintilla.SCI_ENDUNDOACTION)
            return

        if event.key() in (Qt.Key_Backspace, Qt.Key_Delete, Qt.Key_Space,Qt.Key_Tab, Qt.Key_Backtab):
            self.syntaxToCheck = True

        if event.key() in (Qt.Key_Up, Qt.Key_Down, Qt.Key_Left, Qt.Key_Right):
            self.syntaxToCheck = True
            self.checkSyntax()

        super(TextEditor, self).keyPressEvent(event)

        if self.shouldShowAutocompletion():
            self.updateCompleter()
            self.autoCompleteFromAPIs()

        # print(time.time()-start,event.key())

    def keyReleaseEvent(self, event):

        key = event.key()

        if key == 16777249:
            self.ctrlPressed = False
        if key == 16777248:
            self.shiftPressed = False

    def mousePressEvent(self, event):

        if self.ctrlPressed:
            x, y = event.x(), event.y()
            charPos = self.SendScintilla(QsciScintilla.SCI_POSITIONFROMPOINT, x, y)
            if charPos == -1:
                return

            line = self.SendScintilla(QsciScintilla.SCI_LINEFROMPOSITION, charPos)
            index = charPos - self.SendScintilla(QsciScintilla.SCI_POSITIONFROMLINE, line)

            word = self.wordAt(line, index)

            self.onWordClicked(word)

        else:
            self.onClick()
            super(TextEditor, self).mousePressEvent(event)

    def wordAt(self, line, index):

        lineLength = self.SendScintilla(QsciScintilla.SCI_LINELENGTH, line)

        buffer = bytearray(lineLength)
        self.SendScintilla(QsciScintilla.SCI_GETLINE, line, buffer)

        lineText = buffer.decode("utf-8").rstrip("\x00")

        start = index
        while start > 0 and (lineText[start - 1].isalnum() or lineText[start - 1] == "_"):
            start -= 1

        end = index
        while end < len(lineText) and (lineText[end].isalnum() or lineText[end] == "_"):
            end += 1

        return lineText[start:end]

    def onWordClicked(self, word):

        if word in auto.dictImport:
            importStatement = auto.dictImport[word]

            fullText = self.getText()
            lines = fullText.splitlines()

            if any(line.strip() == importStatement for line in lines):
                return

            currentLine, currentIndex = self.getCursorPosition()

            for i in range(len(lines)):
                if lines[i].startswith("import PyIPSDK"):
                    insertLine = i + 1
                    self.setCursorPosition(insertLine, 0)
                    if lines[i+1].strip() == "" or  lines[i+1].startswith("import PyIPSDK"):
                        self.insert(importStatement + "\n")
                        if currentLine >= insertLine:
                            currentLine += 1
                    else:
                        self.insert(importStatement + "\n\n")
                        if currentLine >= insertLine:
                            currentLine += 2
                    self.setCursorPosition(currentLine, currentIndex)
                    return

            self.setCursorPosition(0, 0)
            if len(lines) > 0 and lines[0].strip() == "":
                self.insert(importStatement + "\n")
                currentLine += 1
            else:
                self.insert(importStatement + "\n\n")
                currentLine += 2
            self.setCursorPosition(currentLine, currentIndex)

            self.syntaxToCheck = True
            self.checkSyntax()

        if word in auto.functionsList:

            docPath = None
            for file in os.listdir(vrb.folderFunctions):
                folder = vrb.folderFunctions + "/" + file
                mhoFile = folder + "/" + word + ".mho"
                if os.path.exists(mhoFile):
                    xmlFile = xmlet.parse(mhoFile)
                    xmlElement = xmlFile.getroot()
                    docTitle = Dfct.childText(xmlElement, 'DocTitle')
                    docPath = vrb.dictLinks[docTitle]

            if docPath is not None:
                pagePath = vrb.folderDocEssential + '/' + docPath
                webbrowser.open_new_tab(pagePath)

    def onClick(self):

        self.syntaxToCheck = True
        self.checkSyntax()
        self.highlightKeywords()

    def findNext(self, text):

        cursorPos = self.SendScintilla(QsciScintilla.SCI_GETCURRENTPOS)

        self.SendScintilla(QsciScintilla.SCI_SETSEL, cursorPos, cursorPos)

        self.SendScintilla(QsciScintilla.SCI_SEARCHANCHOR)

        searchFlags = 0
        pos = self.SendScintilla(QsciScintilla.SCI_SEARCHNEXT, searchFlags, text.encode('utf-8'))

        if pos == -1:
            self.SendScintilla(QsciScintilla.SCI_GOTOPOS, 0)
        else:
            self.SendScintilla(QsciScintilla.SCI_GOTOPOS, pos + len(text))
            self.SendScintilla(QsciScintilla.SCI_SETSEL, pos, pos + len(text))

    def findPrevious(self, text):

        # pass

        self.SendScintilla(QsciScintilla.SCI_SEARCHANCHOR)

        searchFlags = 0

        pos = self.SendScintilla(QsciScintilla.SCI_SEARCHPREV, searchFlags, text.encode('utf-8'))

        if pos == -1:
            endPos = self.SendScintilla(QsciScintilla.SCI_GETTEXTLENGTH)
            self.SendScintilla(QsciScintilla.SCI_GOTOPOS, endPos)

    def swapLineWithAbove(self):

        currentLine, currentIndex = self.getCursorPosition()

        if currentLine == 0:
            return

        self.SendScintilla(QsciScintilla.SCI_BEGINUNDOACTION)

        try:
            currentLineText = self.text(currentLine).rstrip()
            aboveLineText = self.text(currentLine - 1).rstrip()

            self.setSelection(currentLine - 1, 0, currentLine - 1, len(aboveLineText))
            self.replaceSelectedText(currentLineText)

            self.setSelection(currentLine, 0, currentLine, len(currentLineText))
            self.replaceSelectedText(aboveLineText)

            self.setCursorPosition(currentLine - 1, currentIndex)
        finally:
            self.SendScintilla(QsciScintilla.SCI_ENDUNDOACTION)

    def swapLineWithBelow(self):
        currentLine, currentIndex = self.getCursorPosition()

        if currentLine == self.lines() - 1:
            return

        self.SendScintilla(QsciScintilla.SCI_BEGINUNDOACTION)

        try:
            currentLineText = self.text(currentLine).rstrip()
            belowLineText = self.text(currentLine + 1).rstrip()

            self.setSelection(currentLine + 1, 0, currentLine + 1, len(belowLineText))
            self.replaceSelectedText(currentLineText)

            self.setSelection(currentLine, 0, currentLine, len(currentLineText))
            self.replaceSelectedText(belowLineText)

            self.setCursorPosition(currentLine + 1, currentIndex)

        finally:
            self.SendScintilla(QsciScintilla.SCI_ENDUNDOACTION)

    def shouldShowAutocompletion(self):

        currentPos = self.getCursorPosition()
        currentLine = self.text(currentPos[0])

        cursorIndex = currentPos[1]

        if cursorIndex < len(currentLine) and currentLine[cursorIndex].isalnum():
            cursorIndex += 1

        wordStartPos = cursorIndex
        while wordStartPos > 0 and currentLine[wordStartPos - 1].isalnum():
            wordStartPos -= 1

        currentWord = currentLine[wordStartPos:cursorIndex].strip()

        if len(currentWord) < 1:
            return False

        pythonKeywords = keyword.kwlist
        for kw in pythonKeywords:
            if kw.startswith(currentWord):
                return True

        script = jedi.Script(self.getCleanCode())
        variables = self.getDefinedVariables(script)

        for var in variables:
            if var.startswith(currentWord) and var != currentWord:
                return True

        return False

    def commentOrUncomment(self):

        cursor = self.getCursorPosition()

        if self.hasSelectedText():
            startLine, startIndex, endLine, endIndex = self.getSelection()
        else:
            startLine, endLine = cursor[0], cursor[0]

        self.SendScintilla(QsciScintilla.SCI_BEGINUNDOACTION)

        try:
            isCommenting = False
            for line in range(startLine, endLine + 1):
                if not self.text(line).strip().startswith("#"):
                    isCommenting = True
                    break

            for line in range(startLine, endLine + 1):
                lineText = self.text(line).rstrip()  # Supprimer les espaces à droite

                self.setSelection(line, 0, line, len(lineText))

                if isCommenting:
                    self.replaceSelectedText("#" + lineText)
                else:
                    # Supprimer uniquement le premier caractère "#" s'il est présent
                    if lineText.lstrip().startswith("#"):
                        uncommentedLine = lineText.lstrip()[1:]  # Supprimer le premier caractère "#"
                        # Remettre les espaces en début de ligne pour conserver l'indentation
                        leadingSpaces = len(lineText) - len(lineText.lstrip())
                        uncommentedLine = " " * leadingSpaces + uncommentedLine
                        self.replaceSelectedText(uncommentedLine)
        finally:
            self.SendScintilla(QsciScintilla.SCI_ENDUNDOACTION)

    def handlePaste(self,text=None):

        if text is None:
            clipboard = qt.QApplication.clipboard()
            text = clipboard.text()

        if self.hasSelectedText():
            self.removeSelectedText()

        currentLine, currentIndex = self.getCursorPosition()

        currentLineText = self.text(currentLine)

        currentIndentation = currentLineText.count('\t', 0, len(currentLineText) - len(currentLineText.lstrip()))

        text = text.replace("    ", "\t")  # Optionnel : transformer 4 espaces en une tabulation

        textLines = text.splitlines()
        minIndentation = float('inf')
        for line in textLines:
            strippedLine = line.lstrip()
            if strippedLine:  # Ignorer les lignes vides
                lineIndentation = len(line) - len(strippedLine)
                minIndentation = min(minIndentation, lineIndentation)
        if minIndentation == float('inf'):
            minIndentation = 0

        newText = ""
        firstLine = True
        for line in textLines:
            if newText != "":
                newText += "\n"
            strippedLine = line.lstrip()
            if strippedLine:  # Ignorer les lignes vides
                lineIndentation = len(line) - len(strippedLine)
                if firstLine:
                    indentationNumber = 0
                else:
                    indentationNumber = lineIndentation - minIndentation + currentIndentation

                newLine = "\t" * indentationNumber + strippedLine
                newText += newLine

            firstLine = False

        text = newText

        self.insert(text)

        linesAdded = text.count('\n')
        if linesAdded > 0:
            newLine = currentLine + linesAdded
            newIndex = len(text.split('\n')[-1])  # Longueur de la dernière ligne collée
        else:
            newLine = currentLine
            newIndex = currentIndex + len(text)

        self.setCursorPosition(newLine, newIndex)

    def getCleanCode(self):

        lines = []

        for i in range(self.lines()):
            line = self.text(i).rstrip()
            lines.append(line)

        return "\n".join(lines)

    def rgbToBgr(self,color):

        r = color.red()
        g = color.green()
        b = color.blue()

        return (b << 16) | (g << 8) | r

    def checkUndefinedVariables(self, script,code):

        # start = time.time()

        posRef = self.SendScintilla(QsciScintilla.SCI_GETCURRENTPOS)

        definedNames = script.get_names(all_scopes=True, definitions=True)

        definedDict = {}
        currentDefined = set()

        for name in definedNames:
            for line in range(name.line, code.count("\n") + 2):
                if line not in definedDict:
                    definedDict[line] = set(currentDefined)
                definedDict[line].add(name.name)
            currentDefined.add(name.name)

        names = script.get_names(all_scopes=True, references=True)

        for name in names:
            lineText = self.text(name.line - 1)

            if not name.is_definition():
                if name.column > 0 and lineText[name.column - 1] == '.':
                    continue

                inferred = name.infer()

                if not inferred or inferred[0].name == 'unknown':
                    if self.isArgumentInFunctionCall(name, lineText):
                        continue

                    if name.name not in definedDict.get(name.line, set()):
                        startPos = self.positionFromLineIndex(name.line - 1, name.column)
                        length = len(name.name)

                        self.SendScintilla(QsciScintilla.SCI_SETINDICATORCURRENT, self.errorIndicator)
                        self.SendScintilla(QsciScintilla.SCI_INDICATORFILLRANGE, startPos, length)

        self.SendScintilla(QsciScintilla.SCI_GOTOPOS, posRef)

        # print("checkUndefinedVariables",time.time() - start)

    def isArgumentInFunctionCall(self, name, lineText):

        if name.column < len(lineText):
            variableName = name.name
            lineAfterVariable = lineText[name.column + len(variableName):]

        argumentPattern = r'^\s*=\s*[^\),]*'

        return re.match(argumentPattern, lineAfterVariable) is not None

    def connectTextChanged(self):

        self.textHasChanged = True
        self.checkSyntax()
        self.highlightKeywords()

    def checkSyntax(self,fromTimer=False):

        if fromTimer == False:
            self.syntaxTimer = time.time()
        else:
            if time.time() - self.syntaxTimer > self.timeBeforeActualize:
                if self.syntaxToCheck and self.textHasChanged:

                    self.clearHighlights(clear="Syntax")

                    code = self.getCleanCode()
                    script = jedi.Script(code)

                    errors = script.get_syntax_errors()

                    for error in errors:
                        if "Error" in error.get_message():
                            self.applyIndicator(error.line - 1, self.errorIndicator)
                        else:
                            self.applyIndicator(error.line - 1, self.warningIndicator)

                    self.checkUndefinedVariables(script,code)

                    self.textHasChanged = False
                    self.syntaxToCheck = False

                    self.syntaxTimer = time.time()

        # if self.syntaxToCheck and self.textHasChanged:
        #
        #     self.clearHighlights(clear="Syntax")
        #
        #     code = self.getCleanCode()
        #     script = jedi.Script(code)
        #
        #     errors = script.get_syntax_errors()
        #
        #     for error in errors:
        #         if "Error" in error.get_message():
        #             self.applyIndicator(error.line - 1, self.errorIndicator)
        #         else:
        #             self.applyIndicator(error.line - 1, self.warningIndicator)
        #
        #     self.checkUndefinedVariables(script,code)
        #
        #     self.textHasChanged = False
        #     self.syntaxToCheck = False

    def applyIndicator(self, line, indicator):

        startPos = self.positionFromLineIndex(line, 0)
        endPos = self.positionFromLineIndex(line, len(self.text(line)))

        self.SendScintilla(QsciScintilla.SCI_SETINDICATORCURRENT, indicator)
        self.SendScintilla(QsciScintilla.SCI_INDICATORFILLRANGE, startPos, endPos - startPos)

    def placeCursorOnLine(self, lineNumber):

        if lineNumber < 0:
            print("Invalid line number. Line numbers must be >= 0.")
            return
        totalLines = self.lines()
        if lineNumber >= totalLines:
            print(f"Invalid line number. The editor has only {totalLines} lines.")
            return

        self.setFocus()
        qt.QApplication.processEvents()

        self.setCursorPosition(lineNumber, 0)
        self.ensureLineVisible(lineNumber)

    def text(self, line=None):
        if line is None:
            lines = [super(TextEditor, self).text(i) for i in range(self.lines())]
            return "\n".join(lines)
        else:
            return super(TextEditor, self).text(line)

if __name__ == '__main__':
    app = qt.QApplication(sys.argv)

    editor = TextEditor()
    editor.show()

    sys.exit(app.exec_())
