import sys
import traceback
import random
from PyQt5.QtCore import pyqtSignal,QPointF,Qt,QCoreApplication
from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore

import DatabaseFunction as Dfct
import xml.etree.ElementTree as xmlet

import PyIPSDK

import UsefullVariables as vrb
import UsefullFunctions as fct
import UsefullWidgets as wgt
import UsefullDisplay as display


class StatisticsWidget(qt.QWidget):
    """
    Displays IPSDK statistics
    Adapts itself to the multisliced or not
    Is a kind of Singleton in case of not multislice
    """

    # Contains the unique instance of StatisticsWidget
    __instance = None

    @staticmethod
    def getInstance(ipsdkStat, valueSpoiler):
        # If the stats correspond to a multiSlice, a new StatisticsWidgetSliced is created
        # Otherwise the same StatisticsWidget instance is always returned
        if (0, 0, 0) in PyIPSDK.toPyDict(ipsdkStat):
            # multiSlice
            return StatisticsWidgetSliced(ipsdkStat)
        else:
            # Whole image
            if StatisticsWidget.__instance is None:
                StatisticsWidget.__instance = StatisticsWidget(valueSpoiler)
            return StatisticsWidget.__instance

    def __init__(self, valueSpoiler):
        super().__init__()

        self.valueSpoiler = valueSpoiler

        self.statsToDisplay = []

        self.buttonSave = wgt.PushButtonImage(vrb.folderImages + "/Save.png", margins=2)
        self.buttonSave.setFixedSize(30*vrb.ratio, 30*vrb.ratio)

        self.tableView = qt.QTableWidget(0, 7)
        self.headerNames = ['Name', 'Min', 'Max', 'Sum', 'PixCount', 'Mean', 'StdDev']
        self.tableView.horizontalHeader().setDefaultSectionSize(self.tableView.width()/8)
        self.tableView.setHorizontalHeaderLabels(self.headerNames)
        self.tableView.setAlternatingRowColors(True)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.buttonSave, 0, 0)
        self.layout.addWidget(self.tableView, 1, 0)
        self.setLayout(self.layout)

        self.setMinimumSize(600, 300)

        self.valueSpoiler.SignalLabelOverwritten.connect(self.loadStatistics)
        self.valueSpoiler.signalLabelDeletePosition.connect(self.loadStatistics)
        self.buttonSave.clicked.connect(self.saveFileResult)

        self.setWindowTitle("Statistics")

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def loadStatistics(self,boolShow = False):
        self.tableView.setRowCount(0)
        row = 0
        for itemNum in range(self.valueSpoiler.widget.layout.count()):
            item = self.valueSpoiler.widget.layout.itemAt(itemNum)
            if item is not None:
                labelValue = item.widget()
                xmlElement = labelValue.xmlElement
                if labelValue.object is not None:
                    ipsdkStats = PyIPSDK.toPyDict(labelValue.object)
                    if (0, 0, 0) not in ipsdkStats and labelValue.labelButtonTableDisplay.state == 1:
                        self.tableView.insertRow(row)
                        cell = qt.QTableWidgetItem(Dfct.childText(xmlElement, 'Name'))
                        self.tableView.setItem(row, 0, cell)
                        for i in range(1, len(self.headerNames)):
                            cell = qt.QTableWidgetItem(str(fct.numberCalibration(ipsdkStats[self.headerNames[i]])))
                            cell.setTextAlignment(Qt.AlignRight)
                            self.tableView.setItem(row, i, cell)
                        row += 1

        if row == 0:
            self.close()

        if boolShow and row != 0:
            if self.isMaximized():
                self.showMaximized()
            else:
                self.showNormal()
            self.window().raise_()
            self.window().activateWindow()

    def closeEvent(self, a0: QtGui.QCloseEvent) -> None:

        for itemNum in range(self.valueSpoiler.widget.layout.count()):
            item = self.valueSpoiler.widget.layout.itemAt(itemNum)
            if item is not None:
                labelValue = item.widget()
                if labelValue.object is not None:
                    ipsdkStats = PyIPSDK.toPyDict(labelValue.object)
                    if (0, 0, 0) not in ipsdkStats and labelValue.labelButtonTableDisplay.state == 1:
                        labelValue.labelButtonTableDisplay.setState(0)

    def display(self, xmlElement, boolShow=True):
        self.loadStatistics(boolShow = boolShow)

    def stopDisplay(self):
        self.loadStatistics()

    def saveFileResult(self):

        filename = qt.QFileDialog.getSaveFileName(self, "Save file as", "","(*.xls);;(*.csv)")
        if filename[1] == "(*.xls)":
            fct.convertToXlsFile(self.tableView,filename[0])
        if filename[1] == "(*.csv)":
            fct.convertToCsvFile(self.tableView,filename[0])

class StatisticsWidgetSliced(qt.QWidget):
    """
    Displays stats plan by plan (and for each plan, the colors)
    """

    def __init__(self, ipsdkStatistics):
        super().__init__()

        self.colors = [QtGui.QColor.fromRgb(242, 139, 133), QtGui.QColor.fromRgb(151, 242, 133), QtGui.QColor.fromRgb(133, 211, 242)]

        self.buttonSave = wgt.PushButtonImage(vrb.folderImages + "/Save.png", margins=2)
        self.buttonSave.setFixedSize(30*vrb.ratio, 30*vrb.ratio)

        self.buttonProfile = wgt.PushButtonImage(vrb.folderImages + "/Profile.png", margins=2)
        self.buttonProfile.setFixedSize(30*vrb.ratio, 30*vrb.ratio)

        self.labelMeanValue = qt.QLabel("Mean value")
        self.labelMeanValue.setFixedSize(45*vrb.ratio, 30*vrb.ratio)
        self.lineEditMeanValue = qt.QLineEdit()
        self.lineEditMeanValue.setPlaceholderText("1")
        self.lineEditMeanValue.setFixedSize(30*vrb.ratio, 30*vrb.ratio)

        try:
            file = xmlet.parse(vrb.folderInformation + "/Settings.mho")
            settingsElement = file.getroot()
        except:
            settingsElement = xmlet.Element('Settings')

        smoothingProfileValue = Dfct.SubElement(settingsElement, "SmoothingProfile")
        try:
            smoothingProfileValue = int(smoothingProfileValue.text)
        except:
            smoothingProfileValue = 1
        if smoothingProfileValue < 1:
            smoothingProfileValue = 1
        self.lineEditMeanValue.setText(str(smoothingProfileValue))

        self.allProfiles = []

        #self.tableView = qt.QTableWidget(0, 7)
        # self.headerNames = ['Plan', 'Min', 'Max', 'Sum', 'PixCount', 'Mean', 'StdDev']
        self.headerNames = ['Plan', 'ChannelColorName', 'Min', 'Max', 'Sum', 'PixCount', 'Mean', 'StdDev']
        self.tableView = qt.QTableWidget(0, len(self.headerNames))
        
        self.tableView.horizontalHeader().setDefaultSectionSize(self.tableView.width()/8)
        self.tableView.setHorizontalHeaderLabels(self.headerNames)
        self.tableView.setAlternatingRowColors(True)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.buttonSave, 0, 0)
        self.layout.addWidget(self.buttonProfile, 0, 1)
        self.layout.addWidget(self.labelMeanValue, 0, 2)
        self.layout.addWidget(self.lineEditMeanValue, 0, 3,Qt.AlignLeft)
        self.layout.addWidget(self.tableView, 1,0,1,4)
        self.setLayout(self.layout)

        self.setMinimumSize(600, 300)
        self.addStatistics(ipsdkStatistics)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonSave.clicked.connect(self.saveFileResult)
        self.buttonProfile.clicked.connect(self.displayProfile)

    def addStatistics(self, ipsdkStatistics):
        random.seed(1)

        sizeZ = ipsdkStatistics.getMaxSizeZ()
        image = ipsdkStatistics.image

        ipsdkStatistics = PyIPSDK.toPyDict(ipsdkStatistics)
        nbRow = len(ipsdkStatistics)
        self.tableView.setRowCount(nbRow)

        nbColor = 1
        if (0, 2, 0) in ipsdkStatistics:
            nbColor = 2
            for z, c, t in ipsdkStatistics:
                nbColor = max(nbColor, c)
            nbColor += 1
        nbPlan = 1
        if (1, 0, 0) in ipsdkStatistics:
            for z, c, t in ipsdkStatistics:
                nbPlan = max(nbPlan, z)
            nbPlan += 1

        for slice in ipsdkStatistics:
            stats = ipsdkStatistics[slice]
            #cell = qt.QTableWidgetItem(str(slice[0]+slice[2]*sizeZ))
            cell = qt.QTableWidgetItem(str(slice[0]))
            cell.setTextAlignment(Qt.AlignRight)
            self.tableView.setItem(slice[2]*nbColor*nbPlan + slice[0] * nbColor + slice[1], 0, cell)

            cell = qt.QTableWidgetItem(str(image.getChannelName(slice[1])))
            cell.setTextAlignment(Qt.AlignRight)
            self.tableView.setItem(slice[2] * nbColor * nbPlan + slice[0] * nbColor + slice[1], 1, cell)

            if nbColor != 1:
                try:
                    cell.setBackground(self.colors[slice[1]])
                except:
                    # If more than RGB colors, then a random one is chosen
                    self.colors.append(QtGui.QColor.fromRgb(random.randint(0, 255), random.randint(0, 255), random.randint(0, 255)))
                    cell.setBackground(self.colors[slice[1]])
            # for i in range(1, len(self.headerNames)):
            for i in range(2, len(self.headerNames)):
                cell = qt.QTableWidgetItem(str(stats[self.headerNames[i]]))
                cell.setTextAlignment(Qt.AlignRight)
                self.tableView.setItem(slice[2]*nbColor*nbPlan + slice[0] * nbColor + slice[1], i, cell)
        return self

    def display(self, xmlElement, boolShow=True):
        self.setWindowTitle(Dfct.childText(xmlElement, 'Name'))
        if boolShow:
            if self.isMaximized():
                self.showMaximized()
            else:
                self.showNormal()
            self.window().raise_()
            self.window().activateWindow()

    def stopDisplay(self):
        self.close()

    def saveFileResult(self):

        filename = qt.QFileDialog.getSaveFileName(self, "Save file as", "","(*.xls);;(*.csv)")
        if filename[1] == "(*.xls)":
            fct.convertToXlsFile(self.tableView,filename[0])
        if filename[1] == "(*.csv)":
            fct.convertToCsvFile(self.tableView,filename[0])

    def displayProfile(self):

        try:
            numColumn = self.tableView.currentColumn()
            nbRow = self.tableView.rowCount()

            title = self.tableView.horizontalHeaderItem(numColumn).text()
            values = []
            legend = []
            for numRow in range(nbRow):
                value = float(self.tableView.item(numRow,numColumn).text())
                legend.append(numRow)
                values.append(value)

            smoothingProfileValue = self.lineEditMeanValue.text()

            try:
                smoothingProfileValue = int(smoothingProfileValue)
            except:
                smoothingProfileValue = 1
            if smoothingProfileValue <1:
                smoothingProfileValue = 1

            try:
                file = xmlet.parse(vrb.folderInformation + "/Settings.mho")
                settingsElement = file.getroot()
            except:
                settingsElement = xmlet.Element('Settings')

            smoothingProfileElement = Dfct.SubElement(settingsElement, "SmoothingProfile")
            smoothingProfileElement.text = str(smoothingProfileValue)
            Dfct.saveXmlElement(settingsElement, vrb.folderInformation + "/Settings.mho")

            if smoothingProfileValue != 1:
                newValues = []
                for i in range(len(values)):
                    newValue = 0
                    nbElement = 0
                    for step in range(-smoothingProfileValue + 1, smoothingProfileValue):
                        if i+step>=0 and i+step<len(values):
                            newValue+=values[i+step]
                            nbElement+=1
                    newValue = newValue/nbElement
                    newValues.append(newValue)
                values = newValues

            profile = display.WidgetProfile()
            profile.drawLineProfile(values,legend,title=title,xlabel="Z",yLabel=title)
            profile.signalDelete.connect(self.removeLineProfile)
            self.allProfiles.append(profile)

            profile.show()

        except:
            traceback.print_exc(file=sys.stderr)

    def removeLineProfile(self,profile):

        try:
            self.allProfiles.remove(profile)
        except:
            pass
