import sys
import os
import traceback

from PyQt5.QtCore import pyqtSignal,QPointF,Qt,QCoreApplication
from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore

import xml.etree.ElementTree as xmlet

import WidgetTypes
import CustomWidgets

import UsefullWidgets as wgt
import UsefullVariables as vrb
import DatabaseFunction as Dfct
import UsefullFunctions as fct
import UsefullTexts as txt

from RangeSlider import RangeSlider

import PyIPSDK
import PyIPSDK.IPSDKIPLBasicMorphology as morpho
import PyIPSDK.IPSDKIPLAdvancedMorphology as advmorpho
import PyIPSDK.IPSDKIPLColor as color
import PyIPSDK.IPSDKIPLUtility as util
import PyIPSDK.IPSDKIPLFiltering as filtering
import PyIPSDK.IPSDKIPLBinarization as bin
import PyIPSDK.IPSDKIPLArithmetic as arithm
import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr
import PyIPSDK.IPSDKIPLGeometricTransform as gtrans
import PyIPSDK.IPSDKIPLShapeAnalysis as shapeanalysis
import PyIPSDK.IPSDKIPLShapeSegmentation as shapesegmentation
import PyIPSDK.IPSDKIPLIntensityTransform as itrans
import PyIPSDK.IPSDKIPLLogical as logic
import PyIPSDK.IPSDKIPLFeatureDetection as fd
import PyIPSDK.IPSDKIPLStats as stats
import PyIPSDK.IPSDKIPLClassification as classif

import numpy as np

class SieveWidget(qt.QWidget):

    SignalSieveChanged = pyqtSignal()

    def __init__(self, infoSet=None):
        qt.QWidget.__init__(self)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        try:
            file = xmlet.parse(vrb.folderInformation + "/UserSieves.mho")
            self.xmlElement = file.getroot()
        except:
            self.xmlElement = xmlet.Element('UserSieves')

        self.layout = qt.QGridLayout()

        self.sieveSelection = SieveSelection(parent = self)
        self.sieveParameters = SieveParameters(parent=self)

        self.layout.addWidget(self.sieveSelection, 0, 0, Qt.AlignLeft)
        self.layout.addWidget(self.sieveParameters, 1, 0, Qt.AlignLeft | Qt.AlignTop)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(10*vrb.ratio,10*vrb.ratio,10*vrb.ratio,10*vrb.ratio)

        self.setWindowTitle("Sieve selection")

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.setFixedSize(500*vrb.ratio, 320*vrb.ratio)

        self.loadXmlElement()

        self.sieveSelection.comboBoxSelection.currentIndexChanged.connect(self.changeSieve)

        self.sieveParameters.updateVisible()

    def loadXmlElement(self):

        self.sieveSelection.comboBoxSelection.clear()

        for child in self.xmlElement:
            name = Dfct.childText(child,"Name")
            self.sieveSelection.comboBoxSelection.addItem(name,child)
            if self.sieveSelection.comboBoxSelection.count() > 1:
                self.sieveSelection.buttonDelete.setEnabled(True)

        if self.sieveSelection.comboBoxSelection.count() == 0:
            self.addDefaultSieve()
        else:
            self.changeSieve()

    def addElement(self):

        self.sieveParameters.writeXmlElement()

        newName = "sieve 1"
        num = 1
        while self.sieveSelection.comboBoxSelection.findText(newName) != -1:
            num+=1
            newName = "sieve " + str(num)

        child = xmlet.SubElement(self.xmlElement, "Sieve")
        nameElement = Dfct.SubElement(child, "Name")
        nameElement.text = newName
        self.setDefaultValues(child)

        self.sieveSelection.comboBoxSelection.addItem(newName, child)
        self.sieveSelection.comboBoxSelection.setCurrentIndex(self.sieveSelection.comboBoxSelection.count()-1)

        self.sieveSelection.buttonSave.setEnabled(True)
        self.changeSieve()

    def addDefaultSieve(self):

        child = xmlet.SubElement(self.xmlElement, "Sieve")
        Dfct.SubElement(child, "Name").text = "Default sieve"
        self.setDefaultValues(child)

        self.sieveSelection.comboBoxSelection.addItem("Default sieve", child)
        self.sieveSelection.comboBoxSelection.setCurrentIndex(self.sieveSelection.comboBoxSelection.count()-1)

        self.changeSieve()
        self.saveXmlElement()
        self.sieveParameters.updateVisible()

    def setDefaultValues(self,child):

        Dfct.SubElement(child, "Type").text = "1"
        Dfct.SubElement(child, "Min").text = ""
        Dfct.SubElement(child, "Max").text = ""
        Dfct.SubElement(child, "BinChecked").text = "False"
        Dfct.SubElement(child, "BinValue").text = "1"
        Dfct.SubElement(child, "NbClassesChecked").text = "True"
        Dfct.SubElement(child, "NbClassesValue").text = "20"
        Dfct.SubElement(child, "RaisonChecked").text = "True"
        Dfct.SubElement(child, "RaisonValue").text = "1.41"
        Dfct.SubElement(child, "NbClassesGeometricChecked").text = "False"
        Dfct.SubElement(child, "NbClassesGeometricValue").text = ""
        Dfct.SubElement(child, "Sieve").text = ""

    def changeSieve(self):

        try:
            if self.sieveParameters.xmlElement is not None:

                self.sieveParameters.writeXmlElement()

            self.sieveSelection.buttonDelete.setEnabled(self.sieveSelection.comboBoxSelection.currentIndex()!=0)

            element = self.sieveSelection.comboBoxSelection.currentData()
            self.sieveParameters.loadSieve(element)

        except:
            traceback.print_exc(file=sys.stderr)

    def saveXmlElement(self):

        self.sieveParameters.writeXmlElement()

        Dfct.saveXmlElement(self.xmlElement,vrb.folderInformation + "/UserSieves.mho")
        self.sieveSelection.buttonSave.setEnabled(False)

        self.SignalSieveChanged.emit()

        fct.updateSievesComboBox()

class NameSieveWidget(qt.QWidget):

    def __init__(self):
        qt.QWidget.__init__(self)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.layout = qt.QGridLayout()

        self.labelName = qt.QLabel("Name sieve")
        self.labelName.setFixedSize(100*vrb.ratio,30*vrb.ratio)
        self.lineEditName = qt.QLineEdit()
        self.lineEditName.setFixedSize(160*vrb.ratio,30*vrb.ratio)
        self.buttonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=2)
        self.buttonValidate.setFixedSize(30*vrb.ratio,30*vrb.ratio)
        self.buttonValidate.setToolTip("Validate")

        self.layout.addWidget(self.labelName, 0, 0)
        self.layout.addWidget(self.lineEditName, 0, 1)
        self.layout.addWidget(self.buttonValidate, 0, 2)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3, 0, 3, 0)
        self.layout.setHorizontalSpacing(3)

        self.setWindowTitle("Name sieve")

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

class SieveSelection(qt.QGroupBox):

    def __init__(self,parent = None):
        qt.QGroupBox.__init__(self)

        self.parent = parent

        self.layout = qt.QGridLayout()

        self.nameSieveWidget = NameSieveWidget()

        self.labelSelection = qt.QLabel("Select your sieve")
        self.labelSelection.setFixedSize(110*vrb.ratio,30*vrb.ratio)
        self.comboBoxSelection = qt.QComboBox()
        self.comboBoxSelection.setFixedSize(140*vrb.ratio,30*vrb.ratio)
        self.buttonAdd = wgt.PushButtonImage(vrb.folderImages + "/Add_4.png", margins=2)
        self.buttonAdd.setFixedSize(30*vrb.ratio,30*vrb.ratio)
        self.buttonAdd.setToolTip("Add a new sieve")
        self.buttonEdit = wgt.PushButtonImage(vrb.folderImages + "/Edit_3.png", margins=2)
        self.buttonEdit.setFixedSize(30*vrb.ratio,30*vrb.ratio)
        self.buttonEdit.setToolTip("Change the name of the sieve")
        self.buttonSave = wgt.PushButtonImage(vrb.folderImages + "/Save.png", margins=2)
        self.buttonSave.setFixedSize(30*vrb.ratio,30*vrb.ratio)
        self.buttonSave.setToolTip("Save this sieve")
        self.buttonDelete = wgt.PushButtonImage(vrb.folderImages + "/Delete.png", margins=2)
        self.buttonDelete.setFixedSize(30*vrb.ratio,30*vrb.ratio)
        self.buttonDelete.setToolTip("Delete this sieve")
        self.buttonDelete.setEnabled(False)

        self.layout.addWidget(self.labelSelection, 0, 0)
        self.layout.addWidget(self.comboBoxSelection, 0, 1)
        self.layout.addWidget(self.buttonAdd, 0, 2)
        self.layout.addWidget(self.buttonEdit, 0, 3)
        self.layout.addWidget(self.buttonSave, 0, 4)
        self.layout.addWidget(self.buttonDelete, 0, 5, Qt.AlignLeft)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3*vrb.ratio, 0, 3*vrb.ratio, 0)

        self.setStyleSheet("QGroupBox {border: 0px solid gray; }")

        self.buttonAdd.clicked.connect(self.parent.addElement)
        self.buttonEdit.clicked.connect(self.editElement)
        self.buttonSave.clicked.connect(self.parent.saveXmlElement)
        self.buttonDelete.clicked.connect(self.deleteElement)

        self.nameSieveWidget.buttonValidate.clicked.connect(self.validateName)

        self.setFixedSize(475*vrb.ratio,50*vrb.ratio)

    def editElement(self):

        self.nameSieveWidget.lineEditName.setText(self.comboBoxSelection.currentText())
        fct.showWidget(self.nameSieveWidget)

    def validateName(self):

        self.nameSieveWidget.close()
        self.comboBoxSelection.setItemText(self.comboBoxSelection.currentIndex(),self.nameSieveWidget.lineEditName.text())
        element = self.comboBoxSelection.currentData()
        Dfct.SubElement(element,"Name").text = self.nameSieveWidget.lineEditName.text()
        # self.parent.saveXmlElement()

    def deleteElement(self):

        element = self.comboBoxSelection.currentData()
        self.comboBoxSelection.removeItem(self.comboBoxSelection.currentIndex())
        Dfct.removeElement(self.parent.xmlElement,element)
        # self.parent.saveXmlElement()

class SieveParameters(qt.QGroupBox):

    def __init__(self,parent = None):
        qt.QGroupBox.__init__(self)

        self.xmlElement = None

        self.parent = parent

        self.layout = qt.QGridLayout()

        self.labelType = qt.QLabel("Type")
        self.labelType.setFixedSize(75*vrb.ratio,30*vrb.ratio)
        self.comboBoxType = qt.QComboBox()
        self.comboBoxType.setFixedSize(120*vrb.ratio,30*vrb.ratio)
        self.comboBoxType.addItem("Linear Min/Max")
        self.comboBoxType.addItem("Linear Auto")
        self.comboBoxType.addItem("Geometric Min/Max")
        self.comboBoxType.addItem("Geometric Auto")
        self.comboBoxType.addItem("Custom")

        self.labelWithLineEditMin = wgt.LabelWithLineEdit(20,"Min","")
        self.labelWithLineEditMin.setFixedSize(75*vrb.ratio, 30*vrb.ratio)
        self.labelWithLineEditMax = wgt.LabelWithLineEdit(20,"Max","")
        self.labelWithLineEditMax.setFixedSize(75*vrb.ratio, 30*vrb.ratio)

        self.radioButtonWithLineEditBin = wgt.RadioButtonWithLineEdit(70,"Bin","1")
        self.radioButtonWithLineEditBin.setFixedSize(130*vrb.ratio, 30*vrb.ratio)
        self.radioButtonWithLineEditNbClasses = wgt.RadioButtonWithLineEdit(70,"Nb Classes","20")
        self.radioButtonWithLineEditNbClasses.setFixedSize(130*vrb.ratio, 30*vrb.ratio)

        self.radioButtonWithLineEditRaison = wgt.RadioButtonWithLineEdit(70,"Raison","1.41")
        self.radioButtonWithLineEditRaison.setFixedSize(130*vrb.ratio, 30*vrb.ratio)
        self.radioButtonWithLineEditNbClassesGeometric = wgt.RadioButtonWithLineEdit(70,"Nb Classes","20")
        self.radioButtonWithLineEditNbClassesGeometric.setFixedSize(130*vrb.ratio, 30*vrb.ratio)

        font = QtGui.QFont()
        font.setPointSize(11)
        self.labelSieveValue = wgt.ScrollableLabel()
        self.labelSieveValue.label.setText("Sieve values = [ ]")
        self.labelSieveValue.setFixedSize(350*vrb.ratio, 60*vrb.ratio)
        self.labelSieveValue.label.setFont(font)

        self.labelWithLineEditSieve = wgt.LabelWithLineEdit(40,"Sieve","1,5,20,...",typeValue="text")
        self.labelWithLineEditSieve.setFixedSize(350*vrb.ratio, 60*vrb.ratio)

        self.emptyLabel1 = qt.QLabel("")
        self.emptyLabel1.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        self.emptyLabel2 = qt.QLabel("")
        self.emptyLabel2.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        self.emptyLabel3 = qt.QLabel("")
        self.emptyLabel3.setFixedSize(85*vrb.ratio, 30*vrb.ratio)
        self.emptyLabel4 = qt.QLabel("")
        self.emptyLabel4.setFixedSize(75*vrb.ratio, 30*vrb.ratio)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3*vrb.ratio, 0, 3*vrb.ratio, 0)

        self.setFixedSize(475*vrb.ratio,200*vrb.ratio)

        self.comboBoxType.currentIndexChanged.connect(self.updateVisible)

        self.radioButtonWithLineEditBin.radioButton.clicked.connect(self.binClicked)
        self.radioButtonWithLineEditNbClasses.radioButton.clicked.connect(self.nbClassesClicked)
        self.radioButtonWithLineEditBin.lineEdit.textChanged.connect(self.computeSieve)
        self.radioButtonWithLineEditNbClasses.lineEdit.textChanged.connect(self.computeSieve)

        self.radioButtonWithLineEditRaison.radioButton.clicked.connect(self.raisonClicked)
        self.radioButtonWithLineEditNbClassesGeometric.radioButton.clicked.connect(self.nbClassesGeometricClicked)
        self.radioButtonWithLineEditRaison.lineEdit.textChanged.connect(self.computeSieve)
        self.radioButtonWithLineEditNbClassesGeometric.lineEdit.textChanged.connect(self.computeSieve)

        self.labelWithLineEditMin.lineEdit.textChanged.connect(self.computeSieve)
        self.labelWithLineEditMax.lineEdit.textChanged.connect(self.computeSieve)
        self.labelWithLineEditSieve.lineEdit.textChanged.connect(self.buttonSaveEnabled)

        self.binClicked()
        self.raisonClicked()

    def binClicked(self):

        self.radioButtonWithLineEditBin.radioButton.setChecked(True)
        self.radioButtonWithLineEditBin.lineEdit.setEnabled(True)
        self.radioButtonWithLineEditNbClasses.radioButton.setChecked(False)
        self.radioButtonWithLineEditNbClasses.lineEdit.setEnabled(False)

        self.computeSieve()

    def nbClassesClicked(self):

        self.radioButtonWithLineEditNbClasses.radioButton.setChecked(True)
        self.radioButtonWithLineEditNbClasses.lineEdit.setEnabled(True)
        self.radioButtonWithLineEditBin.radioButton.setChecked(False)
        self.radioButtonWithLineEditBin.lineEdit.setEnabled(False)

        self.computeSieve()

    def raisonClicked(self):

        self.radioButtonWithLineEditRaison.radioButton.setChecked(True)
        self.radioButtonWithLineEditRaison.lineEdit.setEnabled(True)
        self.radioButtonWithLineEditNbClassesGeometric.radioButton.setChecked(False)
        self.radioButtonWithLineEditNbClassesGeometric.lineEdit.setEnabled(False)

        self.computeSieve()

    def nbClassesGeometricClicked(self):

        self.radioButtonWithLineEditNbClassesGeometric.radioButton.setChecked(True)
        self.radioButtonWithLineEditNbClassesGeometric.lineEdit.setEnabled(True)
        self.radioButtonWithLineEditRaison.radioButton.setChecked(False)
        self.radioButtonWithLineEditRaison.lineEdit.setEnabled(False)

        self.computeSieve()

    def updateVisible(self):

        # Clear the existing layout
        for i in reversed(range(self.layout.count())):
            self.layout.itemAt(i).widget().setParent(None)

        self.layout.addWidget(self.labelType, 0, 0)
        self.layout.addWidget(self.comboBoxType, 0, 1, 1, 2, Qt.AlignLeft)

        if self.comboBoxType.currentIndex() == 0:
            self.layout.addWidget(self.labelWithLineEditMin, 1, 0)
            self.layout.addWidget(self.labelWithLineEditMax, 2, 0)
            self.layout.addWidget(self.emptyLabel1, 1, 1)
            self.layout.addWidget(self.radioButtonWithLineEditBin, 1, 2, Qt.AlignLeft)
            self.layout.addWidget(self.radioButtonWithLineEditNbClasses, 2, 2, Qt.AlignLeft)
            self.layout.addWidget(self.labelSieveValue, 3, 0, 1, 3, Qt.AlignLeft)

            self.labelWithLineEditMin.setVisible(True)
            self.labelWithLineEditMax.setVisible(True)
            self.radioButtonWithLineEditBin.setVisible(True)
            self.radioButtonWithLineEditNbClasses.setVisible(True)
            self.radioButtonWithLineEditRaison.setVisible(False)
            self.radioButtonWithLineEditNbClassesGeometric.setVisible(False)
            self.labelSieveValue.setVisible(True)
            self.labelWithLineEditSieve.setVisible(False)
            self.radioButtonWithLineEditBin.lineEdit.setEnabled(self.radioButtonWithLineEditBin.radioButton.isChecked())
            self.radioButtonWithLineEditNbClasses.lineEdit.setEnabled(self.radioButtonWithLineEditNbClasses.radioButton.isChecked())

        if self.comboBoxType.currentIndex() == 1:

            self.layout.addWidget(self.emptyLabel4, 1, 0)
            self.layout.addWidget(self.emptyLabel1, 1, 1)
            self.layout.addWidget(self.radioButtonWithLineEditBin, 1, 2, Qt.AlignLeft)
            self.layout.addWidget(self.radioButtonWithLineEditNbClasses, 2, 2, Qt.AlignLeft)
            self.layout.addWidget(self.labelSieveValue, 3, 0, 1, 3, Qt.AlignLeft)

            self.labelWithLineEditMin.setVisible(False)
            self.labelWithLineEditMax.setVisible(False)
            self.radioButtonWithLineEditBin.setVisible(True)
            self.radioButtonWithLineEditNbClasses.setVisible(True)
            self.radioButtonWithLineEditRaison.setVisible(False)
            self.radioButtonWithLineEditNbClassesGeometric.setVisible(False)
            self.labelSieveValue.setVisible(True)
            self.labelWithLineEditSieve.setVisible(False)
            self.radioButtonWithLineEditBin.lineEdit.setEnabled(self.radioButtonWithLineEditBin.radioButton.isChecked())
            self.radioButtonWithLineEditNbClasses.lineEdit.setEnabled(self.radioButtonWithLineEditNbClasses.radioButton.isChecked())

        elif self.comboBoxType.currentIndex() == 2:
            self.layout.addWidget(self.labelWithLineEditMin, 1, 0)
            self.layout.addWidget(self.labelWithLineEditMax, 2, 0)
            self.layout.addWidget(self.emptyLabel1, 1, 1)
            self.layout.addWidget(self.radioButtonWithLineEditRaison, 1, 2, Qt.AlignLeft)
            self.layout.addWidget(self.radioButtonWithLineEditNbClassesGeometric, 2, 2, Qt.AlignLeft)
            self.layout.addWidget(self.labelSieveValue, 3, 0, 1, 3, Qt.AlignLeft)

            self.labelWithLineEditMin.setVisible(True)
            self.labelWithLineEditMax.setVisible(True)
            self.radioButtonWithLineEditBin.setVisible(False)
            self.radioButtonWithLineEditNbClasses.setVisible(False)
            self.radioButtonWithLineEditRaison.setVisible(True)
            self.radioButtonWithLineEditNbClassesGeometric.setVisible(True)
            self.labelSieveValue.setVisible(True)
            self.labelWithLineEditSieve.setVisible(False)
            self.radioButtonWithLineEditRaison.lineEdit.setEnabled(self.radioButtonWithLineEditRaison.radioButton.isChecked())
            self.radioButtonWithLineEditNbClassesGeometric.lineEdit.setEnabled(self.radioButtonWithLineEditNbClassesGeometric.radioButton.isChecked())

        elif self.comboBoxType.currentIndex() == 3:
            self.layout.addWidget(self.emptyLabel4, 1, 0)
            self.layout.addWidget(self.emptyLabel1, 1, 1)
            self.layout.addWidget(self.radioButtonWithLineEditRaison, 1, 2, Qt.AlignLeft)
            self.layout.addWidget(self.radioButtonWithLineEditNbClassesGeometric, 2, 2, Qt.AlignLeft)
            self.layout.addWidget(self.labelSieveValue, 3, 0, 1, 3, Qt.AlignLeft)

            self.labelWithLineEditMin.setVisible(False)
            self.labelWithLineEditMax.setVisible(False)
            self.radioButtonWithLineEditBin.setVisible(False)
            self.radioButtonWithLineEditNbClasses.setVisible(False)
            self.radioButtonWithLineEditRaison.setVisible(True)
            self.radioButtonWithLineEditNbClassesGeometric.setVisible(True)
            self.labelSieveValue.setVisible(True)
            self.labelWithLineEditSieve.setVisible(False)
            self.radioButtonWithLineEditRaison.lineEdit.setEnabled(self.radioButtonWithLineEditRaison.radioButton.isChecked())
            self.radioButtonWithLineEditNbClassesGeometric.lineEdit.setEnabled(self.radioButtonWithLineEditNbClassesGeometric.radioButton.isChecked())

        elif self.comboBoxType.currentIndex() == 4:
            self.layout.addWidget(self.emptyLabel4, 1, 0)
            self.layout.addWidget(self.emptyLabel1, 1, 1)
            self.layout.addWidget(self.emptyLabel3, 1, 2, Qt.AlignLeft)
            self.layout.addWidget(self.emptyLabel2, 2, 2, Qt.AlignLeft)
            self.layout.addWidget(self.labelWithLineEditSieve, 3, 0, 1, 3, Qt.AlignLeft)

            self.labelWithLineEditMin.setVisible(False)
            self.labelWithLineEditMax.setVisible(False)
            self.radioButtonWithLineEditBin.setVisible(False)
            self.radioButtonWithLineEditNbClasses.setVisible(False)
            self.radioButtonWithLineEditRaison.setVisible(False)
            self.radioButtonWithLineEditNbClassesGeometric.setVisible(False)
            self.labelSieveValue.setVisible(False)
            self.labelWithLineEditSieve.setVisible(True)

        self.computeSieve()

    def writeXmlElement(self):

        Dfct.createSubElementComboBox(self.xmlElement,self.comboBoxType,"Type")
        Dfct.createSubElementLineEdit(self.xmlElement,self.labelWithLineEditMin.lineEdit,"Min")
        Dfct.createSubElementLineEdit(self.xmlElement,self.labelWithLineEditMax.lineEdit,"Max")
        Dfct.createSubElementCheckBox(self.xmlElement,self.radioButtonWithLineEditBin.radioButton,"BinChecked")
        Dfct.createSubElementLineEdit(self.xmlElement,self.radioButtonWithLineEditBin.lineEdit,"BinValue")
        Dfct.createSubElementCheckBox(self.xmlElement,self.radioButtonWithLineEditNbClasses.radioButton,"NbClassesChecked")
        Dfct.createSubElementLineEdit(self.xmlElement,self.radioButtonWithLineEditNbClasses.lineEdit,"NbClassesValue")
        Dfct.createSubElementCheckBox(self.xmlElement,self.radioButtonWithLineEditRaison.radioButton,"RaisonChecked")
        Dfct.createSubElementLineEdit(self.xmlElement,self.radioButtonWithLineEditRaison.lineEdit,"RaisonValue")
        Dfct.createSubElementCheckBox(self.xmlElement,self.radioButtonWithLineEditNbClassesGeometric.radioButton,"NbClassesGeometricChecked")
        Dfct.createSubElementLineEdit(self.xmlElement,self.radioButtonWithLineEditNbClassesGeometric.lineEdit,"NbClassesGeometricValue")
        Dfct.createSubElementLineEdit(self.xmlElement,self.labelWithLineEditSieve.lineEdit,"Sieve")

    def loadXmlElement(self):

        Dfct.readElementComboBox(self.xmlElement,self.comboBoxType,"Type")

        Dfct.readElementLineEdit(self.xmlElement,self.labelWithLineEditMin.lineEdit,"Min")
        Dfct.readElementLineEdit(self.xmlElement,self.labelWithLineEditMax.lineEdit,"Max")
        Dfct.readElementCheckBox(self.xmlElement,self.radioButtonWithLineEditBin.radioButton,"BinChecked")
        Dfct.readElementLineEdit(self.xmlElement,self.radioButtonWithLineEditBin.lineEdit,"BinValue")
        Dfct.readElementCheckBox(self.xmlElement,self.radioButtonWithLineEditNbClasses.radioButton,"NbClassesChecked")
        Dfct.readElementLineEdit(self.xmlElement,self.radioButtonWithLineEditNbClasses.lineEdit,"NbClassesValue")
        Dfct.readElementCheckBox(self.xmlElement,self.radioButtonWithLineEditRaison.radioButton,"RaisonChecked")
        Dfct.readElementLineEdit(self.xmlElement,self.radioButtonWithLineEditRaison.lineEdit,"RaisonValue")
        Dfct.readElementCheckBox(self.xmlElement,self.radioButtonWithLineEditNbClassesGeometric.radioButton,"NbClassesGeometricChecked")
        Dfct.readElementLineEdit(self.xmlElement,self.radioButtonWithLineEditNbClassesGeometric.lineEdit,"NbClassesGeometricValue")
        Dfct.readElementLineEdit(self.xmlElement,self.labelWithLineEditSieve.lineEdit,"Sieve")

    def computeSieve(self):

        if self.comboBoxType.currentIndex() in [1,3]:
            text = ""
        else:
            try:

                text = "Sieve values = [ ]"

                mini = float(self.labelWithLineEditMin.lineEdit.text())
                maxi = float(self.labelWithLineEditMax.lineEdit.text())

                if abs(round(mini)-mini)<0.00001 and abs(round(maxi)-maxi)<0.00001:
                    isInterger = True
                    mini = round(mini)
                    maxi = round(maxi)
                else:
                    isInterger = False

                if maxi > mini:
                    if self.comboBoxType.currentIndex()==0:
                        if self.radioButtonWithLineEditBin.radioButton.isChecked():
                            if self.radioButtonWithLineEditBin.lineEdit.text() == "":
                                binValue = int(self.radioButtonWithLineEditBin.lineEdit.placeholderText())
                            else:
                                binValue = float(self.radioButtonWithLineEditBin.lineEdit.text())
                        else:
                            if self.radioButtonWithLineEditNbClasses.lineEdit.text() == "":
                                nbClasses = int(self.radioButtonWithLineEditNbClasses.lineEdit.placeholderText())
                            else:
                                nbClasses = int(self.radioButtonWithLineEditNbClasses.lineEdit.text())
                            binValue = (maxi-mini) / nbClasses

                        if abs(round(binValue) - binValue) > 0.00001:
                            isInterger = False
                        else:
                            binValue = round(binValue)

                        if binValue > 0:
                            text = "Sieve values = ["
                            value = mini
                            while value < maxi:
                                if isInterger:
                                    text += str(round(value)) + ", "
                                else:
                                    text += str(round(value*10)/10) + ", "
                                value += binValue

                            text+= str(maxi) + "]"
                        else:
                            text = "Sieve values = [ ]"

                    if self.comboBoxType.currentIndex()==2 and mini > 0:

                        if self.radioButtonWithLineEditNbClassesGeometric.radioButton.isChecked():
                            if self.radioButtonWithLineEditNbClassesGeometric.lineEdit.text() == "":
                                nbClasses = int(self.radioButtonWithLineEditNbClassesGeometric.lineEdit.placeholderText())
                            else:
                                nbClasses = int(self.radioButtonWithLineEditNbClassesGeometric.lineEdit.text())

                            r = (maxi / mini) ** (1 / (nbClasses - 1))
                        else:
                            if self.radioButtonWithLineEditRaison.lineEdit.text() == "":
                                r = float(self.radioButtonWithLineEditRaison.lineEdit.placeholderText())
                            else:
                                r = float(self.radioButtonWithLineEditRaison.lineEdit.text())

                        if r>1:
                            text = "Sieve values = ["
                            value = mini
                            while value < maxi:
                                text += str(round(value)) + ", "
                                value *= r
                            text+= str(maxi) + "]"

                        else:
                            text = "Sieve values = [ ]"
                else:
                    text = "Sieve values = [ ]"

            except:
                # traceback.print_exc(file=sys.stderr)
                text = "Sieve values = [ ]"

        self.labelSieveValue.label.setText(text)
        self.buttonSaveEnabled()

    def buttonSaveEnabled(self):

        self.parent.sieveSelection.buttonSave.setEnabled(True)

    def loadSieve(self,element):

        self.parent.sieveSelection.buttonSave.setEnabled(False)
        self.xmlElement = element
        self.loadXmlElement()
        # print(element)

class ComboBoxSieveWithButton(qt.QGroupBox):

    def __init__(self, xmlElement, type):
        super().__init__()

        self.layout = qt.QGridLayout()

        self.comboBoxSieve = qt.QComboBox()
        self.comboBoxSieve.setFixedSize(120*vrb.ratio,20*vrb.ratio)
        self.buttonEdit = wgt.PushButtonImage(vrb.folderImages + "/Edit_3.png", margins=2)
        self.buttonEdit.setFixedSize(20*vrb.ratio,20*vrb.ratio)

        self.layout.addWidget(self.comboBoxSieve, 0, 0)
        self.layout.addWidget(self.buttonEdit, 0, 1)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(0,0,0,0)

        self.setStyleSheet("QGroupBox {border: 0px solid gray; }")

        self.setFixedHeight(vrb.DEFAULT_SIZE)

        self.buttonEdit.clicked.connect(vrb.mainWindow.openSievePreferences)

    def userValueToInterface(self, xmlElement):
        pass

    def interfaceToXml(self, number):

        paramNode = xmlet.Element('Parameter_' + str(number))
        Dfct.SubElement(paramNode, 'Type').text = WidgetTypes.InputType.SIEVE.value
        value = self.comboBoxSieve.currentText()
        Dfct.SubElement(paramNode, 'Value').text = value

        return paramNode

    @staticmethod
    def xmlToValue(xmlElement):
        value = Dfct.childText(xmlElement, 'Value')

        return value


if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook

    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    foo = SieveWidget()

    foo.show()

    foo.sieveParameters.updateVisible()
    foo.sieveSelection.buttonSave.setEnabled(False)

    app.exec_()
