import sys,traceback
import os
import webbrowser

import PyQt5.QtCore as QtCore
from PyQt5.QtCore import pyqtSignal, Qt, QCoreApplication
import PyQt5.QtWidgets as qt
from PyQt5 import QtGui

import WidgetTypes
import UsefullVariables as vrb
import UsefullFunctionsForWidget as fctWidget
from CustomShapeAnalysisWidget import CustomShapeAnalysisWidget
from ShapeAnalysisTreeItemsWidgets import MyTreeWidgetItem, MeasureLabel, CustomTreeWidgetItem
# from SearchBox import SearchBoxMeasure
import UsefullWidgets as wgt

import DatabaseFunction as Dfct
import UsefullFunctions as fct
import UsefullTexts as txt

import xml.etree.ElementTree as xmlet

from ConversionIPSDK import dictConversion


class MeasureInfoSetWidget(qt.QGroupBox):

    SignalInfoSetChanged = pyqtSignal()

    def __init__(self, xmlElement, type,hasConstraint = False):
        if type != WidgetTypes.InputType.MEASURE_INFO_SET:
            print('ERROR : InputType does not match', file=sys.stderr)
        super().__init__()
        self.type = type
        self.hasConstraint = hasConstraint
        self.xmlElement = xmlElement
        self.additionalFolder = None

        if self.xmlElement is not None:
            self.dimension = Dfct.childText(self.xmlElement, 'Constraints')
        else:
            self.dimension = "2D"

        self.comboBoxInfoSet = qt.QComboBox()
        self.comboBoxInfoSet.setFixedHeight(20*vrb.ratio)
        # self.labelButtonAdd = wgt.LabelAdd()
        # self.labelButtonDel = wgt.LabelDelete('-')
        # self.labelButtonEdit = wgt.LabelEdit()
        self.pushButtonAdd = wgt.PushButtonImage(vrb.folderImages + "/Add_Calibration.png", margins=2)
        self.pushButtonAdd.setFixedSize(20*vrb.ratio,20*vrb.ratio)
        self.pushButtonAdd.setToolTip(txt.dictToolTips["NewInfoset"])
        self.pushButtonEdit = wgt.PushButtonImage(vrb.folderImages + "/Edit_2.png", margins=2)
        self.pushButtonEdit.setFixedSize(20*vrb.ratio,20*vrb.ratio)
        self.pushButtonEdit.setToolTip(txt.dictToolTips["EditInfoset"])
        self.pushButtonDel = wgt.PushButtonImage(vrb.folderImages + "/Delete_Calibration.png", margins=2)
        self.pushButtonDel.setFixedSize(20*vrb.ratio,20*vrb.ratio)
        self.pushButtonDel.setToolTip(txt.dictToolTips["DeleteInfoset"])

        self.infoSetSelector = None

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.comboBoxInfoSet, 0, 0)
        self.layout.addWidget(self.pushButtonAdd, 0, 1)
        self.layout.addWidget(self.pushButtonEdit, 0, 2)
        self.layout.addWidget(self.pushButtonDel, 0, 3)
        self.setLayout(self.layout)

        self.setFixedHeight(vrb.DEFAULT_SIZE)
        self.layout.setContentsMargins(0, 0, 0, 0)

        self.pushButtonAdd.clicked.connect(self.addClicked)
        self.pushButtonEdit.clicked.connect(self.editClicked)
        self.pushButtonDel.clicked.connect(self.deleteInfoSet)
        self.comboBoxInfoSet.currentIndexChanged.connect(self.emitInfoSetChanged)

        self.fillComboBox()

    def showEvent(self, a0: QtGui.QShowEvent) -> None:
        self.fillComboBox()

    def fillComboBox(self):
        self.comboBoxInfoSet.clear()
        folder = vrb.folderUserData + '/InfoSet'
        if self.additionalFolder is not None:
            self.fillFromFolder(self.additionalFolder,additional = True)
        self.fillFromFolder(folder)

    def fillFromFolder(self,folder,additional = False):

        for fileName in sorted(os.listdir(folder),key=str.casefold):
            filePath = folder + '/' + fileName
            if filePath.endswith('.mho'):
                try:
                    xmlInfoSet = xmlet.parse(filePath).getroot()
                    if xmlInfoSet.tag == 'AllMeasures' and xmlInfoSet.get('Dimension') == str(self.dimension):
                        if additional:
                            self.comboBoxInfoSet.addItem(fileName.replace('.mho', ''), [xmlInfoSet,folder])
                        else:
                            self.comboBoxInfoSet.addItem(fileName.replace('.mho', ''), [xmlInfoSet,None])
                except:
                    pass

    def interfaceToXml(self, number):
        try:
            paramNode = xmlet.Element('Parameter_' + str(number))
            Dfct.SubElement(paramNode, 'Type').text = WidgetTypes.InputType.MEASURE_INFO_SET.value
            value = self.comboBoxInfoSet.currentData()[0]
            copy = Dfct.copyXmlElement(value)
            valueNode = Dfct.SubElement(paramNode, 'Value')
            valueNode.append(copy)
            return paramNode
        except:
            pass

    @staticmethod
    def xmlToValue(xmlElement):
        valueParam = Dfct.SubElement(xmlElement, 'Value')
        value = Dfct.SubElement(valueParam, 'AllMeasures')
        return value

    def deleteInfoSet(self):
        try:
            folder = vrb.folderUserData + '/InfoSet'
            fileName = self.comboBoxInfoSet.currentText() + '.mho'
            filePath = folder + '/' + fileName
            os.remove(filePath)
            self.comboBoxInfoSet.removeItem(self.comboBoxInfoSet.currentIndex())
        except:
            vrb.printDebug('ERROR: Could not delete ' + str(self.comboBoxInfoSet.currentIndex()))

    def editClicked(self):
        xmlInfoSet = self.comboBoxInfoSet.currentData()[0]
        additionalFolder = self.comboBoxInfoSet.currentData()[1]
        name = self.comboBoxInfoSet.currentText()
        if self.comboBoxInfoSet.currentIndex() == -1:
            name = 'DefaultName_' + str(self.dimension)
        self.showInfoSetSelector(xmlInfoSet, name,additionalFolder = additionalFolder)

    def addClicked(self):
        i = self.comboBoxInfoSet.count() + 1
        while self.comboBoxInfoSet.findText('Set_'+str(self.dimension)+'_' + str(i)) != -1:
            i += 1
        self.showInfoSetSelector(name='Set_'+str(self.dimension)+'_' + str(i))

    def showInfoSetSelector(self, xmlInfoSet=None, name='',additionalFolder = None):
        self.infoSetSelector = InfoSetSelector(xmlInfoSet, self.dimension, name,additionalFolder=additionalFolder,hasConstraint = self.hasConstraint)
        self.infoSetSelector.SignalValidate.connect(self.validateInfoSetSelector)
        self.infoSetSelector.SignalCancel.connect(self.cancelInfoSetSelector)
        self.setEnabled(False)
        self.infoSetSelector.show()

    def validateInfoSetSelector(self, xmlInfoSet):

        if self.infoSetSelector is None:
            name = self.comboBoxInfoSet.currentText()
            xmlInfoSet = self.comboBoxInfoSet.currentData()[0]
            additionalFolder = self.comboBoxInfoSet.currentData()[1]
            self.infoSetSelector = InfoSetSelector(xmlInfoSet, self.dimension, name, additionalFolder=additionalFolder, hasConstraint=self.hasConstraint)

        name = self.infoSetSelector.lineEditName.text()

        if self.infoSetSelector.additionalFolder is None:
            Dfct.saveXmlElement(xmlInfoSet, vrb.folderUserData + '/InfoSet/' + name + '.mho')
        else:
            Dfct.saveXmlElement(xmlInfoSet, self.infoSetSelector.additionalFolder+'/Model_Infoset.mho')
        self.fillComboBox()
        self.comboBoxInfoSet.setCurrentIndex(self.comboBoxInfoSet.findText(name))
        self.infoSetSelector.SignalValidate.disconnect(self.validateInfoSetSelector)
        self.infoSetSelector.SignalCancel.disconnect(self.cancelInfoSetSelector)
        self.infoSetSelector.close()
        self.setEnabled(True)
        self.SignalInfoSetChanged.emit()

    def cancelInfoSetSelector(self):

        self.infoSetSelector.SignalValidate.disconnect(self.validateInfoSetSelector)
        self.infoSetSelector.SignalCancel.disconnect(self.cancelInfoSetSelector)
        self.infoSetSelector.close()
        self.setEnabled(True)

    def emitInfoSetChanged(self):

        self.SignalInfoSetChanged.emit()

class InfoSetSelector(qt.QWidget):
    SignalValidate = pyqtSignal(xmlet.Element)
    SignalCancel = pyqtSignal()

    def __init__(self, xmlInfoSet=None, analysisDimension='2D', name='',additionalFolder = None,hasConstraint = False):
        super().__init__()

        self.hasConstraint = hasConstraint

        self.additionalFolder = additionalFolder
        self.dim = analysisDimension
        self.copyCurrentCustomMeasure = None

        # self.searchBoxMeasure = SearchBoxMeasure()

        self.treeWidget = MeasureTreeWidget(xmlInfoSet, self.dim, hasConstraint=hasConstraint)

        self.infoPanel = InfoPanel()

        self.labelName = qt.QLabel('Infoset name')
        self.labelName.setFixedWidth(55*vrb.ratio)
        self.lineEditName = qt.QLineEdit(name)
        self.lineEditName.setFixedWidth(80*vrb.ratio)
        self.pushButtonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=2)
        self.pushButtonValidate.setFixedSize(20*vrb.ratio,20*vrb.ratio)
        self.pushButtonValidate.setToolTip("Validate")

        self.customShapeAnalysisWidget = CustomShapeAnalysisWidget()
        self.customShapeAnalysisWidget.setVisible(False)

        self.groupBoxTreeWidget = qt.QGroupBox()
        self.layoutTreeWidget = qt.QGridLayout()
        self.layoutTreeWidget.addWidget(self.treeWidget, 0, 0)
        self.layoutTreeWidget.addWidget(self.customShapeAnalysisWidget, 1, 0)
        self.layoutTreeWidget.setContentsMargins(0,0,0,0)
        self.layoutTreeWidget.setSizeConstraint(1)
        self.groupBoxTreeWidget.setLayout(self.layoutTreeWidget)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.groupBoxTreeWidget, 0, 0 ,2 ,1)
        self.layout.addWidget(self.infoPanel, 0, 1)
        # self.layout.addWidget(self.searchBoxMeasure,1,1,Qt.AlignRight | Qt.AlignBottom)
        # self.layout.addWidget(self.customShapeAnalysisWidget, 1, 0)
        self.layoutBottom = qt.QGridLayout()
        self.groupBoxBottom = qt.QGroupBox()
        self.layoutBottom.addWidget(self.labelName, 0, 0,Qt.AlignRight)
        self.layoutBottom.addWidget(self.lineEditName, 0, 1)
        self.layoutBottom.addWidget(self.pushButtonValidate, 0, 2)
        self.layoutBottom.setHorizontalSpacing(15*vrb.ratio)
        self.layoutBottom.setContentsMargins(0,0,0,0)
        self.groupBoxBottom.setLayout(self.layoutBottom)
        self.layout.addWidget(self.groupBoxBottom, 2, 0, 1, 2)
        self.setLayout(self.layout)

        self.layout.setContentsMargins(10*vrb.ratio, 10*vrb.ratio, 10*vrb.ratio, 10*vrb.ratio)
        self.layout.setSizeConstraint(1)
        # self.layout.setSizeConstraint(2)
        # self.layout.setVerticalSpacing(10*vrb.ratio)

        self.treeWidget.currentItemChanged.connect(self.updateInfoPanel)
        self.treeWidget.SignalCreateCustomMeasure.connect(self.createCustomMeasure)
        self.pushButtonValidate.clicked.connect(self.saveInfoSet)
        self.infoPanel.pushButtonAddVariableToCustom.clicked.connect(self.addVariableToCustom)

        self.customShapeAnalysisWidget.pushButtonCancel.clicked.connect(self.cancelCustomMeasure)
        self.customShapeAnalysisWidget.pushButtonValidate.clicked.connect(self.validateCustomMeasure)

        self.setWindowTitle("Infoset")

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.resize(580*vrb.ratio,660*vrb.ratio)

    def resizeEvent(self, event):

        widthInfoPanel = 170*vrb.ratio
        self.infoPanel.setFixedSize(widthInfoPanel,self.height() - 60*vrb.ratio)
        if self.customShapeAnalysisWidget.isVisible():
            self.customShapeAnalysisWidget.setFixedSize(self.width() - widthInfoPanel-  30*vrb.ratio,self.height()*50/100)
            self.treeWidget.setFixedSize(self.width() - widthInfoPanel - 30*vrb.ratio,self.height()*50/100 - 60*vrb.ratio)
        else:
            self.treeWidget.setFixedSize(self.width() - widthInfoPanel - 30*vrb.ratio,self.height() - 40*vrb.ratio)

    def createCustomMeasure(self, xmlElement, isEdition):
        # xmlElement is either xmlParent (False) or xmlMeasure (True)
        if isEdition:
            self.customShapeAnalysisWidget.xmlMeasure = xmlElement
            self.copyCurrentCustomMeasure = Dfct.copyXmlElement(xmlElement)
        else:
            xmlParent = xmlElement
            self.customShapeAnalysisWidget.createDefaultXml(xmlParent, self.dim)
        self.customShapeAnalysisWidget.loadXml(self.customShapeAnalysisWidget.xmlMeasure)
        self.customShapeAnalysisWidget.setVisible(True)
        self.treeWidget.toggleDeleteButtons(self.customShapeAnalysisWidget.xmlMeasure)
        self.updateInfoPanel()

    def cancelCustomMeasure(self):
        xmlParent = Dfct.removeElement(self.treeWidget.xmlElement, self.customShapeAnalysisWidget.xmlMeasure)
        if self.copyCurrentCustomMeasure is not None:
            xmlParent.append(self.copyCurrentCustomMeasure)
            parent = self.treeWidget.findParentItem(self.customShapeAnalysisWidget.xmlMeasure)
            for nb in range(parent.childCount()):
                item = parent.child(nb)
                if item.widget.xmlElement == self.customShapeAnalysisWidget.xmlMeasure:
                    item.widget.xmlElement = self.copyCurrentCustomMeasure
                    break
            self.copyCurrentCustomMeasure = None
        self.customShapeAnalysisWidget.setVisible(False)
        self.updateInfoPanel()
        self.treeWidget.toggleDeleteButtons(None)

    def validateCustomMeasure(self):
        self.copyCurrentCustomMeasure = None
        customMeasure = self.customShapeAnalysisWidget.xmlMeasure
        #customMeasure.set('CheckState', '2')
        self.customShapeAnalysisWidget.saveCustomMeasure(self.treeWidget.findParentItem(customMeasure).widget.xmlElement)
        self.treeWidget.addCustomWidget(customMeasure,checkMeasure=True)
        self.customShapeAnalysisWidget.setVisible(False)
        self.updateInfoPanel()
        self.treeWidget.toggleDeleteButtons(None)

    def addVariableToCustom(self):
        self.updateInfoPanel()
        xmlMeasure = self.treeWidget.currentItem().widget.xmlElement
        self.customShapeAnalysisWidget.addVariable(xmlMeasure)

    def saveInfoSet(self):
        self.infoPanel.saveMeasureParameters()
        self.treeWidget.saveInfoSet()
        self.SignalValidate.emit(self.treeWidget.xmlElement)
        # self.SignalValidate.emit(self.treeWidget.xmlElement, self.lineEditName.text())

    def closeEvent(self, a0: QtGui.QCloseEvent) -> None:

        self.SignalCancel.emit()
        #self.saveInfoSet()
        super().closeEvent(a0)

    def updateInfoPanel(self):
        xmlMeasure = None
        try:
            xmlMeasure = self.treeWidget.currentItem().widget.xmlElement
        except Exception as e:
            pass
        self.infoPanel.changeMeasure(xmlMeasure)
        # Button is enabled only if a custom is shown
        if xmlMeasure is not None and xmlMeasure.tag == 'Measure':
            self.infoPanel.pushButtonAddVariableToCustom.setVisible(self.infoPanel.pushButtonAddVariableToCustom.isEnabled() and self.customShapeAnalysisWidget.isVisible())
        else:
            self.infoPanel.pushButtonAddVariableToCustom.setVisible(False)
        self.resizeEvent(None)

class MeasureTreeWidget(qt.QTreeWidget):
    SignalCreateCustomMeasure = pyqtSignal(xmlet.Element, bool)

    def __init__(self, xmlInfoSet, dim,hasConstraint=False):
        super().__init__()
        self.dim = dim
        self.hasConstraint=hasConstraint

        self.xmlElement = xmlInfoSet
        if self.xmlElement is not None:
            measureIterator = Dfct.childIterator(xmlInfoSet, "Measure")
            self.allMeasures = []
            for measure in measureIterator:
                self.allMeasures.append(measure)
        else:
            self.xmlElement = xmlet.Element('AllMeasures')
            self.xmlElement.set('Dimension', str(self.dim))
            self.allMeasures = []

        self.createXmlElement(vrb.folderShapeAnalysis, self.xmlElement)

        self.setHeaderHidden(True)
        self.loadXmlElement(None, self.xmlElement)
        self.addPlusButtons()

        # self.setMinimumSize(210,280)

    def createXmlElement(self, folder, xmlElement):
        for fileName in sorted(os.listdir(folder),key=str.casefold):
            filePath = folder + '/' + fileName
            if os.path.isdir(filePath):
                while fileName[0] in vrb.maskNumbers:
                    fileName = fileName[1:]
                fileName = fileName.split('_')
                if len(fileName) == 1 or fileName[1] == self.dim:
                    #Exception !!
                    if self.dim != "3D" or fileName[0] not in ["Angularity","Matching","Skeleton"]:
                        xmlCategory = Dfct.SubElement(xmlElement, fileName[0])
                        self.createXmlElement(filePath, xmlCategory)
            elif filePath.endswith('.mho'):
                try:
                    xmlMeasure = xmlet.parse(filePath).getroot()
                    objectElement = Dfct.SubElement(xmlMeasure,"Object")
                    nameMeasure = Dfct.childText(objectElement,"Name")
                    # 2D/3D
                    if Dfct.childText(xmlMeasure, 'Dimension') == self.dim or Dfct.childText(xmlMeasure, 'Dimension') == 'Both':
                        alreadyInInfoset = False
                        if nameMeasure != None and nameMeasure != "":
                            for currentMeasure in self.allMeasures:
                                currentObject = Dfct.SubElement(currentMeasure,"Object")
                                currentName = Dfct.childText(currentObject,"Name")
                                if currentName == nameMeasure:
                                    alreadyInInfoset = True
                                    # Dfct.copySubElement(xmlElement,currentMeasure)
                        if alreadyInInfoset == False:
                            xmlElement.append(xmlMeasure)
                except:
                    pass

    def loadXmlElement(self, parent, xmlElement):
        for childNode in xmlElement:
            # Create the item
            if childNode.tag == 'Measure' and childNode.get('isCustom') != str(True):
                # non custom

                widget = MeasureLabel(childNode, Dfct.childText(childNode, 'UserName'), triState=False)

                if self.hasConstraint and Dfct.childText(childNode,'UserName') in vrb.measureException:
                    widget.setEnabled(False)

                item = MyTreeWidgetItem(widget)
                widget.item = item
                widget.treeWidget = self
                self.addItem(item, parent)
            elif childNode.tag == 'Measure' and childNode.get('isCustom') == str(True):
                # add custom
                self.addCustomWidget(childNode, parent)
            else:
                # not measure
                widget = MeasureLabel(childNode, childNode.tag, triState=True)
                item = MyTreeWidgetItem(widget)
                widget.item = item
                widget.treeWidget = self
                self.addItem(item, parent)
                self.loadXmlElement(item, childNode)

    def addPlusButtons(self):
        # add '+' buttons for custom
        topLevelCustomTreeWidgetItem = self.findTopLevelCustomTreeWidgetItem()
        if topLevelCustomTreeWidgetItem is not None:
            for nb in range(topLevelCustomTreeWidgetItem.childCount()):
                child = topLevelCustomTreeWidgetItem.child(nb)
                item = CustomTreeWidgetItem(child.widget.xmlElement, isButton=True, parentItem=child)
                self.addItem(item, child)
                item.widget.pushButtonPlus.clicked.connect(self.emitSignalCreateCustomMeasure)

    def findTopLevelCustomTreeWidgetItem(self):
        topLevelCustomTreeWidgetItem = None
        for nb in range(self.topLevelItemCount()):
            # find the top level with custom
            if self.topLevelItem(nb).widget.xmlElement.tag == 'Custom':
                topLevelCustomTreeWidgetItem = self.topLevelItem(nb)
                break
        return topLevelCustomTreeWidgetItem

    def addItem(self, item, parent):
        # Insert the item in the list
        if parent is None:
            self.addTopLevelItem(item)
        else:
            parent.addChild(item)
        self.setItemWidget(item, 0, item.widget)
        checkstate = item.widget.xmlElement.get('CheckState')
        if checkstate is not None:
            if item.widget.isEnabled():
                item.widget.checkBox.setCheckState(int(checkstate))
                if checkstate != '0':
                    item.setExpanded(True)
            else:
                item.widget.checkBox.setCheckState(0)

    def emitSignalCreateCustomMeasure(self):
        xmlCurrentCustomMeasure = self.sender().xmlElement
        self.SignalCreateCustomMeasure.emit(xmlCurrentCustomMeasure, False)

    def emitSignalCreateCustomMeasure_edit(self, xmlElement):
        self.SignalCreateCustomMeasure.emit(xmlElement, True)

    def saveInfoSet(self):
        for nb in range(self.topLevelItemCount()):
            item = self.topLevelItem(nb)
            item.saveMeasure()

    def addCustomWidget(self, xmlMeasure, parent=None,checkMeasure=False):

        if parent is None:
            parent = self.findParentItem(xmlMeasure)
        for nb in range(parent.childCount()):
            if parent.child(nb).widget.xmlElement == xmlMeasure:
                # If the xmlMeasure is already displayed in the tree, nothing to do
                parent.child(nb).widget.label.setText(Dfct.childText(xmlMeasure, 'UserName'))
                return
        item = CustomTreeWidgetItem(xmlMeasure, isButton=False,checkMeasure=checkMeasure)
        item.widget.item = item
        item.widget.treeWidget = self

        parent.insertChild(0, item)
        self.setItemWidget(item, 0, item.widget)
        parent.checkChildrenState()

        item.widget.SignalDeleteClicked.connect(self.deleteCustomMeasure)
        item.widget.SignalEditClicked.connect(self.emitSignalCreateCustomMeasure_edit)

    def deleteCustomMeasure(self, customMeasureLabel):
        xmlMeasure = customMeasureLabel.xmlElement
        parent = customMeasureLabel.treeItem.parent()
        parent.widget.xmlElement.remove(xmlMeasure)
        parent.removeChild(customMeasureLabel.treeItem)
        parent.checkChildrenState()
        try:
            fileName = Dfct.childText(xmlMeasure, 'UserName') + '.mho'
            os.remove(vrb.folderShapeAnalysis + '/3Custom/' + parent.widget.xmlElement.tag + '/' + fileName)
        except Exception as e:
            print(e, file=sys.stderr)

    def findParentItem(self, xmlMeasure):
        for nb in range(self.topLevelItemCount()):
            parent = self.findParentItemReccursive(self.topLevelItem(nb), xmlMeasure)
            if parent is not None:
                return parent
        return None

    def findParentItemReccursive(self, treeItem, xmlMeasure):
        if treeItem.childCount() != 0:
            for childNode in treeItem.widget.xmlElement:
                if childNode == xmlMeasure:
                    # treeItem is the parent
                    return treeItem
            # Not found -> search in all children
            for nb in range(treeItem.childCount()):
                childItem = treeItem.child(nb)
                if childItem.widget.xmlElement == xmlMeasure:
                    return treeItem
                else:
                    res = self.findParentItemReccursive(childItem, xmlMeasure)
                    if res is not None:
                        return res
        # Not found at all
        return None

    def toggleDeleteButtons(self, newCurrentXmlCustomMeasure, treeItem=None):
        if treeItem is None:
            treeItem = self.findTopLevelCustomTreeWidgetItem()
        if treeItem.widget.xmlElement.tag == 'Measure':
            treeItem.widget.labelButtonDelete.setEnabled(treeItem.widget.xmlElement != newCurrentXmlCustomMeasure)
            treeItem.widget.labelButtonEdit.setEnabled(treeItem.widget.xmlElement != newCurrentXmlCustomMeasure)
        else:
            for nb in range(treeItem.childCount()):
                child = treeItem.child(nb)
                self.toggleDeleteButtons(newCurrentXmlCustomMeasure, child)


class InfoPanel(qt.QGroupBox):
    def __init__(self):
        super().__init__()
        self.currentMeasureXml = None
        self.currentParameters = None

        self.defaultTextLabelTop = ''
        self.defaultTextLabelDescription = ''

        # self.labelTop = qt.QLabel(self.defaultTextLabelTop)
        # # self.labelTop.setWordWrap(True)
        # titleFont = self.labelTop.font()
        # titleFont.setPixelSize(12*vrb.ratio)
        # self.labelTop.setFont(titleFont)

        self.labelTop = qt.QTextEdit()
        self.labelTop.setFixedHeight(45*vrb.ratio)
        self.labelTop.setEnabled(False)
        titleFont = self.labelTop.font()
        titleFont.setPixelSize(12*vrb.ratio)
        self.labelTop.setFont(titleFont)
        self.labelTop.setStyleSheet("QTextEdit{background-color: transparent;border :0px}")

        # self.labelDescription = qt.QLabel(self.defaultTextLabelDescription)
        # self.labelDescription.setWordWrap(True)

        self.labelDescription = qt.QTextEdit()
        self.labelDescription.setFixedHeight(55*vrb.ratio)
        self.labelDescription.setEnabled(False)
        self.labelDescription.setVerticalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)
        descriptionFont = self.labelDescription.font()
        descriptionFont.setPixelSize(9*vrb.ratio)
        self.labelDescription.setFont(descriptionFont)
        self.labelDescription.setStyleSheet("QTextEdit{background-color: transparent;border :0px}")
        # self.labelDescription.viewport().setAutoFillBackground(False)

        self.groupBoxParameters = qt.QGroupBox()
        self.groupBoxParameters.setVisible(False)
        self.groupBoxParameters.setFixedHeight(150*vrb.ratio)

        self.pushButtonDoc = wgt.PushButtonImage(margins=1, filename=vrb.folderImages + '/Interrogation.png')
        self.pushButtonDoc.setStyleSheet("background-color : transparent; border :0px")
        self.pushButtonDoc.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)
        self.pushButtonDoc.setToolTip("Open documentation")
        self.pushButtonAddVariableToCustom = wgt.PushButtonImage(vrb.folderImages + "/Add_2.png", margins=2)
        self.pushButtonAddVariableToCustom.setFixedSize(20*vrb.ratio,20*vrb.ratio)
        self.pushButtonAddVariableToCustom.setToolTip("Add as variable to current custom measure")

        emptyLabel = qt.QLabel()
        emptyLabel.setFixedHeight(20*vrb.ratio)

        self.layout = qt.QGridLayout()
        self.layout.setAlignment(Qt.AlignTop)
        self.layout.addWidget(self.labelTop, 0, 0,Qt.AlignTop)
        self.layout.addWidget(self.pushButtonDoc, 0, 1,Qt.AlignRight | Qt.AlignTop)
        self.layout.addWidget(self.labelDescription, 1, 0,Qt.AlignTop)
        self.layout.addWidget(self.groupBoxParameters, 2, 0,Qt.AlignTop)
        self.layout.addWidget(emptyLabel, 3, 0,Qt.AlignTop)
        self.layout.addWidget(self.pushButtonAddVariableToCustom, 4, 0,Qt.AlignTop)
        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)

        self.pushButtonDoc.clicked.connect(self.showDocumentation)

    def changeMeasure(self, xmlMeasure):

        self.saveMeasureParameters()
        self.currentParameters = None
        self.currentMeasureXml = xmlMeasure
        if xmlMeasure is not None and xmlMeasure.tag == 'Measure':

            self.labelTop.setText(Dfct.childText(xmlMeasure, 'UserName'))
            self.labelDescription.setText(Dfct.childText(xmlMeasure, 'Description'))

            self.groupBoxParameters.deleteLater()
            self.groupBoxParameters = ParametersGroupBox(xmlMeasure)
            self.layout.addWidget(self.groupBoxParameters, 2, 0)
            self.groupBoxParameters.setVisible(self.groupBoxParameters.groupBoxLayout.count() != 0)
            self.currentParameters = self.groupBoxParameters.currentParameters

            self.groupBoxParameters.setFixedHeight(150*vrb.ratio)

            if xmlMeasure.get('isCustom') != str(True):
                self.pushButtonAddVariableToCustom.setVisible(True)
            else:
                self.pushButtonAddVariableToCustom.setVisible(False)
        else:
            self.groupBoxParameters.setVisible(False)
            try:
                self.labelTop.setText(dictConversion[xmlMeasure.tag]['Username'])
            except:
                try:
                    self.labelTop.setText(xmlMeasure.tag)
                except:
                    self.labelTop.setText(self.defaultTextLabelTop)
            try:
                self.labelDescription.setText(dictConversion[xmlMeasure.tag]['Description'])
            except:
                self.labelDescription.setText(self.defaultTextLabelDescription)
            self.pushButtonAddVariableToCustom.setVisible(False)

    def saveMeasureParameters(self):
        if self.currentParameters is not None:
            self.groupBoxParameters.saveMeasureParameters(self.currentMeasureXml)

    def showDocumentation(self):
        try:
            """
            Opens a tab on the default browser with the doc
            If the docPath has not been set, shows the module page
            """
            objectElement = Dfct.SubElement(self.currentMeasureXml, "Object")
            docTitle = Dfct.childText(self.currentMeasureXml, "DocTitle")
            docPath = vrb.dictLinks[docTitle]

            docFound = False
            if docPath is not None:
                pagePath = vrb.folderDocEssential + '/' + docPath

                if os.path.exists(pagePath):
                    docFound=True
                    webbrowser.open_new_tab(pagePath)

            if docFound == False:
                print('Documentation not available, opening first doc page', file=sys.stderr)
                pagePath = vrb.folderDocEssential + '/measures.html'
                if os.path.exists(pagePath):
                    webbrowser.open_new_tab(pagePath)
                else:
                    print('Documentation not available, no page found', file=sys.stderr)

        except:
            traceback.print_exc(file=sys.stderr)

    def findDocLink(self, xmlElementToFind, xmlElement=None):

        if xmlElement is None:
            xmlElement = xmlet.parse(vrb.folderInformation + '/DirectoryLinks.mho').getroot()
            xmlElement = Dfct.SubElement(xmlElement, WidgetTypes.OutputType.SHAPE_ANALYSIS.value)
        if xmlElement.tag == xmlElementToFind.tag:
            return xmlElement.text
        else:
            for childNode in xmlElement:
                link = self.findDocLink(xmlElementToFind, childNode)
                if link is not None:
                    return link
        return None


class ParametersGroupBox(qt.QGroupBox):
    def __init__(self, xmlMeasure):
        super(ParametersGroupBox, self).__init__()
        self.groupBoxLayout = qt.QGridLayout()
        self.setLayout(self.groupBoxLayout)
        self.groupBoxLayout.setAlignment(Qt.AlignTop)
        self.currentParameters = None

        for objectArg in Dfct.SubElement(xmlMeasure, 'Object'):
            if objectArg.tag == 'Parameters':
                self.currentParameters = {}
                self.groupBoxLayout.setContentsMargins(0, 0, 0, 0)
                for nbParam, paramNode in enumerate(objectArg, 0):
                    try:
                        label, widget, constraints, type,defaultValue = fctWidget.generateParamWidget(paramNode)
                        if isinstance(dictConversion[Dfct.childText(paramNode, 'Name')]['UserName'], list):
                            label = qt.QLabel(dictConversion[Dfct.childText(paramNode, 'Name')]['UserName'][nbParam])
                        else:
                            label = qt.QLabel(dictConversion[Dfct.childText(paramNode, 'Name')]['UserName'])
                        if widget is not None:
                            self.currentParameters[nbParam] = {}
                            self.currentParameters[nbParam]['Type'] = type
                            self.currentParameters[nbParam]['Label'] = label
                            self.currentParameters[nbParam]['Widget'] = widget
                            self.currentParameters[nbParam]['Constraints'] = constraints
                            self.groupBoxLayout.addWidget(self.currentParameters[nbParam]['Label'], nbParam, 0, Qt.AlignHCenter)
                            self.groupBoxLayout.addWidget(self.currentParameters[nbParam]['Widget'], nbParam, 1, Qt.AlignLeft)
                    except:
                        traceback.print_exc(file=sys.stderr)

    def saveMeasureParameters(self, xmlMeasureParameters):
        if self.currentParameters is not None:
            for objectArg in Dfct.SubElement(xmlMeasureParameters, 'Object'):
                if objectArg.tag == 'Parameters':
                    for nbParam, paramNode in enumerate(objectArg, 0):
                        try:
                            xmlValue = self.currentParameters[nbParam]['Widget'].interfaceToXml(0)
                            Dfct.SubElement(paramNode, 'Value').text = Dfct.SubElement(xmlValue, 'Value').text
                        except:
                            pass

if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook

    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    foo = InfoSetSelector()
    foo.show()

    app.exec_()