import sys
import os
from pathlib import Path

import PyQt5.QtWidgets as qt
from PyQt5 import QtCore
from PyQt5 import QtGui
from PyQt5.QtCore import pyqtSignal,QPointF,Qt,QCoreApplication

import UsefullVariables as vrb
import UsefullTexts as txt
import UsefullWidgets as wgt
import InterfaceFunction as uifct

import DatabaseFunction as Dfct
import UsefullFunctions as fct
import xml.etree.ElementTree as xmlet

import shutil

import PIL
from PIL import Image, ImageFont, ImageDraw

import numpy as np
import cv2

import PyIPSDK.IPSDKIPLUtility as util

import PyIPSDK.IPSDKUI as ui
import WidgetTypes

from FunctionsDictionary import nativeModules

import PyIPSDK

class MovieMakerPreferencesWidget(qt.QWidget):

    def __init__(self):
        super().__init__()

        try:
            file = xmlet.parse(vrb.folderInformation + "/MovieMakerPreferences.mho")
            self.settingsElement = file.getroot()
        except:
            self.settingsElement = xmlet.Element('MovieMakerPreferences')

        self.checkBoxLogo = qt.QCheckBox("Add logo to the movie")
        self.checkBoxLogo.setFixedSize(100*vrb.ratio,20*vrb.ratio)
        self.checkBoxLogo.setChecked(True)

        self.labelLogo = qt.QLabel("Logo path")
        self.labelLogo.setFixedSize(100*vrb.ratio,20*vrb.ratio)
        self.lineEditLogoPath = qt.QLineEdit()
        self.lineEditLogoPath.setPlaceholderText(vrb.folderImages + "/Explorer_Logo_Bleu.png")
        self.lineEditLogoPath.setFixedSize(140*vrb.ratio,20*vrb.ratio)
        self.buttonOpenFolder = wgt.PushButtonImage(vrb.folderImages + "/Folder.png")
        self.buttonOpenFolder.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.positionLogo = qt.QLabel("Position logo")
        self.positionLogo.setFixedSize(100*vrb.ratio,20*vrb.ratio)
        self.comboBoxPosition = qt.QComboBox()
        self.positionLogo.setFixedSize(100*vrb.ratio,30*vrb.ratio)
        self.comboBoxPosition.addItem("Bottom Right")
        self.comboBoxPosition.addItem("Bottom Left")
        self.comboBoxPosition.addItem("Top Left")
        self.comboBoxPosition.addItem("Top Right")

        self.labelSizeLogo = qt.QLabel("Size logo (%)")
        self.labelSizeLogo.setFixedSize(100*vrb.ratio,20*vrb.ratio)
        self.spinBoxSizeLogo = qt.QSpinBox()
        self.spinBoxSizeLogo.setRange(10,50)
        self.spinBoxSizeLogo.setValue(25)
        self.spinBoxSizeLogo.setFixedSize(40*vrb.ratio,20*vrb.ratio)

        self.buttonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=0)
        self.buttonValidate.setFixedSize(30, 30)
        self.buttonValidate.setToolTip("Validate")

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.checkBoxLogo, 0, 0)
        self.layout.addWidget(self.labelLogo, 1, 0)
        self.layout.addWidget(self.lineEditLogoPath, 1, 1)
        self.layout.addWidget(self.buttonOpenFolder, 1, 2)
        self.layout.addWidget(self.positionLogo,2,0)
        self.layout.addWidget(self.comboBoxPosition,2,1)
        self.layout.addWidget(self.labelSizeLogo,3,0)
        self.layout.addWidget(self.spinBoxSizeLogo,3,1)
        self.layout.addWidget(self.buttonValidate, 4, 2,Qt.AlignRight)

        self.layout.setAlignment(Qt.AlignTop)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10*vrb.ratio,10*vrb.ratio,10*vrb.ratio,10*vrb.ratio)
        self.resize(200*vrb.ratio,120*vrb.ratio)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setWindowTitle("Movie Maker Preferences")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.checkBoxLogo.stateChanged.connect(self.checkBoxLogoChanged)
        self.buttonOpenFolder.clicked.connect(self.openQDialog)
        self.buttonValidate.clicked.connect(self.validate)

        self.readXmlElement()
        self.checkBoxLogoChanged()

    def openQDialog(self):

        defaultFolder = os.path.dirname(vrb.folderImages)
        filename = qt.QFileDialog.getOpenFileName(self, "Select your logo", defaultFolder)
        if filename[0] != [] and filename[0] != '' and filename[0] != None:
            self.lineEditLogoPath.setText(filename[0])

    def readXmlElement(self):

        addLogoElement = Dfct.SubElement(self.settingsElement,"AddLogo")
        pathLogoElement = Dfct.SubElement(self.settingsElement,"PathLogo")
        positionLogoElement = Dfct.SubElement(self.settingsElement,"PositionLogo")
        sizeLogoElement = Dfct.SubElement(self.settingsElement,"SizeLogo")

        self.checkBoxLogo.setChecked(addLogoElement.text != "False")

        if pathLogoElement.text is not None and pathLogoElement.text != "":
            self.lineEditLogoPath.setText(pathLogoElement.text)
        if positionLogoElement.text is not None and positionLogoElement.text != "":
            self.comboBoxPosition.setCurrentIndex(int(positionLogoElement.text))
        if sizeLogoElement.text is not None and sizeLogoElement.text != "":
            self.spinBoxSizeLogo.setValue(int(sizeLogoElement.text))

    def validate(self):

        addLogoElement = Dfct.SubElement(self.settingsElement,"AddLogo")
        pathLogoElement = Dfct.SubElement(self.settingsElement,"PathLogo")
        positionLogoElement = Dfct.SubElement(self.settingsElement,"PositionLogo")
        sizeLogoElement = Dfct.SubElement(self.settingsElement,"SizeLogo")

        if self.checkBoxLogo.isChecked():
            addLogoElement.text = "True"
        else:
            addLogoElement.text = "False"
        pathLogoElement.text = self.lineEditLogoPath.text()
        positionLogoElement.text = str(self.comboBoxPosition.currentIndex())
        sizeLogoElement.text = str(self.spinBoxSizeLogo.value())

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/MovieMakerPreferences.mho")

        self.close()

    def closeEvent(self, a0: QtGui.QCloseEvent) -> None:

        self.readXmlElement()

    def checkBoxLogoChanged(self):

        enabled = self.checkBoxLogo.isChecked()
        self.lineEditLogoPath.setEnabled(enabled)
        self.buttonOpenFolder.setEnabled(enabled)
        self.comboBoxPosition.setEnabled(enabled)
        self.spinBoxSizeLogo.setEnabled(enabled)

class DiskImagesPreferencesWidget(qt.QWidget):

    def __init__(self):
        super().__init__()

        try:
            file = xmlet.parse(vrb.folderInformation + "/DiskImage.mho")
            self.settingsElement = file.getroot()
        except:
            self.settingsElement = xmlet.Element('DiskImage')

        self.folderTmp = PyIPSDK.getIPSDKDefaultDirectory(PyIPSDK.eDefaultExternalDirectory.eDED_Tmp)

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/DiskImage.mho")

        self.labelFolderToSave = qt.QLabel("Folder to save disk images")
        self.labelFolderToSave.setFixedSize(160*vrb.ratio,20*vrb.ratio)
        self.lineEditFolderToSave = qt.QLineEdit()
        # self.lineEditFolderToSave.setPlaceholderText(str(Path.home()) + "AppData/Local/Temp/ReactivIP/IPSDK")
        self.lineEditFolderToSave.setPlaceholderText(self.folderTmp)
        self.lineEditFolderToSave.setFixedSize(200*vrb.ratio,20*vrb.ratio)
        self.buttonOpenFolderToSave = wgt.PushButtonImage(vrb.folderImages + "/Folder.png")
        self.buttonOpenFolderToSave.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.labelConvertAfterCrop = qt.QLabel("Automatically convert into memory\nimage after cropping")
        self.labelConvertAfterCrop.setFixedSize(160 * vrb.ratio, 50 * vrb.ratio)
        self.checkBoxConvertAfterCrop = qt.QCheckBox()
        self.checkBoxConvertAfterCrop.setChecked(True)

        self.labelConvertAfterExtract = qt.QLabel("Automatically convert into memory\nimage after extracting a plan")
        self.labelConvertAfterExtract.setFixedSize(160 * vrb.ratio, 50 * vrb.ratio)
        self.checkBoxConvertAfterExtract = qt.QCheckBox()
        self.checkBoxConvertAfterExtract.setChecked(True)

        self.buttonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=0)
        self.buttonValidate.setFixedSize(30, 30)
        self.buttonValidate.setToolTip("Validate")

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelFolderToSave, 0, 0)
        self.layout.addWidget(self.lineEditFolderToSave, 0, 1)
        self.layout.addWidget(self.buttonOpenFolderToSave, 0, 2)
        self.layout.addWidget(self.labelConvertAfterCrop,1,0)
        self.layout.addWidget(self.checkBoxConvertAfterCrop,1,1)
        self.layout.addWidget(self.labelConvertAfterExtract,2,0)
        self.layout.addWidget(self.checkBoxConvertAfterExtract,2,1)
        self.layout.addWidget(self.buttonValidate, 3, 2,Qt.AlignRight)

        self.layout.setAlignment(Qt.AlignTop)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10*vrb.ratio,10*vrb.ratio,10*vrb.ratio,10*vrb.ratio)
        self.resize(240*vrb.ratio,165*vrb.ratio)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setWindowTitle("Disk images Preferences")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonOpenFolderToSave.clicked.connect(self.openQDialog)
        self.buttonValidate.clicked.connect(self.validate)

        self.readXmlElement()

    def openQDialog(self):

        defaultFolder = self.lineEditFolderToSave.text()
        if defaultFolder == "" or os.path.exists(defaultFolder) == False:
            # defaultFolder = str(Path.home()) + "/AppData/Local/Temp/ReactivIP/IPSDK"
            defaultFolder = self.folderTmp
            if not os.path.exists(defaultFolder):
                try:
                    os.makedirs(defaultFolder)
                except:
                    pass

        filename = qt.QFileDialog.getExistingDirectory(self, "Select the folder to save disk images", defaultFolder)

        if filename != "" and filename != None:
            self.lineEditFolderToSave.setText(filename)

    def readXmlElement(self):

        folderToSaveElement = Dfct.SubElement(self.settingsElement,"FolderToSave")
        convertAfterCropElement = Dfct.SubElement(self.settingsElement,"ConvertAfterCrop")
        convertAfterExtractElement = Dfct.SubElement(self.settingsElement,"ConvertAfterExtract")

        if folderToSaveElement.text is not None and folderToSaveElement.text != "":
            self.lineEditFolderToSave.setText(folderToSaveElement.text)
            if os.path.exists(folderToSaveElement.text):
                PyIPSDK.setAllocatedDiskImagePath(folderToSaveElement.text)
            else:
                # PyIPSDK.setAllocatedDiskImagePath(str(Path.home()) + "/AppData/Local/Temp/ReactivIP/IPSDK")
                PyIPSDK.setAllocatedDiskImagePath(self.folderTmp)
        if convertAfterCropElement.text == "False":
            self.checkBoxConvertAfterCrop.setChecked(False)
        else:
            self.checkBoxConvertAfterCrop.setChecked(True)
        if convertAfterExtractElement.text == "False":
            self.checkBoxConvertAfterExtract.setChecked(False)
        else:
            self.checkBoxConvertAfterExtract.setChecked(True)

    def validate(self):

        folderToSaveElement = Dfct.SubElement(self.settingsElement,"FolderToSave")
        convertAfterCropElement = Dfct.SubElement(self.settingsElement,"ConvertAfterCrop")
        convertAfterExtractElement = Dfct.SubElement(self.settingsElement,"ConvertAfterExtract")

        if self.lineEditFolderToSave.text() != "":
            folderToSaveElement.text = self.lineEditFolderToSave.text()
            if os.path.exists(folderToSaveElement.text):
                PyIPSDK.setAllocatedDiskImagePath(folderToSaveElement.text)
        else:
            folderToSaveElement.text = ""
            # PyIPSDK.setAllocatedDiskImagePath(str(Path.home()) + "/AppData/Local/Temp/ReactivIP/IPSDK")
            PyIPSDK.setAllocatedDiskImagePath(self.folderTmp)

        if self.checkBoxConvertAfterCrop.isChecked():
            convertAfterCropElement.text = "True"
        else:
            convertAfterCropElement.text = "False"

        if self.checkBoxConvertAfterExtract.isChecked():
            convertAfterExtractElement.text = "True"
        else:
            convertAfterExtractElement.text = "False"

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/DiskImage.mho")

        self.close()


class ZStackFocusWidget(qt.QWidget):

    def __init__(self):
        super().__init__()

        try:
            file = xmlet.parse(vrb.folderInformation + "/Settings.mho")
            self.settingsElement = file.getroot()
        except:
            self.settingsElement = xmlet.Element('Settings')

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/Settings.mho")

        self.labelMethod = qt.QLabel("Method")
        self.labelMethod.setFixedSize(150,30)
        self.comboBoxMethod = qt.QComboBox()
        self.comboBoxMethod.addItem("Ponderation")
        self.comboBoxMethod.addItem("Maximum")
        self.comboBoxMethod.setFixedSize(150,30)

        self.labelContrastSize = qt.QLabel("Contrast size")
        self.labelContrastSize.setFixedSize(150,30)
        self.lineEditContrastSize = wgt.NumberLineEdit(constraint="Natural")
        self.lineEditContrastSize.setPlaceholderText("10")
        self.lineEditContrastSize.setFixedSize(50,30)

        self.labelMeanSmoothing = qt.QLabel("Mean smoothing")
        self.labelMeanSmoothing.setFixedSize(150,30)
        self.lineEditMeanSmoothing = wgt.NumberLineEdit(constraint="Natural")
        self.lineEditMeanSmoothing.setPlaceholderText("1")
        self.lineEditMeanSmoothing.setFixedSize(50,30)

        self.labelZoomFactor = qt.QLabel("Zoom factor")
        self.labelZoomFactor.setFixedSize(150,30)
        self.lineEditZoomFactor = wgt.NumberLineEdit()
        self.lineEditZoomFactor.setPlaceholderText("0.25")
        self.lineEditZoomFactor.setFixedSize(50,30)

        self.buttonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=0)
        self.buttonValidate.setFixedSize(30, 30)
        self.buttonValidate.setToolTip("Validate")

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelMethod, 0, 0)
        self.layout.addWidget(self.comboBoxMethod, 0, 1)
        self.layout.addWidget(self.labelContrastSize, 1, 0)
        self.layout.addWidget(self.lineEditContrastSize, 1, 1)
        self.layout.addWidget(self.labelMeanSmoothing, 2, 0)
        self.layout.addWidget(self.lineEditMeanSmoothing, 2, 1)
        self.layout.addWidget(self.labelZoomFactor, 3, 0)
        self.layout.addWidget(self.lineEditZoomFactor, 3, 1)
        self.layout.addWidget(self.buttonValidate, 4,2,Qt.AlignRight)

        self.layout.setAlignment(Qt.AlignTop)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10,10,10,10)
        self.resize(320,250)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setWindowTitle("Z Stack Focus Preferences")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonValidate.clicked.connect(self.validate)

        self.readXmlElement()

    def readXmlElement(self):

        methodElement = Dfct.SubElement(self.settingsElement,"ZStackMethod")
        contrastSizeElement = Dfct.SubElement(self.settingsElement,"ZStackContrastSize")
        meanSmoothingElement = Dfct.SubElement(self.settingsElement,"ZStackMeanSmoothing")
        zoomFactorElement = Dfct.SubElement(self.settingsElement,"ZStackZoomFactor")

        if methodElement.text is not None:
           self.comboBoxMethod.setCurrentIndex(int(methodElement.text))
        else:
            self.comboBoxMethod.setCurrentIndex(0)
        if contrastSizeElement.text is not None:
           self.lineEditContrastSize.setText(contrastSizeElement.text)
        else:
            self.lineEditContrastSize.setText("")
        if meanSmoothingElement.text is not None:
           self.lineEditMeanSmoothing.setText(meanSmoothingElement.text)
        else:
            self.lineEditMeanSmoothing.setText("")
        if zoomFactorElement.text is not None:
           self.lineEditZoomFactor.setText(zoomFactorElement.text)
        else:
            self.lineEditZoomFactor.setText("")

    def validate(self):

        methodElement = Dfct.SubElement(self.settingsElement,"ZStackMethod")
        contrastSizeElement = Dfct.SubElement(self.settingsElement,"ZStackContrastSize")
        meanSmoothingElement = Dfct.SubElement(self.settingsElement,"ZStackMeanSmoothing")
        zoomFactorElement = Dfct.SubElement(self.settingsElement,"ZStackZoomFactor")

        methodElement.text = str(self.comboBoxMethod.currentIndex())

        if self.lineEditContrastSize.text() is not None and self.lineEditContrastSize.text() != "":
            contrastSizeElement.text = self.lineEditContrastSize.text()
        else:
            contrastSizeElement.text = "10"
        if self.lineEditMeanSmoothing.text() is not None and self.lineEditMeanSmoothing.text() != "":
            meanSmoothingElement.text = self.lineEditMeanSmoothing.text()
        else:
            meanSmoothingElement.text = "1"
        if self.lineEditZoomFactor.text() is not None and self.lineEditZoomFactor.text() != "":
            zoomFactorElement.text = self.lineEditZoomFactor.text()
        else:
            zoomFactorElement.text = "0.25"

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/Settings.mho")

        self.close()

class WidgetDeleteModule(qt.QWidget):

    def __init__(self):
        super().__init__()

        self.label = qt.QLabel("Module to delete")
        self.comboBox = qt.QComboBox()
        self.buttonDelete = wgt.PushButtonImage(vrb.folderImages + "/Remove.png", margins=2)
        self.buttonDelete.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.label, 0, 0)
        self.layout.addWidget(self.comboBox, 0, 1)
        self.layout.addWidget(self.buttonDelete, 0, 2)

        self.layout.setAlignment(Qt.AlignTop)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10, 10, 10, 10)
        self.resize(250 * vrb.ratio, 40 * vrb.ratio)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setWindowTitle("Remove Module")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonDelete.clicked.connect(self.deleteModule)

    def display(self):

        self.comboBox.clear()
        for i in range(vrb.mainWindow.categorySpoilersContainer.layout.count() + 1):
            item = vrb.mainWindow.categorySpoilersContainer.layout.itemAt(i)
            if item is not None:
                spoilerCategory = item.widget()
                if spoilerCategory.title not in nativeModules and spoilerCategory.isVisible():
                    self.comboBox.addItem(spoilerCategory.title,spoilerCategory)

        self.show()

    def deleteModule(self):

        try:
            filePath = vrb.folderFunctions + "/" + self.comboBox.currentText()
            try:
                shutil.rmtree(filePath)
            except:
                import traceback
                traceback.print_exc(file=sys.stderr)
            self.comboBox.currentData().setVisible(False)
            self.close()
        except:
            pass

class WidgetAddModule(qt.QWidget):

    def __init__(self):
        super().__init__()


        self.label = qt.QLabel("Module name")
        self.lineEdit = qt.QLineEdit()
        self.buttonAdd = wgt.PushButtonImage(vrb.folderImages + "/Add_2.png", margins=2)
        self.buttonAdd.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.label, 0, 0)
        self.layout.addWidget(self.lineEdit, 0, 1)
        self.layout.addWidget(self.buttonAdd, 0, 2)

        self.layout.setAlignment(Qt.AlignTop)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10, 10, 10, 10)
        self.resize(250 * vrb.ratio, 40 * vrb.ratio)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setWindowTitle("Add Module")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonAdd.clicked.connect(self.addModule)

    def addModule(self):

        moduleName = self.lineEdit.text()
        if moduleName != "":
            vrb.mainWindow.addModule(moduleName)
            self.close()


class Folder3DWidget(qt.QWidget):
    """
    Interactive code editor
    Has a quick guide
    """
    def __init__(self):
        super().__init__()

        try:
            file = xmlet.parse(vrb.folderInformation + "/folder3D.mho")
            self.folder3DElement = file.getroot()
        except:
            self.folder3DElement = xmlet.Element('folder3D')
        path = Dfct.childText(self.folder3DElement, "Path")

        # Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/Settings.mho")

        self.labelFolder3D = qt.QLabel("Viewer 3D path")
        self.lineEditFolder3D = qt.QLineEdit()
        if path is not None:
            self.lineEditFolder3D.setText(path)
        self.buttonFolder3D = qt.QPushButton("...")

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelFolder3D, 0, 0)
        self.layout.addWidget(self.lineEditFolder3D, 0, 1)
        self.layout.addWidget(self.buttonFolder3D, 0,2)

        self.layout.setAlignment(Qt.AlignTop)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10,10,10,10)
        self.resize(350*vrb.ratio,40*vrb.ratio)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setWindowTitle("Settings viewer 3D")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)
        
        self.buttonFolder3D.clicked.connect(self.getPath)
    
    def getPath(self):
        
        path = Dfct.childText(self.folder3DElement, "Path")
        if path is not None:
            defaultFolder = os.path.dirname(path)
        else:
            defaultFolder = "C:/Program Files/"
        filenameExe = qt.QFileDialog.getOpenFileName(self, "Select your 3D viewer", defaultFolder)
        if len(filenameExe[0]) != 0 and filenameExe[0] != '' and filenameExe[0] != None:
            Dfct.SubElement(self.folder3DElement, "Path").text = filenameExe[0]
            Dfct.saveXmlElement(self.folder3DElement, vrb.folderInformation + "/folder3D.mho")
            self.lineEditFolder3D.setText(filenameExe[0])
    
class InterfacePreferencesWidget(qt.QWidget):

    def __init__(self):
        super().__init__()

        try:
            file = xmlet.parse(vrb.folderInformation + "/Settings.mho")
            self.settingsElement = file.getroot()
        except:
            self.settingsElement = xmlet.Element('Settings')

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/Settings.mho")

        self.labelRatioDimensions = qt.QLabel("Ratio dimensions")
        self.labelRatioDimensions.setFixedSize(160*vrb.ratio,30*vrb.ratio)
        self.lineEditRatioDimensions = wgt.NumberLineEdit()
        self.lineEditRatioDimensions.setPlaceholderText("1")
        self.lineEditRatioDimensions.setFixedSize(50*vrb.ratio,30*vrb.ratio)
        # self.labeledLineEditRatioDimensions = wgt.LabeledWidget(self.lineEditRatioDimensions,title="Ratio dimensions")

        self.labelProcessFunctions = qt.QLabel("Font process functions")
        self.labelProcessFunctions.setFixedSize(160*vrb.ratio,30*vrb.ratio)
        self.lineEditProcessFunctions = wgt.NumberLineEdit()
        self.lineEditProcessFunctions.setPlaceholderText("8")
        self.lineEditProcessFunctions.setFixedSize(50*vrb.ratio,30*vrb.ratio)

        self.labelMachineLearning = qt.QLabel("Font Machine Learning")
        self.labelMachineLearning.setFixedSize(160*vrb.ratio,30*vrb.ratio)
        self.lineEditMachineLearning = wgt.NumberLineEdit()
        self.lineEditMachineLearning.setPlaceholderText("10")
        self.lineEditMachineLearning.setFixedSize(50*vrb.ratio,30*vrb.ratio)

        self.labelDecimalSeparator = qt.QLabel("Decimal separator")
        self.labelDecimalSeparator.setFixedSize(160*vrb.ratio,30*vrb.ratio)
        groupBoxPoint = qt.QGroupBox()
        layoutPoint = qt.QHBoxLayout()
        groupBoxPoint.setLayout(layoutPoint)
        self.radioButtonPoint = qt.QRadioButton('.')
        self.radioButtonPoint.setChecked(True)
        self.radioButtonVirgule = qt.QRadioButton(',')
        layoutPoint.addWidget(self.radioButtonPoint)
        layoutPoint.addWidget(self.radioButtonVirgule)

        self.labelSignificantDigits = qt.QLabel("Significant digits")
        self.labelSignificantDigits.setFixedSize(160*vrb.ratio,30*vrb.ratio)
        self.lineEditSignificantDigits = wgt.NumberLineEdit(constraint="Natural")
        self.lineEditSignificantDigits.setPlaceholderText("3")
        self.lineEditSignificantDigits.setFixedSize(50*vrb.ratio,30*vrb.ratio)

        # self.labelMergeAnalysisFormat = qt.QLabel("Merge analysis format")
        self.labelMergeAnalysisFormat = qt.QLabel("Table format")
        self.labelMergeAnalysisFormat.setFixedSize(160*vrb.ratio,30*vrb.ratio)
        groupBoxFormat = qt.QGroupBox()
        layoutFormat = qt.QHBoxLayout()
        groupBoxFormat.setLayout(layoutFormat)
        self.radioButtonCsv = qt.QRadioButton('csv')
        self.radioButtonCsv.setChecked(True)
        self.radioButtonXls = qt.QRadioButton('xls')
        layoutFormat.addWidget(self.radioButtonCsv)
        layoutFormat.addWidget(self.radioButtonXls)

        self.labelBatchAnalysisFormat = qt.QLabel("Batch analysis format")
        self.labelBatchAnalysisFormat.setFixedSize(160*vrb.ratio,30*vrb.ratio)
        groupBoxBatchAnalysisFormat = qt.QGroupBox()
        layoutBatchAnalysisFormat = qt.QHBoxLayout()
        groupBoxBatchAnalysisFormat.setLayout(layoutBatchAnalysisFormat)
        self.radioButtonBatchAnalysisFormatCsv = qt.QRadioButton('csv')
        self.radioButtonBatchAnalysisFormatCsv.setChecked(True)
        self.radioButtonBatchAnalysisFormatXls = qt.QRadioButton('xls')
        self.radioButtonBatchAnalysisFormatXml = qt.QRadioButton('xml')
        layoutBatchAnalysisFormat.addWidget(self.radioButtonBatchAnalysisFormatCsv)
        layoutBatchAnalysisFormat.addWidget(self.radioButtonBatchAnalysisFormatXls)
        layoutBatchAnalysisFormat.addWidget(self.radioButtonBatchAnalysisFormatXml)

        self.labelNbShapes = qt.QLabel("Number of shapes\nin analysis table")
        self.labelNbShapes.setFixedSize(160*vrb.ratio,30*vrb.ratio)
        self.lineNbShapes = wgt.NumberLineEdit()
        self.lineNbShapes.setPlaceholderText("100")
        self.lineNbShapes.setFixedSize(50*vrb.ratio,30*vrb.ratio)

        self.labelDisplay3DShapes = qt.QLabel("Automatically display\n3D shapes")
        self.labelDisplay3DShapes.setFixedSize(160*vrb.ratio,50*vrb.ratio)
        self.checkBoxDisplay3DShapes = qt.QCheckBox()
        self.checkBoxDisplay3DShapes.setChecked(True)

        self.labelWarningMessageExplorerClosed = qt.QLabel("Warning message\nwhen Explorer is closed")
        self.labelWarningMessageExplorerClosed.setFixedSize(160*vrb.ratio,50*vrb.ratio)
        self.checkBoxWarningMessageExplorerClosed = qt.QCheckBox()
        self.checkBoxWarningMessageExplorerClosed.setChecked(True)

        self.labelBackupSessions = qt.QLabel("Save backup\nsessions automaticaly")
        self.labelBackupSessions.setFixedSize(160*vrb.ratio,50*vrb.ratio)
        self.checkBoxBackupSessions = qt.QCheckBox()
        self.checkBoxBackupSessions.setChecked(True)

        self.buttonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=0)
        self.buttonValidate.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        self.buttonValidate.setToolTip("Validate")

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelRatioDimensions, 0, 0)
        self.layout.addWidget(self.lineEditRatioDimensions, 0, 1)
        # self.layout.addWidget(self.labeledLineEditRatioDimensions, 0, 1,1,2)
        self.layout.addWidget(self.labelProcessFunctions, 1, 0)
        self.layout.addWidget(self.lineEditProcessFunctions, 1, 1)
        self.layout.addWidget(self.labelMachineLearning, 2, 0)
        self.layout.addWidget(self.lineEditMachineLearning, 2, 1)
        self.layout.addWidget(self.labelDecimalSeparator, 3, 0)
        self.layout.addWidget(groupBoxPoint, 3, 1)
        self.layout.addWidget(self.labelSignificantDigits, 4, 0)
        self.layout.addWidget(self.lineEditSignificantDigits, 4, 1)
        self.layout.addWidget(self.labelMergeAnalysisFormat, 5, 0)
        self.layout.addWidget(groupBoxFormat, 5, 1)
        self.layout.addWidget(self.labelBatchAnalysisFormat, 6, 0)
        self.layout.addWidget(groupBoxBatchAnalysisFormat, 6, 1)
        self.layout.addWidget(self.labelNbShapes, 7, 0)
        self.layout.addWidget(self.lineNbShapes, 7, 1)
        self.layout.addWidget(self.labelDisplay3DShapes, 8, 0)
        self.layout.addWidget(self.checkBoxDisplay3DShapes, 8, 1)
        self.layout.addWidget(self.labelWarningMessageExplorerClosed, 9, 0)
        self.layout.addWidget(self.checkBoxWarningMessageExplorerClosed, 9, 1)
        self.layout.addWidget(self.labelBackupSessions, 10, 0)
        self.layout.addWidget(self.checkBoxBackupSessions, 10, 1)
        self.layout.addWidget(self.buttonValidate, 11, 1,Qt.AlignRight)
        # self.layout.addWidget(self.buttonReinit, 1, 1,Qt.AlignRight)

        self.layout.setAlignment(Qt.AlignTop)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10*vrb.ratio,10*vrb.ratio,10*vrb.ratio,10*vrb.ratio)
        self.resize(240*vrb.ratio,280*vrb.ratio)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setWindowTitle("Interface Preferences")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonValidate.clicked.connect(self.validate)

        self.readXmlElement()

    def readXmlElement(self):

        ratioDimensionElement = Dfct.SubElement(self.settingsElement,"RatioDimensions")
        fontProcessFunctionsElement = Dfct.SubElement(self.settingsElement,"FontProcessFunctions")
        fontMachineLearningElement = Dfct.SubElement(self.settingsElement,"FontMachineLearning")
        decimalSeparatorElement = Dfct.SubElement(self.settingsElement,"DecimalSeparator")
        significantDigitsElement = Dfct.SubElement(self.settingsElement,"NumberOfDigits")
        mergeAnalysisFormatElement = Dfct.SubElement(self.settingsElement,"MergeAnalysisFormat")
        batchAnalysisFormatElement = Dfct.SubElement(self.settingsElement,"BatchAnalysisFormat")
        nbShapesElement = Dfct.SubElement(self.settingsElement,"NbShapes")
        warningMessageExplorerClosedElement = Dfct.SubElement(self.settingsElement,"WarningMessageExplorerClosed")
        saveBackupSessionsElement = Dfct.SubElement(self.settingsElement,"SaveBackupSessions")
        display3DShapesElement = Dfct.SubElement(self.settingsElement,"Display3DShapes")

        if ratioDimensionElement.text is not None:
           self.lineEditRatioDimensions.setText(ratioDimensionElement.text)
        if fontProcessFunctionsElement.text is not None:
           self.lineEditProcessFunctions.setText(fontProcessFunctionsElement.text)
        if fontMachineLearningElement.text is not None:
           self.lineEditMachineLearning.setText(fontMachineLearningElement.text)
        if decimalSeparatorElement.text == ",":
            self.radioButtonVirgule.setChecked(True)
        else:
            self.radioButtonPoint.setChecked(True)
        if significantDigitsElement.text is not None:
           self.lineEditSignificantDigits.setText(significantDigitsElement.text)
        if mergeAnalysisFormatElement.text == "csv":
            self.radioButtonCsv.setChecked(True)
        else:
            self.radioButtonXls.setChecked(True)
        if batchAnalysisFormatElement.text == "xml":
            self.radioButtonBatchAnalysisFormatXml.setChecked(True)
        elif batchAnalysisFormatElement.text == "xls":
            self.radioButtonBatchAnalysisFormatXls.setChecked(True)
        else:
            self.radioButtonBatchAnalysisFormatCsv.setChecked(True)
        if nbShapesElement.text is not None:
           self.lineNbShapes.setText(nbShapesElement.text)
        if display3DShapesElement.text == "False":
            self.checkBoxDisplay3DShapes.setChecked(False)
        else:
            self.checkBoxDisplay3DShapes.setChecked(True)
        if warningMessageExplorerClosedElement.text == "False":
            self.checkBoxWarningMessageExplorerClosed.setChecked(False)
        else:
            self.checkBoxWarningMessageExplorerClosed.setChecked(True)
        if saveBackupSessionsElement.text == "False":
            self.checkBoxBackupSessions.setChecked(False)
        else:
            self.checkBoxBackupSessions.setChecked(True)

    def validate(self):

        ratioDimensionElement = Dfct.SubElement(self.settingsElement, "RatioDimensions")
        fontProcessFunctionsElement = Dfct.SubElement(self.settingsElement,"FontProcessFunctions")
        fontMachineLearningElement = Dfct.SubElement(self.settingsElement,"FontMachineLearning")
        decimalSeparatorElement = Dfct.SubElement(self.settingsElement, "DecimalSeparator")
        significantDigitsElement = Dfct.SubElement(self.settingsElement,"NumberOfDigits")
        mergeAnalysisFormatElement = Dfct.SubElement(self.settingsElement,"MergeAnalysisFormat")
        batchAnalysisFormatElement = Dfct.SubElement(self.settingsElement,"BatchAnalysisFormat")
        nbShapesElement = Dfct.SubElement(self.settingsElement,"NbShapes")
        warningMessageExplorerClosedElement = Dfct.SubElement(self.settingsElement,"WarningMessageExplorerClosed")
        saveBackupSessionsElement = Dfct.SubElement(self.settingsElement,"SaveBackupSessions")
        display3DShapesElement = Dfct.SubElement(self.settingsElement,"Display3DShapes")

        try:
            if ratioDimensionElement.text == self.lineEditRatioDimensions.text():
                showMessage = False
            else:
                showMessage = True
        except:
            showMessage = True

        if self.lineEditRatioDimensions.text() is not None and self.lineEditRatioDimensions.text() != "":
            ratioDimensionElement.text = self.lineEditRatioDimensions.text()
        else:
            ratioDimensionElement.text = "1"

        if self.lineEditProcessFunctions.text() is not None and self.lineEditProcessFunctions.text() != "":
            fontProcessFunctionsElement.text = self.lineEditProcessFunctions.text()
        else:
            fontProcessFunctionsElement.text = "8"

        if self.lineEditMachineLearning.text() is not None and self.lineEditMachineLearning.text() != "":
            fontMachineLearningElement.text = self.lineEditMachineLearning.text()
        else:
            fontMachineLearningElement.text = "10"

        if self.radioButtonVirgule.isChecked():
            decimalSeparatorElement.text = ","
        else:
            decimalSeparatorElement.text = "."

        if self.lineEditSignificantDigits.text() is not None and self.lineEditSignificantDigits.text() != "":
            significantDigitsElement.text = self.lineEditSignificantDigits.text()
        else:
            significantDigitsElement.text = "3"

        if self.radioButtonCsv.isChecked():
            mergeAnalysisFormatElement.text = "csv"
        else:
            mergeAnalysisFormatElement.text = "xls"

        if self.radioButtonBatchAnalysisFormatCsv.isChecked():
            batchAnalysisFormatElement.text = "csv"
        elif self.radioButtonBatchAnalysisFormatXls.isChecked():
            batchAnalysisFormatElement.text = "xls"
        elif self.radioButtonBatchAnalysisFormatXml.isChecked():
            batchAnalysisFormatElement.text = "xml"

        if self.lineNbShapes.text() is not None and self.lineNbShapes.text() != "":
            nbShapesElement.text = self.lineNbShapes.text()
        else:
            nbShapesElement.text = "100"

        if self.checkBoxDisplay3DShapes.isChecked():
            display3DShapesElement.text = "True"
        else:
            display3DShapesElement.text = "False"

        if self.checkBoxWarningMessageExplorerClosed.isChecked():
            warningMessageExplorerClosedElement.text = "True"
        else:
            warningMessageExplorerClosedElement.text = "False"

        if self.checkBoxBackupSessions.isChecked():
            saveBackupSessionsElement.text = "True"
        else:
            saveBackupSessionsElement.text = "False"

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/Settings.mho")

        if showMessage:
            self.messageBox = wgt.MessageBox("Interface preferences saved.\n Please restart Explorer to apply modifications", '', buttons=[qt.QMessageBox.Ok],windowTitle="Interface Preferences")
            self.messageBox.exec()

        self.close()

class ColorWidget(qt.QWidget):

    def __init__(self):
        super().__init__()

        try:
            file = xmlet.parse(vrb.folderInformation + "/Settings.mho")
            self.settingsElement = file.getroot()
        except:
            self.settingsElement = xmlet.Element('Settings')
        self.xmlElement = Dfct.SubElement(self.settingsElement,"Colors")

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/Settings.mho")

        self.labelLineMeasure = qt.QLabel("Line measure color")
        self.buttonLineMeasure = qt.QPushButton("")
        self.buttonLineMeasure.setFixedSize(20*vrb.ratio,20*vrb.ratio)

        self.labelPolygon = qt.QLabel("Polygon color")
        self.buttonPolygon = qt.QPushButton("")
        self.buttonPolygon.setFixedSize(20*vrb.ratio,20*vrb.ratio)

        # self.buttonReinit = qt.QPushButton("Reinit")
        self.buttonReinit = wgt.PushButtonImage(vrb.folderImages + "/Refresh.png")
        self.buttonReinit.setFixedSize(20 * vrb.ratio, 20 * vrb.ratio)

        self.groupboxMeasureColors = qt.QGroupBox()
        self.measureColorsLayout = qt.QGridLayout()
        self.measureColorsLayout.addWidget(self.labelLineMeasure, 0,0)
        self.measureColorsLayout.addWidget(self.buttonLineMeasure, 0,1)
        self.measureColorsLayout.addWidget(self.labelPolygon, 1,0)
        self.measureColorsLayout.addWidget(self.buttonPolygon, 1,1)
        self.measureColorsLayout.addWidget(self.buttonReinit, 2,1)
        self.groupboxMeasureColors.setLayout(self.measureColorsLayout)
        self.groupboxMeasureColors.setTitle("Measure colors")
        self.groupboxMeasureColors.setStyleSheet(
            f'QGroupBox:title {{font-size: {10 * vrb.ratio}px; left: 20px;padding-left: 10px;padding-right: 10px;padding-top: {-10 * vrb.ratio}px; color:rgb(52, 152, 219)}}  QGroupBox {{font-weight: bold; border: 1px solid gray; margin-top: {10 * vrb.ratio}px }}')

        self.checkboxSwitchMeasuresMode = qt.QCheckBox("Switch mode after drawing")
        self.checkboxSwitchMeasuresMode.setToolTip(txt.dictToolTips["MeasuresSwitchMode"])
        self.checkboxSwitchMeasuresMode.setChecked(True)

        self.layout = qt.QGridLayout()
        # self.layout.addWidget(self.labelLineMeasure, 0, 0)
        # self.layout.addWidget(self.buttonLineMeasure, 0, 1)
        # self.layout.addWidget(self.labelPolygon, 1, 0)
        # self.layout.addWidget(self.buttonPolygon, 1, 1)
        # self.layout.addWidget(self.buttonReinit, 2, 1,Qt.AlignRight)
        self.layout.addWidget(self.checkboxSwitchMeasuresMode, 0, 0)
        self.layout.addWidget(self.groupboxMeasureColors, 1, 0)

        self.layout.setAlignment(Qt.AlignTop)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10,10,10,10)
        self.resize(170*vrb.ratio,120*vrb.ratio)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setWindowTitle("Measures Preferences")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonLineMeasure.clicked.connect(self.getColor)
        self.buttonPolygon.clicked.connect(self.getColor)
        self.buttonReinit.clicked.connect(self.reinit)

        self.readXmlElement()

    def getColor(self):

        # try:
        #     self.setWindowFlag(Qt.WindowStaysOnBottomHint)
        # except:
        #     self.setWindowFlags(Qt.WindowStaysOnBottomHint)
        #
        # self.show()

        self.hide()

        color = qt.QColorDialog.getColor()

        self.show()

        # try:
        #     self.setWindowFlag(Qt.WindowStaysOnTopHint)
        # except:
        #     self.setWindowFlags(Qt.WindowStaysOnTopHint)
        #
        # self.show()

        if (color.isValid()):

            if self.sender() == self.buttonLineMeasure:
                lineMeasureElement = Dfct.SubElement(self.xmlElement, "LineMeasure")
                lineMeasureElement.text = str(color.red())+","+str(color.green())+","+str(color.blue())
            if self.sender() == self.buttonPolygon:
                polygonElement = Dfct.SubElement(self.xmlElement, "Polygon")
                polygonElement.text = str(color.red())+","+str(color.green())+","+str(color.blue())

        self.readXmlElement()

    def readXmlElement(self):

        lineMeasureElement = Dfct.SubElement(self.xmlElement,"LineMeasure")
        if lineMeasureElement.text is not None:
            vrb.lineMeasureColor = lineMeasureElement.text.split(",")
        self.buttonLineMeasure.setStyleSheet('QPushButton {background-color: rgb(' + str(vrb.lineMeasureColor[0]) + ',' + str(vrb.lineMeasureColor[1]) + ',' + str(vrb.lineMeasureColor[2]) + ');}')
        polygonElement = Dfct.SubElement(self.xmlElement,"Polygon")
        if polygonElement.text is not None:
            vrb.polygonColor = polygonElement.text.split(",")
        self.buttonPolygon.setStyleSheet('QPushButton {background-color: rgb(' + str(vrb.polygonColor[0]) + ',' + str(vrb.polygonColor[1]) + ',' + str(vrb.polygonColor[2]) + ');}')

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/Settings.mho")
        if vrb.mainWindow is not None:
            vrb.mainWindow.imageViewer.getRoiImage(changeRoiImage=False)

    def reinit(self):

        lineMeasureElement = Dfct.SubElement(self.xmlElement, "LineMeasure")
        lineMeasureElement.text = "240,204,0"
        polygonElement = Dfct.SubElement(self.xmlElement, "Polygon")
        polygonElement.text = "39,98,150"
        self.readXmlElement()

class PostProcessSettingsWidget(qt.QWidget):
    """
    Interactive code editor
    Has a quick guide
    """
    def __init__(self):
        super().__init__()

        try:
            file = xmlet.parse(vrb.folderInformation + "/Settings.mho")
            self.settingsElement = file.getroot()
        except:
            self.settingsElement = xmlet.Element('Settings')
        self.xmlElement = Dfct.SubElement(self.settingsElement,"PostProcess")

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/Settings.mho")

        self.labelMerge = qt.QLabel("Merge color")
        self.buttonMerge = qt.QPushButton("")
        self.buttonMerge.setFixedSize(20*vrb.ratio,20*vrb.ratio)

        self.labelSplit2D = qt.QLabel("Split 2D color")
        self.buttonSplit2D = qt.QPushButton("")
        self.buttonSplit2D.setFixedSize(20*vrb.ratio,20*vrb.ratio)

        self.labelSplit3D = qt.QLabel("Split 3D color")
        self.buttonSplit3D = qt.QPushButton("")
        self.buttonSplit3D.setFixedSize(20*vrb.ratio,20*vrb.ratio)

        self.labelAddLabel = qt.QLabel("Add label color")
        self.buttonAddLabel = qt.QPushButton("")
        self.buttonAddLabel.setFixedSize(20*vrb.ratio,20*vrb.ratio)

        self.labelKeep = qt.QLabel("Keep label color")
        self.buttonKeep = qt.QPushButton("")
        self.buttonKeep.setFixedSize(20*vrb.ratio,20*vrb.ratio)

        self.labelDelete = qt.QLabel("Delete color")
        self.buttonDelete = qt.QPushButton("")
        self.buttonDelete.setFixedSize(20*vrb.ratio,20*vrb.ratio)

        self.labelUndo = qt.QLabel("Number of undo")
        self.lineEditUndo = qt.QLineEdit()
        self.lineEditUndo.setText("5")
        self.lineEditUndo.setAlignment(Qt.AlignRight)

        self.buttonReinit = qt.QPushButton("Reinit")

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelMerge, 0, 0)
        self.layout.addWidget(self.buttonMerge, 0, 1)
        self.layout.addWidget(self.labelSplit2D, 1, 0)
        self.layout.addWidget(self.buttonSplit2D, 1, 1)
        self.layout.addWidget(self.labelSplit3D, 2, 0)
        self.layout.addWidget(self.buttonSplit3D, 2, 1)
        self.layout.addWidget(self.labelAddLabel, 3, 0)
        self.layout.addWidget(self.buttonAddLabel, 3, 1)
        self.layout.addWidget(self.labelKeep, 4, 0)
        self.layout.addWidget(self.buttonKeep, 4, 1)
        self.layout.addWidget(self.labelDelete, 5, 0)
        self.layout.addWidget(self.buttonDelete, 5, 1)
        self.layout.addWidget(self.labelUndo, 6, 0)
        self.layout.addWidget(self.lineEditUndo, 6, 1)
        self.layout.addWidget(self.buttonReinit, 7, 1,Qt.AlignRight)

        self.layout.setAlignment(Qt.AlignTop)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10,10,10,10)
        self.resize(170*vrb.ratio,60*vrb.ratio)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setWindowTitle("Color Preference")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonMerge.clicked.connect(self.getColor)
        self.buttonSplit2D.clicked.connect(self.getColor)
        self.buttonSplit3D.clicked.connect(self.getColor)
        self.buttonAddLabel.clicked.connect(self.getColor)
        self.buttonKeep.clicked.connect(self.getColor)
        self.buttonDelete.clicked.connect(self.getColor)
        self.lineEditUndo.textChanged.connect(self.changeUndo)
        self.buttonReinit.clicked.connect(self.reinit)

        self.readXmlElement()

    def getColor(self):

        try:
            self.setWindowFlag(Qt.WindowStaysOnBottomHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnBottomHint)

        self.show()

        color = qt.QColorDialog.getColor()

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.show()

        if (color.isValid()):

            if self.sender() == self.buttonMerge:
                element = Dfct.SubElement(self.xmlElement, "Merge")
                element.text = str(color.red())+","+str(color.green())+","+str(color.blue())
            if self.sender() == self.buttonSplit2D:
                element = Dfct.SubElement(self.xmlElement, "Split2D")
                element.text = str(color.red())+","+str(color.green())+","+str(color.blue())
            if self.sender() == self.buttonSplit3D:
                element = Dfct.SubElement(self.xmlElement, "Split3D")
                element.text = str(color.red())+","+str(color.green())+","+str(color.blue())
            if self.sender() == self.buttonAddLabel:
                element = Dfct.SubElement(self.xmlElement, "AddLabel")
                element.text = str(color.red())+","+str(color.green())+","+str(color.blue())
            if self.sender() == self.buttonKeep:
                element = Dfct.SubElement(self.xmlElement, "Keep")
                element.text = str(color.red())+","+str(color.green())+","+str(color.blue())
            if self.sender() == self.buttonDelete:
                element = Dfct.SubElement(self.xmlElement, "Delete")
                element.text = str(color.red())+","+str(color.green())+","+str(color.blue())

        self.readXmlElement()

    def changeUndo(self):

        undoElement = Dfct.SubElement(self.xmlElement, "Undo")
        try:
            undoValue = int(self.lineEditUndo.text())
            undoElement.text = str(undoValue)
        except:
            undoElement.text = "5"

        self.readXmlElement()

    def readXmlElement(self):

        element = Dfct.SubElement(self.xmlElement,"Merge")
        if element.text is not None:
            vrb.mergeColor = element.text.split(",")
        self.buttonMerge.setStyleSheet('QPushButton {background-color: rgb(' + str(vrb.mergeColor[0]) + ',' + str(vrb.mergeColor[1]) + ',' + str(vrb.mergeColor[2]) + ');}')

        element = Dfct.SubElement(self.xmlElement,"Split2D")
        if element.text is not None:
            vrb.split2DColor = element.text.split(",")
        self.buttonSplit2D.setStyleSheet('QPushButton {background-color: rgb(' + str(vrb.split2DColor[0]) + ',' + str(vrb.split2DColor[1]) + ',' + str(vrb.split2DColor[2]) + ');}')

        element = Dfct.SubElement(self.xmlElement,"Split3D")
        if element.text is not None:
            vrb.split3DColor = element.text.split(",")
        self.buttonSplit3D.setStyleSheet('QPushButton {background-color: rgb(' + str(vrb.split3DColor[0]) + ',' + str(vrb.split3DColor[1]) + ',' + str(vrb.split3DColor[2]) + ');}')

        element = Dfct.SubElement(self.xmlElement,"AddLabel")
        if element.text is not None:
            vrb.addColor = element.text.split(",")
        self.buttonAddLabel.setStyleSheet('QPushButton {background-color: rgb(' + str(vrb.addColor[0]) + ',' + str(vrb.addColor[1]) + ',' + str(vrb.addColor[2]) + ');}')

        element = Dfct.SubElement(self.xmlElement,"Keep")
        if element.text is not None:
            vrb.keepColor = element.text.split(",")
        self.buttonKeep.setStyleSheet('QPushButton {background-color: rgb(' + str(vrb.keepColor[0]) + ',' + str(vrb.keepColor[1]) + ',' + str(vrb.keepColor[2]) + ');}')

        element = Dfct.SubElement(self.xmlElement,"Delete")
        if element.text is not None:
            vrb.deleteColor = element.text.split(",")
        self.buttonDelete.setStyleSheet('QPushButton {background-color: rgb(' + str(vrb.deleteColor[0]) + ',' + str(vrb.deleteColor[1]) + ',' + str(vrb.deleteColor[2]) + ');}')

        element = Dfct.SubElement(self.xmlElement, "Undo")
        try:
            vrb.numberPreviousImages = int(element.text)
            self.lineEditUndo.setText(element.text)
        except:
            vrb.numberPreviousImages = 5
            self.lineEditUndo.setText("5")

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/Settings.mho")
        if vrb.mainWindow is not None:
            vrb.mainWindow.imageViewer.getRoiImage(changeRoiImage=False)

    def reinit(self):

        Dfct.SubElement(self.xmlElement, "Merge").text = "0,0,220"
        Dfct.SubElement(self.xmlElement, "Split2D").text = "30,30,30"
        Dfct.SubElement(self.xmlElement, "Split3D").text = "0,220,0"
        Dfct.SubElement(self.xmlElement, "AddLabel").text = "178,115,0"
        Dfct.SubElement(self.xmlElement, "KeepLabel").text = "0,220,0"
        Dfct.SubElement(self.xmlElement, "Delete").text = "220,0,0"
        Dfct.SubElement(self.xmlElement, "Undo").text = "5"
        self.readXmlElement()

class calibrationSettingsWidget(qt.QWidget):

    def __init__(self):
        super().__init__()

        try:
            file = xmlet.parse(vrb.folderInformation + "/CalibrationSettings.mho")
            self.settingsElement = file.getroot()
        except:
            self.settingsElement = xmlet.Element('CalibrationSettings')

        self.image = None
        self.isPreviewed = False

        self.colorElement = Dfct.SubElement(self.settingsElement, "Colors")
        self.positionElement = Dfct.SubElement(self.settingsElement, "Position")
        self.sizePercentageElement = Dfct.SubElement(self.settingsElement, "SizePercentage")
        self.textPositionElement = Dfct.SubElement(self.settingsElement, "TextPosition")
        self.textWidthElement = Dfct.SubElement(self.settingsElement, "TextWidth")
        self.transparentBackgroundElement = Dfct.SubElement(self.settingsElement, "TransparentBackground")

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/CalibrationSettings.mho")


        self.labelPosition = qt.QLabel("Position")
        self.labelPosition.setFixedSize(80*vrb.ratio,30*vrb.ratio)
        self.comboBoxPosition = qt.QComboBox()
        self.comboBoxPosition.setFixedSize(90 * vrb.ratio, 20 * vrb.ratio)

        self.labelSizePercentage = qt.QLabel("Size (%)")
        self.labelSizePercentage.setFixedSize(80*vrb.ratio,30*vrb.ratio)
        self.lineEditSizePercentage = qt.QLineEdit()
        self.lineEditSizePercentage.setText(str(50))
        self.lineEditSizePercentage.setAlignment(Qt.AlignRight)
        self.lineEditSizePercentage.setFixedSize(50*vrb.ratio, 20*vrb.ratio)

        self.labelTextColor = qt.QLabel("Text color")
        self.labelTextColor.setFixedSize(80*vrb.ratio, 20*vrb.ratio)
        self.buttonTextColor = qt.QPushButton("")
        self.buttonTextColor.setFixedSize(40*vrb.ratio,20*vrb.ratio)

        self.labelBackgroundColor = qt.QLabel("Background color")
        self.labelBackgroundColor.setFixedSize(80 * vrb.ratio, 20 * vrb.ratio)
        if self.transparentBackgroundElement.text == "True":
            self.labelBackgroundColor.setEnabled(False)
            self.labelBackgroundColor.setStyleSheet("QLabel { color : grey; }")

        # self.buttonBackgroundColor = qt.QPushButton("")
        # self.buttonBackgroundColor.setFixedSize(40 * vrb.ratio, 20 * vrb.ratio)
        # self.checkBoxTransparentBackground = qt.QCheckBox("Transparent")

        self.buttonBackgroundColor = qt.QPushButton("")
        self.buttonBackgroundColor.setFixedSize(40 * vrb.ratio, 20 * vrb.ratio)
        self.checkBoxBackgroundColor = qt.QCheckBox("Background color")
        self.checkBoxBackgroundColor.setChecked(True)

        # groupBoxBackgroundColor = qt.QGroupBox()
        # layoutBackgroundColor = qt.QGridLayout()
        # layoutBackgroundColor.setContentsMargins(10, 10, 10, 20)
        # layoutBackgroundColor.addWidget(self.labelBackgroundColor, 0, 0)
        # layoutBackgroundColor.addWidget(self.buttonBackgroundColor, 0, 1)
        # layoutBackgroundColor.addWidget(self.checkBoxTransparentBackground, 0, 2, Qt.AlignRight)
        # groupBoxBackgroundColor.setLayout(layoutBackgroundColor)
        # groupBoxBackgroundColor.setFixedWidth(300 * vrb.ratio)

        # self.labelTextFont = qt.QLabel("Font")
        # self.labelTextFont.setFixedSize(80 * vrb.ratio, 20 * vrb.ratio)

        self.labelTextPosition = qt.QLabel("Text position")
        self.labelTextPosition.setFixedSize(80 * vrb.ratio, 20 * vrb.ratio)
        self.comboBoxTextPosition = qt.QComboBox()
        self.comboBoxTextPosition.setFixedSize(90 * vrb.ratio, 20 * vrb.ratio)

        self.labelTextWidth = qt.QLabel("Text width")
        self.labelTextWidth.setFixedSize(80 * vrb.ratio, 20 * vrb.ratio)
        self.comboBoxTextWidth = qt.QComboBox()
        self.comboBoxTextWidth.setFixedSize(90 * vrb.ratio, 20 * vrb.ratio)

        self.buttonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=0)
        self.buttonValidate.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        self.buttonValidate.setToolTip("Validate")

        self.buttonPreview = wgt.SwitchingLabel(wgt.LabelImage(vrb.folderImages + '/eye_close.png'), wgt.LabelImage(vrb.folderImages + '/eye_open.png'))
        self.buttonPreview.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.buttonPreview.setToolTip("Preview")

        self.buttonRefresh = wgt.PushButtonImage(vrb.folderImages + "/Refresh.png", margins=0)
        self.buttonRefresh.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.buttonRefresh.setToolTip("Refresh")
        self.buttonRefresh.hide()

        groupBoxDisplaySettings = qt.QGroupBox()
        layoutDisplaySettings = qt.QGridLayout()
        layoutDisplaySettings.setContentsMargins(10, 10, 10, 10)
        layoutDisplaySettings.addWidget(self.labelPosition, 0, 0)
        layoutDisplaySettings.addWidget(self.comboBoxPosition, 0, 1)
        layoutDisplaySettings.addWidget(self.labelSizePercentage, 1, 0)
        layoutDisplaySettings.addWidget(self.lineEditSizePercentage, 1, 1, Qt.AlignRight)
        layoutDisplaySettings.addWidget(self.labelTextPosition, 2, 0)
        layoutDisplaySettings.addWidget(self.comboBoxTextPosition, 2, 1)
        layoutDisplaySettings.addWidget(self.labelTextWidth, 3, 0)
        layoutDisplaySettings.addWidget(self.comboBoxTextWidth, 3, 1)
        groupBoxDisplaySettings.setLayout(layoutDisplaySettings)
        #groupBoxDisplaySettings.setFixedSize(220 * vrb.ratio, 180 * vrb.ratio)
        groupBoxDisplaySettings.setTitle("Display settings :")
        groupBoxDisplaySettings.setStyleSheet(
            'QGroupBox:title {left: 20px ;padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {font: bold; border: 1px solid gray; margin-top: 12 px}')


        groupBoxColorSettings = qt.QGroupBox()
        layoutColorSettings = qt.QGridLayout()
        layoutColorSettings.setContentsMargins(10, 10, 10, 10)
        layoutColorSettings.addWidget(self.labelTextColor, 0, 0)
        layoutColorSettings.addWidget(self.buttonTextColor, 0, 1)
        layoutColorSettings.addWidget(self.checkBoxBackgroundColor, 1, 0)
        layoutColorSettings.addWidget(self.buttonBackgroundColor, 1, 1)
        groupBoxColorSettings.setLayout(layoutColorSettings)
        #groupBoxColorSettings.setFixedSize(220 * vrb.ratio, 150 * vrb.ratio)
        groupBoxColorSettings.setTitle("Colors :")
        groupBoxColorSettings.setStyleSheet(
            'QGroupBox:title {left: 20px ;padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {font: bold; border: 1px solid gray; margin-top: 12 px}')


        groupBoxValidateButtons = qt.QGroupBox()
        # layoutValidateButtons = qt.QGridLayout()
        # layoutValidateButtons.setContentsMargins(0, 0, 0, 0)
        # layoutValidateButtons.addWidget(self.buttonRefresh, 0, 0, Qt.AlignRight)
        # layoutValidateButtons.addWidget(self.buttonPreview, 0, 1, Qt.AlignRight)
        # layoutValidateButtons.addWidget(self.buttonValidate, 0, 2)
        # groupBoxValidateButtons.setLayout(layoutValidateButtons)
        # groupBoxValidateButtons.setFixedWidth(220 * vrb.ratio)

        layoutValidateButtons = qt.QHBoxLayout()
        layoutValidateButtons.setContentsMargins(0, 0, 0, 0)
        layoutValidateButtons.addStretch()
        layoutValidateButtons.addWidget(self.buttonRefresh)
        layoutValidateButtons.addWidget(self.buttonPreview)
        layoutValidateButtons.addWidget(self.buttonValidate)
        groupBoxValidateButtons.setLayout(layoutValidateButtons)
        #groupBoxValidateButtons.setFixedWidth(220 * vrb.ratio)

        self.layout = qt.QVBoxLayout()
        self.layout.addWidget(groupBoxDisplaySettings)
        self.layout.addWidget(groupBoxColorSettings)
        self.layout.addWidget(groupBoxValidateButtons)

        self.layout.setAlignment(Qt.AlignTop)

        self.setLayout(self.layout)
        self.layout.setSizeConstraint(1)
        # self.layout.setContentsMargins(10,10,10,10)
        # self.resize(240*vrb.ratio,380*vrb.ratio)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setWindowTitle("Calibration Display Settings")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.buttonValidate.clicked.connect(self.validate)
        self.checkBoxBackgroundColor.stateChanged.connect(self.setTransparent)
        #self.buttonPreview.clicked.connect(self.validate)
        self.buttonPreview.SignalWidgetSwitched.connect(self.validate)
        self.buttonTextColor.clicked.connect(self.getColor)
        self.buttonBackgroundColor.clicked.connect(self.getColor)
        self.buttonRefresh.clicked.connect(self.validate)

        self.readXmlColorElement()


    def closeEvent(self, event):

        vrb.mainWindow.changeCurrentXmlElement(vrb.mainWindow.currentLabel)
        self.close()
        
    def validate(self):

        vrb.mainWindow.toggleGroupBoxProcessing(True)

        # show and hides buttons depending if preview mode is on
        if self.sender()==self.buttonPreview:
            if self.isPreviewed:
                self.buttonRefresh.hide()
                self.isPreviewed = False
            else:
                self.buttonRefresh.show()
                self.isPreviewed = True


        hasOverlay = vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.isChecked()
        vrb.mainWindow.changeCurrentXmlElement(vrb.mainWindow.currentLabel)

        if hasOverlay:
            vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.setChecked(True)

        #get current image
        try:
            currentImg = vrb.mainWindow.currentLabel.image
            self.image=currentImg
            currentName = vrb.mainWindow.currentLabel.name
        except:
            self.messageBox = wgt.MessageBox(
                "No image to insert the calibration.", '',
                buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Erreur")
            self.messageBox.exec()

        self.applyCalibrationImg(currentImg)

        vrb.mainWindow.toggleGroupBoxProcessing(False)


    def applyCalibrationImg(self, inImg):
        try:
            # # copy and conversion from ipsdk image to PIL image
            # img = util.copyImg(inImg)

            if inImg.getBufferType() in [PyIPSDK.eImageBufferType.eIBT_Label8,
                                         PyIPSDK.eImageBufferType.eIBT_Label16,
                                         PyIPSDK.eImageBufferType.eIBT_Label32]:
                img = fct.lutToColor(inImg, vrb.mainWindow.currentLabel.groupBoxLut)
            else:
                img=inImg

            # set overlay image
            if vrb.mainWindow.widgetImage.imageOverlayLabel is not None and vrb.mainWindow.widgetImage.imageOverlayLabel.image is not None:
                imageOverlay = vrb.mainWindow.widgetImage.imageOverlayLabel.image
            else:
                imageOverlay = None

            if imageOverlay is not None:
                img = vrb.mainWindow.widgetImage.transformImage(img, imageOverlay, None, None, None, extractImage=True)
            else:
                if img.getColorGeometryType() == PyIPSDK.eCGT_Grey:
                    img = vrb.mainWindow.widgetImage.transformImage(img, img, None, None, None, extractImage=True)

            # retrieve calibration
            if inImg.hasGeometricCalibration():
                self.calibration = inImg.getGeometricCalibration()
            else:
                self.calibration = PyIPSDK.createGeometricCalibration2d(1, 1, 'px')

            textColor = (int(vrb.calibrationTextColor[0]), int(vrb.calibrationTextColor[1]), int(vrb.calibrationTextColor[2]), 255)
            backgroundColor=(int(vrb.calibrationBackgroundColor[0]),int(vrb.calibrationBackgroundColor[1]),int(vrb.calibrationBackgroundColor[2]),255)

            # creates an image with inlaid calibration
            ipsdkImg = fct.calibrationIncrustationImg(img,self.calibration.getXScale(), self.calibration.getUnitStr(), self.comboBoxPosition.currentText(),
                                                      int(self.lineEditSizePercentage.text()), self.comboBoxTextPosition.currentText(),
                                                      self.comboBoxTextWidth.currentText(), textColor, backgroundColor,
                                                      self.checkBoxBackgroundColor.isChecked())

            # ui.displayImg(ipsdkImg, "output image ipsdk")

            if self.sender()==self.buttonPreview:
                if self.isPreviewed:
                    vrb.mainWindow.widgetImage.setImage(ipsdkImg)
                else:
                    img = vrb.mainWindow.widgetImage.transformImage(inImg, inImg, None, None, None,
                                                                    extractImage=True)
                    vrb.mainWindow.widgetImage.setImage(img)
                    # inImg = vrb.mainWindow.widgetImage.transformImage(inImg, inImg, None, None, None,
                    #                                                 extractImage=True)
                    # vrb.mainWindow.widgetImage.setImage(inImg)
            elif self.sender() == self.buttonRefresh:
                vrb.mainWindow.widgetImage.setImage(ipsdkImg)
            else:
                self.buttonPreview.clickedWidget()
                outputDict = {}
                outputDict["Name"] = ""
                outputDict["Type"] = "Image"

                functionXmlElement = xmlet.Element('FunctionCall')
                Dfct.SubElement(functionXmlElement, 'Name').text = 'GenerateImageCalibration'
                paramsNode = Dfct.SubElement(functionXmlElement, 'Parameters')

                paramNode0 = Dfct.SubElement(paramsNode, 'Parameter_0')
                typeNode = Dfct.SubElement(paramNode0, 'Type')
                typeNode.text = "Image"
                valueNode = Dfct.SubElement(paramNode0, 'Value')
                valueNode.set('isElementID', str(True))
                valueNode.text = Dfct.childText(vrb.mainWindow.currentLabel.xmlElement, 'ElementID')

                paramNode1 = Dfct.SubElement(paramsNode, 'Parameter_1')
                Dfct.SubElement(paramNode1, 'Type').text = "String"
                Dfct.SubElement(paramNode1, 'Value').text = str(self.calibration.getXScale())

                paramNode2 = Dfct.SubElement(paramsNode, 'Parameter_2')
                Dfct.SubElement(paramNode2, 'Type').text = "String"
                Dfct.SubElement(paramNode2, 'Value').text = str(self.calibration.getUnitStr())

                paramNode3 = Dfct.SubElement(paramsNode, 'Parameter_3')
                Dfct.SubElement(paramNode3, 'Type').text = "String"
                Dfct.SubElement(paramNode3, 'Value').text = str(self.comboBoxPosition.currentText())

                paramNode4 = Dfct.SubElement(paramsNode, 'Parameter_4')
                Dfct.SubElement(paramNode4, 'Type').text = "String"
                Dfct.SubElement(paramNode4, 'Value').text = str(int(self.lineEditSizePercentage.text()))

                paramNode5 = Dfct.SubElement(paramsNode, 'Parameter_5')
                Dfct.SubElement(paramNode5, 'Type').text = "String"
                Dfct.SubElement(paramNode5, 'Value').text = str(self.comboBoxTextPosition.currentText())

                paramNode6 = Dfct.SubElement(paramsNode, 'Parameter_6')
                Dfct.SubElement(paramNode6, 'Type').text = "String"
                Dfct.SubElement(paramNode6, 'Value').text = str(self.comboBoxTextWidth.currentText())

                paramNode7 = Dfct.SubElement(paramsNode, 'Parameter_7')
                Dfct.SubElement(paramNode7, 'Type').text = "String"
                Dfct.SubElement(paramNode7, 'Value').text = str(textColor)

                paramNode8 = Dfct.SubElement(paramsNode, 'Parameter_8')
                Dfct.SubElement(paramNode8, 'Type').text = "String"
                Dfct.SubElement(paramNode8, 'Value').text = str(backgroundColor)

                paramNode9 = Dfct.SubElement(paramsNode, 'Parameter_9')
                Dfct.SubElement(paramNode9, 'Type').text = "String"
                Dfct.SubElement(paramNode9, 'Value').text = str(self.checkBoxBackgroundColor.isChecked())

                label, realOutputName = vrb.mainWindow.addResult(ipsdkImg, outputDict, functionXmlElement)

                xmlElement = label.xmlElement
                Dfct.SubElement(xmlElement, 'AssociatedID').text = Dfct.childText(label.xmlElement, 'ElementID')

            self.writeXmlElement()

        except:
            import traceback
            traceback.print_exc(file=sys.stderr)
            self.close()


    def writeXmlElement(self):

        self.positionElement.text = str(self.comboBoxPosition.currentIndex())
        self.sizePercentageElement.text = self.lineEditSizePercentage.text()
        self.textPositionElement.text = str(self.comboBoxTextPosition.currentIndex())
        self.textWidthElement.text = str(self.comboBoxTextWidth.currentIndex())
        self.transparentBackgroundElement.text = str(self.checkBoxBackgroundColor.isChecked())

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/CalibrationSettings.mho")

    def readXmlColorElement(self):
        textColorElement = Dfct.SubElement(self.colorElement, "TextColor")
        if textColorElement.text is not None:
            vrb.calibrationTextColor = textColorElement.text.split(",")
        self.buttonTextColor.setStyleSheet(
            'QPushButton {background-color: rgb(' + str(vrb.calibrationTextColor[0]) + ',' + str(
                vrb.calibrationTextColor[1]) + ',' + str(vrb.calibrationTextColor[2]) + ');}')
        backgroundColorElement = Dfct.SubElement(self.colorElement, "BackgroundColor")
        if backgroundColorElement.text is not None:
            vrb.calibrationBackgroundColor = backgroundColorElement.text.split(",")
        self.buttonBackgroundColor.setStyleSheet('QPushButton {background-color: rgb(' + str(vrb.calibrationBackgroundColor[0]) + ',' + str(
            vrb.calibrationBackgroundColor[1]) + ',' + str(vrb.calibrationBackgroundColor[2]) + ');}')

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/CalibrationSettings.mho")
        if vrb.mainWindow is not None:
            vrb.mainWindow.imageViewer.getRoiImage(changeRoiImage=False)

    def readXmlElement(self):

        if self.positionElement.text is not None:
            self.comboBoxPosition.setCurrentIndex(int(self.positionElement.text))
        if self.sizePercentageElement.text is not None:
            self.lineEditSizePercentage.setText(self.sizePercentageElement.text)
        if self.textPositionElement.text is not None:
            self.comboBoxTextPosition.setCurrentIndex(int(self.textPositionElement.text))
        if self.textWidthElement.text is not None:
            self.comboBoxTextWidth.setCurrentIndex(int(self.textWidthElement.text))
        if self.transparentBackgroundElement.text is not None:
            if self.transparentBackgroundElement.text == "False":
                self.checkBoxBackgroundColor.setChecked(False)
            else:
                self.checkBoxBackgroundColor.setChecked(True)

        Dfct.saveXmlElement(self.settingsElement, vrb.folderInformation + "/CalibrationSettings.mho")

    def setTransparent(self):
        if(self.checkBoxBackgroundColor.isChecked()):
            self.buttonBackgroundColor.setEnabled(True)
            self.checkBoxBackgroundColor.setStyleSheet("QCheckBox { color : white; }")
        else:
            self.buttonBackgroundColor.setEnabled(False)
            self.checkBoxBackgroundColor.setStyleSheet("QCheckBox { color : grey; }")
        self.update()

    def loadComboboxTypes(self):
        self.comboBoxPosition.clear()
        self.comboBoxPosition.addItem("Top left")
        self.comboBoxPosition.addItem("Top right")
        self.comboBoxPosition.addItem("Bottom left")
        self.comboBoxPosition.addItem("Bottom right")
        self.comboBoxPosition.setCurrentIndex(0)

        self.comboBoxTextWidth.clear()
        self.comboBoxTextWidth.addItem("Large")
        self.comboBoxTextWidth.addItem("Medium")
        self.comboBoxTextWidth.addItem("Small")
        self.comboBoxTextWidth.setCurrentIndex(1)

        self.comboBoxTextPosition.clear()
        self.comboBoxTextPosition.addItem("Up")
        self.comboBoxTextPosition.addItem("Left")
        self.comboBoxTextPosition.addItem("Right")
        self.comboBoxTextPosition.addItem("Bottom")
        self.comboBoxTextPosition.setCurrentIndex(0)

        self.readXmlElement()

    def getColor(self):

        self.setVisible(False)

        color = qt.QColorDialog.getColor()

        self.setVisible(True)

        if (color.isValid()):
            if self.sender() == self.buttonTextColor:
                textColorElement = Dfct.SubElement(self.colorElement, "TextColor")
                textColorElement.text = str(color.red())+","+str(color.green())+","+str(color.blue())
            if self.sender() == self.buttonBackgroundColor:
                backgroundColorElement = Dfct.SubElement(self.colorElement, "BackgroundColor")
                backgroundColorElement.text = str(color.red())+","+str(color.green())+","+str(color.blue())

        self.readXmlColorElement()

if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    # foo = LineEditIf()
    # foo.layout.setContentsMargins(5, 5, 5, 5)

    # foo = Folder3DWidget()
    # foo = InterfacePreferencesWidget()
    # foo = ZStackFocusWidget()
    # import UsefullFunctions as fct
    # style = fct.getStyleSheet()
    # foo.setStyleSheet(style)

    # foo = DiskImagesPreferencesWidget()
    foo = MovieMakerPreferencesWidget()

    foo.show()

    app.exec_()