import sys
import os
import webbrowser

import PyQt5.QtCore as QtCore
from PyQt5.QtCore import pyqtSignal, Qt, QCoreApplication
import PyQt5.QtWidgets as qt
from PyQt5 import QtGui

import WidgetTypes
from UsefullWidgets import NumberLineEdit,PushButtonImage
import UsefullVariables as vrb
import UsefullTexts as txt

import DatabaseFunction as Dfct
import xml.etree.ElementTree as xmlet


class SelectorWidget(qt.QGroupBox):
    # [self] [True if start of selection (False if end)] [0 if cancel, 1 if validate]
    SignalSelector = pyqtSignal(qt.QGroupBox, bool, int)

    def __init__(self, xmlElement, type):
        if type != WidgetTypes.InputType.SELECTOR:
            print('ERROR : InputType does not match', file=sys.stderr)
        super().__init__()
        self.type = type
        self.xmlElement = xmlElement

        self.selectorNode = Dfct.SubElement(xmlElement, 'Selector')

        self.layout = qt.QGridLayout()

        self.stackedWidget = qt.QStackedWidget()

        self.groupBoxSelector = WidgetTypes.SelectorType.selectorWidget(Dfct.childText(self.selectorNode, 'SelectorType'))(self.selectorNode)
        self.groupBoxText = GroupBoxText(Dfct.childText(self.selectorNode, 'Text'))

        self.stackedWidget.addWidget(self.groupBoxSelector)
        self.stackedWidget.addWidget(self.groupBoxText)

        self.layout.addWidget(self.stackedWidget)
        self.setLayout(self.layout)
        self.layout.setSizeConstraint(1)

        self.groupBoxSelector.pushButtonSelectionRequest.clicked.connect(self.selectionClicked)
        self.groupBoxText.pushButtonCancel.clicked.connect(self.cancelClicked)
        self.groupBoxText.pushButtonValidate.clicked.connect(self.validateClicked)
        self.setMaximumHeight(100*vrb.ratio)

    def selectionClicked(self):
        self.SignalSelector.emit(self, True, 2)
        self.stackedWidget.setCurrentWidget(self.groupBoxText)

    def cancelClicked(self):
        self.SignalSelector.emit(self, False, 0)
        self.stackedWidget.setCurrentWidget(self.groupBoxSelector)

    def validateClicked(self):
        self.SignalSelector.emit(self, False, 1)
        self.stackedWidget.setCurrentWidget(self.groupBoxSelector)

    def interfaceToXml(self, number):
        paramNode = xmlet.Element('Parameter_' + str(number))
        Dfct.SubElement(paramNode, 'Type').text = WidgetTypes.InputType.SELECTOR.value
        Dfct.SubElement(paramNode, 'SelectorType').text = Dfct.childText(self.groupBoxSelector.xmlSelector, 'SelectorType')
        valueNode = self.groupBoxSelector.interfaceToXml()
        paramNode.append(valueNode)
        return paramNode

    @staticmethod
    def xmlToValue(xmlElement):
        valueParam = Dfct.SubElement(xmlElement, 'Value')
        selectorType = Dfct.childText(xmlElement, 'SelectorType')
        value = WidgetTypes.SelectorType.selectorWidget(selectorType).xmlToValue(valueParam)
        return value


class GroupBoxText(qt.QGroupBox):
    def __init__(self, text):
        super(GroupBoxText, self).__init__()

        self.labelText = qt.QLabel(text)
        self.labelText.setWordWrap(True)
        self.pushButtonCancel = qt.QPushButton('Cancel')
        self.pushButtonValidate = qt.QPushButton('Validate')

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.labelText, 0, 0, 1, 2)
        self.layout.addWidget(self.pushButtonCancel, 1, 0)
        self.layout.addWidget(self.pushButtonValidate, 1, 1)
        self.setLayout(self.layout)


class GenericSelectorWidget(qt.QGroupBox):
    def __init__(self, xmlSelector):
        super(GenericSelectorWidget, self).__init__()
        self.xmlSelector = xmlSelector

        #self.pushButtonSelectionRequest = qt.QPushButton('↗')  # ⇗

        self.pushButtonSelectionRequest = PushButtonImage(vrb.folderImages + "/Edit_3.png", margins=2)
        self.pushButtonSelectionRequest.setFixedSize(20*vrb.ratio, 20*vrb.ratio)
        self.pushButtonSelectionRequest.setToolTip(txt.dictToolTips["SelectThreshold"])

        self.layout = qt.QGridLayout()

        layout = qt.QGridLayout()
        layout.addLayout(self.layout, 0, 0)
        layout.addWidget(self.pushButtonSelectionRequest, 0, 1)
        self.setLayout(layout)

    def interfaceToXml(self):
        pass
        # print('ERROR: Must be redefined in subclass')

    @staticmethod
    def xmlToValue(xmlValue):
        pass
        # print('ERROR: Must be redefined in subclass')


class RangeSelectorWidget(GenericSelectorWidget):
    def __init__(self, xmlSelector):
        super(RangeSelectorWidget, self).__init__(xmlSelector)

        self.selectorType = "Range"

        # self.labelMin = qt.QLabel('Min')
        self.checkBoxMin = qt.QCheckBox('Min')
        self.checkBoxMin.setChecked(True)
        self.lineEditMin = NumberLineEdit(constraint=WidgetTypes.ScalarConstraint.REAL.value)
        self.lineEditMin.setText('127')

        # self.labelMax = qt.QLabel('Max')
        self.checkBoxMax = qt.QCheckBox('Max')
        self.checkBoxMax.setChecked(True)
        self.lineEditMax = NumberLineEdit(constraint=WidgetTypes.ScalarConstraint.REAL.value)
        self.lineEditMax.setText('255')

        self.layout.addWidget(self.checkBoxMin, 0, 0)
        self.layout.addWidget(self.lineEditMin, 0, 1)
        self.layout.addWidget(self.checkBoxMax, 1, 0)
        self.layout.addWidget(self.lineEditMax, 1, 1)

        self.checkBoxMin.stateChanged.connect(self.checkBoxMinChanged)
        self.checkBoxMax.stateChanged.connect(self.checkBoxMaxChanged)

    def checkBoxMinChanged(self):

        self.lineEditMin.setEnabled(self.checkBoxMin.isChecked())
        if self.checkBoxMin.isChecked() == False:
            self.checkBoxMax.setChecked(True)

    def checkBoxMaxChanged(self):
        self.lineEditMax.setEnabled(self.checkBoxMax.isChecked())
        if self.checkBoxMax.isChecked() == False:
            self.checkBoxMin.setChecked(True)

    def interfaceToXml(self):
        valueNode = xmlet.Element('Value')

        if self.checkBoxMin.isChecked():
            textMin = self.lineEditMin.text()
        else:
            textMin = ""
        if self.checkBoxMax.isChecked():
            textMax = self.lineEditMax.text()
        else:
            textMax = ""

        valueNode.text = textMin + '_' + textMax

        return valueNode

    @staticmethod
    def xmlToValue(xmlValue):
        textSplit = xmlValue.text.split('_')
        res = []
        for i in range(len(textSplit)):
            if textSplit[i] == '':
                res.append(None)
            else:
                res.append(float(textSplit[i]))

        return res

class HysteresisSelectorWidget(GenericSelectorWidget):
    def __init__(self, xmlSelector):
        super(HysteresisSelectorWidget, self).__init__(xmlSelector)

        self.selectorType = "Hysteresis"

        self.comboBoxType = qt.QComboBox()
        self.comboBoxType.addItem("Light")
        self.comboBoxType.addItem("Dark")
        self.comboBoxType.setFixedHeight(20*vrb.ratio)

        self.lineEditMin = NumberLineEdit(constraint=WidgetTypes.ScalarConstraint.REAL.value)
        self.lineEditMin.setText('100')

        self.lineEditMax = NumberLineEdit(constraint=WidgetTypes.ScalarConstraint.REAL.value)
        self.lineEditMax.setText('200')

        self.layout.addWidget(self.comboBoxType, 0,0)
        self.layout.addWidget(self.lineEditMin, 1,0)
        self.layout.addWidget(self.lineEditMax, 2, 0)

    def interfaceToXml(self):
        valueNode = xmlet.Element('Value')

        textMin = self.lineEditMin.text()
        textMax = self.lineEditMax.text()

        valueNode.text = self.comboBoxType.currentText() + '_' + textMin + '_' + textMax

        return valueNode

    @staticmethod
    def xmlToValue(xmlValue):
        textSplit = xmlValue.text.split('_')
        res = []
        for i in range(len(textSplit)):
            if i == 0:
                res.append(textSplit[i])
            else:
                if textSplit[i] == '':
                    res.append(None)
                else:
                    res.append(float(textSplit[i]))

        return res


class SimpleRangeSelectorWidget(GenericSelectorWidget):
    def __init__(self, xmlSelector):
        super(SimpleRangeSelectorWidget, self).__init__(xmlSelector)

        self.selectorType = "Simple"

        self.lineEdit = NumberLineEdit(constraint=WidgetTypes.ScalarConstraint.REAL.value)
        self.lineEdit.setText('3')

        self.layout.addWidget(self.lineEdit, 0, 0)


    def interfaceToXml(self):
        valueNode = xmlet.Element('Value')
        valueNode.text = str(self.lineEdit.text())
        return valueNode

    @staticmethod
    def xmlToValue(xmlValue):

        text = xmlValue.text
        if text == '':
            res = 3
        else:
            res = float(text)

        return res


if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])
    sys._excepthook = sys.excepthook

    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)
    sys.excepthook = exception_hook

    xmlElement = xmlet.fromstring('<Parameter_0><Name>Threshold</Name><Type>Selector</Type><Constraints/><Default/><Mandatory>True</Mandatory><Selector><Text>Please select the threshold range then click on validate.</Text><SelectorType>RangeSelector</SelectorType></Selector></Parameter_0>')
    foo = SelectorWidget(xmlElement, WidgetTypes.InputType.SELECTOR)


    foo.show()

    app.exec_()