import sys, os, traceback
import re as regex
import xml.etree.ElementTree as xmlet

from PyQt5 import QtWidgets as qt, QtGui
from PyQt5.QtCore import pyqtSignal, Qt, QCoreApplication

import DatabaseFunction as Dfct

from FunctionsDictionary import dictAllFunctions
import UsefullVariables as vrb
import UsefullWidgets as wgt
import UsefullFunctions as fct
import UsefullTexts as txt
import WidgetTypes

import ElementIDManager as Eidm

from TextEditor import TextEditor
from HighlightTextEditor import PythonHighlighter

from MacroWidgets import EditScriptWindow

def scriptFromXmlElement(xmlElement):
    try:
        script = '###import PyIPSDK\n'
        functionNode = Dfct.SubElement(xmlElement, 'FunctionCall')
        functionName = Dfct.childText(functionNode, 'Name')
        associatedIDs = Dfct.childText(xmlElement, 'AssociatedID')
        assoSplit = associatedIDs.split(',')
        if len(assoSplit) == 1:
            imageID = "ID_" + xmlElement.tag + "_" + Dfct.childText(xmlElement, "ElementID") + "_ID"
            script += dictAllFunctions[functionName](functionNode, process=False, outImage=imageID)
        else:
            types = []
            outputsElement = Dfct.SubElement(functionNode, 'Outputs')
            for i in range(len(assoSplit)):
                outputElement = Dfct.SubElement(outputsElement, "Parameter_" + str(i))
                #outputElement = Dfct.SubElement(outputsElement, "Output_" + str(i))
                types.append(Dfct.childText(outputElement, "Type"))
            imageID = []
            for i in range(len(assoSplit)):
                imageID.append("ID_" + types[i] + "_" + assoSplit[i] + "_ID")
            script += dictAllFunctions[functionName](functionNode, process=False, outImage=imageID)
    except Exception as e:
        traceback.print_exc(file=sys.stderr)
        vrb.printDebug('ERROR: could not generate script')
        vrb.printDebug(e)
        script = 'ERROR: could not generate script'
    return script


def fillDictWithElement(dictScript, listKeys, xmlElement, xmlAllElements, isCompleteScript=True):
    """
    Fill dictScript recursively, adding xmlElement and all its parents

    :param dictScript: dictionary associating ID -> xmlElement
    :param xmlElement: element to add, and its parents
    :param xmlAllElements:
    :param isCompleteScript: if true: When an element is added in the dict, all its associated node are added with None (So we know they are siblings)
                             else: the associated nodes are added normally (add the xmlElement)
    """
    elementID = Dfct.childText(xmlElement, 'ElementID')

    addElement(listKeys, elementID)

    if elementID not in dictScript:
        dictScript[elementID] = xmlElement
        associatedID = Dfct.SubElement(xmlElement, "AssociatedID").text
        if associatedID is not None and associatedID != "":
            assoSplit = associatedID.split(',')
            for id in assoSplit:
                if id != elementID:
                    if isCompleteScript:
                        dictScript[id] = None
                    else:
                        dictScript[id] = Eidm.getElementWithID(xmlAllElements, id)

    parentIds = Eidm.findParentIDs(xmlElement)
    for parentId in parentIds:
        parentId = parentId.text
        if parentId is not None:
            parentXmlElement = Eidm.getElementWithID(xmlAllElements, parentId)
            fillDictWithElement(dictScript, listKeys, parentXmlElement, xmlAllElements)

def addElement(liste,element):
    if element in liste:
        liste.remove(element)
    liste.insert(0, element)

def nameToScriptName(name):
    name = name.replace(' ', '_')
    finalName = ''
    for c in name:
        if regex.match('[a-zA-Z0-9]', c) or c == "_":
            finalName += c
    if regex.match('[0-9]', finalName[0]):
        finalName = '_' + finalName
    return finalName

def functionScript(xmlElement, xmlAllElements):
    dictScript = {}
    listKeys = []
    fillDictWithElement(dictScript, listKeys, xmlElement, xmlAllElements, isCompleteScript=False)
    script = scriptFromXmlElement(xmlElement)

    for key in dictScript:
        xmlElement = dictScript[key]
        if xmlElement is not None:
            scriptName = 'ID_' + xmlElement.tag + '_' + str(key) + '_ID'
            script = script.replace(scriptName, nameToScriptName(Dfct.childText(xmlElement, 'Name')))
        else:
            # In case they don't have an xmlElement anymore (siblings were deleted and were useless)
            script = regex.sub('ID_.*_' + str(key) + '_ID', 'deletedOutput', script)

    scriptBody, scriptImports = separateImports(script)
    return scriptImports + '\n' + scriptBody


def completeScript(xmlElement, xmlAllElements):

    dictScript = {}
    keys = []
    fillDictWithElement(dictScript, keys, xmlElement,xmlAllElements)

    scriptTitle = "def My_function( "
    scriptBody = ""
    inputImages = []
    for key in keys:
        if key in dictScript and dictScript[key] is not None:
            functionName = Dfct.childText(Dfct.SubElement(dictScript[key], 'FunctionCall'), "Name")
            if functionName == "LoadImage":
                scriptTitle += "ID_Image_"+str(key)+"_ID,"
                inputImages.append("ID_Image_"+str(key)+"_ID")
            else:
                script = scriptFromXmlElement(dictScript[key]) + "\n"
                scriptBody += script
    scriptTitle = scriptTitle[:-1] + " ):"
    elementToReturn = "ID_" + xmlElement.tag + "_" + str(Dfct.childText(xmlElement, "ElementID"))+"_ID"

    scriptBody, scriptImports = separateImports(scriptBody)
    scriptBody = scriptBody.replace("\n", "\n\t")

    scriptComplete = scriptImports + '\n' + scriptTitle + "\n\n\t" + scriptBody + "return " + elementToReturn

    scriptComplete = renameInScript(inputImages, "in", scriptComplete)

    scriptComplete = scriptComplete.replace(elementToReturn, "output")

    allIDElements = []
    scriptSplit = scriptComplete.split('ID_')
    for splited in scriptSplit[1:]:
        element = "ID_"+splited.split('_ID')[0]+"_ID"
        if element not in allIDElements and element not in inputImages:
            allIDElements.append(element)

    scriptComplete = renameInScript(allIDElements, "", scriptComplete)

    return scriptComplete

def renameInScript(inputElements, prefix, scriptComplete):

    print("inputElements",inputElements)
    print("prefix",prefix)
    print("scriptComplete",scriptComplete)

    dictInputImages = {}
    for inputImage in inputElements:
        inputImage = inputImage[3:-3]
        dictInputImages[inputImage.split("_")[1]] = inputImage.split("_")[0]
    numImage = 1
    # keys = sorted(dictInputImages.keys())
    keys = dictInputImages.keys()
    for key in keys:
        newName = WidgetTypes.OutputType.typeToPrefix(dictInputImages[key])
        if prefix == '':
            newName = newName + str(numImage)
        else:
            newName = prefix + newName.capitalize() + str(numImage)
        scriptComplete = scriptComplete.replace("ID_"+dictInputImages[key]+"_"+str(key)+"_ID", newName)
        numImage += 1

    return scriptComplete

def separateImports(scriptBody):
    allImports = []
    scriptSplit = scriptBody.split('###')
    for splited in scriptSplit[1:]:
        element = splited.split('\n')[0] + '\n'
        if element not in allImports:
            allImports.append(element)
            scriptBody = scriptBody.replace('###' + element, '')

    scriptImports = ''
    for _import in sorted(allImports):
        scriptImports += _import

    return scriptBody, scriptImports

def findFunctionsAndArguments(textScript):
    textSplit = textScript.split('\ndef ')
    functionsDict = {}
    if textSplit[0].startswith('def '):
        textSplit[0] = textSplit[0].replace('def ', '')
        functionHeader = textSplit[0].replace(' ', '').split('):\n')[0]
        functionSplit = functionHeader.split('(')
        functionName = functionSplit[0]
        args = functionSplit[1].split(',')
        functionsDict[functionName] = args

    for i in range(1, len(textSplit)):
        functionHeader = textSplit[i].replace(' ', '').split('):\n')[0]
        functionSplit = functionHeader.split('(')
        functionName = functionSplit[0]
        args = functionSplit[1].split(',')
        functionsDict[functionName] = args
    return functionsDict


class ScriptWindow(qt.QWidget):

    def __init__(self, mainWindow, xmlElement):
        super().__init__()
        self.mainWindow = mainWindow
        self.xmlElement = xmlElement

        self.editScriptWindow = None

        # self.textEditScript = qt.QTextEdit()
        self.textEditScript = TextEditor()
        # self.highlight = PythonHighlighter(self.textEditScript.document())

        # ###
        # font = QtGui.QFont()
        # # font.setFamily("Courier")
        # font.setStyleHint(QtGui.QFont.Monospace)
        # font.setFixedPitch(True)
        # font.setPointSize(10)
        # self.textEditScript.setFont(font)
        # metrics = QtGui.QFontMetrics(font)
        # self.textEditScript.setTabStopWidth(4*metrics.width(' '))
        # ###
        #
        # self.textEditScript.setReadOnly(True)
        # self.textEditScript.setLineWrapMode(False)

        self.groupBoxButtons = qt.QGroupBox()
        layoutButtons = qt.QGridLayout()
        layoutButtons.setAlignment(Qt.AlignLeft)
        self.pushButtonExportScript = wgt.PushButtonImage(vrb.folderImages + "/Save.png")
        self.pushButtonExportScript.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        self.pushButtonExportScript.setToolTip(txt.dictToolTips["ExportScript"])
        self.pushButtonEditMacro = wgt.PushButtonImage(vrb.folderImages + "/Edit.png")
        self.pushButtonEditMacro.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        self.pushButtonEditMacro.setToolTip(txt.dictToolTips["CreateMacro"])

        layoutButtons.addWidget(self.pushButtonExportScript, 0, 0)
        layoutButtons.addWidget(self.pushButtonEditMacro, 0, 1)
        self.groupBoxButtons.setLayout(layoutButtons)
        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.groupBoxButtons, 0, 0)
        self.layout.addWidget(self.textEditScript, 1, 0)
        self.setLayout(self.layout)

        self.pushButtonEditMacro.clicked.connect(self.showEditMacroWindow)
        self.pushButtonExportScript.clicked.connect(self.exportScript)

        self.setWindowTitle('Function script')
        self.setMinimumSize(640, 360)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def exportScript(self):
        filename = qt.QFileDialog.getSaveFileName(self, 'Save as', vrb.folderDefault, "Python file (*.py)")
        if filename[0] is not None and filename[0] != '':
            with open(filename[0], 'w+') as file:
                file.write(self.textEditScript.getText() + '\n')

    def showEditMacroWindow(self):

        self.editScriptWindow = EditScriptWindow(xmlElement = None)
        self.editScriptWindow.textEditScript.setText(self.textEditScript.getText())
        self.editScriptWindow.groupParameters.actualizeInterface()
        self.editScriptWindow.show()
        # self.editScriptWindow.window().setWindowModality(Qt.ApplicationModal)
        self.editScriptWindow.textHasChanged = False

        self.close()


    def setText(self, text):
        # text = text.replace('\t', '    ')
        self.textEditScript.setText(text)


if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook


    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)


    sys.excepthook = exception_hook

    filename = vrb.folderFunctions + "/Macro/Vsnr2d.mho"
    xmlFile = xmlet.parse(filename)
    xmlElement = xmlFile.getroot()

    foo = EditScriptWindow(xmlFunctionDescription=xmlElement)

    #foo = EditScriptWindow(xmlFunctionDescription=None)

    #highlight = PythonHighlighter(foo.document())

    foo.show()

    app.exec_()