import PyQt5.QtWidgets as qt
from PyQt5 import QtCore
from PyQt5 import QtGui

import PyIPSDK
import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr
import PyIPSDK.IPSDKIPLUtility as util

import numpy as np
import math

import UsefullVariables as vrb
import UsefullWidgets as wgt
import UsefullFunctions as fct

class RangeSlider(qt.QGroupBox):

    signalValueChanged = QtCore.pyqtSignal(float, float)

    def __init__(self,fontSize = None,customResize = [33,67], transparency=128,style = "Classic",labelSize = 50*vrb.ratio):
        qt.QGroupBox.__init__(self)

        self.sliderStyle = style

        self.customResize = customResize
        self.fontSize = fontSize
        self.type = PyIPSDK.eIBT_UInt8
        self.isFixedMin = True
        self.isFixedMax = True
        self.isLockedMin = False
        self.isLockedMax = False
        self.currentIndex = 0

        self.labelSize = labelSize
        self.marginLabelSize = 6*vrb.ratio
        self.handleSize = 4*vrb.ratio
        self.minValue = 0
        self.maxValue = 255
        # self.currentValueMin = int(self.maxValue/2)
        self.currentValueMin = self.minValue
        self.currentValueMax = self.maxValue
        self.allowConnect = True

        self.setStyleSheet('QGroupBox {border: 0px transparent; }')

        self.groupLabelValue = GroupValueLabel(labelSize=self.labelSize,fontSize = fontSize)
        self.groupLabelValue.setFixedHeight(self.height()*33/100)

        self.groupSlider = GroupSlider(parent=self,labelSize=self.labelSize,handleSize=self.handleSize,minValue=self.minValue,maxValue=self.maxValue,fontSize = fontSize, transparency=transparency )
        self.groupSlider.setFixedHeight(self.height()*67/100)

        self.layout = qt.QGridLayout()
        self.setLayout(self.layout)

        self.layout.setContentsMargins(0, 0, 0, 0)
        self.groupSlider.setContentsMargins(0, 0, 0, 0)
        self.groupLabelValue.setContentsMargins(0, 0, 0, 0)
        self.setContentsMargins(0,0,0,0)
        self.layout.setVerticalSpacing(0)

        if self.sliderStyle == "Classic":
            self.layout.addWidget(self.groupLabelValue,0,0)
            self.layout.addWidget(self.groupSlider,1,0)
        elif self.sliderStyle == "Reduced":
            self.layout.addWidget(self.groupSlider,0,0)

        self.groupSlider.signalValueChanged.connect(self.changeLabelsInterface)
        self.groupSlider.signalMouseClicked.connect(self.applyClick)

        self.refreshValues()

    def resizeEvent(self,event):

        if self.sliderStyle == "Classic":
            self.groupLabelValue.setFixedHeight(self.height() * self.customResize[0] / 100)
            self.groupSlider.setFixedHeight(self.height() * self.customResize[1] / 100)
        elif self.sliderStyle == "Reduced":
            self.groupSlider.setFixedHeight(self.height())

        # self.setValues(self.currentValueMin,self.currentValueMax)

    def applyClick(self,ratio):

        value = self.minValue + ratio*(self.maxValue - self.minValue)
        if self.type != PyIPSDK.eIBT_Real32:
            value = round(value)
        if value <= (self.currentValueMin + self.currentValueMax)/2 and self.isLockedMin == False:
            self.isFixedMin = False
            self.setValueMin(value)
        elif value > (self.currentValueMin + self.currentValueMax)/2 and self.isLockedMax == False:
            self.isFixedMax = False
            self.setValueMax(value)
        elif self.isLockedMin:
            self.isFixedMax = False
            self.setValueMax(value)
        elif self.isLockedMax:
            self.isFixedMin = False
            self.setValueMin(value)

    def setMinMaxValues(self,minValue,maxValue):

        if minValue != self.minValue or maxValue != self.maxValue:
            self.minValue = float(fct.adjustedNumberCalibration(minValue,mode = 'min',numberMeaningful=3))
            self.maxValue = float(fct.adjustedNumberCalibration(maxValue,mode = 'max',numberMeaningful=3))
            self.groupSlider.labelMin.setText(fct.adjustedNumberCalibration(minValue,mode = 'min',numberMeaningful=3))
            self.groupSlider.labelMax.setText(fct.adjustedNumberCalibration(maxValue,mode = 'max',numberMeaningful=3))

            self.setValues(self.currentValueMin, self.currentValueMax)

    def resetValues(self):

        self.setValues(self.minValue,self.maxValue)

    def refreshValues(self):

        self.setValues(self.currentValueMin,self.currentValueMax)

    def changeLabelsInterface(self):

        if self.allowConnect:
            if self.currentIndex == 1:
                self.isFixedMin = False
            elif self.currentIndex == 2:
                self.isFixedMax = False
            self.changeLabels()

    def changeLabels(self):

        if self.allowConnect:
            # start = self.groupSlider.groupBoxMiddle.pos().x() - self.groupSlider.handleSize - 2*vrb.ratio
            start = self.groupSlider.groupBoxMiddle.pos().x() - self.groupSlider.handleSize
            # end = start + self.groupSlider.groupBoxMiddle.width() + 2*vrb.ratio
            end = start + self.groupSlider.groupBoxMiddle.width()

            maxSize = self.width() - 2 * (self.labelSize + self.handleSize) - 2 * self.marginLabelSize + 2*vrb.ratio

            # maxSize = max(maxSize,end)

            minValue = start * (self.maxValue-self.minValue) / maxSize + self.minValue
            minValue = max(minValue,self.minValue)
            maxValue = end * (self.maxValue-self.minValue) / maxSize + self.minValue
            maxValue = min(maxValue,self.maxValue)
            minValue = min(minValue,self.maxValue)
            maxValue = max(maxValue,self.minValue)

            if self.type != PyIPSDK.eIBT_Real32:
                minValue = round(minValue)
                maxValue = round(maxValue)

            if self.isFixedMin:
                minValue = self.currentValueMin
            if self.isFixedMax:
                maxValue = self.currentValueMax

            if self.sliderStyle == "Classic":
                self.groupLabelValue.labelMinValue.setText(fct.numberCalibration(minValue,3))
                self.groupLabelValue.labelMaxValue.setText(fct.numberCalibration(maxValue,3))
            elif self.sliderStyle == "Reduced":
                self.groupSlider.labelMin.setText(fct.numberCalibration(minValue,3))
                self.groupSlider.labelMax.setText(fct.numberCalibration(maxValue,3))

            self.currentValueMin = minValue
            self.currentValueMax = maxValue

            self.groupLabelValue.layout.setStretch(0, start + (self.marginLabelSize + self.handleSize))
            self.groupLabelValue.layout.setStretch(2, end -start)
            self.groupLabelValue.layout.setStretch(4,maxSize-end + (self.marginLabelSize + self.handleSize))

            self.signalValueChanged.emit(minValue,maxValue)

    def setValues(self,minValue=None,maxValue=None):

        self.allowConnect = False
        maxSize = self.width() - 2 * (self.labelSize + self.handleSize) - 2 * self.marginLabelSize + 2*vrb.ratio
        # maxSize = self.width() - 2 * self.labelSize - 2 * self.marginLabelSize + 2*vrb.ratio
        denominator = (self.maxValue - self.minValue)
        if denominator == 0:
            denominator = 1

        if minValue is not None:
            minValue = max(minValue, self.minValue)
            self.groupSlider.splitter.moveSplitter(0, 1)
            # if abs(minValue-self.minValue)>0.0001:
            #     start = (minValue - self.minValue) * maxSize / denominator
            #     self.groupSlider.splitter.moveSplitter(start, 1)

            start = (minValue - self.minValue) * maxSize / denominator
            self.groupSlider.splitter.moveSplitter(start, 1)

            self.isFixedMin = True
            self.currentValueMin = minValue
        if maxValue is not None:
            maxValue = min(maxValue, self.maxValue)
            self.groupSlider.splitter.moveSplitter(maxSize + self.handleSize + 4*vrb.ratio, 2)

            if abs(maxValue-self.maxValue)>0.0001:
                end = (maxValue - self.minValue) * maxSize / denominator
                # self.groupSlider.splitter.moveSplitter(end + self.handleSize + 4*vrb.ratio, 2)
                self.groupSlider.splitter.moveSplitter(end + self.handleSize , 2)
            else:
                end = (maxValue - self.minValue) * maxSize / denominator
                self.groupSlider.splitter.moveSplitter(end + self.handleSize + 4*vrb.ratio, 2)

            self.isFixedMax = True
            self.currentValueMax = maxValue

        # self.groupSlider.splitter.moveSplitter(maxSize + self.handleSize+10, 2)

        self.allowConnect = True

        self.changeLabels()

    def setValueMin(self,minValue):

        self.setValues(minValue=minValue)

    def setValueMax(self,maxValue):

        self.setValues(maxValue=maxValue)

class GroupSlider(qt.QWidget):

    signalValueChanged = QtCore.pyqtSignal()
    signalMouseClicked = QtCore.pyqtSignal(float)

    def __init__(self,parent=None,labelSize = 30,handleSize = 3, minValue = 0, maxValue = 255,fontSize = None,transparency = 128):
        qt.QWidget.__init__(self)

        self.parent = parent
        self.labelSize = labelSize
        self.handleSize = handleSize
        self.minValue = minValue
        self.maxValue = maxValue

        self.groupBoxStart = qt.QWidget()
        self.groupBoxStart.setStyleSheet('QWidget {background-color: transparent;}')

        self.groupBoxMiddle = qt.QWidget()
        # self.groupBoxMiddle.setStyleSheet('QWidget {background-color: rgba(255, 255, 255,128);}')
        self.groupBoxMiddle.setStyleSheet('QWidget {background-color: rgba(255, 255, 255,'+str(transparency)+');}')

        self.groupBoxEnd = qt.QWidget()
        self.groupBoxEnd.setStyleSheet('QWidget {background-color: transparent;}')

        image = np.zeros((1000, 1000, 3))
        image = np.asarray(image, dtype=np.uint8)
        self.labelBackground = wgt.LabelImage(image=image)
        #self.labelBackground = wgt.LabelImage(str(vrb.folderButtons)+"threshold.jpg")
        #self.groupBoxBackground.setStyleSheet('QWidget {background-image: url("'+str(vrb.folderButtons)+'threshold.jpg");}')

        self.splitter = qt.QSplitter()
        self.splitter.setStyleSheet('QSplitter::handle:vertical {height: '+ str(self.handleSize) + 'px;} QSplitter::handle {background-color: darkblue;}') # QSplitter::handle:pressed {background-color: rgb(0, 100, 0);}')

        self.splitter.addWidget(self.groupBoxStart)
        self.splitter.addWidget(self.groupBoxMiddle)
        self.splitter.addWidget(self.groupBoxEnd)

        font = QtGui.QFont()
        if fontSize is not None:
            font.setPointSize(fontSize)
        font.setBold(True)

        self.labelMin = qt.QLabel(str(self.minValue))
        self.labelMin.setFixedWidth(labelSize)
        self.labelMin.setAlignment(QtCore.Qt.AlignCenter)
        self.labelMin.setFont(font)

        self.labelMax = qt.QLabel(str(self.maxValue))
        self.labelMax.setFixedWidth(labelSize)
        self.labelMax.setAlignment(QtCore.Qt.AlignCenter)
        self.labelMax.setFont(font)

        self.layout = qt.QGridLayout()

        self.layout.setContentsMargins(0,0,0,0)
        self.setContentsMargins(0,0,0,0)

        self.layout.addWidget(self.labelMin,0,0)
        self.layout.addWidget(self.labelBackground, 0, 1)
        self.layout.addWidget(self.splitter, 0, 1)
        self.layout.addWidget(self.labelMax,0,2)

        self.setLayout(self.layout)

        self.splitter.splitterMoved.connect(self.handleMoveSplitter)

    # def resizeEvent(self, a0: QtGui.QResizeEvent) -> None:
    #
    #     self.splitter.setFixedWidth(50*vrb.ratio)
    #     self.labelBackground.setFixedWidth(50*vrb.ratio)

    def applyExclusive(self,exclusive):

        if exclusive:
            self.groupBoxStart.setStyleSheet('QWidget {background-color: rgba(255, 255, 255,128);}')
            self.groupBoxMiddle.setStyleSheet('QWidget {background-color: transparent;}')
            self.groupBoxEnd.setStyleSheet('QWidget {background-color: rgba(255, 255, 255,128);}')
        else:
            self.groupBoxStart.setStyleSheet('QWidget {background-color: transparent;}')
            self.groupBoxMiddle.setStyleSheet('QWidget {background-color: rgba(255, 255, 255,128);}')
            self.groupBoxEnd.setStyleSheet('QWidget {background-color: transparent;}')

    def handleMoveSplitter(self, xpos, index):

        if index == 1:
            self.groupBoxEnd.setMinimumWidth(self.groupBoxEnd.width())
            self.groupBoxStart.setMinimumWidth(0)
        if index == 2:
            self.groupBoxStart.setMinimumWidth(self.groupBoxStart.width())
            self.groupBoxEnd.setMinimumWidth(0)

        self.parent.currentIndex = index

        self.signalValueChanged.emit()

    def mousePressEvent(self,event):

        posX = event.pos().x()-self.splitter.pos().x()
        ratio = posX/self.splitter.width()
        if ratio >= 0 and ratio <= 1:
            self.signalMouseClicked.emit(ratio)

    def getHistogram(self,histogram,color = [255,0,0],image = None, output = False, contour = False):

        newHistogram = []
        for i in range(len(histogram)):
            newHistogram.append(math.sqrt(histogram[i]))

        if len(newHistogram)>=2:
            maxValue = max(newHistogram)
            index = newHistogram.index(maxValue)
            newHistogram[index] = 0
            maxValue2 = max(newHistogram)
            newHistogram[index] = min(3*maxValue2,maxValue)

        sizeX, sizeY = 1000, 1000
        if image is None:
            image = np.zeros((sizeY, sizeX, 3))
            image = np.asarray(image, dtype=np.uint8)

        maxFreq = max(newHistogram)
        maxFreq = max(1,maxFreq)
        step = sizeX/len(newHistogram)

        for i in range(len(newHistogram)):
            for c in range(len(color)):
                if color[c]!=0:
                    if contour == False:
                        image[sizeY-int((newHistogram[i]/maxFreq)*sizeY):sizeY,int(step*i):int(step*(i+1)),c] = color[c]
                    else:
                        image[sizeY - int((newHistogram[i] / maxFreq) * sizeY)-5:sizeY - int((newHistogram[i] / maxFreq) * sizeY)+5, int(step * i):int(step * (i + 1)), c] = color[c]
                        try:
                            image[min(sizeY - int((newHistogram[i] / maxFreq) * sizeY),sizeY - int((newHistogram[i+1] / maxFreq) * sizeY)):
                                  max(sizeY - int((newHistogram[i] / maxFreq) * sizeY),sizeY - int((newHistogram[i+1] / maxFreq) * sizeY)), int(step * (i + 1))-1:int(step * (i + 1))+1, c] = color[c]
                        except:
                            pass

        if output == False:
            self.labelBackground.image = image
            self.labelBackground.resizeEvent(None)
        else:
            return image

    def getColorHistogram(self,image,contour=False):

        histogramMsrParams = PyIPSDK.createHistoMsrParamsWithBinWidth(0, 255, 1)
        colors = [[255,0,0],[0,255,0],[0,0,255]]
        outImage = None

        for c in range(3):

            plan = PyIPSDK.extractPlan(0,c,0,image)
            plan = util.copyImg(plan)
            histogramMsr2dResult = glbmsr.histogramMsr2d(plan, histogramMsrParams)
            outImage = self.getHistogram(histogramMsr2dResult.frequencies, contour=contour, color=colors[c],image=outImage,output=True)

        vector = outImage[:,:,2] == 255
        outImage[vector, 0] = 0
        outImage[vector, 1] = 0
        vector = outImage[:, :, 1] == 255
        outImage[vector, 0] = 0

        self.labelBackground.image = outImage
        self.labelBackground.resizeEvent(None)


class GroupValueLabel(qt.QWidget):

    def __init__(self, labelSize = 30,fontSize = None):
        qt.QWidget.__init__(self)

        self.labelSize = labelSize

        self.layout = qt.QHBoxLayout()

        font = QtGui.QFont()
        if fontSize is not None:
            font.setPointSize(fontSize)
        font.setBold(True)

        self.labelMinValue = qt.QLabel('0')
        self.labelMinValue.setFixedWidth(self.labelSize)
        self.labelMinValue.setAlignment(QtCore.Qt.AlignRight)
        self.labelMinValue.setFont(font)

        self.labelMaxValue = qt.QLabel('255')
        self.labelMaxValue.setFixedWidth(self.labelSize)
        self.labelMaxValue.setAlignment(QtCore.Qt.AlignLeft)
        self.labelMaxValue.setFont(font)

        self.layout.addStretch()
        self.layout.addWidget(self.labelMinValue)
        self.layout.addStretch()
        self.layout.addWidget(self.labelMaxValue)
        self.layout.addStretch()

        self.setLayout(self.layout)

        self.layout.setContentsMargins(0, 0, 0, 0)
        self.setContentsMargins(0, 0, 0, 0)




if __name__ == '__main__':

    app = QtCore.QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    foo = RangeSlider(style = "Reduced")
    foo.setFixedSize(300,50)
    foo.show()

    foo.setMinMaxValues(50,100)
    foo.setValues(50,81)

    app.exec_()