import time
start = time.time()

import sys
import os
import traceback

from PyQt5.QtCore import pyqtSignal,QPointF,Qt,QCoreApplication
from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore

import xml.etree.ElementTree as xmlet

import UsefullWidgets as wgt
import UsefullVariables as vrb
import UsefullFunctions as fct
import DatabaseFunction as Dfct
import UsefullTexts as txt

import PyIPSDK
import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr
import PyIPSDK.IPSDKIPLUtility as util

from GroupCalibration import GroupBoxCalibration
from RangeSlider import RangeSlider
from SettingsWidget import MovieMakerPreferencesWidget

import numpy as np
import random

import napari
from napari.layers import Image, Surface, Points, Shapes,Labels
import imageio
from scipy import ndimage as ndi

try:
    import cv2
    from PIL import Image as PILImage
except:
    pass

import vispy

import copy

# Liste d'attributs complexes ou internes qui ne doivent pas être mis à jour manuellement
ATTRIBUTES_TO_IGNORE = ['_data', '_data_raw', '_slice', 'events', 'extent', '_thumbnail', 'thumbnail', '_slice_input','_pad_width','pad_width']


class MovieMakerWithMenu(qt.QGroupBox):

    def __init__(self, parent=None,position = 0):
        qt.QGroupBox.__init__(self)

        self.menu = MenuMovieMaker()
        self.movieMakerScrollArea = MovieMakerScrollArea()

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.menu,0,0,Qt.AlignLeft)
        self.layout.addWidget(self.movieMakerScrollArea,1,0,Qt.AlignTop)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(5*vrb.ratio, 5*vrb.ratio, 5*vrb.ratio, 5*vrb.ratio)
        self.layout.setContentsMargins(0,0,0,0)
        self.layout.setSpacing(0)

        self.setLayout(self.layout)

        self.menu.buttonDelete.clicked.connect(self.movieMakerScrollArea.centralWidget.removeCurrentFrame)
        self.menu.buttonAdd.clicked.connect(self.movieMakerScrollArea.centralWidget.addFrame)
        self.menu.buttonRotate.clicked.connect(self.movieMakerScrollArea.centralWidget.rotateFrame)
        self.menu.buttonPlayMovie.clicked.connect(self.movieMakerScrollArea.centralWidget.playMovie)
        self.menu.buttonPauseMovie.clicked.connect(self.movieMakerScrollArea.centralWidget.pauseMovie)
        self.menu.buttonSaveMovie.clicked.connect(self.movieMakerScrollArea.centralWidget.saveMovie)
        self.menu.buttonDeleteAll.clicked.connect(self.movieMakerScrollArea.centralWidget.deleteAll)
        self.menu.buttonSaveScenario.clicked.connect(self.movieMakerScrollArea.centralWidget.saveScenario)
        self.menu.buttonSaveScenarioFolder.clicked.connect(self.movieMakerScrollArea.centralWidget.saveScenarioFolder)
        self.menu.buttonLoadScenario.clicked.connect(self.movieMakerScrollArea.centralWidget.loadScenario)
        self.menu.buttonSettings.clicked.connect(self.movieMakerScrollArea.centralWidget.openSettings)

        self.setFixedHeight(165*vrb.ratio)

        # self.setStyleSheet("QGroupBox {background-color: rgb(100,0,0); }")

    def resizeEvent(self, a0: QtGui.QResizeEvent) -> None:

        self.movieMakerScrollArea.setFixedHeight(max(0,self.height()-35*vrb.ratio))

class FrameWidget(qt.QGroupBox):

    def __init__(self, parent=None,position = 0):
        qt.QGroupBox.__init__(self)

        self.position = position

        self.isFrame = False

        self.parent = parent

        emptyLabel = qt.QLabel()
        emptyLabel.setFixedHeight(15*vrb.ratio)

        self.label = qt.QLabel()
        self.label.setFixedSize(40*vrb.ratio,20*vrb.ratio)

        self.labelRotation = wgt.LabelImage(vrb.folderImages + '/Rotation.png')
        self.labelRotation.setFixedSize(20*vrb.ratio,20*vrb.ratio)

        self.imageThumbnails = wgt.LabelImage()
        self.imageThumbnails.setFixedSize(50*vrb.ratio,50*vrb.ratio)

        self.contextMenu = qt.QMenu(self)
        self.contextMenu.addAction('Update frame', self.updateFrameContent)
        self.contextMenu.addAction('Remove frame', self.removeFrame)
        self.contextMenu.addAction('Duplicate frame', self.duplicateFrame)
        self.contextMenu.addAction('Edit frame', self.editFrame)
        self.actionRotation = qt.QAction("Change rotation", self.contextMenu)
        self.actionRotation.triggered.connect(self.changeRotation)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(emptyLabel,0,0,1,2)
        self.layout.addWidget(self.label,1,0)
        self.layout.addWidget(self.labelRotation,1,1)
        self.layout.addWidget(self.imageThumbnails,2,0,1,2)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(5*vrb.ratio, 5*vrb.ratio, 4*vrb.ratio, 4*vrb.ratio)
        self.layout.setSpacing(0)

        self.setFixedSize(75*vrb.ratio, 95*vrb.ratio)

        # self.setStyleSheet("QGroupBox {border: 1px solid gray; }")

    def changeRotation(self):

        dictInfo = self.parent.dictAllFrames[self.position][2]
        direction = dictInfo["Camera"]["angles"].direction
        if direction[2] is not None:
            self.parent.rotateFrameWidget.comboBoxAxis.setCurrentText("X")
            self.parent.rotateFrameWidget.spinBoxAngle.setValue(direction[2])
        elif direction[1] is not None:
            self.parent.rotateFrameWidget.comboBoxAxis.setCurrentText("Y")
            self.parent.rotateFrameWidget.spinBoxAngle.setValue(direction[1])
        elif direction[0] is not None:
            self.parent.rotateFrameWidget.comboBoxAxis.setCurrentText("Z")
            self.parent.rotateFrameWidget.spinBoxAngle.setValue(direction[0])

        self.parent.frameWithUpdatedRotation = self

        fct.showWidget(self.parent.rotateFrameWidget)

    def updateFrameContent(self,pixmapOnly=False):

        viewerNapari = vrb.mainWindow.viewerNapariQt
        imageNp = viewerNapari.screenshot(flash=False)
        image = fct.numpyImageToIPSDKImage(imageNp, invertChannels=False)
        outImage = PyIPSDK.createImageRgb(image.getBufferType(), max(image.getSizeX(), image.getSizeY()), max(image.getSizeX(), image.getSizeY()))
        util.eraseImg(outImage, 0)
        util.putROI2dImg(outImage, image, int((max(image.getSizeX(), image.getSizeY()) - image.getSizeX()) / 2), int((max(image.getSizeX(), image.getSizeY()) - image.getSizeY()) / 2), outImage)
        pixmapImage = fct.IPSDKImageToPixmap(outImage)

        dictInfo = self.parent.createDictInfo()
        self.parent.dictAllFrames[self.position][1] = pixmapImage
        if pixmapOnly == False:
            self.parent.dictAllFrames[self.position][2] = dictInfo

        self.parent.updateAllFrames(self.parent.dictAllFrames)

    def removeFrame(self):

        self.parent.dictAllFrames[self.position] = None
        self.parent.updateAllFrames(self.parent.dictAllFrames)

    def duplicateFrame(self):

        try:
            if self.parent.dictAllFrames[self.position+1] is None:
                pixmapCopy = self.parent.dictAllFrames[self.position][1].copy()
                dictInfoCopy = self.parent.dictAllFrames[self.position][2].copy()
                self.parent.dictAllFrames[self.position+1] = [str(int(((self.position+1)*vrb.movieMakerFrameTime)*10)/10) + " s",pixmapCopy,dictInfoCopy]
            else:
                self.messageBox = wgt.MessageBox("You can't duplicate the frame if there is another one directly after it", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                self.messageBox.exec()

            self.parent.updateAllFrames(self.parent.dictAllFrames)

        except:
            traceback.print_exc(file=sys.stderr)

    def editFrame(self):

        self.parent.frameEditorWidget.show()
        self.parent.frameEditorWidget.loadFromFrame(self)

    def mouseMoveEvent(self, e):

        if self.isFrame:
            if e.buttons() == Qt.LeftButton:
                drag = QtGui.QDrag(self)
                mime = QtCore.QMimeData()
                drag.setMimeData(mime)
                drag.exec_(Qt.MoveAction)

    def mousePressEvent(self, e):

        if self.isFrame and e.button() == 1:
            self.parent.currentPosition = self.position
        elif self.isFrame == False and e.button() == 1:
            self.parent.currentPosition = None
        elif self.isFrame and e.button() == 2:
            self.contextMenu.exec(self.contextMenu.parent().mapToGlobal(e.pos()))

        # self.parent.dictAllFrames = self.parent.adjustDict(self.parent.dictAllFrames)
        self.parent.updateAllFrames(self.parent.dictAllFrames)

    def mouseDoubleClickEvent(self, e):

        if self.isFrame:
            # self.sliderPlay.pos().x()
            self.parent.sliderPlay.splitter.moveSplitter(self.pos().x()-self.parent.sliderPlay.pos().x(), 1)
            self.parent.updateMovie()
            # qt.QApplication.processEvents()
            # self.parent.updateMovie()

    def updateFrame(self,frame):

        self.label.setVisible(frame is not None)
        self.imageThumbnails.setVisible(frame is not None)
        if frame is None:
            if self.isFrame:
                self.setStyleSheet("QGroupBox {background-color: transparent;border: 0px transparent; }")
            self.setFixedWidth(1*vrb.ratio)
            self.isFrame = False
        else:
            self.isFrame = True
            self.label.setText(frame[0])

            if frame[2]["Camera"]["angles"].direction[0] is not None or frame[2]["Camera"]["angles"].direction[1] is not None or\
                frame[2]["Camera"]["angles"].direction[2] is not None:
                self.labelRotation.setVisible(True)
                self.contextMenu.addAction(self.actionRotation)
            else:
                self.labelRotation.setVisible(False)
                self.contextMenu.removeAction(self.actionRotation)

            # self.imageThumbnails.filename = frame[1]
            if frame[1] is None:
                self.parent.applyDictToNapari(frame[2])
                viewerNapari = vrb.mainWindow.viewerNapariQt
                imageNp = viewerNapari.screenshot(flash=False)
                image = fct.numpyImageToIPSDKImage(imageNp, invertChannels=False)
                outImage = PyIPSDK.createImageRgb(image.getBufferType(), max(image.getSizeX(), image.getSizeY()), max(image.getSizeX(), image.getSizeY()))
                util.eraseImg(outImage, 0)
                util.putROI2dImg(outImage, image, int((max(image.getSizeX(), image.getSizeY()) - image.getSizeX()) / 2), int((max(image.getSizeX(), image.getSizeY()) - image.getSizeY()) / 2), outImage)
                pixmapImage = fct.IPSDKImageToPixmap(outImage)
                frame[1] = pixmapImage

            self.imageThumbnails.pixmap = frame[1]
            self.imageThumbnails.resizeEvent(None)
            if self.parent.currentPosition == self.position:
                self.setStyleSheet("QGroupBox {background-color: rgb(100, 100, 100);border: 3px solid rgb(6, 115, 186); }")
                self.layout.setContentsMargins(3 * vrb.ratio, 3 * vrb.ratio, 2 * vrb.ratio, 2 * vrb.ratio)
                self.parentCurrentWidget = self
            else:
                self.setStyleSheet("QGroupBox {background-color: rgb(100, 100, 100);border: 0px transparent; }")
                self.layout.setContentsMargins(5 * vrb.ratio, 5 * vrb.ratio, 5 * vrb.ratio, 5 * vrb.ratio)
            self.setFixedWidth(60*vrb.ratio)

class MovieMakerScrollArea(qt.QScrollArea):

    def __init__(self):
        qt.QScrollArea.__init__(self)

        self.setWidgetResizable(True)
        self.setVerticalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)

        self.centralWidget = MovieMaker()

        self.setWidget(self.centralWidget)

        # self.setStyleSheet("QScrollArea {background-color: rgb(100,0,0); }")

        # self.setFixedHeight(70*vrb.ratio)

    def keyPressEvent(self, event):
        key = event.key()

        if key == 16777234: #arrow left
            self.centralWidget.moveFrame(-1)
        if key == 16777236: #arrow right
            self.centralWidget.moveFrame(1)
        if key == 16777223: #Suppr
            self.centralWidget.removeCurrentFrame()
        if key == 16777249:
            self.centralWidget.ctrlPressed = True

        self.centralWidget.dictAllFrames = self.centralWidget.adjustDict(self.centralWidget.dictAllFrames)
        self.centralWidget.updateAllFrames(self.centralWidget.dictAllFrames)

    def keyReleaseEvent(self, event):

        key = event.key()

        if key == 16777249:
            self.centralWidget.ctrlPressed = False


class MenuMovieMaker(qt.QGroupBox):

    def __init__(self):
        qt.QGroupBox.__init__(self)

        self.layout = qt.QHBoxLayout()

        self.buttonAdd = wgt.PushButtonImage(vrb.folderImages + "/Add_4.png", margins=2)
        self.buttonAdd.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonAdd.setToolTip(txt.dictToolTips["AddNewFrame"])

        self.buttonDelete = wgt.PushButtonImage(vrb.folderImages + "/Trash2.png", margins=2)
        self.buttonDelete.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonDelete.setToolTip(txt.dictToolTips["RemoveFrame"])

        self.buttonRotate = wgt.PushButtonImage(vrb.folderImages + "/Rotation.png", margins=2)
        self.buttonRotate.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonRotate.setToolTip(txt.dictToolTips["RotationFrame"])

        self.buttonPlayMovie = wgt.PushButtonImage(vrb.folderImages + "/Lecture.png", margins=2)
        self.buttonPlayMovie.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonPlayMovie.setToolTip(txt.dictToolTips["PlayMovie"])

        self.buttonPauseMovie = wgt.PushButtonImage(vrb.folderImages + "/Pause.png", margins=2)
        self.buttonPauseMovie.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonPauseMovie.setToolTip(txt.dictToolTips["PauseMovie"])

        self.buttonSaveMovie = wgt.PushButtonImage(vrb.folderImages + "/Save_Movie.png", margins=2)
        self.buttonSaveMovie.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonSaveMovie.setToolTip(txt.dictToolTips["SaveMovie"])

        self.buttonDeleteAll = wgt.PushButtonImage(vrb.folderImages + "/Delete.png", margins=2)
        self.buttonDeleteAll.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonDeleteAll.setToolTip(txt.dictToolTips["DeleteAllFrames"])

        self.buttonSaveScenario = wgt.PushButtonImage(vrb.folderImages + "/Save.png", margins=2)
        self.buttonSaveScenario.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonSaveScenario.setToolTip(txt.dictToolTips["SaveScenario"])

        self.buttonSaveScenarioFolder = wgt.PushButtonImage(vrb.folderImages + "/Save_Folder.png", margins=2)
        self.buttonSaveScenarioFolder.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonSaveScenarioFolder.setToolTip(txt.dictToolTips["SaveScenarioFolder"])

        self.buttonLoadScenario = wgt.PushButtonImage(vrb.folderImages + "/Folder.png", margins=2)
        self.buttonLoadScenario.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonAdd.setToolTip(txt.dictToolTips["LoadScenario"])

        self.buttonSettings = wgt.PushButtonImage(vrb.folderImages + "/Settings.png", margins=2)
        self.buttonSettings.setFixedSize(25*vrb.ratio,25*vrb.ratio)
        self.buttonAdd.setToolTip(txt.dictToolTips["SettingsMovieMaker"])

        self.layout.addWidget(self.buttonAdd)
        self.layout.addWidget(self.buttonDelete)
        self.layout.addSpacing(10*vrb.ratio)
        self.layout.addWidget(self.buttonRotate)
        self.layout.addSpacing(10*vrb.ratio)
        self.layout.addWidget(self.buttonPlayMovie)
        self.layout.addWidget(self.buttonPauseMovie)
        self.layout.addSpacing(10*vrb.ratio)
        self.layout.addWidget(self.buttonSaveMovie)
        self.layout.addSpacing(10*vrb.ratio)
        self.layout.addWidget(self.buttonDeleteAll)
        self.layout.addSpacing(10*vrb.ratio)
        self.layout.addWidget(self.buttonSaveScenario)
        self.layout.addWidget(self.buttonSaveScenarioFolder)
        self.layout.addWidget(self.buttonLoadScenario)
        self.layout.addSpacing(10*vrb.ratio)
        self.layout.addWidget(self.buttonSettings)

        # self.layout.insertSpacing(2,20*vrb.ratio)
        self.layout.setAlignment(Qt.AlignLeft)
        self.setContentsMargins(0,0,0,0)

        self.setLayout(self.layout)

        self.setFixedHeight(35*vrb.ratio)

class SliderPlay(qt.QGroupBox):

    def __init__(self,parent = None):
        qt.QGroupBox.__init__(self)

        self.parent = parent

        self.isHandlingSplitter = False

        self.groupBoxStart = qt.QWidget()
        self.groupBoxStart.setStyleSheet('QWidget {background-color: transparent;}')

        self.groupBoxMiddle = qt.QWidget()
        self.groupBoxMiddle.setStyleSheet('QWidget {background-color: rgba(0, 255, 0,'+str(128)+');}')
        self.groupBoxMiddle.setFixedWidth(10*vrb.ratio)

        self.groupBoxEnd = qt.QWidget()
        self.groupBoxEnd.setStyleSheet('QWidget {background-color: transparent;}')

        self.splitter = qt.QSplitter()
        self.splitter.setStyleSheet('QSplitter::handle:vertical {height: '+ str(0) + 'px;} QSplitter::handle {background-color: darkblue;}') # QSplitter::handle:pressed {background-color: rgb(0, 100, 0);}')

        self.splitter.addWidget(self.groupBoxStart)
        self.splitter.addWidget(self.groupBoxMiddle)
        self.splitter.addWidget(self.groupBoxEnd)

        self.layout = qt.QGridLayout()

        self.layout.setContentsMargins(0, 0, 0, 0)
        self.setContentsMargins(0, 0, 0, 0)

        self.layout.addWidget(self.splitter, 0, 0)

        self.setLayout(self.layout)

        self.splitter.splitterMoved.connect(self.handleMoveSplitter)

        self.setFixedHeight(15*vrb.ratio)

    def handleMoveSplitter(self):
        if self.isHandlingSplitter == False:

            self.isHandlingSplitter = True

            if self.groupBoxStart.width() == 0:
                self.splitter.moveSplitter(self.groupBoxMiddle.width(),2)
            if self.groupBoxEnd.width() == 0:
                self.splitter.moveSplitter(self.width()-self.groupBoxMiddle.width(),1)

            self.isHandlingSplitter = False

        self.parent.updateMovie()

    def mousePressEvent(self,event):

        self.splitter.moveSplitter(event.pos().x(),1)

class MovieMaker(qt.QGroupBox):

    def __init__(self):
        qt.QGroupBox.__init__(self)

        self.settingsMovieMaker = MovieMakerPreferencesWidget()
        self.frameEditorWidget = FrameEditorWidget()
        self.rotateFrameWidget = RotateFrameWidget()
        self.frameWithUpdatedRotation = None

        self.firstUpdate = True

        self.ctrlPressed = False

        self.stopMovie = True

        self.currentPosition = None
        self.currentWidget = None

        self.setAcceptDrops(True)

        # self.layout = qt.QHBoxLayout()
        self.layout = qt.QGridLayout()
        self.layout.setAlignment(Qt.AlignLeft)
        # self.layout.setSpacing(0)
        self.layout.setHorizontalSpacing(0)
        self.setContentsMargins(0,0,0,0)

        self.setLayout(self.layout)

        self.dictAllFrames = {}
        self.allFramesList = []
        for i in range(vrb.movieMakerNbFrames):
            frame = FrameWidget(parent = self,position=i)
            # self.layout.addWidget(frame)
            label = qt.QLabel()
            label.setStyleSheet("QLabel {background-color: rgb(100,100,100); }")
            label.setFixedHeight(15*vrb.ratio)
            if i % 20 == 0:
                label.setText(str(int(i*vrb.movieMakerFrameTime)) + " s")
            self.layout.addWidget(label,0,i)
            self.layout.addWidget(frame,1,i, Qt.AlignTop)
            self.allFramesList.append(frame)
            self.dictAllFrames[i] = None

        self.sliderPlay = SliderPlay(parent = self)

        self.layout.addWidget(self.sliderPlay,1,0,1,vrb.movieMakerNbFrames, Qt.AlignTop)

        self.previousDict = self.dictAllFrames.copy()
        self.backUpDict = self.dictAllFrames.copy()

        self.previousPosition = self.currentPosition
        self.backUpPosition = self.currentPosition

        self.updateAllFrames(self.dictAllFrames)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        # self.setFixedHeight(80*vrb.ratio)

        self.setStyleSheet("QGroupBox {background-color: rgb(25,25,25); }")

        self.rotateFrameWidget.buttonValidate.clicked.connect(self.addFrameWithRotation)

    def moveFrame(self,value):

        if self.currentPosition is not None:
            pos = self.currentPosition
            while pos >= 0 and pos < len(self.dictAllFrames) and self.dictAllFrames[pos] is not None:
                pos += value
            if pos >=0 and pos < len(self.dictAllFrames):
                self.dictAllFrames[pos] = [self.dictAllFrames[self.currentPosition][0],self.dictAllFrames[self.currentPosition][1],self.dictAllFrames[self.currentPosition][2]]
                self.dictAllFrames[self.currentPosition] = None
                self.currentPosition = pos

        self.updateAllFrames(self.dictAllFrames)

    def removeCurrentFrame(self):

        if self.currentPosition is not None:
            self.dictAllFrames[self.currentPosition] = None

        self.updateAllFrames(self.dictAllFrames)

    def getTimeSlider(self):

        posX = self.sliderPlay.groupBoxMiddle.pos().x() + self.sliderPlay.pos().x()

        currentTime = 0

        for n in range(len(self.allFramesList)):
            w = self.allFramesList[n]

            if n == len(self.allFramesList)-1:
                width = w.size().width()
            else:
                width=self.allFramesList[n+1].x()-self.allFramesList[n].x()

            if posX >= w.x() and posX < w.x() + width:
                index = n
                ratio = (posX-w.x())/width
                currentTime = (index+ratio)*vrb.movieMakerFrameTime

        return currentTime

    def timeToPosition(self,t):

        index = int((t/vrb.movieMakerFrameTime)+0.000001)

        rest = t-(index*vrb.movieMakerFrameTime)
        #get frame widget
        w = self.allFramesList[index]

        if index == len(self.allFramesList) - 1:
            width = w.size().width()
        else:
            width = self.allFramesList[index + 1].x() - self.allFramesList[index].x()

        # posX = w.pos().x() + (rest/vrb.movieMakerFrameTime)*w.size().width() - self.sliderPlay.pos().x()
        posX = w.pos().x() + (rest/vrb.movieMakerFrameTime)*width - self.sliderPlay.pos().x()

        posX += 0.000001

        return posX

    def playMovie(self):
        self.stopMovie = False

        splitSecond = 10
        startTime = self.getTimeSlider()

        maxTime = (self.getLenDict())*vrb.movieMakerFrameTime

        nbFrames = int(maxTime*splitSecond)+2

        # vrb.mainWindow.viewer3dSettingsWidget.loadingSettings = True

        for i in range(int(startTime*splitSecond),nbFrames):

            if self.stopMovie == False:
                t = i/splitSecond

                t = min(t,maxTime)

                posX = self.timeToPosition(t)

                self.sliderPlay.splitter.moveSplitter(posX,1)

                qt.QApplication.processEvents()

                # try:
                #     vrb.mainWindow.viewer3dSettingsWidget.loadSettings(vrb.mainWindow.currentLabel,update=False)
                # except:
                #     traceback.print_exc(file=sys.stderr)
                # time.sleep(1/splitSecond)
            else:
                break

        # vrb.mainWindow.viewer3dSettingsWidget.loadingSettings = False
        # vrb.mainWindow.viewer3dSettingsWidget.updateVisualization(currentLabel)
        self.stopMovie = True

    def pauseMovie(self):

        self.stopMovie = True

    def deleteAll(self):

        messageBox = wgt.MessageBox('Do you want to remove all the frames ?', '',buttons=[qt.QMessageBox.Yes, qt.QMessageBox.No], icon=qt.QMessageBox.Warning)
        res = messageBox.exec()
        if res == qt.QMessageBox.Yes:

            self.currentPosition = None
            self.currentWidget = None

            for i in self.dictAllFrames:
                self.dictAllFrames[i] = None

            self.updateAllFrames(self.dictAllFrames)

            self.sliderPlay.splitter.moveSplitter(0, 1)

    def saveScenario(self):

        try:

            filenameDialog = qt.QFileDialog.getSaveFileName(self, "Save scenario", vrb.folderMovieMaker, "txt (*.txt)")
            filename = filenameDialog[0]
            if filename != None and filename != '':

                textDict = fct.writeElement(self.dictAllFrames)

                file1 = open(filename, "w")
                file1.write(textDict)
                file1.close()

        except:
            traceback.print_exc(file=sys.stderr)

    def saveScenarioFolder(self):

        filenameDialog = qt.QFileDialog.getSaveFileName(self, "Save scenario", vrb.folderMovieMaker, "txt (*.txt)")
        filename = filenameDialog[0]
        if filename != None and filename != '':

            vrb.mainWindow.groupBoxProcessing.setText('Save scenario...\nPlease wait.')
            vrb.modeProcess = "SavingScenario"
            vrb.mainWindow.groupBoxProcessing.buttonStop.setVisible(True)
            vrb.mainWindow.groupBoxProcessing.progressBar.setVisible(True)
            vrb.mainWindow.toggleGroupBoxProcessing(True)

            try:
                basename = os.path.basename(filename)
                basename = basename.split('.')[0]
                folderFilename = vrb.folderMovieMaker + "/" + basename

                if not os.path.exists(folderFilename):
                    os.makedirs(folderFilename)

                textDict = fct.writeElement(self.dictAllFrames)

                file1 = open(folderFilename + "/scenario.txt", "w")
                file1.write(textDict)
                file1.close()

                listObjectsElement = xmlet.Element('ListObjects')
                vrb.breakScenario=False
                nbObjects = vrb.mainWindow.widgetLabelImage.layout.count()
                vrb.mainWindow.groupBoxProcessing.progressBar.setRange(0, nbObjects)
                for num in range(vrb.mainWindow.widgetLabelImage.layout.count()):
                    if vrb.breakScenario:
                        break
                    else:
                        vrb.mainWindow.groupBoxProcessing.progressBar.setValue(num+1)
                        vrb.mainWindow.groupBoxProcessing.setText('Save scenario ('+str(num+1)+'/'+str(nbObjects)+')...\nPlease wait.')
                        qt.QApplication.processEvents()
                    try:
                        item = vrb.mainWindow.widgetLabelImage.layout.itemAt(num)
                        if item is not None:
                            label = item.widget()
                            objectElement = Dfct.SubElement(listObjectsElement, "Object_"+str(num))
                            Dfct.SubElement(objectElement, 'Name').text = Dfct.convertTextToAscii(label.name)
                            Dfct.SubElement(objectElement, 'Name').set('ASCII', str(True))
                            Dfct.SubElement(objectElement, 'ElementID').text = Dfct.childText(label.xmlElement,"ElementID")
                            if label.objectType == "Image":
                                Dfct.SubElement(objectElement,"Type").text = "Image"
                                PyIPSDK.saveTiffImageFile(folderFilename + "/" + label.name + ".tif", label.image)
                            elif label.objectType == "Mesh":
                                Dfct.SubElement(objectElement,"Type").text = "Mesh"
                                Dfct.SubElement(objectElement,"Ratio").text = str(label.ratio)
                                Dfct.SubElement(objectElement,"MinFaces").text = str(label.minFaces)
                                PyIPSDK.writeToBinaryFile(folderFilename + "/" + label.name + ".bin", label.mesh)
                    except:
                        traceback.print_exc(file=sys.stderr)
                        # pass

                Dfct.saveXmlElement(listObjectsElement,folderFilename + "/listObjects.mho")

            except:
                traceback.print_exc(file=sys.stderr)

        vrb.mainWindow.toggleGroupBoxProcessing(False)
        vrb.mainWindow.groupBoxProcessing.setText('')
        vrb.mainWindow.groupBoxProcessing.buttonStop.setVisible(False)
        vrb.mainWindow.groupBoxProcessing.progressBar.setVisible(False)
        vrb.modeProcess = "ProcessingImages"

    def loadScenario(self):

        vrb.mainWindow.groupBoxProcessing.setText('Loading scenario...\nPlease wait.')
        vrb.mainWindow.groupBoxProcessing.buttonStop.setVisible(False)
        vrb.mainWindow.groupBoxProcessing.progressBar.setVisible(False)
        vrb.mainWindow.toggleGroupBoxProcessing(True)

        try:

            filename = qt.QFileDialog.getOpenFileNames(self, "Select your scenario", vrb.folderMovieMaker,"txt (*.txt)")
            if filename[0] != [] and filename[0] != '' and filename[0] != None:

                folder = os.path.dirname(filename[0][0])
                if os.path.exists(folder + "/listObjects.mho"):
                    self.loadScenarioFolder(folder)

                file = open(filename[0][0], "r")
                textDict = file.read()
                self.dictAllFrames = fct.readElement(textDict)
                for i in self.dictAllFrames:
                    if self.dictAllFrames[i] is not None:
                        eulerAngles = vrb.EulerAngles()
                        eulerAngles.angles = self.dictAllFrames[i][2]["Camera"]["angles"]
                        self.dictAllFrames[i][2]["Camera"]["angles"] = eulerAngles

                self.updateAllFrames(self.dictAllFrames)

        except:
            traceback.print_exc(file=sys.stderr)

        vrb.mainWindow.toggleGroupBoxProcessing(False)
        vrb.mainWindow.groupBoxProcessing.setText('')

    def loadScenarioFolder(self,folder):

        file = xmlet.parse(folder + "/listObjects.mho")
        listObjectsElement = file.getroot()
        for child in listObjectsElement:
            if Dfct.childText(child,"Type") == "Image":
                nameImage = Dfct.childText(child, 'Name')
                label,image = vrb.mainWindow.urlToImage(folder + "/" + nameImage + ".tif")
                Dfct.SubElement(label.xmlElement, "ElementID").text = Dfct.childText(child, "ElementID")
                label.addToNapari()
                label.addOrthosliceToNapari()

            elif Dfct.childText(child,"Type") == "Mesh":
                nameImage = Dfct.childText(child, 'Name')

                label,objectValue = vrb.mainWindow.urlToObject(folder + "/" + nameImage + ".bin")
                Dfct.SubElement(label.xmlElement, "ElementID").text = Dfct.childText(child, "ElementID")
                label.addToNapari()
                try:
                    ratio = float(Dfct.childText(child, 'Ratio'))
                    minFaces = int(float(Dfct.childText(child, 'MinFaces')))

                    if ratio != 0 or minFaces != 1000:
                        vertices, faces, values, nbFaces = fct.createSurfaceForNapari(label.mesh, ratio=ratio,minFaces=minFaces)
                        label.surface = (vertices, faces, values)
                        label.ratio = ratio
                        label.minFaces = minFaces
                        label.layer.data = label.surface
                        label.nbFacesSimplify = len(faces)
                except:
                    traceback.print_exc(file=sys.stderr)

        vrb.mainWindow.viewer3dSettingsWidget.loadSettings()

    def openSettings(self):

        self.settingsMovieMaker.show()
        self.settingsMovieMaker.readXmlElement()

    def saveMovieGIF(self, filename, viewerNapari, logo, fps, startTime, splitSecond, maxTime):
        """
        Saves the movis in a GIF image
        """
        imgs = []
        for i in range(int(startTime * splitSecond), int(maxTime * splitSecond) + 1):
            t = i / splitSecond
            posX = self.timeToPosition(t)
            self.sliderPlay.splitter.moveSplitter(posX, 1)
            qt.QApplication.processEvents()

            frame = viewerNapari.screenshot(flash=False)
            if self.settingsMovieMaker.checkBoxLogo.isChecked():
                self.addLogoImage(frame, logo)
            b, g, r, a = cv2.split(frame)
            frameRGB = cv2.merge([r, g, b])
            imgs.append(PILImage.fromarray(frameRGB))
            time.sleep(0.05)

        # https://pillow.readthedocs.io/en/stable/handbook/image-file-formats.html#gif
        imgs[0].save(fp=filename, format="GIF", save_all=True, append_images=imgs, duration=1000.0*maxTime/fps, optimize=False, loop=0)


    def saveMovieGeneric(self, filename, viewerNapari, logo, fps, startTime, splitSecond, maxTime):
        """
        Saves the movis in an avi, mp4, mov or mkv file
        """
        first_image = viewerNapari.screenshot(flash=False)
        h, w, _ = first_image.shape
        fourcc = cv2.VideoWriter_fourcc(*'mp4v')
        vid_writer = cv2.VideoWriter(filename, fourcc, fps, (w, h))
        for i in range(int(startTime * splitSecond), int(maxTime * splitSecond) + 1):

            t = i / splitSecond
            posX = self.timeToPosition(t)
            self.sliderPlay.splitter.moveSplitter(posX, 1)
            qt.QApplication.processEvents()

            frame = viewerNapari.screenshot(flash=False)
            if self.settingsMovieMaker.checkBoxLogo.isChecked():
                self.addLogoImage(frame, logo)

            b, g, r, a = cv2.split(frame)
            frameRGB = cv2.merge([r, g, b])
            vid_writer.write(frameRGB)

            time.sleep(0.05)

        vid_writer.release()

    def saveMovie(self):

        filenameDialog = qt.QFileDialog.getSaveFileName(self, "Save movie as", "","avi (*.avi);;mp4 (*.mp4);;gif (*.gif);;mov (*.mov);;mkv (*.mkv)")
        filename = filenameDialog[0]
        if filename != None and filename != '':

            file_extension = filenameDialog[1].split(" ")[0]
            if filename != None and filename != '':
                if not filename.endswith(file_extension):
                    filename += "." + file_extension

            if self.settingsMovieMaker.checkBoxLogo.isChecked():
                if self.settingsMovieMaker.lineEditLogoPath.text() == "":
                    logo = PILImage.open(self.settingsMovieMaker.lineEditLogoPath.placeholderText())
                else:
                    logo = PILImage.open(self.settingsMovieMaker.lineEditLogoPath.text())
                logo = logo.convert('RGBA')
            else:
                logo = None

            self.stopMovie = False

            splitSecond = 20
            startTime = 0
            self.sliderPlay.splitter.moveSplitter(startTime, 1)

            viewerNapari = vrb.mainWindow.viewerNapariQt
            maxTime = (self.getLenDict()) * vrb.movieMakerFrameTime

            fps = 20
            if file_extension == "gif":
                self.saveMovieGIF(filename, viewerNapari, logo, fps, startTime, splitSecond, maxTime)
            else:
                self.saveMovieGeneric(filename, viewerNapari, logo, fps, startTime, splitSecond, maxTime)


    def addLogoImage(self,image,imageLogo):

        logoNp = np.array(imageLogo)
        shapeLogo = logoNp.shape
        shape = image.shape

        ratio = self.settingsMovieMaker.spinBoxSizeLogo.value()/100
        resizeFactor = (min(shape[0], shape[1]) / (max(logoNp.shape[0], logoNp.shape[1]))) * ratio
        logoNp = cv2.resize(logoNp, dsize=(int(shapeLogo[1] * resizeFactor), int(shapeLogo[0] * resizeFactor)), interpolation=cv2.INTER_AREA)

        shapeLogo = logoNp.shape
        if self.settingsMovieMaker.comboBoxPosition.currentIndex() == 0:
            cropImage = image[int(shape[0] * 0.95 - shapeLogo[0]):int(shape[0] * 0.95), int(shape[1] * 0.95 - shapeLogo[1]):int(shape[1] * 0.95), :]
        elif self.settingsMovieMaker.comboBoxPosition.currentIndex() == 1:
            cropImage = image[int(shape[0] * 0.95 - shapeLogo[0]):int(shape[0] * 0.95), int(shape[1] * 0.05):int(shape[1] * 0.05 + shapeLogo[1]), :]
        elif self.settingsMovieMaker.comboBoxPosition.currentIndex() == 2:
            cropImage = image[int(shape[0] * 0.05):int(shape[0] * 0.05 + shapeLogo[0]), int(shape[1] * 0.05):int(shape[1] * 0.05 + shapeLogo[1]), :]
        elif self.settingsMovieMaker.comboBoxPosition.currentIndex() == 3:
            cropImage = image[int(shape[0] * 0.05):int(shape[0] * 0.05 + shapeLogo[0]), int(shape[1] * 0.95 - shapeLogo[1]):int(shape[1] * 0.95), :]

        mergedImage = np.where(cropImage[:, :] == [0, 0, 0, 255], logoNp, cropImage)
        cropImage[:, :, :] = mergedImage

    def updateMovie(self):
        currentTime = self.getTimeSlider()
        self.getDictInterpolated(currentTime)

    def getLenDict(self):

        lenDict = 0
        for i in range(len(self.dictAllFrames)):
            if self.dictAllFrames[i] is not None:
                lenDict = i

        return lenDict

    def adjustDict(self,inputDict):

        outputDict = {}
        for i in range(len(inputDict)):
            outputDict[i] = None
        firstIndexNotEmpty = None
        for i in range(len(inputDict)):
            if inputDict[i] is not None:
                firstIndexNotEmpty = i
                break
        if firstIndexNotEmpty is not None:
            for i in range(len(inputDict)):
                if i == 0:
                    outputDict[0] = ["0 s" , inputDict[firstIndexNotEmpty][1],inputDict[firstIndexNotEmpty][2]]
                elif i > firstIndexNotEmpty and inputDict[i] is not None:
                    outputDict[i] = [str(int(i * vrb.movieMakerFrameTime * 10) / 10) + " s",inputDict[i][1],inputDict[i][2]]

        if self.currentPosition == firstIndexNotEmpty:
            self.currentPosition = 0

        return outputDict

    def addFrame(self):

        try:
            viewerNapari = vrb.mainWindow.viewerNapariQt
            # imageNp = viewerNapari.screenshot(canvas_only=True,flash=False)
            imageNp = viewerNapari.screenshot(flash=False)
            image = fct.numpyImageToIPSDKImage(imageNp,invertChannels=False)
            outImage = PyIPSDK.createImageRgb(image.getBufferType(),max(image.getSizeX(),image.getSizeY()),max(image.getSizeX(),image.getSizeY()))
            util.eraseImg(outImage,0)
            util.putROI2dImg(outImage, image, int((max(image.getSizeX(),image.getSizeY())-image.getSizeX())/2), int((max(image.getSizeX(),image.getSizeY())-image.getSizeY())/2), outImage)

            pixmapImage = fct.IPSDKImageToPixmap(outImage)

            dictInfo = self.createDictInfo()

            # state = self.saveSceneState()
            # print('state :',state)

            if self.dictAllFrames[0] is None:
                self.dictAllFrames[0] = ["0 s", pixmapImage,dictInfo]
                self.currentPosition = 0
            else:
                lenDict = self.getLenDict()
                self.dictAllFrames[lenDict+10] = [str(int(((lenDict+10)*vrb.movieMakerFrameTime)*10)/10) + " s",pixmapImage,dictInfo]
                self.currentPosition = lenDict+10

            self.updateAllFrames(self.dictAllFrames)

        except:
            traceback.print_exc(file=sys.stderr)

    # def saveSceneState(self):
    #
    #     # viewer = vrb.mainWindow.viewerNapariQt
    #     viewer = vrb.mainWindow.viewerNapari
    #
    #     state = []
    #     for layer in viewer.layers:
    #         # Utiliser __dict__ pour accéder à tous les attributs de la couche
    #         layer_state = {
    #             key: copy.deepcopy(value)
    #             for key, value in layer.__dict__.items()
    #             if key not in ['data', 'events','_data','_data_raw']  # Exclure les attributs spécifiques indésirables
    #         }
    #
    #         # Ajouter le type de couche pour pouvoir la recréer plus tard
    #         layer_state['type'] = type(layer).__name__
    #
    #         state.append(layer_state)
    #
    #     # print('iciii',state)
    #
    #     return state

    # def saveSceneState(self):
    #
    #     viewer = vrb.mainWindow.viewerNapari
    #
    #     state = []
    #     for layer in viewer.layers:
    #         # Utiliser __dict__ pour accéder à tous les attributs de la couche
    #         layer_state = {
    #             key: copy.deepcopy(value)
    #             for key, value in layer.__dict__.items()
    #             if key not in ['_data', '_data_raw', 'data', 'events']  # Exclure les attributs indésirables
    #         }
    #
    #         # Ajouter le type de couche pour pouvoir la recréer plus tard
    #         layer_state['type'] = type(layer).__name__
    #         state.append(layer_state)
    #
    #     # Sauvegarder aussi l'état de la caméra
    #     camera_state = {
    #         'center': viewer.camera.center,
    #         'zoom': viewer.camera.zoom,
    #         'angles': viewer.camera.angles if hasattr(viewer.camera, 'angles') else None,
    #         'perspective': viewer.camera.perspective
    #     }
    #
    #     return [state, camera_state]

    # def applyStateToViewer(self,state):
    #
    #     viewer = vrb.mainWindow.viewerNapari
    #
    #
    #     if len(viewer.layers) != len(state):
    #         raise ValueError("Le nombre de couches dans le viewer ne correspond pas à l'état sauvegardé.")
    #
    #     for layer, layer_state in zip(viewer.layers, state):
    #         for key, value in layer_state.items():
    #             if key == 'type':  # Ignorer le type de couche, car ce n'est pas un attribut modifiable
    #                 continue
    #
    #             # Gestion des attributs commençant par _
    #             public_key = key.lstrip('_')  # Retirer le _ initial pour obtenir la version publique potentielle
    #
    #             # Vérifier si une propriété publique existe pour cet attribut
    #             if hasattr(layer, public_key):
    #                 try:
    #                     setattr(layer, public_key, value)  # Utiliser la propriété publique
    #                 except AttributeError:
    #                     print(f"Le paramètre {public_key} ne peut pas être appliqué à la couche {layer.name}")
    #             else:
    #                 # Sinon, essayer d'utiliser setattr sur l'attribut original
    #                 try:
    #                     setattr(layer, key, value)
    #                 except AttributeError:
    #                     print(f"Le paramètre {key} ne peut pas être appliqué à la couche {layer.name}")
    #
    # def applyStateToViewer(self,stateList):
    #
    #     viewer = vrb.mainWindow.viewerNapari
    #
    #     state = stateList[0]
    #     camera_state = stateList[1]
    #
    #     if len(viewer.layers) != len(state):
    #         raise ValueError("Le nombre de couches dans le viewer ne correspond pas à l'état sauvegardé.")
    #
    #     for layer, layer_state in zip(viewer.layers, state):
    #         for key, value in layer_state.items():
    #             if key == 'type':  # Ignorer le type de couche, car ce n'est pas un attribut modifiable
    #                 continue
    #
    #             # Gestion des attributs commençant par _
    #             public_key = key.lstrip('_')  # Retirer le _ initial pour obtenir la version publique potentielle
    #             applied = False
    #
    #             # Vérifier si une propriété publique existe pour cet attribut
    #             if hasattr(layer, public_key):
    #                 try:
    #                     setattr(layer, public_key, value)  # Utiliser la propriété publique
    #                     print(f"Attribut appliqué : {public_key} = {value}")
    #                     applied = True
    #                 except AttributeError:
    #                     print(f"Le paramètre {public_key} ne peut pas être appliqué à la couche {layer.name}")
    #             else:
    #                 # Sinon, essayer d'utiliser setattr sur l'attribut original
    #                 try:
    #                     setattr(layer, key, value)
    #                     print(f"Attribut appliqué : {key} = {value}")
    #                     applied = True
    #                 except AttributeError:
    #                     print(f"Le paramètre {key} ne peut pas être appliqué à la couche {layer.name}")
    #
    #             if not applied:
    #                 print(f"Attribut ignoré : {key}")
    #
    #     # Appliquer l'état de la caméra
    #     viewer.camera.center = camera_state['center']
    #     viewer.camera.zoom = camera_state['zoom']
    #     if camera_state['angles'] is not None:
    #         viewer.camera.angles = camera_state['angles']
    #     viewer.camera.perspective = camera_state['perspective']
    #
    #     # Imprimer les attributs de la caméra appliqués
    #     print("Attributs de la caméra appliqués :")
    #     print(f"  center = {camera_state['center']}")
    #     print(f"  zoom = {camera_state['zoom']}")
    #     if camera_state['angles'] is not None:
    #         print(f"  angles = {camera_state['angles']}")
    #     print(f"  perspective = {camera_state['perspective']}")

    # def applyStateToViewer(self, stateList):
    #     viewer = vrb.mainWindow.viewerNapari
    #
    #     state = stateList[0]
    #     camera_state = stateList[1]
    #
    #     if len(viewer.layers) != len(state):
    #         raise ValueError("Le nombre de couches dans le viewer ne correspond pas à l'état sauvegardé.")
    #
    #     for layer, layer_state in zip(viewer.layers, state):
    #         try:
    #             for key, value in layer_state.items():
    #                 if key == 'type' or key in ATTRIBUTES_TO_IGNORE:  # Ignorer le type de couche et les attributs complexes
    #                     print(f"Attribut ignoré car complexe ou non modifiable directement : {key}")
    #                     continue
    #
    #                 # Gestion des attributs commençant par _
    #                 public_key = key.lstrip('_')  # Retirer le _ initial pour obtenir la version publique potentielle
    #                 applied = False
    #
    #                 # Vérifier si une propriété publique existe pour cet attribut
    #                 if hasattr(layer, public_key):
    #                     try:
    #                         # Appliquer la propriété publique
    #                         setattr(layer, public_key, value)
    #                         print(f"Attribut appliqué : {public_key} = {value}")
    #                         applied = True
    #                     except Exception as e:
    #                         print(f"Erreur en appliquant {public_key}, with value : {value} à la couche {layer.name}: {e}")
    #                 else:
    #                     # Sinon, essayer d'utiliser setattr sur l'attribut original
    #                     try:
    #                         setattr(layer, key, value)
    #                         print(f"Attribut appliqué : {key} = {value}")
    #                         applied = True
    #                     except Exception as e:
    #                         print(f"Erreur _ en appliquant {key}, with value : {value} à la couche {layer.name}: {e}")
    #
    #                 if not applied:
    #                     print(f"Attribut ignoré : {key}")
    #
    #         except Exception as e:
    #             print(f"Erreur générale en appliquant l'état à la couche {layer.name}: {e}")
    #
    #     # Appliquer l'état de la caméra
    #     try:
    #         viewer.camera.center = camera_state['center']
    #         viewer.camera.zoom = camera_state['zoom']
    #         if camera_state['angles'] is not None:
    #             viewer.camera.angles = camera_state['angles']
    #         viewer.camera.perspective = camera_state['perspective']
    #
    #         # Imprimer les attributs de la caméra appliqués
    #         print("Attributs de la caméra appliqués :")
    #         print(f"  center = {camera_state['center']}")
    #         print(f"  zoom = {camera_state['zoom']}")
    #         if camera_state['angles'] is not None:
    #             print(f"  angles = {camera_state['angles']}")
    #         print(f"  perspective = {camera_state['perspective']}")
    #
    #     except Exception as e:
    #         print(f"Erreur en appliquant l'état de la caméra : {e}")
    #
    # def interpolateState(self,stateList1, stateList2, t):
    #     """
    #     Interpole entre deux états de viewer (couches et caméra).
    #
    #     :param stateList1: Le premier état [state_layers, camera_state].
    #     :param stateList2: Le second état [state_layers, camera_state].
    #     :param t: Le facteur d'interpolation (entre 0 et 1).
    #     :return: L'état interpolé [state_layers_interpolated, camera_state_interpolated].
    #     """
    #
    #     state1, camera_state1 = stateList1
    #     state2, camera_state2 = stateList2
    #
    #     # Interpolation des couches
    #     interpolated_layers = []
    #     for layer1, layer2 in zip(state1, state2):
    #         interpolated_layer = {}
    #         for key in layer1:
    #             value1 = layer1[key]
    #             value2 = layer2[key]
    #
    #             # Si les deux valeurs sont des nombres ou des vecteurs, interpole
    #             if isinstance(value1, (int, float)) and isinstance(value2, (int, float)):
    #                 interpolated_layer[key] = (1 - t) * value1 + t * value2
    #             elif isinstance(value1, (list, tuple)) and isinstance(value2, (list, tuple)):
    #                 if len(value1) == len(value2):
    #                     interpolated_layer[key] = [(1 - t) * v1 + t * v2 for v1, v2 in zip(value1, value2)]
    #                 else:
    #                     interpolated_layer[key] = value1  # Conserver inchangé si les longueurs diffèrent
    #             else:
    #                 # Si ce n'est pas interpolable, conserver la valeur de state1
    #                 interpolated_layer[key] = value1
    #
    #         interpolated_layers.append(interpolated_layer)
    #
    #     # Interpolation de l'état de la caméra
    #     interpolated_camera_state = {}
    #     for key in camera_state1:
    #         value1 = camera_state1[key]
    #         value2 = camera_state2[key]
    #
    #         if isinstance(value1, (int, float)) and isinstance(value2, (int, float)):
    #             interpolated_camera_state[key] = (1 - t) * value1 + t * value2
    #         elif isinstance(value1, (list, tuple)) and isinstance(value2, (list, tuple)):
    #             if len(value1) == len(value2):
    #                 interpolated_camera_state[key] = [(1 - t) * v1 + t * v2 for v1, v2 in zip(value1, value2)]
    #             else:
    #                 interpolated_camera_state[key] = value1  # Conserver inchangé si les longueurs diffèrent
    #         else:
    #             # Si ce n'est pas interpolable, conserver la valeur de camera_state1
    #             interpolated_camera_state[key] = value1
    #
    #     return [interpolated_layers, interpolated_camera_state]

    def rotateFrame(self):

        self.frameWithUpdatedRotation = None
        fct.showWidget(self.rotateFrameWidget)

    def addFrameWithRotation(self):

        try:
            axis = self.rotateFrameWidget.comboBoxAxis.currentText()
            angle = self.rotateFrameWidget.spinBoxAngle.value()

            self.rotateFrameWidget.close()

            if self.frameWithUpdatedRotation is not None:

                dictInfo = self.dictAllFrames[self.frameWithUpdatedRotation.position][2]

                if axis == "X":
                    dictInfo["Camera"]["angles"].direction = [None,None,angle]
                if axis == "Y":
                    dictInfo["Camera"]["angles"].direction = [None,angle,None]
                if axis == "Z":
                    dictInfo["Camera"]["angles"].direction = [angle,None,None]

            else:

                viewerNapari = vrb.mainWindow.viewerNapariQt
                # imageNp = viewerNapari.screenshot(canvas_only=True,flash=False)
                imageNp = viewerNapari.screenshot(flash=False)
                image = fct.numpyImageToIPSDKImage(imageNp, invertChannels=False)
                outImage = PyIPSDK.createImageRgb(image.getBufferType(), max(image.getSizeX(), image.getSizeY()),max(image.getSizeX(), image.getSizeY()))
                util.eraseImg(outImage, 0)
                util.putROI2dImg(outImage, image, int((max(image.getSizeX(), image.getSizeY()) - image.getSizeX()) / 2),
                                 int((max(image.getSizeX(), image.getSizeY()) - image.getSizeY()) / 2), outImage)

                pixmapImage = fct.IPSDKImageToPixmap(outImage)

                dictInfo = self.createDictInfo()

                if axis == "X":
                    dictInfo["Camera"]["angles"].angles = (dictInfo["Camera"]["angles"].angles[0],dictInfo["Camera"]["angles"].angles[1],dictInfo["Camera"]["angles"].angles[2]+angle)
                    dictInfo["Camera"]["angles"].direction = [None,None,angle]
                if axis == "Y":
                    dictInfo["Camera"]["angles"].angles = (dictInfo["Camera"]["angles"].angles[0],dictInfo["Camera"]["angles"].angles[1]+angle,dictInfo["Camera"]["angles"].angles[2])
                    dictInfo["Camera"]["angles"].direction = [None,angle,None]
                if axis == "Z":
                    dictInfo["Camera"]["angles"].angles = (dictInfo["Camera"]["angles"].angles[0]+angle,dictInfo["Camera"]["angles"].angles[1],dictInfo["Camera"]["angles"].angles[2])
                    dictInfo["Camera"]["angles"].direction = [angle,None,None]

                if self.dictAllFrames[0] is None:
                    self.dictAllFrames[0] = ["0 s", pixmapImage, dictInfo]
                    self.currentPosition = 0
                else:
                    lenDict = self.getLenDict()
                    self.dictAllFrames[lenDict + 10] = [str(int(((lenDict + 10) * vrb.movieMakerFrameTime) * 10) / 10) + " s",pixmapImage, dictInfo]
                    self.currentPosition = lenDict + 10

            self.updateAllFrames(self.dictAllFrames)

            frame = self.allFramesList[self.currentPosition]

            self.sliderPlay.splitter.moveSplitter(frame.pos().x() - self.sliderPlay.pos().x(), 1)
            self.updateMovie()
            qt.QApplication.processEvents()
            frame.updateFrameContent(pixmapOnly=True)

            self.frameWithUpdatedRotation = None

        except:
            traceback.print_exc(file=sys.stderr)
            self.frameWithUpdatedRotation = None

        # self.currentFrame.parent.dictAllFrames[self.currentFrame.position][1] = None
        # self.currentFrame.updateFrame(self.currentFrame.parent.dictAllFrames[self.currentFrame.position])
        #
        # self.close()
        #
        # print(axis,angle)

    def createDictInfo(self):
        viewer = vrb.mainWindow.viewerNapari

        dictInfo = {}
        dictInfo["Layers"] = {}
        dictInfo["Camera"] = {}
        dictInfo["Viewer"] = {}
        dictInfo["Viewer"]["step"] = vrb.mainWindow.viewerNapari.dims.current_step[0]

        for layer in viewer.layers:
            try:
                idElement = layer.name
                id = idElement.split("_")[0]

                # info = idElement.split("_")[1]
                label = vrb.dictElements[id][3]

                dictInfo["Layers"][idElement] = {}
                dictInfo["Layers"][idElement]["visible"] = layer.visible
                if isinstance(layer, Surface):
                    dictInfo["Layers"][idElement]["objectType"] = "Mesh"
                    dictInfo["Layers"][idElement]["blending"] = layer.blending
                    dictInfo["Layers"][idElement]["opacity"] = layer.opacity
                    dictInfo["Layers"][idElement]["shading"] = layer.shading

                    dictInfo["Layers"][idElement]["wireframeVisible"] = layer.wireframe.visible
                    dictInfo["Layers"][idElement]["wireframeRed"] = layer.wireframe.color[0]
                    dictInfo["Layers"][idElement]["wireframeGreen"] = layer.wireframe.color[1]
                    dictInfo["Layers"][idElement]["wireframeBlue"] = layer.wireframe.color[2]
                    dictInfo["Layers"][idElement]["colormap"] = layer.colormap.colors
                elif isinstance(layer, Image) or isinstance(layer, Labels):

                    if label.image.getBufferType() not in [PyIPSDK.eIBT_Label8, PyIPSDK.eIBT_Label16,
                                                           PyIPSDK.eIBT_Label32]:
                        dictInfo["Layers"][idElement]["objectType"] = "Image"
                        dictInfo["Layers"][idElement]["gamma"] = layer.gamma
                        dictInfo["Layers"][idElement]["attenuation"] = layer.attenuation
                        dictInfo["Layers"][idElement]["iso_threshold"] = layer.iso_threshold
                    else:
                        dictInfo["Layers"][idElement]["objectType"] = "ImageLabel"

                    dictInfo["Layers"][idElement]["rendering"] = layer.rendering
                    dictInfo["Layers"][idElement]["blending"] = layer.blending
                    dictInfo["Layers"][idElement]["opacity"] = layer.opacity
                    if label.image.getBufferType() not in [PyIPSDK.eIBT_Label8, PyIPSDK.eIBT_Label16,
                                                           PyIPSDK.eIBT_Label32]:
                        dictInfo["Layers"][idElement]["colormap"] = layer.colormap.colors
                    else:
                        dictInfo["Layers"][idElement]["colormap"] = None
                    if label.image.getBufferType() not in [PyIPSDK.eIBT_Label8, PyIPSDK.eIBT_Label16,
                                                           PyIPSDK.eIBT_Label32]:
                        dictInfo["Layers"][idElement]["contrast_limits"] = layer.contrast_limits
                    else:
                        dictInfo["Layers"][idElement]["contrast_limits"] = None
                else:
                    continue

                experimental_clipping_planes = []
                for i in range(6):
                    if len(layer.experimental_clipping_planes) == 0:
                        experimental_clipping_planes = fct.createDefaultExperimentalClippingPlanes(label)
                    else:
                        experimental_clipping_planes.append(
                            {"position": layer.experimental_clipping_planes[i].position,
                             "normal": layer.experimental_clipping_planes[i].normal,
                             "enabled": layer.experimental_clipping_planes[i].enabled})
                dictInfo["Layers"][idElement]["experimental_clipping_planes"] = experimental_clipping_planes

            except:
                # pass
                traceback.print_exc(file=sys.stderr)

        dictInfo["Camera"]["zoom"] = vrb.mainWindow.viewerNapari.camera.zoom
        dictInfo["Camera"]["center"] = vrb.mainWindow.viewerNapari.camera.center
        angles = vrb.EulerAngles()
        angles.angles = vrb.mainWindow.viewerNapari.camera.angles
        dictInfo["Camera"]["angles"] = angles

        dictInfo["Camera"]["perspective"] = vrb.mainWindow.viewerNapari.camera.perspective
        dictInfo["Camera"]["interactive"] = vrb.mainWindow.viewerNapari.camera.interactive

        try:
            if vrb.mainWindow.viewer3dSettingsWidget.viewerSettings.radioButtonCameraFly.isChecked():
                rotation1 = vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera.rotation1
                rotation2 = vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera.rotation2

                dictInfo["Camera"]["rotation1"] = [self.correctQuaternionValue(rotation1.w),
                                                   self.correctQuaternionValue(rotation1.x),
                                                   self.correctQuaternionValue(rotation1.y),
                                                   self.correctQuaternionValue(rotation1.z, offset=-0.5)]

                dictInfo["Camera"]["rotation2"] = [rotation2.w,rotation2.x,rotation2.y,rotation2.z]
        except:
            # pass
            traceback.print_exc(file=sys.stderr)

        if vrb.mainWindow.viewerNapariQt.canvas.camera._3D_camera == vrb.mainWindow.viewerNapariQt.canvas.camera._3D_flyCamera:
            dictInfo["Camera"]["mode"] = "Fly"
        else:
            dictInfo["Camera"]["mode"] = "Turn"

        return dictInfo

    def correctQuaternionValue(self,value,offset=0.5):

        if abs(-value - offset) < abs(value - offset):
            value = -value

        return value

    def getDictInterpolated(self,currentTime):
        indexBefore = None
        indexAfter = None
        for i in range(len(self.dictAllFrames)):
            if self.dictAllFrames[i] is not None:
                if i*vrb.movieMakerFrameTime<currentTime:
                    indexBefore = i
                elif i*vrb.movieMakerFrameTime>=currentTime:
                    indexAfter = i
                    break

        if indexBefore is None and indexAfter is not None:
            self.applyDictToNapari(self.dictAllFrames[indexAfter][2])

        elif indexBefore is not None and indexAfter is not None:

            ratio = (currentTime-indexBefore*vrb.movieMakerFrameTime)/(indexAfter*vrb.movieMakerFrameTime-indexBefore*vrb.movieMakerFrameTime)
            if self.dictAllFrames[indexBefore][2]["Camera"]["mode"] != self.dictAllFrames[indexAfter][2]["Camera"]["mode"] and ratio !=0:
                dictInterpolated = fct.blendValues(self.dictAllFrames[indexBefore][2],self.dictAllFrames[indexAfter][2], 1)
            else:
                dictInterpolated = fct.blendValues(self.dictAllFrames[indexBefore][2],self.dictAllFrames[indexAfter][2],ratio)

            self.applyDictToNapari(dictInterpolated)

            if ratio == 1 and (self.dictAllFrames[indexAfter][2]["Camera"]["angles"].direction[0] is not None or\
                self.dictAllFrames[indexAfter][2]["Camera"]["angles"].direction[1] is not None or\
                self.dictAllFrames[indexAfter][2]["Camera"]["angles"].direction[2] is not None) and \
                (self.dictAllFrames[indexAfter][2]["Camera"]["angles"].angles[0] != dictInterpolated["Camera"]["angles"].angles[0] or\
                 self.dictAllFrames[indexAfter][2]["Camera"]["angles"].angles[1] != dictInterpolated["Camera"]["angles"].angles[1] or\
                 self.dictAllFrames[indexAfter][2]["Camera"]["angles"].angles[2] != dictInterpolated["Camera"]["angles"].angles[2]):

                self.dictAllFrames[indexAfter][2]["Camera"]["angles"].angles = (dictInterpolated["Camera"]["angles"].angles[0],
                                                                                dictInterpolated["Camera"]["angles"].angles[1],dictInterpolated["Camera"]["angles"].angles[2])
                frame = self.allFramesList[indexAfter]
                frame.updateFrameContent(pixmapOnly=True)

        elif indexBefore is not None and indexAfter is None:
            self.applyDictToNapari(self.dictAllFrames[indexBefore][2])

    def applyDictToNapari(self,dictInfo):
        viewer = vrb.mainWindow.viewerNapari

        idsList = []
        for num in range(vrb.mainWindow.widgetLabelImage.layout.count()):
            try:
                item = vrb.mainWindow.widgetLabelImage.layout.itemAt(num)
                if item is not None:
                    label = item.widget()
                    idElement = Dfct.childText(label.xmlElement, "ElementID")
                    idsList.append(idElement)
            except:
                traceback.print_exc(file=sys.stderr)
                # pass

        vrb.mainWindow.viewer3dSettingsWidget.loadingCamera = True

        try:
            if dictInfo["Camera"]["mode"] == "Turn" and vrb.mainWindow.viewerNapariQt.canvas.camera._3D_camera != vrb.mainWindow.viewerNapariQt.canvas.camera._3D_turnCamera:
                vrb.mainWindow.viewerNapariQt.canvas.camera._3D_camera = vrb.mainWindow.viewerNapariQt.canvas.camera._3D_turnCamera
                vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera = vrb.mainWindow.viewerNapariQt.canvas.camera._3D_turnCamera
                vrb.mainWindow.viewer3dSettingsWidget.viewerSettings.radioButtonCameraNormal.setChecked(True)
            if dictInfo["Camera"]["mode"] == "Fly" and vrb.mainWindow.viewerNapariQt.canvas.camera._3D_camera != vrb.mainWindow.viewerNapariQt.canvas.camera._3D_flyCamera:
                vrb.mainWindow.viewerNapariQt.canvas.camera._3D_camera = vrb.mainWindow.viewerNapariQt.canvas.camera._3D_flyCamera
                vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera = vrb.mainWindow.viewerNapariQt.canvas.camera._3D_flyCamera
                vrb.mainWindow.viewer3dSettingsWidget.viewerSettings.radioButtonCameraFly.setChecked(True)

            if vrb.mainWindow.viewerNapari.camera.zoom != dictInfo["Camera"]["zoom"]:
                vrb.mainWindow.viewerNapari.camera.zoom = dictInfo["Camera"]["zoom"]
            if vrb.mainWindow.viewerNapari.camera.center != dictInfo["Camera"]["center"]:
                vrb.mainWindow.viewerNapari.camera.center = dictInfo["Camera"]["center"]
            if vrb.mainWindow.viewerNapari.camera.angles != dictInfo["Camera"]["angles"].angles:
                vrb.mainWindow.viewerNapari.camera.angles = dictInfo["Camera"]["angles"].angles
            if vrb.mainWindow.viewerNapari.camera.perspective != dictInfo["Camera"]["perspective"]:
                vrb.mainWindow.viewerNapari.camera.perspective = dictInfo["Camera"]["perspective"]
                vrb.mainWindow.viewer3dSettingsWidget.viewerSettings.sliderFov.slider.setValue(dictInfo["Camera"]["perspective"])

            if vrb.mainWindow.viewerNapari.camera.interactive != dictInfo["Camera"]["interactive"]:
                vrb.mainWindow.viewerNapari.camera.interactive = dictInfo["Camera"]["interactive"]
            if "rotation1" in dictInfo["Camera"]:
                rotation1 = vispy.util.quaternion.Quaternion(dictInfo["Camera"]["rotation1"][0],dictInfo["Camera"]["rotation1"][1],
                                                              dictInfo["Camera"]["rotation1"][2],dictInfo["Camera"]["rotation1"][3])
                if vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera == vrb.mainWindow.viewerNapariQt.canvas.camera._3D_flyCamera and rotation1 != vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera.rotation1:
                    vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera.rotation1 = rotation1
            if "rotation2" in dictInfo["Camera"]:
                rotation2 = vispy.util.quaternion.Quaternion(dictInfo["Camera"]["rotation2"][0],dictInfo["Camera"]["rotation2"][1],
                                                              dictInfo["Camera"]["rotation2"][2],dictInfo["Camera"]["rotation2"][3])
                if vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera == vrb.mainWindow.viewerNapariQt.canvas.camera._3D_flyCamera and rotation2 != vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera.rotation1:
                    vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera.rotation2 = rotation2
        except:
            traceback.print_exc(file=sys.stderr)

        # vrb.mainWindow.viewerNapariQt.camera._3D_camera.update()
        # vrb.mainWindow.viewerNapariQt.camera._view.camera.update()

        vrb.mainWindow.viewer3dSettingsWidget.loadingCamera = False
        
        if vrb.mainWindow.viewerNapari.dims.set_current_step != dictInfo["Viewer"]["step"]:
            vrb.mainWindow.viewerNapari.dims.set_current_step(0,dictInfo["Viewer"]["step"])

        for layer in viewer.layers:
            try:
                idElement = layer.name

                id = idElement.split("_")[0]
                try:
                    info = idElement.split("_")[1]
                except:
                    info = None
                label = vrb.dictElements[id][3]

                if idElement in dictInfo["Layers"]:

                    if layer.visible != dictInfo["Layers"][idElement]["visible"]:
                        layer.visible = dictInfo["Layers"][idElement]["visible"]
                        if info is not None and info in ["orthosliceX","orthosliceY","orthosliceZ"]:
                            label.buttonOrthoslices.setActivation(not dictInfo["Layers"][idElement]["visible"])
                        else:
                            label.buttonEye.setActivation(not dictInfo["Layers"][idElement]["visible"])

                    if isinstance(layer, Surface):

                        if layer.blending != dictInfo["Layers"][idElement]["blending"]:
                            layer.blending = dictInfo["Layers"][idElement]["blending"]
                        if layer.opacity != dictInfo["Layers"][idElement]["opacity"]:
                            layer.opacity = dictInfo["Layers"][idElement]["opacity"]
                        if layer.shading != dictInfo["Layers"][idElement]["shading"]:
                            layer.shading = dictInfo["Layers"][idElement]["shading"]
                        if layer.wireframe.visible != dictInfo["Layers"][idElement]["wireframeVisible"]:
                            layer.wireframe.visible = dictInfo["Layers"][idElement]["wireframeVisible"]
                        if ((layer.wireframe.color[0] != dictInfo["Layers"][idElement]["wireframeRed"])
                                or (layer.wireframe.color[0] != dictInfo["Layers"][idElement]["wireframeRed"])
                                or (layer.wireframe.color[0] != dictInfo["Layers"][idElement]["wireframeRed"])):
                            layer.wireframe.color = [dictInfo["Layers"][idElement]["wireframeRed"],
                                                           dictInfo["Layers"][idElement]["wireframeGreen"],
                                                           dictInfo["Layers"][idElement]["wireframeBlue"], 1.]

                    elif isinstance(layer, Image) or isinstance(layer, Labels):

                        if label.image.getBufferType() not in [PyIPSDK.eIBT_Label8, PyIPSDK.eIBT_Label16, PyIPSDK.eIBT_Label32]:
                            if layer.gamma != dictInfo["Layers"][idElement]["gamma"]:
                                layer.gamma = dictInfo["Layers"][idElement]["gamma"]
                            if layer.attenuation != dictInfo["Layers"][idElement]["attenuation"]:
                                layer.attenuation = dictInfo["Layers"][idElement]["attenuation"]
                            if layer.iso_threshold != dictInfo["Layers"][idElement]["iso_threshold"]:
                                layer.iso_threshold = dictInfo["Layers"][idElement]["iso_threshold"]
                            if layer.contrast_limits is not None and dictInfo["Layers"][idElement]["contrast_limits"] is not None and (
                            (layer.contrast_limits[0] != dictInfo["Layers"][idElement]["contrast_limits"][0] or
                            layer.contrast_limits[1] != dictInfo["Layers"][idElement]["contrast_limits"][1])):
                                layer.contrast_limits = dictInfo["Layers"][idElement]["contrast_limits"]
                        if layer.rendering != dictInfo["Layers"][idElement]["rendering"]:
                            layer.rendering = dictInfo["Layers"][idElement]["rendering"]
                        if layer.blending != dictInfo["Layers"][idElement]["blending"]:
                            layer.blending = dictInfo["Layers"][idElement]["blending"]
                        if layer.opacity != dictInfo["Layers"][idElement]["opacity"]:
                            layer.opacity = dictInfo["Layers"][idElement]["opacity"]

                    if dictInfo["Layers"][idElement]["colormap"] is not None:
                        colormap = napari.utils.Colormap(dictInfo["Layers"][idElement]["colormap"])
                        #if colormap is not None and fct.compareColorMapNapari(layer.colormap,colormap) == False:
                        layer.colormap = colormap

                    try:
                        if fct.compareExperimentalClippingPlanes(layer.experimental_clipping_planes,dictInfo["Layers"][idElement]["experimental_clipping_planes"]) == False:
                            if len(label.linkedLayers) == 0:

                                layer.experimental_clipping_planes = dictInfo["Layers"][idElement]["experimental_clipping_planes"]

                                layer.refresh()
                            else:
                                for layer in label.linkedLayers:
                                    layer.experimental_clipping_planes = dictInfo["Layers"][idElement]["experimental_clipping_planes"]
                                    layer.refresh()


                    except:
                        traceback.print_exc(file=sys.stderr)

                else:

                    if layer.visible:
                        layer.visible = False
                        if info in ["orthosliceX", "orthosliceY", "orthosliceZ"]:
                            label.buttonOrthoslices.setActivation(True)
                        else:
                            label.buttonEye.setActivation(True)

                # vrb.mainWindow.viewer3dSettingsWidget.loadSettings(label)

            except:
                traceback.print_exc(file=sys.stderr)
                # pass

        # try:
        #     vrb.mainWindow.viewer3dSettingsWidget.loadSettings(vrb.mainWindow.currentLabel)
        # except:
        #     traceback.print_exc(file=sys.stderr)

        vrb.mainWindow.viewerNapariQt.canvas.camera._3D_camera.view_changed()
        vrb.mainWindow.viewerNapariQt.canvas.camera._view.camera.view_changed()

        if self.firstUpdate:
            qt.QApplication.processEvents()
            self.firstUpdate = False



    def dragEnterEvent(self, e):

        e.accept()
        self.previousDict = self.dictAllFrames.copy()
        self.backUpDict = self.dictAllFrames.copy()

        self.previousPosition = self.currentPosition
        self.backUpPosition = self.currentPosition

    def dragMoveEvent(self, e):

        pos = e.pos()
        widget = e.source()

        ddictCopy = self.dictAllFrames.copy()

        refIndex = None
        newIndex = None
        for n in range(len(self.allFramesList)):
            w = self.allFramesList[n]

            if n == len(self.allFramesList)-1:
                width = w.size().width()
            else:
                width=self.allFramesList[n+1].x()-self.allFramesList[n].x()

            if w == widget:
                refIndex = n
            if pos.x() > w.x() and pos.x() < w.x() + width and w.isFrame == False:
                newIndex = n

        if newIndex is None:
            ddictCopy = self.previousDict.copy()
            self.currentPosition = self.previousPosition

        elif refIndex is not None and newIndex is not None and refIndex != newIndex:

            diffIndex = newIndex - refIndex

            if self.ctrlPressed and diffIndex > 0:
                for n in range(len(self.allFramesList) - 1, refIndex, -1):
                    if ddictCopy[n] is not None:
                        currentRefIndex = n
                        currentNewIndex = min(n + diffIndex, len(self.allFramesList) - 1)
                        currentTime = int(currentNewIndex * vrb.movieMakerFrameTime * 10) / 10

                        ddictCopy[currentNewIndex] = [str(currentTime) + " s", ddictCopy[currentRefIndex][1], ddictCopy[currentRefIndex][2]]
                        ddictCopy[currentRefIndex] = None

            currentTime = int(newIndex * vrb.movieMakerFrameTime * 10) / 10
            ddictCopy[newIndex] = [str(currentTime) +" s",ddictCopy[refIndex][1],ddictCopy[refIndex][2]]
            ddictCopy[refIndex] = None
            self.currentPosition = newIndex
            # self.previousDict = ddictCopy.copy()
            self.previousPosition = newIndex

            if self.ctrlPressed and diffIndex < 0:
                for n in range(refIndex+1,len(self.allFramesList)):
                    if ddictCopy[n] is not None:
                        currentRefIndex = n
                        currentNewIndex = min(n+diffIndex,len(self.allFramesList)-1)
                        currentTime = int(currentNewIndex * vrb.movieMakerFrameTime * 10) / 10

                        ddictCopy[currentNewIndex] = [str(currentTime) + " s", ddictCopy[currentRefIndex][1], ddictCopy[currentRefIndex][2]]
                        ddictCopy[currentRefIndex] = None

            self.previousDict = ddictCopy.copy()

        e.accept()

        self.updateAllFrames(ddictCopy)

    def dragLeaveEvent(self,e):

        self.currentPosition = self.backUpPosition
        self.backUpDict = self.adjustDict(self.backUpDict)
        self.updateAllFrames(self.backUpDict)

    def dropEvent(self, e):

        pos = e.pos()
        widget = e.source()

        refIndex = None
        newIndex = None
        for n in range(len(self.allFramesList)):
            w = self.allFramesList[n]

            if n == len(self.allFramesList)-1:
                width = w.size().width()
            else:
                width=self.allFramesList[n+1].x()-self.allFramesList[n].x()

            if w == widget:
                refIndex = n
            if pos.x() > w.x() and pos.x() < w.x() + width and w.isFrame == False:
                newIndex = n

        if newIndex is None:
            self.dictAllFrames = self.previousDict.copy()
            self.currentPosition = self.previousPosition

        elif refIndex is not None and newIndex is not None and refIndex != newIndex:
            currentTime = int(newIndex*vrb.movieMakerFrameTime*10)/10
            self.dictAllFrames[newIndex] = [str(currentTime) +" s",self.dictAllFrames[refIndex][1],self.dictAllFrames[refIndex][2]]
            self.dictAllFrames[refIndex] = None
            self.currentPosition = newIndex

        e.accept()

        self.dictAllFrames = self.adjustDict(self.dictAllFrames)
        self.updateAllFrames(self.dictAllFrames)

    def updateAllFrames(self,ddict):

        self.currentWidget = None

        for n in range(len(self.allFramesList)):
            w = self.allFramesList[n]
            w.updateFrame(ddict[n])

class FrameEditorWidget(qt.QWidget):

    def __init__(self,parent = None):
        qt.QWidget.__init__(self)

        self.currentFrame = None

        self.labelZoom = qt.QLabel("Zoom")
        self.labelZoom.setFixedSize(45*vrb.ratio,30*vrb.ratio)
        self.lineEditZoom = qt.QLineEdit()
        self.lineEditZoom.setFixedSize(30*vrb.ratio,30*vrb.ratio)

        self.labelCenterX = qt.QLabel("Center X")
        self.labelCenterX.setFixedSize(45*vrb.ratio,30*vrb.ratio)
        self.lineEditCenterX = qt.QLineEdit()
        self.lineEditCenterX.setFixedSize(30*vrb.ratio,30*vrb.ratio)
        self.labelCenterY = qt.QLabel("Center Y")
        self.labelCenterY.setFixedSize(45*vrb.ratio,30*vrb.ratio)
        self.lineEditCenterY = qt.QLineEdit()
        self.lineEditCenterY.setFixedSize(30*vrb.ratio,30*vrb.ratio)
        self.labelCenterZ = qt.QLabel("Center Z")
        self.labelCenterZ.setFixedSize(45*vrb.ratio,30*vrb.ratio)
        self.lineEditCenterZ = qt.QLineEdit()
        self.lineEditCenterZ.setFixedSize(30*vrb.ratio,30*vrb.ratio)

        self.labelAnglePrecession = qt.QLabel("Angle α")
        self.labelAnglePrecession.setFixedSize(45*vrb.ratio,30*vrb.ratio)
        self.lineEditAnglePrecession = qt.QLineEdit()
        self.lineEditAnglePrecession.setFixedSize(30*vrb.ratio,30*vrb.ratio)
        self.labelAngleNutation = qt.QLabel("Angle β")
        self.labelAngleNutation.setFixedSize(45*vrb.ratio,30*vrb.ratio)
        self.lineEditAngleNutation = qt.QLineEdit()
        self.lineEditAngleNutation.setFixedSize(30*vrb.ratio,30*vrb.ratio)
        self.labelAngleIntrinsicRotation = qt.QLabel("Angle γ")
        self.labelAngleIntrinsicRotation.setFixedSize(45*vrb.ratio,30*vrb.ratio)
        self.lineEditAngleIntrinsicRotation = qt.QLineEdit()
        self.lineEditAngleIntrinsicRotation.setFixedSize(30*vrb.ratio,30*vrb.ratio)

        self.labelPerspective = qt.QLabel("Perspective")
        self.labelPerspective.setFixedSize(45*vrb.ratio,30*vrb.ratio)
        self.lineEditPerspective = qt.QLineEdit()
        self.lineEditPerspective.setFixedSize(30*vrb.ratio,30*vrb.ratio)

        self.pushButtonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=2)
        self.pushButtonValidate.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)

        self.layout = qt.QGridLayout()

        self.layout.setContentsMargins(0, 0, 0, 0)
        self.setContentsMargins(0, 0, 0, 0)

        self.layout.addWidget(self.labelZoom, 0, 0)
        self.layout.addWidget(self.lineEditZoom, 0, 1)
        self.layout.addWidget(self.labelCenterX, 1, 0)
        self.layout.addWidget(self.lineEditCenterX, 1, 1)
        self.layout.addWidget(self.labelCenterY, 1, 2)
        self.layout.addWidget(self.lineEditCenterY, 1, 3)
        self.layout.addWidget(self.labelCenterZ, 1, 4)
        self.layout.addWidget(self.lineEditCenterZ, 1, 5)
        self.layout.addWidget(self.labelAnglePrecession, 2, 0)
        self.layout.addWidget(self.lineEditAnglePrecession, 2, 1)
        self.layout.addWidget(self.labelAngleNutation, 2, 2)
        self.layout.addWidget(self.lineEditAngleNutation, 2, 3)
        self.layout.addWidget(self.labelAngleIntrinsicRotation, 2, 4)
        self.layout.addWidget(self.lineEditAngleIntrinsicRotation, 2, 5)
        self.layout.addWidget(self.labelPerspective, 3, 0)
        self.layout.addWidget(self.lineEditPerspective, 3, 1)
        self.layout.addWidget(self.pushButtonValidate, 4, 5,Qt.AlignRight)

        self.setLayout(self.layout)

        self.setFixedSize(300*vrb.ratio,300*vrb.ratio)

        self.pushButtonValidate.clicked.connect(self.applyToFrame)

        self.setWindowTitle("Frame editor")

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def loadFromFrame(self,frame):
        self.currentFrame = frame

        ddict = frame.parent.dictAllFrames[frame.position][2]["Camera"]
        self.lineEditZoom.setText(str(int(ddict["zoom"]*100)/100))
        self.lineEditCenterX.setText(str(ddict["center"][0]))
        self.lineEditCenterY.setText(str(ddict["center"][1]))
        self.lineEditCenterZ.setText(str(ddict["center"][2]))
        self.lineEditAnglePrecession.setText(str(int(ddict["angles"].angles[0]*100)/100))
        self.lineEditAngleNutation.setText(str(int(ddict["angles"].angles[1]*100)/100))
        self.lineEditAngleIntrinsicRotation.setText(str(int(ddict["angles"].angles[2]*100)/100))
        self.lineEditPerspective.setText(str(ddict["perspective"]))

    def applyToFrame(self):
        if self.currentFrame is not None:

            ddict = self.currentFrame.parent.dictAllFrames[self.currentFrame.position][2]["Camera"]
            ddict["zoom"] = float(self.lineEditZoom.text())
            ddict["center"] = [float(self.lineEditCenterX.text()),float(self.lineEditCenterY.text()),float(self.lineEditCenterZ.text())]
            ddict["angles"].angles = (float(self.lineEditAnglePrecession.text()),float(self.lineEditAngleNutation.text()),float(self.lineEditAngleIntrinsicRotation.text()))
            ddict["perspective"] = float(self.lineEditPerspective.text())

            self.currentFrame.parent.dictAllFrames[self.currentFrame.position][1] = None
            self.currentFrame.updateFrame(self.currentFrame.parent.dictAllFrames[self.currentFrame.position])

            self.close()

class RotateFrameWidget(qt.QWidget):

    def __init__(self,parent = None):
        qt.QWidget.__init__(self)

        self.labelAxis = qt.QLabel("Axis")
        self.labelAxis.setFixedSize(30*vrb.ratio,25*vrb.ratio)
        self.comboBoxAxis = qt.QComboBox()
        self.comboBoxAxis.addItem("X")
        self.comboBoxAxis.addItem("Y")
        self.comboBoxAxis.addItem("Z")
        self.comboBoxAxis.setFixedSize(40*vrb.ratio,20*vrb.ratio)

        self.labelAngle = qt.QLabel("Angle value")
        self.labelAngle.setFixedSize(55 * vrb.ratio, 25 * vrb.ratio)
        self.spinBoxAngle = qt.QSpinBox()
        self.spinBoxAngle.setRange(-360,360)
        self.spinBoxAngle.setValue(90)
        self.spinBoxAngle.setSingleStep(90)
        self.spinBoxAngle.setFixedSize(40*vrb.ratio,20*vrb.ratio)
        self.spinBoxAngle.setAlignment(Qt.AlignRight)

        self.buttonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=2)
        self.buttonValidate.setFixedSize(25*vrb.ratio,25*vrb.ratio)

        self.layout = qt.QGridLayout()

        self.layout.setContentsMargins(15*vrb.ratio, 0, 15*vrb.ratio, 0)

        self.layout.addWidget(self.labelAxis, 0, 0)
        self.layout.addWidget(self.comboBoxAxis, 0, 1,Qt.AlignLeft)
        self.layout.addWidget(self.labelAngle, 0, 2)
        self.layout.addWidget(self.spinBoxAngle, 0, 3, Qt.AlignLeft)
        self.layout.addWidget(self.buttonValidate, 0, 4,Qt.AlignRight)

        self.setLayout(self.layout)

        self.setFixedSize(250 * vrb.ratio, 50 * vrb.ratio)

        self.setWindowTitle("Widget rotation")

        style = fct.getStyleSheet()
        self.setStyleSheet(style)


if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook

    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    # foo = MovieMakerScrollArea()
    # widget = qt.QWidget()
    # layout = qt.QGridLayout()
    # layout.addWidget(foo)
    # layout.setContentsMargins(0,0,0,0)
    # widget.setLayout(layout)

    # widget = SliderPlay()

    # widget = MovieMaker()

    widget = MovieMakerWithMenu()

    style = fct.getStyleSheet()
    widget.setStyleSheet(style)

    widget.show()

    #foo.centralWidget.sliderPlay.splitter.moveSplitter(0, 1)
    widget.movieMakerScrollArea.centralWidget.sliderPlay.splitter.moveSplitter(0, 1)

    app.exec_()

