import sys

from PyQt5.QtCore import pyqtSignal, Qt, QCoreApplication
import PyQt5.QtWidgets as qt
from PyQt5 import QtGui

import WidgetTypes
from UsefullWidgets import DecimalLineEdit, NumberLineEdit
import UsefullVariables as vrb
import UsefullFunctions as fct

import DatabaseFunction as Dfct
import xml.etree.ElementTree as xmlet


class LabelMatrixCreator(qt.QLabel):
    """
    A clickable label that will open a MatrixCreator
    """
    def __init__(self, xmlElement, type):
        if type != WidgetTypes.InputType.KERNEL_2D:
            print('ERROR : InputType does not match', file=sys.stderr)
        super().__init__()
        self.type = type
        self.kernelXml = Dfct.SubElement(xmlElement, 'Element')
        self.setText(Dfct.childText(self.kernelXml, 'Name'))
        self.setFixedHeight(vrb.DEFAULT_SIZE)
        self.matrixCreator = MyMatrixCreator(self.kernelXml)
        self.matrixCreator.pushButtonCancel.clicked.connect(self.cancel)
        self.matrixCreator.SignalValidate.connect(self.closeMatrixCreator)
        self.mouseDoubleClickEvent = self.showMatrixCreator

        font = self.font()
        font.setUnderline(True)
        self.setFont(font)
        self.setStyleSheet("QLabel {color:rgb(6, 115, 186);}")

    def interfaceToXml(self, number):
        paramNode = xmlet.Element('Parameter_' + str(number))
        Dfct.SubElement(paramNode, 'Type').text = WidgetTypes.InputType.KERNEL_2D.value
        value = self.kernelXml
        Dfct.SubElement(paramNode, 'Value').append(value)
        return paramNode

    @staticmethod
    def xmlToValue(xmlElement):
        valueParam = Dfct.SubElement(xmlElement, 'Value')
        value = Dfct.SubElement(valueParam, 'Element')
        return LabelMatrixCreator.valueXmlToKernel(value)

    @staticmethod
    def valueXmlToKernel(kernelXml):
        halfX = int(Dfct.childText(kernelXml, 'HalfSizeX'))
        halfY = int(Dfct.childText(kernelXml, 'HalfSizeY'))
        valuesXml = Dfct.SubElement(kernelXml, 'Values')
        values = []
        for row in range(-halfY, halfY + 1):
            for col in range(-halfX, halfX + 1):
                value = Dfct.childText(valuesXml, 'Pos_' + str(row) + '_' + str(col))
                if value is None:
                    values.append(0)
                else:
                    values.append(fct.fractionToFloat(value))
        return 'PyIPSDK.rectangularKernelXY(' + str(halfX) + ', ' + str(halfY) + ', ' + str(values) + ')\n'

    def showMatrixCreator(self, event):
        self.matrixCreator.loadXmlToInterface()
        self.matrixCreator.show()
        self.matrixCreator.saveElement = Dfct.copyXmlElement(self.kernelXml)

    def closeMatrixCreator(self):
        self.matrixCreator.close()
        self.setText(Dfct.childText(self.kernelXml, 'Name'))

    def cancel(self):
        self.kernelXml = Dfct.copyXmlElement(self.matrixCreator.saveElement)
        self.matrixCreator.xmlElement = self.kernelXml
        self.matrixCreator.close()


class MyMatrixCreator(qt.QGroupBox):

    SignalValidate = pyqtSignal()

    def __init__(self, xmlElement):
        super().__init__()
        self.spacing = 5
        self.margins = 5
        self.xmlElement = xmlElement
        self.saveElement = None

        self.isLoaded = False

        # Left
        self.labelNbRow = qt.QLabel('Half Size Y')
        self.lineEditNbRow = NumberLineEdit()
        self.labelNbCol = qt.QLabel('Half Size X')
        self.lineEditNbCol = NumberLineEdit()
        # self.labelNbPlan = qt.QLabel("Half Size Z")
        # self.lineEditNbPlan = NumberLineEdit()
        self.pushButtonResize = qt.QPushButton("Resize")

        self.groupBoxSizes = qt.QGroupBox()
        self.groupBoxSizes.layout = qt.QGridLayout()
        self.groupBoxSizes.layout.addWidget(self.labelNbCol, 0, 0)
        self.groupBoxSizes.layout.addWidget(self.lineEditNbCol, 0, 1)
        self.groupBoxSizes.layout.addWidget(self.labelNbRow, 1, 0)
        self.groupBoxSizes.layout.addWidget(self.lineEditNbRow, 1, 1)
        # self.groupBoxSizes.layout.addWidget(self.labelNbPlan, 2, 0)
        # self.groupBoxSizes.layout.addWidget(self.lineEditNbPlan, 2, 1)
        self.groupBoxSizes.layout.addWidget(self.pushButtonResize, 0, 2, 3, 1)
        self.groupBoxSizes.setLayout(self.groupBoxSizes.layout)

        # Right
        self.labelFillWith = qt.QLabel('Fill with')
        self.lineEditFillWith = qt.QLineEdit()
        self.pushButtonFillWith = qt.QPushButton("OK")


        self.groupBoxTools = qt.QGroupBox()
        self.groupBoxTools.layout = qt.QGridLayout()
        self.groupBoxTools.layout.addWidget(self.labelFillWith, 0, 0)
        self.groupBoxTools.layout.addWidget(self.lineEditFillWith, 0, 1)
        self.groupBoxTools.layout.addWidget(self.pushButtonFillWith, 0, 2)

        self.groupBoxTools.setLayout(self.groupBoxTools.layout)

        self.tableViewMatrix = qt.QTableWidget(3, 3)
        self.tableViewMatrix.horizontalHeader().setDefaultSectionSize(45)
        self.tableViewMatrix.verticalHeader().setDefaultSectionSize(45)

        # Bottom
        self.pushButtonValidate = qt.QPushButton('Validate')
        self.pushButtonCancel = qt.QPushButton('Cancel')
        self.labelElementName = qt.QLabel('Element name')
        self.lineEditElementName = qt.QLineEdit()

        self.groupBoxBottom = qt.QGroupBox()
        self.groupBoxBottom.layout = qt.QGridLayout()
        self.groupBoxBottom.layout.addWidget(self.labelElementName, 0, 0)
        self.groupBoxBottom.layout.addWidget(self.lineEditElementName, 0, 1)
        self.groupBoxBottom.layout.addWidget(self.pushButtonCancel, 0, 2)
        self.groupBoxBottom.layout.addWidget(self.pushButtonValidate, 0, 3)
        self.groupBoxBottom.setLayout(self.groupBoxBottom.layout)

        # layout
        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.groupBoxSizes, 0, 0)
        self.layout.addWidget(self.groupBoxTools, 0, 1)

        self.layout.addWidget(self.tableViewMatrix, 1, 0, 1, 2)

        self.layout.addWidget(self.groupBoxBottom, 2, 0, 1, 2)

        self.setLayout(self.layout)
        self.layout.setSizeConstraint(1)
        self.layout.setVerticalSpacing(self.spacing)
        self.layout.setContentsMargins(self.margins, self.margins, self.margins, self.margins)

        # connects
        self.pushButtonFillWith.clicked.connect(self.pushButtonFillWithClicked)
        self.pushButtonValidate.clicked.connect(self.pushButtonValidateClicked)
        self.pushButtonResize.clicked.connect(self.resizeMatrix)
        self.tableViewMatrix.cellChanged.connect(self.verifyCell)
        self.lineEditFillWith.textChanged.connect(self.verifyLineEditFillWith)
        self.lineEditElementName.textChanged.connect(self.updateElementName)

        self.loadXmlToInterface()

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def updateElementName(self):
        text = self.lineEditElementName.text()
        Dfct.SubElement(self.xmlElement, 'Name').text = text


    def verifyLineEditFillWith(self):
        text = self.lineEditFillWith.text()
        self.lineEditFillWith.setText(self.verifyText(text))

    def verifyCell(self, row, col):
        if not self.isLoaded:
            text = self.tableViewMatrix.item(row, col).text()
            newText = self.verifyText(text)
            halfSizeX = int(Dfct.childText(self.xmlElement, 'HalfSizeX'))
            halfSizeY = int(Dfct.childText(self.xmlElement, 'HalfSizeY'))
            valuesNode = Dfct.SubElement(self.xmlElement, 'Values')
            Dfct.SubElement(valuesNode, 'Pos_' + str(row-halfSizeY) + '_' + str(col-halfSizeX)).text = newText
            self.loadXmlToInterface()

    def verifyText(self, text):
        newText = ''
        numberPass = False
        nbPoint = 0
        nbSlash = 0
        for i in range(len(text)):
            if text[i] == '-' and i == 0:
                newText += '-'
            if text[i] in vrb.maskNumbers:
                newText += text[i]
                numberPass = True
            if text[i] == '.' and numberPass and nbPoint == 0:
                newText += '.'
                nbPoint += 1
            if text[i] == '/' and numberPass and nbSlash == 0:
                newText += '/'
                numberPass = False
                nbPoint = 0
                nbSlash += 1
        return newText

    def highlightCenter(self):
        nbRow = self.tableViewMatrix.rowCount()
        nbCol = self.tableViewMatrix.columnCount()
        centerRow = nbRow // 2 + 1
        centerCol = nbCol // 2 + 1
        for row in range(nbRow):
            for col in range(nbCol):
                if row == centerRow-1 and col == centerCol-1:
                    self.tableViewMatrix.item(row, col).setBackground(QtGui.QColor.fromRgb(255, 255, 153))
                else:
                    self.tableViewMatrix.item(row, col).setBackground(Qt.white)


    def pushButtonFillWithClicked(self):
        value = self.lineEditFillWith.text()
        valuesNode = Dfct.SubElement(self.xmlElement, 'Values')
        for child in valuesNode:
            child.text = value
        self.loadXmlToInterface()

    def cleanXml(self):
        halfSizeX = int(Dfct.SubElement(self.xmlElement, 'HalfSizeX').text)
        halfSizeY = int(Dfct.SubElement(self.xmlElement, 'HalfSizeY').text)
        valuesXml = Dfct.SubElement(self.xmlElement, 'Values')

        toRemoveList = []
        for valueNode in valuesXml:
            valueTag = valueNode.tag.replace('Pos_', '')
            valueTagSplit = valueTag.split('_')
            row = int(valueTagSplit[0])
            col = int(valueTagSplit[1])
            if (row < -halfSizeY or row > halfSizeY) or (col < -halfSizeX or col > halfSizeX):
                toRemoveList.append(valueNode)
        while len(toRemoveList) != 0:
            valuesXml.remove(toRemoveList.pop())

    def loadXmlToInterface(self):
        self.isLoaded = True
        self.lineEditElementName.setText(Dfct.childText(self.xmlElement, 'Name'))
        halfSizeX = Dfct.SubElement(self.xmlElement, 'HalfSizeX').text
        halfSizeY = Dfct.SubElement(self.xmlElement, 'HalfSizeY').text
        self.lineEditNbCol.setText(halfSizeX)
        self.lineEditNbRow.setText(halfSizeY)
        halfSizeX = int(halfSizeX)
        halfSizeY = int(halfSizeY)
        self.tableViewMatrix.setColumnCount(halfSizeX * 2 + 1)
        self.tableViewMatrix.setRowCount(halfSizeY * 2 + 1)
        # self.resizeMatrix(False, False)
        valuesNode = Dfct.SubElement(self.xmlElement, 'Values')
        for row in range(-halfSizeY, halfSizeY+1):
            for col in range(-halfSizeX, halfSizeX+1):
                valueElement = Dfct.SubElement(valuesNode, 'Pos_' + str(row) + '_' + str(col))
                if valueElement.text is None:
                    valueElement.text = '0'
                cell = qt.QTableWidgetItem(valueElement.text)
                self.tableViewMatrix.setItem(row+halfSizeY, col+halfSizeX, cell)
        self.highlightCenter()
        self.isLoaded = False


    def pushButtonValidateClicked(self):
        nameNode = Dfct.SubElement(self.xmlElement, 'Name')
        if nameNode.text is None or nameNode.text == '':
            sizeX = int(Dfct.childText(self.xmlElement, 'HalfSizeX')) * 2 + 1
            sizeY = int(Dfct.childText(self.xmlElement, 'HalfSizeY')) * 2 + 1
            nameNode.text = 'Kernel_' + str(sizeX) + 'x' + str(sizeY)
        self.SignalValidate.emit()

    def resizeMatrix(self):
        row = self.lineEditNbRow.text()
        if row != "":
            Dfct.SubElement(self.xmlElement, 'HalfSizeY').text = str(row)
        else:
            self.lineEditNbRow.setText(Dfct.SubElement(self.xmlElement, 'HalfSizeY').text)

        col = self.lineEditNbCol.text()
        if col != '':
            Dfct.SubElement(self.xmlElement, 'HalfSizeX').text = str(col)
        else:
            self.lineEditNbCol.setText(Dfct.SubElement(self.xmlElement, 'HalfSizeX').text)

        self.cleanXml()
        self.loadXmlToInterface()




if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook


    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook


    string = '<Element><Name>UnNomCommeça</Name><HalfSizeX>2</HalfSizeX><HalfSizeY>1</HalfSizeY><Values><Pos_0_-1>0</Pos_0_-1><Pos_0_0>1</Pos_0_0><Pos_0_1>2</Pos_0_1></Values></Element>'
    xmlElementKernel = xmlet.fromstring(string)

    # foo = MyMatrixCreator(xmlElementKernel)
    # foo.show()


    xmlElement = xmlet.Element('Param_0')
    Dfct.SubElement(xmlElement, 'Name').text = 'Kernel'
    Dfct.SubElement(xmlElement, 'Type').text = WidgetTypes.InputType.KERNEL_2D.value
    xmlElement.append(xmlElementKernel)
    foo = LabelMatrixCreator(xmlElement, WidgetTypes.InputType.KERNEL_2D)
    foo.show()

    app.exec_()

