import os
import sys

from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore
from PyQt5.QtCore import pyqtSignal, QPointF, Qt, QCoreApplication

import PyIPSDK
import PyIPSDK.IPSDKIPLUtility as util
import PyIPSDK.IPSDKIPLIntensityTransform as itrans
import PyIPSDK.IPSDKIPLGeometricTransform as gtrans

import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr
import PyIPSDK.IPSDKIPLArithmetic as arithm
import PyIPSDK.IPSDKIPLBinarization as bin
import PyIPSDK.IPSDKIPLColor as colorIP
import PyIPSDK.IPSDKIPLBasicMorphology as morpho
import PyIPSDK.IPSDKIPLLogical as logic

import numpy
import xml.etree.ElementTree as xmlet

import UsefullTexts as txt

import UsefullVariables as vrb
import UsefullWidgets as wgt
import UsefullFunctions as fct
import DatabaseFunction as Dfct

import DrawFunctions as DWfct

import time
import traceback

from Sliders import SimpleHorizontalSliderLabel
from RangeSlider import RangeSlider
# from RangeSliderIDS import RangeSlider
from ImageViewer import ImageViewer,ImageViewerWithScrollBar,WidgetInfo

try:
    from pyueye import ueye
except:
    pass
import numpy
from PIL import Image, ImageQt
from PyQt5.QtCore import Qt, QRectF
from PyQt5.QtGui import QPixmap
from PyQt5.QtWidgets import QWidget, QGraphicsScene, QGraphicsView, QPushButton, QVBoxLayout, QApplication, QMessageBox
import sys
import platform

try:
    if platform.system() == 'Windows':
        import win32event
except:
    pass

try:
    import cv2
except:
    pass

class WidgetIDS(qt.QWidget):

    def __init__(self, parent=None):
        qt.QWidget.__init__(self)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        try:
            file = xmlet.parse(vrb.folderInformation + "/Camera.mho")
            self.xmlElement = file.getroot()
        except:
            self.xmlElement = xmlet.Element('Camera')

        self.parent = parent

        self.acquireImage = False
        self.image = None

        self.hasHistogram = False
        self.histogramFirstTime = True
        self.histogramWidget = HistogramWidget(parent=self)

        self.scene = qt.QGraphicsScene()
        self.imageViewerWithScrollBar = ImageViewerWithScrollBar(self.scene, widgetImage=self, parent=self)
        self.imageViewer = self.imageViewerWithScrollBar.imageViewer

        self.graphicElements = GraphicElements(widgetImage=self)

        self.camera = CaptureVideoWidget(widgetImage= self)

        self.groupMenu = GroupMenu(parent=self)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.groupMenu,0,0)
        self.layout.addWidget(self.imageViewerWithScrollBar,1,0,Qt.AlignCenter)

        self.setLayout(self.layout)
        self.layout.setContentsMargins(5* vrb.ratio, 5* vrb.ratio, 5* vrb.ratio, 15* vrb.ratio)
        self.layout.setSizeConstraint(1)

        # self.camera.display_image()

        self.groupMenu.pushButtonLecure.clicked.connect(self.changeLecture)
        self.groupMenu.pushButtonColor.clicked.connect(self.changeColor)
        self.groupMenu.pushButtonRoi.clicked.connect(self.changeRoi)
        self.groupMenu.pushButtonHistogram.clicked.connect(self.showHistogram)
        #self.groupMenu.pushButtonWhiteBalanced.clicked.connect(self.setWhiteBalance)
        self.groupMenu.pushButtonExport.clicked.connect(self.exportImage)

        self.imageViewer.signalEmitMouse.connect(self.graphicElements.selection)

        self.groupMenu.sliderGain.slider.valueChanged.connect(self.changeGain)
        self.groupMenu.sliderExposureTime.slider.valueChanged.connect(self.changeFrameRate)

        # if vrb.hasIDS == False:
            # self.groupMenu.sliderGain.setVisible(False)
            # self.groupMenu.sliderExposureTime.setVisible(False)

        self.setWindowTitle("Camera")

        self.resize(800*vrb.ratio,600*vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def resizeEvent(self,event):

        self.imageViewerWithScrollBar.setFixedSize(max(1,self.width() * 100 / 100 - 30* vrb.ratio), max(1, self.height() * 100 / 100 - 90*vrb.ratio))

    def closeEvent(self, event):

        self.camera.closeEvent(event)

    def setImage(self,image):

        # self.image = fct.getMultiResolutionImage(image)
        self.image =image
        self.image = fct.getMultiResolutionImage(self.image)

        self.imageViewer.setImage(self.image)
        self.imageViewer.getRoiImage(changeRoiImage=False)

        if self.hasHistogram:
            if image.getSizeC() == 1:
                histogramMsrParams = PyIPSDK.createHistoMsrParamsWithBinWidth(0,255,1)
                histogramMsr2dResult = glbmsr.histogramMsr2d(image, histogramMsrParams)
                # self.histogramWidget.rangeSlider.groupSlider.getHistogram(histogramMsr2dResult.frequencies,contour=self.histogramWidget.checkBoxContour.isChecked(),color=[255,0,0])
                # self.histogramWidget.rangeSlider.groupSlider.getHistogram(histogramMsr2dResult.frequencies,contour=self.histogramWidget.checkBoxContour.isChecked(),color=[128,128,128])
                self.histogramWidget.rangeSlider.groupSlider.getHistogram(histogramMsr2dResult.frequencies,contour=self.histogramWidget.checkBoxContour.isChecked(),color=[200,200,200])
            else:
                self.histogramWidget.rangeSlider.groupSlider.getColorHistogram(image,contour=self.histogramWidget.checkBoxContour.isChecked())

            if self.histogramFirstTime:
                self.histogramWidget.rangeSlider.setValues(self.histogramWidget.rangeSlider.minValue,self.histogramWidget.rangeSlider.maxValue)
                self.histogramWidget.rangeSlider.groupLabelValue.labelMinValue.setText("0")
                self.histogramFirstTime = False

        self.createXmlElement()

        qt.QApplication.processEvents()
        if self.acquireImage:
            self.exportImage()
            self.acquireImage = False

    def transformImage(self,image,imageOverlay = None,imageDraw = None,imageProbabilities = None,imageThreshold = None, imageSuperPixels = None):

        if vrb.hasIDS == False:
            if self.groupMenu.pushButtonColor.activate:
                image = colorIP.lightnessImg(image)

        outImage = fct.applyImageContrast(image, self.histogramWidget.rangeSlider.currentValueMin, self.histogramWidget.rangeSlider.currentValueMax)

        return outImage

    def createXmlElement(self):

        Dfct.SubElement(self.xmlElement,"Color").text = str(self.groupMenu.pushButtonColor.activate)
        Dfct.SubElement(self.xmlElement,"Roi").text = str(self.groupMenu.pushButtonRoi.activate)
        Dfct.SubElement(self.xmlElement,"Contour").text = str(self.histogramWidget.checkBoxContour.isChecked())
        Dfct.SubElement(self.xmlElement,"MinValue").text = str(self.histogramWidget.rangeSlider.currentValueMin)
        Dfct.SubElement(self.xmlElement,"MaxValue").text = str(self.histogramWidget.rangeSlider.currentValueMax)
        Dfct.SubElement(self.xmlElement,"Gain").text = str(self.groupMenu.sliderGain.slider.value())
        Dfct.SubElement(self.xmlElement,"ExposureTime").text = str(self.groupMenu.sliderExposureTime.slider.value())

        Dfct.saveXmlElement(self.xmlElement,vrb.folderInformation + "/Camera.mho")

    def loadXmlElement(self):

        color = Dfct.childText(self.xmlElement,"Color")
        if color == "True":
            self.changeColor()
        roi = Dfct.childText(self.xmlElement,"Roi")
        if roi == "True":
            self.changeRoi()
        contour = Dfct.childText(self.xmlElement,"Contour")
        if contour == "True":
            self.histogramWidget.checkBoxContour.setChecked(True)
        try:
            minValue = int(Dfct.childText(self.xmlElement,"MinValue"))
            maxValue = int(Dfct.childText(self.xmlElement,"MaxValue"))
            self.histogramWidget.rangeSlider.currentValueMin = minValue
            self.histogramWidget.rangeSlider.currentValueMax = maxValue
            self.histogramWidget.rangeSlider.setValues(minValue,maxValue)
            self.histogramFirstTime = False
        except:
            pass
        try:
            gain = int(Dfct.childText(self.xmlElement,"Gain"))
            self.groupMenu.sliderGain.slider.setValue(gain)
            exposureTime = int(Dfct.childText(self.xmlElement, "ExposureTime"))
            self.groupMenu.sliderExposureTime.slider.setValue(exposureTime)
        except:
            pass

    def exportImage(self):

        try:
            outImage = self.transformImage(self.image[0])

            if self.groupMenu.pushButtonRoi.activate:
                try:
                    minX = None
                    minY = None
                    maxX = None
                    maxY = None
                    allPoints = Dfct.SubElement(self.graphicElements.roiSelection, "AllPoints")
                    for child in allPoints:
                        pointX = float(Dfct.childText(child, "PointX"))
                        pointY = float(Dfct.childText(child, "PointY"))
                        if minX is None:
                            minX = pointX
                            minY = pointY
                            maxX = pointX
                            maxY = pointY
                        else:
                            minX = min(minX,pointX)
                            minY = min(minY,pointY)
                            maxX = max(maxX,pointX)
                            maxY = max(maxY,pointY)

                    outImage = util.getROI2dImg(outImage, int(minX), int(minY), int(maxX - minX), int(maxY - minY))

                except:
                    pass

            vrb.mainWindow.addImageFromCamera(outImage,"Capture_"+str(vrb.captureNum))
            vrb.captureNum+=1
        except:
            traceback.print_exc(file=sys.stderr)

    def showHistogram(self):

        if self.histogramWidget.isMaximized():
            self.histogramWidget.showMaximized()
        else:
            self.histogramWidget.showNormal()
        self.histogramWidget.window().raise_()
        self.histogramWidget.window().activateWindow()

        self.hasHistogram = True

    def setWhiteBalance(self):

        pVal1 = ueye.DOUBLE(1)
        pVal2 = ueye.DOUBLE(0)

        # ret = ueye.is_SetAutoParameter(self.camera.hCam, ueye.IS_SET_ENABLE_AUTO_WHITEBALANCE, pVal1, pVal2)
        ret = ueye.is_SetAutoParameter(self.camera.hCam, ueye.IS_SET_ENABLE_AUTO_SENSOR_WHITEBALANCE, pVal1, pVal2)
        #print (ret)

    def changeLecture(self):

        self.groupMenu.pushButtonLecure.changeActivation()
        self.camera.switch_capturing()

    def changeRoi(self):

        self.groupMenu.pushButtonRoi.changeActivation()

        self.imageViewer.getRoiImage(changeRoiImage=False)
        if self.groupMenu.pushButtonRoi.activate:

            allPoints = Dfct.SubElement(self.graphicElements.roiSelection,"AllPoints")

            if len(allPoints) == 0:

                DWfct.addPointToPolygon(self.graphicElements.roiSelection, QtCore.QPointF(0, 0))
                DWfct.addPointToPolygon(self.graphicElements.roiSelection, QtCore.QPointF(self.image[0].getSizeX(), 0))
                DWfct.addPointToPolygon(self.graphicElements.roiSelection, QtCore.QPointF(self.image[0].getSizeX(), self.image[0].getSizeY()))
                DWfct.addPointToPolygon(self.graphicElements.roiSelection, QtCore.QPointF(0, self.image[0].getSizeY()))

        self.imageViewer.getRoiImage(changeRoiImage=False)

    def changeColor(self):

        self.groupMenu.pushButtonColor.changeActivation()

        if vrb.hasIDS:
            if self.groupMenu.pushButtonColor.activate:

                ueye.is_SetColorMode(self.camera.hCam, ueye.IS_CM_MONO8)
                self.camera.bytes_per_pixel = 1
                self.camera.packMode = ueye.IS_CM_MONO8

            else:

                ueye.is_SetColorMode(self.camera.hCam, ueye.IS_CM_SENSOR_RAW8)
                self.camera.bytes_per_pixel = 3
                self.camera.packMode = ueye.IS_CM_RGB8_PACKED

            self.camera.alloc(self.camera.hCam)

    def changeGain(self):

        try:
            gain = int(self.groupMenu.sliderGain.slider.value())
            gain = ueye.INT(gain)

            ret = ueye.is_SetHWGainFactor(self.camera.hCam, ueye.IS_SET_MASTER_GAIN_FACTOR, gain)

        except:
            pass

    def changeFrameRate(self):

        exposureTime = int(self.groupMenu.sliderExposureTime.slider.value())
        framesPerSeconds = 1000/exposureTime

        fps = ueye.DOUBLE(framesPerSeconds)
        newFps = ueye.DOUBLE()
        ueye.is_SetFrameRate(self.camera.hCam, fps,newFps)

        exp = ueye.DOUBLE(exposureTime)
        newExp = ueye.DOUBLE()
        try:
            ueye.is_SetExposureTime(self.camera.hCam, exp,newExp)
        except:
            pass


class HistogramWidget(qt.QWidget):

    def __init__(self, parent=None):
        qt.QWidget.__init__(self)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.parent = parent

        fontSize = 10

        self.rangeSlider = RangeSlider(fontSize=fontSize)
        self.rangeSlider.currentValueMin = 0
        self.rangeSlider.currentValueMax = 255

        self.checkBoxContour = qt.QCheckBox("Contour")
        font = QtGui.QFont()
        font.setPointSize(fontSize)
        self.checkBoxContour.setFont(font)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.checkBoxContour, 0, 0)
        self.layout.addWidget(self.rangeSlider, 1, 0)

        self.setLayout(self.layout)
        self.layout.setHorizontalSpacing(20 * vrb.ratio)
        self.layout.setContentsMargins(10, 10, 10, 10)
        self.layout.setSizeConstraint(1)

        self.setWindowTitle("Histogram")

        self.resize(600*vrb.ratio,160*vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def closeEvent(self, a0: QtGui.QCloseEvent):

        self.parent.hasHistogram = False

class GroupMenu(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.parent = parent

        self.pushButtonLecure = wgt.PushButtonDoubleImage(vrb.folderImages + "/Pause.png", vrb.folderImages + "/Lecture.png")
        self.pushButtonLecure.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.pushButtonLecure.setToolTip(txt.dictToolTips["CameraLecture"])

        self.pushButtonColor = wgt.PushButtonDoubleImage(vrb.folderImages + "/Grey.png", vrb.folderImages + "/Color.png")
        self.pushButtonColor.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.pushButtonColor.setToolTip(txt.dictToolTips["CameraColor"])

        self.pushButtonRoi = wgt.PushButtonDoubleImage(vrb.folderImages + "/Zone.png", vrb.folderImages + "/Zone_Pressed.png")
        self.pushButtonRoi.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.pushButtonRoi.setToolTip(txt.dictToolTips["CameraRoi"])

        self.pushButtonHistogram = wgt.PushButtonImage(vrb.folderImages + "/Histogram_2.png", margins=2)
        self.pushButtonHistogram.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.pushButtonHistogram.setToolTip(txt.dictToolTips["DisplayHistogram"])

        self.sliderGain = SimpleHorizontalSliderLabel(label="Gain", titleSize= 30*vrb.ratio, minimum=100, maximum=2000, defaultValue=100, ratio=100)
        self.sliderGain.setFixedWidth(150*vrb.ratio)

        self.sliderExposureTime = SimpleHorizontalSliderLabel(label="ExposureTime", titleSize= 150*vrb.ratio, minimum=40, maximum=200, defaultValue=40, ratio=1)
        self.sliderExposureTime.setFixedWidth(270*vrb.ratio)

        self.pushButtonExport = wgt.PushButtonImage(vrb.folderImages + "/Camera.png", margins=2)
        self.pushButtonExport.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.pushButtonExport.setToolTip(txt.dictToolTips["Camera"])

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.pushButtonLecure,0,0)
        self.layout.addWidget(self.pushButtonColor,0,1)
        self.layout.addWidget(self.pushButtonRoi,0,2)
        self.layout.addWidget(self.pushButtonHistogram,0,3)
        self.layout.addWidget(self.sliderGain,0,4)
        self.layout.addWidget(self.sliderExposureTime,0,5,Qt.AlignLeft)
        self.layout.addWidget(self.pushButtonExport,0,6,Qt.AlignRight)

        self.setLayout(self.layout)
        self.layout.setHorizontalSpacing(20*vrb.ratio)
        self.layout.setContentsMargins(10, 10, 10, 10)
        self.layout.setSizeConstraint(1)

        # self.setFixedSize(400* vrb.ratio,40* vrb.ratio)

class GraphicElements(QtCore.QObject):

    def __init__(self,widgetImage = None):
        super(GraphicElements, self).__init__()

        self.widgetImage = widgetImage
        self.imageViewer = widgetImage.imageViewer
        self.xmlElement = widgetImage.xmlElement

        self.roiSelection = Dfct.SubElement(self.xmlElement, "RoiSelection")

        self.distMinSelection = 10

        self.currentPointSelected = None
        self.currentPolygonSelected = None

        self.drawing = False
        self.hasMoving = False

        self.selectionRectangle = None
        self.refPosX = None
        self.refPosY = None

    def applyGraphicElement(self,imageViewer):

        try:
            if self.widgetImage.groupMenu.pushButtonRoi.activate:
                DWfct.addPolygonToScene(imageViewer, self.roiSelection, closed="True", color=[float(vrb.polygonColor[0]), float(vrb.polygonColor[1]),
                                                                                          float(vrb.polygonColor[2])], showPoints=True, lineSize=2, pointMode="circle", pointSize=6)
        except:
            traceback.print_exc(file=sys.stderr)
            pass

    def selection(self, ddict, imageViewer):

        distMoveRef = self.distMinSelection

        if ddict["Button"] == 1 and ddict["Event"] == "MousePressed":

            self.resetCurrentPolygonSelected()

            point, distMove = DWfct.selectPointPolygon(self.imageViewer, QtCore.QPointF(ddict["Position"][0], ddict["Position"][1]), self.roiSelection, distMin=self.distMinSelection)
            if distMove < distMoveRef:
                self.currentPointSelected = point
                self.currentPolygonSelected = self.roiSelection
                distMoveRef = distMove

            if self.currentPointSelected is None:
                polygon, distMove = DWfct.selectPolygon(self.imageViewer, QtCore.QPointF(ddict["Position"][0], ddict["Position"][1]), self.roiSelection, distMin=self.distMinSelection)
                if distMove < distMoveRef:
                    self.currentPolygonSelected = self.roiSelection
                    distMoveRef = distMove

            if self.currentPolygonSelected != None:
                self.refPosX = ddict["Position"][0]
                self.refPosY = ddict["Position"][1]
                selectionElement = Dfct.SubElement(self.currentPolygonSelected, "Selection")
                selectionElement.text = "True"
            else:
                self.imageViewer.standAlone = True
                self.imageViewer.viewport().setCursor(Qt.ClosedHandCursor)

            self.imageViewer.getRoiImage(changeRoiImage=False)

        if ddict["Button"] == 0 and ddict["Event"] == "Move":

            if self.currentPointSelected is not None:
                # Moving this point
                pointXElement = Dfct.SubElement(self.currentPointSelected, "PointX")
                pointXElement.text = str(ddict["Position"][0])
                pointYElement = Dfct.SubElement(self.currentPointSelected, "PointY")
                pointYElement.text = str(ddict["Position"][1])

                if self.currentPolygonSelected == self.roiSelection:
                    # In case of the roiSelection, the polygon is a rectangle, so moving a point moves 2 other points
                    numPointSelected = int(Dfct.childText(self.currentPointSelected, 'Num'))
                    for point in Dfct.SubElement(self.roiSelection, 'AllPoints'):
                        if int(Dfct.childText(point, 'Num')) == (numPointSelected + 1) % 4:
                            if numPointSelected % 2 == 1:
                                pointXElement = Dfct.SubElement(point, "PointX")
                                pointXElement.text = str(ddict["Position"][0])
                            else:
                                pointYElement = Dfct.SubElement(point, "PointY")
                                pointYElement.text = str(ddict["Position"][1])
                        if int(Dfct.childText(point, 'Num')) == (numPointSelected - 1) % 4:
                            if numPointSelected % 2 == 0:
                                pointXElement = Dfct.SubElement(point, "PointX")
                                pointXElement.text = str(ddict["Position"][0])
                            else:
                                pointYElement = Dfct.SubElement(point, "PointY")
                                pointYElement.text = str(ddict["Position"][1])

                self.imageViewer.getRoiImage(changeRoiImage=False)

            else:
                # Not moving a point
                if self.currentPolygonSelected is not None and self.refPosX is not None and self.refPosY is not None:
                    # moving a rectangle
                    deltaX = ddict["Position"][0] - self.refPosX
                    deltaY = ddict["Position"][1] - self.refPosY
                    allPointsElement = Dfct.SubElement(self.currentPolygonSelected, "AllPoints")
                    for child in allPointsElement:
                        if child.tag == "Point":
                            pointXElement = Dfct.SubElement(child, "PointX")
                            deltaX = min(deltaX, self.imageViewer.image[0].getSizeX() - float(pointXElement.text))
                            deltaX = max(deltaX, -float(pointXElement.text))
                            pointYElement = Dfct.SubElement(child, "PointY")
                            deltaY = min(deltaY, self.imageViewer.image[0].getSizeY() - float(pointYElement.text))
                            deltaY = max(deltaY, -float(pointYElement.text))

                    for child in allPointsElement:
                        if child.tag == "Point":
                            pointXElement = Dfct.SubElement(child, "PointX")
                            pointXElement.text = str(float(pointXElement.text) + deltaX)
                            pointYElement = Dfct.SubElement(child, "PointY")
                            pointYElement.text = str(float(pointYElement.text) + deltaY)
                    self.refPosX = ddict["Position"][0]
                    self.refPosY = ddict["Position"][1]

                    self.imageViewer.getRoiImage(changeRoiImage=False)

        if ddict["Button"] == 1 and ddict["Event"] == "Release":
            self.currentPointSelected = None
            #self.currentPolygonSelected = None
            self.refPosX = None
            self.refPosY = None
            self.imageViewer.standAlone = False


    def resetCurrentPolygonSelected(self):
        if self.currentPolygonSelected != None:
            selectionElement = Dfct.SubElement(self.currentPolygonSelected, "Selection")
            selectionElement.text = "False"
            self.currentPolygonSelected = None

class ImageData:
    memory_pointer = None
    memory_id = None
    width = None
    height = None
    bits_per_pixel = None


class CaptureVideoWidget():
    def __init__(self, widgetImage=None):

        self.widgetImage = widgetImage
        self.imageViewer = self.widgetImage.imageViewer

        self.bytes_per_pixel = 1
        self.bytes_per_pixel = 3

        self.packMode = ueye.IS_CM_RGB8_PACKED
        # self.packMode = ueye.IS_CM_MONO8

        self.converted_memory_pointer = ueye.c_mem_p()
        self.converted_memory_id = ueye.int()
        self.img_data = ImageData()
        # self.capturing = False
        self.capturing = True
        self.hCam = 0
        self.init()
        ueye.is_SetColorMode(self.hCam, ueye.IS_CM_SENSOR_RAW8)
        # ueye.is_SetColorMode(self.hCam, ueye.IS_CM_MONO8)
        self.alloc(self.hCam)
        ueye.is_CaptureVideo(self.hCam, Wait=True)

        self.event = ueye.HANDLE(int())
        self.frame_event_id = ueye.IS_SET_EVENT_FRAME

    def init(self):

        h_cam = ueye.HIDS(0 | ueye.IS_USE_DEVICE_ID)
        ret = ueye.is_InitCamera(h_cam, None)

        if ret == ueye.IS_SUCCESS:
            vrb.hasIDS = True

        try:
            import cv2
            cap = cv2.VideoCapture(0)
            vrb.hasCamera = cap.isOpened()
        except:
            pass

        # if ret != ueye.IS_SUCCESS:
        #     error_message = QMessageBox()
        #     result = QMessageBox.critical(error_message, 'Error', "Invalid device id!", buttons=QMessageBox.Ok)
        #     if result == QMessageBox.Ok:
        #         exit()

        self.hCam = h_cam
        return h_cam

    # def init(self):
    #
    #     try:
    #         import cv2
    #         cap = cv2.VideoCapture(0)
    #         vrb.hasCamera = cap.isOpened()
    #     except:
    #         pass

    def alloc(self, h_cam):
        rect_aoi = ueye.IS_RECT()
        memory_id = ueye.int()
        memory_pointer = ueye.c_mem_p()
        bits_per_pixel = 8
        ueye.is_AOI(h_cam, ueye.IS_AOI_IMAGE_GET_AOI, rect_aoi, ueye.sizeof(rect_aoi))
        ueye.is_AllocImageMem(h_cam, rect_aoi.s32Width, rect_aoi.s32Height, bits_per_pixel, memory_pointer, memory_id)
        ueye.is_SetImageMem(h_cam, memory_pointer, memory_id)
        self.img_data.memory_pointer = memory_pointer
        self.img_data.memory_id = memory_id
        self.img_data.width = rect_aoi.s32Width
        self.img_data.height = rect_aoi.s32Height
        self.img_data.bits_per_pixel = bits_per_pixel
        ueye.is_AllocImageMem(h_cam, rect_aoi.s32Width, rect_aoi.s32Height, 8*self.bytes_per_pixel,
                              self.converted_memory_pointer, self.converted_memory_id)

    def closeEvent(self, event):
        self.capturing = False
        ueye.is_DisableEvent(self.hCam, self.frame_event_id)
        ueye.is_ExitEvent(self.hCam, self.frame_event_id)
        if ueye.is_CaptureVideo(self.hCam, ueye.IS_GET_LIVE):
            ueye.is_StopLiveVideo(self.hCam, Wait=True)

    def switch_capturing(self):
        if self.capturing:
            ueye.is_StopLiveVideo(self.hCam, Wait=True)
            self.capturing = False
        else:
            ueye.is_CaptureVideo(self.hCam, Wait=True)
            self.capturing = True
            self.display_image()

    def convert_image_data(self):
        rect_aoi = ueye.IS_RECT()
        converted_image_data = ImageData()
        conversion_params = ueye.BUFFER_CONVERSION_PARAMS()
        ueye.is_AOI(self.hCam, ueye.IS_AOI_IMAGE_GET_AOI, rect_aoi, ueye.sizeof(rect_aoi))
        converted_image_data.memory_pointer = self.converted_memory_pointer
        converted_image_data.memory_id = self.converted_memory_id
        converted_image_data.width = rect_aoi.s32Width
        converted_image_data.height = rect_aoi.s32Height
        converted_image_data.bits_per_pixel = 8*self.bytes_per_pixel
        # conversion_params.nDestPixelFormat = ueye.IS_CM_MONO8
        # conversion_params.nDestPixelFormat = ueye.IS_CM_RGB8_PACKED
        conversion_params.nDestPixelFormat = self.packMode
        conversion_params.pSourceBuffer = self.img_data.memory_pointer
        conversion_params.pDestBuffer = converted_image_data.memory_pointer
        conversion_params.nDestPixelConverter = ueye.IS_CONV_MODE_SOFTWARE_3X3
        conversion_params.nDestColorCorrectionMode = ueye.IS_CCOR_DISABLE
        conversion_params.nDestGamma = ueye.INT(100)
        conversion_params.nDestSaturationU = ueye.INT(100)
        conversion_params.nDestSaturationV = ueye.INT(100)
        conversion_params.nDestEdgeEnhancement = ueye.INT(0)
        ueye.is_Convert(self.hCam, ueye.IS_CONVERT_CMD_APPLY_PARAMS_AND_CONVERT_BUFFER,
                        conversion_params, ueye.sizeof(conversion_params))

        return converted_image_data

    def free_image_mem(self, mempointer, memid):
        ueye.is_FreeImageMem(self.hCam, mempointer, memid)

    def display_image(self):

        if vrb.hasIDS:

            fps = ueye.DOUBLE()
            ueye.is_GetFramesPerSecond(self.hCam, fps)

            timeout = int((5/fps)*1000)
            h_event = None
            if platform.system() == 'Windows':
                h_event = win32event.CreateEvent(None, False, False, None)
                self.event = ueye.HANDLE(int(h_event))
                ueye.is_InitEvent(self.hCam, self.event, self.frame_event_id)
            ueye.is_EnableEvent(self.hCam, self.frame_event_id)

            while True:

                ret = None
                if not self.capturing:
                    break
                if platform.system() == 'Windows':
                    ret = win32event.WaitForSingleObject(h_event, timeout)
                elif platform.system() == 'Linux':
                    ret = ueye.is_WaitEvent(self.hCam, self.frame_event_id, timeout)

                if ret == 0:

                    converted_image_data = self.convert_image_data()

                    image = (ueye.CHAR * int(self.img_data.width * self.img_data.height * self.bytes_per_pixel))()
                    ueye.is_CopyImageMem(hCam=self.hCam, pcSource=converted_image_data.memory_pointer, nID=converted_image_data.memory_id, pcDest=image)
                    image = numpy.reshape(image,(int(self.img_data.height), int(self.img_data.width), self.bytes_per_pixel))
                    image = image.view(numpy.uint8)

                    if image.shape[2] >= 3:
                        if image.shape[2] == 3:
                            r, g, b = cv2.split(image)

                        allChannels = [PyIPSDK.fromArray(r), PyIPSDK.fromArray(g), PyIPSDK.fromArray(b)]

                        # imageIP = PyIPSDK.createImageRgb(PyIPSDK.eImageBufferType.eIBT_UInt8, image.shape[1], image.shape[0])
                        imageIP = PyIPSDK.createImageRgb(allChannels[0].getBufferType(), image.shape[1], image.shape[0])
                        util.eraseImg(imageIP, 0)
                        for c in range(3):
                            plan = PyIPSDK.extractPlan(0, c, 0, imageIP)
                            util.copyImg(allChannels[c], plan)
                    else:
                        imageIP = PyIPSDK.fromArray(image[:,:,0])

                    self.widgetImage.setImage(imageIP)
                    self.imageViewer.getRoiImage()

                    # self.pil_image = Image.fromarray(self.image_data_copy)
                    # self.graphics_scene.clear()
                    # self.width, self.height = self.pil_image.size
                    # self.qt_image = ImageQt.ImageQt(self.pil_image)
                    # self.pix_map = QPixmap.fromImage(self.qt_image)
                    # self.graphics_scene.addPixmap(self.pix_map)
                    # self.graphics_view.fitInView(QRectF(0, 0, self.width, self.height), Qt.KeepAspectRatio)
                    # self.graphics_scene.update()
                    # qt.QApplication.processEvents()

        else:

            cap = cv2.VideoCapture(0)

            cap.set(cv2.CAP_PROP_FRAME_WIDTH, 1600)
            cap.set(cv2.CAP_PROP_FRAME_HEIGHT, 1200)

            ret, frame = cap.read()

            if ret == False:
                cap = cv2.VideoCapture(0)

            while (True):

                if not self.capturing:
                    break

                #gain = float(int(10*(self.widgetImage.groupMenu.sliderGain.slider.value()-100)/1900))
                gain = self.widgetImage.groupMenu.sliderGain.slider.value()
                #exposureTime = float(int(100*(self.widgetImage.groupMenu.sliderExposureTime.slider.value()-40)/160))
                exposureTime = self.widgetImage.groupMenu.sliderExposureTime.slider.value()

                cap.set(cv2.CAP_PROP_GAIN, gain/1000)
                cap.set(cv2.CAP_PROP_EXPOSURE, exposureTime)
                # cap.set(cv2.CAP_PROP_FPS, exposureTime)

                # Capture frame-by-frame
                ret, image = cap.read()

                if image.shape[2] >= 3:
                    if image.shape[2] == 3:
                        r, g, b = cv2.split(image)

                    allChannels = [PyIPSDK.fromArray(b), PyIPSDK.fromArray(g), PyIPSDK.fromArray(r)]

                    # imageIP = PyIPSDK.createImageRgb(PyIPSDK.eImageBufferType.eIBT_UInt8, image.shape[1], image.shape[0])
                    imageIP = PyIPSDK.createImageRgb(allChannels[0].getBufferType(), image.shape[1], image.shape[0])
                    util.eraseImg(imageIP, 0)
                    for c in range(3):
                        plan = PyIPSDK.extractPlan(0, c, 0, imageIP)
                        util.copyImg(allChannels[c], plan)
                else:
                    imageIP = PyIPSDK.fromArray(image[:, :, 0])

                self.widgetImage.setImage(imageIP)
                self.imageViewer.getRoiImage()



if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook


    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    # foo = GroupMenu()

    foo = WidgetIDS()

    # img = PyIPSDK.loadTiffImageFile("C:/ExampleImages/blobs2d.tif")
    # img.valueMin, img.valueMax, img.valueMinMask = fct.getMinMaxValue(img)
    #
    # imageVisualization = fct.getMultiResolutionImage(img)
    #
    # foo.imageViewer.setImage(imageVisualization)

    # foo.resize(800*vrb.ratio, 800)
    foo.show()

    foo.camera.display_image()

    app.exec_()