import sys,traceback

from PyQt5.QtCore import pyqtSignal,QPointF,Qt,QCoreApplication
from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore

import numpy as np

import time

import Sliders

try:
    import matplotlib
    matplotlib.use('QT5Agg')
    import matplotlib.pylab as plt
    from matplotlib.backends.qt_compat import QtCore, QtWidgets#, is_pyqt5
    from matplotlib.backends.backend_qt5agg import FigureCanvas, NavigationToolbar2QT as NavigationToolbar
    from matplotlib.figure import Figure
except:
    pass

import UsefullWidgets as wgt
from RangeSlider import RangeSlider

import DatabaseFunction as Dfct

import UsefullFunctions as fct

import PyIPSDK
import UsefullVariables as vrb

import UsefullDisplay as display

class HistogramMeasureWidget(qt.QWidget):

    def __init__(self, histogram,xmlElement = None):
        super(HistogramMeasureWidget, self).__init__()

        self.histogram = histogram

        self.fig = Figure()
        self.plotWidget = FigureCanvas(self.fig)
        self.axes = self.fig.add_subplot(111)

        self.toolbar = NavigationToolbar(self.plotWidget, self)

        defaultValueSpacing = 8/(len(self.histogram.tamis) - 3)
        self.sliderSpacing = Sliders.SimpleHorizontalSliderLabel(label="Lengend percentage ", minimum=0, maximum=100, defaultValue=defaultValueSpacing*100, ratio=100,titleSize=130*vrb.ratio)
        self.sliderSpacing.setFixedSize(260*vrb.ratio,30*vrb.ratio)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.toolbar, 0, 0, 1, 2)
        self.layout.addWidget(self.plotWidget, 1, 0, 1, 2)
        self.layout.addWidget(self.sliderSpacing,2, 0, 1, 2)

        self.setLayout(self.layout)

        self.layout.setContentsMargins(10*vrb.ratio,10*vrb.ratio,10*vrb.ratio,10*vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.sliderSpacing.slider.valueChanged.connect(self.computeChart)

        self.computeChart()

    def computeChart(self):

        tick_label = []

        value = 1 + (len(self.histogram.tamis) - 3)*float(self.sliderSpacing.slider.value() / 100)

        tickLabelNum = 0
        for i in range(len(self.histogram.tamis) - 1):
            if int((i / (len(self.histogram.tamis) - 2)) * value) >= tickLabelNum:
                tick_label.append(str(round(((self.histogram.tamis[i] + self.histogram.tamis[i + 1]) / 2) * 10) / 10))
                tickLabelNum += 1
            else:
                tick_label.append("")

        try:
            self.axes.clear()

            if self.histogram.title is not None:
                self.axes.set_title(self.histogram.title)
            if self.histogram.xLegend is not None:
                self.axes.set_xlabel(self.histogram.xLegend)
            if self.histogram.yLegend is not None:
                self.axes.set_ylabel(self.histogram.yLegend)

            self.axes.bar(self.histogram.xValues, self.histogram.barsetValues, tick_label=tick_label, color="blue")
            if self.histogram.sizeText is not None:
                self.axes.tick_params(labelsize=self.histogram.sizeText)
            self.plotWidget.draw()

        except:
            traceback.print_exc(file=sys.stderr)

    def display(self,xmlElement=None):

        try:
            self.setWindowTitle(Dfct.childText(xmlElement, 'Name'))
        except:
            pass
        fct.showWidget(self)

    def stopDisplay(self):

        self.close()

class HistogramMeasureTableWidget(display.TableWidgetWindow):

    def __init__(self, histogram):
        super().__init__()

        self.histogram = histogram

        tick_label = []

        for i in range(len(self.histogram.tamis) - 1):
            tick_label.append(round(((self.histogram.tamis[i] + self.histogram.tamis[i + 1]) / 2) * 10) / 10)

        ddict = {}
        ddict["x values"] = tick_label
        ddict["y values"] = self.histogram.barsetValues

        self.tableWidget.ddict = ddict
        self.tableWidget.loadDictionary()
        self.resize(300*vrb.ratio,500*vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

class HistogramWidget(qt.QWidget):
    """
    Widget used to display an IPSDK histogram as qtChart
    Adapts itself to the type of histogram
    """

    def __init__(self, ipsdkHistogram,xmlElement = None,keep=False):
        from WidgetTypes import ScalarConstraint

        self.keep = keep
        self.refHistogram = ipsdkHistogram
        self.isList = False

        self.widgetHistogramSettings = wgt.WidgetHistogramSettings()

        if self.keep:
            self.ipsdkHistogram = ipsdkHistogram
        else:
            try:
                self.ipsdkHistogram = PyIPSDK.toPyDict(ipsdkHistogram)
            except:
                self.ipsdkHistogram = fct.listToHistogram(ipsdkHistogram.value,nbValues = ipsdkHistogram.nbValues)
                self.isList = True

        self.title = None
        try:
            if xmlElement is not None:
                functionCallElement = Dfct.SubElement(xmlElement,"FunctionCall")
                pararmetersElement = Dfct.SubElement(functionCallElement,"Parameters")
                pararmeter0Element = Dfct.SubElement(pararmetersElement,"Parameter_0")
                if Dfct.childText(pararmeter0Element,"Type") == "ShapeAnalysis":
                    pararmeter1Element = Dfct.SubElement(pararmetersElement, "Parameter_1")
                    measureName = Dfct.childText(pararmeter1Element,"Value")
                    try:
                        try:
                            unit = " (" + self.refHistogram.infoset.getMeasure(measureName).getMsrUnitStr() + ")"
                            unit = unit.replace('^2', '²')
                            unit = unit.replace('^3', '³')
                        except:
                            dictAnalysis = PyIPSDK.toPyDict(self.refHistogram.infoset)
                            infoset = dictAnalysis[0,0,0]
                            unit = " (" + infoset.getMeasure(measureName).getMsrUnitStr() + ")"
                            unit = unit.replace('^2', '²')
                            unit = unit.replace('^3', '³')
                    except:
                        unit = ""

                    userName, isFound = fct.findUserName(measureName)
                    self.title = userName + unit
        except:
            pass

        if self.title is None:
            try:
                self.title = self.refHistogram.title
            except:
                pass


        self.colorNames = []

        self.tables = []

        self.currentLabels = []
        self.currentValues = []
        self.currentNames = []

        self.numberMeaningful = 3

        self.isMultiSlice = (0, 0, 0) in self.ipsdkHistogram
        self.sizeZ = 0
        self.sizeC = 0
        self.sizeT = 0
        if self.isMultiSlice:
            for z, c, t in self.ipsdkHistogram:
                self.sizeZ = max(self.sizeZ, z)
                self.sizeC = max(self.sizeC, c)
                self.sizeT = max(self.sizeT, t)
        self.sizeZ += 1
        self.sizeC += 1
        self.sizeT += 1

        qt.QWidget.__init__(self)

        self.animation = True
        self.isFirstDisplay = True
        # empiric value
        self.magicRatio = 5 * self.sizeC
        self.labelWidths = [82*vrb.ratio, 114*vrb.ratio, 75*vrb.ratio]
        #self.barColors = [QtGui.QColor(66, 134, 244)]
        self.barColors = ["blue"]

        self.fig = Figure()
        self.plotWidget = FigureCanvas(self.fig)
        self.axes = self.fig.add_subplot(111)

        self.toolbar = wgt.NavigationToolbarWidget(self.plotWidget, self)

        self.buttonTable = wgt.PushButtonImage(vrb.folderImages + "/tableau.png")
        self.buttonTable.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)

        self.buttonSettings = wgt.PushButtonImage(vrb.folderImages + "/Settings.png")
        self.buttonSettings.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)

        self.labelMultiplyBin = qt.QLabel('Bin width')
        self.labelMultiplyBin.setFixedSize(self.labelWidths[0], 20*vrb.ratio)
        self.labelMultiplyBin.setAlignment(Qt.AlignBottom)
        self.sliderBinWidth = wgt.SliderShowingValue()
        self.sliderBinWidth.setFixedHeight(50*vrb.ratio)

        self.labelLegendSpacing = qt.QLabel('Legend spacing')
        self.labelLegendSpacing.setFixedSize(self.labelWidths[1], 20*vrb.ratio)
        self.labelLegendSpacing.setAlignment(Qt.AlignBottom)
        self.sliderLegendSpacing = wgt.SliderShowingValue(1)
        self.sliderLegendSpacing.setFixedHeight(50*vrb.ratio)

        self.labelBounds = qt.QLabel('Bounds')
        self.labelBounds.setFixedSize(self.labelWidths[2], 20*vrb.ratio)
        self.labelBounds.setAlignment(Qt.AlignBottom)
        self.rangeSliderBounds = RangeSlider()
        self.rangeSliderBounds.setFixedHeight(50*vrb.ratio)

        self.rangeSliderBounds.type = PyIPSDK.eIBT_Real32
        try:
            if int(self.ipsdkHistogram["BinWidth"]) == self.ipsdkHistogram["BinWidth"]:
                self.rangeSliderBounds.type = PyIPSDK.eIBT_UInt8
            else:
                self.rangeSliderBounds.type = PyIPSDK.eIBT_Real32
        except:
            try:
                if int(self.ipsdkHistogram[(0, 0, 0)]["BinWidth"]) == self.ipsdkHistogram[(0, 0, 0)]["BinWidth"]:
                    self.rangeSliderBounds.type = PyIPSDK.eIBT_UInt8
                else:
                    self.rangeSliderBounds.type = PyIPSDK.eIBT_Real32
            except:
                self.rangeSliderBounds.type = PyIPSDK.eIBT_Real32

        self.checkBoxLog = qt.QCheckBox("Log")

        self.groupBoxBottom = qt.QGroupBox()
        layoutBottom = qt.QGridLayout()
        layoutBottom.addWidget(self.labelMultiplyBin, 0, 0)
        layoutBottom.addWidget(self.sliderBinWidth, 0, 1)

        layoutBottom.addWidget(self.labelLegendSpacing, 0, 2)
        layoutBottom.addWidget(self.sliderLegendSpacing, 0, 3)

        layoutBottom.addWidget(self.labelBounds, 0, 4)
        layoutBottom.addWidget(self.rangeSliderBounds, 0, 5)
        layoutBottom.addWidget(self.checkBoxLog, 0, 6)
        self.groupBoxBottom.setLayout(layoutBottom)
        # self.groupBoxBottom.setMaximumHeight(90*vrb.ratio)
        self.groupBoxBottom.setStyleSheet('QGroupBox {border: 0px transparent; }')
        self.groupBoxBottom.setFixedHeight(90*vrb.ratio)

        if self.sizeZ != 1:
            self.labelPlan = qt.QLabel('Plan')
            self.sliderPlan = wgt.SliderShowingValue(1, Qt.Vertical, True)
            slider = self.sliderPlan.slider
            slider.setRange(0, self.sizeZ-1)
            slider.setValue(slider.maximum())
            self.groupBoxPlan = qt.QGroupBox()
            layoutPlan = qt.QGridLayout()
            layoutPlan.addWidget(self.labelPlan, 0, 0)
            layoutPlan.addWidget(self.sliderPlan, 0, 1)
            self.groupBoxPlan.setLayout(layoutPlan)
            self.sliderPlan.SignalSliderChangedValue.connect(self.initValues)

        if self.sizeT != 1:
            self.labelSequence = qt.QLabel('Sequence')
            self.sliderSequence = wgt.SliderShowingValue(1, Qt.Vertical, True)
            slider = self.sliderSequence.slider
            slider.setRange(0, self.sizeT-1)
            slider.setValue(slider.maximum())
            self.groupBoxSequence = qt.QGroupBox()
            layoutSequence = qt.QGridLayout()
            layoutSequence.addWidget(self.labelSequence, 0, 0)
            layoutSequence.addWidget(self.sliderSequence, 0, 1)
            self.groupBoxSequence.setLayout(layoutSequence)
            self.sliderSequence.SignalSliderChangedValue.connect(self.initValues)

        if self.sizeC != 1:
            try:
                image = self.refHistogram.image
                for c in range(self.sizeC):
                    self.colorNames.append(image.getChannelName(c))
            except:
                self.colorNames = []
                if self.sizeC == 3:
                    self.colorNames = ["Red","Green","Blue"]
                else:
                    for c in range(self.sizeC):
                        self.colorNames.append("Channel "+str(c))

            self.checkBoxColors = []
            for c in range(self.sizeC):
                self.checkBoxColors.append(qt.QCheckBox(self.colorNames[c]))

            self.groupBoxColor = qt.QGroupBox()
            layoutColor = qt.QGridLayout()
            layoutColor.setAlignment(Qt.AlignBottom)
            for checkBox in self.checkBoxColors:
                checkBox.setChecked(True)
                layoutColor.addWidget(checkBox)
                checkBox.stateChanged.connect(self.computeChart)
            self.groupBoxColor.setLayout(layoutColor)
            # self.barColors = [QtGui.QColor(255, 0, 0), QtGui.QColor(0, 255, 0), QtGui.QColor(0, 0, 255), QtGui.QColor(160, 160, 160)]
            self.barColors = ["red","lime","blue"]

        if self.sizeC != 1 or self.sizeZ != 1 or self.sizeT != 1:
            self.groupBoxRight = qt.QGroupBox()
            self.groupBoxRight.setStyleSheet(('QGroupBox {border: 0px transparent; }'))
            if self.sizeZ != 1 and self.sizeT != 1:
                self.groupBoxRight.setFixedWidth(180*vrb.ratio)
            else:
                self.groupBoxRight.setFixedWidth(100*vrb.ratio)
            layoutRight = qt.QGridLayout()
            if self.sizeZ != 1:
                layoutRight.addWidget(self.groupBoxPlan, 0, 0)
                if self.sizeT != 1:
                    layoutRight.addWidget(self.groupBoxSequence, 0, 1)
            else:
                if self.sizeT != 1:
                    layoutRight.addWidget(self.groupBoxSequence, 0, 0)
            if self.sizeC != 1:
                if self.sizeZ != 1 and self.sizeT != 1:
                    layoutRight.addWidget(self.groupBoxColor, 1, 0, 1 ,2)
                else:
                    layoutRight.addWidget(self.groupBoxColor, 1, 0)
            self.groupBoxRight.setLayout(layoutRight)

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.toolbar,0,0)
        self.layout.addWidget(self.buttonTable,0,1,Qt.AlignRight)
        if self.isList:
            self.layout.addWidget(self.buttonSettings,0,2,Qt.AlignRight)

        if self.sizeC != 1 or self.sizeZ != 1 or self.sizeT != 1:
            self.layout.addWidget(self.plotWidget, 1, 0, 1, 1)
            self.layout.addWidget(self.groupBoxRight, 1, 1,Qt.AlignRight)
            self.layout.addWidget(self.groupBoxBottom, 2, 0,1,2)
        else:
            self.layout.addWidget(self.plotWidget, 1, 0, 1, 2)
            self.layout.addWidget(self.groupBoxBottom, 2, 0)

        self.layout.setContentsMargins(15*vrb.ratio, 0, 15*vrb.ratio, 0)
        self.layout.setSizeConstraint(1)

        self.setLayout(self.layout)

        self.sliderLegendSpacing.SignalSliderChangedValue.connect(self.computeChart)
        self.sliderBinWidth.SignalSliderChangedValue.connect(self.computeChart)
        self.rangeSliderBounds.signalValueChanged.connect(self.computeChart)
        self.checkBoxLog.stateChanged.connect(self.computeChart)
        self.buttonTable.clicked.connect(self.setTableWidget)
        self.buttonSettings.clicked.connect(self.loadSettings)
        self.widgetHistogramSettings.buttonValidate.clicked.connect(self.applySettings)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.resize(1200*vrb.ratio, 600*vrb.ratio)

    def resizeEvent(self, event):
        availableSpace = self.groupBoxBottom.width() - sum(self.labelWidths)
        availableSpace += - self.groupBoxBottom.layout().spacing() * 5
        availableSpace = max(0,availableSpace)
        self.sliderBinWidth.setFixedWidth(availableSpace * 22 / 100)
        self.sliderLegendSpacing.setFixedWidth(availableSpace * 22 / 100)
        self.rangeSliderBounds.setFixedWidth(availableSpace * 42 / 100)
        self.checkBoxLog.setFixedWidth(availableSpace * 8 / 100)

        self.toolbar.setFixedSize(self.width()-80*vrb.ratio,50*vrb.ratio)

    def closeEvent(self, a0: QtGui.QCloseEvent):

        for table in self.tables:
            table.close()

        self.tables = []

    def loadSettings(self):

        self.widgetHistogramSettings.lineEditMinValue.setText(fct.numberCalibration(self.ipsdkHistogram["Min"]))
        self.widgetHistogramSettings.lineEditMaxValue.setText(fct.numberCalibration(self.ipsdkHistogram["Max"]))
        self.widgetHistogramSettings.lineEditBinWidth.setText(fct.numberCalibration(self.ipsdkHistogram["BinWidth"]))

        self.widgetHistogramSettings.show()

    def applySettings(self):

        try:
            self.widgetHistogramSettings.close()

            try:
                minValue = float(self.widgetHistogramSettings.lineEditMinValue.text())
            except:
                minValue = None
            try:
                maxValue = float(self.widgetHistogramSettings.lineEditMaxValue.text())
            except:
                maxValue = None
            if self.widgetHistogramSettings.radioButtonNbValues.isChecked():
                try:
                    nbValues = int(self.widgetHistogramSettings.lineEditNbValues.text())
                except:
                    nbValues = None
                binWidth = None
            else:
                try:
                    binWidth = float(self.widgetHistogramSettings.lineEditBinWidth.text())
                except:
                    binWidth = None
                nbValues = None

            self.ipsdkHistogram = fct.listToHistogram(self.refHistogram.value, minValue=minValue,maxValue=maxValue,binWidth=binWidth,nbValues=nbValues)

            self.minValue = self.ipsdkHistogram["Min"]
            self.maxValue = self.ipsdkHistogram["Max"]
            self.binWidth = self.ipsdkHistogram["BinWidth"]
            self.values = [self.ipsdkHistogram["Frequencies"]]

            self.sliderBinWidth.setInitialValue(self.binWidth)
            self.rangeSliderBounds.setMinMaxValues(self.minValue, self.maxValue)
            self.rangeSliderBounds.setValues(self.minValue, self.maxValue)
            self.computeChart()
        except:
            pass


    def stopDisplay(self):
        self.close()

    def initValues(self):
        """
        Called everytime the values to display are changed
        """

        if self.sizeZ == 1 and self.sizeC == 1 and self.sizeT == 1:
            self.minValue = self.ipsdkHistogram["Min"]
            self.maxValue = self.ipsdkHistogram["Max"]
            self.binWidth = self.ipsdkHistogram["BinWidth"]
            self.values = [self.ipsdkHistogram["Frequencies"]]
        else:
            if self.sizeZ != 1:
                selectedPlan = self.sliderPlan.getValue()
            else:
                selectedPlan = 0
            if self.sizeT != 1:
                selectedSequence = self.sliderSequence.getValue()
            else:
                selectedSequence = 0

            self.minValue = fct.getHistogramMinValue(self.ipsdkHistogram)
            self.maxValue = fct.getHistogramMaxValue(self.ipsdkHistogram)
            self.binWidth = fct.getHistogramBinValue(self.ipsdkHistogram)

            if self.sizeC == 1:
                self.values = [self.ipsdkHistogram[(selectedPlan, 0, selectedSequence)]["Frequencies"]]
            else:
                self.values = []

                nbFrequencies = int(((self.maxValue-self.minValue)/self.binWidth) + 1)

                for c in range(self.sizeC):

                    histogram = self.ipsdkHistogram[(selectedPlan, c, selectedSequence)]

                    value = np.zeros((nbFrequencies))
                    for i in range(len(histogram["Frequencies"])):

                        pos = int((histogram["Min"] + (i+0.5)*histogram["BinWidth"] - self.minValue)/self.binWidth)
                        value[pos] += histogram["Frequencies"][i]

                    self.values.append(value)

        if self.isFirstDisplay:
            self.sliderBinWidth.setInitialValue(self.binWidth)
            self.rangeSliderBounds.setMinMaxValues(self.minValue, self.maxValue)
            self.rangeSliderBounds.setValues(self.minValue, self.maxValue)
        else:
            self.computeChart()

    def computeChart(self):
        """
        Creates the chart from the initialised values
        """

        try:

            self.axes.clear()

            if self.title is not None:
                self.axes.set_title(self.title)

            binWidthFactor = self.sliderBinWidth.getSliderPosition()
            minX = self.rangeSliderBounds.currentValueMin
            maxX = self.rangeSliderBounds.currentValueMax
            nbValuesDisplayed = 0

            #series = QBarSeries()
            maxY = -float('inf')  # Will contain the greatest number of frequency
            minY = float('inf')   # Will contain the minimal number of frequency

            nbColor = 0
            if self.sizeC != 1:
                for c in range(len(self.values)):
                    if self.checkBoxColors[c].isChecked():
                        nbColor += 1
            if nbColor == 0:
                width = 0.7
            else:
                width = 0.7 / nbColor

            vectorColor = []
            for colorNum in range(nbColor):
                vectorColor.append((-((nbColor-1)/2)+colorNum)*width)

            currentNumColor = 0
            self.currentValues = []
            self.currentNames = []
            for c in range(len(self.values)):
                if self.sizeC == 1 or (self.sizeC != 1 and self.checkBoxColors[c].isChecked()):
                    nbValuesDisplayed = 0
                    barsetValues = []
                    labels = []
                    xValues = []
                    nbAdd = 0
                    sum = 0
                    for i in range(len(self.values[c])):
                        value = self.values[c][i]
                        nb = self.minValue + i*self.binWidth

                        if minX <= nb <= maxX:
                            sum += value
                            nbAdd += 1

                            if nbAdd == binWidthFactor or i == len(self.values[c])-1:
                                barsetValues.append(sum)
                                labels.append(fct.numberCalibration(nb,self.numberMeaningful))
                                xValues.append(nbValuesDisplayed)
                                nbValuesDisplayed += 1
                                maxY = max(maxY, sum)
                                minY = min(minY, sum)
                                nbAdd = 0
                                sum = 0

                    oneLegendEachX = self.sliderLegendSpacing.getValue()
                    newLabels = []
                    nbins = 0

                    for i in range(len(labels)):

                        if i%oneLegendEachX == 0:
                            newLabels.append(labels[i])
                            nbins+=1
                        else:
                            try:
                                if (i%oneLegendEachX)/oneLegendEachX<0.5:
                                    newLabels.append(labels[oneLegendEachX*(i//oneLegendEachX)])
                                else:
                                    newLabels.append(labels[oneLegendEachX*((i//oneLegendEachX)+1)])
                            except:
                                newLabels.append("")

                    while len(newLabels)<len(labels):
                        newLabels.append("")

                    self.currentLabels = labels
                    self.currentValues.append(barsetValues)
                    if self.sizeC == 1:
                        self.currentNames.append("Intensities")
                    else:
                        self.currentNames.append(self.colorNames[c])

                    if self.sizeC == 1:
                        self.axes.bar(xValues,barsetValues,width=width,tick_label=newLabels,color="blue")
                        self.axes.locator_params(axis='x',nbins=nbins)
                    elif self.sizeC > 1 and self.checkBoxColors[c].isChecked():
                        if self.sizeC <= 3:
                            self.axes.bar([x + vectorColor[currentNumColor] for x in xValues], barsetValues, width=width, tick_label=newLabels, color=self.barColors[c])
                        else:
                            valColor = c%10
                            self.axes.bar([x + vectorColor[currentNumColor] for x in xValues], barsetValues, width=width, tick_label=newLabels,color = "C"+str(valColor),label=self.colorNames[c])
                        currentNumColor += 1

                    if self.checkBoxLog.isChecked():
                        self.axes.set_yscale("log")

            if self.sizeC >3:
                self.axes.legend(loc=1, prop={'size': 10})

            if nbValuesDisplayed != 0:
                try:
                    self.axes.locator_params(axis='x', nbins=nbins)
                except:
                    pass
            else:
                self.axes.clear()
                self.plotWidget.draw()
                return

            oneLegendEachX = self.sliderLegendSpacing.getValue()
            legendNumber = max(round((nbValuesDisplayed)/oneLegendEachX), 1)

            if self.isFirstDisplay:
                # Tries of optimize the display to show bars and in a beautiful way
                self.isFirstDisplay = False
                binFactorOptimized = ((maxX - minX) * self.magicRatio / self.plotWidget.width()) / self.binWidth
                if self.sliderBinWidth.getSliderPosition() < binFactorOptimized:
                    posSliderBinWidthOptiDisplay = int(binFactorOptimized) + 1
                    self.sliderBinWidth.slider.setValue(min(max(posSliderBinWidthOptiDisplay, 1), 20))
                while ((self.window().width() * 80 / 100) / legendNumber) < 50/len(self.values):
                    oneLegendEachX += 1
                    legendNumber = max(round(nbValuesDisplayed / oneLegendEachX), 1)
                # self.sliderLegendSpacing.slider.setValue(oneLegendEachX*4)
                self.sliderLegendSpacing.slider.setValue(10)

            try:
                self.plotWidget.draw()
            except:
                print("Can't plot histogram")

        except:
            traceback.print_exc(file=sys.stderr)

    def display(self, xmlElement=None, boolShow=True):

        try:
            self.setWindowTitle(Dfct.childText(xmlElement, 'Name'))
        except:
            pass
        #if self.isFirstDisplay:
        self.isFirstDisplay = True
        self.initValues()
        self.computeChart()

        if boolShow:

            if self.isMaximized():
                self.showMaximized()
            else:
                self.showNormal()
            self.window().raise_()
            self.window().activateWindow()

        self.rangeSliderBounds.isFixedMin = False
        self.rangeSliderBounds.isFixedMax = False

        self.rangeSliderBounds.setValues(self.rangeSliderBounds.minValue,self.rangeSliderBounds.maxValue)

    def setTableWidget(self):

        try:
            ddict = {}
            ddict["Intensities"] = self.currentLabels

            if len(self.currentValues) == 1:
                ddict["Number of pixels"] = self.currentValues[0]
            else:
                for c in range(len(self.currentValues)):
                    ddict[self.currentNames[c]] = self.currentValues[c]

            table = display.TableWidgetWindow(ddict)

            table.setWindowTitle("Histogram table")

            # table.resize(78*(len(self.currentValues)+1)*vrb.ratio,500*vrb.ratio)

            self.tables.append(table)

            table.show()
        except:
            traceback.print_exc(file=sys.stderr)

class HistogramTableWidget(qt.QWidget):
    """
    Widget used to display an IPSDK histogram as table
    Adapts itself to the type of histogram
    """

    def __init__(self, ipsdkHistogram):

        qt.QWidget.__init__(self)

        self.refHistogram = ipsdkHistogram

        try:
            self.ipsdkHistogram = PyIPSDK.toPyDict(ipsdkHistogram)
        except:
            self.ipsdkHistogram = fct.listToHistogram(ipsdkHistogram.value,nbValues = ipsdkHistogram.nbValues)

        self.colorNames = []

        self.isMultiSlice = (0, 0, 0) in self.ipsdkHistogram
        self.sizeZ = 0
        self.sizeC = 0
        self.sizeT = 0
        if self.isMultiSlice:
            for z, c, t in self.ipsdkHistogram:
                self.sizeZ = max(self.sizeZ, z)
                self.sizeC = max(self.sizeC, c)
                self.sizeT = max(self.sizeC, t)
        self.sizeZ += 1
        self.sizeC += 1
        self.sizeT += 1

        if self.sizeC > 1:
            try:
                image = self.refHistogram.image
                for c in range(self.sizeC):
                    self.colorNames.append(image.getChannelName(c))
            except:
                self.colorNames = []
                if self.sizeC == 3:
                    self.colorNames = ["Red", "Green", "Blue"]
                else:
                    for c in range(self.sizeC):
                        self.colorNames.append("Channel " + str(c))

        self.tableWidgetWindow = display.TableWidgetWindow()
        self.tableWidget = self.tableWidgetWindow.tableWidget

        if self.sizeZ != 1:
            self.labelPlan = qt.QLabel('Plan')
            self.sliderPlan = wgt.SliderShowingValue(1, Qt.Vertical, True)
            slider = self.sliderPlan.slider
            slider.setRange(0, self.sizeZ-1)
            slider.setValue(slider.maximum())
            self.groupBoxPlan = qt.QGroupBox()
            layoutPlan = qt.QGridLayout()
            layoutPlan.addWidget(self.labelPlan, 0, 0)
            layoutPlan.addWidget(self.sliderPlan, 0, 1)
            self.groupBoxPlan.setLayout(layoutPlan)
            self.sliderPlan.SignalSliderChangedValue.connect(self.initValues)
            sizeGroupBoxPlan = 65*vrb.ratio
        else:
            sizeGroupBoxPlan = 0

        if self.sizeT != 1:
            self.labelSequence = qt.QLabel('Sequence')
            self.sliderSequence = wgt.SliderShowingValue(1, Qt.Vertical, True)
            slider = self.sliderSequence.slider
            slider.setRange(0, self.sizeT-1)
            slider.setValue(slider.maximum())
            self.groupBoxSequence = qt.QGroupBox()
            layoutSequence = qt.QGridLayout()
            layoutSequence.addWidget(self.labelSequence, 0, 0)
            layoutSequence.addWidget(self.sliderSequence, 0, 1)
            self.groupBoxSequence.setLayout(layoutSequence)
            self.sliderSequence.SignalSliderChangedValue.connect(self.initValues)
            sizeGroupBoxSequence = 65*vrb.ratio
        else:
            sizeGroupBoxSequence = 0

        self.initValues()

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.tableWidgetWindow, 0, 0)
        if self.sizeZ != 1:
            self.layout.addWidget(self.groupBoxPlan, 0, 1)
            if self.sizeT !=1:
                self.layout.addWidget(self.groupBoxSequence, 0, 2)
        else:
            if self.sizeT !=1:
                self.layout.addWidget(self.groupBoxSequence, 0, 1)

        self.layout.setContentsMargins(0, 0, 0, 0)
        self.layout.setSizeConstraint(1)

        self.setLayout(self.layout)

        self.resize(78 * (self.sizeC + 1) * vrb.ratio + 30*vrb.ratio + sizeGroupBoxPlan + sizeGroupBoxSequence, 500 * vrb.ratio)

        style = fct.getStyleSheet()
        self.setStyleSheet(style)

    def initValues(self):

        ddictResult = {}

        if self.sizeC == 1:
            if self.isMultiSlice == False:
                ddict = self.ipsdkHistogram
            else:
                if self.sizeZ != 1:
                    selectedPlan = self.sliderPlan.getValue()
                else:
                    selectedPlan = 0
                if self.sizeT != 1:
                    selectedSequence = self.sliderSequence.getValue()
                else:
                    selectedSequence = 0
                ddict = self.ipsdkHistogram[(selectedPlan, 0, selectedSequence)]
            frequencies = ddict["Frequencies"]
            intensities = []
            for i in range(len(frequencies)):
                intensities.append(ddict["Min"] + i*ddict["BinWidth"])

            ddictResult ["Intensities"] = intensities
            ddictResult ["Number of pixels"] = frequencies

        else:
            if self.sizeZ == 1:
                selectedPlan = 0
            else:
                selectedPlan = self.sliderPlan.getValue()
            if self.sizeT == 1:
                selectedSequence = 0
            else:
                selectedSequence = self.sliderSequence.getValue()

            minValue = fct.getHistogramMinValue(self.ipsdkHistogram)
            maxValue = fct.getHistogramMaxValue(self.ipsdkHistogram)
            binWidth = fct.getHistogramBinValue(self.ipsdkHistogram)

            nbFrequencies = int(((maxValue - minValue) / binWidth) + 1)

            intensities = []
            for i in range(nbFrequencies):
                intensities.append(minValue + i * binWidth)
            ddictResult["Intensities"] = intensities

            for c in range(self.sizeC):

                histogram = self.ipsdkHistogram[(selectedPlan, c, selectedSequence)]
                value = np.zeros((int(nbFrequencies)))

                for i in range(len(histogram["Frequencies"])):
                    pos = int((histogram["Min"] + (i+0.5)*histogram["BinWidth"] - minValue)/binWidth)
                    value[pos] += histogram["Frequencies"][i]

                ddictResult[self.colorNames[c]] = value

        self.tableWidget.ddict = ddictResult
        self.tableWidget.loadDictionary()

    def stopDisplay(self):
        self.close()

    def display(self, xmlElement=None, boolShow=True):
        self.setWindowTitle(Dfct.childText(xmlElement, 'Name'))
        self.initValues()
        if boolShow:
            if self.isMaximized():
                self.showMaximized()
            else:
                self.showNormal()
            self.window().raise_()
            self.window().activateWindow()

if __name__ == '__main__':

    import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook

    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    histogramMsrParams = PyIPSDK.createHistoMsrParamsWithBinWidth(0, 255, 1)
    inImage = PyIPSDK.loadTiffImageFile("D:/ExampleImages/Lena_510x509_UInt8.tif")
    histogramMsr2dResult = glbmsr.histogramMsr2d(inImage, histogramMsrParams)
    # inImage = PyIPSDK.loadTiffImageFile("C:/ExampleImages/Lena_RGB_510x509_UInt8.tif")
    # histogramMsr2dResult = glbmsr.multiSlice_histogramMsr2d(inImage, histogramMsrParams)

    foo = HistogramWidget(histogramMsr2dResult)

    foo.show()
    foo.initValues()

    app.exec_()