import sys
import os
import traceback

from PyQt5.QtCore import pyqtSignal,QPointF,Qt,QCoreApplication
from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore

import xml.etree.ElementTree as xmlet

import UsefullWidgets as wgt
import UsefullVariables as vrb
import UsefullFunctions as fct
import DatabaseFunction as Dfct
import UsefullTexts as txt

import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr
import PyIPSDK


import numpy as np


class GroupColorCalibration(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.colorCalibrationSettingsWidget = ColorCalibrationSettingsWidget()

        totalWidth = 0

        self.buttonColorCalibrationSettings = wgt.PushButtonImage(vrb.folderImages + "/Lut_Color_Settings.png")
        self.buttonColorCalibrationSettings.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.buttonColorCalibrationSettings.setToolTip(txt.dictToolTips["Lut_Color_Settings"])
        totalWidth += 30 * vrb.ratio

        self.buttonColorCalibrationApply = wgt.PushButtonImage(vrb.folderImages + "/Lut_Color_Apply.png")
        self.buttonColorCalibrationApply.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.buttonColorCalibrationApply.setToolTip(txt.dictToolTips["Lut_Color_Apply"])
        totalWidth += 30 * vrb.ratio

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.buttonColorCalibrationSettings, 0, 0)
        self.layout.addWidget(self.buttonColorCalibrationApply, 0, 1)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(5 * vrb.ratio, 0, 5 * vrb.ratio, 0)
        self.layout.setHorizontalSpacing(3 * vrb.ratio)
        self.layout.setAlignment(Qt.AlignCenter)

        totalWidth += 15 * vrb.ratio

        self.setStyleSheet("QGroupBox {border: 1px solid gray; }")

        self.setFixedSize(totalWidth, 40 * vrb.ratio)

        self.buttonColorCalibrationSettings.clicked.connect(self.showColorCalibrationSettings)
        self.buttonColorCalibrationApply.clicked.connect(self.colorCalibrationSettingsWidget.applyColorCalibration)

    def showColorCalibrationSettings(self):

        fct.showWidget(self.colorCalibrationSettingsWidget)

class ColorCalibrationSettingsWidget(qt.QWidget):

    def __init__(self):
        super().__init__()

        try:
            file = xmlet.parse(vrb.folderInformation + "/ColorCalibrationSettings.mho")
            self.xmlElement = file.getroot()
        except:
            self.xmlElement = xmlet.Element('ColorCalibrationSettings')

        self.image = None
        self.isPreviewed = False

        self.labelPosition = qt.QLabel("Position")
        self.labelPosition.setFixedSize(80*vrb.ratio,30*vrb.ratio)
        self.comboBoxPosition = qt.QComboBox()
        self.comboBoxPosition.addItem("Top left")
        self.comboBoxPosition.addItem("Top right")
        self.comboBoxPosition.addItem("Bottom left")
        self.comboBoxPosition.addItem("Bottom right")
        self.comboBoxPosition.setCurrentIndex(0)
        self.comboBoxPosition.setFixedSize(90 * vrb.ratio, 20 * vrb.ratio)

        self.labelSizePercentage = qt.QLabel("Size (%)")
        self.labelSizePercentage.setFixedSize(80*vrb.ratio,30*vrb.ratio)
        self.lineEditSizePercentage = qt.QLineEdit()
        self.lineEditSizePercentage.setText(str(50))
        self.lineEditSizePercentage.setPlaceholderText(str(50))
        self.lineEditSizePercentage.setAlignment(Qt.AlignRight)
        self.lineEditSizePercentage.setFixedSize(50*vrb.ratio, 20*vrb.ratio)

        self.radioButtonVertical = qt.QRadioButton("Vertical")
        self.radioButtonVertical.setFixedSize(80*vrb.ratio, 20*vrb.ratio)
        self.radioButtonVertical.setChecked(True)
        self.radioButtonHorizontal = qt.QRadioButton("Horizontal")
        self.radioButtonHorizontal.setFixedSize(80*vrb.ratio, 20*vrb.ratio)

        self.labelTextColor = qt.QLabel("Text color")
        self.labelTextColor.setFixedSize(80*vrb.ratio, 20*vrb.ratio)
        self.buttonTextColor = qt.QPushButton("")
        self.buttonTextColor.setFixedSize(40*vrb.ratio,20*vrb.ratio)
        self.buttonTextColor.setStyleSheet('QPushButton {background-color: rgb(0,0,0);}')

        self.labelBackgroundColor = qt.QLabel("Background color")
        self.labelBackgroundColor.setFixedSize(80 * vrb.ratio, 20 * vrb.ratio)
        self.buttonBackgroundColor = qt.QPushButton("")
        self.buttonBackgroundColor.setFixedSize(40 * vrb.ratio, 20 * vrb.ratio)
        self.buttonBackgroundColor.setStyleSheet('QPushButton {background-color: rgb(255,255,255);}')

        self.buttonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=0)
        self.buttonValidate.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        self.buttonValidate.setToolTip("Validate")

        self.buttonPreview = wgt.SwitchingLabel(wgt.LabelImage(vrb.folderImages + '/eye_close.png'), wgt.LabelImage(vrb.folderImages + '/eye_open.png'))
        self.buttonPreview.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.buttonPreview.setToolTip("Preview")

        self.buttonRefresh = wgt.PushButtonImage(vrb.folderImages + "/Refresh.png", margins=0)
        self.buttonRefresh.setFixedSize(30 * vrb.ratio, 30 * vrb.ratio)
        self.buttonRefresh.setToolTip("Refresh")
        self.buttonRefresh.setVisible(False)

        groupBoxDisplaySettings = qt.QGroupBox()
        layoutDisplaySettings = qt.QGridLayout()
        layoutDisplaySettings.setContentsMargins(10, 10, 10, 10)
        layoutDisplaySettings.addWidget(self.labelPosition, 0, 0)
        layoutDisplaySettings.addWidget(self.comboBoxPosition, 0, 1)
        layoutDisplaySettings.addWidget(self.labelSizePercentage, 1, 0)
        layoutDisplaySettings.addWidget(self.lineEditSizePercentage, 1, 1, Qt.AlignRight)
        layoutDisplaySettings.addWidget(self.radioButtonVertical, 3, 0)
        layoutDisplaySettings.addWidget(self.radioButtonHorizontal, 3, 1)
        groupBoxDisplaySettings.setLayout(layoutDisplaySettings)

        groupBoxDisplaySettings.setTitle("Display settings :")
        groupBoxDisplaySettings.setStyleSheet(
            'QGroupBox:title {left: 20px ;padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {font: bold; border: 1px solid gray; margin-top: 12 px}')

        groupBoxColorSettings = qt.QGroupBox()
        layoutColorSettings = qt.QGridLayout()
        layoutColorSettings.setContentsMargins(10, 10, 10, 10)
        layoutColorSettings.addWidget(self.labelTextColor, 0, 0)
        layoutColorSettings.addWidget(self.buttonTextColor, 0, 1)
        layoutColorSettings.addWidget(self.labelBackgroundColor, 1, 0)
        layoutColorSettings.addWidget(self.buttonBackgroundColor, 1, 1)
        groupBoxColorSettings.setLayout(layoutColorSettings)
        #groupBoxColorSettings.setFixedSize(220 * vrb.ratio, 150 * vrb.ratio)
        groupBoxColorSettings.setTitle("Colors :")
        groupBoxColorSettings.setStyleSheet(
            'QGroupBox:title {left: 20px ;padding-left: 10px;padding-right: 10px; padding-top: -12px; color:rgb(6, 115, 186)}  QGroupBox {font: bold; border: 1px solid gray; margin-top: 12 px}')

        groupBoxValidateButtons = qt.QGroupBox()

        layoutValidateButtons = qt.QHBoxLayout()
        layoutValidateButtons.setContentsMargins(0, 0, 0, 0)
        layoutValidateButtons.addStretch()
        layoutValidateButtons.addWidget(self.buttonRefresh)
        layoutValidateButtons.addWidget(self.buttonPreview)
        layoutValidateButtons.addWidget(self.buttonValidate)
        groupBoxValidateButtons.setLayout(layoutValidateButtons)

        self.layout = qt.QVBoxLayout()
        self.layout.addWidget(groupBoxDisplaySettings)
        self.layout.addWidget(groupBoxColorSettings)
        self.layout.addWidget(groupBoxValidateButtons)

        self.layout.setAlignment(Qt.AlignTop)

        self.setLayout(self.layout)
        self.layout.setSizeConstraint(1)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setWindowTitle("Calibration Display Settings")
        style = fct.getStyleSheet()
        self.setStyleSheet(style)

        self.loadXmlElement()

        self.buttonTextColor.clicked.connect(self.getColor)
        self.buttonBackgroundColor.clicked.connect(self.getColor)
        self.buttonPreview.SignalWidgetSwitched.connect(self.previewColorCalibration)
        self.buttonRefresh.clicked.connect(self.refreshColorCalibration)
        self.buttonValidate.clicked.connect(self.applyColorCalibration)

    def closeEvent(self, a0: QtGui.QCloseEvent):

        checkBoxOverlay = vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.isChecked()
        currentText = vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.currentText()
        vrb.mainWindow.changeCurrentXmlElement(vrb.mainWindow.currentLabel)
        if checkBoxOverlay == True:
            vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.setChecked(checkBoxOverlay)
            vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.setCurrentText(currentText)

        self.buttonRefresh.setVisible(False)
        self.buttonPreview.setState(0)

    def writeXmlElement(self):

        Dfct.createSubElementComboBox(self.xmlElement,self.comboBoxPosition,"Position")
        Dfct.createSubElementLineEdit(self.xmlElement,self.lineEditSizePercentage,"Size")
        if self.radioButtonHorizontal.isChecked():
            Dfct.SubElement(self.xmlElement,"Direction").text = "Horizontal"
        else:
            Dfct.SubElement(self.xmlElement,"Direction").text = "Vertical"

    def loadXmlElement(self):

        Dfct.readElementComboBox(self.xmlElement,self.comboBoxPosition,"Position")
        Dfct.readElementLineEdit(self.xmlElement,self.lineEditSizePercentage,"Size")
        if Dfct.childText(self.xmlElement,"Direction") == "Horizontal":
            self.radioButtonHorizontal.setChecked(True)
        else:
            self.radioButtonVertical.setChecked(True)

        self.loadXmlColorElement()

    def getColor(self):

        self.setVisible(False)

        color = qt.QColorDialog.getColor()

        self.setVisible(True)

        if (color.isValid()):
            if self.sender() == self.buttonTextColor:
                textColorElement = Dfct.SubElement(self.xmlElement, "TextColor")
                textColorElement.text = str(color.red())+","+str(color.green())+","+str(color.blue())
            if self.sender() == self.buttonBackgroundColor:
                backgroundColorElement = Dfct.SubElement(self.xmlElement, "BackgroundColor")
                backgroundColorElement.text = str(color.red())+","+str(color.green())+","+str(color.blue())

        self.loadXmlColorElement()

    def loadXmlColorElement(self):

        textColorElement = Dfct.SubElement(self.xmlElement, "TextColor")
        if textColorElement.text is not None:
            textColorElementSplit = textColorElement.text.split(",")
            self.buttonTextColor.setStyleSheet(
                'QPushButton {background-color: rgb(' + str(textColorElementSplit[0]) + ',' + str(
                    textColorElementSplit[1]) + ',' + str(textColorElementSplit[2]) + ');}')

        backgroundColorElement = Dfct.SubElement(self.xmlElement, "BackgroundColor")
        if backgroundColorElement.text is not None:
            backgroundColorElementSplit = backgroundColorElement.text.split(",")
            self.buttonBackgroundColor.setStyleSheet('QPushButton {background-color: rgb(' + str(backgroundColorElementSplit[0]) + ',' + str(
                backgroundColorElementSplit[1]) + ',' + str(backgroundColorElementSplit[2]) + ');}')

    def previewColorCalibration(self):

        try:
            self.writeXmlElement()
            Dfct.saveXmlElement(self.xmlElement, vrb.folderInformation + "/ColorCalibrationSettings.mho")

            if self.buttonPreview.state == 1:
                checkBoxOverlay = vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.isChecked()
                currentText = vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.currentText()
                vrb.mainWindow.changeCurrentXmlElement(vrb.mainWindow.currentLabel)
                if checkBoxOverlay == True:
                    vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.setChecked(checkBoxOverlay)
                    vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.setCurrentText(currentText)
                self.buttonRefresh.setVisible(True)
                imageColorCalibration = self.createImageWithColorCalibration()
                vrb.mainWindow.widgetImage.setImage(imageColorCalibration)
            else:
                self.buttonRefresh.setVisible(False)
                checkBoxOverlay = vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.isChecked()
                currentText = vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.currentText()
                vrb.mainWindow.changeCurrentXmlElement(vrb.mainWindow.currentLabel)
                if checkBoxOverlay == True:
                    vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.setChecked(checkBoxOverlay)
                    vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.setCurrentText(currentText)
        except:
            traceback.print_exc(file=sys.stderr)

    def refreshColorCalibration(self):

        try:
            self.writeXmlElement()
            Dfct.saveXmlElement(self.xmlElement, vrb.folderInformation + "/ColorCalibrationSettings.mho")
            checkBoxOverlay = vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.isChecked()
            currentText = vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.currentText()
            vrb.mainWindow.changeCurrentXmlElement(vrb.mainWindow.currentLabel)
            if checkBoxOverlay == True:
                vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.setChecked(checkBoxOverlay)
                vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.setCurrentText(currentText)
            imageColorCalibration = self.createImageWithColorCalibration()
            vrb.mainWindow.widgetImage.setImage(imageColorCalibration)
        except:
            traceback.print_exc(file=sys.stderr)

    def applyColorCalibration(self):

        try:
            self.writeXmlElement()
            Dfct.saveXmlElement(self.xmlElement, vrb.folderInformation + "/ColorCalibrationSettings.mho")

            checkBoxOverlay = vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.isChecked()
            currentText = vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.currentText()
            vrb.mainWindow.changeCurrentXmlElement(vrb.mainWindow.currentLabel)
            if checkBoxOverlay == True:
                vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.setChecked(checkBoxOverlay)
                vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.setCurrentText(currentText)

            if vrb.mainWindow.currentLabel.image.getSizeC() > 1:
                self.messageBox = wgt.MessageBox("You can't create a color calibration with a color/multichannel image", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                self.messageBox.exec()
                return
            elif vrb.mainWindow.currentLabel.image.getSizeZ() > 1:
                self.messageBox = wgt.MessageBox("You can't create a color calibration with a 3D image", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                self.messageBox.exec()
                return
            elif vrb.mainWindow.currentLabel.image.getSizeT() > 1:
                self.messageBox = wgt.MessageBox("You can't create a color calibration with a sequence image", '', buttons=[qt.QMessageBox.Ok], icon=qt.QMessageBox.Warning, windowTitle="Error")
                self.messageBox.exec()
                return
            else:
                imageColorCalibration = self.createImageWithColorCalibration()

                outputDict = {}
                outputDict["Name"] = ""
                outputDict["Type"] = "Image"

                functionXmlElement = xmlet.Element('FunctionCall')
                Dfct.SubElement(functionXmlElement, 'Name').text = 'CreateColorCalibrationImage'
                # paramsNode = Dfct.SubElement(functionXmlElement, 'Parameters')
                # paramNode = Dfct.SubElement(paramsNode, 'Parameter_0')
                # Dfct.SubElement(paramNode, 'Value').text = str(polygon)

                label, realOutputName = vrb.mainWindow.addResult(imageColorCalibration, outputDict, functionXmlElement)

        except:
            traceback.print_exc(file=sys.stderr)

    def createImageWithColorCalibration(self):

        currentLabel = vrb.mainWindow.currentLabel
        image = currentLabel.image

        if vrb.mainWindow.widgetImage.groupBoxOverlay.checkBoxOverlay.isChecked() and vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.count()>=1:
            imageLabelOverlay = vrb.mainWindow.widgetImage.groupBoxOverlay.comboBoxOverlay.currentData()
            imageOverlay = imageLabelOverlay.image
        else:
            imageOverlay = None

        if imageOverlay is not None:
            lutVector = imageLabelOverlay.groupBoxLut.labelLut.image
            valueMin = fct.numberCalibration(imageLabelOverlay.valueMin)
            valueMax = fct.numberCalibration(imageLabelOverlay.valueMax)
        else:
            lutVector = currentLabel.groupBoxLut.labelLut.image
            valueMin = fct.numberCalibration(currentLabel.valueMin)
            valueMax = fct.numberCalibration(currentLabel.valueMax)

        position = self.comboBoxPosition.currentText()
        size = self.lineEditSizePercentage.text()
        if size == "":
            size = int(self.lineEditSizePercentage.placeholderText())
        else:
            size = int(size)
        if self.radioButtonHorizontal.isChecked():
            direction = "Horizontal"
        else:
            direction = "Vertical"

        textColor = Dfct.childText(self.xmlElement, "TextColor")
        if textColor is None:
            textColor = "0,0,0"
        textColorSplit = textColor.split(",")
        textColor = (int(textColorSplit[0]), int(textColorSplit[1]), int(textColorSplit[2]),255)

        backgroundColor = Dfct.childText(self.xmlElement, "BackgroundColor")
        if backgroundColor is None:
            backgroundColor = "255,255,255"
        backgroundColorSplit = backgroundColor.split(",")
        backgroundColor = [int(backgroundColorSplit[0]), int(backgroundColorSplit[1]), int(backgroundColorSplit[2])]

        imageColorCalibration = fct.integrateColorCalibration(image, imageOverlay, lutVector, valueMin, valueMax, position=position, size=size,
                                      direction=direction, textColor=textColor, backgroundColor=backgroundColor)

        return imageColorCalibration

if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])


    sys._excepthook = sys.excepthook


    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    foo = ColorCalibrationSettingsWidget()

    foo.show()

    app.exec_()
