import sys
import os
import traceback

from PyQt5.QtCore import pyqtSignal,QPointF,Qt,QCoreApplication
from PyQt5 import QtGui
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore

import xml.etree.ElementTree as xmlet

import UsefullWidgets as wgt
import UsefullVariables as vrb
import DatabaseFunction as Dfct
import UsefullFunctions as fct
import UsefullTexts as txt

import SettingsWidget

import PyIPSDK.IPSDKIPLGlobalMeasure as glbmsr
import PyIPSDK

import numpy as np

class GroupBoxCalibration(qt.QGroupBox):

    def __init__(self, mainWindow=None,parent=None):
        qt.QGroupBox.__init__(self)

        self.parent = parent
        self.mainWindow = mainWindow
        self.defaultCalibration = None
        self.refWidth = 300*vrb.ratio
        self.secondWidth = 371*vrb.ratio

        self.numberMeaningful = 4

        self.labelCalibration = qt.QLabel("Calibration : 1px = 1px")
        font = QtGui.QFont()
        font.setPixelSize(12*vrb.ratio)
        font.setBold(True)
        self.labelCalibration.setFont(font)
        self.labelCalibration.setFixedSize(250*vrb.ratio, 30*vrb.ratio)
        self.labelCalibration.setAlignment(Qt.AlignCenter)

        self.buttonCalibration = wgt.PushButtonImage(vrb.folderImages + "/Labels.png",margins=10)
        self.buttonCalibration.setFixedSize(60*vrb.ratio, 25*vrb.ratio)
        self.buttonCalibration.setToolTip(txt.dictToolTips["CalibrationWindow"])

        self.buttonCalibrationSettings = wgt.PushButtonImage(vrb.folderImages + "/calibration_settings.png")
        self.buttonCalibrationSettings.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)
        self.buttonCalibrationSettings.setToolTip(txt.dictToolTips["calibration_settings"])

        self.buttonApplyCalibrationSettings = wgt.PushButtonImage(vrb.folderImages + "/calibration_apply.png")
        self.buttonApplyCalibrationSettings.setFixedSize(25 * vrb.ratio, 25 * vrb.ratio)
        self.buttonApplyCalibrationSettings.setToolTip(txt.dictToolTips["calibration_apply"])

        self.widgetCalibration = WidgetCalibration(mainWindow = mainWindow)
        self.widgetCalibrationSettings = SettingsWidget.calibrationSettingsWidget()
        self.widgetCalibrationSettings.loadComboboxTypes()

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelCalibration, 0, 0)
        self.layout.addWidget(self.buttonCalibration, 0, 1)
        self.layout.addWidget(self.buttonCalibrationSettings, 0, 2)
        self.layout.addWidget(self.buttonApplyCalibrationSettings, 0, 3)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(5*vrb.ratio, 0, 5*vrb.ratio, 0)
        self.layout.setHorizontalSpacing(3*vrb.ratio)

        self.setStyleSheet("QGroupBox {border: 1px solid gray; }")

        self.setFixedSize(351*vrb.ratio, 40*vrb.ratio)

        self.buttonCalibration.clicked.connect(self.showWidgetCalibration)
        self.buttonCalibrationSettings.clicked.connect(self.showWidgetCalibrationSettings)
        self.buttonApplyCalibrationSettings.clicked.connect(self.widgetCalibrationSettings.validate)
        self.widgetCalibration.groupBoxValidate.buttonValidate.clicked.connect(self.applyCalibration)
        self.widgetCalibration.groupBoxValidate.buttonCancel.clicked.connect(self.cancelCalibration)

    def showWidgetCalibration(self):

        self.widgetCalibration.showNormal()
        self.widgetCalibration.activateWindow()
        self.imageToInterfaceCalibration()

    def showWidgetCalibrationSettings(self):
        self.widgetCalibrationSettings.loadComboboxTypes()
        self.widgetCalibrationSettings.showNormal()
        self.widgetCalibrationSettings.activateWindow()

    def cancelCalibration(self):

        self.loadImageCalibration()
        self.widgetCalibration.close()
        self.mainWindow.imageViewer.getRoiImage(changeRoiImage=False)

    def applyCalibration(self):

        groupBoxDefineCalibration = self.widgetCalibration.tabWidgetCalibration.currentWidget().groupBoxDefineCalibration
        self.defaultCalibration = xmlet.Element('Default_Calibration')
        groupBoxDefineCalibration.interfaceToXml(self.defaultCalibration)

        try:
            groupBoxDefineCalibration = self.widgetCalibration.tabWidgetCalibration.currentWidget().groupBoxDefineCalibration
            if groupBoxDefineCalibration.groupTypeCalibration.comboBoxPixelType.currentIndex() == 0:
                xScale = float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquareValue.text())/max(1,float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquarePixels.text()))
                yScale = float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquareValue.text())/max(1,float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquarePixels.text()))
                zScale = float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquareValue.text())/max(1,float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquarePixels.text()))
            else:
                xScale = float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueX.text()) / max(1,float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsX.text()))
                yScale = float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueY.text()) / max(1,float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsY.text()))
                zScale = float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueZ.text()) / max(1,float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsZ.text()))
            unit = groupBoxDefineCalibration.groupTypeCalibration.comboBoxUnit.currentText()

            currentImage = self.mainWindow.currentLabel.image
            currentImage.setScaleGeometricCalibration(float(fct.numberCalibration(xScale,self.numberMeaningful)), float(fct.numberCalibration(yScale,self.numberMeaningful)),
                                                         float(fct.numberCalibration(zScale,self.numberMeaningful)),unit)

            for num in range(self.mainWindow.widgetLabelImage.layout.count()):
                item = self.mainWindow.widgetLabelImage.layout.itemAt(num)
                if item is not None:
                    image = item.widget().image
                    if image != currentImage and image.hasGeometricCalibration():
                        calib = image.getGeometricCalibration()
                        if calib.getUnitStr() == "px" and calib.getXScale() == 1 and calib.getYScale() == 1 and calib.getZScale() == 1:
                            image.setScaleGeometricCalibration(float(fct.numberCalibration(xScale,self.numberMeaningful)), float(fct.numberCalibration(yScale,self.numberMeaningful)),
                                                      float(fct.numberCalibration(zScale,self.numberMeaningful)),unit)

        except:
            #traceback.print_exc(file=sys.stderr)
            pass

        self.widgetCalibration.tabWidgetCalibration.groupBoxUserCalibration.saveCalibration()

        self.loadImageCalibration()
        self.widgetCalibration.close()
        self.mainWindow.imageViewer.getRoiImage(changeRoiImage=False)
        self.mainWindow.updateLineProfile()

        # for 3D visualization
        if vrb.mainWindow.currentLabel is not None and vrb.mainWindow.currentLabel.layer is not None:
            # scales the volume from a factor
            # vrb.mainWindow.currentLabel.layer.scale = [xScale/xScale, yScale/xScale, zScale/xScale]
            # minScale = min(zScale, yScale, xScale)
            # vrb.mainWindow.currentLabel.layer.scale = [zScale/minScale, yScale/minScale, xScale/minScale]
            if zScale>0 or yScale>0 or xScale>0:
	            vrb.mainWindow.currentLabel.layer.scale = [zScale, yScale, xScale]
	            vrb.mainWindow.resetView()
            # vrb.mainWindow.viewerNapari.camera.zoom = vrb.mainWindow.viewerNapari.camera.zoom * ((xScale+yScale+zScale)/3)*2

            vrb.mainWindow.viewer3dSettingsWidget.loadSettings(vrb.mainWindow.currentLabel)
            vrb.mainWindow.viewer3dSettingsWidget.resetCliping()

            # scale the XYZ planes layers and the 3D ROI if it exist
            if vrb.mainWindow.widgetImage.modeViewer == "4Views":
                if vrb.mainWindow.viewer3dSettingsWidget.xPlane is not None and \
                vrb.mainWindow.viewer3dSettingsWidget.yPlane is not None and \
                vrb.mainWindow.viewer3dSettingsWidget.zPlane is not None:

                    vrb.mainWindow.viewer3dSettingsWidget.xPlane.scale = vrb.mainWindow.currentLabel.layer.scale
                    vrb.mainWindow.viewer3dSettingsWidget.yPlane.scale = vrb.mainWindow.currentLabel.layer.scale
                    vrb.mainWindow.viewer3dSettingsWidget.zPlane.scale = vrb.mainWindow.currentLabel.layer.scale

                if vrb.mainWindow.viewer3dSettingsWidget.roi3D is not None:
                    vrb.mainWindow.viewer3dSettingsWidget.roi3D.scale = vrb.mainWindow.currentLabel.layer.scale


    def loadImageCalibration(self):

        try:
            image = self.mainWindow.currentLabel.image

            if image.hasGeometricCalibration():
                calib = image.getGeometricCalibration()
                try:
                    unit = calib.getUnitStr()
                except:
                    unit = "px"
                if unit.lower() == "inch":
                    image.setGeometricCalibration(None)

            if image.hasGeometricCalibration():

                calib = image.getGeometricCalibration()
                try:
                    unit = calib.getUnitStr()
                except:
                    unit = "px"

                if unit == "":
                    unit = "px"

                xScale = calib.getXScale()
                yScale = calib.getYScale()
                zScale = calib.getZScale()

                defaultName = None
                epsilon = 0.00001
                for i in range(self.widgetCalibration.tabWidgetCalibration.groupBoxUserCalibration.comboBoxCalibration.count()):
                    element = self.widgetCalibration.tabWidgetCalibration.groupBoxUserCalibration.comboBoxCalibration.itemData(i)
                    if unit == Dfct.childText(element,"Unit"):
                        if Dfct.childText(element,"Choice") == str(0):
                            ratio = float(Dfct.childText(element,"SquareValue"))/float(Dfct.childText(element,"SquarePixel"))
                            if abs(ratio-xScale)<epsilon and abs(ratio-yScale)<epsilon and (image.getVolumeGeometryType() == PyIPSDK.eVGT_2d or (image.getVolumeGeometryType() == PyIPSDK.eVGT_3d and abs(ratio-zScale)<epsilon)):
                                defaultName = Dfct.childText(element,"Name")
                        if Dfct.childText(element,"Choice") == str(1):
                            ratioX = float(Dfct.childText(element, "XValue")) / float(Dfct.childText(element, "XPixel"))
                            ratioY = float(Dfct.childText(element, "YValue")) / float(Dfct.childText(element, "YPixel"))
                            ratioZ = float(Dfct.childText(element, "ZValue")) / float(Dfct.childText(element, "ZPixel"))
                            if abs(ratioX-xScale)<epsilon and abs(ratioY-yScale)<epsilon and (image.getVolumeGeometryType() == PyIPSDK.eVGT_2d or (image.getVolumeGeometryType() == PyIPSDK.eVGT_3d and abs(ratioZ-zScale)<epsilon)):
                                defaultName = Dfct.childText(element,"Name")
                if defaultName is not None:
                    self.labelCalibration.setText(defaultName)
                    self.labelCalibration.setFixedWidth(self.refWidth)
                    self.setFixedWidth(self.secondWidth)
                else:
                    if xScale == yScale and (image.getVolumeGeometryType() == PyIPSDK.eVGT_2d or (image.getVolumeGeometryType() == PyIPSDK.eVGT_3d and yScale==zScale)):
                        self.labelCalibration.setText("Calibration : 1px = "+fct.numberCalibration(xScale,self.numberMeaningful)+unit)
                        self.labelCalibration.setFixedWidth(self.refWidth)
                        self.setFixedWidth(self.secondWidth)
                    else:
                        text = "Calibration : "
                        text += "X : 1px = "+fct.numberCalibration(xScale,self.numberMeaningful)+unit+" , "
                        self.labelCalibration.setFixedWidth(self.refWidth)
                        self.setFixedWidth(self.secondWidth)
                        if image.getVolumeGeometryType() == PyIPSDK.eVGT_2d:
                            text += "Y : 1px = " + fct.numberCalibration(yScale, self.numberMeaningful) + unit
                            self.labelCalibration.setFixedWidth(self.refWidth+100*vrb.ratio)
                            self.setFixedWidth(self.secondWidth+100*vrb.ratio)
                        else:
                            text += "Y : 1px = "+fct.numberCalibration(yScale,self.numberMeaningful)+unit+" , "
                            text += "Z : 1px = "+fct.numberCalibration(zScale,self.numberMeaningful)+unit
                            self.labelCalibration.setFixedWidth(self.refWidth+200*vrb.ratio)
                            self.setFixedWidth(self.secondWidth+200*vrb.ratio)
                        self.labelCalibration.setText(text)
            else:
                self.labelCalibration.setText("No calibration")
                self.labelCalibration.setFixedWidth(self.refWidth)
                self.setFixedWidth(self.secondWidth)
        except:
            # traceback.print_exc(file=sys.stderr)
            self.applyInterfaceCalibration()

    def applyInterfaceCalibration(self):

        try:
            unit = Dfct.childText(self.defaultCalibration,"Unit")
            if Dfct.childText(self.defaultCalibration,"Choice") == "0":
                self.labelCalibration.setText("Calibration : " + Dfct.childText(self.defaultCalibration,"SquarePixel") + "px = "+Dfct.childText(self.defaultCalibration,"SquareValue")+unit)
                self.labelCalibration.setFixedWidth(self.refWidth)
                self.setFixedWidth(self.secondWidth)
            else:
                text = "Calibration : "
                text += "X : " + Dfct.childText(self.defaultCalibration,"XPixel") + "px = "+Dfct.childText(self.defaultCalibration,"XValue")+unit + " , "
                text += "Y : " + Dfct.childText(self.defaultCalibration,"YPixel") + "px = "+Dfct.childText(self.defaultCalibration,"YValue")+unit + " , "
                text += "Z : " + Dfct.childText(self.defaultCalibration,"ZPixel") + "px = "+Dfct.childText(self.defaultCalibration,"ZValue")+unit
                self.labelCalibration.setText(text)
                self.labelCalibration.setFixedWidth(self.refWidth + 200*vrb.ratio)
                self.setFixedWidth(self.secondWidth + 200*vrb.ratio)
        except:
            #traceback.print_exc(file=sys.stderr)
            self.labelCalibration.setText("Calibration : 1px = 1px")

    def applyDefaultCalibration(self):

        if self.defaultCalibration is not None:
            try:
                image = self.mainWindow.currentLabel.image
                groupBoxDefineCalibration = self.widgetCalibration.tabWidgetCalibration.currentWidget().groupBoxDefineCalibration
                if groupBoxDefineCalibration.groupTypeCalibration.comboBoxPixelType.currentIndex() == 0:
                    xScale = float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquareValue.text())/max(1,float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquarePixels.text()))
                    yScale = float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquareValue.text())/max(1,float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquarePixels.text()))
                    zScale = float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquareValue.text())/max(1,float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquarePixels.text()))
                else:
                    xScale = float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueX.text()) / max(1,float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsX.text()))
                    yScale = float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueY.text()) / max(1,float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsY.text()))
                    zScale = float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueZ.text()) / max(1,float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsZ.text()))
                unit = groupBoxDefineCalibration.groupTypeCalibration.comboBoxUnit.currentText()

                calib = PyIPSDK.createGeometricCalibration3d(float(fct.numberCalibration(xScale,self.numberMeaningful)), float(fct.numberCalibration(yScale,self.numberMeaningful)),
                                                             float(fct.numberCalibration(zScale,self.numberMeaningful)),unit)
                image.setGeometricCalibration(calib)
            except:
                #traceback.print_exc(file=sys.stderr)
                pass

    def getCalibration(self):

        groupBoxDefineCalibration = self.widgetCalibration.tabWidgetCalibration.currentWidget().groupBoxDefineCalibration
        if groupBoxDefineCalibration.groupTypeCalibration.comboBoxPixelType.currentIndex() == 0:
            xScale = float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquareValue.text()) / max(1, float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquarePixels.text()))
            yScale = float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquareValue.text()) / max(1, float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquarePixels.text()))
            zScale = float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquareValue.text()) / max(1, float(groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquarePixels.text()))
        else:
            xScale = float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueX.text()) / max(1, float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsX.text()))
            yScale = float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueY.text()) / max(1, float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsY.text()))
            zScale = float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueZ.text()) / max(1, float(groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsZ.text()))
        unit = groupBoxDefineCalibration.groupTypeCalibration.comboBoxUnit.currentText()

        calib = PyIPSDK.createGeometricCalibration3d(float(fct.numberCalibration(xScale, self.numberMeaningful)), float(fct.numberCalibration(yScale, self.numberMeaningful)),
                                                     float(fct.numberCalibration(zScale, self.numberMeaningful)), unit)

        return calib

    def imageToInterfaceCalibration(self):

        self.widgetCalibration.tabWidgetCalibration.setCurrentIndex(0)
        groupBoxDefineCalibration = self.widgetCalibration.tabWidgetCalibration.groupBoxCurrentCalibration.groupBoxDefineCalibration
        try:
            image = self.mainWindow.currentLabel.image
            if image.hasGeometricCalibration():
                calib = image.getGeometricCalibration()
                xScale = calib.getXScale()
                yScale = calib.getYScale()
                zScale = calib.getZScale()
                if xScale == yScale and (image.getVolumeGeometryType() == PyIPSDK.eVGT_2d or (image.getVolumeGeometryType() == PyIPSDK.eVGT_3d and yScale == zScale)):
                    groupBoxDefineCalibration.groupTypeCalibration.comboBoxPixelType.setCurrentIndex(0)
                    groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquarePixels.setText("1")
                    groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquareValue.setText(fct.numberCalibration(xScale, self.numberMeaningful))
                else:
                    groupBoxDefineCalibration.groupTypeCalibration.comboBoxPixelType.setCurrentIndex(1)
                    groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsX.setText("1")
                    groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueX.setText(fct.numberCalibration(xScale, self.numberMeaningful))
                    groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsY.setText("1")
                    groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueY.setText(fct.numberCalibration(yScale, self.numberMeaningful))
                    groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsZ.setText("1")
                    groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueZ.setText(fct.numberCalibration(zScale, self.numberMeaningful))
                try:
                    unit = calib.getUnitStr()
                except:
                    unit = "um"
                groupBoxDefineCalibration.groupTypeCalibration.comboBoxUnit.setCurrentText(unit)
                if groupBoxDefineCalibration.groupTypeCalibration.comboBoxUnit.currentIndex() == -1:
                    groupBoxDefineCalibration.groupTypeCalibration.comboBoxUnit.setCurrentIndex(0)
            self.widgetCalibration.tabWidgetCalibration.groupBoxCurrentCalibration.labelCurrentCalibration.setText("Calibration of "+Dfct.childText(self.mainWindow.currentLabel.xmlElement,"Name"))
        except:
            #traceback.print_exc(file=sys.stderr)
            pass

class WidgetCalibration(qt.QWidget):

    def __init__(self, mainWindow=None):
        qt.QWidget.__init__(self)

        try:
            self.setWindowFlag(Qt.WindowStaysOnTopHint)
        except:
            self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.mainWindow = mainWindow

        self.tabWidgetCalibration = TabWidgetCalibration()
        self.groupBoxValidate = GroupBoxValidate()

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.tabWidgetCalibration, 0, 0)
        self.layout.addWidget(self.groupBoxValidate, 1, 0,Qt.AlignLeft)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(10, 10, 10, 10)
        self.layout.setHorizontalSpacing(3)
        self.layout.setVerticalSpacing(10)

        height = 20*vrb.ratio + 10*2*vrb.ratio +self.tabWidgetCalibration.height() + self.groupBoxValidate.height()

        if self.mainWindow is not None:
            self.tabWidgetCalibration.groupBoxCurrentCalibration.groupBoxDefineCalibration.groupBoxSquarePixels.buttonMeasureCalibration.clicked.connect(self.mainWindow.graphicElements.setCalibrationLine)
            self.tabWidgetCalibration.groupBoxUserCalibration.groupBoxDefineCalibration.groupBoxSquarePixels.buttonMeasureCalibration.clicked.connect(self.mainWindow.graphicElements.setCalibrationLine)

        style = fct.getStyleSheet()
        style += "QGroupBox {border: 1px solid gray; }"
        self.setStyleSheet(style)
        #self.setStyleSheet("QGroupBox {border: 1px solid gray; }")
        self.setWindowTitle("Calibration")
        self.setFixedSize(380*vrb.ratio, height)

    def closeEvent(self,event):

        self.mainWindow.graphicElements.stopCalibrationLine()

class TabWidgetCalibration(qt.QTabWidget):

    def __init__(self, parent=None):
        qt.QTabWidget.__init__(self)

        try:
            file = xmlet.parse(vrb.folderInformation + "/UserCalibrations.mho")
            self.calibrationsElement = file.getroot()
        except:
            self.calibrationsElement = xmlet.Element('Calibrations')
            newCalibration = xmlet.SubElement(self.calibrationsElement,"Calibration")
            createEmptyCalibration(newCalibration,name = "No calibration")

        self.groupBoxCurrentCalibration = GroupBoxCurrentCalibration(calibrationsElement=self.calibrationsElement)
        self.groupBoxUserCalibration = GroupBoxUserCalibration(calibrationsElement=self.calibrationsElement)

        self.addTab(self.groupBoxCurrentCalibration,"Current calibration")
        self.addTab(self.groupBoxUserCalibration,"User calibration")

        self.setFixedSize(360*vrb.ratio,310*vrb.ratio)

        self.groupBoxCurrentCalibration.groupBoxDefineCalibration.groupBoxSave.buttonSave.clicked.connect(self.addAndSaveCalibration)

    def addAndSaveCalibration(self):

        element = xmlet.SubElement(self.calibrationsElement,"Calibration")
        name = self.groupBoxCurrentCalibration.groupBoxDefineCalibration.groupBoxSave.lineEditSave.text()
        if name == "":
            name = "New calibration"
        self.groupBoxCurrentCalibration.groupBoxDefineCalibration.interfaceToXml(element,name=name)
        Dfct.saveXmlElement(self.calibrationsElement, vrb.folderInformation + "/UserCalibrations.mho")
        index = self.groupBoxUserCalibration.comboBoxCalibration.currentIndex()
        self.groupBoxUserCalibration.fillComboBoxWithElement()
        self.groupBoxUserCalibration.comboBoxCalibration.setCurrentIndex(index)

        self.groupBoxCurrentCalibration.groupBoxDefineCalibration.groupBoxSave.buttonSave.setEnabled(False)

def createEmptyCalibration(element,name = "New calibration"):

    Dfct.SubElement(element, "Name").text = name
    Dfct.SubElement(element, "Choice").text = "0"
    Dfct.SubElement(element, "Unit").text = "px"
    Dfct.SubElement(element, "SquarePixel").text = "1"
    Dfct.SubElement(element, "SquareValue").text = "1"
    Dfct.SubElement(element, "XPixel").text = "1"
    Dfct.SubElement(element, "XValue").text = "1"
    Dfct.SubElement(element, "YPixel").text = "1"
    Dfct.SubElement(element, "YValue").text = "1"
    Dfct.SubElement(element, "ZPixel").text = "1"
    Dfct.SubElement(element, "ZValue").text = "1"

class GroupBoxUserCalibration(qt.QGroupBox):

    def __init__(self, calibrationsElement=None,parent=None):

        qt.QGroupBox.__init__(self)

        self.loading = False

        self.calibrationsElement = calibrationsElement

        self.labelUserCalibration = qt.QLabel("User calibration ")
        font = QtGui.QFont()
        font.setPixelSize(14*vrb.ratio)
        font.setBold(True)
        self.labelUserCalibration.setFont(font)
        self.labelUserCalibration.setContentsMargins(5,0,0,0)
        self.labelUserCalibration.setFixedSize(112*vrb.ratio,30*vrb.ratio)

        self.comboBoxCalibration = qt.QComboBox()
        self.comboBoxCalibration.setFixedSize(145*vrb.ratio,30*vrb.ratio)

        self.buttonAddCalibration = wgt.PushButtonImage(vrb.folderImages + "/Add_Calibration.png")
        self.buttonAddCalibration.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        self.buttonAddCalibration.setToolTip(txt.dictToolTips["NewCalibration"])

        self.buttonDeleteCalibration = wgt.PushButtonImage(vrb.folderImages + "/Delete_Calibration.png")
        self.buttonDeleteCalibration.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        self.buttonDeleteCalibration.setToolTip(txt.dictToolTips["DeleteCalibration"])

        self.groupBoxDefineCalibration = GroupBoxDefineCalibration(type="user")

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelUserCalibration, 0, 0)
        self.layout.addWidget(self.comboBoxCalibration, 0, 1)
        self.layout.addWidget(self.buttonAddCalibration, 0, 2)
        self.layout.addWidget(self.buttonDeleteCalibration, 0, 3)
        self.layout.addWidget(self.groupBoxDefineCalibration, 1, 0,1,4)
        self.layout.setAlignment(Qt.AlignLeft)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3, 3, 3, 3)
        self.layout.setHorizontalSpacing(3)
        self.layout.setVerticalSpacing(8)

        self.setStyleSheet("QGroupBox {border: 0px transparent;}")
        # self.setStyleSheet("QGroupBox {border: 0px transparent; background:rgb(220, 220, 220) }")

        height = 6*vrb.ratio + 3*3*vrb.ratio + self.groupBoxDefineCalibration.height()

        self.setFixedSize(360*vrb.ratio, height + 60*vrb.ratio)

        self.fillComboBoxWithElement()

        self.buttonAddCalibration.clicked.connect(self.addCalibration)
        self.buttonDeleteCalibration.clicked.connect(self.deleteCalibration)
        self.groupBoxDefineCalibration.groupBoxSave.buttonSave.clicked.connect(self.saveCalibration)
        self.comboBoxCalibration.currentIndexChanged.connect(self.changeComboBoxIndex)

        self.groupBoxDefineCalibration.groupTypeCalibration.comboBoxPixelType.currentIndexChanged.connect(self.enableSaveCalibration)
        self.groupBoxDefineCalibration.groupTypeCalibration.comboBoxUnit.currentIndexChanged.connect(self.enableSaveCalibration)
        self.groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsX.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsY.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditPixelsZ.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueX.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueY.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxDefineCalibration.groupBoxRectangularPixels.lineEditValueZ.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquarePixels.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxDefineCalibration.groupBoxSquarePixels.lineEditSquareValue.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxDefineCalibration.groupBoxSave.lineEditSave.textChanged.connect(self.enableSaveCalibration)

        self.changeComboBoxIndex()

    def enableSaveCalibration(self):

        self.groupBoxDefineCalibration.groupBoxSave.buttonSave.setEnabled(True)

    def fillComboBoxWithElement(self):

        self.comboBoxCalibration.clear()
        for child in self.calibrationsElement:
            self.comboBoxCalibration.addItem(Dfct.childText(child,"Name"),child)
        Dfct.saveXmlElement(self.calibrationsElement, vrb.folderInformation + "/UserCalibrations.mho")

    def addCalibration(self):

        newCalibration = xmlet.SubElement(self.calibrationsElement, "Calibration")
        createEmptyCalibration(newCalibration)

        self.loading = True
        self.fillComboBoxWithElement()
        self.loading = False
        self.comboBoxCalibration.setCurrentIndex(self.comboBoxCalibration.count()-1)
        Dfct.saveXmlElement(self.calibrationsElement, vrb.folderInformation + "/UserCalibrations.mho")

        self.enableSaveCalibration()

    def changeComboBoxIndex(self):

        if self.loading == False:
            self.buttonDeleteCalibration.setEnabled(self.comboBoxCalibration.currentIndex()!=0)
            self.groupBoxDefineCalibration.setEnabled(self.comboBoxCalibration.currentIndex()!=0)

            element = self.comboBoxCalibration.currentData()
            self.groupBoxDefineCalibration.xmlToInterface(element)

            self.enableSaveCalibration()

    def saveCalibration(self):

        element = self.comboBoxCalibration.currentData()
        name = self.groupBoxDefineCalibration.groupBoxSave.lineEditSave.text()
        if name == "":
            name = "New calibration"
        self.groupBoxDefineCalibration.interfaceToXml(element,name=name)
        Dfct.saveXmlElement(self.calibrationsElement, vrb.folderInformation + "/UserCalibrations.mho")
        index = self.comboBoxCalibration.currentIndex()
        self.fillComboBoxWithElement()
        self.comboBoxCalibration.setCurrentIndex(index)

        self.groupBoxDefineCalibration.groupBoxSave.buttonSave.setEnabled(False)

    def deleteCalibration(self):

        index = self.comboBoxCalibration.currentIndex()
        Dfct.removeElement(self.calibrationsElement, self.comboBoxCalibration.currentData())
        Dfct.saveXmlElement(self.calibrationsElement, vrb.folderInformation + "/UserCalibrations.mho")
        self.fillComboBoxWithElement()
        self.comboBoxCalibration.setCurrentIndex(min(index,self.comboBoxCalibration.count()-1))

        self.enableSaveCalibration()

class GroupBoxCurrentCalibration(qt.QGroupBox):

    def __init__(self, calibrationsElement,parent=None):
        qt.QGroupBox.__init__(self)

        self.calibrationsElement = calibrationsElement

        self.labelCurrentCalibration = qt.QLabel("Calibration of the current image")
        font = QtGui.QFont()
        font.setPixelSize(14*vrb.ratio)
        font.setBold(True)
        self.labelCurrentCalibration.setFont(font)
        self.labelCurrentCalibration.setContentsMargins(5,0,0,0)
        self.labelCurrentCalibration.setFixedSize(400*vrb.ratio,30*vrb.ratio)

        self.groupBoxDefineCalibration = GroupBoxDefineCalibration(type ="current")

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelCurrentCalibration, 0, 0)
        self.layout.addWidget(self.groupBoxDefineCalibration, 1, 0)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3, 3, 3, 3)
        self.layout.setHorizontalSpacing(3)
        self.layout.setVerticalSpacing(8)

        self.setStyleSheet("QGroupBox {border: 0px transparent;}")
        # self.setStyleSheet("QGroupBox {border: 0px transparent; background:rgb(220, 220, 220) }")

        height = 6*vrb.ratio + 3*3*vrb.ratio + self.groupBoxDefineCalibration.height()

        self.setFixedSize(360*vrb.ratio, height + 60*vrb.ratio)

class GroupBoxDefineCalibration(qt.QGroupBox):

    def __init__(self, parent=None, type = "current"):
        qt.QGroupBox.__init__(self)

        self.groupTypeCalibration = GroupBoxTypeCalibration()
        self.emptyLabel = qt.QLabel()
        self.emptyLabel.setFixedHeight(3*vrb.ratio)
        self.groupBoxSquarePixels = GroupBoxSquarePixels()
        self.groupBoxRectangularPixels = GroupBoxRectangularPixels()
        self.groupBoxRectangularPixels.setVisible(False)
        self.groupBoxSave = GroupBoxSave(type=type)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.groupTypeCalibration, 0, 0)
        self.layout.addWidget(self.emptyLabel, 1, 0)
        self.layout.addWidget(self.groupBoxSquarePixels, 2, 0,Qt.AlignTop)
        self.layout.addWidget(self.groupBoxRectangularPixels, 2, 0,Qt.AlignTop)
        self.layout.addWidget(self.groupBoxSave, 3, 0)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3, 3, 3, 3)
        self.layout.setHorizontalSpacing(3)
        self.layout.setVerticalSpacing(8)

        self.setStyleSheet("QGroupBox {border: 0px transparent;}")

        height = 6*vrb.ratio + 3*3*vrb.ratio + self.groupTypeCalibration.height() + self.groupBoxSquarePixels.height() + self.groupBoxSave.height()

        self.setFixedSize(360*vrb.ratio, height + 10*vrb.ratio)

        self.groupTypeCalibration.comboBoxPixelType.currentIndexChanged.connect(self.changePixelsType)
        self.groupTypeCalibration.comboBoxUnit.currentIndexChanged.connect(self.changePixelsUnit)

        self.groupTypeCalibration.comboBoxPixelType.currentIndexChanged.connect(self.enableSaveCalibration)
        self.groupTypeCalibration.comboBoxUnit.currentIndexChanged.connect(self.enableSaveCalibration)
        self.groupBoxRectangularPixels.lineEditPixelsX.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxRectangularPixels.lineEditPixelsY.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxRectangularPixels.lineEditPixelsZ.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxRectangularPixels.lineEditValueX.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxRectangularPixels.lineEditValueY.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxRectangularPixels.lineEditValueZ.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxSquarePixels.lineEditSquarePixels.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxSquarePixels.lineEditSquareValue.textChanged.connect(self.enableSaveCalibration)
        self.groupBoxSave.lineEditSave.textChanged.connect(self.enableSaveCalibration)

    def enableSaveCalibration(self):

        self.groupBoxSave.buttonSave.setEnabled(True)

    def changePixelsType(self):

        if self.groupTypeCalibration.comboBoxPixelType.currentIndex() == 0:
            self.groupBoxSquarePixels.setVisible(True)
            self.groupBoxRectangularPixels.setVisible(False)
        if self.groupTypeCalibration.comboBoxPixelType.currentIndex() == 1:
            self.groupBoxSquarePixels.setVisible(False)
            self.groupBoxRectangularPixels.setVisible(True)

    def changePixelsUnit(self):

        for label in [self.groupBoxSquarePixels.labelSquareValue,self.groupBoxRectangularPixels.labelValueX,self.groupBoxRectangularPixels.labelValueY,self.groupBoxRectangularPixels.labelValueZ]:
            label.setText(self.groupTypeCalibration.comboBoxUnit.currentText())

    def interfaceToXml(self,element,name = "New calibration"):

        Dfct.SubElement(element, "Name").text = name
        Dfct.SubElement(element, "Choice").text = str(self.groupTypeCalibration.comboBoxPixelType.currentIndex())
        Dfct.SubElement(element, "Unit").text = str(self.groupTypeCalibration.comboBoxUnit.currentText())
        Dfct.SubElement(element, "SquarePixel").text = self.groupBoxSquarePixels.lineEditSquarePixels.text()
        Dfct.SubElement(element, "SquareValue").text = self.groupBoxSquarePixels.lineEditSquareValue.text()
        Dfct.SubElement(element, "XPixel").text = self.groupBoxRectangularPixels.lineEditPixelsX.text()
        Dfct.SubElement(element, "XValue").text = self.groupBoxRectangularPixels.lineEditValueX.text()
        Dfct.SubElement(element, "YPixel").text = self.groupBoxRectangularPixels.lineEditPixelsY.text()
        Dfct.SubElement(element, "YValue").text = self.groupBoxRectangularPixels.lineEditValueY.text()
        Dfct.SubElement(element, "ZPixel").text = self.groupBoxRectangularPixels.lineEditPixelsZ.text()
        Dfct.SubElement(element, "ZValue").text = self.groupBoxRectangularPixels.lineEditValueZ.text()
        for child in element:
            if child.text == "" or child.text is None:
                child.text = "1"

    def xmlToInterface(self,element):

        try:
            self.groupBoxSave.lineEditSave.setText(Dfct.childText(element, "Name"))
            self.groupTypeCalibration.comboBoxPixelType.setCurrentIndex(int(Dfct.childText(element, "Choice")))
            self.groupTypeCalibration.comboBoxUnit.setCurrentText(Dfct.childText(element, "Unit"))
            self.groupBoxSquarePixels.lineEditSquarePixels.setText(Dfct.childText(element, "SquarePixel"))
            self.groupBoxSquarePixels.lineEditSquareValue.setText(Dfct.childText(element, "SquareValue"))
            self.groupBoxRectangularPixels.lineEditPixelsX.setText(Dfct.childText(element, "XPixel"))
            self.groupBoxRectangularPixels.lineEditValueX.setText(Dfct.childText(element, "XValue"))
            self.groupBoxRectangularPixels.lineEditPixelsY.setText(Dfct.childText(element, "YPixel"))
            self.groupBoxRectangularPixels.lineEditValueY.setText(Dfct.childText(element, "YValue"))
            self.groupBoxRectangularPixels.lineEditPixelsZ.setText(Dfct.childText(element, "ZPixel"))
            self.groupBoxRectangularPixels.lineEditValueZ.setText(Dfct.childText(element, "ZValue"))
        except:
            pass


class GroupBoxTypeCalibration(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        totalWidth = 0

        self.labelPixelType = qt.QLabel("Pixel type : ")
        self.labelPixelType.setFixedSize(80*vrb.ratio, 30*vrb.ratio)
        self.labelPixelType.setAlignment(Qt.AlignRight | Qt.AlignVCenter)
        totalWidth += 80*vrb.ratio

        self.comboBoxPixelType = qt.QComboBox()
        self.comboBoxPixelType.addItem("Square/Cubic")
        self.comboBoxPixelType.addItem("Rectangular")
        self.comboBoxPixelType.setFixedSize(130*vrb.ratio, 30*vrb.ratio)
        totalWidth += 130*vrb.ratio

        self.emptyLabel = qt.QLabel()
        self.emptyLabel.setFixedSize(5*vrb.ratio,30*vrb.ratio)
        totalWidth += 5*vrb.ratio

        self.labelUnit = qt.QLabel("Unit : ")
        self.labelUnit.setFixedSize(40*vrb.ratio, 30*vrb.ratio)
        self.labelUnit.setAlignment(Qt.AlignRight | Qt.AlignVCenter)
        totalWidth += 40*vrb.ratio

        self.comboBoxUnit = qt.QComboBox()
        self.comboBoxUnit.setFixedSize(50*vrb.ratio, 30*vrb.ratio)
        self.comboBoxUnit.addItem("px")
        self.comboBoxUnit.addItem("m")
        self.comboBoxUnit.addItem("cm")
        self.comboBoxUnit.addItem("mm")
        self.comboBoxUnit.addItem("um")
        self.comboBoxUnit.addItem("nm")
        self.comboBoxUnit.setCurrentIndex(4)
        totalWidth += 40*vrb.ratio

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelPixelType, 0, 0)
        self.layout.addWidget(self.comboBoxPixelType, 0, 1)
        self.layout.addWidget(self.emptyLabel, 0, 2)
        self.layout.addWidget(self.labelUnit, 0, 3)
        self.layout.addWidget(self.comboBoxUnit, 0, 4)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3, 0, 3, 0)
        self.layout.setHorizontalSpacing(3)

        totalWidth += 3 * 4*vrb.ratio + 8*vrb.ratio

        self.setStyleSheet("QGroupBox {border: 0px transparent; }")

        self.setFixedSize(totalWidth, 36*vrb.ratio)

class GroupBoxSquarePixels(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        totalWidth = 0

        self.lineEditSquarePixels = qt.QLineEdit()
        self.lineEditSquarePixels.setText("1")
        self.lineEditSquarePixels.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.lineEditSquarePixels.setAlignment(Qt.AlignRight | Qt.AlignVCenter)
        totalWidth += 60*vrb.ratio

        self.labelSquarePixels = qt.QLabel("pixels = ")
        self.labelSquarePixels.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.labelSquarePixels.setAlignment(Qt.AlignCenter)
        totalWidth += 60*vrb.ratio

        self.lineEditSquareValue = qt.QLineEdit()
        self.lineEditSquareValue.setText("1")
        self.lineEditSquareValue.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.lineEditSquareValue.setAlignment(Qt.AlignRight | Qt.AlignVCenter)
        totalWidth += 60*vrb.ratio

        self.labelSquareValue = qt.QLabel("um")
        self.labelSquareValue.setFixedSize(40*vrb.ratio, 30*vrb.ratio)
        self.labelSquareValue.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
        totalWidth += 60*vrb.ratio

        self.buttonMeasureCalibration = wgt.PushButtonImage(vrb.folderImages + "/Measure_Calibration.png")
        self.buttonMeasureCalibration.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        self.buttonMeasureCalibration.setToolTip(txt.dictToolTips["DrawCalibration"])
        totalWidth += 30*vrb.ratio

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.lineEditSquarePixels, 0, 0)
        self.layout.addWidget(self.labelSquarePixels, 0, 1)
        self.layout.addWidget(self.lineEditSquareValue, 0, 2)
        self.layout.addWidget(self.labelSquareValue, 0, 3)
        self.layout.addWidget(self.buttonMeasureCalibration, 0, 4)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3, 0, 3, 0)
        self.layout.setHorizontalSpacing(3)

        totalWidth += 3 * 4*vrb.ratio + 8*vrb.ratio

        self.setStyleSheet("QGroupBox {border: 1px solid gray; }")

        self.setFixedSize(totalWidth + 30*vrb.ratio, 122*vrb.ratio)

class GroupBoxRectangularPixels(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.labelX = qt.QLabel("X : ")
        self.labelX.setFixedSize(10*vrb.ratio, 30*vrb.ratio)

        self.lineEditPixelsX = qt.QLineEdit()
        self.lineEditPixelsX.setText("1")
        self.lineEditPixelsX.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.lineEditPixelsX.setAlignment(Qt.AlignRight | Qt.AlignVCenter)

        self.labelPixelsX = qt.QLabel("pixels = ")
        self.labelPixelsX.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.labelPixelsX.setAlignment(Qt.AlignCenter)

        self.lineEditValueX = qt.QLineEdit()
        self.lineEditValueX.setText("1")
        self.lineEditValueX.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.lineEditValueX.setAlignment(Qt.AlignRight | Qt.AlignVCenter)

        self.labelValueX = qt.QLabel("um")
        self.labelValueX.setFixedSize(40*vrb.ratio, 30*vrb.ratio)
        self.labelValueX.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)

        self.labelY = qt.QLabel("Y : ")
        self.labelY.setFixedSize(10*vrb.ratio, 30*vrb.ratio)

        self.lineEditPixelsY = qt.QLineEdit()
        self.lineEditPixelsY.setText("1")
        self.lineEditPixelsY.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.lineEditPixelsY.setAlignment(Qt.AlignRight | Qt.AlignVCenter)

        self.labelPixelsY = qt.QLabel("pixels = ")
        self.labelPixelsY.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.labelPixelsY.setAlignment(Qt.AlignCenter)

        self.lineEditValueY = qt.QLineEdit()
        self.lineEditValueY.setText("1")
        self.lineEditValueY.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.lineEditValueY.setAlignment(Qt.AlignRight | Qt.AlignVCenter)

        self.labelValueY = qt.QLabel("um")
        self.labelValueY.setFixedSize(40*vrb.ratio, 30*vrb.ratio)
        self.labelValueY.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)

        self.labelZ = qt.QLabel("Z : ")
        self.labelZ.setFixedSize(10*vrb.ratio, 30*vrb.ratio)

        self.lineEditPixelsZ = qt.QLineEdit()
        self.lineEditPixelsZ.setText("1")
        self.lineEditPixelsZ.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.lineEditPixelsZ.setAlignment(Qt.AlignRight | Qt.AlignVCenter)

        self.labelPixelsZ = qt.QLabel("pixels = ")
        self.labelPixelsZ.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.labelPixelsZ.setAlignment(Qt.AlignCenter)

        self.lineEditValueZ = qt.QLineEdit()
        self.lineEditValueZ.setText("1")
        self.lineEditValueZ.setFixedSize(60*vrb.ratio, 30*vrb.ratio)
        self.lineEditValueZ.setAlignment(Qt.AlignRight | Qt.AlignVCenter)

        self.labelValueZ = qt.QLabel("um")
        self.labelValueZ.setFixedSize(40*vrb.ratio, 30*vrb.ratio)
        self.labelValueZ.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.labelX, 0, 0)
        self.layout.addWidget(self.lineEditPixelsX, 0, 1)
        self.layout.addWidget(self.labelPixelsX, 0, 2)
        self.layout.addWidget(self.lineEditValueX, 0, 3)
        self.layout.addWidget(self.labelValueX, 0, 4)
        self.layout.addWidget(self.labelY, 1, 0)
        self.layout.addWidget(self.lineEditPixelsY, 1, 1)
        self.layout.addWidget(self.labelPixelsY, 1, 2)
        self.layout.addWidget(self.lineEditValueY, 1, 3)
        self.layout.addWidget(self.labelValueY, 1, 4)
        self.layout.addWidget(self.labelZ, 2, 0)
        self.layout.addWidget(self.lineEditPixelsZ, 2, 1)
        self.layout.addWidget(self.labelPixelsZ, 2, 2)
        self.layout.addWidget(self.lineEditValueZ, 2, 3)
        self.layout.addWidget(self.labelValueZ, 2, 4)

        self.layout.setAlignment(Qt.AlignLeft)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3, 0, 3, 0)
        self.layout.setHorizontalSpacing(3)
        self.layout.setVerticalSpacing(3)

        self.setStyleSheet("QGroupBox {border: 1px solid gray; }")

        self.setFixedSize(320*vrb.ratio, 122*vrb.ratio)

class GroupBoxSave(qt.QGroupBox):

    def __init__(self, parent=None,type=None):
        qt.QGroupBox.__init__(self)

        totalWidth = 0

        self.lineEditSave = qt.QLineEdit()
        self.lineEditSave.setFixedSize(170*vrb.ratio, 30*vrb.ratio)
        self.lineEditSave.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
        totalWidth += 170*vrb.ratio

        if type == "current":
            self.buttonSave = wgt.PushButtonImage(vrb.folderImages + "/Add_2.png")
            self.buttonSave.setToolTip(txt.dictToolTips["AddCalibration"])
            self.lineEditSave.setPlaceholderText("Add this calibration")
        else:
            self.buttonSave = wgt.PushButtonImage(vrb.folderImages + "/Save.png")
            self.buttonSave.setToolTip(txt.dictToolTips["SaveCalibration"])
            self.lineEditSave.setPlaceholderText("Save this calibration")
        self.buttonSave.setFixedSize(30*vrb.ratio, 30*vrb.ratio)
        totalWidth += 30*vrb.ratio

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.lineEditSave, 0, 0)
        self.layout.addWidget(self.buttonSave, 0, 1)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3, 0, 3, 0)
        self.layout.setHorizontalSpacing(3)

        totalWidth += 3 * 1*vrb.ratio + 8*vrb.ratio

        self.setStyleSheet("QGroupBox {border: 0px transparent; }")

        self.setFixedSize(totalWidth, 36*vrb.ratio)

class GroupBoxValidate(qt.QGroupBox):

    def __init__(self, parent=None):
        qt.QGroupBox.__init__(self)

        self.buttonValidate = wgt.PushButtonImage(vrb.folderImages + "/Validate.png", margins=2)
        self.buttonValidate.setFixedSize(30*vrb.ratio,30*vrb.ratio)
        self.buttonValidate.setToolTip("Validate")
        self.buttonCancel = wgt.PushButtonImage(vrb.folderImages + "/Back.png", margins=2)
        self.buttonCancel.setFixedSize(30*vrb.ratio,30*vrb.ratio)
        self.buttonCancel.setToolTip("Validate")

        self.layout = qt.QGridLayout()

        self.layout.addWidget(self.buttonValidate, 0, 0)
        self.layout.addWidget(self.buttonCancel, 0, 1, Qt.AlignLeft)

        self.setLayout(self.layout)

        self.layout.setSizeConstraint(1)
        self.layout.setContentsMargins(3, 0, 3, 0)
        self.layout.setHorizontalSpacing(10*vrb.ratio)

        self.setStyleSheet("QGroupBox {border: 0px transparent; }")

        self.setFixedSize(120*vrb.ratio + 1*3*vrb.ratio + 8*vrb.ratio,36*vrb.ratio)

if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    sys._excepthook = sys.excepthook

    def exception_hook(exctype, value, traceback):
        print(exctype, value, traceback)
        sys._excepthook(exctype, value, traceback)
        sys.exit(1)

    sys.excepthook = exception_hook

    # foo = GroupBoxCalibration()
    #foo = GroupBoxUserCalibration()
    # foo = GroupBoxNewCalibration()
    # foo = GroupBoxSquarePixels()
    #foo = GroupBoxRectangularPixels()
    # foo = GroupBoxSave()
    # foo = GroupBoxNewCalibration()
    #foo = WidgetCalibration()
    foo = GroupBoxCalibration()

    foo.show()

    app.exec_()
