import sys
import PyQt5.QtWidgets as qt
from PyQt5 import QtCore
from PyQt5 import QtGui
from PyQt5.QtCore import pyqtSignal,QPointF,Qt,QCoreApplication

import UsefullVariables as vrb
import UsefullTexts as txt

import DatabaseFunction as Dfct
import xml.etree.ElementTree as xmlet

import WidgetTypes


class FormulaWidget(qt.QGroupBox):
    """
    Widget used to edit formulas, shows a simple line edit and a button for further edits
    """
    def __init__(self, xmlElement, type):
        super().__init__()
        if type != WidgetTypes.InputType.FORMULA:
            print('ERROR : InputType does not match', file=sys.stderr)
        self.lineEditFormula = qt.QLineEdit()

        self.pushButtonEdit = qt.QPushButton("Edit...")
        self.formulaEditionWidget = FormulaEditionWidget()

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.lineEditFormula, 0, 0)
        self.layout.addWidget(self.pushButtonEdit, 0, 1)

        self.setLayout(self.layout)
        self.layout.setContentsMargins(0, 0, 0, 0)
        self.setStyleSheet("FormulaWidget {border: 0px transparent; }")

        self.setFixedHeight(vrb.DEFAULT_SIZE)
        self.pushButtonEdit.clicked.connect(self.showFormulaEditionWindow)
        self.formulaEditionWidget.pushButtonValidate.clicked.connect(self.closeFormulaEditionWindow)

    def showFormulaEditionWindow(self):
        self.formulaEditionWidget.setText(self.lineEditFormula.text())
        self.formulaEditionWidget.show()

    def closeFormulaEditionWindow(self):
        text = self.formulaEditionWidget.lineEditIf.getText()
        self.lineEditFormula.setText(text)
        self.formulaEditionWidget.close()

    def interfaceToXml(self, number):
        paramNode = xmlet.Element('Parameter_' + str(number))
        Dfct.SubElement(paramNode, 'Type').text = WidgetTypes.InputType.FORMULA.value
        value = self.lineEditFormula.text()
        Dfct.SubElement(paramNode, 'Value').text = value
        return paramNode

    @staticmethod
    def xmlToValue(xmlElement):
        value = Dfct.childText(xmlElement, 'Value')
        return value


class FormulaEditionWidget(qt.QWidget):
    """
    Interactive code editor
    Has a quick guide
    """
    def __init__(self):
        super().__init__()
        self.lineEditIf = LineEditIf()
        self.helpPanel = qt.QTextBrowser()
        self.helpPanel.setLineWrapMode(False)
        self.helpPanel.setHtml(quickGuideHTML())
        self.pushButtonValidate = qt.QPushButton('OK')

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.lineEditIf, 0, 0)
        self.layout.addWidget(self.helpPanel, 0, 1, 2, 1)
        self.layout.addWidget(self.pushButtonValidate, 1, 0)

        self.setLayout(self.layout)

        self.lineEditIf.layout.setContentsMargins(5, 5, 5, 5)

    def setText(self, text):
        self.lineEditIf.clear()
        self.lineEditIf.setText(text.replace(' ', ''))


def quickGuideHTML():
    style = 'h1{color:red;font-size:20px;}\n'
    style += '.function{color:blue}\n'
    style += '.argument{color:green}\n'
    style += '.description{color:dark-grey}\n'

    def functionHTML(name, args, description, parenthesis=True):
        textFunction = '<span class="function">' + name + '</span>'
        if parenthesis:
            textFunction += '('
        textFunction += '<span class="argument"> ' + args + '</span>'
        if parenthesis:
            textFunction += ')'
        textFunction += '<span class="description">' + description + '</span>'
        return textFunction

    text = '<!DOCTYPE html>\n'
    text += '<head>\n'
    text += '<title>Quick Guide</title>\n'
    text += '<style>\n'
    text += style
    text += '</style>\n'
    text += '</head>\n'
    text += '<body>\n'
    text += '\n'
    text += '<h1>Quick Guide</h1><br/>\n'
    text += functionHTML('', 'I1', ': Input image 1', False) + '\n<br/>'
    text += functionHTML('', 'I2', ': Input image 2', False) + '\n<br/>'
    text += functionHTML('', 'I3', ': Input image 3', False) + '\n'
    text += '<h2>Dynamic Constants:</h2><br/>\n'
    text += functionHTML('', 'x', ': X coordinates', False) + '\n<br/>'
    text += functionHTML('', 'y', ': Y coordinates', False) + '\n<br/>'
    text += functionHTML('', 'Sx', ': Width of an image', False) + '\n<br/>'
    text += functionHTML('', 'Sy', ': Height of an image', False) + '\n<br/>'
    text += functionHTML('', 'Cx', ': Width center', False) + '\n<br/>'
    text += functionHTML('', 'Cy', ': Height center', False) + '\n<br/>'
    text += '<h2>Image Constants:</h2><br/>\n'
    text += functionHTML('GLMin', 'I1', ': Minimum grey level of I1') + '\n<br/>'
    text += functionHTML('GLMax', 'I1', ': Maximum grey level of I1') + '\n<br/>'
    text += functionHTML('GLMean', 'I1', ': Mean grey level of I1') + '\n<br/>'
    text += functionHTML('GLStdDev', 'I1', ', ') + functionHTML('GLVar', 'I1', ': Standard deviation, Variance grey level of I1') + '\n'
    text += '<h2>Static Constants:</h2><br/>\n'
    text += functionHTML('', 'pi', ': Pi value', False) + '\n<br/>'
    text += functionHTML('', 'e', ': Euler number value', False) + '\n'
    text += '<h2>Booleans:</h2><br/>\n'
    text += functionHTML('', 'true, false', '', False) + '\n'
    text += '<h2>Logic Functions:</h2><br/>\n'
    text += functionHTML('if', 'C, A1, A2', ': if(C) then A1 else A2') + '\n<br/>'
    text += functionHTML('lut', 'C1:V1, C2:V2, ..., Cn:Vn; defaultValue', ': If C1 is True, returns V1...<br/>Returns defaultValue if none of the previous conditions are True') + '\n'
    text += '<h2>Functions:</h2>\n'
    text += '<h3>Unary:</h3>\n'
    text += functionHTML('abs', 'V', ', ') + functionHTML('sqrt', 'V', ', ') + functionHTML('exp', 'V', ', ') + functionHTML('log', 'V', '') + '\n<br/>'
    text += functionHTML('cos', 'V', ', ') + functionHTML('sin', 'V', ', ') + functionHTML('tan', 'V', '') + '\n<br/>'
    text += functionHTML('acos', 'V', ', ') + functionHTML('asin', 'V', ', ') + functionHTML('atan', 'V', '') + '\n'
    text += '<h3>Binary:</h3>\n'
    text += functionHTML('atan2', 'V1, V2', ', ') + '\n'
    text += '<h3>Multiple:</h3>\n'
    text += functionHTML('min', 'V1, ..., Vn', ', ') + functionHTML('max', 'V1, ..., Vn', ', ') + functionHTML('sum', 'V1, ..., Vn', ', ') + functionHTML('mean', 'V1, ..., Vn', '') + '\n<br/>'
    text += functionHTML('median', 'V1, ..., Vn', ', ') + functionHTML('stddev', 'V1, ..., Vn', ', ') + functionHTML('variance', 'V1, ..., Vn', '') + '\n<br/>'
    text += functionHTML('normL1', 'V1, ..., Vn', ', ') + functionHTML('normL2', 'V1, ..., Vn', ': Absolute value norm, Euclidian norm') + '\n'
    text += '<h3>Coupled</h3>\n'
    text += functionHTML('distance', 'x0, y0, x1, y1, ...', ': Distance value computation')
    text += '</body>\n'
    text += '\n'

    return text


class LineEditIf(qt.QWidget):
    """
    ComboBox + label visible
    id the comboBox is on If, shows a groupBoxThen
    """
    def __init__(self):
        super().__init__()

        self.comboBoxIf = qt.QComboBox()
        self.comboBoxIf.addItem('Do')
        self.comboBoxIf.addItem('If')
        self.lineEditCondition = qt.QLineEdit()
        self.groupBoxElse = None

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.comboBoxIf, 0, 0)
        self.layout.addWidget(self.lineEditCondition, 0, 1)

        self.setLayout(self.layout)
        self.layout.setAlignment(Qt.AlignTop)
        self.layout.setContentsMargins(0, 0, 0, 0)

        self.comboBoxIf.currentIndexChanged.connect(self.showHideIf)

    def showHideIf(self):
        if self.comboBoxIf.currentIndex() == 1:
            if self.groupBoxElse is None:
                self.groupBoxElse = GroupBoxThen()
                self.layout.addWidget(self.groupBoxElse, 1, 0, 1, 2)
            self.groupBoxElse.setVisible(True)
        else:
            if self.groupBoxElse is not None:
                self.groupBoxElse.setVisible(False)

    def getText(self):
        if self.comboBoxIf.currentIndex() == 0:
            return self.lineEditCondition.text()
        else:
            if self.groupBoxElse is not None:
                text = 'if ('
                text += self.lineEditCondition.text() + ', '
                text += self.groupBoxElse.lineEditThen.getText() + ', '
                text += self.groupBoxElse.lineEditElse.getText() + ')'
                return text

    def setText(self, text):
        if text.startswith('if('):
            self.comboBoxIf.setCurrentIndex(1)
            text = text[3:-1]
            nbParentheses = 0
            nbOfIfVirgule = 0
            currentText = ''
            for i in range(len(text)):
                if text[i] != ',' or nbParentheses > 0:
                    currentText += text[i]
                    if text[i] == '(':
                        nbParentheses += 1
                    elif text[i] == ')':
                        nbParentheses -= 1
                else:
                    if nbOfIfVirgule == 0:
                        self.lineEditCondition.setText(currentText)
                    else:
                        self.groupBoxElse.lineEditThen.setText(currentText)
                    nbOfIfVirgule += 1
                    currentText = ''
            self.groupBoxElse.lineEditElse.setText(currentText)

        else:
            self.lineEditCondition.setText(text)

    def clear(self):
        self.lineEditCondition.setText('')
        self.comboBoxIf.setCurrentIndex(0)
        if self.groupBoxElse is not None:
            self.groupBoxElse.lineEditThen.clear()
            self.groupBoxElse.lineEditElse.clear()


class GroupBoxThen(qt.QGroupBox):
    """
    2 groupBoxIf separated by an else
    """
    def __init__(self):
        super().__init__()

        self.lineEditThen = LineEditIf()
        self.labelElse = qt.QLabel('Else')
        self.lineEditElse = LineEditIf()

        self.layout = qt.QGridLayout()
        self.layout.addWidget(self.lineEditThen, 0, 1)
        self.layout.addWidget(self.labelElse, 1, 0)
        self.layout.addWidget(self.lineEditElse, 2, 1)

        self.setLayout(self.layout)
        self.layout.setAlignment(Qt.AlignTop)
        self.layout.setContentsMargins(0, 0, 0, 0)
        self.setStyleSheet("GroupBoxThen {border: 0px transparent; }")

if __name__ == '__main__':

    app = QCoreApplication.instance()
    if app is None:
        app = qt.QApplication([])

    # foo = LineEditIf()
    # foo.layout.setContentsMargins(5, 5, 5, 5)

    foo = FormulaWidget(None, WidgetTypes.InputType.FORMULA)
    foo.show()

    app.exec_()